/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain.view;

import java.awt.*;
import java.awt.geom.*;
import java.awt.font.*;
import javax.swing.tree.TreeSelectionModel;

import com.redhat.rhdb.explain.ExplainTreeNode;
import com.redhat.rhdb.treedisplay.DefaultTreeRenderer;
import com.redhat.rhdb.treedisplay.TreeLayoutNode;
import com.redhat.rhdb.treedisplay.TreeLayoutEdge;
import com.redhat.rhdb.treedisplay.TreeDisplay;

/**
 * This class is the renderer class that the ExplainView's TreeDisplay
 * component uses when drawing the plan tree. Nodes are currently
 * drawn as the name of the node (a string) and edges are arrows.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 1.0
 */
public class ExplainTreeRenderer extends DefaultTreeRenderer {

	/**
	 * Code for drawing a node.
	 *
	 * @param node a <code>TreeLayoutNode</code> value
	 * @param g a <code>Graphics</code> value
	 * @param m a <code>TreeSelectionModel</code> value
	 */
	public void drawNode(TreeLayoutNode node, Graphics g, TreeSelectionModel m)
	{
		Graphics2D g2d = (Graphics2D) g;

		if (node == null)
			return;

		ExplainTreeLayoutNode etln = (ExplainTreeLayoutNode) node;
		ExplainTreeNode n = (ExplainTreeNode) node.getData();
		Rectangle rect = new Rectangle(node.getX(), node.getY(), node.getWidth(), node.getHeight());
		Object data = node.getData();

		if (m.getLeadSelectionPath() != null &&
			m.getLeadSelectionPath().getLastPathComponent() == data)
		{
			rect.x -= 5;
			rect.y += 5;
			rect.width += 10;
			rect.height -= 10;
		}

		if (! rect.intersects(g.getClipBounds()))
			return;

		// selected?
		if (m.getLeadSelectionPath() != null &&
			m.getLeadSelectionPath().getLastPathComponent() == data)
		{
			g2d.setColor(new Color(204, 204, 255));
			g2d.fill(rect);
		}

		// draw image
		Image i = etln.getIcon();
		if (i != null)
		{
			int x = (etln.getWidth() - i.getWidth(null)) / 2;
			g2d.drawImage(i, node.getX() + x, node.getY() + ExplainTreeLayoutNode.ICON_SPACE, null);
		}

		// draw text
		g2d.setColor(Color.black);

		String text = n.toString();
		if (text == null)
			text = "<null>";

		TextLayout tl = new TextLayout(text,
									   new Font("Helvetica", Font.PLAIN, 12),
									   g2d.getFontRenderContext());
		Rectangle2D bounds = tl.getBounds();

		tl.draw(g2d,
				(float) (node.getX() + (node.getWidth()/2 - bounds.getWidth()/2)),
				(float) (node.getY() + i.getHeight(null) + 2*ExplainTreeLayoutNode.ICON_SPACE + bounds.getHeight()/2));
	}

	/**
	 * Code for drawing an edge. Draws an arrow.
	 *
	 * @param edge a <code>TreeLayoutEdge</code> value
	 * @param g a <code>Graphics</code> value
	 * @param m a <code>TreeSelectionModel</code> value
	 */
	public void drawEdge(TreeLayoutEdge edge, Graphics g, TreeSelectionModel m)
	{
		Graphics2D g2d = (Graphics2D) g;
		int epsilon = 5;
		int l = 15;
		double alpha = Math.toRadians(22.5);
		double w, h;
		double dx, dy;
		Point p, c, a1, a2;

		TreeLayoutNode parent = edge.getParent();
		TreeLayoutNode child = edge.getChild();
		int offset = edge.getParentOffset();

		// compute line endpoints

		switch (edge.getTreeLayoutModel().getOrientation())
		{
			case TreeDisplay.TOP:
				p = new Point(parent.getX() + offset, parent.getY() + parent.getHeight());
				c = new Point(child.getX() + (child.getWidth() / 2), child.getY());
				break;

			case TreeDisplay.BOTTOM:
				p = new Point(parent.getX() + offset, parent.getY());
				c = new Point(child.getX() + (child.getWidth() / 2), child.getY() + child.getHeight());
				break;

			case TreeDisplay.LEFT:
				p = new Point(parent.getX() + parent.getWidth() + 10, parent.getY() + offset);
				c = new Point(child.getX() - 10, child.getY() + (child.getHeight() / 2));
				break;

			case TreeDisplay.RIGHT:
				p = new Point(parent.getX() - 10, parent.getY() + offset);
				c = new Point(child.getX() + child.getWidth() + 10, child.getY() + (child.getHeight() / 2));
				break;

			default:
				p = new Point(parent.getX() + offset, parent.getY() + parent.getHeight());
				c = new Point(child.getX() + (child.getWidth() / 2), child.getY());
				break;
		}

		// clipping
		Shape clip = g2d.getClipBounds();
		Rectangle rect = new Line2D.Double(p, c).getBounds();

		if (rect.width < epsilon)
			rect = new Rectangle(rect.x - 6, rect.y, rect.width + 13, rect.height);
		if (rect.height < epsilon)
			rect = new Rectangle(rect.x, rect.y - 6, rect.width, rect.height + 13);

		if (! clip.intersects(rect))
			return;

		// for drawing a straight line to child.. hmmm.. this is
		// making the assumption that child is direcly below us,
		// which is not always true. so give a little bit of leeway.
		// really only for TOP, BOTTOM orientations.
		if (parent.getChildCount() == 1 && Math.abs((parent.getX() + (parent.getWidth() / 2)) -
													(child.getX() + (child.getWidth() / 2))) < epsilon)
			c.x = p.x;

		// arrow lines
		dx = p.x - c.x;
		dy = p.y - c.y;
		double len = Math.sqrt(dx*dx + dy*dy);
		h = l * Math.cos(alpha);
		w = l * Math.sin(alpha);

		// points on a horizontal line
		double a1x = len - h;
		double a1y = w;
		double a2x = a1x;
		double a2y = -a1y;

		double sin_theta = dy / len;
		double cos_theta = dx / len;

		// rotate points
		int result1_x = (int)(a1x*cos_theta - a1y*sin_theta + 0.5) + c.x;
		int result1_y = (int)(a1x*sin_theta + a1y*cos_theta + 0.5) + c.y;
		int result2_x = (int)(a2x*cos_theta - a2y*sin_theta + 0.5) + c.x;
		int result2_y = (int)(a2x*sin_theta + a2y*cos_theta + 0.5) + c.y;

		a1 = new Point(result1_x, result1_y);
		a2 = new Point(result2_x, result2_y);

		GeneralPath path = new GeneralPath();
		path.moveTo(a1.x, a1.y);
		path.lineTo(p.x, p.y);
		path.lineTo(a2.x, a2.y);

		// draw
		g2d.setColor(Color.black);
		g2d.setStroke(new BasicStroke(1.0f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
		g2d.drawLine(p.x, p.y, c.x, c.y);
		g2d.draw(path);
	}
}// ExplainTreeRenderer
