/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLSubject.cpp - SAML subject implementation

   Scott Cantor
   5/8/02

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;


SAMLSubject::SAMLSubject(
    SAMLNameIdentifier* name,
    const Iterator<const XMLCh*>& confirmationMethods,
    DOMElement* confirmationData,
    DOMElement* keyInfo
    ) : m_name(name), m_confirmationData(NULL), m_keyInfo(NULL), m_scratch(NULL)
{
    RTTI(SAMLSubject);
    if (confirmationData && (!XML::isElementNamed(confirmationData, XML::SAML_NS, L(SubjectConfirmationData))))
        throw SAMLException("confirmationData must be a saml:SubjectConfirmationData element");

    if (name)
        name->setParent(this);
    while (confirmationMethods.hasNext())
        m_confirmationMethods.push_back(XML::assign(confirmationMethods.next()));
    if (confirmationData) {
        m_scratch=DOMImplementationRegistry::getDOMImplementation(NULL)->createDocument();
        m_confirmationData=static_cast<DOMElement*>(m_scratch->importNode(confirmationData,true));
    }
    if (keyInfo) {
        if (!m_scratch)
            m_scratch=DOMImplementationRegistry::getDOMImplementation(NULL)->createDocument();
        m_keyInfo=static_cast<DOMElement*>(m_scratch->importNode(keyInfo,true));
    }
}

SAMLSubject::SAMLSubject(DOMElement* e) : m_name(NULL), m_confirmationData(NULL), m_keyInfo(NULL), m_scratch(NULL)
{
    RTTI(SAMLSubject);
    fromDOM(e);
}

SAMLSubject::SAMLSubject(istream& in) : SAMLObject(in), m_name(NULL), m_confirmationData(NULL), m_keyInfo(NULL), m_scratch(NULL)
{
    RTTI(SAMLSubject);
    fromDOM(m_document->getDocumentElement());
}

SAMLSubject::~SAMLSubject()
{
    if (m_scratch)
        m_scratch->release();
    delete m_name;
    if (m_bOwnStrings) {
        for (vector<const XMLCh*>::const_iterator i=m_confirmationMethods.begin(); i!=m_confirmationMethods.end(); i++) {
            XMLCh* temp=const_cast<XMLCh*>(*i);
            XMLString::release(&temp);
        }
    }
}

void SAMLSubject::ownStrings()
{
    if (!m_bOwnStrings) {
        for (vector<const XMLCh*>::iterator i=m_confirmationMethods.begin(); i!=m_confirmationMethods.end(); i++)
            (*i)=XML::assign(*i);
        m_bOwnStrings=true;
    }
}

void SAMLSubject::fromDOM(DOMElement* e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking && !XML::isElementNamed(e,XML::SAML_NS,L(Subject)))
        throw MalformedException("SAMLSubject::fromDOM() requires saml:Subject at root");

    // Look for NameIdentifier.
    DOMElement* n = XML::getFirstChildElement(e,XML::SAML_NS,L(NameIdentifier));
    if (n) {
        m_name = SAMLNameIdentifier::getInstance(n);
        m_name->setParent(this);
        n = XML::getNextSiblingElement(n);
    }

    // Look for SubjectConfirmation.
    if (n && XML::isElementNamed(n,XML::SAML_NS,L(SubjectConfirmation))) {
        // Iterate over ConfirmationMethods.
        DOMElement* cm=XML::getFirstChildElement(n,XML::SAML_NS,L(ConfirmationMethod));
        while (cm) {
            if (cm->hasChildNodes())
                m_confirmationMethods.push_back(cm->getFirstChild()->getNodeValue());
            cm=XML::getNextSiblingElement(cm,XML::SAML_NS,L(ConfirmationMethod));
        }

        // Extract optional SubjectConfirmationData.
        m_confirmationData=XML::getFirstChildElement(n,XML::SAML_NS,L(SubjectConfirmationData));
        
        // Extract optional ds:KeyInfo.
        m_keyInfo=XML::getFirstChildElement(n,XML::XMLSIG_NS,L(KeyInfo));
    }
    checkValidity();
}

void SAMLSubject::setNameIdentifier(SAMLNameIdentifier* name)
{
    delete m_name;
    m_name=NULL;
    if (name)
        m_name=static_cast<SAMLNameIdentifier*>(name->setParent(this));
    ownStrings();
    setDirty();
}

void SAMLSubject::setConfirmationMethods(const Iterator<const XMLCh*>& methods)
{
    while (m_confirmationMethods.size())
        removeConfirmationMethod(0);
    while (methods.hasNext())
        addConfirmationMethod(methods.next());
}

void SAMLSubject::addConfirmationMethod(const XMLCh* method)
{
    if (XML::isEmpty(method))
        throw SAMLException("confirmation method cannot be null or empty");
    
    ownStrings();
    m_confirmationMethods.push_back(XML::assign(method));
    setDirty();
}

void SAMLSubject::removeConfirmationMethod(unsigned long index)
{
    if (m_bOwnStrings) {
        XMLCh* ch=const_cast<XMLCh*>(m_confirmationMethods[index]);
        XMLString::release(&ch);
    }
    m_confirmationMethods.erase(m_confirmationMethods.begin()+index);
    ownStrings();
    setDirty();
}

void SAMLSubject::setConfirmationData(DOMElement* data)
{
    if (data && (!XML::isElementNamed(data, XML::SAML_NS, L(SubjectConfirmationData))))
        throw SAMLException("confirmationData must be a saml:SubjectConfirmationData element");

    if (m_confirmationData) {
        if (m_confirmationData->getParentNode())
            m_confirmationData->getParentNode()->removeChild(m_confirmationData);
        m_confirmationData->release();
        m_confirmationData=NULL;
    }
    if (m_document)
        m_confirmationData = static_cast<DOMElement*>(m_document->importNode(data,true));
    else {
        if (!m_scratch)
            m_scratch=DOMImplementationRegistry::getDOMImplementation(NULL)->createDocument();
        m_confirmationData = static_cast<DOMElement*>(m_scratch->importNode(data,true));
    }
    ownStrings();
    setDirty();
}

void SAMLSubject::setKeyInfo(DOMElement* keyInfo)
{
    if (m_keyInfo) {
        if (m_keyInfo->getParentNode())
            m_keyInfo->getParentNode()->removeChild(m_keyInfo);
        m_keyInfo->release();
        m_keyInfo=NULL;
    }
    if (m_document)
        m_keyInfo = static_cast<DOMElement*>(m_document->importNode(keyInfo,true));
    else {
        if (!m_scratch)
            m_scratch=DOMImplementationRegistry::getDOMImplementation(NULL)->createDocument();
        m_keyInfo = static_cast<DOMElement*>(m_scratch->importNode(keyInfo,true));
    }
    ownStrings();
    setDirty();
}

DOMElement* SAMLSubject::buildRoot(DOMDocument* doc, bool xmlns) const
{ 
    DOMElement* s = doc->createElementNS(XML::SAML_NS, L(Subject));
    if (xmlns)
        s->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    return s;
}

DOMNode* SAMLSubject::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc, xmlns);
    DOMElement* s=static_cast<DOMElement*>(m_root);
    doc=s->getOwnerDocument();

    if (m_bDirty) {
        if (m_name)
            s->appendChild(m_name->toDOM(doc,false));

        if (!m_confirmationMethods.empty()) {
            DOMElement* conf = doc->createElementNS(XML::SAML_NS,L(SubjectConfirmation));
            for (vector<const XMLCh*>::const_iterator i=m_confirmationMethods.begin(); i!=m_confirmationMethods.end(); i++)
                conf->appendChild(doc->createElementNS(XML::SAML_NS,L(ConfirmationMethod)))->appendChild(doc->createTextNode(*i));
            if (m_confirmationData) {
                if (m_confirmationData->getOwnerDocument() != doc) {
                    DOMElement* copy=static_cast<DOMElement*>(doc->importNode(m_confirmationData,true));
                    if (m_confirmationData->getParentNode())
                        m_confirmationData->getParentNode()->removeChild(m_confirmationData);
                    m_confirmationData->release();
                    m_confirmationData=copy;
                }
                conf->appendChild(m_confirmationData);
            }
            if (m_keyInfo) {
                if (m_keyInfo->getOwnerDocument() != doc) {
                    DOMElement* copy=static_cast<DOMElement*>(doc->importNode(m_keyInfo,true));
                    if (m_keyInfo->getParentNode())
                        m_keyInfo->getParentNode()->removeChild(m_keyInfo);
                    m_keyInfo->release();
                    m_keyInfo=copy;
                }
                conf->appendChild(m_keyInfo);
            }
            s->appendChild(conf);
        }
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(s,XML::SAML_NS);
    }

    return m_root;
}

void SAMLSubject::checkValidity() const
{
    if (!m_name && m_confirmationMethods.empty())
        throw MalformedException("Subject is invalid, requires either NameIdentifier or at least one ConfirmationMethod");
}

SAMLObject* SAMLSubject::clone() const
{
    return new SAMLSubject(
        static_cast<SAMLNameIdentifier*>(m_name->clone()),
        m_confirmationMethods,
        m_confirmationData,
        m_keyInfo
        );
}

const XMLCh SAMLSubject::CONF_ARTIFACT01[] = // urn:oasis:names:tc:SAML:1.0:cm:artifact-01
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_c, chLatin_m, chColon, chLatin_a, chLatin_r, chLatin_t, chLatin_i, chLatin_f, chLatin_a, chLatin_c, chLatin_t,
        chDash, chDigit_0, chDigit_1, chNull
};

const XMLCh SAMLSubject::CONF_ARTIFACT[] = // urn:oasis:names:tc:SAML:1.0:cm:artifact
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_c, chLatin_m, chColon, chLatin_a, chLatin_r, chLatin_t, chLatin_i, chLatin_f, chLatin_a, chLatin_c, chLatin_t, chNull
};

const XMLCh SAMLSubject::CONF_BEARER[] = // urn:oasis:names:tc:SAML:1.0:cm:bearer
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_c, chLatin_m, chColon, chLatin_b, chLatin_e, chLatin_a, chLatin_r, chLatin_e, chLatin_r, chNull
};

const XMLCh SAMLSubject::CONF_HOLDER_KEY[] = // urn:oasis:names:tc:SAML:1.0:cm:holder-of-key
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_c, chLatin_m, chColon, chLatin_h, chLatin_o, chLatin_l, chLatin_d, chLatin_e, chLatin_r, chDash,
        chLatin_o, chLatin_f, chDash, chLatin_k, chLatin_e, chLatin_y, chNull
};

const XMLCh SAMLSubject::CONF_SENDER_VOUCHES[] = // urn:oasis:names:tc:SAML:1.0:cm:sender-vouches
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_c, chLatin_m, chColon, chLatin_s, chLatin_e, chLatin_n, chLatin_d, chLatin_e, chLatin_r, chDash,
        chLatin_v, chLatin_o, chLatin_u, chLatin_c, chLatin_h, chLatin_e, chLatin_s, chNull
};
