//
// SignedCmsTest.cs - NUnit tests for SignedCms
//
// Author:
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// (C) 2003 Motus Technologies Inc. (http://www.motus.com)
// Copyright (C) 2004-2005 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

#if NET_2_0

using NUnit.Framework;

using System;
using System.Collections;
using System.IO;
using System.Security.Cryptography;
using System.Security.Cryptography.Pkcs;
using System.Security.Cryptography.X509Certificates;

namespace MonoTests.System.Security.Cryptography.Pkcs {

	[TestFixture]
	public class SignedCmsTest {

		static byte[] asnNull = { 0x05, 0x00 };
		static string pkcs7Oid = "1.2.840.113549.1.7.1";
		static string pkcs7Name = "PKCS 7 Data";
		static string rsaOid = "1.2.840.113549.1.1.1";
		static string rsaName = "RSA";

		static public byte[] farscape_p12_pfx = { 
			0x30, 0x82, 0x07, 0x17, 0x02, 0x01, 0x03, 0x30, 0x82, 0x06, 0xD3, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x06, 0xC4, 0x04, 0x82, 0x06, 0xC0, 0x30, 0x82, 0x06, 0xBC, 0x30, 0x82, 0x03, 0xCD, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0xBE, 0x04, 0x82, 0x03, 0xBA, 0x30, 0x82, 0x03, 0xB6, 0x30, 0x82, 0x03, 0xB2, 0x06, 0x0B, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x0A, 0x01, 0x02, 0xA0, 0x82, 0x02, 0xB6, 0x30, 0x82, 0x02, 0xB2, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 
			0x0E, 0x04, 0x08, 0x86, 0x2A, 0xA9, 0x71, 0x6D, 0xA4, 0xB8, 0x2D, 0x02, 0x02, 0x07, 0xD0, 0x04, 0x82, 0x02, 0x90, 0x90, 0x14, 0xB5, 0xF0, 0xB6, 0x86, 0x56, 0xCB, 0xFA, 0x63, 0xAD, 0x9F, 0x5A, 0x59, 0x6C, 0xAD, 0x00, 0x3C, 0x37, 0x8A, 0xC3, 0x88, 0x58, 0x8B, 0xD7, 0x48, 0x53, 0x7A, 0xC8, 0x5B, 0x0D, 0x98, 0xDD, 0x8B, 0xB3, 0xEC, 0x4C, 0xAC, 0x61, 0x18, 0xE3, 0x5E, 0x47, 0xAD, 0xC7, 0x92, 0xBC, 0xD3, 0x00, 0x07, 0xFF, 0x1A, 0x68, 0x74, 0x45, 0x8E, 0xD8, 0x7C, 0x9F, 0x18, 0x7B, 0xD7, 0xC8, 0x47, 0xBA, 0x6B, 0x19, 0xF2, 0xBF, 0x7E, 0x51, 0x0B, 0x4B, 0x43, 0xE8, 0xB9, 0x56, 0x7E, 0xD0, 0x74, 0xC7, 
			0xDE, 0x76, 0xDB, 0xFF, 0x5C, 0x6B, 0x53, 0xBE, 0x31, 0x06, 0xAE, 0x6C, 0x8F, 0xDC, 0x49, 0x04, 0x71, 0x74, 0xEE, 0xB8, 0x06, 0xCB, 0xAD, 0x86, 0xB4, 0x4E, 0xB9, 0x46, 0xA1, 0x03, 0x5E, 0x0E, 0xA7, 0xC7, 0x37, 0x6B, 0xB0, 0x8D, 0x2D, 0x81, 0x1F, 0xE3, 0xC2, 0x05, 0xDE, 0xEF, 0x51, 0x07, 0x70, 0x6E, 0x35, 0x9A, 0xAD, 0x19, 0x5E, 0xAF, 0xEB, 0x7F, 0xEF, 0xE4, 0xAB, 0x07, 0xF3, 0xF6, 0xEA, 0xFA, 0x0E, 0x83, 0x65, 0x06, 0x3C, 0xF3, 0xBD, 0x96, 0x08, 0x14, 0xC5, 0x34, 0x26, 0xED, 0xC0, 0x10, 0xCC, 0xAE, 0x2D, 0x8F, 0xBE, 0xED, 0x98, 0x0D, 0x88, 0x1B, 0x1E, 0xC7, 0x37, 0xF0, 0xFC, 0xDB, 0x3C, 0xE3, 
			0x1B, 0x66, 0x52, 0x45, 0x6E, 0x05, 0xA6, 0xD9, 0x12, 0x23, 0x05, 0x5F, 0xE3, 0x9F, 0x7D, 0x21, 0x9B, 0x2E, 0x3E, 0x9E, 0x3C, 0xEE, 0xD1, 0x9B, 0x55, 0xDE, 0x57, 0x60, 0xA5, 0x24, 0x2D, 0xC7, 0x94, 0xEC, 0xFC, 0xB1, 0x6A, 0x65, 0xBD, 0x85, 0x02, 0x5C, 0x58, 0xAA, 0x5A, 0x6A, 0xF3, 0xAC, 0x6B, 0xDD, 0x0E, 0x63, 0xB2, 0x4B, 0x5B, 0x67, 0x3D, 0xC3, 0xBF, 0xE4, 0xC8, 0xEF, 0x3F, 0x89, 0x5A, 0xCD, 0x6D, 0xEF, 0x05, 0x22, 0x2B, 0x72, 0xFF, 0x80, 0x7A, 0xDD, 0xF1, 0x59, 0xA7, 0x6F, 0x00, 0xB1, 0xBD, 0x4D, 0x88, 0xD6, 0xE4, 0x8A, 0xDD, 0xA9, 0xFC, 0xD9, 0x01, 0x0A, 0x65, 0x8E, 0x52, 0xF9, 0x7E, 0x20, 
			0x72, 0x67, 0x0D, 0x5B, 0xEE, 0x67, 0x5B, 0x46, 0x4A, 0x15, 0xA2, 0x6F, 0x15, 0x2B, 0x5B, 0x9A, 0x93, 0x12, 0x4F, 0xF4, 0xAD, 0x49, 0xD0, 0x11, 0xF1, 0x7E, 0x40, 0xDE, 0x32, 0x96, 0x2E, 0xB3, 0xE8, 0x71, 0x60, 0x27, 0x6E, 0xA2, 0x71, 0x83, 0xC7, 0xFE, 0x0E, 0x8B, 0x31, 0x06, 0x64, 0xE1, 0x19, 0x02, 0xB9, 0x44, 0x25, 0x0C, 0x94, 0x64, 0x7E, 0x5F, 0x89, 0x4D, 0x7E, 0x99, 0x0B, 0x91, 0xB8, 0x22, 0xA5, 0x33, 0x92, 0xD3, 0x49, 0x07, 0x1D, 0xC6, 0x25, 0x4A, 0xD7, 0x6D, 0xE2, 0x94, 0x3F, 0xFA, 0x10, 0x72, 0x59, 0x62, 0xF5, 0xC6, 0xD4, 0x3A, 0xEE, 0x8F, 0xBC, 0x9C, 0xBC, 0xFC, 0xC7, 0x37, 0xBF, 0x7C, 
			0xA0, 0x67, 0xB0, 0xFF, 0x0F, 0x29, 0xA0, 0xA2, 0x71, 0x6B, 0x21, 0x00, 0xF4, 0x54, 0xD9, 0x3D, 0x1B, 0xCE, 0xF4, 0xFE, 0x6F, 0xF5, 0x21, 0xCB, 0x47, 0x58, 0x17, 0xF6, 0x45, 0x2F, 0xA0, 0x3B, 0x8B, 0xD9, 0xB8, 0x8A, 0x33, 0x3F, 0x16, 0xE0, 0xC7, 0x8A, 0xB8, 0x11, 0x2F, 0xA8, 0x7E, 0x7D, 0xA7, 0x7B, 0x65, 0x27, 0x89, 0x3C, 0x67, 0x4D, 0xD5, 0x70, 0x28, 0x76, 0x60, 0x96, 0x68, 0xBF, 0xFB, 0xCD, 0x49, 0xE0, 0x8A, 0x7C, 0x6F, 0x76, 0x06, 0x48, 0x6D, 0x63, 0x67, 0x8A, 0x47, 0x82, 0x5E, 0x7F, 0x0E, 0xAC, 0x46, 0xB6, 0xBC, 0x0A, 0x6D, 0xE2, 0x1A, 0x3A, 0x20, 0xA5, 0xC7, 0x81, 0x71, 0x6E, 0x2B, 0x16, 
			0x97, 0xD4, 0xFA, 0xC0, 0xDD, 0x72, 0x5B, 0x9F, 0xA3, 0x43, 0xF4, 0x85, 0xB1, 0xC6, 0xA8, 0xE0, 0x62, 0x81, 0x5D, 0xA5, 0x07, 0x29, 0x6A, 0x6A, 0x2D, 0xE1, 0x1D, 0xBE, 0x12, 0x6D, 0x42, 0x58, 0x6F, 0x4E, 0x30, 0x3D, 0xBF, 0x32, 0x11, 0x38, 0xBC, 0x36, 0x76, 0x60, 0xFC, 0x57, 0x2F, 0xD3, 0x9E, 0xC4, 0x1A, 0x92, 0xEA, 0xDE, 0x85, 0xFD, 0xE7, 0xAA, 0x30, 0xA6, 0x97, 0x2C, 0x36, 0x3B, 0x3B, 0x0E, 0x92, 0x52, 0xFF, 0x42, 0xD7, 0x62, 0x6C, 0xC1, 0x3A, 0xE7, 0x1B, 0x4E, 0x13, 0x8C, 0x95, 0xB3, 0x4B, 0xA7, 0x9E, 0x42, 0x75, 0xA8, 0xCA, 0x63, 0x76, 0xC4, 0x45, 0x74, 0x96, 0x43, 0xD8, 0x86, 0x82, 0xBE, 
			0x37, 0xFF, 0x9B, 0xEB, 0xB7, 0x18, 0xA1, 0x2F, 0xE3, 0x6C, 0x08, 0xE8, 0x11, 0x96, 0x8C, 0x5E, 0x9E, 0x2B, 0xE7, 0xDB, 0x7D, 0x54, 0xE1, 0xDB, 0x1E, 0xD3, 0x8F, 0xB5, 0x19, 0x4B, 0xB2, 0x16, 0xDB, 0xCF, 0xEC, 0x88, 0x0B, 0x6C, 0x3C, 0xE4, 0xF2, 0xC4, 0xFF, 0x4D, 0x3E, 0x53, 0x52, 0x3A, 0x81, 0x0B, 0x6E, 0xAC, 0x95, 0xEA, 0x5A, 0x6E, 0x4D, 0x83, 0x23, 0x82, 0xC9, 0x90, 0x02, 0x74, 0x10, 0x2A, 0x6C, 0xFB, 0x97, 0x4F, 0x5F, 0x70, 0x8E, 0xF0, 0xB9, 0x31, 0x81, 0xE8, 0x30, 0x0D, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x02, 0x31, 0x00, 0x30, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 
			0x86, 0xF7, 0x0D, 0x01, 0x09, 0x15, 0x31, 0x06, 0x04, 0x04, 0x01, 0x00, 0x00, 0x00, 0x30, 0x57, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x14, 0x31, 0x4A, 0x1E, 0x48, 0x00, 0x64, 0x00, 0x64, 0x00, 0x62, 0x00, 0x30, 0x00, 0x65, 0x00, 0x64, 0x00, 0x31, 0x00, 0x64, 0x00, 0x2D, 0x00, 0x32, 0x00, 0x36, 0x00, 0x30, 0x00, 0x34, 0x00, 0x2D, 0x00, 0x34, 0x00, 0x32, 0x00, 0x35, 0x00, 0x66, 0x00, 0x2D, 0x00, 0x38, 0x00, 0x31, 0x00, 0x35, 0x00, 0x66, 0x00, 0x2D, 0x00, 0x34, 0x00, 0x39, 0x00, 0x35, 0x00, 0x61, 0x00, 0x37, 0x00, 0x64, 0x00, 0x65, 0x00, 0x65, 0x00, 0x37, 0x00, 0x61, 0x00, 
			0x64, 0x00, 0x30, 0x30, 0x69, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x01, 0x31, 0x5C, 0x1E, 0x5A, 0x00, 0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20, 0x00, 0x52, 0x00, 0x53, 0x00, 0x41, 0x00, 0x20, 0x00, 0x53, 0x00, 0x43, 0x00, 0x68, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x6E, 0x00, 0x65, 0x00, 0x6C, 0x00, 0x20, 0x00, 0x43, 0x00, 0x72, 0x00, 0x79, 0x00, 0x70, 0x00, 0x74, 0x00, 0x6F, 0x00, 0x67, 0x00, 0x72, 0x00, 0x61, 0x00, 0x70, 0x00, 0x68, 0x00, 0x69, 0x00, 0x63, 0x00, 0x20, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 
			0x00, 0x76, 0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x72, 0x30, 0x82, 0x02, 0xE7, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x06, 0xA0, 0x82, 0x02, 0xD8, 0x30, 0x82, 0x02, 0xD4, 0x02, 0x01, 0x00, 0x30, 0x82, 0x02, 0xCD, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x06, 0x30, 0x0E, 0x04, 0x08, 0x0B, 0x02, 0xBA, 0x83, 0x5D, 0x71, 0x1D, 0xBD, 0x02, 0x02, 0x07, 0xD0, 0x80, 0x82, 0x02, 0xA0, 0x52, 0xD1, 0x51, 0x2A, 0xD1, 0x6D, 0x7E, 0xB0, 0x0A, 0x62, 0x6C, 0x0B, 0xE6, 0x6C, 0x72, 
			0x65, 0x3F, 0x89, 0x36, 0x1F, 0x71, 0x85, 0x00, 0x45, 0xC9, 0x56, 0x65, 0xC6, 0x43, 0xF6, 0xC1, 0x56, 0x81, 0xF0, 0xF5, 0x73, 0x57, 0xA5, 0x47, 0x45, 0xB6, 0xBD, 0xC3, 0xEB, 0xE0, 0xE0, 0x0F, 0x23, 0xCE, 0x95, 0xD7, 0x85, 0xCA, 0x73, 0xC0, 0x0E, 0x36, 0x7A, 0xF1, 0x01, 0x3F, 0x0B, 0x1C, 0xC2, 0x6C, 0x09, 0xC2, 0x43, 0x98, 0x14, 0x10, 0x80, 0x35, 0xF0, 0x45, 0x7A, 0x4F, 0x1F, 0x90, 0x3F, 0xD8, 0x08, 0xC6, 0x09, 0x22, 0xA3, 0xFD, 0x7A, 0x54, 0xB4, 0x27, 0x53, 0x20, 0x42, 0xE8, 0x89, 0xBE, 0xDC, 0x13, 0xCE, 0x9B, 0x76, 0x8F, 0xBB, 0x19, 0xA3, 0x54, 0x6E, 0xCB, 0x3C, 0x09, 0x7C, 0xC1, 0xD4, 0xCE, 
			0xF0, 0xFF, 0x95, 0xA0, 0xB6, 0x41, 0x07, 0xC0, 0xDD, 0x57, 0x36, 0xDC, 0x45, 0x65, 0xA2, 0xC8, 0xB3, 0x72, 0x3F, 0x99, 0xCA, 0x2C, 0xA0, 0x24, 0x06, 0x1E, 0xF9, 0xD3, 0xBB, 0xF4, 0x70, 0xA5, 0x2B, 0xCB, 0xFE, 0x14, 0x73, 0x8E, 0x83, 0x7A, 0x50, 0xA0, 0xB3, 0x80, 0xBC, 0xDA, 0xEF, 0x1D, 0x68, 0x35, 0xE9, 0x62, 0x3D, 0xA6, 0x0D, 0x0A, 0xF3, 0x06, 0x2C, 0x87, 0x7B, 0xC6, 0x83, 0x27, 0x1F, 0x22, 0x1E, 0xC3, 0x37, 0xD1, 0xB3, 0x81, 0x2B, 0x0E, 0xCA, 0x36, 0x2B, 0x45, 0x2C, 0xAE, 0x09, 0x23, 0xA4, 0xFF, 0xB0, 0xE6, 0x13, 0x70, 0x89, 0xB7, 0x2A, 0xD7, 0x94, 0x35, 0x1F, 0x73, 0x0E, 0x50, 0xF3, 0x5E, 
			0x92, 0xC3, 0xE7, 0x8E, 0x29, 0x32, 0xED, 0x3F, 0xCC, 0x34, 0x53, 0x54, 0xE5, 0xA1, 0x50, 0x93, 0x40, 0x95, 0x47, 0x29, 0x4B, 0x59, 0x4D, 0x28, 0xBC, 0x2F, 0xA9, 0x5F, 0xF8, 0x27, 0x22, 0x49, 0xDB, 0x66, 0xA6, 0x24, 0xE0, 0xF2, 0xF0, 0x0F, 0xCC, 0x7B, 0xE4, 0x55, 0x0D, 0xB4, 0x20, 0x73, 0xB9, 0x29, 0xA4, 0x7F, 0xDD, 0x46, 0xA0, 0x47, 0x3A, 0x03, 0x20, 0xBD, 0x6E, 0xF6, 0x88, 0x18, 0x02, 0xD2, 0xD9, 0x4F, 0xC6, 0x55, 0xA7, 0x82, 0xDB, 0x32, 0x5B, 0x1A, 0x74, 0x8D, 0xBD, 0xD8, 0x66, 0x3D, 0x0E, 0x43, 0xFE, 0x6A, 0x5E, 0xD8, 0x23, 0x04, 0x6A, 0x0F, 0x75, 0xC1, 0xCA, 0xD1, 0x04, 0xDB, 0x8D, 0x7F, 
			0x21, 0xCA, 0xE6, 0xF0, 0x3D, 0x15, 0x23, 0x87, 0x52, 0xBE, 0x8E, 0xAA, 0x4B, 0xA2, 0xFA, 0xAE, 0x33, 0xD3, 0xB4, 0x9A, 0x54, 0xCC, 0xA3, 0xE1, 0xB1, 0x6C, 0xD7, 0xA4, 0x51, 0x7B, 0x8F, 0x58, 0x01, 0x8C, 0xC3, 0xE9, 0x49, 0xB8, 0xB8, 0x01, 0x3B, 0x0D, 0x94, 0x16, 0xF4, 0x47, 0xA4, 0x9C, 0x20, 0x97, 0x35, 0x2A, 0x10, 0xCA, 0xA8, 0xB5, 0xDA, 0x0F, 0x2D, 0x0C, 0x7D, 0xA0, 0x55, 0x17, 0x9C, 0x55, 0xEA, 0x6F, 0x7D, 0xE3, 0x3B, 0xB3, 0x81, 0x0F, 0x4E, 0xD0, 0x0B, 0x88, 0x1A, 0xF6, 0xB4, 0x0F, 0x15, 0x18, 0xC5, 0x54, 0x4C, 0xF1, 0x15, 0x88, 0xAD, 0x03, 0x7E, 0x0E, 0x88, 0x34, 0xB6, 0xCF, 0x96, 0x9B, 
			0x70, 0xC9, 0x16, 0x8D, 0x63, 0xB2, 0xF6, 0x4C, 0x05, 0x7D, 0x45, 0x5F, 0xD7, 0xA7, 0xE0, 0xBC, 0xA0, 0xBE, 0xBF, 0x8B, 0x70, 0x08, 0x90, 0x93, 0x32, 0xE0, 0x23, 0x84, 0x26, 0x76, 0x85, 0x03, 0x19, 0xF7, 0xE3, 0x66, 0x41, 0xAD, 0x60, 0xEE, 0xED, 0x4D, 0x7F, 0xC7, 0xB7, 0xE4, 0xE2, 0x0B, 0xCC, 0x5C, 0x12, 0x18, 0xD8, 0xF8, 0x2E, 0x24, 0x7A, 0x4D, 0x66, 0x10, 0x9C, 0xAC, 0xF6, 0xD8, 0x51, 0x69, 0x77, 0x58, 0xD0, 0xF5, 0x15, 0xB7, 0xF0, 0xA0, 0x2F, 0xB9, 0x13, 0x8B, 0x65, 0x77, 0x1A, 0x02, 0xB1, 0xD1, 0x86, 0x25, 0xFB, 0xD5, 0x44, 0x9D, 0xBB, 0x2D, 0xF9, 0x7D, 0x77, 0xB8, 0x7F, 0x5A, 0x34, 0x08, 
			0x0B, 0x8C, 0xBE, 0x6C, 0xBD, 0xF4, 0xD0, 0x9A, 0x1E, 0x77, 0x94, 0xB3, 0x37, 0x5F, 0xED, 0x4C, 0x0D, 0x18, 0x58, 0xD1, 0x5F, 0x7D, 0xD7, 0x1A, 0xBD, 0x6D, 0x3A, 0xEF, 0xAA, 0x7B, 0xAF, 0x60, 0xB9, 0x6A, 0x89, 0x36, 0x27, 0xF1, 0xCA, 0x0F, 0xD4, 0x8D, 0x75, 0xA7, 0x62, 0x0C, 0x95, 0x4E, 0xA1, 0x03, 0xEE, 0x06, 0x5C, 0x6C, 0x3F, 0x6F, 0x37, 0x3E, 0xCE, 0x9B, 0x26, 0x89, 0x4E, 0xDD, 0x9E, 0x57, 0x72, 0xB7, 0xD7, 0xE6, 0x25, 0xB8, 0xDA, 0x91, 0x11, 0xB2, 0xB6, 0x89, 0x18, 0x42, 0xDF, 0xA6, 0x1E, 0xB5, 0x13, 0x1D, 0x90, 0x21, 0x48, 0x75, 0x58, 0x0C, 0x0A, 0x22, 0xC2, 0x07, 0x12, 0x9B, 0x73, 0x6E, 
			0x0F, 0xCE, 0x10, 0x28, 0x3D, 0x2A, 0x45, 0x64, 0x60, 0xE3, 0xB7, 0xE1, 0x76, 0x90, 0xEC, 0x5B, 0xC6, 0xA1, 0xF0, 0xC4, 0xE8, 0x12, 0xD9, 0xC6, 0x22, 0x80, 0xB5, 0x30, 0xE5, 0x17, 0xAE, 0x05, 0x96, 0xBB, 0x4E, 0xBB, 0x33, 0xBB, 0xB0, 0x63, 0x29, 0x74, 0x11, 0x06, 0x23, 0x36, 0xB4, 0xA1, 0x25, 0xD5, 0x2A, 0xF3, 0x90, 0x38, 0x18, 0x02, 0x62, 0x30, 0x3B, 0x30, 0x1F, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x04, 0x14, 0xDC, 0x3A, 0xAB, 0x36, 0xD7, 0x3E, 0xF4, 0x6C, 0x52, 0xC9, 0x89, 0x37, 0xFE, 0x71, 0x71, 0x83, 0xC6, 0x09, 0x88, 0xDD, 0x04, 0x14, 0xF5, 0x76, 0xC2, 0xCC, 0xB9, 0xE5, 
			0xF5, 0x28, 0xA3, 0x2D, 0x55, 0xDC, 0xDE, 0x3B, 0xCF, 0x53, 0xEE, 0x4B, 0x8F, 0x6F, 0x02, 0x02, 0x07, 0xD0 };

		static public byte [] intca_cer = { 0x30, 0x82, 0x02, 0x31, 0x30, 0x82, 0x01, 0x9A, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x1B, 0xFC, 0x61, 0x41, 0xD0, 0xCF, 0x41, 0xB9, 0x47, 0xEF, 0x21, 0x7D, 0x6A, 0x53, 0x32, 0x76, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x04, 0x05, 0x00, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x16, 0x4D, 0x6F, 0x6E, 0x6F, 0x20, 0x55, 0x6E, 0x69, 0x74, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x33, 0x31, 0x31, 0x32, 0x37, 0x30, 0x32, 0x33, 0x32, 0x32, 0x34, 0x5A, 0x17, 0x0D, 0x33, 
			0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x23, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x18, 0x4D, 0x6F, 0x6E, 0x6F, 0x20, 0x53, 0x53, 0x4C, 0x20, 0x49, 0x6E, 0x74, 0x65, 0x72, 0x6D, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x43, 0x41, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xB9, 0xA0, 0xA4, 0x9D, 0xC5, 0xF8, 0x60, 0x9E, 0xCB, 0x2C, 0xC5, 0xF0, 0xF7, 0x36, 0x72, 0x1B, 0x2B, 0x6E, 0x28, 0xEA, 0x40, 0x4A, 
			0x9F, 0xD1, 0x50, 0x8C, 0xB0, 0xD7, 0xAB, 0x44, 0x0C, 0x86, 0x32, 0x6D, 0x79, 0xA9, 0xFB, 0x67, 0x7E, 0x7E, 0x88, 0xE6, 0x58, 0x85, 0x47, 0x54, 0x78, 0xC2, 0xE0, 0x22, 0xA5, 0xB2, 0x83, 0x2C, 0x67, 0xF1, 0x64, 0xAC, 0xA4, 0xE5, 0x14, 0x3E, 0x3A, 0xB5, 0xC1, 0xD6, 0xD3, 0x11, 0xA6, 0x6D, 0x0D, 0x59, 0x4F, 0xE7, 0x93, 0x60, 0x45, 0xE3, 0xD5, 0x73, 0x83, 0xB5, 0xCA, 0x9E, 0xD1, 0x69, 0x12, 0x56, 0x19, 0x49, 0x19, 0xBD, 0x11, 0x91, 0x54, 0x36, 0x31, 0x97, 0x9A, 0x91, 0x4A, 0x66, 0x8C, 0x67, 0x5F, 0xE3, 0x30, 0xBC, 0xDA, 0xE5, 0x41, 0xC9, 0x6B, 0xD5, 0x7B, 0x3F, 0x08, 0x14, 0x59, 0x8C, 0xD5, 0x99, 
			0x13, 0xE3, 0x02, 0xDE, 0xB0, 0xCD, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x68, 0x30, 0x66, 0x30, 0x10, 0x06, 0x03, 0x55, 0x1D, 0x0A, 0x04, 0x09, 0x30, 0x07, 0x03, 0x02, 0x07, 0x80, 0x02, 0x01, 0x01, 0x30, 0x52, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x4B, 0x30, 0x49, 0x80, 0x10, 0xD5, 0x6B, 0x08, 0x45, 0x51, 0x7C, 0x7E, 0xE7, 0x36, 0x07, 0x6E, 0xF7, 0x07, 0xE5, 0xC3, 0xE0, 0xA1, 0x23, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x16, 0x4D, 0x6F, 0x6E, 0x6F, 0x20, 0x55, 0x6E, 0x69, 0x74, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x41, 0x82, 
			0x10, 0xAF, 0x9A, 0xC0, 0xAC, 0xB0, 0x96, 0x43, 0x85, 0x43, 0xF5, 0x85, 0xD0, 0x24, 0xB4, 0x24, 0x9C, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x04, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x5B, 0xE9, 0x92, 0x57, 0xF1, 0x5F, 0xB6, 0x1D, 0x43, 0x2E, 0xB8, 0xE6, 0xA9, 0x68, 0x76, 0x0D, 0x41, 0xF4, 0x46, 0x0D, 0x32, 0x6A, 0xA1, 0x52, 0x30, 0x31, 0x4F, 0x0E, 0x4B, 0x91, 0x83, 0x6A, 0x18, 0x18, 0x48, 0x9C, 0xBB, 0xDB, 0x1D, 0x8B, 0xB3, 0x2B, 0x7A, 0x34, 0x57, 0xD7, 0x2F, 0xE6, 0x0A, 0xD9, 0x98, 0xAA, 0xD0, 0x2C, 0x5F, 0xBC, 0x36, 0xB6, 0xC4, 0x9B, 0xF1, 0x11, 0x9F, 0x9C, 
			0xAE, 0x9F, 0x05, 0x40, 0xC5, 0x71, 0xA4, 0x0D, 0xA5, 0x2C, 0x9A, 0x77, 0x5F, 0xD5, 0x04, 0xA4, 0xE7, 0xAB, 0x9C, 0x65, 0xC8, 0x26, 0x4C, 0xF6, 0x0C, 0x0A, 0x6A, 0x50, 0x4E, 0x21, 0x9A, 0x23, 0xD2, 0xDB, 0xDC, 0x0A, 0xC8, 0xFC, 0x6D, 0x9C, 0x24, 0xCB, 0x2F, 0x3B, 0x63, 0xE1, 0x2E, 0xDA, 0xA5, 0x97, 0x90, 0xB8, 0x3E, 0xAE, 0xE2, 0x18, 0x08, 0xB9, 0x19, 0x97, 0xE0, 0x21, 0xA0, 0xFB };

		static public byte [] root_cer = { 0x30, 0x82, 0x02, 0x1D, 0x30, 0x82, 0x01, 0x86, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0xAF, 0x9A, 0xC0, 0xAC, 0xB0, 0x96, 0x43, 0x85, 0x43, 0xF5, 0x85, 0xD0, 0x24, 0xB4, 0x24, 0x9C, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x04, 0x05, 0x00, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x16, 0x4D, 0x6F, 0x6E, 0x6F, 0x20, 0x55, 0x6E, 0x69, 0x74, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x33, 0x31, 0x31, 0x32, 0x37, 0x30, 0x32, 0x32, 0x38, 0x30, 0x36, 0x5A, 0x17, 0x0D, 0x33, 
			0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x16, 0x4D, 0x6F, 0x6E, 0x6F, 0x20, 0x55, 0x6E, 0x69, 0x74, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x41, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xCC, 0x7C, 0x6B, 0xFE, 0xC4, 0x76, 0x23, 0x15, 0xBD, 0x7C, 0xD5, 0x58, 0x47, 0x92, 0x46, 0xA6, 0x8A, 0x7A, 0xA1, 0xEC, 0x8A, 0xB6, 0x35, 0x17, 
			0x9E, 0x4C, 0xD8, 0x1B, 0xCB, 0x89, 0x6A, 0x55, 0x2E, 0x0B, 0x13, 0x68, 0xCA, 0xBC, 0x33, 0xC2, 0xA9, 0x92, 0x70, 0xDF, 0xFA, 0xD9, 0x1F, 0x10, 0x5F, 0xAE, 0x2B, 0xD4, 0x43, 0x64, 0x44, 0xE3, 0x05, 0xF5, 0x11, 0xFD, 0x17, 0x29, 0x55, 0xF4, 0x60, 0x00, 0x9D, 0x23, 0x27, 0x1E, 0x73, 0x64, 0x97, 0xAB, 0x9A, 0x1B, 0x71, 0x65, 0x48, 0x25, 0x8B, 0x27, 0x3F, 0x02, 0xA1, 0xF1, 0xF0, 0x9A, 0x6A, 0x52, 0x02, 0x19, 0x0B, 0x18, 0xB8, 0xCF, 0x42, 0x2E, 0x8A, 0x8C, 0x83, 0x62, 0x12, 0x5D, 0xBA, 0x1D, 0xC1, 0x54, 0xC6, 0x03, 0x3B, 0x17, 0xD3, 0xB8, 0x78, 0x31, 0xB9, 0x7A, 0xAC, 0x84, 0xE5, 0x51, 0x84, 0xD8, 
			0x5A, 0x16, 0x40, 0xA1, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x56, 0x30, 0x54, 0x30, 0x52, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x4B, 0x30, 0x49, 0x80, 0x10, 0xD5, 0x6B, 0x08, 0x45, 0x51, 0x7C, 0x7E, 0xE7, 0x36, 0x07, 0x6E, 0xF7, 0x07, 0xE5, 0xC3, 0xE0, 0xA1, 0x23, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x16, 0x4D, 0x6F, 0x6E, 0x6F, 0x20, 0x55, 0x6E, 0x69, 0x74, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x41, 0x82, 0x10, 0xAF, 0x9A, 0xC0, 0xAC, 0xB0, 0x96, 0x43, 0x85, 0x43, 0xF5, 0x85, 0xD0, 0x24, 0xB4, 0x24, 0x9C, 0x30, 0x0D, 0x06, 
			0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x04, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0xA9, 0xE3, 0x44, 0x93, 0x54, 0x8E, 0x50, 0x93, 0xAC, 0xBE, 0x91, 0xA4, 0x99, 0x37, 0x3B, 0xAA, 0xEA, 0x22, 0x13, 0x14, 0xD3, 0x3A, 0xB8, 0xFB, 0xA3, 0x0F, 0x55, 0xB8, 0x83, 0xA1, 0x56, 0xA7, 0x51, 0xE3, 0x22, 0x14, 0x44, 0x41, 0x3E, 0xDC, 0x3B, 0xF3, 0x03, 0x89, 0xF4, 0xF2, 0x55, 0x70, 0x9A, 0x02, 0x48, 0x25, 0x0D, 0x66, 0x40, 0x21, 0x70, 0xF0, 0x0E, 0x79, 0xF5, 0x48, 0xD5, 0x19, 0xB8, 0xCC, 0xBD, 0x5D, 0xAC, 0xA5, 0x33, 0xD5, 0x02, 0x4F, 0x3F, 0x16, 0x42, 0xE6, 0x32, 0xFA, 0xEC, 0xAF, 0xE6, 0x12, 
			0xD0, 0x75, 0xE7, 0x4F, 0x0D, 0x50, 0xF4, 0xFC, 0x82, 0xD9, 0x82, 0xAF, 0x15, 0xDA, 0x74, 0xB5, 0xBF, 0xA9, 0x7A, 0xE2, 0xD0, 0x60, 0x7F, 0xEF, 0x48, 0x98, 0x03, 0x33, 0xDA, 0xBA, 0x12, 0x5E, 0x2F, 0xD7, 0x7F, 0xB6, 0xD3, 0xDF, 0x11, 0x93, 0xF8, 0x7B, 0xA9, 0xA8 };

		private void DefaultProperties (SignedCms sp, int version) 
		{
			// unaffected by constructors
			Assert.AreEqual (0, sp.Certificates.Count, "Certificates");
			Assert.AreEqual (0, sp.SignerInfos.Count, "SignerInfos");
			Assert.AreEqual (version, sp.Version, "Version");
		}

		private X509Certificate2 GetCertificate (bool includePrivateKey) 
		{
			return new X509Certificate2 (farscape_p12_pfx, "farscape");
		}

		private void AddChain (X509Certificate2Collection coll)
		{
			coll.Add (new X509Certificate2 (intca_cer));
			coll.Add (new X509Certificate2 (root_cer));
		}

		[Test]
		public void ConstructorEmpty () 
		{
			SignedCms sp = new SignedCms ();
			// default properties
			Assert.AreEqual (pkcs7Name, sp.ContentInfo.ContentType.FriendlyName, "ContentInfo.ContentType.FriendlyName");
			Assert.AreEqual (pkcs7Oid, sp.ContentInfo.ContentType.Value, "ContentInfo.ContentType.Value");
			Assert.AreEqual (0, sp.ContentInfo.Content.Length, "ContentInfo.Content");
			Assert.IsTrue (!sp.Detached, "Detached");
			DefaultProperties (sp, 0);
		}

		[Test]
		public void ConstructorContentInfo () 
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (ci);
			// default properties
			Assert.AreEqual (rsaName, sp.ContentInfo.ContentType.FriendlyName, "ContentInfo.ContentType.FriendlyName");
			Assert.AreEqual (rsaOid, sp.ContentInfo.ContentType.Value, "ContentInfo.ContentType.Value");
			Assert.AreEqual (2, sp.ContentInfo.Content.Length, "ContentInfo.Content");
			Assert.IsTrue (!sp.Detached, "Detached");
			DefaultProperties (sp, 0);
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void ConstructorContentInfoNull () 
		{
			SignedCms sp = new SignedCms (null);
		}

		[Test]
		public void ConstructorContentInfoDetachedTrue () 
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (ci, true);
			// default properties
			Assert.AreEqual (rsaName, sp.ContentInfo.ContentType.FriendlyName, "ContentInfo.ContentType.FriendlyName");
			Assert.AreEqual (rsaOid, sp.ContentInfo.ContentType.Value, "ContentInfo.ContentType.Value");
			Assert.AreEqual (2, sp.ContentInfo.Content.Length, "ContentInfo.Content");
			Assert.IsTrue (sp.Detached, "Detached");
			DefaultProperties (sp, 0);
		}

		[Test]
		public void ConstructorContentInfoDetachedFalse () 
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (ci, false);
			// default properties
			Assert.AreEqual (rsaName, sp.ContentInfo.ContentType.FriendlyName, "ContentInfo.ContentType.FriendlyName");
			Assert.AreEqual (rsaOid, sp.ContentInfo.ContentType.Value, "ContentInfo.ContentType.Value");
			Assert.AreEqual (2, sp.ContentInfo.Content.Length, "ContentInfo.Content");
			Assert.IsTrue (!sp.Detached, "Detached");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void ConstructorContentInfoNullDetached () 
		{
			SignedCms sp = new SignedCms (null, true);
		}

		private void DefaultSubjectIdentifierTypePropertiesCms (SignedCms sp, int version) 
		{
			Assert.AreEqual (pkcs7Name, sp.ContentInfo.ContentType.FriendlyName, "ContentInfo.ContentType.FriendlyName");
			Assert.AreEqual (pkcs7Oid, sp.ContentInfo.ContentType.Value, "ContentInfo.ContentType.Value");
			Assert.AreEqual (0, sp.ContentInfo.Content.Length, "ContentInfo.Content");
			Assert.IsTrue (!sp.Detached, "Detached");
			DefaultProperties (sp, version);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeIssuerAndSerialNumber ()
		{
			SignedCms sp = new SignedCms (SubjectIdentifierType.IssuerAndSerialNumber);
			// default properties
			DefaultSubjectIdentifierTypePropertiesCms (sp, 0);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeSubjectKeyIdentifier () 
		{
			SignedCms sp = new SignedCms (SubjectIdentifierType.SubjectKeyIdentifier);
			// default properties
			DefaultSubjectIdentifierTypePropertiesCms (sp, 0);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeUnknown () 
		{
			SignedCms sp = new SignedCms (SubjectIdentifierType.Unknown);
			// default properties
			DefaultSubjectIdentifierTypePropertiesCms (sp, 0);
		}

		private void DefaultSubjectIdentifierTypeProperties (SignedCms sp, int version) 
		{
			Assert.AreEqual (rsaName, sp.ContentInfo.ContentType.FriendlyName, "ContentInfo.ContentType.FriendlyName");
			Assert.AreEqual (rsaOid, sp.ContentInfo.ContentType.Value, "ContentInfo.ContentType.Value");
			Assert.AreEqual (2, sp.ContentInfo.Content.Length, "ContentInfo.Content");
			DefaultProperties (sp, version);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeIssuerAndSerialNumberContentInfo () 
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (SubjectIdentifierType.IssuerAndSerialNumber, ci);
			// default properties
			Assert.IsTrue (!sp.Detached, "Detached");
			DefaultSubjectIdentifierTypeProperties (sp, 0);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeSubjectKeyIdentifierContentInfo ()
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (SubjectIdentifierType.SubjectKeyIdentifier, ci);
			// default properties
			Assert.IsTrue (!sp.Detached, "Detached");
			DefaultSubjectIdentifierTypeProperties (sp, 0);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeUnknownContentInfo ()
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (SubjectIdentifierType.Unknown, ci);
			// default properties
			Assert.IsTrue (!sp.Detached, "Detached");
			DefaultSubjectIdentifierTypeProperties (sp, 0);
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void ConstructorSubjectIdentifierTypeContentInfoNull () 
		{
			SignedCms sp = new SignedCms (SubjectIdentifierType.Unknown, null);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeIssuerAndSerialNumberContentInfoDetached () 
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (SubjectIdentifierType.IssuerAndSerialNumber, ci, true);
			// default properties
			Assert.IsTrue (sp.Detached, "Detached");
			DefaultSubjectIdentifierTypeProperties (sp, 0);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeSubjectKeyIdentifierContentInfoDetached () 
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (SubjectIdentifierType.SubjectKeyIdentifier, ci, true);
			// default properties
			Assert.IsTrue (sp.Detached, "Detached");
			DefaultSubjectIdentifierTypeProperties (sp, 0);
		}

		[Test]
		public void ConstructorSubjectIdentifierTypeUnknownContentInfoDetached () 
		{
			Oid oid = new Oid (rsaOid);
			ContentInfo ci = new ContentInfo (oid, asnNull);
			SignedCms sp = new SignedCms (SubjectIdentifierType.Unknown, ci, true);
			// default properties
			Assert.IsTrue (sp.Detached, "Detached");
			DefaultSubjectIdentifierTypeProperties (sp, 0);
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void ConstructorSubjectIdentifierTypeContentInfoNullDetached () 
		{
			SignedCms sp = new SignedCms (SubjectIdentifierType.Unknown, null, true);
		}

		private void RoundTrip (byte[] encoded) 
		{
			SignedCms sp = new SignedCms ();
			sp.Decode (encoded);
			sp.CheckSignature (true);
		}

		[Test]
		[Ignore ("now broken everywhere")]
		public void ComputeSignatureCmsSignerIssuerAndSerialNumber () 
		{
			ContentInfo ci = new ContentInfo (asnNull);
			SignedCms sp = new SignedCms (ci);

			CmsSigner signer = new CmsSigner (SubjectIdentifierType.IssuerAndSerialNumber, GetCertificate (true));
			signer.Certificates.Add (new X509Certificate2 (intca_cer));
			signer.Certificates.Add (new X509Certificate2 (root_cer));
			sp.ComputeSignature (signer);

			byte[] encoded = sp.Encode ();
			string s = BitConverter.ToString (encoded);
#if DEBUG
			FileStream fs = File.OpenWrite ("ComputeSignatureCmsSignerIssuerAndSerialNumber.der");
			fs.Write (encoded, 0, encoded.Length);
			fs.Close ();
#endif
			RoundTrip (encoded);
		}

		[Test]
		[Ignore ("now broken everywhere")]
		public void ComputeSignatureCmsSignerSubjectKeyIdentifier () 
		{
			ContentInfo ci = new ContentInfo (asnNull);
			SignedCms sp = new SignedCms (ci);

			CmsSigner signer = new CmsSigner (SubjectIdentifierType.SubjectKeyIdentifier, GetCertificate (true));
			signer.Certificates.Add (new X509Certificate2 (intca_cer));
			signer.Certificates.Add (new X509Certificate2 (root_cer));
			sp.ComputeSignature (signer);

			byte[] encoded = sp.Encode ();
			string s = BitConverter.ToString (encoded);
#if DEBUG
			FileStream fs = File.OpenWrite ("ComputeSignaturePkcs7SignerSubjectKeyIdentifier.der");
			fs.Write (encoded, 0, encoded.Length);
			fs.Close ();
#endif
			RoundTrip (encoded);
		}

		[Test]
		[Ignore ("now broken everywhere")]
		public void ComputeSignatureCmsSignerUnknown () 
		{
			ContentInfo ci = new ContentInfo (asnNull);
			SignedCms sp = new SignedCms (ci);

			CmsSigner signer = new CmsSigner (SubjectIdentifierType.Unknown, GetCertificate (true));
			signer.Certificates.Add (new X509Certificate2 (intca_cer));
			signer.Certificates.Add (new X509Certificate2 (root_cer));
			sp.ComputeSignature (signer);

			byte[] encoded = sp.Encode ();
			string s = BitConverter.ToString (encoded);
#if DEBUG
			FileStream fs = File.OpenWrite ("ComputeSignaturePkcs7SignerUnknown.der");
			fs.Write (encoded, 0, encoded.Length);
			fs.Close ();
#endif
			RoundTrip (encoded);
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void ComputeEmptySignatureCmsSigner () 
		{
			CmsSigner signer = new CmsSigner ();
			SignedCms sp = new SignedCms ();
			sp.ComputeSignature (signer);
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void ComputeEmptySignature ()
		{
			SignedCms sp = new SignedCms ();
			sp.ComputeSignature ();
		}

		private void CheckSignatureProperties (SignedCms sp, int version) 
		{
			Assert.AreEqual (1, sp.Certificates.Count, "Certificates");
			Assert.AreEqual (2, sp.ContentInfo.Content.Length, "ContentInfo.Content");
			Assert.IsTrue (!sp.Detached, "Detached");
			Assert.AreEqual (1, sp.SignerInfos.Count, "SignerInfos");
			Assert.AreEqual (version, sp.Version, "Version");
		}

		[Test]
		public void CheckSignatureCmsSignerIssuerAndSerialNumber () 
		{
			byte[] signature = { 0x30, 0x82, 0x03, 0x4C, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0, 0x82, 0x03, 0x3D, 0x30, 0x82, 0x03, 0x39, 0x02, 0x01, 0x01, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x11, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x04, 0x04, 0x02, 0x05, 0x00, 0xA0, 0x82, 0x02, 0x2E, 0x30, 0x82, 0x02, 0x2A, 0x30, 0x82, 0x01, 0x97, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65,
				0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x33, 0x30, 0x38, 0x31, 0x33, 0x30, 0x30, 0x34, 0x33, 0x34, 0x37, 0x5A, 0x17, 0x0D, 0x33, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x13, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x08, 0x46, 0x41, 0x52, 0x53, 0x43, 0x41, 0x50, 0x45, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xD2, 0xCB, 0x47, 0x21, 0xF5, 0x62, 0xDD, 0x35, 0xBF, 0x1D, 0xEC, 0x9A, 0x4C, 0x07, 0x2C, 0x01, 0xF0, 0x28, 0xC2, 0x82, 0x17, 0x8E, 0x58, 0x32, 
				0xD5, 0x4C, 0xAC, 0x86, 0xB4, 0xC9, 0xEB, 0x21, 0x26, 0xF3, 0x22, 0x30, 0xC5, 0x7A, 0xA3, 0x5A, 0xDD, 0x53, 0xAB, 0x1C, 0x06, 0x3E, 0xB2, 0x13, 0xC4, 0x05, 0x1D, 0x95, 0x8B, 0x0A, 0x71, 0x71, 0x11, 0xA7, 0x47, 0x26, 0x61, 0xF1, 0x76, 0xBE, 0x35, 0x72, 0x32, 0xC5, 0xCB, 0x47, 0xA4, 0x22, 0x41, 0x1E, 0xAD, 0x29, 0x11, 0x0D, 0x39, 0x22, 0x0C, 0x79, 0x90, 0xC6, 0x52, 0xA1, 0x10, 0xF6, 0x55, 0x09, 0x4E, 0x51, 0x26, 0x47, 0x0E, 0x94, 0xE6, 0x81, 0xF5, 0x18, 0x6B, 0x99, 0xF0, 0x76, 0xF3, 0xB2, 0x4C, 0x91, 0xE9, 0xBA, 0x3B, 0x3F, 0x6E, 0x63, 0xDA, 0x12, 0xD1, 0x0B, 0x73, 0x0E, 0x12, 0xC7, 0x70, 0x77, 0x22, 0x03, 0x9D, 0x5D, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x72, 0x30, 0x70, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 
				0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x59, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x52, 0x30, 0x50, 0x80, 0x10, 0xAE, 0xD7, 0x80, 0x88, 0xA6, 0x3D, 0xBA, 0x50, 0xA1, 0x7E, 0x57, 0xE5, 0x40, 0xC9, 0x6F, 0xC5, 0xA1, 0x2A, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x82, 0x10, 0x9D, 0xAE, 0xA3, 0x39, 0x47, 0x0E, 0xD4, 0xA2, 0x49, 0x78, 0xEA, 0x6C, 0xBA, 0x0D, 0xDE, 0x9C, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x32, 0x8A, 0x7E, 0xAD, 0xE7, 0x67, 0x9E, 0x5C, 0x4C, 0xD8, 0x33, 0x59, 0x68, 0xCF, 
				0x94, 0xC0, 0x36, 0x47, 0x7A, 0xA7, 0x85, 0xC2, 0xDD, 0xD8, 0xDA, 0x11, 0x3C, 0x66, 0xC1, 0x83, 0xE3, 0xAB, 0x33, 0x06, 0x7C, 0xE3, 0x6A, 0x15, 0x72, 0xB8, 0x83, 0x3D, 0x0B, 0xAB, 0x3C, 0xEE, 0x75, 0x13, 0xBD, 0x5C, 0x96, 0x25, 0x56, 0x36, 0x05, 0xFA, 0xAE, 0xD4, 0xF4, 0xCF, 0x52, 0xEC, 0x11, 0xB5, 0xEA, 0x9F, 0x20, 0xA3, 0xC8, 0x34, 0x72, 0x59, 0x09, 0x51, 0xE7, 0x36, 0x87, 0x86, 0x86, 0x98, 0xB5, 0x30, 0x7B, 0xFB, 0x3D, 0xCC, 0x5E, 0xE8, 0xC9, 0x49, 0xE0, 0xC6, 0xEA, 0x02, 0x76, 0x01, 0xE0, 0xBB, 0x8A, 0x70, 0xEB, 0x07, 0x86, 0xE8, 0x04, 0xE7, 0x48, 0xE4, 0x6C, 0x90, 0xE6, 0x16, 0x42, 0xB4, 0xBB, 0xC0, 0xC4, 0x82, 0x5F, 0xF8, 0xFB, 0x7E, 0xB2, 0x9E, 0xC2, 0x78, 0x26, 0x86, 0x31, 0x81, 0xE1, 0x30, 0x81, 0xDE, 0x02, 0x01, 0x01, 0x30, 0x3C, 0x30, 0x28, 
				0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x81, 0x80, 0x45, 0x88, 0x80, 0x58, 0xC7, 0x4F, 0xE4, 0xD8, 0x88, 0xB0, 0xC0, 0x08, 0x70, 0x84, 0xCC, 0x8E, 0xA7, 0xF1, 0xA4, 0x07, 0x41, 0x14, 0x3E, 0xF5, 0xEA, 0x6E, 0x05, 0x75, 0xB8, 0x58, 0xAA, 0x5C, 0x0E, 0xFD, 0x7A, 0x07, 0x09, 0xE1, 0x80, 0x94, 
				0xBD, 0xAA, 0x45, 0xBB, 0x55, 0x9C, 0xC2, 0xD9, 0x72, 0x14, 0x4B, 0xA4, 0x64, 0xFB, 0x38, 0x9F, 0xD3, 0x22, 0xED, 0xB3, 0x0B, 0xF7, 0xAE, 0x4D, 0xE6, 0x65, 0x4D, 0x2A, 0x31, 0x18, 0xB5, 0xB4, 0x2D, 0x9E, 0x4E, 0xD7, 0xC0, 0x44, 0x5F, 0xAC, 0x43, 0xDC, 0x4F, 0x3D, 0x6D, 0x2C, 0x8C, 0xA1, 0xFE, 0x08, 0x38, 0xB7, 0xC4, 0xC4, 0x08, 0xDB, 0xF8, 0xF0, 0xC1, 0x55, 0x54, 0x49, 0x9D, 0xA4, 0x7F, 0x76, 0xDE, 0xF4, 0x29, 0x1C, 0x0B, 0x95, 0x10, 0x90, 0xB5, 0x0A, 0x9A, 0xEC, 0xCA, 0x89, 0x9A, 0x85, 0x92, 0x76, 0x78, 0x6F, 0x97, 0x67 };
			SignedCms sp = new SignedCms ();
			sp.Decode (signature);
			sp.CheckSignature (true);
			CheckSignatureProperties (sp, 1);
		}

		[Test]
		public void CheckSignatureCmsSignerSubjectKeyIdentifier () 
		{
			byte[] signature = { 0x30, 0x82, 0x03, 0x24, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0, 0x82, 0x03, 0x15, 0x30, 0x82, 0x03, 0x11, 0x02, 0x01, 0x03, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x11, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x04, 0x04, 0x02, 0x05, 0x00, 0xA0, 0x82, 0x02, 0x2E, 0x30, 0x82, 0x02, 0x2A, 0x30, 0x82, 0x01, 0x97, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 
				0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x33, 0x30, 0x38, 0x31, 0x33, 0x30, 0x30, 0x34, 0x33, 0x34, 0x37, 0x5A, 0x17, 0x0D, 0x33, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x13, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x08, 0x46, 0x41, 0x52, 0x53, 0x43, 0x41, 0x50, 0x45, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xD2, 0xCB, 0x47, 0x21, 0xF5, 0x62, 0xDD, 0x35, 0xBF, 0x1D, 0xEC, 0x9A, 0x4C, 0x07, 0x2C, 0x01, 0xF0, 0x28, 0xC2, 0x82, 0x17, 0x8E, 0x58, 0x32, 
				0xD5, 0x4C, 0xAC, 0x86, 0xB4, 0xC9, 0xEB, 0x21, 0x26, 0xF3, 0x22, 0x30, 0xC5, 0x7A, 0xA3, 0x5A, 0xDD, 0x53, 0xAB, 0x1C, 0x06, 0x3E, 0xB2, 0x13, 0xC4, 0x05, 0x1D, 0x95, 0x8B, 0x0A, 0x71, 0x71, 0x11, 0xA7, 0x47, 0x26, 0x61, 0xF1, 0x76, 0xBE, 0x35, 0x72, 0x32, 0xC5, 0xCB, 0x47, 0xA4, 0x22, 0x41, 0x1E, 0xAD, 0x29, 0x11, 0x0D, 0x39, 0x22, 0x0C, 0x79, 0x90, 0xC6, 0x52, 0xA1, 0x10, 0xF6, 0x55, 0x09, 0x4E, 0x51, 0x26, 0x47, 0x0E, 0x94, 0xE6, 0x81, 0xF5, 0x18, 0x6B, 0x99, 0xF0, 0x76, 0xF3, 0xB2, 0x4C, 0x91, 0xE9, 0xBA, 0x3B, 0x3F, 0x6E, 0x63, 0xDA, 0x12, 0xD1, 0x0B, 0x73, 0x0E, 0x12, 0xC7, 0x70, 0x77, 0x22, 0x03, 0x9D, 0x5D, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x72, 0x30, 0x70, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 
				0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x59, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x52, 0x30, 0x50, 0x80, 0x10, 0xAE, 0xD7, 0x80, 0x88, 0xA6, 0x3D, 0xBA, 0x50, 0xA1, 0x7E, 0x57, 0xE5, 0x40, 0xC9, 0x6F, 0xC5, 0xA1, 0x2A, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x82, 0x10, 0x9D, 0xAE, 0xA3, 0x39, 0x47, 0x0E, 0xD4, 0xA2, 0x49, 0x78, 0xEA, 0x6C, 0xBA, 0x0D, 0xDE, 0x9C, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x32, 0x8A, 0x7E, 0xAD, 0xE7, 0x67, 0x9E, 0x5C, 0x4C, 0xD8, 0x33, 0x59, 0x68, 0xCF, 
				0x94, 0xC0, 0x36, 0x47, 0x7A, 0xA7, 0x85, 0xC2, 0xDD, 0xD8, 0xDA, 0x11, 0x3C, 0x66, 0xC1, 0x83, 0xE3, 0xAB, 0x33, 0x06, 0x7C, 0xE3, 0x6A, 0x15, 0x72, 0xB8, 0x83, 0x3D, 0x0B, 0xAB, 0x3C, 0xEE, 0x75, 0x13, 0xBD, 0x5C, 0x96, 0x25, 0x56, 0x36, 0x05, 0xFA, 0xAE, 0xD4, 0xF4, 0xCF, 0x52, 0xEC, 0x11, 0xB5, 0xEA, 0x9F, 0x20, 0xA3, 0xC8, 0x34, 0x72, 0x59, 0x09, 0x51, 0xE7, 0x36, 0x87, 0x86, 0x86, 0x98, 0xB5, 0x30, 0x7B, 0xFB, 0x3D, 0xCC, 0x5E, 0xE8, 0xC9, 0x49, 0xE0, 0xC6, 0xEA, 0x02, 0x76, 0x01, 0xE0, 0xBB, 0x8A, 0x70, 0xEB, 0x07, 0x86, 0xE8, 0x04, 0xE7, 0x48, 0xE4, 0x6C, 0x90, 0xE6, 0x16, 0x42, 0xB4, 0xBB, 0xC0, 0xC4, 0x82, 0x5F, 0xF8, 0xFB, 0x7E, 0xB2, 0x9E, 0xC2, 0x78, 0x26, 0x86, 0x31, 0x81, 0xB9, 0x30, 0x81, 0xB6, 0x02, 0x01, 0x03, 0x80, 0x14, 0x02, 0xE1, 
				0xA7, 0x32, 0x54, 0xAE, 0xFD, 0xC0, 0xA4, 0x32, 0x36, 0xF6, 0xFE, 0x23, 0x6A, 0x03, 0x72, 0x28, 0xB1, 0xF7, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x81, 0x80, 0x45, 0x88, 0x80, 0x58, 0xC7, 0x4F, 0xE4, 0xD8, 0x88, 0xB0, 0xC0, 0x08, 0x70, 0x84, 0xCC, 0x8E, 0xA7, 0xF1, 0xA4, 0x07, 0x41, 0x14, 0x3E, 0xF5, 0xEA, 0x6E, 0x05, 0x75, 0xB8, 0x58, 0xAA, 0x5C, 0x0E, 0xFD, 0x7A, 0x07, 0x09, 0xE1, 0x80, 0x94, 0xBD, 0xAA, 0x45, 0xBB, 0x55, 0x9C, 0xC2, 0xD9, 0x72, 0x14, 0x4B, 0xA4, 0x64, 0xFB, 0x38, 0x9F, 0xD3, 0x22, 0xED, 0xB3, 0x0B, 0xF7, 0xAE, 0x4D, 0xE6, 0x65, 0x4D, 0x2A, 0x31, 0x18, 0xB5, 0xB4, 0x2D, 0x9E, 0x4E, 0xD7, 0xC0, 0x44, 0x5F, 0xAC, 
				0x43, 0xDC, 0x4F, 0x3D, 0x6D, 0x2C, 0x8C, 0xA1, 0xFE, 0x08, 0x38, 0xB7, 0xC4, 0xC4, 0x08, 0xDB, 0xF8, 0xF0, 0xC1, 0x55, 0x54, 0x49, 0x9D, 0xA4, 0x7F, 0x76, 0xDE, 0xF4, 0x29, 0x1C, 0x0B, 0x95, 0x10, 0x90, 0xB5, 0x0A, 0x9A, 0xEC, 0xCA, 0x89, 0x9A, 0x85, 0x92, 0x76, 0x78, 0x6F, 0x97, 0x67 };
			SignedCms sp = new SignedCms ();
			sp.Decode (signature);
			sp.CheckSignature (true);
			CheckSignatureProperties (sp, 3);
		}

		[Test]
		public void CheckSignatureCmsSignerUnknown () 
		{
			byte[] signature = { 0x30, 0x82, 0x03, 0x4C, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0, 0x82, 0x03, 0x3D, 0x30, 0x82, 0x03, 0x39, 0x02, 0x01, 0x01, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x11, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x04, 0x04, 0x02, 0x05, 0x00, 0xA0, 0x82, 0x02, 0x2E, 0x30, 0x82, 0x02, 0x2A, 0x30, 0x82, 0x01, 0x97, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 
				0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x33, 0x30, 0x38, 0x31, 0x33, 0x30, 0x30, 0x34, 0x33, 0x34, 0x37, 0x5A, 0x17, 0x0D, 0x33, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x13, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x08, 0x46, 0x41, 0x52, 0x53, 0x43, 0x41, 0x50, 0x45, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xD2, 0xCB, 0x47, 0x21, 0xF5, 0x62, 0xDD, 0x35, 0xBF, 0x1D, 0xEC, 0x9A, 0x4C, 0x07, 0x2C, 0x01, 0xF0, 0x28, 0xC2, 0x82, 0x17, 0x8E, 0x58, 0x32, 
				0xD5, 0x4C, 0xAC, 0x86, 0xB4, 0xC9, 0xEB, 0x21, 0x26, 0xF3, 0x22, 0x30, 0xC5, 0x7A, 0xA3, 0x5A, 0xDD, 0x53, 0xAB, 0x1C, 0x06, 0x3E, 0xB2, 0x13, 0xC4, 0x05, 0x1D, 0x95, 0x8B, 0x0A, 0x71, 0x71, 0x11, 0xA7, 0x47, 0x26, 0x61, 0xF1, 0x76, 0xBE, 0x35, 0x72, 0x32, 0xC5, 0xCB, 0x47, 0xA4, 0x22, 0x41, 0x1E, 0xAD, 0x29, 0x11, 0x0D, 0x39, 0x22, 0x0C, 0x79, 0x90, 0xC6, 0x52, 0xA1, 0x10, 0xF6, 0x55, 0x09, 0x4E, 0x51, 0x26, 0x47, 0x0E, 0x94, 0xE6, 0x81, 0xF5, 0x18, 0x6B, 0x99, 0xF0, 0x76, 0xF3, 0xB2, 0x4C, 0x91, 0xE9, 0xBA, 0x3B, 0x3F, 0x6E, 0x63, 0xDA, 0x12, 0xD1, 0x0B, 0x73, 0x0E, 0x12, 0xC7, 0x70, 0x77, 0x22, 0x03, 0x9D, 0x5D, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x72, 0x30, 0x70, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 
				0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x59, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x52, 0x30, 0x50, 0x80, 0x10, 0xAE, 0xD7, 0x80, 0x88, 0xA6, 0x3D, 0xBA, 0x50, 0xA1, 0x7E, 0x57, 0xE5, 0x40, 0xC9, 0x6F, 0xC5, 0xA1, 0x2A, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x82, 0x10, 0x9D, 0xAE, 0xA3, 0x39, 0x47, 0x0E, 0xD4, 0xA2, 0x49, 0x78, 0xEA, 0x6C, 0xBA, 0x0D, 0xDE, 0x9C, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x32, 0x8A, 0x7E, 0xAD, 0xE7, 0x67, 0x9E, 0x5C, 0x4C, 0xD8, 0x33, 0x59, 0x68, 0xCF, 
				0x94, 0xC0, 0x36, 0x47, 0x7A, 0xA7, 0x85, 0xC2, 0xDD, 0xD8, 0xDA, 0x11, 0x3C, 0x66, 0xC1, 0x83, 0xE3, 0xAB, 0x33, 0x06, 0x7C, 0xE3, 0x6A, 0x15, 0x72, 0xB8, 0x83, 0x3D, 0x0B, 0xAB, 0x3C, 0xEE, 0x75, 0x13, 0xBD, 0x5C, 0x96, 0x25, 0x56, 0x36, 0x05, 0xFA, 0xAE, 0xD4, 0xF4, 0xCF, 0x52, 0xEC, 0x11, 0xB5, 0xEA, 0x9F, 0x20, 0xA3, 0xC8, 0x34, 0x72, 0x59, 0x09, 0x51, 0xE7, 0x36, 0x87, 0x86, 0x86, 0x98, 0xB5, 0x30, 0x7B, 0xFB, 0x3D, 0xCC, 0x5E, 0xE8, 0xC9, 0x49, 0xE0, 0xC6, 0xEA, 0x02, 0x76, 0x01, 0xE0, 0xBB, 0x8A, 0x70, 0xEB, 0x07, 0x86, 0xE8, 0x04, 0xE7, 0x48, 0xE4, 0x6C, 0x90, 0xE6, 0x16, 0x42, 0xB4, 0xBB, 0xC0, 0xC4, 0x82, 0x5F, 0xF8, 0xFB, 0x7E, 0xB2, 0x9E, 0xC2, 0x78, 0x26, 0x86, 0x31, 0x81, 0xE1, 0x30, 0x81, 0xDE, 0x02, 0x01, 0x01, 0x30, 0x3C, 0x30, 0x28, 
				0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x81, 0x80, 0x45, 0x88, 0x80, 0x58, 0xC7, 0x4F, 0xE4, 0xD8, 0x88, 0xB0, 0xC0, 0x08, 0x70, 0x84, 0xCC, 0x8E, 0xA7, 0xF1, 0xA4, 0x07, 0x41, 0x14, 0x3E, 0xF5, 0xEA, 0x6E, 0x05, 0x75, 0xB8, 0x58, 0xAA, 0x5C, 0x0E, 0xFD, 0x7A, 0x07, 0x09, 0xE1, 0x80, 0x94, 
				0xBD, 0xAA, 0x45, 0xBB, 0x55, 0x9C, 0xC2, 0xD9, 0x72, 0x14, 0x4B, 0xA4, 0x64, 0xFB, 0x38, 0x9F, 0xD3, 0x22, 0xED, 0xB3, 0x0B, 0xF7, 0xAE, 0x4D, 0xE6, 0x65, 0x4D, 0x2A, 0x31, 0x18, 0xB5, 0xB4, 0x2D, 0x9E, 0x4E, 0xD7, 0xC0, 0x44, 0x5F, 0xAC, 0x43, 0xDC, 0x4F, 0x3D, 0x6D, 0x2C, 0x8C, 0xA1, 0xFE, 0x08, 0x38, 0xB7, 0xC4, 0xC4, 0x08, 0xDB, 0xF8, 0xF0, 0xC1, 0x55, 0x54, 0x49, 0x9D, 0xA4, 0x7F, 0x76, 0xDE, 0xF4, 0x29, 0x1C, 0x0B, 0x95, 0x10, 0x90, 0xB5, 0x0A, 0x9A, 0xEC, 0xCA, 0x89, 0x9A, 0x85, 0x92, 0x76, 0x78, 0x6F, 0x97, 0x67 };
			SignedCms sp = new SignedCms ();
			sp.Decode (signature);
			sp.CheckSignature (true);
			CheckSignatureProperties (sp, 1);
		}
	}
}

#endif
