/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999  Espen Sand, espensa@online.no
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <string.h>

#include <klocale.h>

#include "conversion.h"


CConversion::CConversion( void )
{
  setMode( cnvDefault );
}


CConversion::EMode CConversion::mode( void )
{
  return( (EMode)mState.mode );
}

const SEncodeState &CConversion::state( void )
{
  return( mState );
}

QString CConversion::names( unsigned int index )
{
  static const QString strings[4] = 
  {
    i18n("Default encoding", "Default"),
    i18n("EBCDIC"),
    i18n("US-ASCII (7 bit)"),
    i18n("Unknown"),
  };
 
  return( strings[ index > cnvUsAscii ? 3 : index ] );
}


const unsigned char *CConversion::tables( EMode cnvMode )
{
  /*
  static unsigned char ebcdicToLatin1[256] = 
  {
    0x00, 0x01, 0x02, 0x03, 0x9C, 0x09, 0x86, 0x7F, 
    0x97, 0x8D, 0x8E, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
    0x10, 0x11, 0x12, 0x13, 0x9D, 0x85, 0x08, 0x87, 
    0x18, 0x19, 0x92, 0x8F, 0x1C, 0x1D, 0x1E, 0x1F,
    0x80, 0x81, 0x82, 0x83, 0x84, 0x0A, 0x17, 0x1B, 
    0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x05, 0x06, 0x07,
    0x90, 0x91, 0x16, 0x93, 0x94, 0x95, 0x96, 0x04, 
    0x98, 0x99, 0x9A, 0x9B, 0x14, 0x15, 0x9E, 0x1A,
    0x20, 0xA0, 0xE2, 0xE4, 0xE0, 0xE1, 0xE3, 0xE5, 
    0xE7, 0xF1, 0xA2, 0x2E, 0x3C, 0x28, 0x2B, 0x7C,
    0x26, 0xE9, 0xEA, 0xEB, 0xE8, 0xED, 0xEE, 0xEF, 
    0xEC, 0xDF, 0x21, 0x24, 0x2A, 0x29, 0x3B, 0x5E,
    0x2D, 0x2F, 0xC2, 0xC4, 0xC0, 0xC1, 0xC3, 0xC5, 
    0xC7, 0xD1, 0xA6, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,
    0xF8, 0xC9, 0xCA, 0xCB, 0xC8, 0xCD, 0xCE, 0xCF, 
    0xCC, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,
    0xD8, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
    0x68, 0x69, 0xAB, 0xBB, 0xF0, 0xFD, 0xFE, 0xB1,
    0xB0, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 
    0x71, 0x72, 0xAA, 0xBA, 0xE6, 0xB8, 0xC6, 0xA4,
    0xB5, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 
    0x79, 0x7A, 0xA1, 0xBF, 0xD0, 0x5B, 0xDE, 0xAE,
    0xAC, 0xA3, 0xA5, 0xB7, 0xA9, 0xA7, 0xB6, 0xBC, 
    0xBD, 0xBE, 0xDD, 0xA8, 0xAF, 0x5D, 0xB4, 0xD7,
    0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
    0x48, 0x49, 0xAD, 0xF4, 0xF6, 0xF2, 0xF3, 0xF5,
    0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 
    0x51, 0x52, 0xB9, 0xFB, 0xFC, 0xF9, 0xFA, 0xFF,
    0x5C, 0xF7, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 
    0x59, 0x5A, 0xB2, 0xD4, 0xD6, 0xD2, 0xD3, 0xD5,
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
    0x38, 0x39, 0xB3, 0xDB, 0xDC, 0xD9, 0xDA, 0x9F
  };
  */

  static unsigned char ebcdicToLatin1[256] = 
  {
    0x00, 0x01, 0x02, 0x03, 0x9C, 0x09, 0x86, 0x7F, 
    0x97, 0x8D, 0x8E, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 
    0x10, 0x11, 0x12, 0x13, 0x9D, 0x85, 0x08, 0x87, 
    0x18, 0x19, 0x92, 0x8F, 0x1C, 0x1D, 0x1E, 0x1F, 
    0x80, 0x81, 0x82, 0x83, 0x84, 0x0A, 0x17, 0x1B, 
    0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x05, 0x06, 0x07, 
    0x90, 0x91, 0x16, 0x93, 0x94, 0x95, 0x96, 0x04, 
    0x98, 0x99, 0x9A, 0x9B, 0x14, 0x15, 0x9E, 0x1A, 
    0x20, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 
    0xA7, 0xA8, 0x5B, 0x2E, 0x3C, 0x28, 0x2B, 0x21, 
    0x26, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 
    0xB0, 0xB1, 0x5D, 0x24, 0x2A, 0x29, 0x3B, 0x5E, 
    0x2D, 0x2F, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 
    0xB8, 0xB9, 0x7C, 0x2C, 0x25, 0x5F, 0x3E, 0x3F, 
    0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1, 
    0xC2, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22, 
    0xC3, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
    0x68, 0x69, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 
    0xCA, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 
    0x71, 0x72, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 
    0xD1, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 
    0x79, 0x7A, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 
    0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 
    0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 
    0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
    0x48, 0x49, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 
    0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 
    0x51, 0x52, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2, 0xF3, 
    0x5C, 0x9F, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 
    0x59, 0x5A, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
    0x38, 0x39, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF, 
  }; 

  static unsigned char buf[256]; 


  if( cnvMode == cnvEbcdic )
  {
    return( ebcdicToLatin1 );
  }
  else if( cnvMode == cnvUsAscii )
  {
    memset( buf, 0, 256 );
    for( unsigned int i=0; i<127; i++ ) { buf[i] = i; }
    return( buf );
  }
  else if( cnvMode == cnvDefault )
  {
    for( unsigned int i=0; i<256; i++ ) { buf[i] = i; }
    return( buf );
  }
  else
  {
    return( 0 );
  }
}


void CConversion::setMode( EMode cnvMode )
{
  const unsigned char *table = tables( cnvMode );
  if( table == 0 )
  {
    cnvMode = cnvDefault;
    table = tables( cnvMode );
  }  

  mState.mode = cnvMode;
  mState.name = names( mState.mode );
  memcpy( mData, table, 256 );
}


bool CConversion::lossless( EMode cnvMode )
{
  const unsigned char *table = tables( cnvMode );
  if( table == 0 )
  {
    return( false );
  }

  unsigned char flag[256];
  memset( flag, 0, sizeof( flag ) );

  for( uint i=0; i<256; i++ )
  {
    int index = table[i];
    if( flag[index] != 0 )
    {
      return( false );
    }
    flag[index] += 1;
  }
  
  return( true );
}



#if 0
void CConversion::setMode( EMode cnvMode )
{
  /*
  static unsigned char ebcdicToLatin1[256] = 
  {
    0x00, 0x01, 0x02, 0x03, 0x9C, 0x09, 0x86, 0x7F, 
    0x97, 0x8D, 0x8E, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
    0x10, 0x11, 0x12, 0x13, 0x9D, 0x85, 0x08, 0x87, 
    0x18, 0x19, 0x92, 0x8F, 0x1C, 0x1D, 0x1E, 0x1F,
    0x80, 0x81, 0x82, 0x83, 0x84, 0x0A, 0x17, 0x1B, 
    0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x05, 0x06, 0x07,
    0x90, 0x91, 0x16, 0x93, 0x94, 0x95, 0x96, 0x04, 
    0x98, 0x99, 0x9A, 0x9B, 0x14, 0x15, 0x9E, 0x1A,
    0x20, 0xA0, 0xE2, 0xE4, 0xE0, 0xE1, 0xE3, 0xE5, 
    0xE7, 0xF1, 0xA2, 0x2E, 0x3C, 0x28, 0x2B, 0x7C,
    0x26, 0xE9, 0xEA, 0xEB, 0xE8, 0xED, 0xEE, 0xEF, 
    0xEC, 0xDF, 0x21, 0x24, 0x2A, 0x29, 0x3B, 0x5E,
    0x2D, 0x2F, 0xC2, 0xC4, 0xC0, 0xC1, 0xC3, 0xC5, 
    0xC7, 0xD1, 0xA6, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,
    0xF8, 0xC9, 0xCA, 0xCB, 0xC8, 0xCD, 0xCE, 0xCF, 
    0xCC, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,
    0xD8, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
    0x68, 0x69, 0xAB, 0xBB, 0xF0, 0xFD, 0xFE, 0xB1,
    0xB0, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 
    0x71, 0x72, 0xAA, 0xBA, 0xE6, 0xB8, 0xC6, 0xA4,
    0xB5, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 
    0x79, 0x7A, 0xA1, 0xBF, 0xD0, 0x5B, 0xDE, 0xAE,
    0xAC, 0xA3, 0xA5, 0xB7, 0xA9, 0xA7, 0xB6, 0xBC, 
    0xBD, 0xBE, 0xDD, 0xA8, 0xAF, 0x5D, 0xB4, 0xD7,
    0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
    0x48, 0x49, 0xAD, 0xF4, 0xF6, 0xF2, 0xF3, 0xF5,
    0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 
    0x51, 0x52, 0xB9, 0xFB, 0xFC, 0xF9, 0xFA, 0xFF,
    0x5C, 0xF7, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 
    0x59, 0x5A, 0xB2, 0xD4, 0xD6, 0xD2, 0xD3, 0xD5,
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
    0x38, 0x39, 0xB3, 0xDB, 0xDC, 0xD9, 0xDA, 0x9F
  };
  */

  static unsigned char ebcdicToLatin1[256] = 
  {
    0x00, 0x01, 0x02, 0x03, 0x9C, 0x09, 0x86, 0x7F, 
    0x97, 0x8D, 0x8E, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 
    0x10, 0x11, 0x12, 0x13, 0x9D, 0x85, 0x08, 0x87, 
    0x18, 0x19, 0x92, 0x8F, 0x1C, 0x1D, 0x1E, 0x1F, 
    0x80, 0x81, 0x82, 0x83, 0x84, 0x0A, 0x17, 0x1B, 
    0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x05, 0x06, 0x07, 
    0x90, 0x91, 0x16, 0x93, 0x94, 0x95, 0x96, 0x04, 
    0x98, 0x99, 0x9A, 0x9B, 0x14, 0x15, 0x9E, 0x1A, 
    0x20, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 
    0xA7, 0xA8, 0x5B, 0x2E, 0x3C, 0x28, 0x2B, 0x21, 
    0x26, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 
    0xB0, 0xB1, 0x5D, 0x24, 0x2A, 0x29, 0x3B, 0x5E, 
    0x2D, 0x2F, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 
    0xB8, 0xB9, 0x7C, 0x2C, 0x25, 0x5F, 0x3E, 0x3F, 
    0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1, 
    0xC2, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22, 
    0xC3, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
    0x68, 0x69, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 
    0xCA, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 
    0x71, 0x72, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 
    0xD1, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 
    0x79, 0x7A, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 
    0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 
    0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 
    0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
    0x48, 0x49, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 
    0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 
    0x51, 0x52, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2, 0xF3, 
    0x5C, 0x9F, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 
    0x59, 0x5A, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
    0x38, 0x39, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF, 
  }; 


  if( cnvMode == cnvEbcdic )
  {
    memcpy( mData, ebcdicToLatin1, 256 );
  }
  else if( cnvMode == cnvUsAscii )
  {
    memset( mData, 0, 256 );
    for( unsigned int i=0; i<127; i++ ) { mData[i] = i; }
  }
  else
  {
    for( unsigned int i=0; i<256; i++ ) { mData[i] = i; }
    cnvMode = cnvDefault;
  }

  mState.mode = cnvMode;
  mState.name = names( mState.mode );
}
#endif




int CConversion::convert( QByteArray &buf, EMode newMode, CProgress &p )
{
  if( newMode == mode() )
  {
    p.finish();
    return( Err_Success );
  }

  if( buf.isEmpty() )
  {
    p.finish();
    setMode( newMode );
    return( Err_Success );
  }

  //
  // Make backup
  //
  const QByteArray tmp( buf.copy() );
  if( tmp.isEmpty() )
  {
    p.finish();
    return( Err_NoMemory );
  }

  unsigned char rev[256];
  uint i, sum;

  sum = 0;

  const float total = buf.size() * 2.0;

  //
  // Prepare reverse buffer.
  //
  for( i=0; i < 256; i++ ) 
  { 
    rev[(unsigned char)mData[i]] = i; 
  }

  //
  // Normalize
  //
  for( i=0; i<buf.size(); i++, sum++ ) 
  {
    buf[i] = rev[ (unsigned char)buf[i] ];

    //
    // The expired() function introduces too much overhead in this case
    // so it is only executed every 100'th character
    //
    if( i % 100 == 0 && p.expired() == true )
    {
      const int errCode = p.step( (float)sum/ total );
      if( errCode == Err_Stop )
      {
        buf = tmp;
	p.finish();
	return( Err_OperationAborted );
      }
    }
  }

  EMode origMode = mode();
  setMode( newMode );

  //
  // Apply new encoding.
  //
  for( uint i=0; i<buf.size(); i++, sum++ )
  {
    buf[i] = mData[ (unsigned char)buf[i] ];

    //
    // The expired() function introduces too much overhead in this case
    // so it is only executed every 100'th character
    //
    if( i % 100 == 0 && p.expired() == true )
    {
      const int errCode = p.step( (float)sum/ total );
      if( errCode == Err_Stop )
      {
        buf = tmp;
	setMode( origMode );
	p.finish();
	return( Err_OperationAborted );
      }
    }
  }

  p.finish();
  return( Err_Success );
}


