/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.s3;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.solr.common.util.StrUtils;
import org.apache.solr.common.util.SuppressForbidden;
import org.apache.solr.s3.S3Exception;
import org.apache.solr.s3.S3NotFoundException;
import org.apache.solr.s3.S3OutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.core.retry.RetryPolicy;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.http.apache.ApacheHttpClient;
import software.amazon.awssdk.http.apache.ProxyConfiguration;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;
import software.amazon.awssdk.services.s3.S3Configuration;
import software.amazon.awssdk.services.s3.model.CommonPrefix;
import software.amazon.awssdk.services.s3.model.Delete;
import software.amazon.awssdk.services.s3.model.DeleteObjectsRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectsResponse;
import software.amazon.awssdk.services.s3.model.DeletedObject;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;
import software.amazon.awssdk.services.s3.model.NoSuchBucketException;
import software.amazon.awssdk.services.s3.model.NoSuchKeyException;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.services.s3.paginators.ListObjectsV2Iterable;

public class S3StorageClient {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    static final String S3_FILE_PATH_DELIMITER = "/";
    private static final int MAX_KEYS_PER_BATCH_DELETE = 1000;
    private static final String S3_DIR_CONTENT_TYPE = "application/x-directory";
    private static final Set<String> NOT_FOUND_CODES = Set.of("NoSuchKey", "404 Not Found");
    private final S3Client s3Client;
    private final String bucketName;

    S3StorageClient(String bucketName, String profile, String region, String proxyUrl, boolean proxyUseSystemSettings, String endpoint, boolean disableRetries) {
        this(S3StorageClient.createInternalClient(profile, region, proxyUrl, proxyUseSystemSettings, endpoint, disableRetries), bucketName);
    }

    @VisibleForTesting
    S3StorageClient(S3Client s3Client, String bucketName) {
        this.s3Client = s3Client;
        this.bucketName = bucketName;
    }

    private static S3Client createInternalClient(String profile, String region, String proxyUrl, boolean proxyUseSystemSettings, String endpoint, boolean disableRetries) {
        RetryPolicy retryPolicy;
        S3Configuration.Builder configBuilder = S3Configuration.builder().pathStyleAccessEnabled(Boolean.valueOf(true));
        if (StrUtils.isNotNullOrEmpty((String)profile)) {
            configBuilder.profileName(profile);
        }
        ApacheHttpClient.Builder sdkHttpClientBuilder = ApacheHttpClient.builder();
        ProxyConfiguration.Builder proxyConfigurationBuilder = ProxyConfiguration.builder();
        if (StrUtils.isNotNullOrEmpty((String)proxyUrl)) {
            proxyConfigurationBuilder.endpoint(URI.create(proxyUrl));
        } else {
            proxyConfigurationBuilder.useSystemPropertyValues(Boolean.valueOf(proxyUseSystemSettings));
        }
        sdkHttpClientBuilder.proxyConfiguration((ProxyConfiguration)proxyConfigurationBuilder.build());
        sdkHttpClientBuilder.useIdleConnectionReaper(Boolean.valueOf(false));
        if (disableRetries) {
            retryPolicy = RetryPolicy.none();
        } else {
            RetryMode.Resolver retryModeResolver = RetryMode.resolver();
            if (StrUtils.isNotNullOrEmpty((String)profile)) {
                retryModeResolver.profileName(profile);
            }
            RetryMode retryMode = retryModeResolver.resolve();
            RetryPolicy.Builder retryPolicyBuilder = RetryPolicy.builder((RetryMode)retryMode);
            if (retryMode == RetryMode.ADAPTIVE) {
                retryPolicyBuilder.fastFailRateLimiting(Boolean.valueOf(false));
            }
            retryPolicy = retryPolicyBuilder.build();
        }
        DefaultCredentialsProvider.Builder credentialsProviderBuilder = DefaultCredentialsProvider.builder();
        if (StrUtils.isNotNullOrEmpty((String)profile)) {
            credentialsProviderBuilder.profileName(profile);
        }
        S3ClientBuilder clientBuilder = (S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)S3Client.builder().credentialsProvider((AwsCredentialsProvider)credentialsProviderBuilder.build())).overrideConfiguration(builder -> builder.retryPolicy(retryPolicy))).serviceConfiguration((S3Configuration)configBuilder.build())).httpClient(sdkHttpClientBuilder.build());
        if (StrUtils.isNotNullOrEmpty((String)endpoint)) {
            clientBuilder.endpointOverride(URI.create(endpoint));
        }
        if (StrUtils.isNotNullOrEmpty((String)region)) {
            clientBuilder.region(Region.of((String)region));
        }
        return (S3Client)clientBuilder.build();
    }

    void createDirectory(String path) throws S3Exception {
        String sanitizedDirPath = this.sanitizedDirPath(path);
        if (!this.pathExists(sanitizedDirPath)) {
            this.createDirectory(this.getParentDirectory(sanitizedDirPath));
            try {
                PutObjectRequest putRequest = (PutObjectRequest)PutObjectRequest.builder().bucket(this.bucketName).contentType(S3_DIR_CONTENT_TYPE).key(sanitizedDirPath).build();
                this.s3Client.putObject(putRequest, RequestBody.empty());
            }
            catch (SdkClientException ase) {
                throw S3StorageClient.handleAmazonException((SdkException)((Object)ase));
            }
        }
    }

    void delete(Collection<String> paths) throws S3Exception {
        HashSet<String> entries = new HashSet<String>();
        for (String path : paths) {
            entries.add(this.sanitizedFilePath(path));
        }
        Collection<String> deletedPaths = this.deleteObjects(entries);
        if (entries.size() != deletedPaths.size()) {
            HashSet<String> notDeletedPaths = new HashSet<String>(entries);
            entries.removeAll(deletedPaths);
            throw new S3NotFoundException(((Object)notDeletedPaths).toString());
        }
    }

    void deleteDirectory(String path) throws S3Exception {
        path = this.sanitizedDirPath(path);
        Set<String> entries = this.listAll(path);
        if (this.pathExists(path)) {
            entries.add(path);
        }
        this.deleteObjects(entries);
    }

    String[] listDir(String path) throws S3Exception {
        String prefix = path = this.sanitizedDirPath(path);
        try {
            ListObjectsV2Iterable objectListing = this.s3Client.listObjectsV2Paginator(builder -> builder.bucket(this.bucketName).prefix(prefix).delimiter(S3_FILE_PATH_DELIMITER).build());
            return (String[])Stream.concat(objectListing.contents().stream().map(S3Object::key), objectListing.commonPrefixes().stream().map(CommonPrefix::prefix)).filter(s -> s.startsWith(prefix)).map(s -> s.substring(prefix.length())).filter(s -> !s.isEmpty()).filter(s -> {
                int slashIndex = s.indexOf(S3_FILE_PATH_DELIMITER);
                return slashIndex == -1 || slashIndex == s.length() - 1;
            }).map(s -> {
                if (s.endsWith(S3_FILE_PATH_DELIMITER)) {
                    return s.substring(0, s.length() - 1);
                }
                return s;
            }).toArray(String[]::new);
        }
        catch (SdkException sdke) {
            throw S3StorageClient.handleAmazonException(sdke);
        }
    }

    boolean pathExists(String path) throws S3Exception {
        String s3Path = this.sanitizedPath(path);
        if (s3Path.isEmpty() || S3_FILE_PATH_DELIMITER.equals(s3Path)) {
            return true;
        }
        try {
            this.s3Client.headObject(builder -> builder.bucket(this.bucketName).key(s3Path));
            return true;
        }
        catch (NoSuchKeyException e) {
            return false;
        }
        catch (SdkException sdke) {
            throw S3StorageClient.handleAmazonException(sdke);
        }
    }

    boolean isDirectory(String path) throws S3Exception {
        String s3Path = this.sanitizedDirPath(path);
        try {
            HeadObjectResponse objectMetadata = this.s3Client.headObject(builder -> builder.bucket(this.bucketName).key(s3Path));
            String contentType = objectMetadata.contentType();
            return StrUtils.isNotNullOrEmpty((String)contentType) && contentType.equalsIgnoreCase(S3_DIR_CONTENT_TYPE);
        }
        catch (NoSuchKeyException nske) {
            return false;
        }
        catch (SdkException sdke) {
            throw S3StorageClient.handleAmazonException(sdke);
        }
    }

    long length(String path) throws S3Exception {
        String s3Path = this.sanitizedFilePath(path);
        if (this.isDirectory(s3Path)) {
            throw new S3Exception("Path is Directory");
        }
        try {
            HeadObjectResponse objectMetadata = this.s3Client.headObject(b -> b.bucket(this.bucketName).key(s3Path));
            String contentType = objectMetadata.contentType();
            if (StrUtils.isNullOrEmpty((String)contentType) || !contentType.equalsIgnoreCase(S3_DIR_CONTENT_TYPE)) {
                return objectMetadata.contentLength();
            }
            throw new S3Exception("Path is Directory");
        }
        catch (SdkException sdke) {
            throw S3StorageClient.handleAmazonException(sdke);
        }
    }

    InputStream pullStream(String path) throws S3Exception {
        String s3Path = this.sanitizedFilePath(path);
        try {
            return this.s3Client.getObject(b -> b.bucket(this.bucketName).key(s3Path));
        }
        catch (SdkException sdke) {
            throw S3StorageClient.handleAmazonException(sdke);
        }
    }

    OutputStream pushStream(String path) throws S3Exception {
        if (!this.parentDirectoryExist(path = this.sanitizedFilePath(path))) {
            throw new S3Exception("Parent directory doesn't exist of path: " + path);
        }
        try {
            return new S3OutputStream(this.s3Client, path, this.bucketName);
        }
        catch (SdkException sdke) {
            throw S3StorageClient.handleAmazonException(sdke);
        }
    }

    void close() {
        this.s3Client.close();
    }

    private Collection<String> deleteObjects(Collection<String> paths) throws S3Exception {
        try {
            return this.deleteObjects(paths, 1000);
        }
        catch (SdkException sdke) {
            throw S3StorageClient.handleAmazonException(sdke);
        }
    }

    @VisibleForTesting
    @SuppressForbidden(reason="Lists.partition")
    Collection<String> deleteObjects(Collection<String> entries, int batchSize) throws S3Exception {
        List keysToDelete = entries.stream().map(s -> (ObjectIdentifier)ObjectIdentifier.builder().key(s).build()).sorted(Comparator.comparing(ObjectIdentifier::key).reversed()).collect(Collectors.toList());
        List partitions = Lists.partition(keysToDelete, (int)batchSize);
        HashSet<String> deletedPaths = new HashSet<String>();
        boolean deleteIndividually = false;
        for (List partition : partitions) {
            DeleteObjectsRequest request = this.createBatchDeleteRequest(partition);
            try {
                DeleteObjectsResponse response = this.s3Client.deleteObjects(request);
                response.deleted().stream().map(DeletedObject::key).forEach(deletedPaths::add);
            }
            catch (AwsServiceException ase) {
                if (ase.statusCode() == 501) {
                    deleteIndividually = true;
                    break;
                }
                throw S3StorageClient.handleAmazonException((SdkException)((Object)ase));
            }
            catch (SdkException sdke) {
                throw S3StorageClient.handleAmazonException(sdke);
            }
        }
        if (deleteIndividually) {
            for (ObjectIdentifier k : keysToDelete) {
                try {
                    this.s3Client.deleteObject(b -> b.bucket(this.bucketName).key(k.key()));
                    deletedPaths.add(k.key());
                }
                catch (SdkException sdke) {
                    throw new S3Exception("Could not delete object with key: " + k.key(), sdke);
                }
            }
        }
        return deletedPaths;
    }

    private DeleteObjectsRequest createBatchDeleteRequest(List<ObjectIdentifier> keysToDelete) {
        return (DeleteObjectsRequest)DeleteObjectsRequest.builder().bucket(this.bucketName).delete((Delete)Delete.builder().objects(keysToDelete).build()).build();
    }

    private Set<String> listAll(String path) throws S3Exception {
        String prefix = this.sanitizedDirPath(path);
        try {
            ListObjectsV2Iterable objectListing = this.s3Client.listObjectsV2Paginator(builder -> builder.bucket(this.bucketName).prefix(prefix).build());
            return objectListing.contents().stream().map(S3Object::key).filter(s -> s.startsWith(prefix)).collect(Collectors.toSet());
        }
        catch (SdkException sdke) {
            throw S3StorageClient.handleAmazonException(sdke);
        }
    }

    private boolean parentDirectoryExist(String path) throws S3Exception {
        String parentDirectory = this.getParentDirectory(path);
        if (parentDirectory.isEmpty() || parentDirectory.equals(S3_FILE_PATH_DELIMITER)) {
            return true;
        }
        return this.pathExists(parentDirectory);
    }

    private String getParentDirectory(String path) {
        if (!path.contains(S3_FILE_PATH_DELIMITER)) {
            return "";
        }
        int fromEnd = path.length() - 1;
        if (path.endsWith(S3_FILE_PATH_DELIMITER)) {
            --fromEnd;
        }
        return fromEnd > 0 ? path.substring(0, path.lastIndexOf(S3_FILE_PATH_DELIMITER, fromEnd) + 1) : "";
    }

    String sanitizedPath(String path) throws S3Exception {
        String sanitizedPath = path.trim();
        if (sanitizedPath.startsWith(S3_FILE_PATH_DELIMITER)) {
            sanitizedPath = sanitizedPath.substring(1).trim();
        }
        return sanitizedPath;
    }

    String sanitizedFilePath(String path) throws S3Exception {
        String sanitizedPath = this.sanitizedPath(path);
        if (sanitizedPath.endsWith(S3_FILE_PATH_DELIMITER)) {
            throw new S3Exception("Invalid Path. Path for file can't end with '/'");
        }
        if (sanitizedPath.isEmpty()) {
            throw new S3Exception("Invalid Path. Path cannot be empty");
        }
        return sanitizedPath;
    }

    String sanitizedDirPath(String path) throws S3Exception {
        Object sanitizedPath = this.sanitizedPath(path);
        if (!((String)sanitizedPath).endsWith(S3_FILE_PATH_DELIMITER)) {
            sanitizedPath = (String)sanitizedPath + S3_FILE_PATH_DELIMITER;
        }
        return sanitizedPath;
    }

    static S3Exception handleAmazonException(SdkException sdke) {
        if (sdke instanceof AwsServiceException) {
            AwsServiceException ase = (AwsServiceException)((Object)sdke);
            String errMessage = String.format(Locale.ROOT, "An AmazonServiceException was thrown! [serviceName=%s] [awsRequestId=%s] [httpStatus=%s] [s3ErrorCode=%s] [message=%s]", ase.awsErrorDetails().serviceName(), ase.requestId(), ase.statusCode(), ase.awsErrorDetails().errorCode(), ase.awsErrorDetails().errorMessage());
            log.error(errMessage);
            if (sdke instanceof NoSuchKeyException || sdke instanceof NoSuchBucketException || ase.statusCode() == 404 && NOT_FOUND_CODES.contains(ase.awsErrorDetails().errorCode())) {
                return new S3NotFoundException(errMessage, ase);
            }
            return new S3Exception(errMessage, ase);
        }
        return new S3Exception(sdke);
    }
}

