/*	$NetBSD$	*/

#define ACCELERATE

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/ioctl.h>
#include <sys/types.h>		/* for makedev() */
#include <sys/conf.h>		/* cdevsw[] */

#include <machine/bus.h>
#include <machine/intr.h>
#include <machine/autoconf.h>	/* for conscode */

#include <arch/hp300/dev/intiovar.h>
#include <arch/hp300/dev/diovar.h>
#include <arch/hp300/dev/diodevs.h>
#include <arch/hp300/dev/xtopcatreg.h>

#include <dev/cons.h>
#include <dev/wsfont/wsfont.h>
#include <dev/wscons/wsconsio.h>
#include <dev/wscons/wsdisplayvar.h>
#include <dev/rasops/rasops.h>

#include "wsdisplay.h"
#include "xhilkbd.h"

#define READTYPE(base,reg,type)				\
	(*(volatile type *)((base)+(reg)))
#define WRITETYPE(base,reg,val,type)			\
	(*(volatile type *)((base)+(reg)) = (val))
#define READBYTE(base,reg)		READTYPE(base, reg, u_int8_t)
#define READWORD(base,reg)		READTYPE(base, reg, u_int16_t)
#define WRITEBYTE(base,reg,val)		WRITETYPE(base, reg, val, u_int8_t)
#define WRITEWORD(base,reg,val)		WRITETYPE(base, reg, val, u_int16_t)

#define WAITBUSY(base, planes)				\
	while (READBYTE((base), XTOPCAT_BUSY) & (planes)) DELAY(100)
#define WAITCMAPBUSY(base)				\
	while (READWORD(base, XTOPCAT_CMAP_BUSY) & 0x04) DELAY(100)

struct xtopcat_softc {
	struct device sc_dev;
	caddr_t sc_baseaddr;
	int sc_model;
	int sc_scode;
	int sc_planemask;
	struct rasops_info *sc_ri;
	int sc_nscreens;
	int sc_blanked;
};

static int  xtopcat_intio_match(struct device *, struct cfdata *, void *);
static void xtopcat_intio_attach(struct device *, struct device *, void *);
static int  xtopcat_dio_match(struct device *, struct cfdata *, void *);
static void xtopcat_dio_attach(struct device *, struct device *, void *);

const struct cfattach xtopcat_intio_ca = {
        sizeof(struct xtopcat_softc),
	xtopcat_intio_match,
	xtopcat_intio_attach,
};

const struct cfattach xtopcat_dio_ca = {
	sizeof(struct xtopcat_softc),
	xtopcat_dio_match,
	xtopcat_dio_attach,
};

void xtopcatcnprobe(struct consdev *cp);
void xtopcatcninit(struct consdev *cp);

static void xtopcat_common_attach(struct xtopcat_softc *sc);
static void xtopcat_common_init(struct rasops_info *ri);

static int xtopcatioctl(void *, u_long, caddr_t, int, struct proc *);
static paddr_t xtopcatmmap(void *, off_t, int);
static int xtopcat_alloc_screen(void *, const struct wsscreen_descr *,
	void **, int *, int *, long *);
static void xtopcat_free_screen(void *, void *);
static int xtopcat_show_screen(void *, void *, int,
	void (*) (void *, int, int), void *);

static void loadromfont(struct xtopcat_softc *sc);
#ifdef ACCELERATE
static void copyrows(void *v, int src, int dest, int num);
#endif

static struct wsscreen_descr xtopcat_stdscreen = {
	"std",				/* name */
	0, 0,				/* nrows, ncols */
	0,				/* textops */
	0, 0,				/* font width, font height */
	WSSCREEN_REVERSE		/* capabilities */
};

static const struct wsscreen_descr *xtopcat_scrlist[] = {
	&xtopcat_stdscreen,
};

static const struct wsscreen_list xtopcat_screenlist = {
	sizeof(xtopcat_scrlist) / sizeof(struct wsscreen_descr *),
	xtopcat_scrlist,
};

static const struct wsdisplay_accessops xtopcat_accessops = {
	xtopcatioctl,
	xtopcatmmap,
	xtopcat_alloc_screen,
	xtopcat_free_screen,
	xtopcat_show_screen,
	0 /* load_font */
};

static struct wsdisplay_font xtopcat_internal_font = {
	"internal",			/* name */
	'\0',				/* firstchar */
	0,				/* numchars */
	WSDISPLAY_FONTENC_ISO,		/* encoding */
	0,				/* width */
	0,				/* height */
	2,				/* stride */
	WSDISPLAY_FONTORDER_L2R,	/* bitorder */
	WSDISPLAY_FONTORDER_L2R,	/* byteorder */
	NULL,				/* data */
};

/* XXX can't be static for multiple screens (virtual consoles) */
static struct rasops_info xtopcat_console_ri;

static u_int8_t XXX_planemask;

static int
xtopcat_intio_match(struct device *parent, struct cfdata *match, void *aux)
{
	struct intio_attach_args *ia = aux;
	bus_space_handle_t bst = ia->ia_bst;
	bus_space_handle_t bsh;
	u_int8_t priid, secid;
	int found = 0;

	if (strcmp("fb      ",ia->ia_modname) != 0)
		return (0);

	if (bus_space_map(bst, ia->ia_iobase, XTOPCAT_IOSIZE, 0, &bsh))
		return (0);

	priid = bus_space_read_1(bst, bsh, XTOPCAT_PRIID);
	secid = bus_space_read_1(bst, bsh, XTOPCAT_SECID);

	if (priid == XTOPCAT_PRIID_DISPLAY) {
		switch(secid) {
                case XTOPCAT_SECID_TOPCAT:
                case XTOPCAT_SECID_LRCATSEYE:
                case XTOPCAT_SECID_HRCCATSEYE:
                case XTOPCAT_SECID_HRMCATSEYE:
                        found = 2;	/* beat ite driver */
                }
        }

	bus_space_unmap(bst, bsh, XTOPCAT_IOSIZE);
        return (found);
}

static void
xtopcat_intio_attach(struct device *parent, struct device *self, void *aux)
{
	struct intio_attach_args *ia = aux;
        struct xtopcat_softc *sc = (struct xtopcat_softc *)self;
	bus_space_tag_t bst;
	bus_space_handle_t bsh;

	bst = ia->ia_bst;
	if (bus_space_map(bst, ia->ia_iobase, XTOPCAT_IOSIZE, 0, &bsh)) {
		printf(": can't map framebuffer\n");
		return;
	}

	sc->sc_baseaddr = (caddr_t)bus_space_vaddr(bst, bsh);
	sc->sc_model = bus_space_read_1(bst, bsh, XTOPCAT_SECID);
	sc->sc_scode = -1;	/* XXX internal i/o */

	xtopcat_common_attach(sc);
}

static int
xtopcat_dio_match(struct device *parent, struct cfdata *match, void *aux)
{
	struct dio_attach_args *da = aux;

	if (da->da_id == XTOPCAT_PRIID_DISPLAY) {
		switch(da->da_secid) {
                case XTOPCAT_SECID_TOPCAT:
                case XTOPCAT_SECID_LRCATSEYE:
                case XTOPCAT_SECID_HRCCATSEYE:
                case XTOPCAT_SECID_HRMCATSEYE:
                        return (1);
                }
        }

        return (0);
}

static void
xtopcat_dio_attach(struct device *parent, struct device *self, void *aux)
{
        struct dio_attach_args *da = aux;
        struct xtopcat_softc *sc = (struct xtopcat_softc *)self;
	bus_space_handle_t bsh;
	bus_addr_t addr;

	sc->sc_model = da->da_secid;
	sc->sc_scode = da->da_scode;

        if (sc->sc_scode == conscode)
                sc->sc_baseaddr = (caddr_t)conaddr;	/* XXX */
        else {
		addr = (bus_addr_t)dio_scodetopa(sc->sc_scode);
		if (bus_space_map(da->da_bst, addr, da->da_size, 0, &bsh)) {
                        printf(": can't map framebuffer\n");
                        return;
                }
		sc->sc_baseaddr = (caddr_t)bus_space_vaddr(da->da_bst, bsh);
        }

        xtopcat_common_attach(sc);
}

static void
xtopcat_common_attach(struct xtopcat_softc *sc)
{
	struct rasops_info *ri;
	struct wsemuldisplaydev_attach_args waa;
	int isconsole;
	int depth;

	isconsole = (sc->sc_scode == conscode);

	if (isconsole) {
		sc->sc_ri = ri = &xtopcat_console_ri;
		sc->sc_nscreens = 1;
	} else {
		MALLOC(ri, struct rasops_info *, sizeof(struct rasops_info),
		    M_DEVBUF, M_NOWAIT);
		if (ri == NULL) {
			printf(": can't allocate memory\n");
			return;
		}
		memset(ri, 0, sizeof(struct rasops_info));
		ri->ri_hw = sc->sc_baseaddr;
		xtopcat_common_init(ri);
		sc->sc_ri = ri;
	}

	/*
	 * Determine display depth and mask.
	 */
	depth = READBYTE(sc->sc_baseaddr, XTOPCAT_DEPTH);
	WRITEBYTE(sc->sc_baseaddr, XTOPCAT_PRR, RR_COPY);
	*(ri->ri_bits) = 0xff;
	XXX_planemask = *(ri->ri_bits);

	/*
	 * Announce ourselves.
	 */
	printf(": %s %dx%d ", (sc->sc_model == XTOPCAT_SECID_TOPCAT ?
	    "TopCat" : "CatsEye"), ri->ri_width, ri->ri_height);
	if (depth == 1)
		printf("monochrome");
	else
		printf("%d-bit colour", depth);
	printf(" display\n");

	/*
	 * Make the ROM font useable
	 */
	loadromfont(sc);

	/*
	 * Attach wsdisplay driver.
	 */
	waa.console = isconsole;
	waa.scrdata = &xtopcat_screenlist;
	waa.accessops = &xtopcat_accessops;
	waa.accesscookie = sc;

	config_found((struct device *)sc, &waa, wsemuldisplaydevprint);
}

/*
 *  Initialise the hardware.  Called for console initialisation and
 *  for display initialisation.
 */
static void
xtopcat_common_init(struct rasops_info *ri)
{
	caddr_t baseaddr, fbbaseaddr;
	int fbwidth, fbheight, dpywidth, dpyheight;
	int fboff, fbaddr;
	u_int8_t planemask;
	int error, cookie;

	baseaddr = (caddr_t)ri->ri_hw;

	if (READBYTE(baseaddr, XTOPCAT_SECID) != XTOPCAT_SECID_TOPCAT) {
		while (READWORD(baseaddr, XTOPCAT_CATSEYE_STATUS)&1)
			;
		WRITEWORD(baseaddr, XTOPCAT_CATSEYE_STATUS, 0);
		WRITEWORD(baseaddr, XTOPCAT_VB_SELECT, 0);
		WRITEWORD(baseaddr, XTOPCAT_TCTL, 0);
		WRITEWORD(baseaddr, XTOPCAT_ACTL, 0);
		WRITEWORD(baseaddr, XTOPCAT_PNCTL, 0);
		WRITEWORD(baseaddr, XTOPCAT_RUG_CMDSTAT, 0x90);
	}

	fbwidth = (READBYTE(baseaddr, XTOPCAT_FBWMSB) << 8) |
	    READBYTE(baseaddr, XTOPCAT_FBWLSB);
	fbheight = (READBYTE(baseaddr, XTOPCAT_FBHMSB) << 8) |
	    READBYTE(baseaddr, XTOPCAT_FBHLSB);
	dpywidth = (READBYTE(baseaddr, XTOPCAT_DPYWMSB) << 8) |
	    READBYTE(baseaddr, XTOPCAT_DPYWLSB);
	dpyheight = (READBYTE(baseaddr, XTOPCAT_DPYHMSB) << 8) |
	    READBYTE(baseaddr, XTOPCAT_DPYHLSB);

	fboff = (READBYTE(baseaddr,XTOPCAT_FBOMSB) << 8) |
	    READBYTE(baseaddr,XTOPCAT_FBOLSB);
	fbaddr = READBYTE(baseaddr, fboff) << 16;

	/* XXX */
	if ((caddr_t)IIOP(baseaddr) >= (caddr_t)DIOIIBASE)
		fbbaseaddr = (caddr_t)(baseaddr + fbaddr);
	else
		fbbaseaddr = iomap((caddr_t)fbaddr, fbwidth*fbheight);

	/*
	 * Determine the number of planes by writing to the first
	 * frame buffer display location, then reading it back.
	 */
	WRITEBYTE(baseaddr, XTOPCAT_WEN, ~0);
	WRITEBYTE(baseaddr, XTOPCAT_PRR, RR_COPY);
	WRITEBYTE(baseaddr, XTOPCAT_FBEN, ~0);
	*fbbaseaddr = 0xff;
	planemask = *fbbaseaddr;
	XXX_planemask = planemask;

	/*
	 * Enable reading/writing of all the planes
	 */
	WRITEBYTE(baseaddr, XTOPCAT_FBEN, planemask);
	WRITEBYTE(baseaddr, XTOPCAT_WEN, planemask);
	WRITEBYTE(baseaddr, XTOPCAT_REN, planemask);
	WRITEBYTE(baseaddr, XTOPCAT_PRR, RR_COPY);

	/*
	 * Initialise colour map for colour displays
	 */
	if (planemask != 1) {
		WAITBUSY(baseaddr, planemask);
		WRITEBYTE(baseaddr, XTOPCAT_NBLANK, 0x01);

		WAITCMAPBUSY(baseaddr);
		WRITEWORD(baseaddr, XTOPCAT_RDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_GDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_BDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_CINDEX, 0xff);
		WRITEWORD(baseaddr, XTOPCAT_STROBE, 0xff);

		DELAY(100);
		WAITCMAPBUSY(baseaddr);
		WRITEWORD(baseaddr, XTOPCAT_RDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_GDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_BDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_CINDEX, 0x00);

		DELAY(100);
		WAITCMAPBUSY(baseaddr);
		WRITEWORD(baseaddr, XTOPCAT_RDATA, 0xff);
		WRITEWORD(baseaddr, XTOPCAT_GDATA, 0xff);
		WRITEWORD(baseaddr, XTOPCAT_BDATA, 0xff);
		WRITEWORD(baseaddr, XTOPCAT_CINDEX, 0xfe);
		WRITEWORD(baseaddr, XTOPCAT_STROBE, 0xff);

		DELAY(100);
		WAITCMAPBUSY(baseaddr);
		WRITEWORD(baseaddr, XTOPCAT_RDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_GDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_BDATA, 0x00);
		WRITEWORD(baseaddr, XTOPCAT_CINDEX, 0x00);
	}

	/*
	 * Setup wscons fonts.
	 */
	wsfont_init();

	/* prefer 12 pixel wide font */
	if ((cookie = wsfont_find(NULL, 12, 0, 0)) <= 0)
		cookie = wsfont_find(NULL, 0, 0, 0);

	if (cookie <= 0) {
 		printf("xtopcat: font table is empty\n");
		return;
	}

	if (wsfont_lock(cookie, &ri->ri_font,
	    WSDISPLAY_FONTORDER_L2R, WSDISPLAY_FONTORDER_L2R) <= 0) {
		printf("xtopcat: couldn't lock font\n");
		return;
	}
	ri->ri_wsfcookie = cookie;

	/*
	 * Setup raster console.
	 */
	ri->ri_flg = RI_CENTER | RI_CLEAR;
	ri->ri_depth = 8;
	ri->ri_width = dpywidth;
	ri->ri_height = dpyheight;
	ri->ri_stride = fbwidth;
	ri->ri_bits = fbbaseaddr;
	error = rasops_init(ri, 34, 80);

#ifdef ACCELERATE
	ri->ri_ops.copyrows = copyrows;
#endif

	if (error && baseaddr == conaddr)
		printf("Unable to initialise rasops\n");

	/* XXX shouldn't be global for multiple screens */
	xtopcat_stdscreen.nrows = ri->ri_rows;
	xtopcat_stdscreen.ncols = ri->ri_cols;
	xtopcat_stdscreen.textops = &ri->ri_ops;
	xtopcat_stdscreen.capabilities = ri->ri_caps;
}

static int
xtopcatioctl(void *v, u_long cmd, caddr_t data, int flag, struct proc *p)
{
	struct xtopcat_softc *sc = v;
	struct rasops_info *ri = sc->sc_ri;
	int turnoff;

	switch (cmd) {
	case WSDISPLAYIO_GTYPE:
		switch (sc->sc_model) {
		case XTOPCAT_SECID_TOPCAT:
			*(u_int *)data = WSDISPLAY_TYPE_TOPCAT;
			break;
		case XTOPCAT_SECID_LRCATSEYE:
		case XTOPCAT_SECID_HRCCATSEYE:
		case XTOPCAT_SECID_HRMCATSEYE:
			*(u_int *)data = WSDISPLAY_TYPE_CATSEYE;
			break;
		}
		return (0);

	case WSDISPLAYIO_GINFO:
#define wsd_fbip ((struct wsdisplay_fbinfo *)data)
		wsd_fbip->height = ri->ri_height;
		wsd_fbip->width = ri->ri_width;
		wsd_fbip->depth = ri->ri_depth;
		wsd_fbip->cmsize =
		    (1 << READBYTE(sc->sc_baseaddr, XTOPCAT_DEPTH));
		return (0);

	case WSDISPLAYIO_SVIDEO:
		turnoff = *(int *)data = WSDISPLAYIO_VIDEO_OFF;
		WAITBUSY(sc->sc_baseaddr, XXX_planemask);
		if ((sc->sc_blanked == 0) && turnoff)
			WRITEWORD(sc->sc_baseaddr, XTOPCAT_NBLANK, 0x00);
		else
			WRITEWORD(sc->sc_baseaddr, XTOPCAT_NBLANK, 0x01);
		sc->sc_blanked = turnoff;
		return (0);

	case WSDISPLAYIO_GVIDEO:
		*(u_int *)data = sc->sc_blanked ?
		    WSDISPLAYIO_VIDEO_OFF : WSDISPLAYIO_VIDEO_ON;
		return (0);

	case WSDISPLAYIO_GETCMAP:
/*		return get_cmap(sc, (struct wsdisplay_cmap *)data); */

	case WSDISPLAYIO_PUTCMAP:
/*		return put_cmap(sc, (struct wsdisplay_cmap *)data); */

	case WSDISPLAYIO_GCURPOS:
	case WSDISPLAYIO_SCURPOS:
	case WSDISPLAYIO_GCURMAX:
	case WSDISPLAYIO_GCURSOR:
	case WSDISPLAYIO_SCURSOR:
		return (0);
	}
	return (ENOTTY);
}

static paddr_t
xtopcatmmap(void *v, off_t offset, int prot)
{
	struct xtopcat_softc *sc = v;
	struct rasops_info *ri = sc->sc_ri;

	return ((paddr_t)ri->ri_bits);
}

static int
xtopcat_alloc_screen(void *v, const struct wsscreen_descr *type,
	void **cookiep, int *curxp, int *curyp, long *attrp)
{
	struct xtopcat_softc *sc = v;
	struct rasops_info *ri = sc->sc_ri;
	long defattr;

	if (sc->sc_nscreens > 0)
		return (ENOMEM);

	*cookiep = ri;
	*curxp = 0;
	*curyp = 0;
	(*ri->ri_ops.alloc_attr)(ri, 0, 0, 0, &defattr);
	*attrp = defattr;
	sc->sc_nscreens++;
	return (0);	
}

static void
xtopcat_free_screen(void *v, void *cookie)
{
	struct xtopcat_softc *sc = v;

	if (sc->sc_ri == &xtopcat_console_ri)
		panic("xtopcat_free_screen: console");

	sc->sc_nscreens--;
}

static int
xtopcat_show_screen(void *v, void *cookie, int waitok,
	void (*cb) (void *, int, int), void *cbarg)
{
	return (0);
}

static void
loadromfont(struct xtopcat_softc *sc)
{
	caddr_t baseaddr = sc->sc_baseaddr;
	int romoff, fontoff;

	romoff = (READBYTE(baseaddr, XTOPCAT_ROMOFFMSB) << 8) |
	    READBYTE(baseaddr, XTOPCAT_ROMOFFLSB);
	fontoff = ((READBYTE(baseaddr, romoff + ROM_FONTOFFMSB) << 8) |
	    READBYTE(baseaddr, romoff + ROM_FONTOFFLSB));

	xtopcat_internal_font.fontwidth =
	    READBYTE(baseaddr, fontoff + FONT_WIDTH);
	xtopcat_internal_font.fontheight =
	    READBYTE(baseaddr, fontoff + FONT_HEIGHT);
	xtopcat_internal_font.data = (void *)(baseaddr + fontoff + FONT_DATA);
	(void) wsfont_add(&xtopcat_internal_font, 0);
}

#ifdef ACCELERATE
static void
copyrows(void *v, int src, int dst, int num)
{
	struct rasops_info *ri = v;
	caddr_t baseaddr = ri->ri_hw;
	u_int16_t xsrc, ysrc, xdst, ydst, width, height;

	if (num == 0)
		return;

	xsrc = ri->ri_xorigin;
	ysrc = src * ri->ri_font->fontheight + ri->ri_yorigin;
	xdst = ri->ri_xorigin;
	ydst = dst * ri->ri_font->fontheight + ri->ri_yorigin;
	width = ri->ri_emuwidth;
	height = num * ri->ri_font->fontheight;

	WAITBUSY(baseaddr, XXX_planemask);
	WRITEBYTE(baseaddr, XTOPCAT_WMRR, RR_COPY);
	WRITEWORD(baseaddr, XTOPCAT_SOURCE_X, xsrc);
	WRITEWORD(baseaddr, XTOPCAT_SOURCE_Y, ysrc);
	WRITEWORD(baseaddr, XTOPCAT_DEST_X, xdst);
	WRITEWORD(baseaddr, XTOPCAT_DEST_Y, ydst);
	WRITEWORD(baseaddr, XTOPCAT_WWIDTH, width);
	WRITEWORD(baseaddr, XTOPCAT_WHEIGHT, height);
	WRITEWORD(baseaddr, XTOPCAT_WMOVE, XXX_planemask);
	WAITBUSY(baseaddr, XXX_planemask);
}
#endif




/*
 * Console support.
 *
 * The console code is totally screwed.  What we do instead is let ite
 * attach as the console, but override the device match during
 * autoconfiguration.
 *
 */

extern int wsdisplayopen(dev_t, int, int, struct proc *);
extern int topcatcnprobe(struct consdev *);
extern caddr_t conaddr;
void xhilkbdcninit(void);

void
xtopcatcnprobe(struct consdev *cp)
{
	int maj;

	printf("xtopcatcnprobe(): looking for console\n");

	/* let topcat driver do the hard stuff for us ATM */
	topcatcnprobe(cp);

	for (maj=0; maj < nchrdev; maj++)
		if (cdevsw[maj].d_open == wsdisplayopen)
			break;

	cp->cn_dev = makedev(maj, 0);
}


void
xtopcatcninit(struct consdev *cp)
{
	caddr_t baseaddr = (caddr_t)IIOV(0x560000);
	struct rasops_info *ri;
	long defattr;

#if NXHILKBD > 0
	xhilkbdcninit();
#endif

	ri = &xtopcat_console_ri;
	ri->ri_hw = (void *)baseaddr;
	xtopcat_common_init(ri);
	(*ri->ri_ops.alloc_attr)(ri, 0, 0, 0, &defattr);
	wsdisplay_cnattach(&xtopcat_stdscreen, ri, 0, 0, defattr);
}
