/*	$NetBSD$	*/

#include "wskbd.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/ioctl.h>

#include <machine/cpu.h>			/* for IIOV */

#include <arch/hp300/dev/xhilreg.h>
#include <arch/hp300/dev/xhilvar.h>
#include <arch/hp300/dev/xhilkbd_keymaps.h>

#include <arch/hp300/dev/hilreg.h>		/* XXX should go away */

#include <dev/wscons/wsconsio.h>
#include <dev/wscons/wskbdvar.h>
#include <dev/wscons/wsksymdef.h>
#include <dev/wscons/wsksymvar.h>


static int xhilkbdmatch(struct device *, struct cfdata *, void *);
static void xhilkbdattach(struct device *, struct device *, void *aux);

struct xhilkbd_softc {
	struct device sc_dev;
	struct device *sc_parent;
	struct device *sc_wskbddev;
	int sc_enabled;
};

const struct cfattach xhilkbd_ca = {
	sizeof (struct xhilkbd_softc), xhilkbdmatch, xhilkbdattach, 
};

static int xhilkbd_handler(void *arg, u_int8_t *pkt, int len);

static int xhilkbd_enable(void *, int);
static void xhilkbd_set_leds(void *, int);
static int xhilkbd_ioctl(void *, u_long, caddr_t, int, struct proc *);

static const struct wskbd_accessops xhilkbd_accessops = {
	xhilkbd_enable,
	xhilkbd_set_leds,
	xhilkbd_ioctl,
};

static struct wskbd_mapdata xhilkbd_keymapdata = {
	xhilkbd_keydesctab,		/* table of keymaps */
	KB_US,				/* default keymap */
};

void xhilkbdcninit(void);
int xhilkbdcngetc(dev_t dev);

static void xhilkbd_cngetc(void *, u_int *, int *);
static void xhilkbd_cnpollc(void *, int);
static void xhilkbd_cnbell(void *, u_int, u_int, u_int);

static const struct wskbd_consops xhilkbd_consops = {
	xhilkbd_cngetc,
	xhilkbd_cnpollc,
	xhilkbd_cnbell,
};

static int xhilkbd_isconsole;


static int
xhilkbdmatch(struct device *parent, struct cfdata *match, void *aux)
{
	struct xhil_attach_args *ha = aux;

	if (ha->ha_id != 0x40)
		return (0);

	return (1);
}

static void
xhilkbdattach(struct device *parent, struct device *self, void *aux)
{
	struct xhilkbd_softc *sc = (struct xhilkbd_softc *)self;
	struct wskbddev_attach_args waa;

	printf("\n");

	sc->sc_parent = parent;

	xhilregister(parent, xhilkbd_handler, sc, 0x40);

	waa.console = xhilkbd_isconsole;
	waa.keymap = &xhilkbd_keymapdata;
	waa.accessops = &xhilkbd_accessops;
	waa.accesscookie = sc;

	sc->sc_wskbddev = config_found(self, &waa, wskbddevprint);
}

static int
xhilkbd_handler(void *arg, u_int8_t *pkt, int len)
{
	struct xhilkbd_softc *sc = arg;
	u_int wstype;
	int wsdata;	

	wstype = (pkt[1]&1 ? WSCONS_EVENT_KEY_UP : WSCONS_EVENT_KEY_DOWN);
	wsdata = (pkt[1]>>1);
	wskbd_input(sc->sc_wskbddev, wstype, wsdata);

	return (1);
}

static int
xhilkbd_enable(void *cookie, int on)
{
	struct xhilkbd_softc *sc = cookie;

	if (on)
		sc->sc_enabled = 1;
	else
		sc->sc_enabled = 0;

	return (0);
}

static void
xhilkbd_set_leds(void *v, int flags)
{
	/* do nothing */
}

static int
xhilkbd_ioctl(void *v, u_long cmd, caddr_t data, int flag, struct proc *p)
{
	struct xhilkbd_softc *sc = v;

	switch(cmd) {
	case WSKBDIO_GTYPE:
		*(int *)data = WSKBD_TYPE_HP_HIL;
		return (0);
	case WSKBDIO_SETLEDS:
                return 0;
	case WSKBDIO_GETLEDS:
		*(int *)data = 0;
		return 0;
	case WSKBDIO_COMPLEXBELL:
#define d ((struct wskbd_bell_data *)data)
		xhilbell(sc->sc_parent, d->pitch, d->period, d->volume, 0);
		return (0);
	}
	return (-1);
}


/*
 *  Console interface
 */

void
xhilkbdcninit(void)
{
	u_int8_t lang, lpctrl;

	xhilkbd_isconsole = 1;

        HILWAIT(HILADDR);
        WRITEHILCMD(HILADDR, HIL_READLPCTRL);
        HILDATAWAIT(HILADDR);
        lpctrl = READHILDATA(HILADDR);
	lpctrl &= ~LPC_KBDCOOK;
        HILWAIT(HILADDR);
        WRITEHILCMD(HILADDR, HIL_WRITELPCTRL);
        HILWAIT(HILADDR);
        WRITEHILDATA(HILADDR, lpctrl);

        HILWAIT(HILADDR);
        WRITEHILCMD(HILADDR, HIL_WRITEKBDSADR);
        HILWAIT(HILADDR);
        WRITEHILDATA(HILADDR, 0);

        HILWAIT(HILADDR);
        WRITEHILCMD(HILADDR, HIL_READKBDLANG);
        HILDATAWAIT(HILADDR);
        lang = READHILDATA(HILADDR);
        HILWAIT(HILADDR);
        WRITEHILCMD(HILADDR, HIL_INTON);
	switch (lang) {
	case 0x1f:
		xhilkbd_keymapdata.layout = KB_US;
		break;
	case 0x17:
		xhilkbd_keymapdata.layout = KB_UK;
		break;
	case 0x0e:
		xhilkbd_keymapdata.layout = KB_DK;
		break;
	}

	wskbd_cnattach(&xhilkbd_consops, NULL, &xhilkbd_keymapdata);
}

void
xhilkbd_cngetc(void *v, u_int *type, int *datap)
{
	u_int8_t pkt[16];
	u_int8_t *pktp = pkt;
	int data, status;

	while (1) {

	        while (((status = READHILSTAT(HILADDR)) & HIL_DATA_RDY) == 0)
        	        ;
	        data = READHILDATA(HILADDR);

		switch(status>>4) {

		case 0x5:	/* hil status available in data register */

			switch (data & XHIL_STATUS_MASK) {
			case XHIL_STATUS_COMMAND|XHIL_STATUS_POLLDATA: /* 18 */
				if (pkt[0] & 0x40) {
					*type = (pkt[1]&1 ?
					    WSCONS_EVENT_KEY_UP :
					    WSCONS_EVENT_KEY_DOWN);
					*datap = (pkt[1]>>1);
					return;
				}
				break;
			case XHIL_STATUS_POLLDATA: /* 10 */
				pktp = pkt;
				break;
			}
			break;

		case 0x6:	/* hil data available in data register */
			*pktp++ = data;
			break;
		}
	}
}

void
xhilkbd_cnpollc(void *v, int on)
{
	/* nothing */
}

void
xhilkbd_cnbell(void *v, u_int pitch, u_int period, u_int volume)
{
	u_int8_t duration, freq;

	duration = ~(period/10);
	freq = (pitch>>7);
        HILWAIT(HILADDR);
        WRITEHILCMD(HILADDR, XHIL_SETTONE);
        HILWAIT(HILADDR);
        WRITEHILDATA(HILADDR, duration);
        HILWAIT(HILADDR);
        WRITEHILDATA(HILADDR, freq);
}
