/*	$NetBSD$	*/

#include "opt_hildev.h"

#include "rnd.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/queue.h>

#include <machine/intr.h>

#include <arch/hp300/dev/intioreg.h>
#include <arch/hp300/dev/intiovar.h>
#include <arch/hp300/dev/xhilreg.h>
#include <arch/hp300/dev/xhilvar.h>
#ifdef HILDEV
#include <arch/hp300/dev/xhildev.h>
#endif

#if NRND > 0
#include <sys/rnd.h>
#endif

struct xhil_handler {
	LIST_ENTRY(xhil_handler) xh_link;
	int (*xh_func)(void *, u_int8_t *, int);;
	void *xh_arg;
	int xh_id;
};

struct xhil_softc {
	struct device sc_dev;
	bus_space_tag_t sc_bst;
	bus_space_handle_t sc_bsh;
	void *sc_ih;
	LIST_HEAD(, xhil_handler) sc_handlers;
#ifdef HILDEV
	void *sc_hildevcookie;
#endif
#if NRND > 0
	rndsource_element_t rnd_source;
#endif
};

static int xhilmatch(struct device *, struct cfdata *, void *);
static void xhilattach(struct device *, struct device *, void *aux);

const struct cfattach xhil_ca = {
	sizeof (struct xhil_softc), xhilmatch, xhilattach, 
};

static void xhilattach_deferred(struct device *self);
static int xhilprint(void *aux, const char *pnp);
static int xhilintr(void *);
static void xhildispatch(struct xhil_softc *sc, u_int8_t *pkt, int len);

static int
xhilmatch(struct device *parent, struct cfdata *match, void *aux)
{
	struct intio_attach_args *ia = aux;

	if (strcmp("hil     ", ia->ia_modname) != 0)
		return (0);

	return (1);
}

static void
xhilattach(struct device *parent, struct device *self, void *aux)
{
	struct intio_attach_args *ia = aux;
	struct xhil_softc *sc = (struct xhil_softc *)self;

	printf("\n");

	if (bus_space_map(ia->ia_bst, ia->ia_iobase, INTIO_DEVSIZE,
	    0, &sc->sc_bsh)) {
		printf("%s: can't map registers\n",
		    sc->sc_dev.dv_xname);
		return;
	}

	LIST_INIT(&sc->sc_handlers);

	sc->sc_ih = intio_intr_establish(xhilintr, sc, ia->ia_ipl, IPL_TTY);

#if 0
	config_interrupts(self, xhilattach_deferred);
#else
	xhilattach_deferred(self);
#endif
}

static void
xhilattach_deferred(struct device *self)
{
	struct xhil_softc *sc = (struct xhil_softc *)self;
	struct xhil_attach_args ha;
#if 0
	u_int8_t db;

	/* initialise the loop */
	db = XHIL_LPC_RECONF | XHIL_LPC_NOERROR | XHIL_LPC_AUTOPOLL;
	intio_device_write(sc->sc_bst, sc->sc_bsh, XHIL_WRITELPCTRL, &db, 1);

	/*
	 * Delay 1 second for reconfiguration, thean read data to clear
	 * the interrupt
	 */
	DELAY(1000000);
	intio_device_readcmd(sc->sc_bst, sc->sc_bsh, 0, &db);

	/* loop until a successful reconfiguration is reported */ 
	do {
		intio_device_readcmd(sc->sc_bst, sc->sc_bst,
		    XHIL_READLPSTAT, &db);
	} while ((db & (XHIL_LPS_CONFFAIL|XHIL_LPS_CONFGOOD)) == 0);

#endif
	/* turn interrupts on */
	intio_device_writecmd(sc->sc_bst, sc->sc_bsh, XHIL_INTON, NULL, 0);

#if NRND > 0
	rnd_attach_source(&sc->rnd_source, "xhil", RND_TYPE_TTY, 0);
#endif

	/* XXX hack */
	ha.ha_id = 0x40;
	config_found_sm(self, &ha, xhilprint, NULL);

	/* XXX bigger hack */
	ha.ha_id = 0x02;
	config_found_sm(self, &ha, xhilprint, NULL);

#ifdef HILDEV
	sc-sc_hildev_cookie = xhildev_attach();
#endif
}

static int
xhilprint(void *aux, const char *pnp)
{
	return (UNCONF);
}

static int
xhilintr(void *arg)
{
	struct xhil_softc *sc = arg;
        u_int8_t data, status;
	static u_int8_t pkt[16];
	static u_int8_t *pktp = pkt;

        intio_device_readstate(sc->sc_bst, sc->sc_bsh, &status, &data);

	switch ((status>>INTIO_DEV_SRSHIFT) & INTIO_DEV_SRMASK) {
	case XHIL_INTIO_STATAVAIL:
		switch (data & XHIL_STATUS_MASK) {
		case XHIL_STATUS_COMMAND|XHIL_STATUS_POLLDATA:
			xhildispatch(sc, pkt, pktp-pkt);
			break;
		case XHIL_STATUS_COMMAND:
			break;
		case XHIL_STATUS_POLLDATA:
			pktp = pkt;
			break;
		case XHIL_STATUS_ERROR:
			printf("%s: error\n", sc->sc_dev.dv_xname);
			if (data == XHIL_STATUS_RECONFIG)
				/* xhilconfig(); */
			break;
		}
		break;

	case XHIL_INTIO_DATAAVAIL:
		*pktp++ = data;
		break;
	}

#if NRND > 0
	rnd_add_uint32(&sc->rnd_source, (status<<8)|data);
#endif

	return (1);
}

static void
xhildispatch(struct xhil_softc *sc, u_int8_t *pkt, int len)
{
	struct xhil_handler *handler;
	int rv;

	LIST_FOREACH(handler, &sc->sc_handlers, xh_link) {
		if (handler->xh_id == pkt[0]) {
			rv = (handler->xh_func)(handler->xh_arg, pkt, len);
			if (rv == 1)
				return;
		}
	}

#ifdef XHILDEV
	xhildev_dispatch(sc->sc_hildev_cookie, pkt, len);
#else
	printf("%s: spurious packet for 0x%x\n", sc->sc_dev.dv_xname, pkt[0]);
#endif
}

void *
xhilregister(struct device *dev, int (*func)(void *, u_int8_t *, int),
    void *arg, int id)
{
	struct xhil_softc *sc = (struct xhil_softc *)dev;
	struct xhil_handler *handler;

	handler = (struct xhil_handler *)malloc(sizeof(struct xhil_handler),
	    M_DEVBUF, M_NOWAIT);
	if (handler == NULL)
		panic("xhil_register: can't allocate space for handler");

	/* fill in the entry */
	handler->xh_func = func;
	handler->xh_arg = arg;
	handler->xh_id = id;

	LIST_INSERT_HEAD(&sc->sc_handlers, handler, xh_link);

	return ((void *)handler);
}

void
xhilbell(struct device *dev, u_int pitch, u_int period,
	u_int volume, int poll)
{
	struct xhil_softc *sc = (struct xhil_softc *)dev;
	u_int8_t buf[2];

	buf[0] = ~(period/10); 		/* tone duration in hundredths */
	buf[1] = (pitch>>7);		/* tone frequency (0-63) */
	intio_device_writecmd(sc->sc_bst, sc->sc_bsh, XHIL_SETTONE, buf, 2);
}
