/*	$NetBSD$	*/

/*-
 * Copyright (c) 1996, 1998 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason R. Thorpe.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the NetBSD
 *        Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Autoconfiguration support for hp300 internal i/o space.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h> 

#include <machine/hp300spu.h>

#include <hp300/dev/intioreg.h> 
#include <hp300/dev/intiovar.h>

int	intiomatch(struct device *, struct cfdata *, void *);
void	intioattach(struct device *, struct device *, void *);
int	intioprint(void *, const char *);

const struct cfattach intio_ca = {
	sizeof(struct device), intiomatch, intioattach
};

int intio_3xx_device_readcmd(bus_space_tag_t, bus_space_handle_t,
	int, u_int8_t *);
int intio_3xx_device_writecmd(bus_space_tag_t, bus_space_handle_t,
	int, u_int8_t *, int);
int intio_3xx_device_readstate(bus_space_tag_t, bus_space_handle_t,
	u_int8_t *, u_int8_t *);
#define intio_4xx_device_readcmd	intio_3xx_device_readcmd
#define intio_4xx_device_writecmd	intio_3xx_device_writecmd
#define intio_4xx_device_readstate	intio_3xx_device_readstate

const struct intio_accessops intio_3xx_accessops = {
	intio_3xx_device_readcmd,
	intio_3xx_device_writecmd,
	intio_3xx_device_readstate,
};

const struct intio_accessops intio_4xx_accessops = {
	intio_4xx_device_readcmd,
	intio_4xx_device_writecmd,
	intio_4xx_device_readstate,
};

#ifdef notyet
const struct intio_accessops intio_7xx_accessops = {
	intio_7xx_device_readcmd,
	intio_7xx_device_writecmd,
	intio_7xx_device_readstate,
};

const struct intio_accessops intio_8xx_accessops = {
	intio_8xx_device_read,
	intio_8xx_device_write,
	intio_8xx_device_readstate,
};
#endif

const struct intio_builtins intio_3xx_builtins[] = {
	{ "rtc     ",	0x020000,	-1},
	{ "hil     ",	0x028000,	1},
	{ "fb      ",	0x160000,	-1},
};
#define nintio_3xx_builtins \
	(sizeof(intio_3xx_builtins) / sizeof(intio_3xx_builtins[0]))

const struct intio_builtins intio_4xx_builtins[] = {
	{ "rtc     ",	0x020000,	-1},
	{ "frodo   ",	0x01c000,	5},
	{ "hil     ",	0x028000,	1},
	{ "fb      ",	0x160000,	-1},
};
#define nintio_4xx_builtins \
	(sizeof(intio_4xx_builtins) / sizeof(intio_4xx_builtins[0]))

static int intio_matched = 0;
const struct intio_accessops *intio_accessops;


int
intiomatch(parent, match, aux)
	struct device *parent;
	struct cfdata *match;
	void *aux;
{
	/* Allow only one instance. */
	if (intio_matched)
		return (0);

	intio_matched = 1;
	return (1);
}

void
intioattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct intio_attach_args ia;
	const struct intio_builtins *ib;
	int ndevs;
	int i;

	printf("\n");

	switch (machineid) {
	case HP_320:
	case HP_330:
	case HP_340:
	case HP_345:
	case HP_350:
	case HP_360:
	case HP_370:
	case HP_375:
	case HP_380:
	case HP_385:
		intio_accessops = &intio_3xx_accessops;
		ib = intio_3xx_builtins;
		ndevs = nintio_3xx_builtins;
		break;
	case HP_400:
	case HP_425:
	case HP_433:
		intio_accessops = &intio_4xx_accessops;
		ib = intio_4xx_builtins;
		ndevs = nintio_4xx_builtins;
		break;
	}

	bzero(&ia, sizeof(ia));
	ia.ia_bst = HP300_BUS_SPACE_INTIO;

	for (i=0; i<ndevs; i++) {
		strncpy(ia.ia_modname, ib[i].ib_modname, INTIO_MOD_LEN);
		ia.ia_modname[INTIO_MOD_LEN] = '\0';
		ia.ia_iobase = ib[i].ib_offset;
		ia.ia_addr = (bus_addr_t)(intiobase + ib[i].ib_offset);
		ia.ia_ipl = ib[i].ib_ipl;
		config_found(self, &ia, intioprint);
	}
}

int
intioprint(aux, pnp)
	void *aux;
	const char *pnp;
{
	struct intio_attach_args *ia = aux;

	if (pnp)
		printf("%s at %s", ia->ia_modname, pnp);
	if (ia->ia_addr != 0) {
		printf(" addr 0x%lx", INTIOBASE + ia->ia_iobase);
		if (ia->ia_ipl != -1)
			printf(" ipl %d", ia->ia_ipl);
	}
	return (UNCONF);
}

/*
 *  Routines for interface to the wired devices.
 */

#ifdef DIAGNOSTIC
int
intio_device_readcmd(bus_space_tag_t bst, bus_space_handle_t bsh,
	int cmd, u_int8_t *datap)
{
	if (intio_accessops == NULL)
		panic("intio not attached");
	return (*intio_accessops->ia_readcmd)(bst, bsh, cmd, datap);
}

int
intio_device_writecmd(bus_space_tag_t bst, bus_space_handle_t bsh,
	int cmd, u_int8_t *datap, int len)
{
	if (intio_accessops == NULL)
		panic("intio not attached");
	return (*intio_accessops->ia_writecmd)(bst, bsh, cmd, datap, len);
}

int
intio_device_readstate(bus_space_tag_t bst, bus_space_handle_t bsh,
	u_int8_t *statusp, u_int8_t *datap)
{
	if (intio_accessops == NULL)
		panic("intio not attached");
	return (*intio_accessops->ia_readstate)(bst, bsh, statusp, datap);
}
#endif

/*
 *  system-dependent functions for wired devices
 */

#define WAIT(bst,bsh)		\
	while (bus_space_read_1(bst,bsh,INTIO_DEV_3xx_STAT) \
		& INTIO_DEV_BUSY)
#define DATAWAIT(bst,bsh)	\
	while (!(bus_space_read_1(bst, bsh, INTIO_DEV_3xx_STAT) \
		& INTIO_DEV_DATA_READY))

int
intio_3xx_device_readcmd(bus_space_tag_t bst, bus_space_handle_t bsh, int cmd,
	u_int8_t *datap)
{
        u_int8_t status;

	if (cmd != 0) {
		WAIT(bst, bsh);
		bus_space_write_1(bst, bsh, INTIO_DEV_3xx_CMD, cmd);
	}
        do {
		DATAWAIT(bst, bsh);
		status = bus_space_read_1(bst, bsh, INTIO_DEV_3xx_STAT);
		*datap = bus_space_read_1(bst, bsh, INTIO_DEV_3xx_DATA);
	} while (((status >> INTIO_DEV_SRSHIFT) & INTIO_DEV_SRMASK)
		!= INTIO_DEV_SR_DATAAVAIL);
	return (0);		
}

int
intio_3xx_device_writecmd(bus_space_tag_t bst, bus_space_handle_t bsh,
	int cmd, u_int8_t *datap, int len)
{
        WAIT(bst,bsh);
	bus_space_write_1(bst, bsh, INTIO_DEV_3xx_CMD, cmd);
        while (len--) {
                WAIT(bst,bsh);
		bus_space_write_1(bst, bsh, INTIO_DEV_3xx_DATA, *datap++);
        }
	return (0);
}

int
intio_3xx_device_readstate(bus_space_tag_t bst, bus_space_handle_t bsh,
	u_int8_t *statusp, u_int8_t *datap)
{
	*statusp = bus_space_read_1(bst, bsh, INTIO_DEV_3xx_STAT);
	*datap = bus_space_read_1(bst, bsh, INTIO_DEV_3xx_DATA);
	return (0);
}
#undef WAIT
#undef DATAWAIT
