/*
 * Copyright (C) 2007-2009 KenD00
 * 
 * This file is part of DumpHD.
 * 
 * DumpHD is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package dumphd.gui;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.table.AbstractTableModel;

import dumphd.core.DiscSet;

/**
 * FileTableModel used by a SourceTab to display the files of its DiscSet in a table.
 * 
 * Very simple model, does not allow that the file structure of the underlying DiscSet changes.
 *  
 * @author KenD00
 */
public class FileTableModel extends AbstractTableModel {

   private DiscSet ds = null;
   private TreeMap<Integer, Integer> streamSetOffsets = new TreeMap<Integer, Integer>();
   private int fileSetOffset = 0;


   public FileTableModel() {

   }

   public FileTableModel(DiscSet ds) {
      this.ds = ds;
      if (ds != null) {
         calculateOffsets();
      }
   }

   public DiscSet getData() {
      return ds;
   }

   public void setData(DiscSet ds) {
      this.ds = ds;
      if (ds != null) {
         calculateOffsets();
      }
      fireTableDataChanged();
   }

   public Class<?> getColumnClass(int columnIndex) {
      return String.class;
   }

   public String getColumnName(int columnIndex) {
      return "File";
   }

   public boolean isCellEditable(int rowIndex, int columnIndex) {
      return false;
   }

   /* (non-Javadoc)
    * @see javax.swing.table.TableModel#getColumnCount()
    */
   public int getColumnCount() {
      return 1;
   }

   /* (non-Javadoc)
    * @see javax.swing.table.TableModel#getRowCount()
    */
   public int getRowCount() {
      if (ds != null) {
         return fileSetOffset + ds.fileSet.size();
      } else {
         return 0;
      }
   }

   /* (non-Javadoc)
    * @see javax.swing.table.TableModel#getValueAt(int, int)
    */
   public Object getValueAt(int rowIndex, int columnIndex) {
      if (ds != null) {
         if (rowIndex < fileSetOffset) {
            // Row inside streamSets
            int offset = 0;
            Iterator<Map.Entry<Integer, Integer>> it = streamSetOffsets.entrySet().iterator();
            while (it.hasNext()) {
               Map.Entry<Integer, Integer> entry = it.next();
               if (rowIndex < entry.getKey()) {
                  return ds.streamSets.get(entry.getValue()).get(rowIndex - offset);
               } else {
                  offset = entry.getValue();
               }
            }
            // This should never happen!
            return "";
         } else {
            // Row inside fileSet
            return ds.fileSet.get(rowIndex - fileSetOffset);
         }
      } else {
         return "";
      }
   }

   /**
    * Helper Method, calculates the offsets after the DiscSet was changed
    */
   private void calculateOffsets() {
      fileSetOffset = 0;
      Iterator<Map.Entry<Integer, ArrayList<String>>> it = ds.streamSets.entrySet().iterator();
      while (it.hasNext()) {
         Map.Entry<Integer, ArrayList<String>> entry = it.next();
         fileSetOffset += entry.getValue().size();
         streamSetOffsets.put(fileSetOffset, entry.getKey());
      }
   }

}
