/* -*- C -*-
 *
 * This file is part of the Sofia-SIP package
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Pekka Pessi <pekka.pessi@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef SIP_PROTOS_H
/** Defined when <sofia-sip/sip_protos.h> has been included. */
#define SIP_PROTOS_H

/**@file sofia-sip/sip_protos.h
 *
 * SIP prototypes and macros for each header.
 *
 * This file is automatically generated from <sip.h> by msg_parser.awk.
 *
 * @author Pekka Pessi <Pekka.Pessi@nokia.com>.
 *
 */

#include <sofia-sip/su_config.h>

#ifndef SIP_HEADER_H
#include <sofia-sip/sip_header.h>
#endif

#ifndef SIP_HCLASSES_H
#include <sofia-sip/sip_hclasses.h>
#endif

SOFIA_BEGIN_DECLS

#if SU_HAVE_INLINE
/** Get SIP structure from msg. */
su_inline
sip_t *sip_object(msg_t const *msg)
{
  return (sip_t *)msg_public(msg, SIP_PROTOCOL_TAG);
}

/** Insert a (list of) header(s) to the header structure and fragment chain.
 *
 * The function @c sip_header_insert() inserts header or list of headers
 * into a SIP message.  It also inserts them into the the message fragment
 * chain, if it exists.
 *
 * When inserting headers into the fragment chain, a request (or status) is
 * inserted first and replaces the existing request (or status).  The Via
 * headers are inserted after the request or status, and rest of the headers
 * after request, status, or Via headers.
 *
 * If the header is a singleton, existing headers with the same class are
 * removed.
 *
 * @param msg message owning the fragment chain
 * @param sip SIP message structure to which header is added
 * @param h   list of header(s) to be added
 */
su_inline
int sip_header_insert(msg_t *msg, sip_t *sip, sip_header_t *h)
{
  return msg_header_insert(msg, (msg_pub_t *)sip, (msg_header_t *)h);
}

/** Remove a header from a SIP message. */
su_inline
int sip_header_remove(msg_t *msg, sip_t *sip, sip_header_t *h)
{
  return msg_header_remove(msg, (msg_pub_t *)sip, (msg_header_t *)h);
}

/** Return name of the header. */
su_inline
char const *sip_header_name(sip_header_t const *h, int compact)
{
  if (compact && h->sh_class->hc_short[0])
    return h->sh_class->hc_short;
  else
    return h->sh_class->hc_name;
}

/** Return data after header structure. */
su_inline
void *sip_header_data(sip_header_t *h)
{
  return h && h != SIP_NONE ? h->sh_class->hc_size + (char *)h : NULL;
}
#else
sip_t *sip_object(msg_t *msg);
int sip_header_insert(msg_t *msg, sip_t *sip, sip_header_t *h);
int sip_header_remove(msg_t *msg, sip_t *sip, sip_header_t *h);
char const *sip_header_name(sip_header_t const *h, int compact);
void *sip_header_data(sip_header_t *h);
#endif

/**@addtogroup sip_request
 * @{
 */

/** Parse a SIP @ref sip_request "request line". @internal */
SOFIAPUBFUN issize_t sip_request_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_request "request line". @internal */
SOFIAPUBFUN issize_t sip_request_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_request "request line"
 * structure #sip_request_t from #sip_t.
 *
 */
#define sip_request(sip) \
  ((sip_request_t *)msg_header_access((msg_pub_t*)(sip), sip_request_class))

/**Initializer for structure #sip_request_t.
 *
 * A static #sip_request_t structure for
 * @ref sip_request "request line" must be initialized with
 * the SIP_REQUEST_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_request_t sip_request = SIP_REQUEST_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_REQUEST_INIT() SIP_HDR_INIT(request)

/**Initialize a structure #sip_request_t.
 *
 * An #sip_request_t structure for
 * @ref sip_request "request line" can be initialized with the
 * sip_request_init() function/macro. For instance,
 * @code
 *
 *  sip_request_t sip_request;
 *
 *  sip_request_init(&sip_request);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_request_t *sip_request_init(sip_request_t x[1])
{
  return SIP_HEADER_INIT(x, sip_request_class, sizeof(sip_request_t));
}
#else
#define sip_request_init(x) \
  SIP_HEADER_INIT(x, sip_request_class, sizeof(sip_request_t))
#endif

/**Test if header object is instance of #sip_request_t.
 *
 * Check if the header class is an instance of
 * @ref sip_request "request line" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header request
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_request(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_request_hash;
}
#else
int sip_is_request(sip_header_t const *header);
#endif

#define sip_request_p(h) sip_is_request((h))


/**Duplicate a list of @ref sip_request "request line" header structures #sip_request_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   request = sip_request_dup(home, sip->sip_request);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_request_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_request_t *sip_request_dup(su_home_t *home, sip_request_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_request_t *sip_request_dup(su_home_t *home, sip_request_t const *hdr)
{
  return (sip_request_t *)
    msg_header_dup_as(home, sip_request_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_request "request line" header structures #sip_request_t.
 *
 * The function sip_request_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   request = sip_request_copy(home, sip->sip_request);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_request_t *sip_request_copy(su_home_t *home, sip_request_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_request_t *sip_request_copy(su_home_t *home, sip_request_t const *hdr)
{
  return (sip_request_t *)
    msg_header_copy_as(home, sip_request_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_request "request line" structure #sip_request_t.
 *
 * The function sip_request_make() makes a new
 * #sip_request_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_request_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_request_t *sip_request_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_request_t *sip_request_make(su_home_t *home, char const *s)
{
  return (sip_request_t *)sip_header_make(home, sip_request_class, s);
}
#endif

/**Make a @ref sip_request "request line" from formatting result.
 *
 * Make a new #sip_request_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_request_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_request_t *sip_request_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_request_t *sip_request_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_request_class, fmt, ap);
  va_end(ap);

  return (sip_request_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_status
 * @{
 */

/** Parse a SIP @ref sip_status "status line". @internal */
SOFIAPUBFUN issize_t sip_status_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_status "status line". @internal */
SOFIAPUBFUN issize_t sip_status_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_status "status line"
 * structure #sip_status_t from #sip_t.
 *
 */
#define sip_status(sip) \
  ((sip_status_t *)msg_header_access((msg_pub_t*)(sip), sip_status_class))

/**Initializer for structure #sip_status_t.
 *
 * A static #sip_status_t structure for
 * @ref sip_status "status line" must be initialized with
 * the SIP_STATUS_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_status_t sip_status = SIP_STATUS_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_STATUS_INIT() SIP_HDR_INIT(status)

/**Initialize a structure #sip_status_t.
 *
 * An #sip_status_t structure for
 * @ref sip_status "status line" can be initialized with the
 * sip_status_init() function/macro. For instance,
 * @code
 *
 *  sip_status_t sip_status;
 *
 *  sip_status_init(&sip_status);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_status_t *sip_status_init(sip_status_t x[1])
{
  return SIP_HEADER_INIT(x, sip_status_class, sizeof(sip_status_t));
}
#else
#define sip_status_init(x) \
  SIP_HEADER_INIT(x, sip_status_class, sizeof(sip_status_t))
#endif

/**Test if header object is instance of #sip_status_t.
 *
 * Check if the header class is an instance of
 * @ref sip_status "status line" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header status
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_status(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_status_hash;
}
#else
int sip_is_status(sip_header_t const *header);
#endif

#define sip_status_p(h) sip_is_status((h))


/**Duplicate a list of @ref sip_status "status line" header structures #sip_status_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   status = sip_status_dup(home, sip->sip_status);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_status_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_status_t *sip_status_dup(su_home_t *home, sip_status_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_status_t *sip_status_dup(su_home_t *home, sip_status_t const *hdr)
{
  return (sip_status_t *)
    msg_header_dup_as(home, sip_status_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_status "status line" header structures #sip_status_t.
 *
 * The function sip_status_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   status = sip_status_copy(home, sip->sip_status);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_status_t *sip_status_copy(su_home_t *home, sip_status_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_status_t *sip_status_copy(su_home_t *home, sip_status_t const *hdr)
{
  return (sip_status_t *)
    msg_header_copy_as(home, sip_status_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_status "status line" structure #sip_status_t.
 *
 * The function sip_status_make() makes a new
 * #sip_status_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_status_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_status_t *sip_status_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_status_t *sip_status_make(su_home_t *home, char const *s)
{
  return (sip_status_t *)sip_header_make(home, sip_status_class, s);
}
#endif

/**Make a @ref sip_status "status line" from formatting result.
 *
 * Make a new #sip_status_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_status_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_status_t *sip_status_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_status_t *sip_status_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_status_class, fmt, ap);
  va_end(ap);

  return (sip_status_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_via
 * @{
 */

/** Parse a SIP @ref sip_via "Via header". @internal */
SOFIAPUBFUN issize_t sip_via_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_via "Via header". @internal */
SOFIAPUBFUN issize_t sip_via_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_via "Via header"
 * structure #sip_via_t from #sip_t.
 *
 */
#define sip_via(sip) \
  ((sip_via_t *)msg_header_access((msg_pub_t*)(sip), sip_via_class))

/**Initializer for structure #sip_via_t.
 *
 * A static #sip_via_t structure for
 * @ref sip_via "Via header" must be initialized with
 * the SIP_VIA_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_via_t sip_via = SIP_VIA_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_VIA_INIT() SIP_HDR_INIT(via)

/**Initialize a structure #sip_via_t.
 *
 * An #sip_via_t structure for
 * @ref sip_via "Via header" can be initialized with the
 * sip_via_init() function/macro. For instance,
 * @code
 *
 *  sip_via_t sip_via;
 *
 *  sip_via_init(&sip_via);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_via_t *sip_via_init(sip_via_t x[1])
{
  return SIP_HEADER_INIT(x, sip_via_class, sizeof(sip_via_t));
}
#else
#define sip_via_init(x) \
  SIP_HEADER_INIT(x, sip_via_class, sizeof(sip_via_t))
#endif

/**Test if header object is instance of #sip_via_t.
 *
 * Check if the header class is an instance of
 * @ref sip_via "Via header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header via
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_via(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_via_hash;
}
#else
int sip_is_via(sip_header_t const *header);
#endif

#define sip_via_p(h) sip_is_via((h))


/**Duplicate a list of @ref sip_via "Via header" header structures #sip_via_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   via = sip_via_dup(home, sip->sip_via);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_via_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_via_t *sip_via_dup(su_home_t *home, sip_via_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_via_t *sip_via_dup(su_home_t *home, sip_via_t const *hdr)
{
  return (sip_via_t *)
    msg_header_dup_as(home, sip_via_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_via "Via header" header structures #sip_via_t.
 *
 * The function sip_via_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   via = sip_via_copy(home, sip->sip_via);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_via_t *sip_via_copy(su_home_t *home, sip_via_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_via_t *sip_via_copy(su_home_t *home, sip_via_t const *hdr)
{
  return (sip_via_t *)
    msg_header_copy_as(home, sip_via_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_via "Via header" structure #sip_via_t.
 *
 * The function sip_via_make() makes a new
 * #sip_via_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_via_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_via_t *sip_via_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_via_t *sip_via_make(su_home_t *home, char const *s)
{
  return (sip_via_t *)sip_header_make(home, sip_via_class, s);
}
#endif

/**Make a @ref sip_via "Via header" from formatting result.
 *
 * Make a new #sip_via_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_via_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_via_t *sip_via_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_via_t *sip_via_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_via_class, fmt, ap);
  va_end(ap);

  return (sip_via_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_route
 * @{
 */

/** Parse a SIP @ref sip_route "Route header". @internal */
SOFIAPUBFUN issize_t sip_route_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_route "Route header". @internal */
SOFIAPUBFUN issize_t sip_route_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_route "Route header"
 * structure #sip_route_t from #sip_t.
 *
 */
#define sip_route(sip) \
  ((sip_route_t *)msg_header_access((msg_pub_t*)(sip), sip_route_class))

/**Initializer for structure #sip_route_t.
 *
 * A static #sip_route_t structure for
 * @ref sip_route "Route header" must be initialized with
 * the SIP_ROUTE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_route_t sip_route = SIP_ROUTE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ROUTE_INIT() SIP_HDR_INIT(route)

/**Initialize a structure #sip_route_t.
 *
 * An #sip_route_t structure for
 * @ref sip_route "Route header" can be initialized with the
 * sip_route_init() function/macro. For instance,
 * @code
 *
 *  sip_route_t sip_route;
 *
 *  sip_route_init(&sip_route);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_route_t *sip_route_init(sip_route_t x[1])
{
  return SIP_HEADER_INIT(x, sip_route_class, sizeof(sip_route_t));
}
#else
#define sip_route_init(x) \
  SIP_HEADER_INIT(x, sip_route_class, sizeof(sip_route_t))
#endif

/**Test if header object is instance of #sip_route_t.
 *
 * Check if the header class is an instance of
 * @ref sip_route "Route header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header route
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_route(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_route_hash;
}
#else
int sip_is_route(sip_header_t const *header);
#endif

#define sip_route_p(h) sip_is_route((h))


/**Duplicate a list of @ref sip_route "Route header" header structures #sip_route_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   route = sip_route_dup(home, sip->sip_route);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_route_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_route_t *sip_route_dup(su_home_t *home, sip_route_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_route_t *sip_route_dup(su_home_t *home, sip_route_t const *hdr)
{
  return (sip_route_t *)
    msg_header_dup_as(home, sip_route_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_route "Route header" header structures #sip_route_t.
 *
 * The function sip_route_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   route = sip_route_copy(home, sip->sip_route);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_route_t *sip_route_copy(su_home_t *home, sip_route_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_route_t *sip_route_copy(su_home_t *home, sip_route_t const *hdr)
{
  return (sip_route_t *)
    msg_header_copy_as(home, sip_route_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_route "Route header" structure #sip_route_t.
 *
 * The function sip_route_make() makes a new
 * #sip_route_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_route_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_route_t *sip_route_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_route_t *sip_route_make(su_home_t *home, char const *s)
{
  return (sip_route_t *)sip_header_make(home, sip_route_class, s);
}
#endif

/**Make a @ref sip_route "Route header" from formatting result.
 *
 * Make a new #sip_route_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_route_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_route_t *sip_route_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_route_t *sip_route_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_route_class, fmt, ap);
  va_end(ap);

  return (sip_route_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_record_route
 * @{
 */

/** Parse a SIP @ref sip_record_route "Record-Route header". @internal */
SOFIAPUBFUN issize_t sip_record_route_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_record_route "Record-Route header". @internal */
SOFIAPUBFUN issize_t sip_record_route_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_record_route "Record-Route header"
 * structure #sip_record_route_t from #sip_t.
 *
 */
#define sip_record_route(sip) \
  ((sip_record_route_t *)msg_header_access((msg_pub_t*)(sip), sip_record_route_class))

/**Initializer for structure #sip_record_route_t.
 *
 * A static #sip_record_route_t structure for
 * @ref sip_record_route "Record-Route header" must be initialized with
 * the SIP_RECORD_ROUTE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_record_route_t sip_record_route = SIP_RECORD_ROUTE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_RECORD_ROUTE_INIT() SIP_HDR_INIT(record_route)

/**Initialize a structure #sip_record_route_t.
 *
 * An #sip_record_route_t structure for
 * @ref sip_record_route "Record-Route header" can be initialized with the
 * sip_record_route_init() function/macro. For instance,
 * @code
 *
 *  sip_record_route_t sip_record_route;
 *
 *  sip_record_route_init(&sip_record_route);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_record_route_t *sip_record_route_init(sip_record_route_t x[1])
{
  return SIP_HEADER_INIT(x, sip_record_route_class, sizeof(sip_record_route_t));
}
#else
#define sip_record_route_init(x) \
  SIP_HEADER_INIT(x, sip_record_route_class, sizeof(sip_record_route_t))
#endif

/**Test if header object is instance of #sip_record_route_t.
 *
 * Check if the header class is an instance of
 * @ref sip_record_route "Record-Route header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header record_route
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_record_route(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_record_route_hash;
}
#else
int sip_is_record_route(sip_header_t const *header);
#endif

#define sip_record_route_p(h) sip_is_record_route((h))


/**Duplicate a list of @ref sip_record_route "Record-Route header" header structures #sip_record_route_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   record_route = sip_record_route_dup(home, sip->sip_record_route);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_record_route_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_record_route_t *sip_record_route_dup(su_home_t *home, sip_record_route_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_record_route_t *sip_record_route_dup(su_home_t *home, sip_record_route_t const *hdr)
{
  return (sip_record_route_t *)
    msg_header_dup_as(home, sip_record_route_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_record_route "Record-Route header" header structures #sip_record_route_t.
 *
 * The function sip_record_route_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   record_route = sip_record_route_copy(home, sip->sip_record_route);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_record_route_t *sip_record_route_copy(su_home_t *home, sip_record_route_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_record_route_t *sip_record_route_copy(su_home_t *home, sip_record_route_t const *hdr)
{
  return (sip_record_route_t *)
    msg_header_copy_as(home, sip_record_route_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_record_route "Record-Route header" structure #sip_record_route_t.
 *
 * The function sip_record_route_make() makes a new
 * #sip_record_route_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_record_route_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_record_route_t *sip_record_route_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_record_route_t *sip_record_route_make(su_home_t *home, char const *s)
{
  return (sip_record_route_t *)sip_header_make(home, sip_record_route_class, s);
}
#endif

/**Make a @ref sip_record_route "Record-Route header" from formatting result.
 *
 * Make a new #sip_record_route_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_record_route_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_record_route_t *sip_record_route_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_record_route_t *sip_record_route_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_record_route_class, fmt, ap);
  va_end(ap);

  return (sip_record_route_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_max_forwards
 * @{
 */

/** Parse a SIP @ref sip_max_forwards "Max-Forwards header". @internal */
SOFIAPUBFUN issize_t sip_max_forwards_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_max_forwards "Max-Forwards header". @internal */
SOFIAPUBFUN issize_t sip_max_forwards_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_max_forwards "Max-Forwards header"
 * structure #sip_max_forwards_t from #sip_t.
 *
 */
#define sip_max_forwards(sip) \
  ((sip_max_forwards_t *)msg_header_access((msg_pub_t*)(sip), sip_max_forwards_class))

/**Initializer for structure #sip_max_forwards_t.
 *
 * A static #sip_max_forwards_t structure for
 * @ref sip_max_forwards "Max-Forwards header" must be initialized with
 * the SIP_MAX_FORWARDS_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_max_forwards_t sip_max_forwards = SIP_MAX_FORWARDS_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_MAX_FORWARDS_INIT() SIP_HDR_INIT(max_forwards)

/**Initialize a structure #sip_max_forwards_t.
 *
 * An #sip_max_forwards_t structure for
 * @ref sip_max_forwards "Max-Forwards header" can be initialized with the
 * sip_max_forwards_init() function/macro. For instance,
 * @code
 *
 *  sip_max_forwards_t sip_max_forwards;
 *
 *  sip_max_forwards_init(&sip_max_forwards);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_max_forwards_t *sip_max_forwards_init(sip_max_forwards_t x[1])
{
  return SIP_HEADER_INIT(x, sip_max_forwards_class, sizeof(sip_max_forwards_t));
}
#else
#define sip_max_forwards_init(x) \
  SIP_HEADER_INIT(x, sip_max_forwards_class, sizeof(sip_max_forwards_t))
#endif

/**Test if header object is instance of #sip_max_forwards_t.
 *
 * Check if the header class is an instance of
 * @ref sip_max_forwards "Max-Forwards header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header max_forwards
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_max_forwards(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_max_forwards_hash;
}
#else
int sip_is_max_forwards(sip_header_t const *header);
#endif

#define sip_max_forwards_p(h) sip_is_max_forwards((h))


/**Duplicate a list of @ref sip_max_forwards "Max-Forwards header" header structures #sip_max_forwards_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   max_forwards = sip_max_forwards_dup(home, sip->sip_max_forwards);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_max_forwards_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_max_forwards_t *sip_max_forwards_dup(su_home_t *home, sip_max_forwards_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_max_forwards_t *sip_max_forwards_dup(su_home_t *home, sip_max_forwards_t const *hdr)
{
  return (sip_max_forwards_t *)
    msg_header_dup_as(home, sip_max_forwards_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_max_forwards "Max-Forwards header" header structures #sip_max_forwards_t.
 *
 * The function sip_max_forwards_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   max_forwards = sip_max_forwards_copy(home, sip->sip_max_forwards);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_max_forwards_t *sip_max_forwards_copy(su_home_t *home, sip_max_forwards_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_max_forwards_t *sip_max_forwards_copy(su_home_t *home, sip_max_forwards_t const *hdr)
{
  return (sip_max_forwards_t *)
    msg_header_copy_as(home, sip_max_forwards_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_max_forwards "Max-Forwards header" structure #sip_max_forwards_t.
 *
 * The function sip_max_forwards_make() makes a new
 * #sip_max_forwards_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_max_forwards_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_max_forwards_t *sip_max_forwards_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_max_forwards_t *sip_max_forwards_make(su_home_t *home, char const *s)
{
  return (sip_max_forwards_t *)sip_header_make(home, sip_max_forwards_class, s);
}
#endif

/**Make a @ref sip_max_forwards "Max-Forwards header" from formatting result.
 *
 * Make a new #sip_max_forwards_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_max_forwards_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_max_forwards_t *sip_max_forwards_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_max_forwards_t *sip_max_forwards_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_max_forwards_class, fmt, ap);
  va_end(ap);

  return (sip_max_forwards_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_proxy_require
 * @{
 */

/** Parse a SIP @ref sip_proxy_require "Proxy-Require header". @internal */
SOFIAPUBFUN issize_t sip_proxy_require_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_proxy_require "Proxy-Require header". @internal */
SOFIAPUBFUN issize_t sip_proxy_require_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_proxy_require "Proxy-Require header"
 * structure #sip_proxy_require_t from #sip_t.
 *
 */
#define sip_proxy_require(sip) \
  ((sip_proxy_require_t *)msg_header_access((msg_pub_t*)(sip), sip_proxy_require_class))

/**Initializer for structure #sip_proxy_require_t.
 *
 * A static #sip_proxy_require_t structure for
 * @ref sip_proxy_require "Proxy-Require header" must be initialized with
 * the SIP_PROXY_REQUIRE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_proxy_require_t sip_proxy_require = SIP_PROXY_REQUIRE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_PROXY_REQUIRE_INIT() SIP_HDR_INIT(proxy_require)

/**Initialize a structure #sip_proxy_require_t.
 *
 * An #sip_proxy_require_t structure for
 * @ref sip_proxy_require "Proxy-Require header" can be initialized with the
 * sip_proxy_require_init() function/macro. For instance,
 * @code
 *
 *  sip_proxy_require_t sip_proxy_require;
 *
 *  sip_proxy_require_init(&sip_proxy_require);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_proxy_require_t *sip_proxy_require_init(sip_proxy_require_t x[1])
{
  return SIP_HEADER_INIT(x, sip_proxy_require_class, sizeof(sip_proxy_require_t));
}
#else
#define sip_proxy_require_init(x) \
  SIP_HEADER_INIT(x, sip_proxy_require_class, sizeof(sip_proxy_require_t))
#endif

/**Test if header object is instance of #sip_proxy_require_t.
 *
 * Check if the header class is an instance of
 * @ref sip_proxy_require "Proxy-Require header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header proxy_require
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_proxy_require(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_proxy_require_hash;
}
#else
int sip_is_proxy_require(sip_header_t const *header);
#endif

#define sip_proxy_require_p(h) sip_is_proxy_require((h))


/**Duplicate a list of @ref sip_proxy_require "Proxy-Require header" header structures #sip_proxy_require_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   proxy_require = sip_proxy_require_dup(home, sip->sip_proxy_require);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_proxy_require_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_require_t *sip_proxy_require_dup(su_home_t *home, sip_proxy_require_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_proxy_require_t *sip_proxy_require_dup(su_home_t *home, sip_proxy_require_t const *hdr)
{
  return (sip_proxy_require_t *)
    msg_header_dup_as(home, sip_proxy_require_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_proxy_require "Proxy-Require header" header structures #sip_proxy_require_t.
 *
 * The function sip_proxy_require_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   proxy_require = sip_proxy_require_copy(home, sip->sip_proxy_require);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_require_t *sip_proxy_require_copy(su_home_t *home, sip_proxy_require_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_proxy_require_t *sip_proxy_require_copy(su_home_t *home, sip_proxy_require_t const *hdr)
{
  return (sip_proxy_require_t *)
    msg_header_copy_as(home, sip_proxy_require_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_proxy_require "Proxy-Require header" structure #sip_proxy_require_t.
 *
 * The function sip_proxy_require_make() makes a new
 * #sip_proxy_require_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_proxy_require_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_require_t *sip_proxy_require_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_proxy_require_t *sip_proxy_require_make(su_home_t *home, char const *s)
{
  return (sip_proxy_require_t *)sip_header_make(home, sip_proxy_require_class, s);
}
#endif

/**Make a @ref sip_proxy_require "Proxy-Require header" from formatting result.
 *
 * Make a new #sip_proxy_require_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_proxy_require_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_require_t *sip_proxy_require_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_proxy_require_t *sip_proxy_require_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_proxy_require_class, fmt, ap);
  va_end(ap);

  return (sip_proxy_require_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_from
 * @{
 */

/** Parse a SIP @ref sip_from "From header". @internal */
SOFIAPUBFUN issize_t sip_from_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_from "From header". @internal */
SOFIAPUBFUN issize_t sip_from_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_from "From header"
 * structure #sip_from_t from #sip_t.
 *
 */
#define sip_from(sip) \
  ((sip_from_t *)msg_header_access((msg_pub_t*)(sip), sip_from_class))

/**Initializer for structure #sip_from_t.
 *
 * A static #sip_from_t structure for
 * @ref sip_from "From header" must be initialized with
 * the SIP_FROM_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_from_t sip_from = SIP_FROM_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_FROM_INIT() SIP_HDR_INIT(from)

/**Initialize a structure #sip_from_t.
 *
 * An #sip_from_t structure for
 * @ref sip_from "From header" can be initialized with the
 * sip_from_init() function/macro. For instance,
 * @code
 *
 *  sip_from_t sip_from;
 *
 *  sip_from_init(&sip_from);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_from_t *sip_from_init(sip_from_t x[1])
{
  return SIP_HEADER_INIT(x, sip_from_class, sizeof(sip_from_t));
}
#else
#define sip_from_init(x) \
  SIP_HEADER_INIT(x, sip_from_class, sizeof(sip_from_t))
#endif

/**Test if header object is instance of #sip_from_t.
 *
 * Check if the header class is an instance of
 * @ref sip_from "From header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header from
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_from(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_from_hash;
}
#else
int sip_is_from(sip_header_t const *header);
#endif

#define sip_from_p(h) sip_is_from((h))


/**Duplicate a list of @ref sip_from "From header" header structures #sip_from_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   from = sip_from_dup(home, sip->sip_from);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_from_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_from_t *sip_from_dup(su_home_t *home, sip_from_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_from_t *sip_from_dup(su_home_t *home, sip_from_t const *hdr)
{
  return (sip_from_t *)
    msg_header_dup_as(home, sip_from_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_from "From header" header structures #sip_from_t.
 *
 * The function sip_from_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   from = sip_from_copy(home, sip->sip_from);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_from_t *sip_from_copy(su_home_t *home, sip_from_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_from_t *sip_from_copy(su_home_t *home, sip_from_t const *hdr)
{
  return (sip_from_t *)
    msg_header_copy_as(home, sip_from_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_from "From header" structure #sip_from_t.
 *
 * The function sip_from_make() makes a new
 * #sip_from_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_from_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_from_t *sip_from_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_from_t *sip_from_make(su_home_t *home, char const *s)
{
  return (sip_from_t *)sip_header_make(home, sip_from_class, s);
}
#endif

/**Make a @ref sip_from "From header" from formatting result.
 *
 * Make a new #sip_from_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_from_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_from_t *sip_from_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_from_t *sip_from_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_from_class, fmt, ap);
  va_end(ap);

  return (sip_from_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_to
 * @{
 */

/** Parse a SIP @ref sip_to "To header". @internal */
SOFIAPUBFUN issize_t sip_to_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_to "To header". @internal */
SOFIAPUBFUN issize_t sip_to_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_to "To header"
 * structure #sip_to_t from #sip_t.
 *
 */
#define sip_to(sip) \
  ((sip_to_t *)msg_header_access((msg_pub_t*)(sip), sip_to_class))

/**Initializer for structure #sip_to_t.
 *
 * A static #sip_to_t structure for
 * @ref sip_to "To header" must be initialized with
 * the SIP_TO_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_to_t sip_to = SIP_TO_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_TO_INIT() SIP_HDR_INIT(to)

/**Initialize a structure #sip_to_t.
 *
 * An #sip_to_t structure for
 * @ref sip_to "To header" can be initialized with the
 * sip_to_init() function/macro. For instance,
 * @code
 *
 *  sip_to_t sip_to;
 *
 *  sip_to_init(&sip_to);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_to_t *sip_to_init(sip_to_t x[1])
{
  return SIP_HEADER_INIT(x, sip_to_class, sizeof(sip_to_t));
}
#else
#define sip_to_init(x) \
  SIP_HEADER_INIT(x, sip_to_class, sizeof(sip_to_t))
#endif

/**Test if header object is instance of #sip_to_t.
 *
 * Check if the header class is an instance of
 * @ref sip_to "To header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header to
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_to(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_to_hash;
}
#else
int sip_is_to(sip_header_t const *header);
#endif

#define sip_to_p(h) sip_is_to((h))


/**Duplicate a list of @ref sip_to "To header" header structures #sip_to_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   to = sip_to_dup(home, sip->sip_to);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_to_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_to_t *sip_to_dup(su_home_t *home, sip_to_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_to_t *sip_to_dup(su_home_t *home, sip_to_t const *hdr)
{
  return (sip_to_t *)
    msg_header_dup_as(home, sip_to_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_to "To header" header structures #sip_to_t.
 *
 * The function sip_to_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   to = sip_to_copy(home, sip->sip_to);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_to_t *sip_to_copy(su_home_t *home, sip_to_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_to_t *sip_to_copy(su_home_t *home, sip_to_t const *hdr)
{
  return (sip_to_t *)
    msg_header_copy_as(home, sip_to_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_to "To header" structure #sip_to_t.
 *
 * The function sip_to_make() makes a new
 * #sip_to_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_to_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_to_t *sip_to_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_to_t *sip_to_make(su_home_t *home, char const *s)
{
  return (sip_to_t *)sip_header_make(home, sip_to_class, s);
}
#endif

/**Make a @ref sip_to "To header" from formatting result.
 *
 * Make a new #sip_to_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_to_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_to_t *sip_to_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_to_t *sip_to_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_to_class, fmt, ap);
  va_end(ap);

  return (sip_to_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_call_id
 * @{
 */

/** Parse a SIP @ref sip_call_id "Call-ID header". @internal */
SOFIAPUBFUN issize_t sip_call_id_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_call_id "Call-ID header". @internal */
SOFIAPUBFUN issize_t sip_call_id_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_call_id "Call-ID header"
 * structure #sip_call_id_t from #sip_t.
 *
 */
#define sip_call_id(sip) \
  ((sip_call_id_t *)msg_header_access((msg_pub_t*)(sip), sip_call_id_class))

/**Initializer for structure #sip_call_id_t.
 *
 * A static #sip_call_id_t structure for
 * @ref sip_call_id "Call-ID header" must be initialized with
 * the SIP_CALL_ID_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_call_id_t sip_call_id = SIP_CALL_ID_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CALL_ID_INIT() SIP_HDR_INIT(call_id)

/**Initialize a structure #sip_call_id_t.
 *
 * An #sip_call_id_t structure for
 * @ref sip_call_id "Call-ID header" can be initialized with the
 * sip_call_id_init() function/macro. For instance,
 * @code
 *
 *  sip_call_id_t sip_call_id;
 *
 *  sip_call_id_init(&sip_call_id);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_call_id_t *sip_call_id_init(sip_call_id_t x[1])
{
  return SIP_HEADER_INIT(x, sip_call_id_class, sizeof(sip_call_id_t));
}
#else
#define sip_call_id_init(x) \
  SIP_HEADER_INIT(x, sip_call_id_class, sizeof(sip_call_id_t))
#endif

/**Test if header object is instance of #sip_call_id_t.
 *
 * Check if the header class is an instance of
 * @ref sip_call_id "Call-ID header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header call_id
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_call_id(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_call_id_hash;
}
#else
int sip_is_call_id(sip_header_t const *header);
#endif

#define sip_call_id_p(h) sip_is_call_id((h))


/**Duplicate a list of @ref sip_call_id "Call-ID header" header structures #sip_call_id_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   call_id = sip_call_id_dup(home, sip->sip_call_id);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_call_id_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_call_id_t *sip_call_id_dup(su_home_t *home, sip_call_id_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_call_id_t *sip_call_id_dup(su_home_t *home, sip_call_id_t const *hdr)
{
  return (sip_call_id_t *)
    msg_header_dup_as(home, sip_call_id_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_call_id "Call-ID header" header structures #sip_call_id_t.
 *
 * The function sip_call_id_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   call_id = sip_call_id_copy(home, sip->sip_call_id);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_call_id_t *sip_call_id_copy(su_home_t *home, sip_call_id_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_call_id_t *sip_call_id_copy(su_home_t *home, sip_call_id_t const *hdr)
{
  return (sip_call_id_t *)
    msg_header_copy_as(home, sip_call_id_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_call_id "Call-ID header" structure #sip_call_id_t.
 *
 * The function sip_call_id_make() makes a new
 * #sip_call_id_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_call_id_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_call_id_t *sip_call_id_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_call_id_t *sip_call_id_make(su_home_t *home, char const *s)
{
  return (sip_call_id_t *)sip_header_make(home, sip_call_id_class, s);
}
#endif

/**Make a @ref sip_call_id "Call-ID header" from formatting result.
 *
 * Make a new #sip_call_id_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_call_id_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_call_id_t *sip_call_id_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_call_id_t *sip_call_id_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_call_id_class, fmt, ap);
  va_end(ap);

  return (sip_call_id_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_cseq
 * @{
 */

/** Parse a SIP @ref sip_cseq "CSeq header". @internal */
SOFIAPUBFUN issize_t sip_cseq_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_cseq "CSeq header". @internal */
SOFIAPUBFUN issize_t sip_cseq_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_cseq "CSeq header"
 * structure #sip_cseq_t from #sip_t.
 *
 */
#define sip_cseq(sip) \
  ((sip_cseq_t *)msg_header_access((msg_pub_t*)(sip), sip_cseq_class))

/**Initializer for structure #sip_cseq_t.
 *
 * A static #sip_cseq_t structure for
 * @ref sip_cseq "CSeq header" must be initialized with
 * the SIP_CSEQ_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_cseq_t sip_cseq = SIP_CSEQ_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CSEQ_INIT() SIP_HDR_INIT(cseq)

/**Initialize a structure #sip_cseq_t.
 *
 * An #sip_cseq_t structure for
 * @ref sip_cseq "CSeq header" can be initialized with the
 * sip_cseq_init() function/macro. For instance,
 * @code
 *
 *  sip_cseq_t sip_cseq;
 *
 *  sip_cseq_init(&sip_cseq);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_cseq_t *sip_cseq_init(sip_cseq_t x[1])
{
  return SIP_HEADER_INIT(x, sip_cseq_class, sizeof(sip_cseq_t));
}
#else
#define sip_cseq_init(x) \
  SIP_HEADER_INIT(x, sip_cseq_class, sizeof(sip_cseq_t))
#endif

/**Test if header object is instance of #sip_cseq_t.
 *
 * Check if the header class is an instance of
 * @ref sip_cseq "CSeq header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header cseq
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_cseq(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_cseq_hash;
}
#else
int sip_is_cseq(sip_header_t const *header);
#endif

#define sip_cseq_p(h) sip_is_cseq((h))


/**Duplicate a list of @ref sip_cseq "CSeq header" header structures #sip_cseq_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   cseq = sip_cseq_dup(home, sip->sip_cseq);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_cseq_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_cseq_t *sip_cseq_dup(su_home_t *home, sip_cseq_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_cseq_t *sip_cseq_dup(su_home_t *home, sip_cseq_t const *hdr)
{
  return (sip_cseq_t *)
    msg_header_dup_as(home, sip_cseq_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_cseq "CSeq header" header structures #sip_cseq_t.
 *
 * The function sip_cseq_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   cseq = sip_cseq_copy(home, sip->sip_cseq);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_cseq_t *sip_cseq_copy(su_home_t *home, sip_cseq_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_cseq_t *sip_cseq_copy(su_home_t *home, sip_cseq_t const *hdr)
{
  return (sip_cseq_t *)
    msg_header_copy_as(home, sip_cseq_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_cseq "CSeq header" structure #sip_cseq_t.
 *
 * The function sip_cseq_make() makes a new
 * #sip_cseq_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_cseq_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_cseq_t *sip_cseq_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_cseq_t *sip_cseq_make(su_home_t *home, char const *s)
{
  return (sip_cseq_t *)sip_header_make(home, sip_cseq_class, s);
}
#endif

/**Make a @ref sip_cseq "CSeq header" from formatting result.
 *
 * Make a new #sip_cseq_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_cseq_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_cseq_t *sip_cseq_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_cseq_t *sip_cseq_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_cseq_class, fmt, ap);
  va_end(ap);

  return (sip_cseq_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_identity
 * @{
 */

/** Parse a SIP @ref sip_identity "Identity header". @internal */
SOFIAPUBFUN issize_t sip_identity_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_identity "Identity header". @internal */
SOFIAPUBFUN issize_t sip_identity_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_identity "Identity header"
 * structure #sip_identity_t from #sip_t.
 *
 */
#define sip_identity(sip) \
  ((sip_identity_t *)msg_header_access((msg_pub_t*)(sip), sip_identity_class))

/**Initializer for structure #sip_identity_t.
 *
 * A static #sip_identity_t structure for
 * @ref sip_identity "Identity header" must be initialized with
 * the SIP_IDENTITY_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_identity_t sip_identity = SIP_IDENTITY_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_IDENTITY_INIT() SIP_HDR_INIT(identity)

/**Initialize a structure #sip_identity_t.
 *
 * An #sip_identity_t structure for
 * @ref sip_identity "Identity header" can be initialized with the
 * sip_identity_init() function/macro. For instance,
 * @code
 *
 *  sip_identity_t sip_identity;
 *
 *  sip_identity_init(&sip_identity);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_identity_t *sip_identity_init(sip_identity_t x[1])
{
  return SIP_HEADER_INIT(x, sip_identity_class, sizeof(sip_identity_t));
}
#else
#define sip_identity_init(x) \
  SIP_HEADER_INIT(x, sip_identity_class, sizeof(sip_identity_t))
#endif

/**Test if header object is instance of #sip_identity_t.
 *
 * Check if the header class is an instance of
 * @ref sip_identity "Identity header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header identity
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_identity(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_identity_hash;
}
#else
int sip_is_identity(sip_header_t const *header);
#endif

#define sip_identity_p(h) sip_is_identity((h))


/**Duplicate a list of @ref sip_identity "Identity header" header structures #sip_identity_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   identity = sip_identity_dup(home, sip->sip_identity);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_identity_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_identity_t *sip_identity_dup(su_home_t *home, sip_identity_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_identity_t *sip_identity_dup(su_home_t *home, sip_identity_t const *hdr)
{
  return (sip_identity_t *)
    msg_header_dup_as(home, sip_identity_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_identity "Identity header" header structures #sip_identity_t.
 *
 * The function sip_identity_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   identity = sip_identity_copy(home, sip->sip_identity);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_identity_t *sip_identity_copy(su_home_t *home, sip_identity_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_identity_t *sip_identity_copy(su_home_t *home, sip_identity_t const *hdr)
{
  return (sip_identity_t *)
    msg_header_copy_as(home, sip_identity_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_identity "Identity header" structure #sip_identity_t.
 *
 * The function sip_identity_make() makes a new
 * #sip_identity_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_identity_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_identity_t *sip_identity_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_identity_t *sip_identity_make(su_home_t *home, char const *s)
{
  return (sip_identity_t *)sip_header_make(home, sip_identity_class, s);
}
#endif

/**Make a @ref sip_identity "Identity header" from formatting result.
 *
 * Make a new #sip_identity_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_identity_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_identity_t *sip_identity_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_identity_t *sip_identity_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_identity_class, fmt, ap);
  va_end(ap);

  return (sip_identity_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_contact
 * @{
 */

/** Parse a SIP @ref sip_contact "Contact header". @internal */
SOFIAPUBFUN issize_t sip_contact_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_contact "Contact header". @internal */
SOFIAPUBFUN issize_t sip_contact_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_contact "Contact header"
 * structure #sip_contact_t from #sip_t.
 *
 */
#define sip_contact(sip) \
  ((sip_contact_t *)msg_header_access((msg_pub_t*)(sip), sip_contact_class))

/**Initializer for structure #sip_contact_t.
 *
 * A static #sip_contact_t structure for
 * @ref sip_contact "Contact header" must be initialized with
 * the SIP_CONTACT_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_contact_t sip_contact = SIP_CONTACT_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CONTACT_INIT() SIP_HDR_INIT(contact)

/**Initialize a structure #sip_contact_t.
 *
 * An #sip_contact_t structure for
 * @ref sip_contact "Contact header" can be initialized with the
 * sip_contact_init() function/macro. For instance,
 * @code
 *
 *  sip_contact_t sip_contact;
 *
 *  sip_contact_init(&sip_contact);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_contact_t *sip_contact_init(sip_contact_t x[1])
{
  return SIP_HEADER_INIT(x, sip_contact_class, sizeof(sip_contact_t));
}
#else
#define sip_contact_init(x) \
  SIP_HEADER_INIT(x, sip_contact_class, sizeof(sip_contact_t))
#endif

/**Test if header object is instance of #sip_contact_t.
 *
 * Check if the header class is an instance of
 * @ref sip_contact "Contact header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header contact
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_contact(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_contact_hash;
}
#else
int sip_is_contact(sip_header_t const *header);
#endif

#define sip_contact_p(h) sip_is_contact((h))


/**Duplicate a list of @ref sip_contact "Contact header" header structures #sip_contact_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   contact = sip_contact_dup(home, sip->sip_contact);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_contact_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_contact_t *sip_contact_dup(su_home_t *home, sip_contact_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_contact_t *sip_contact_dup(su_home_t *home, sip_contact_t const *hdr)
{
  return (sip_contact_t *)
    msg_header_dup_as(home, sip_contact_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_contact "Contact header" header structures #sip_contact_t.
 *
 * The function sip_contact_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   contact = sip_contact_copy(home, sip->sip_contact);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_contact_t *sip_contact_copy(su_home_t *home, sip_contact_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_contact_t *sip_contact_copy(su_home_t *home, sip_contact_t const *hdr)
{
  return (sip_contact_t *)
    msg_header_copy_as(home, sip_contact_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_contact "Contact header" structure #sip_contact_t.
 *
 * The function sip_contact_make() makes a new
 * #sip_contact_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_contact_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_contact_t *sip_contact_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_contact_t *sip_contact_make(su_home_t *home, char const *s)
{
  return (sip_contact_t *)sip_header_make(home, sip_contact_class, s);
}
#endif

/**Make a @ref sip_contact "Contact header" from formatting result.
 *
 * Make a new #sip_contact_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_contact_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_contact_t *sip_contact_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_contact_t *sip_contact_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_contact_class, fmt, ap);
  va_end(ap);

  return (sip_contact_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_rseq
 * @{
 */

/** Parse a SIP @ref sip_rseq "RSeq header". @internal */
SOFIAPUBFUN issize_t sip_rseq_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_rseq "RSeq header". @internal */
SOFIAPUBFUN issize_t sip_rseq_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_rseq "RSeq header"
 * structure #sip_rseq_t from #sip_t.
 *
 */
#define sip_rseq(sip) \
  ((sip_rseq_t *)msg_header_access((msg_pub_t*)(sip), sip_rseq_class))

/**Initializer for structure #sip_rseq_t.
 *
 * A static #sip_rseq_t structure for
 * @ref sip_rseq "RSeq header" must be initialized with
 * the SIP_RSEQ_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_rseq_t sip_rseq = SIP_RSEQ_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_RSEQ_INIT() SIP_HDR_INIT(rseq)

/**Initialize a structure #sip_rseq_t.
 *
 * An #sip_rseq_t structure for
 * @ref sip_rseq "RSeq header" can be initialized with the
 * sip_rseq_init() function/macro. For instance,
 * @code
 *
 *  sip_rseq_t sip_rseq;
 *
 *  sip_rseq_init(&sip_rseq);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_rseq_t *sip_rseq_init(sip_rseq_t x[1])
{
  return SIP_HEADER_INIT(x, sip_rseq_class, sizeof(sip_rseq_t));
}
#else
#define sip_rseq_init(x) \
  SIP_HEADER_INIT(x, sip_rseq_class, sizeof(sip_rseq_t))
#endif

/**Test if header object is instance of #sip_rseq_t.
 *
 * Check if the header class is an instance of
 * @ref sip_rseq "RSeq header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header rseq
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_rseq(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_rseq_hash;
}
#else
int sip_is_rseq(sip_header_t const *header);
#endif

#define sip_rseq_p(h) sip_is_rseq((h))


/**Duplicate a list of @ref sip_rseq "RSeq header" header structures #sip_rseq_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   rseq = sip_rseq_dup(home, sip->sip_rseq);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_rseq_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_rseq_t *sip_rseq_dup(su_home_t *home, sip_rseq_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_rseq_t *sip_rseq_dup(su_home_t *home, sip_rseq_t const *hdr)
{
  return (sip_rseq_t *)
    msg_header_dup_as(home, sip_rseq_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_rseq "RSeq header" header structures #sip_rseq_t.
 *
 * The function sip_rseq_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   rseq = sip_rseq_copy(home, sip->sip_rseq);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_rseq_t *sip_rseq_copy(su_home_t *home, sip_rseq_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_rseq_t *sip_rseq_copy(su_home_t *home, sip_rseq_t const *hdr)
{
  return (sip_rseq_t *)
    msg_header_copy_as(home, sip_rseq_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_rseq "RSeq header" structure #sip_rseq_t.
 *
 * The function sip_rseq_make() makes a new
 * #sip_rseq_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_rseq_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_rseq_t *sip_rseq_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_rseq_t *sip_rseq_make(su_home_t *home, char const *s)
{
  return (sip_rseq_t *)sip_header_make(home, sip_rseq_class, s);
}
#endif

/**Make a @ref sip_rseq "RSeq header" from formatting result.
 *
 * Make a new #sip_rseq_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_rseq_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_rseq_t *sip_rseq_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_rseq_t *sip_rseq_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_rseq_class, fmt, ap);
  va_end(ap);

  return (sip_rseq_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_rack
 * @{
 */

/** Parse a SIP @ref sip_rack "RAck header". @internal */
SOFIAPUBFUN issize_t sip_rack_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_rack "RAck header". @internal */
SOFIAPUBFUN issize_t sip_rack_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_rack "RAck header"
 * structure #sip_rack_t from #sip_t.
 *
 */
#define sip_rack(sip) \
  ((sip_rack_t *)msg_header_access((msg_pub_t*)(sip), sip_rack_class))

/**Initializer for structure #sip_rack_t.
 *
 * A static #sip_rack_t structure for
 * @ref sip_rack "RAck header" must be initialized with
 * the SIP_RACK_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_rack_t sip_rack = SIP_RACK_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_RACK_INIT() SIP_HDR_INIT(rack)

/**Initialize a structure #sip_rack_t.
 *
 * An #sip_rack_t structure for
 * @ref sip_rack "RAck header" can be initialized with the
 * sip_rack_init() function/macro. For instance,
 * @code
 *
 *  sip_rack_t sip_rack;
 *
 *  sip_rack_init(&sip_rack);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_rack_t *sip_rack_init(sip_rack_t x[1])
{
  return SIP_HEADER_INIT(x, sip_rack_class, sizeof(sip_rack_t));
}
#else
#define sip_rack_init(x) \
  SIP_HEADER_INIT(x, sip_rack_class, sizeof(sip_rack_t))
#endif

/**Test if header object is instance of #sip_rack_t.
 *
 * Check if the header class is an instance of
 * @ref sip_rack "RAck header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header rack
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_rack(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_rack_hash;
}
#else
int sip_is_rack(sip_header_t const *header);
#endif

#define sip_rack_p(h) sip_is_rack((h))


/**Duplicate a list of @ref sip_rack "RAck header" header structures #sip_rack_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   rack = sip_rack_dup(home, sip->sip_rack);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_rack_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_rack_t *sip_rack_dup(su_home_t *home, sip_rack_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_rack_t *sip_rack_dup(su_home_t *home, sip_rack_t const *hdr)
{
  return (sip_rack_t *)
    msg_header_dup_as(home, sip_rack_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_rack "RAck header" header structures #sip_rack_t.
 *
 * The function sip_rack_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   rack = sip_rack_copy(home, sip->sip_rack);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_rack_t *sip_rack_copy(su_home_t *home, sip_rack_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_rack_t *sip_rack_copy(su_home_t *home, sip_rack_t const *hdr)
{
  return (sip_rack_t *)
    msg_header_copy_as(home, sip_rack_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_rack "RAck header" structure #sip_rack_t.
 *
 * The function sip_rack_make() makes a new
 * #sip_rack_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_rack_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_rack_t *sip_rack_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_rack_t *sip_rack_make(su_home_t *home, char const *s)
{
  return (sip_rack_t *)sip_header_make(home, sip_rack_class, s);
}
#endif

/**Make a @ref sip_rack "RAck header" from formatting result.
 *
 * Make a new #sip_rack_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_rack_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_rack_t *sip_rack_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_rack_t *sip_rack_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_rack_class, fmt, ap);
  va_end(ap);

  return (sip_rack_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_request_disposition
 * @{
 */

/** Parse a SIP @ref sip_request_disposition "Request-Disposition header". @internal */
SOFIAPUBFUN issize_t sip_request_disposition_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_request_disposition "Request-Disposition header". @internal */
SOFIAPUBFUN issize_t sip_request_disposition_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_request_disposition "Request-Disposition header"
 * structure #sip_request_disposition_t from #sip_t.
 *
 */
#define sip_request_disposition(sip) \
  ((sip_request_disposition_t *)msg_header_access((msg_pub_t*)(sip), sip_request_disposition_class))

/**Initializer for structure #sip_request_disposition_t.
 *
 * A static #sip_request_disposition_t structure for
 * @ref sip_request_disposition "Request-Disposition header" must be initialized with
 * the SIP_REQUEST_DISPOSITION_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_request_disposition_t sip_request_disposition = SIP_REQUEST_DISPOSITION_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_REQUEST_DISPOSITION_INIT() SIP_HDR_INIT(request_disposition)

/**Initialize a structure #sip_request_disposition_t.
 *
 * An #sip_request_disposition_t structure for
 * @ref sip_request_disposition "Request-Disposition header" can be initialized with the
 * sip_request_disposition_init() function/macro. For instance,
 * @code
 *
 *  sip_request_disposition_t sip_request_disposition;
 *
 *  sip_request_disposition_init(&sip_request_disposition);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_request_disposition_t *sip_request_disposition_init(sip_request_disposition_t x[1])
{
  return SIP_HEADER_INIT(x, sip_request_disposition_class, sizeof(sip_request_disposition_t));
}
#else
#define sip_request_disposition_init(x) \
  SIP_HEADER_INIT(x, sip_request_disposition_class, sizeof(sip_request_disposition_t))
#endif

/**Test if header object is instance of #sip_request_disposition_t.
 *
 * Check if the header class is an instance of
 * @ref sip_request_disposition "Request-Disposition header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header request_disposition
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_request_disposition(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_request_disposition_hash;
}
#else
int sip_is_request_disposition(sip_header_t const *header);
#endif

#define sip_request_disposition_p(h) sip_is_request_disposition((h))


/**Duplicate a list of @ref sip_request_disposition "Request-Disposition header" header structures #sip_request_disposition_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   request_disposition = sip_request_disposition_dup(home, sip->sip_request_disposition);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_request_disposition_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_request_disposition_t *sip_request_disposition_dup(su_home_t *home, sip_request_disposition_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_request_disposition_t *sip_request_disposition_dup(su_home_t *home, sip_request_disposition_t const *hdr)
{
  return (sip_request_disposition_t *)
    msg_header_dup_as(home, sip_request_disposition_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_request_disposition "Request-Disposition header" header structures #sip_request_disposition_t.
 *
 * The function sip_request_disposition_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   request_disposition = sip_request_disposition_copy(home, sip->sip_request_disposition);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_request_disposition_t *sip_request_disposition_copy(su_home_t *home, sip_request_disposition_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_request_disposition_t *sip_request_disposition_copy(su_home_t *home, sip_request_disposition_t const *hdr)
{
  return (sip_request_disposition_t *)
    msg_header_copy_as(home, sip_request_disposition_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_request_disposition "Request-Disposition header" structure #sip_request_disposition_t.
 *
 * The function sip_request_disposition_make() makes a new
 * #sip_request_disposition_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_request_disposition_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_request_disposition_t *sip_request_disposition_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_request_disposition_t *sip_request_disposition_make(su_home_t *home, char const *s)
{
  return (sip_request_disposition_t *)sip_header_make(home, sip_request_disposition_class, s);
}
#endif

/**Make a @ref sip_request_disposition "Request-Disposition header" from formatting result.
 *
 * Make a new #sip_request_disposition_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_request_disposition_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_request_disposition_t *sip_request_disposition_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_request_disposition_t *sip_request_disposition_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_request_disposition_class, fmt, ap);
  va_end(ap);

  return (sip_request_disposition_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_accept_contact
 * @{
 */

/** Parse a SIP @ref sip_accept_contact "Accept-Contact header". @internal */
SOFIAPUBFUN issize_t sip_accept_contact_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_accept_contact "Accept-Contact header". @internal */
SOFIAPUBFUN issize_t sip_accept_contact_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_accept_contact "Accept-Contact header"
 * structure #sip_accept_contact_t from #sip_t.
 *
 */
#define sip_accept_contact(sip) \
  ((sip_accept_contact_t *)msg_header_access((msg_pub_t*)(sip), sip_accept_contact_class))

/**Initializer for structure #sip_accept_contact_t.
 *
 * A static #sip_accept_contact_t structure for
 * @ref sip_accept_contact "Accept-Contact header" must be initialized with
 * the SIP_ACCEPT_CONTACT_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_accept_contact_t sip_accept_contact = SIP_ACCEPT_CONTACT_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ACCEPT_CONTACT_INIT() SIP_HDR_INIT(accept_contact)

/**Initialize a structure #sip_accept_contact_t.
 *
 * An #sip_accept_contact_t structure for
 * @ref sip_accept_contact "Accept-Contact header" can be initialized with the
 * sip_accept_contact_init() function/macro. For instance,
 * @code
 *
 *  sip_accept_contact_t sip_accept_contact;
 *
 *  sip_accept_contact_init(&sip_accept_contact);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_accept_contact_t *sip_accept_contact_init(sip_accept_contact_t x[1])
{
  return SIP_HEADER_INIT(x, sip_accept_contact_class, sizeof(sip_accept_contact_t));
}
#else
#define sip_accept_contact_init(x) \
  SIP_HEADER_INIT(x, sip_accept_contact_class, sizeof(sip_accept_contact_t))
#endif

/**Test if header object is instance of #sip_accept_contact_t.
 *
 * Check if the header class is an instance of
 * @ref sip_accept_contact "Accept-Contact header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header accept_contact
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_accept_contact(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_accept_contact_hash;
}
#else
int sip_is_accept_contact(sip_header_t const *header);
#endif

#define sip_accept_contact_p(h) sip_is_accept_contact((h))


/**Duplicate a list of @ref sip_accept_contact "Accept-Contact header" header structures #sip_accept_contact_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   accept_contact = sip_accept_contact_dup(home, sip->sip_accept_contact);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_accept_contact_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_contact_t *sip_accept_contact_dup(su_home_t *home, sip_accept_contact_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_accept_contact_t *sip_accept_contact_dup(su_home_t *home, sip_accept_contact_t const *hdr)
{
  return (sip_accept_contact_t *)
    msg_header_dup_as(home, sip_accept_contact_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_accept_contact "Accept-Contact header" header structures #sip_accept_contact_t.
 *
 * The function sip_accept_contact_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   accept_contact = sip_accept_contact_copy(home, sip->sip_accept_contact);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_contact_t *sip_accept_contact_copy(su_home_t *home, sip_accept_contact_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_accept_contact_t *sip_accept_contact_copy(su_home_t *home, sip_accept_contact_t const *hdr)
{
  return (sip_accept_contact_t *)
    msg_header_copy_as(home, sip_accept_contact_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_accept_contact "Accept-Contact header" structure #sip_accept_contact_t.
 *
 * The function sip_accept_contact_make() makes a new
 * #sip_accept_contact_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_accept_contact_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_contact_t *sip_accept_contact_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_accept_contact_t *sip_accept_contact_make(su_home_t *home, char const *s)
{
  return (sip_accept_contact_t *)sip_header_make(home, sip_accept_contact_class, s);
}
#endif

/**Make a @ref sip_accept_contact "Accept-Contact header" from formatting result.
 *
 * Make a new #sip_accept_contact_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_accept_contact_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_contact_t *sip_accept_contact_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_accept_contact_t *sip_accept_contact_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_accept_contact_class, fmt, ap);
  va_end(ap);

  return (sip_accept_contact_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_reject_contact
 * @{
 */

/** Parse a SIP @ref sip_reject_contact "Reject-Contact header". @internal */
SOFIAPUBFUN issize_t sip_reject_contact_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_reject_contact "Reject-Contact header". @internal */
SOFIAPUBFUN issize_t sip_reject_contact_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_reject_contact "Reject-Contact header"
 * structure #sip_reject_contact_t from #sip_t.
 *
 */
#define sip_reject_contact(sip) \
  ((sip_reject_contact_t *)msg_header_access((msg_pub_t*)(sip), sip_reject_contact_class))

/**Initializer for structure #sip_reject_contact_t.
 *
 * A static #sip_reject_contact_t structure for
 * @ref sip_reject_contact "Reject-Contact header" must be initialized with
 * the SIP_REJECT_CONTACT_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_reject_contact_t sip_reject_contact = SIP_REJECT_CONTACT_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_REJECT_CONTACT_INIT() SIP_HDR_INIT(reject_contact)

/**Initialize a structure #sip_reject_contact_t.
 *
 * An #sip_reject_contact_t structure for
 * @ref sip_reject_contact "Reject-Contact header" can be initialized with the
 * sip_reject_contact_init() function/macro. For instance,
 * @code
 *
 *  sip_reject_contact_t sip_reject_contact;
 *
 *  sip_reject_contact_init(&sip_reject_contact);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_reject_contact_t *sip_reject_contact_init(sip_reject_contact_t x[1])
{
  return SIP_HEADER_INIT(x, sip_reject_contact_class, sizeof(sip_reject_contact_t));
}
#else
#define sip_reject_contact_init(x) \
  SIP_HEADER_INIT(x, sip_reject_contact_class, sizeof(sip_reject_contact_t))
#endif

/**Test if header object is instance of #sip_reject_contact_t.
 *
 * Check if the header class is an instance of
 * @ref sip_reject_contact "Reject-Contact header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header reject_contact
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_reject_contact(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_reject_contact_hash;
}
#else
int sip_is_reject_contact(sip_header_t const *header);
#endif

#define sip_reject_contact_p(h) sip_is_reject_contact((h))


/**Duplicate a list of @ref sip_reject_contact "Reject-Contact header" header structures #sip_reject_contact_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   reject_contact = sip_reject_contact_dup(home, sip->sip_reject_contact);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_reject_contact_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_reject_contact_t *sip_reject_contact_dup(su_home_t *home, sip_reject_contact_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_reject_contact_t *sip_reject_contact_dup(su_home_t *home, sip_reject_contact_t const *hdr)
{
  return (sip_reject_contact_t *)
    msg_header_dup_as(home, sip_reject_contact_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_reject_contact "Reject-Contact header" header structures #sip_reject_contact_t.
 *
 * The function sip_reject_contact_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   reject_contact = sip_reject_contact_copy(home, sip->sip_reject_contact);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_reject_contact_t *sip_reject_contact_copy(su_home_t *home, sip_reject_contact_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_reject_contact_t *sip_reject_contact_copy(su_home_t *home, sip_reject_contact_t const *hdr)
{
  return (sip_reject_contact_t *)
    msg_header_copy_as(home, sip_reject_contact_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_reject_contact "Reject-Contact header" structure #sip_reject_contact_t.
 *
 * The function sip_reject_contact_make() makes a new
 * #sip_reject_contact_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_reject_contact_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_reject_contact_t *sip_reject_contact_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_reject_contact_t *sip_reject_contact_make(su_home_t *home, char const *s)
{
  return (sip_reject_contact_t *)sip_header_make(home, sip_reject_contact_class, s);
}
#endif

/**Make a @ref sip_reject_contact "Reject-Contact header" from formatting result.
 *
 * Make a new #sip_reject_contact_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_reject_contact_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_reject_contact_t *sip_reject_contact_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_reject_contact_t *sip_reject_contact_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_reject_contact_class, fmt, ap);
  va_end(ap);

  return (sip_reject_contact_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_expires
 * @{
 */

/** Parse a SIP @ref sip_expires "Expires header". @internal */
SOFIAPUBFUN issize_t sip_expires_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_expires "Expires header". @internal */
SOFIAPUBFUN issize_t sip_expires_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_expires "Expires header"
 * structure #sip_expires_t from #sip_t.
 *
 */
#define sip_expires(sip) \
  ((sip_expires_t *)msg_header_access((msg_pub_t*)(sip), sip_expires_class))

/**Initializer for structure #sip_expires_t.
 *
 * A static #sip_expires_t structure for
 * @ref sip_expires "Expires header" must be initialized with
 * the SIP_EXPIRES_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_expires_t sip_expires = SIP_EXPIRES_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_EXPIRES_INIT() SIP_HDR_INIT(expires)

/**Initialize a structure #sip_expires_t.
 *
 * An #sip_expires_t structure for
 * @ref sip_expires "Expires header" can be initialized with the
 * sip_expires_init() function/macro. For instance,
 * @code
 *
 *  sip_expires_t sip_expires;
 *
 *  sip_expires_init(&sip_expires);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_expires_t *sip_expires_init(sip_expires_t x[1])
{
  return SIP_HEADER_INIT(x, sip_expires_class, sizeof(sip_expires_t));
}
#else
#define sip_expires_init(x) \
  SIP_HEADER_INIT(x, sip_expires_class, sizeof(sip_expires_t))
#endif

/**Test if header object is instance of #sip_expires_t.
 *
 * Check if the header class is an instance of
 * @ref sip_expires "Expires header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header expires
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_expires(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_expires_hash;
}
#else
int sip_is_expires(sip_header_t const *header);
#endif

#define sip_expires_p(h) sip_is_expires((h))


/**Duplicate a list of @ref sip_expires "Expires header" header structures #sip_expires_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   expires = sip_expires_dup(home, sip->sip_expires);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_expires_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_expires_t *sip_expires_dup(su_home_t *home, sip_expires_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_expires_t *sip_expires_dup(su_home_t *home, sip_expires_t const *hdr)
{
  return (sip_expires_t *)
    msg_header_dup_as(home, sip_expires_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_expires "Expires header" header structures #sip_expires_t.
 *
 * The function sip_expires_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   expires = sip_expires_copy(home, sip->sip_expires);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_expires_t *sip_expires_copy(su_home_t *home, sip_expires_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_expires_t *sip_expires_copy(su_home_t *home, sip_expires_t const *hdr)
{
  return (sip_expires_t *)
    msg_header_copy_as(home, sip_expires_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_expires "Expires header" structure #sip_expires_t.
 *
 * The function sip_expires_make() makes a new
 * #sip_expires_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_expires_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_expires_t *sip_expires_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_expires_t *sip_expires_make(su_home_t *home, char const *s)
{
  return (sip_expires_t *)sip_header_make(home, sip_expires_class, s);
}
#endif

/**Make a @ref sip_expires "Expires header" from formatting result.
 *
 * Make a new #sip_expires_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_expires_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_expires_t *sip_expires_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_expires_t *sip_expires_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_expires_class, fmt, ap);
  va_end(ap);

  return (sip_expires_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_date
 * @{
 */

/** Parse a SIP @ref sip_date "Date header". @internal */
SOFIAPUBFUN issize_t sip_date_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_date "Date header". @internal */
SOFIAPUBFUN issize_t sip_date_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_date "Date header"
 * structure #sip_date_t from #sip_t.
 *
 */
#define sip_date(sip) \
  ((sip_date_t *)msg_header_access((msg_pub_t*)(sip), sip_date_class))

/**Initializer for structure #sip_date_t.
 *
 * A static #sip_date_t structure for
 * @ref sip_date "Date header" must be initialized with
 * the SIP_DATE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_date_t sip_date = SIP_DATE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_DATE_INIT() SIP_HDR_INIT(date)

/**Initialize a structure #sip_date_t.
 *
 * An #sip_date_t structure for
 * @ref sip_date "Date header" can be initialized with the
 * sip_date_init() function/macro. For instance,
 * @code
 *
 *  sip_date_t sip_date;
 *
 *  sip_date_init(&sip_date);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_date_t *sip_date_init(sip_date_t x[1])
{
  return SIP_HEADER_INIT(x, sip_date_class, sizeof(sip_date_t));
}
#else
#define sip_date_init(x) \
  SIP_HEADER_INIT(x, sip_date_class, sizeof(sip_date_t))
#endif

/**Test if header object is instance of #sip_date_t.
 *
 * Check if the header class is an instance of
 * @ref sip_date "Date header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header date
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_date(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_date_hash;
}
#else
int sip_is_date(sip_header_t const *header);
#endif

#define sip_date_p(h) sip_is_date((h))


/**Duplicate a list of @ref sip_date "Date header" header structures #sip_date_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   date = sip_date_dup(home, sip->sip_date);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_date_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_date_t *sip_date_dup(su_home_t *home, sip_date_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_date_t *sip_date_dup(su_home_t *home, sip_date_t const *hdr)
{
  return (sip_date_t *)
    msg_header_dup_as(home, sip_date_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_date "Date header" header structures #sip_date_t.
 *
 * The function sip_date_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   date = sip_date_copy(home, sip->sip_date);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_date_t *sip_date_copy(su_home_t *home, sip_date_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_date_t *sip_date_copy(su_home_t *home, sip_date_t const *hdr)
{
  return (sip_date_t *)
    msg_header_copy_as(home, sip_date_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_date "Date header" structure #sip_date_t.
 *
 * The function sip_date_make() makes a new
 * #sip_date_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_date_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_date_t *sip_date_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_date_t *sip_date_make(su_home_t *home, char const *s)
{
  return (sip_date_t *)sip_header_make(home, sip_date_class, s);
}
#endif

/**Make a @ref sip_date "Date header" from formatting result.
 *
 * Make a new #sip_date_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_date_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_date_t *sip_date_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_date_t *sip_date_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_date_class, fmt, ap);
  va_end(ap);

  return (sip_date_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_retry_after
 * @{
 */

/** Parse a SIP @ref sip_retry_after "Retry-After header". @internal */
SOFIAPUBFUN issize_t sip_retry_after_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_retry_after "Retry-After header". @internal */
SOFIAPUBFUN issize_t sip_retry_after_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_retry_after "Retry-After header"
 * structure #sip_retry_after_t from #sip_t.
 *
 */
#define sip_retry_after(sip) \
  ((sip_retry_after_t *)msg_header_access((msg_pub_t*)(sip), sip_retry_after_class))

/**Initializer for structure #sip_retry_after_t.
 *
 * A static #sip_retry_after_t structure for
 * @ref sip_retry_after "Retry-After header" must be initialized with
 * the SIP_RETRY_AFTER_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_retry_after_t sip_retry_after = SIP_RETRY_AFTER_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_RETRY_AFTER_INIT() SIP_HDR_INIT(retry_after)

/**Initialize a structure #sip_retry_after_t.
 *
 * An #sip_retry_after_t structure for
 * @ref sip_retry_after "Retry-After header" can be initialized with the
 * sip_retry_after_init() function/macro. For instance,
 * @code
 *
 *  sip_retry_after_t sip_retry_after;
 *
 *  sip_retry_after_init(&sip_retry_after);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_retry_after_t *sip_retry_after_init(sip_retry_after_t x[1])
{
  return SIP_HEADER_INIT(x, sip_retry_after_class, sizeof(sip_retry_after_t));
}
#else
#define sip_retry_after_init(x) \
  SIP_HEADER_INIT(x, sip_retry_after_class, sizeof(sip_retry_after_t))
#endif

/**Test if header object is instance of #sip_retry_after_t.
 *
 * Check if the header class is an instance of
 * @ref sip_retry_after "Retry-After header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header retry_after
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_retry_after(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_retry_after_hash;
}
#else
int sip_is_retry_after(sip_header_t const *header);
#endif

#define sip_retry_after_p(h) sip_is_retry_after((h))


/**Duplicate a list of @ref sip_retry_after "Retry-After header" header structures #sip_retry_after_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   retry_after = sip_retry_after_dup(home, sip->sip_retry_after);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_retry_after_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_retry_after_t *sip_retry_after_dup(su_home_t *home, sip_retry_after_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_retry_after_t *sip_retry_after_dup(su_home_t *home, sip_retry_after_t const *hdr)
{
  return (sip_retry_after_t *)
    msg_header_dup_as(home, sip_retry_after_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_retry_after "Retry-After header" header structures #sip_retry_after_t.
 *
 * The function sip_retry_after_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   retry_after = sip_retry_after_copy(home, sip->sip_retry_after);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_retry_after_t *sip_retry_after_copy(su_home_t *home, sip_retry_after_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_retry_after_t *sip_retry_after_copy(su_home_t *home, sip_retry_after_t const *hdr)
{
  return (sip_retry_after_t *)
    msg_header_copy_as(home, sip_retry_after_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_retry_after "Retry-After header" structure #sip_retry_after_t.
 *
 * The function sip_retry_after_make() makes a new
 * #sip_retry_after_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_retry_after_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_retry_after_t *sip_retry_after_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_retry_after_t *sip_retry_after_make(su_home_t *home, char const *s)
{
  return (sip_retry_after_t *)sip_header_make(home, sip_retry_after_class, s);
}
#endif

/**Make a @ref sip_retry_after "Retry-After header" from formatting result.
 *
 * Make a new #sip_retry_after_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_retry_after_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_retry_after_t *sip_retry_after_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_retry_after_t *sip_retry_after_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_retry_after_class, fmt, ap);
  va_end(ap);

  return (sip_retry_after_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_timestamp
 * @{
 */

/** Parse a SIP @ref sip_timestamp "Timestamp header". @internal */
SOFIAPUBFUN issize_t sip_timestamp_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_timestamp "Timestamp header". @internal */
SOFIAPUBFUN issize_t sip_timestamp_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_timestamp "Timestamp header"
 * structure #sip_timestamp_t from #sip_t.
 *
 */
#define sip_timestamp(sip) \
  ((sip_timestamp_t *)msg_header_access((msg_pub_t*)(sip), sip_timestamp_class))

/**Initializer for structure #sip_timestamp_t.
 *
 * A static #sip_timestamp_t structure for
 * @ref sip_timestamp "Timestamp header" must be initialized with
 * the SIP_TIMESTAMP_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_timestamp_t sip_timestamp = SIP_TIMESTAMP_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_TIMESTAMP_INIT() SIP_HDR_INIT(timestamp)

/**Initialize a structure #sip_timestamp_t.
 *
 * An #sip_timestamp_t structure for
 * @ref sip_timestamp "Timestamp header" can be initialized with the
 * sip_timestamp_init() function/macro. For instance,
 * @code
 *
 *  sip_timestamp_t sip_timestamp;
 *
 *  sip_timestamp_init(&sip_timestamp);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_timestamp_t *sip_timestamp_init(sip_timestamp_t x[1])
{
  return SIP_HEADER_INIT(x, sip_timestamp_class, sizeof(sip_timestamp_t));
}
#else
#define sip_timestamp_init(x) \
  SIP_HEADER_INIT(x, sip_timestamp_class, sizeof(sip_timestamp_t))
#endif

/**Test if header object is instance of #sip_timestamp_t.
 *
 * Check if the header class is an instance of
 * @ref sip_timestamp "Timestamp header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header timestamp
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_timestamp(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_timestamp_hash;
}
#else
int sip_is_timestamp(sip_header_t const *header);
#endif

#define sip_timestamp_p(h) sip_is_timestamp((h))


/**Duplicate a list of @ref sip_timestamp "Timestamp header" header structures #sip_timestamp_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   timestamp = sip_timestamp_dup(home, sip->sip_timestamp);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_timestamp_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_timestamp_t *sip_timestamp_dup(su_home_t *home, sip_timestamp_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_timestamp_t *sip_timestamp_dup(su_home_t *home, sip_timestamp_t const *hdr)
{
  return (sip_timestamp_t *)
    msg_header_dup_as(home, sip_timestamp_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_timestamp "Timestamp header" header structures #sip_timestamp_t.
 *
 * The function sip_timestamp_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   timestamp = sip_timestamp_copy(home, sip->sip_timestamp);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_timestamp_t *sip_timestamp_copy(su_home_t *home, sip_timestamp_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_timestamp_t *sip_timestamp_copy(su_home_t *home, sip_timestamp_t const *hdr)
{
  return (sip_timestamp_t *)
    msg_header_copy_as(home, sip_timestamp_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_timestamp "Timestamp header" structure #sip_timestamp_t.
 *
 * The function sip_timestamp_make() makes a new
 * #sip_timestamp_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_timestamp_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_timestamp_t *sip_timestamp_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_timestamp_t *sip_timestamp_make(su_home_t *home, char const *s)
{
  return (sip_timestamp_t *)sip_header_make(home, sip_timestamp_class, s);
}
#endif

/**Make a @ref sip_timestamp "Timestamp header" from formatting result.
 *
 * Make a new #sip_timestamp_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_timestamp_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_timestamp_t *sip_timestamp_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_timestamp_t *sip_timestamp_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_timestamp_class, fmt, ap);
  va_end(ap);

  return (sip_timestamp_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_min_expires
 * @{
 */

/** Parse a SIP @ref sip_min_expires "Min-Expires header". @internal */
SOFIAPUBFUN issize_t sip_min_expires_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_min_expires "Min-Expires header". @internal */
SOFIAPUBFUN issize_t sip_min_expires_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_min_expires "Min-Expires header"
 * structure #sip_min_expires_t from #sip_t.
 *
 */
#define sip_min_expires(sip) \
  ((sip_min_expires_t *)msg_header_access((msg_pub_t*)(sip), sip_min_expires_class))

/**Initializer for structure #sip_min_expires_t.
 *
 * A static #sip_min_expires_t structure for
 * @ref sip_min_expires "Min-Expires header" must be initialized with
 * the SIP_MIN_EXPIRES_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_min_expires_t sip_min_expires = SIP_MIN_EXPIRES_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_MIN_EXPIRES_INIT() SIP_HDR_INIT(min_expires)

/**Initialize a structure #sip_min_expires_t.
 *
 * An #sip_min_expires_t structure for
 * @ref sip_min_expires "Min-Expires header" can be initialized with the
 * sip_min_expires_init() function/macro. For instance,
 * @code
 *
 *  sip_min_expires_t sip_min_expires;
 *
 *  sip_min_expires_init(&sip_min_expires);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_min_expires_t *sip_min_expires_init(sip_min_expires_t x[1])
{
  return SIP_HEADER_INIT(x, sip_min_expires_class, sizeof(sip_min_expires_t));
}
#else
#define sip_min_expires_init(x) \
  SIP_HEADER_INIT(x, sip_min_expires_class, sizeof(sip_min_expires_t))
#endif

/**Test if header object is instance of #sip_min_expires_t.
 *
 * Check if the header class is an instance of
 * @ref sip_min_expires "Min-Expires header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header min_expires
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_min_expires(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_min_expires_hash;
}
#else
int sip_is_min_expires(sip_header_t const *header);
#endif

#define sip_min_expires_p(h) sip_is_min_expires((h))


/**Duplicate a list of @ref sip_min_expires "Min-Expires header" header structures #sip_min_expires_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   min_expires = sip_min_expires_dup(home, sip->sip_min_expires);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_min_expires_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_min_expires_t *sip_min_expires_dup(su_home_t *home, sip_min_expires_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_min_expires_t *sip_min_expires_dup(su_home_t *home, sip_min_expires_t const *hdr)
{
  return (sip_min_expires_t *)
    msg_header_dup_as(home, sip_min_expires_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_min_expires "Min-Expires header" header structures #sip_min_expires_t.
 *
 * The function sip_min_expires_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   min_expires = sip_min_expires_copy(home, sip->sip_min_expires);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_min_expires_t *sip_min_expires_copy(su_home_t *home, sip_min_expires_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_min_expires_t *sip_min_expires_copy(su_home_t *home, sip_min_expires_t const *hdr)
{
  return (sip_min_expires_t *)
    msg_header_copy_as(home, sip_min_expires_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_min_expires "Min-Expires header" structure #sip_min_expires_t.
 *
 * The function sip_min_expires_make() makes a new
 * #sip_min_expires_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_min_expires_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_min_expires_t *sip_min_expires_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_min_expires_t *sip_min_expires_make(su_home_t *home, char const *s)
{
  return (sip_min_expires_t *)sip_header_make(home, sip_min_expires_class, s);
}
#endif

/**Make a @ref sip_min_expires "Min-Expires header" from formatting result.
 *
 * Make a new #sip_min_expires_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_min_expires_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_min_expires_t *sip_min_expires_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_min_expires_t *sip_min_expires_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_min_expires_class, fmt, ap);
  va_end(ap);

  return (sip_min_expires_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_subject
 * @{
 */

/** Parse a SIP @ref sip_subject "Subject header". @internal */
SOFIAPUBFUN issize_t sip_subject_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_subject "Subject header". @internal */
SOFIAPUBFUN issize_t sip_subject_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_subject "Subject header"
 * structure #sip_subject_t from #sip_t.
 *
 */
#define sip_subject(sip) \
  ((sip_subject_t *)msg_header_access((msg_pub_t*)(sip), sip_subject_class))

/**Initializer for structure #sip_subject_t.
 *
 * A static #sip_subject_t structure for
 * @ref sip_subject "Subject header" must be initialized with
 * the SIP_SUBJECT_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_subject_t sip_subject = SIP_SUBJECT_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SUBJECT_INIT() SIP_HDR_INIT(subject)

/**Initialize a structure #sip_subject_t.
 *
 * An #sip_subject_t structure for
 * @ref sip_subject "Subject header" can be initialized with the
 * sip_subject_init() function/macro. For instance,
 * @code
 *
 *  sip_subject_t sip_subject;
 *
 *  sip_subject_init(&sip_subject);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_subject_t *sip_subject_init(sip_subject_t x[1])
{
  return SIP_HEADER_INIT(x, sip_subject_class, sizeof(sip_subject_t));
}
#else
#define sip_subject_init(x) \
  SIP_HEADER_INIT(x, sip_subject_class, sizeof(sip_subject_t))
#endif

/**Test if header object is instance of #sip_subject_t.
 *
 * Check if the header class is an instance of
 * @ref sip_subject "Subject header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header subject
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_subject(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_subject_hash;
}
#else
int sip_is_subject(sip_header_t const *header);
#endif

#define sip_subject_p(h) sip_is_subject((h))


/**Duplicate a list of @ref sip_subject "Subject header" header structures #sip_subject_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   subject = sip_subject_dup(home, sip->sip_subject);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_subject_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_subject_t *sip_subject_dup(su_home_t *home, sip_subject_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_subject_t *sip_subject_dup(su_home_t *home, sip_subject_t const *hdr)
{
  return (sip_subject_t *)
    msg_header_dup_as(home, sip_subject_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_subject "Subject header" header structures #sip_subject_t.
 *
 * The function sip_subject_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   subject = sip_subject_copy(home, sip->sip_subject);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_subject_t *sip_subject_copy(su_home_t *home, sip_subject_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_subject_t *sip_subject_copy(su_home_t *home, sip_subject_t const *hdr)
{
  return (sip_subject_t *)
    msg_header_copy_as(home, sip_subject_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_subject "Subject header" structure #sip_subject_t.
 *
 * The function sip_subject_make() makes a new
 * #sip_subject_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_subject_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_subject_t *sip_subject_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_subject_t *sip_subject_make(su_home_t *home, char const *s)
{
  return (sip_subject_t *)sip_header_make(home, sip_subject_class, s);
}
#endif

/**Make a @ref sip_subject "Subject header" from formatting result.
 *
 * Make a new #sip_subject_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_subject_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_subject_t *sip_subject_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_subject_t *sip_subject_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_subject_class, fmt, ap);
  va_end(ap);

  return (sip_subject_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_priority
 * @{
 */

/** Parse a SIP @ref sip_priority "Priority header". @internal */
SOFIAPUBFUN issize_t sip_priority_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_priority "Priority header". @internal */
SOFIAPUBFUN issize_t sip_priority_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_priority "Priority header"
 * structure #sip_priority_t from #sip_t.
 *
 */
#define sip_priority(sip) \
  ((sip_priority_t *)msg_header_access((msg_pub_t*)(sip), sip_priority_class))

/**Initializer for structure #sip_priority_t.
 *
 * A static #sip_priority_t structure for
 * @ref sip_priority "Priority header" must be initialized with
 * the SIP_PRIORITY_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_priority_t sip_priority = SIP_PRIORITY_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_PRIORITY_INIT() SIP_HDR_INIT(priority)

/**Initialize a structure #sip_priority_t.
 *
 * An #sip_priority_t structure for
 * @ref sip_priority "Priority header" can be initialized with the
 * sip_priority_init() function/macro. For instance,
 * @code
 *
 *  sip_priority_t sip_priority;
 *
 *  sip_priority_init(&sip_priority);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_priority_t *sip_priority_init(sip_priority_t x[1])
{
  return SIP_HEADER_INIT(x, sip_priority_class, sizeof(sip_priority_t));
}
#else
#define sip_priority_init(x) \
  SIP_HEADER_INIT(x, sip_priority_class, sizeof(sip_priority_t))
#endif

/**Test if header object is instance of #sip_priority_t.
 *
 * Check if the header class is an instance of
 * @ref sip_priority "Priority header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header priority
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_priority(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_priority_hash;
}
#else
int sip_is_priority(sip_header_t const *header);
#endif

#define sip_priority_p(h) sip_is_priority((h))


/**Duplicate a list of @ref sip_priority "Priority header" header structures #sip_priority_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   priority = sip_priority_dup(home, sip->sip_priority);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_priority_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_priority_t *sip_priority_dup(su_home_t *home, sip_priority_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_priority_t *sip_priority_dup(su_home_t *home, sip_priority_t const *hdr)
{
  return (sip_priority_t *)
    msg_header_dup_as(home, sip_priority_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_priority "Priority header" header structures #sip_priority_t.
 *
 * The function sip_priority_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   priority = sip_priority_copy(home, sip->sip_priority);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_priority_t *sip_priority_copy(su_home_t *home, sip_priority_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_priority_t *sip_priority_copy(su_home_t *home, sip_priority_t const *hdr)
{
  return (sip_priority_t *)
    msg_header_copy_as(home, sip_priority_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_priority "Priority header" structure #sip_priority_t.
 *
 * The function sip_priority_make() makes a new
 * #sip_priority_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_priority_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_priority_t *sip_priority_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_priority_t *sip_priority_make(su_home_t *home, char const *s)
{
  return (sip_priority_t *)sip_header_make(home, sip_priority_class, s);
}
#endif

/**Make a @ref sip_priority "Priority header" from formatting result.
 *
 * Make a new #sip_priority_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_priority_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_priority_t *sip_priority_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_priority_t *sip_priority_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_priority_class, fmt, ap);
  va_end(ap);

  return (sip_priority_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_call_info
 * @{
 */

/** Parse a SIP @ref sip_call_info "Call-Info header". @internal */
SOFIAPUBFUN issize_t sip_call_info_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_call_info "Call-Info header". @internal */
SOFIAPUBFUN issize_t sip_call_info_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_call_info "Call-Info header"
 * structure #sip_call_info_t from #sip_t.
 *
 */
#define sip_call_info(sip) \
  ((sip_call_info_t *)msg_header_access((msg_pub_t*)(sip), sip_call_info_class))

/**Initializer for structure #sip_call_info_t.
 *
 * A static #sip_call_info_t structure for
 * @ref sip_call_info "Call-Info header" must be initialized with
 * the SIP_CALL_INFO_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_call_info_t sip_call_info = SIP_CALL_INFO_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CALL_INFO_INIT() SIP_HDR_INIT(call_info)

/**Initialize a structure #sip_call_info_t.
 *
 * An #sip_call_info_t structure for
 * @ref sip_call_info "Call-Info header" can be initialized with the
 * sip_call_info_init() function/macro. For instance,
 * @code
 *
 *  sip_call_info_t sip_call_info;
 *
 *  sip_call_info_init(&sip_call_info);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_call_info_t *sip_call_info_init(sip_call_info_t x[1])
{
  return SIP_HEADER_INIT(x, sip_call_info_class, sizeof(sip_call_info_t));
}
#else
#define sip_call_info_init(x) \
  SIP_HEADER_INIT(x, sip_call_info_class, sizeof(sip_call_info_t))
#endif

/**Test if header object is instance of #sip_call_info_t.
 *
 * Check if the header class is an instance of
 * @ref sip_call_info "Call-Info header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header call_info
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_call_info(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_call_info_hash;
}
#else
int sip_is_call_info(sip_header_t const *header);
#endif

#define sip_call_info_p(h) sip_is_call_info((h))


/**Duplicate a list of @ref sip_call_info "Call-Info header" header structures #sip_call_info_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   call_info = sip_call_info_dup(home, sip->sip_call_info);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_call_info_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_call_info_t *sip_call_info_dup(su_home_t *home, sip_call_info_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_call_info_t *sip_call_info_dup(su_home_t *home, sip_call_info_t const *hdr)
{
  return (sip_call_info_t *)
    msg_header_dup_as(home, sip_call_info_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_call_info "Call-Info header" header structures #sip_call_info_t.
 *
 * The function sip_call_info_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   call_info = sip_call_info_copy(home, sip->sip_call_info);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_call_info_t *sip_call_info_copy(su_home_t *home, sip_call_info_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_call_info_t *sip_call_info_copy(su_home_t *home, sip_call_info_t const *hdr)
{
  return (sip_call_info_t *)
    msg_header_copy_as(home, sip_call_info_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_call_info "Call-Info header" structure #sip_call_info_t.
 *
 * The function sip_call_info_make() makes a new
 * #sip_call_info_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_call_info_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_call_info_t *sip_call_info_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_call_info_t *sip_call_info_make(su_home_t *home, char const *s)
{
  return (sip_call_info_t *)sip_header_make(home, sip_call_info_class, s);
}
#endif

/**Make a @ref sip_call_info "Call-Info header" from formatting result.
 *
 * Make a new #sip_call_info_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_call_info_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_call_info_t *sip_call_info_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_call_info_t *sip_call_info_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_call_info_class, fmt, ap);
  va_end(ap);

  return (sip_call_info_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_organization
 * @{
 */

/** Parse a SIP @ref sip_organization "Organization header". @internal */
SOFIAPUBFUN issize_t sip_organization_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_organization "Organization header". @internal */
SOFIAPUBFUN issize_t sip_organization_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_organization "Organization header"
 * structure #sip_organization_t from #sip_t.
 *
 */
#define sip_organization(sip) \
  ((sip_organization_t *)msg_header_access((msg_pub_t*)(sip), sip_organization_class))

/**Initializer for structure #sip_organization_t.
 *
 * A static #sip_organization_t structure for
 * @ref sip_organization "Organization header" must be initialized with
 * the SIP_ORGANIZATION_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_organization_t sip_organization = SIP_ORGANIZATION_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ORGANIZATION_INIT() SIP_HDR_INIT(organization)

/**Initialize a structure #sip_organization_t.
 *
 * An #sip_organization_t structure for
 * @ref sip_organization "Organization header" can be initialized with the
 * sip_organization_init() function/macro. For instance,
 * @code
 *
 *  sip_organization_t sip_organization;
 *
 *  sip_organization_init(&sip_organization);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_organization_t *sip_organization_init(sip_organization_t x[1])
{
  return SIP_HEADER_INIT(x, sip_organization_class, sizeof(sip_organization_t));
}
#else
#define sip_organization_init(x) \
  SIP_HEADER_INIT(x, sip_organization_class, sizeof(sip_organization_t))
#endif

/**Test if header object is instance of #sip_organization_t.
 *
 * Check if the header class is an instance of
 * @ref sip_organization "Organization header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header organization
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_organization(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_organization_hash;
}
#else
int sip_is_organization(sip_header_t const *header);
#endif

#define sip_organization_p(h) sip_is_organization((h))


/**Duplicate a list of @ref sip_organization "Organization header" header structures #sip_organization_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   organization = sip_organization_dup(home, sip->sip_organization);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_organization_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_organization_t *sip_organization_dup(su_home_t *home, sip_organization_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_organization_t *sip_organization_dup(su_home_t *home, sip_organization_t const *hdr)
{
  return (sip_organization_t *)
    msg_header_dup_as(home, sip_organization_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_organization "Organization header" header structures #sip_organization_t.
 *
 * The function sip_organization_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   organization = sip_organization_copy(home, sip->sip_organization);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_organization_t *sip_organization_copy(su_home_t *home, sip_organization_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_organization_t *sip_organization_copy(su_home_t *home, sip_organization_t const *hdr)
{
  return (sip_organization_t *)
    msg_header_copy_as(home, sip_organization_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_organization "Organization header" structure #sip_organization_t.
 *
 * The function sip_organization_make() makes a new
 * #sip_organization_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_organization_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_organization_t *sip_organization_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_organization_t *sip_organization_make(su_home_t *home, char const *s)
{
  return (sip_organization_t *)sip_header_make(home, sip_organization_class, s);
}
#endif

/**Make a @ref sip_organization "Organization header" from formatting result.
 *
 * Make a new #sip_organization_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_organization_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_organization_t *sip_organization_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_organization_t *sip_organization_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_organization_class, fmt, ap);
  va_end(ap);

  return (sip_organization_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_server
 * @{
 */

/** Parse a SIP @ref sip_server "Server header". @internal */
SOFIAPUBFUN issize_t sip_server_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_server "Server header". @internal */
SOFIAPUBFUN issize_t sip_server_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_server "Server header"
 * structure #sip_server_t from #sip_t.
 *
 */
#define sip_server(sip) \
  ((sip_server_t *)msg_header_access((msg_pub_t*)(sip), sip_server_class))

/**Initializer for structure #sip_server_t.
 *
 * A static #sip_server_t structure for
 * @ref sip_server "Server header" must be initialized with
 * the SIP_SERVER_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_server_t sip_server = SIP_SERVER_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SERVER_INIT() SIP_HDR_INIT(server)

/**Initialize a structure #sip_server_t.
 *
 * An #sip_server_t structure for
 * @ref sip_server "Server header" can be initialized with the
 * sip_server_init() function/macro. For instance,
 * @code
 *
 *  sip_server_t sip_server;
 *
 *  sip_server_init(&sip_server);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_server_t *sip_server_init(sip_server_t x[1])
{
  return SIP_HEADER_INIT(x, sip_server_class, sizeof(sip_server_t));
}
#else
#define sip_server_init(x) \
  SIP_HEADER_INIT(x, sip_server_class, sizeof(sip_server_t))
#endif

/**Test if header object is instance of #sip_server_t.
 *
 * Check if the header class is an instance of
 * @ref sip_server "Server header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header server
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_server(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_server_hash;
}
#else
int sip_is_server(sip_header_t const *header);
#endif

#define sip_server_p(h) sip_is_server((h))


/**Duplicate a list of @ref sip_server "Server header" header structures #sip_server_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   server = sip_server_dup(home, sip->sip_server);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_server_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_server_t *sip_server_dup(su_home_t *home, sip_server_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_server_t *sip_server_dup(su_home_t *home, sip_server_t const *hdr)
{
  return (sip_server_t *)
    msg_header_dup_as(home, sip_server_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_server "Server header" header structures #sip_server_t.
 *
 * The function sip_server_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   server = sip_server_copy(home, sip->sip_server);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_server_t *sip_server_copy(su_home_t *home, sip_server_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_server_t *sip_server_copy(su_home_t *home, sip_server_t const *hdr)
{
  return (sip_server_t *)
    msg_header_copy_as(home, sip_server_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_server "Server header" structure #sip_server_t.
 *
 * The function sip_server_make() makes a new
 * #sip_server_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_server_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_server_t *sip_server_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_server_t *sip_server_make(su_home_t *home, char const *s)
{
  return (sip_server_t *)sip_header_make(home, sip_server_class, s);
}
#endif

/**Make a @ref sip_server "Server header" from formatting result.
 *
 * Make a new #sip_server_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_server_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_server_t *sip_server_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_server_t *sip_server_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_server_class, fmt, ap);
  va_end(ap);

  return (sip_server_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_user_agent
 * @{
 */

/** Parse a SIP @ref sip_user_agent "User-Agent header". @internal */
SOFIAPUBFUN issize_t sip_user_agent_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_user_agent "User-Agent header". @internal */
SOFIAPUBFUN issize_t sip_user_agent_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_user_agent "User-Agent header"
 * structure #sip_user_agent_t from #sip_t.
 *
 */
#define sip_user_agent(sip) \
  ((sip_user_agent_t *)msg_header_access((msg_pub_t*)(sip), sip_user_agent_class))

/**Initializer for structure #sip_user_agent_t.
 *
 * A static #sip_user_agent_t structure for
 * @ref sip_user_agent "User-Agent header" must be initialized with
 * the SIP_USER_AGENT_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_user_agent_t sip_user_agent = SIP_USER_AGENT_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_USER_AGENT_INIT() SIP_HDR_INIT(user_agent)

/**Initialize a structure #sip_user_agent_t.
 *
 * An #sip_user_agent_t structure for
 * @ref sip_user_agent "User-Agent header" can be initialized with the
 * sip_user_agent_init() function/macro. For instance,
 * @code
 *
 *  sip_user_agent_t sip_user_agent;
 *
 *  sip_user_agent_init(&sip_user_agent);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_user_agent_t *sip_user_agent_init(sip_user_agent_t x[1])
{
  return SIP_HEADER_INIT(x, sip_user_agent_class, sizeof(sip_user_agent_t));
}
#else
#define sip_user_agent_init(x) \
  SIP_HEADER_INIT(x, sip_user_agent_class, sizeof(sip_user_agent_t))
#endif

/**Test if header object is instance of #sip_user_agent_t.
 *
 * Check if the header class is an instance of
 * @ref sip_user_agent "User-Agent header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header user_agent
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_user_agent(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_user_agent_hash;
}
#else
int sip_is_user_agent(sip_header_t const *header);
#endif

#define sip_user_agent_p(h) sip_is_user_agent((h))


/**Duplicate a list of @ref sip_user_agent "User-Agent header" header structures #sip_user_agent_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   user_agent = sip_user_agent_dup(home, sip->sip_user_agent);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_user_agent_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_user_agent_t *sip_user_agent_dup(su_home_t *home, sip_user_agent_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_user_agent_t *sip_user_agent_dup(su_home_t *home, sip_user_agent_t const *hdr)
{
  return (sip_user_agent_t *)
    msg_header_dup_as(home, sip_user_agent_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_user_agent "User-Agent header" header structures #sip_user_agent_t.
 *
 * The function sip_user_agent_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   user_agent = sip_user_agent_copy(home, sip->sip_user_agent);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_user_agent_t *sip_user_agent_copy(su_home_t *home, sip_user_agent_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_user_agent_t *sip_user_agent_copy(su_home_t *home, sip_user_agent_t const *hdr)
{
  return (sip_user_agent_t *)
    msg_header_copy_as(home, sip_user_agent_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_user_agent "User-Agent header" structure #sip_user_agent_t.
 *
 * The function sip_user_agent_make() makes a new
 * #sip_user_agent_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_user_agent_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_user_agent_t *sip_user_agent_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_user_agent_t *sip_user_agent_make(su_home_t *home, char const *s)
{
  return (sip_user_agent_t *)sip_header_make(home, sip_user_agent_class, s);
}
#endif

/**Make a @ref sip_user_agent "User-Agent header" from formatting result.
 *
 * Make a new #sip_user_agent_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_user_agent_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_user_agent_t *sip_user_agent_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_user_agent_t *sip_user_agent_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_user_agent_class, fmt, ap);
  va_end(ap);

  return (sip_user_agent_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_in_reply_to
 * @{
 */

/** Parse a SIP @ref sip_in_reply_to "In-Reply-To header". @internal */
SOFIAPUBFUN issize_t sip_in_reply_to_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_in_reply_to "In-Reply-To header". @internal */
SOFIAPUBFUN issize_t sip_in_reply_to_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_in_reply_to "In-Reply-To header"
 * structure #sip_in_reply_to_t from #sip_t.
 *
 */
#define sip_in_reply_to(sip) \
  ((sip_in_reply_to_t *)msg_header_access((msg_pub_t*)(sip), sip_in_reply_to_class))

/**Initializer for structure #sip_in_reply_to_t.
 *
 * A static #sip_in_reply_to_t structure for
 * @ref sip_in_reply_to "In-Reply-To header" must be initialized with
 * the SIP_IN_REPLY_TO_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_in_reply_to_t sip_in_reply_to = SIP_IN_REPLY_TO_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_IN_REPLY_TO_INIT() SIP_HDR_INIT(in_reply_to)

/**Initialize a structure #sip_in_reply_to_t.
 *
 * An #sip_in_reply_to_t structure for
 * @ref sip_in_reply_to "In-Reply-To header" can be initialized with the
 * sip_in_reply_to_init() function/macro. For instance,
 * @code
 *
 *  sip_in_reply_to_t sip_in_reply_to;
 *
 *  sip_in_reply_to_init(&sip_in_reply_to);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_in_reply_to_t *sip_in_reply_to_init(sip_in_reply_to_t x[1])
{
  return SIP_HEADER_INIT(x, sip_in_reply_to_class, sizeof(sip_in_reply_to_t));
}
#else
#define sip_in_reply_to_init(x) \
  SIP_HEADER_INIT(x, sip_in_reply_to_class, sizeof(sip_in_reply_to_t))
#endif

/**Test if header object is instance of #sip_in_reply_to_t.
 *
 * Check if the header class is an instance of
 * @ref sip_in_reply_to "In-Reply-To header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header in_reply_to
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_in_reply_to(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_in_reply_to_hash;
}
#else
int sip_is_in_reply_to(sip_header_t const *header);
#endif

#define sip_in_reply_to_p(h) sip_is_in_reply_to((h))


/**Duplicate a list of @ref sip_in_reply_to "In-Reply-To header" header structures #sip_in_reply_to_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   in_reply_to = sip_in_reply_to_dup(home, sip->sip_in_reply_to);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_in_reply_to_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_in_reply_to_t *sip_in_reply_to_dup(su_home_t *home, sip_in_reply_to_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_in_reply_to_t *sip_in_reply_to_dup(su_home_t *home, sip_in_reply_to_t const *hdr)
{
  return (sip_in_reply_to_t *)
    msg_header_dup_as(home, sip_in_reply_to_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_in_reply_to "In-Reply-To header" header structures #sip_in_reply_to_t.
 *
 * The function sip_in_reply_to_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   in_reply_to = sip_in_reply_to_copy(home, sip->sip_in_reply_to);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_in_reply_to_t *sip_in_reply_to_copy(su_home_t *home, sip_in_reply_to_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_in_reply_to_t *sip_in_reply_to_copy(su_home_t *home, sip_in_reply_to_t const *hdr)
{
  return (sip_in_reply_to_t *)
    msg_header_copy_as(home, sip_in_reply_to_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_in_reply_to "In-Reply-To header" structure #sip_in_reply_to_t.
 *
 * The function sip_in_reply_to_make() makes a new
 * #sip_in_reply_to_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_in_reply_to_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_in_reply_to_t *sip_in_reply_to_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_in_reply_to_t *sip_in_reply_to_make(su_home_t *home, char const *s)
{
  return (sip_in_reply_to_t *)sip_header_make(home, sip_in_reply_to_class, s);
}
#endif

/**Make a @ref sip_in_reply_to "In-Reply-To header" from formatting result.
 *
 * Make a new #sip_in_reply_to_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_in_reply_to_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_in_reply_to_t *sip_in_reply_to_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_in_reply_to_t *sip_in_reply_to_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_in_reply_to_class, fmt, ap);
  va_end(ap);

  return (sip_in_reply_to_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_accept
 * @{
 */

/** Parse a SIP @ref sip_accept "Accept header". @internal */
SOFIAPUBFUN issize_t sip_accept_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_accept "Accept header". @internal */
SOFIAPUBFUN issize_t sip_accept_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_accept "Accept header"
 * structure #sip_accept_t from #sip_t.
 *
 */
#define sip_accept(sip) \
  ((sip_accept_t *)msg_header_access((msg_pub_t*)(sip), sip_accept_class))

/**Initializer for structure #sip_accept_t.
 *
 * A static #sip_accept_t structure for
 * @ref sip_accept "Accept header" must be initialized with
 * the SIP_ACCEPT_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_accept_t sip_accept = SIP_ACCEPT_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ACCEPT_INIT() SIP_HDR_INIT(accept)

/**Initialize a structure #sip_accept_t.
 *
 * An #sip_accept_t structure for
 * @ref sip_accept "Accept header" can be initialized with the
 * sip_accept_init() function/macro. For instance,
 * @code
 *
 *  sip_accept_t sip_accept;
 *
 *  sip_accept_init(&sip_accept);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_accept_t *sip_accept_init(sip_accept_t x[1])
{
  return SIP_HEADER_INIT(x, sip_accept_class, sizeof(sip_accept_t));
}
#else
#define sip_accept_init(x) \
  SIP_HEADER_INIT(x, sip_accept_class, sizeof(sip_accept_t))
#endif

/**Test if header object is instance of #sip_accept_t.
 *
 * Check if the header class is an instance of
 * @ref sip_accept "Accept header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header accept
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_accept(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_accept_hash;
}
#else
int sip_is_accept(sip_header_t const *header);
#endif

#define sip_accept_p(h) sip_is_accept((h))


/**Duplicate a list of @ref sip_accept "Accept header" header structures #sip_accept_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   accept = sip_accept_dup(home, sip->sip_accept);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_accept_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_t *sip_accept_dup(su_home_t *home, sip_accept_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_accept_t *sip_accept_dup(su_home_t *home, sip_accept_t const *hdr)
{
  return (sip_accept_t *)
    msg_header_dup_as(home, sip_accept_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_accept "Accept header" header structures #sip_accept_t.
 *
 * The function sip_accept_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   accept = sip_accept_copy(home, sip->sip_accept);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_t *sip_accept_copy(su_home_t *home, sip_accept_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_accept_t *sip_accept_copy(su_home_t *home, sip_accept_t const *hdr)
{
  return (sip_accept_t *)
    msg_header_copy_as(home, sip_accept_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_accept "Accept header" structure #sip_accept_t.
 *
 * The function sip_accept_make() makes a new
 * #sip_accept_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_accept_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_t *sip_accept_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_accept_t *sip_accept_make(su_home_t *home, char const *s)
{
  return (sip_accept_t *)sip_header_make(home, sip_accept_class, s);
}
#endif

/**Make a @ref sip_accept "Accept header" from formatting result.
 *
 * Make a new #sip_accept_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_accept_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_t *sip_accept_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_accept_t *sip_accept_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_accept_class, fmt, ap);
  va_end(ap);

  return (sip_accept_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_accept_encoding
 * @{
 */

/** Parse a SIP @ref sip_accept_encoding "Accept-Encoding header". @internal */
SOFIAPUBFUN issize_t sip_accept_encoding_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_accept_encoding "Accept-Encoding header". @internal */
SOFIAPUBFUN issize_t sip_accept_encoding_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_accept_encoding "Accept-Encoding header"
 * structure #sip_accept_encoding_t from #sip_t.
 *
 */
#define sip_accept_encoding(sip) \
  ((sip_accept_encoding_t *)msg_header_access((msg_pub_t*)(sip), sip_accept_encoding_class))

/**Initializer for structure #sip_accept_encoding_t.
 *
 * A static #sip_accept_encoding_t structure for
 * @ref sip_accept_encoding "Accept-Encoding header" must be initialized with
 * the SIP_ACCEPT_ENCODING_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_accept_encoding_t sip_accept_encoding = SIP_ACCEPT_ENCODING_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ACCEPT_ENCODING_INIT() SIP_HDR_INIT(accept_encoding)

/**Initialize a structure #sip_accept_encoding_t.
 *
 * An #sip_accept_encoding_t structure for
 * @ref sip_accept_encoding "Accept-Encoding header" can be initialized with the
 * sip_accept_encoding_init() function/macro. For instance,
 * @code
 *
 *  sip_accept_encoding_t sip_accept_encoding;
 *
 *  sip_accept_encoding_init(&sip_accept_encoding);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_accept_encoding_t *sip_accept_encoding_init(sip_accept_encoding_t x[1])
{
  return SIP_HEADER_INIT(x, sip_accept_encoding_class, sizeof(sip_accept_encoding_t));
}
#else
#define sip_accept_encoding_init(x) \
  SIP_HEADER_INIT(x, sip_accept_encoding_class, sizeof(sip_accept_encoding_t))
#endif

/**Test if header object is instance of #sip_accept_encoding_t.
 *
 * Check if the header class is an instance of
 * @ref sip_accept_encoding "Accept-Encoding header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header accept_encoding
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_accept_encoding(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_accept_encoding_hash;
}
#else
int sip_is_accept_encoding(sip_header_t const *header);
#endif

#define sip_accept_encoding_p(h) sip_is_accept_encoding((h))


/**Duplicate a list of @ref sip_accept_encoding "Accept-Encoding header" header structures #sip_accept_encoding_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   accept_encoding = sip_accept_encoding_dup(home, sip->sip_accept_encoding);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_accept_encoding_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_encoding_t *sip_accept_encoding_dup(su_home_t *home, sip_accept_encoding_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_accept_encoding_t *sip_accept_encoding_dup(su_home_t *home, sip_accept_encoding_t const *hdr)
{
  return (sip_accept_encoding_t *)
    msg_header_dup_as(home, sip_accept_encoding_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_accept_encoding "Accept-Encoding header" header structures #sip_accept_encoding_t.
 *
 * The function sip_accept_encoding_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   accept_encoding = sip_accept_encoding_copy(home, sip->sip_accept_encoding);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_encoding_t *sip_accept_encoding_copy(su_home_t *home, sip_accept_encoding_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_accept_encoding_t *sip_accept_encoding_copy(su_home_t *home, sip_accept_encoding_t const *hdr)
{
  return (sip_accept_encoding_t *)
    msg_header_copy_as(home, sip_accept_encoding_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_accept_encoding "Accept-Encoding header" structure #sip_accept_encoding_t.
 *
 * The function sip_accept_encoding_make() makes a new
 * #sip_accept_encoding_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_accept_encoding_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_encoding_t *sip_accept_encoding_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_accept_encoding_t *sip_accept_encoding_make(su_home_t *home, char const *s)
{
  return (sip_accept_encoding_t *)sip_header_make(home, sip_accept_encoding_class, s);
}
#endif

/**Make a @ref sip_accept_encoding "Accept-Encoding header" from formatting result.
 *
 * Make a new #sip_accept_encoding_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_accept_encoding_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_encoding_t *sip_accept_encoding_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_accept_encoding_t *sip_accept_encoding_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_accept_encoding_class, fmt, ap);
  va_end(ap);

  return (sip_accept_encoding_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_accept_language
 * @{
 */

/** Parse a SIP @ref sip_accept_language "Accept-Language header". @internal */
SOFIAPUBFUN issize_t sip_accept_language_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_accept_language "Accept-Language header". @internal */
SOFIAPUBFUN issize_t sip_accept_language_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_accept_language "Accept-Language header"
 * structure #sip_accept_language_t from #sip_t.
 *
 */
#define sip_accept_language(sip) \
  ((sip_accept_language_t *)msg_header_access((msg_pub_t*)(sip), sip_accept_language_class))

/**Initializer for structure #sip_accept_language_t.
 *
 * A static #sip_accept_language_t structure for
 * @ref sip_accept_language "Accept-Language header" must be initialized with
 * the SIP_ACCEPT_LANGUAGE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_accept_language_t sip_accept_language = SIP_ACCEPT_LANGUAGE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ACCEPT_LANGUAGE_INIT() SIP_HDR_INIT(accept_language)

/**Initialize a structure #sip_accept_language_t.
 *
 * An #sip_accept_language_t structure for
 * @ref sip_accept_language "Accept-Language header" can be initialized with the
 * sip_accept_language_init() function/macro. For instance,
 * @code
 *
 *  sip_accept_language_t sip_accept_language;
 *
 *  sip_accept_language_init(&sip_accept_language);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_accept_language_t *sip_accept_language_init(sip_accept_language_t x[1])
{
  return SIP_HEADER_INIT(x, sip_accept_language_class, sizeof(sip_accept_language_t));
}
#else
#define sip_accept_language_init(x) \
  SIP_HEADER_INIT(x, sip_accept_language_class, sizeof(sip_accept_language_t))
#endif

/**Test if header object is instance of #sip_accept_language_t.
 *
 * Check if the header class is an instance of
 * @ref sip_accept_language "Accept-Language header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header accept_language
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_accept_language(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_accept_language_hash;
}
#else
int sip_is_accept_language(sip_header_t const *header);
#endif

#define sip_accept_language_p(h) sip_is_accept_language((h))


/**Duplicate a list of @ref sip_accept_language "Accept-Language header" header structures #sip_accept_language_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   accept_language = sip_accept_language_dup(home, sip->sip_accept_language);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_accept_language_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_language_t *sip_accept_language_dup(su_home_t *home, sip_accept_language_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_accept_language_t *sip_accept_language_dup(su_home_t *home, sip_accept_language_t const *hdr)
{
  return (sip_accept_language_t *)
    msg_header_dup_as(home, sip_accept_language_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_accept_language "Accept-Language header" header structures #sip_accept_language_t.
 *
 * The function sip_accept_language_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   accept_language = sip_accept_language_copy(home, sip->sip_accept_language);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_language_t *sip_accept_language_copy(su_home_t *home, sip_accept_language_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_accept_language_t *sip_accept_language_copy(su_home_t *home, sip_accept_language_t const *hdr)
{
  return (sip_accept_language_t *)
    msg_header_copy_as(home, sip_accept_language_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_accept_language "Accept-Language header" structure #sip_accept_language_t.
 *
 * The function sip_accept_language_make() makes a new
 * #sip_accept_language_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_accept_language_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_language_t *sip_accept_language_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_accept_language_t *sip_accept_language_make(su_home_t *home, char const *s)
{
  return (sip_accept_language_t *)sip_header_make(home, sip_accept_language_class, s);
}
#endif

/**Make a @ref sip_accept_language "Accept-Language header" from formatting result.
 *
 * Make a new #sip_accept_language_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_accept_language_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_accept_language_t *sip_accept_language_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_accept_language_t *sip_accept_language_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_accept_language_class, fmt, ap);
  va_end(ap);

  return (sip_accept_language_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_allow
 * @{
 */

/** Parse a SIP @ref sip_allow "Allow header". @internal */
SOFIAPUBFUN issize_t sip_allow_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_allow "Allow header". @internal */
SOFIAPUBFUN issize_t sip_allow_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_allow "Allow header"
 * structure #sip_allow_t from #sip_t.
 *
 */
#define sip_allow(sip) \
  ((sip_allow_t *)msg_header_access((msg_pub_t*)(sip), sip_allow_class))

/**Initializer for structure #sip_allow_t.
 *
 * A static #sip_allow_t structure for
 * @ref sip_allow "Allow header" must be initialized with
 * the SIP_ALLOW_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_allow_t sip_allow = SIP_ALLOW_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ALLOW_INIT() SIP_HDR_INIT(allow)

/**Initialize a structure #sip_allow_t.
 *
 * An #sip_allow_t structure for
 * @ref sip_allow "Allow header" can be initialized with the
 * sip_allow_init() function/macro. For instance,
 * @code
 *
 *  sip_allow_t sip_allow;
 *
 *  sip_allow_init(&sip_allow);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_allow_t *sip_allow_init(sip_allow_t x[1])
{
  return SIP_HEADER_INIT(x, sip_allow_class, sizeof(sip_allow_t));
}
#else
#define sip_allow_init(x) \
  SIP_HEADER_INIT(x, sip_allow_class, sizeof(sip_allow_t))
#endif

/**Test if header object is instance of #sip_allow_t.
 *
 * Check if the header class is an instance of
 * @ref sip_allow "Allow header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header allow
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_allow(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_allow_hash;
}
#else
int sip_is_allow(sip_header_t const *header);
#endif

#define sip_allow_p(h) sip_is_allow((h))


/**Duplicate a list of @ref sip_allow "Allow header" header structures #sip_allow_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   allow = sip_allow_dup(home, sip->sip_allow);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_allow_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_allow_t *sip_allow_dup(su_home_t *home, sip_allow_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_allow_t *sip_allow_dup(su_home_t *home, sip_allow_t const *hdr)
{
  return (sip_allow_t *)
    msg_header_dup_as(home, sip_allow_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_allow "Allow header" header structures #sip_allow_t.
 *
 * The function sip_allow_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   allow = sip_allow_copy(home, sip->sip_allow);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_allow_t *sip_allow_copy(su_home_t *home, sip_allow_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_allow_t *sip_allow_copy(su_home_t *home, sip_allow_t const *hdr)
{
  return (sip_allow_t *)
    msg_header_copy_as(home, sip_allow_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_allow "Allow header" structure #sip_allow_t.
 *
 * The function sip_allow_make() makes a new
 * #sip_allow_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_allow_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_allow_t *sip_allow_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_allow_t *sip_allow_make(su_home_t *home, char const *s)
{
  return (sip_allow_t *)sip_header_make(home, sip_allow_class, s);
}
#endif

/**Make a @ref sip_allow "Allow header" from formatting result.
 *
 * Make a new #sip_allow_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_allow_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_allow_t *sip_allow_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_allow_t *sip_allow_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_allow_class, fmt, ap);
  va_end(ap);

  return (sip_allow_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_require
 * @{
 */

/** Parse a SIP @ref sip_require "Require header". @internal */
SOFIAPUBFUN issize_t sip_require_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_require "Require header". @internal */
SOFIAPUBFUN issize_t sip_require_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_require "Require header"
 * structure #sip_require_t from #sip_t.
 *
 */
#define sip_require(sip) \
  ((sip_require_t *)msg_header_access((msg_pub_t*)(sip), sip_require_class))

/**Initializer for structure #sip_require_t.
 *
 * A static #sip_require_t structure for
 * @ref sip_require "Require header" must be initialized with
 * the SIP_REQUIRE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_require_t sip_require = SIP_REQUIRE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_REQUIRE_INIT() SIP_HDR_INIT(require)

/**Initialize a structure #sip_require_t.
 *
 * An #sip_require_t structure for
 * @ref sip_require "Require header" can be initialized with the
 * sip_require_init() function/macro. For instance,
 * @code
 *
 *  sip_require_t sip_require;
 *
 *  sip_require_init(&sip_require);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_require_t *sip_require_init(sip_require_t x[1])
{
  return SIP_HEADER_INIT(x, sip_require_class, sizeof(sip_require_t));
}
#else
#define sip_require_init(x) \
  SIP_HEADER_INIT(x, sip_require_class, sizeof(sip_require_t))
#endif

/**Test if header object is instance of #sip_require_t.
 *
 * Check if the header class is an instance of
 * @ref sip_require "Require header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header require
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_require(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_require_hash;
}
#else
int sip_is_require(sip_header_t const *header);
#endif

#define sip_require_p(h) sip_is_require((h))


/**Duplicate a list of @ref sip_require "Require header" header structures #sip_require_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   require = sip_require_dup(home, sip->sip_require);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_require_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_require_t *sip_require_dup(su_home_t *home, sip_require_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_require_t *sip_require_dup(su_home_t *home, sip_require_t const *hdr)
{
  return (sip_require_t *)
    msg_header_dup_as(home, sip_require_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_require "Require header" header structures #sip_require_t.
 *
 * The function sip_require_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   require = sip_require_copy(home, sip->sip_require);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_require_t *sip_require_copy(su_home_t *home, sip_require_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_require_t *sip_require_copy(su_home_t *home, sip_require_t const *hdr)
{
  return (sip_require_t *)
    msg_header_copy_as(home, sip_require_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_require "Require header" structure #sip_require_t.
 *
 * The function sip_require_make() makes a new
 * #sip_require_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_require_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_require_t *sip_require_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_require_t *sip_require_make(su_home_t *home, char const *s)
{
  return (sip_require_t *)sip_header_make(home, sip_require_class, s);
}
#endif

/**Make a @ref sip_require "Require header" from formatting result.
 *
 * Make a new #sip_require_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_require_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_require_t *sip_require_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_require_t *sip_require_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_require_class, fmt, ap);
  va_end(ap);

  return (sip_require_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_supported
 * @{
 */

/** Parse a SIP @ref sip_supported "Supported header". @internal */
SOFIAPUBFUN issize_t sip_supported_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_supported "Supported header". @internal */
SOFIAPUBFUN issize_t sip_supported_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_supported "Supported header"
 * structure #sip_supported_t from #sip_t.
 *
 */
#define sip_supported(sip) \
  ((sip_supported_t *)msg_header_access((msg_pub_t*)(sip), sip_supported_class))

/**Initializer for structure #sip_supported_t.
 *
 * A static #sip_supported_t structure for
 * @ref sip_supported "Supported header" must be initialized with
 * the SIP_SUPPORTED_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_supported_t sip_supported = SIP_SUPPORTED_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SUPPORTED_INIT() SIP_HDR_INIT(supported)

/**Initialize a structure #sip_supported_t.
 *
 * An #sip_supported_t structure for
 * @ref sip_supported "Supported header" can be initialized with the
 * sip_supported_init() function/macro. For instance,
 * @code
 *
 *  sip_supported_t sip_supported;
 *
 *  sip_supported_init(&sip_supported);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_supported_t *sip_supported_init(sip_supported_t x[1])
{
  return SIP_HEADER_INIT(x, sip_supported_class, sizeof(sip_supported_t));
}
#else
#define sip_supported_init(x) \
  SIP_HEADER_INIT(x, sip_supported_class, sizeof(sip_supported_t))
#endif

/**Test if header object is instance of #sip_supported_t.
 *
 * Check if the header class is an instance of
 * @ref sip_supported "Supported header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header supported
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_supported(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_supported_hash;
}
#else
int sip_is_supported(sip_header_t const *header);
#endif

#define sip_supported_p(h) sip_is_supported((h))


/**Duplicate a list of @ref sip_supported "Supported header" header structures #sip_supported_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   supported = sip_supported_dup(home, sip->sip_supported);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_supported_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_supported_t *sip_supported_dup(su_home_t *home, sip_supported_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_supported_t *sip_supported_dup(su_home_t *home, sip_supported_t const *hdr)
{
  return (sip_supported_t *)
    msg_header_dup_as(home, sip_supported_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_supported "Supported header" header structures #sip_supported_t.
 *
 * The function sip_supported_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   supported = sip_supported_copy(home, sip->sip_supported);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_supported_t *sip_supported_copy(su_home_t *home, sip_supported_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_supported_t *sip_supported_copy(su_home_t *home, sip_supported_t const *hdr)
{
  return (sip_supported_t *)
    msg_header_copy_as(home, sip_supported_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_supported "Supported header" structure #sip_supported_t.
 *
 * The function sip_supported_make() makes a new
 * #sip_supported_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_supported_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_supported_t *sip_supported_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_supported_t *sip_supported_make(su_home_t *home, char const *s)
{
  return (sip_supported_t *)sip_header_make(home, sip_supported_class, s);
}
#endif

/**Make a @ref sip_supported "Supported header" from formatting result.
 *
 * Make a new #sip_supported_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_supported_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_supported_t *sip_supported_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_supported_t *sip_supported_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_supported_class, fmt, ap);
  va_end(ap);

  return (sip_supported_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_unsupported
 * @{
 */

/** Parse a SIP @ref sip_unsupported "Unsupported header". @internal */
SOFIAPUBFUN issize_t sip_unsupported_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_unsupported "Unsupported header". @internal */
SOFIAPUBFUN issize_t sip_unsupported_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_unsupported "Unsupported header"
 * structure #sip_unsupported_t from #sip_t.
 *
 */
#define sip_unsupported(sip) \
  ((sip_unsupported_t *)msg_header_access((msg_pub_t*)(sip), sip_unsupported_class))

/**Initializer for structure #sip_unsupported_t.
 *
 * A static #sip_unsupported_t structure for
 * @ref sip_unsupported "Unsupported header" must be initialized with
 * the SIP_UNSUPPORTED_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_unsupported_t sip_unsupported = SIP_UNSUPPORTED_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_UNSUPPORTED_INIT() SIP_HDR_INIT(unsupported)

/**Initialize a structure #sip_unsupported_t.
 *
 * An #sip_unsupported_t structure for
 * @ref sip_unsupported "Unsupported header" can be initialized with the
 * sip_unsupported_init() function/macro. For instance,
 * @code
 *
 *  sip_unsupported_t sip_unsupported;
 *
 *  sip_unsupported_init(&sip_unsupported);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_unsupported_t *sip_unsupported_init(sip_unsupported_t x[1])
{
  return SIP_HEADER_INIT(x, sip_unsupported_class, sizeof(sip_unsupported_t));
}
#else
#define sip_unsupported_init(x) \
  SIP_HEADER_INIT(x, sip_unsupported_class, sizeof(sip_unsupported_t))
#endif

/**Test if header object is instance of #sip_unsupported_t.
 *
 * Check if the header class is an instance of
 * @ref sip_unsupported "Unsupported header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header unsupported
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_unsupported(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_unsupported_hash;
}
#else
int sip_is_unsupported(sip_header_t const *header);
#endif

#define sip_unsupported_p(h) sip_is_unsupported((h))


/**Duplicate a list of @ref sip_unsupported "Unsupported header" header structures #sip_unsupported_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   unsupported = sip_unsupported_dup(home, sip->sip_unsupported);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_unsupported_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_unsupported_t *sip_unsupported_dup(su_home_t *home, sip_unsupported_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_unsupported_t *sip_unsupported_dup(su_home_t *home, sip_unsupported_t const *hdr)
{
  return (sip_unsupported_t *)
    msg_header_dup_as(home, sip_unsupported_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_unsupported "Unsupported header" header structures #sip_unsupported_t.
 *
 * The function sip_unsupported_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   unsupported = sip_unsupported_copy(home, sip->sip_unsupported);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_unsupported_t *sip_unsupported_copy(su_home_t *home, sip_unsupported_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_unsupported_t *sip_unsupported_copy(su_home_t *home, sip_unsupported_t const *hdr)
{
  return (sip_unsupported_t *)
    msg_header_copy_as(home, sip_unsupported_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_unsupported "Unsupported header" structure #sip_unsupported_t.
 *
 * The function sip_unsupported_make() makes a new
 * #sip_unsupported_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_unsupported_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_unsupported_t *sip_unsupported_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_unsupported_t *sip_unsupported_make(su_home_t *home, char const *s)
{
  return (sip_unsupported_t *)sip_header_make(home, sip_unsupported_class, s);
}
#endif

/**Make a @ref sip_unsupported "Unsupported header" from formatting result.
 *
 * Make a new #sip_unsupported_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_unsupported_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_unsupported_t *sip_unsupported_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_unsupported_t *sip_unsupported_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_unsupported_class, fmt, ap);
  va_end(ap);

  return (sip_unsupported_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_event
 * @{
 */

/** Parse a SIP @ref sip_event "Event header". @internal */
SOFIAPUBFUN issize_t sip_event_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_event "Event header". @internal */
SOFIAPUBFUN issize_t sip_event_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_event "Event header"
 * structure #sip_event_t from #sip_t.
 *
 */
#define sip_event(sip) \
  ((sip_event_t *)msg_header_access((msg_pub_t*)(sip), sip_event_class))

/**Initializer for structure #sip_event_t.
 *
 * A static #sip_event_t structure for
 * @ref sip_event "Event header" must be initialized with
 * the SIP_EVENT_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_event_t sip_event = SIP_EVENT_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_EVENT_INIT() SIP_HDR_INIT(event)

/**Initialize a structure #sip_event_t.
 *
 * An #sip_event_t structure for
 * @ref sip_event "Event header" can be initialized with the
 * sip_event_init() function/macro. For instance,
 * @code
 *
 *  sip_event_t sip_event;
 *
 *  sip_event_init(&sip_event);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_event_t *sip_event_init(sip_event_t x[1])
{
  return SIP_HEADER_INIT(x, sip_event_class, sizeof(sip_event_t));
}
#else
#define sip_event_init(x) \
  SIP_HEADER_INIT(x, sip_event_class, sizeof(sip_event_t))
#endif

/**Test if header object is instance of #sip_event_t.
 *
 * Check if the header class is an instance of
 * @ref sip_event "Event header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header event
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_event(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_event_hash;
}
#else
int sip_is_event(sip_header_t const *header);
#endif

#define sip_event_p(h) sip_is_event((h))


/**Duplicate a list of @ref sip_event "Event header" header structures #sip_event_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   event = sip_event_dup(home, sip->sip_event);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_event_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_event_t *sip_event_dup(su_home_t *home, sip_event_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_event_t *sip_event_dup(su_home_t *home, sip_event_t const *hdr)
{
  return (sip_event_t *)
    msg_header_dup_as(home, sip_event_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_event "Event header" header structures #sip_event_t.
 *
 * The function sip_event_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   event = sip_event_copy(home, sip->sip_event);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_event_t *sip_event_copy(su_home_t *home, sip_event_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_event_t *sip_event_copy(su_home_t *home, sip_event_t const *hdr)
{
  return (sip_event_t *)
    msg_header_copy_as(home, sip_event_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_event "Event header" structure #sip_event_t.
 *
 * The function sip_event_make() makes a new
 * #sip_event_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_event_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_event_t *sip_event_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_event_t *sip_event_make(su_home_t *home, char const *s)
{
  return (sip_event_t *)sip_header_make(home, sip_event_class, s);
}
#endif

/**Make a @ref sip_event "Event header" from formatting result.
 *
 * Make a new #sip_event_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_event_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_event_t *sip_event_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_event_t *sip_event_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_event_class, fmt, ap);
  va_end(ap);

  return (sip_event_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_allow_events
 * @{
 */

/** Parse a SIP @ref sip_allow_events "Allow-Events header". @internal */
SOFIAPUBFUN issize_t sip_allow_events_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_allow_events "Allow-Events header". @internal */
SOFIAPUBFUN issize_t sip_allow_events_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_allow_events "Allow-Events header"
 * structure #sip_allow_events_t from #sip_t.
 *
 */
#define sip_allow_events(sip) \
  ((sip_allow_events_t *)msg_header_access((msg_pub_t*)(sip), sip_allow_events_class))

/**Initializer for structure #sip_allow_events_t.
 *
 * A static #sip_allow_events_t structure for
 * @ref sip_allow_events "Allow-Events header" must be initialized with
 * the SIP_ALLOW_EVENTS_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_allow_events_t sip_allow_events = SIP_ALLOW_EVENTS_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ALLOW_EVENTS_INIT() SIP_HDR_INIT(allow_events)

/**Initialize a structure #sip_allow_events_t.
 *
 * An #sip_allow_events_t structure for
 * @ref sip_allow_events "Allow-Events header" can be initialized with the
 * sip_allow_events_init() function/macro. For instance,
 * @code
 *
 *  sip_allow_events_t sip_allow_events;
 *
 *  sip_allow_events_init(&sip_allow_events);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_allow_events_t *sip_allow_events_init(sip_allow_events_t x[1])
{
  return SIP_HEADER_INIT(x, sip_allow_events_class, sizeof(sip_allow_events_t));
}
#else
#define sip_allow_events_init(x) \
  SIP_HEADER_INIT(x, sip_allow_events_class, sizeof(sip_allow_events_t))
#endif

/**Test if header object is instance of #sip_allow_events_t.
 *
 * Check if the header class is an instance of
 * @ref sip_allow_events "Allow-Events header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header allow_events
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_allow_events(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_allow_events_hash;
}
#else
int sip_is_allow_events(sip_header_t const *header);
#endif

#define sip_allow_events_p(h) sip_is_allow_events((h))


/**Duplicate a list of @ref sip_allow_events "Allow-Events header" header structures #sip_allow_events_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   allow_events = sip_allow_events_dup(home, sip->sip_allow_events);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_allow_events_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_allow_events_t *sip_allow_events_dup(su_home_t *home, sip_allow_events_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_allow_events_t *sip_allow_events_dup(su_home_t *home, sip_allow_events_t const *hdr)
{
  return (sip_allow_events_t *)
    msg_header_dup_as(home, sip_allow_events_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_allow_events "Allow-Events header" header structures #sip_allow_events_t.
 *
 * The function sip_allow_events_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   allow_events = sip_allow_events_copy(home, sip->sip_allow_events);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_allow_events_t *sip_allow_events_copy(su_home_t *home, sip_allow_events_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_allow_events_t *sip_allow_events_copy(su_home_t *home, sip_allow_events_t const *hdr)
{
  return (sip_allow_events_t *)
    msg_header_copy_as(home, sip_allow_events_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_allow_events "Allow-Events header" structure #sip_allow_events_t.
 *
 * The function sip_allow_events_make() makes a new
 * #sip_allow_events_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_allow_events_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_allow_events_t *sip_allow_events_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_allow_events_t *sip_allow_events_make(su_home_t *home, char const *s)
{
  return (sip_allow_events_t *)sip_header_make(home, sip_allow_events_class, s);
}
#endif

/**Make a @ref sip_allow_events "Allow-Events header" from formatting result.
 *
 * Make a new #sip_allow_events_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_allow_events_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_allow_events_t *sip_allow_events_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_allow_events_t *sip_allow_events_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_allow_events_class, fmt, ap);
  va_end(ap);

  return (sip_allow_events_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_subscription_state
 * @{
 */

/** Parse a SIP @ref sip_subscription_state "Subscription-State header". @internal */
SOFIAPUBFUN issize_t sip_subscription_state_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_subscription_state "Subscription-State header". @internal */
SOFIAPUBFUN issize_t sip_subscription_state_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_subscription_state "Subscription-State header"
 * structure #sip_subscription_state_t from #sip_t.
 *
 */
#define sip_subscription_state(sip) \
  ((sip_subscription_state_t *)msg_header_access((msg_pub_t*)(sip), sip_subscription_state_class))

/**Initializer for structure #sip_subscription_state_t.
 *
 * A static #sip_subscription_state_t structure for
 * @ref sip_subscription_state "Subscription-State header" must be initialized with
 * the SIP_SUBSCRIPTION_STATE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_subscription_state_t sip_subscription_state = SIP_SUBSCRIPTION_STATE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SUBSCRIPTION_STATE_INIT() SIP_HDR_INIT(subscription_state)

/**Initialize a structure #sip_subscription_state_t.
 *
 * An #sip_subscription_state_t structure for
 * @ref sip_subscription_state "Subscription-State header" can be initialized with the
 * sip_subscription_state_init() function/macro. For instance,
 * @code
 *
 *  sip_subscription_state_t sip_subscription_state;
 *
 *  sip_subscription_state_init(&sip_subscription_state);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_subscription_state_t *sip_subscription_state_init(sip_subscription_state_t x[1])
{
  return SIP_HEADER_INIT(x, sip_subscription_state_class, sizeof(sip_subscription_state_t));
}
#else
#define sip_subscription_state_init(x) \
  SIP_HEADER_INIT(x, sip_subscription_state_class, sizeof(sip_subscription_state_t))
#endif

/**Test if header object is instance of #sip_subscription_state_t.
 *
 * Check if the header class is an instance of
 * @ref sip_subscription_state "Subscription-State header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header subscription_state
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_subscription_state(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_subscription_state_hash;
}
#else
int sip_is_subscription_state(sip_header_t const *header);
#endif

#define sip_subscription_state_p(h) sip_is_subscription_state((h))


/**Duplicate a list of @ref sip_subscription_state "Subscription-State header" header structures #sip_subscription_state_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   subscription_state = sip_subscription_state_dup(home, sip->sip_subscription_state);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_subscription_state_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_subscription_state_t *sip_subscription_state_dup(su_home_t *home, sip_subscription_state_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_subscription_state_t *sip_subscription_state_dup(su_home_t *home, sip_subscription_state_t const *hdr)
{
  return (sip_subscription_state_t *)
    msg_header_dup_as(home, sip_subscription_state_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_subscription_state "Subscription-State header" header structures #sip_subscription_state_t.
 *
 * The function sip_subscription_state_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   subscription_state = sip_subscription_state_copy(home, sip->sip_subscription_state);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_subscription_state_t *sip_subscription_state_copy(su_home_t *home, sip_subscription_state_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_subscription_state_t *sip_subscription_state_copy(su_home_t *home, sip_subscription_state_t const *hdr)
{
  return (sip_subscription_state_t *)
    msg_header_copy_as(home, sip_subscription_state_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_subscription_state "Subscription-State header" structure #sip_subscription_state_t.
 *
 * The function sip_subscription_state_make() makes a new
 * #sip_subscription_state_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_subscription_state_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_subscription_state_t *sip_subscription_state_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_subscription_state_t *sip_subscription_state_make(su_home_t *home, char const *s)
{
  return (sip_subscription_state_t *)sip_header_make(home, sip_subscription_state_class, s);
}
#endif

/**Make a @ref sip_subscription_state "Subscription-State header" from formatting result.
 *
 * Make a new #sip_subscription_state_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_subscription_state_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_subscription_state_t *sip_subscription_state_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_subscription_state_t *sip_subscription_state_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_subscription_state_class, fmt, ap);
  va_end(ap);

  return (sip_subscription_state_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_proxy_authenticate
 * @{
 */

/** Parse a SIP @ref sip_proxy_authenticate "Proxy-Authenticate header". @internal */
SOFIAPUBFUN issize_t sip_proxy_authenticate_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_proxy_authenticate "Proxy-Authenticate header". @internal */
SOFIAPUBFUN issize_t sip_proxy_authenticate_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_proxy_authenticate "Proxy-Authenticate header"
 * structure #sip_proxy_authenticate_t from #sip_t.
 *
 */
#define sip_proxy_authenticate(sip) \
  ((sip_proxy_authenticate_t *)msg_header_access((msg_pub_t*)(sip), sip_proxy_authenticate_class))

/**Initializer for structure #sip_proxy_authenticate_t.
 *
 * A static #sip_proxy_authenticate_t structure for
 * @ref sip_proxy_authenticate "Proxy-Authenticate header" must be initialized with
 * the SIP_PROXY_AUTHENTICATE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_proxy_authenticate_t sip_proxy_authenticate = SIP_PROXY_AUTHENTICATE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_PROXY_AUTHENTICATE_INIT() SIP_HDR_INIT(proxy_authenticate)

/**Initialize a structure #sip_proxy_authenticate_t.
 *
 * An #sip_proxy_authenticate_t structure for
 * @ref sip_proxy_authenticate "Proxy-Authenticate header" can be initialized with the
 * sip_proxy_authenticate_init() function/macro. For instance,
 * @code
 *
 *  sip_proxy_authenticate_t sip_proxy_authenticate;
 *
 *  sip_proxy_authenticate_init(&sip_proxy_authenticate);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_proxy_authenticate_t *sip_proxy_authenticate_init(sip_proxy_authenticate_t x[1])
{
  return SIP_HEADER_INIT(x, sip_proxy_authenticate_class, sizeof(sip_proxy_authenticate_t));
}
#else
#define sip_proxy_authenticate_init(x) \
  SIP_HEADER_INIT(x, sip_proxy_authenticate_class, sizeof(sip_proxy_authenticate_t))
#endif

/**Test if header object is instance of #sip_proxy_authenticate_t.
 *
 * Check if the header class is an instance of
 * @ref sip_proxy_authenticate "Proxy-Authenticate header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header proxy_authenticate
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_proxy_authenticate(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_proxy_authenticate_hash;
}
#else
int sip_is_proxy_authenticate(sip_header_t const *header);
#endif

#define sip_proxy_authenticate_p(h) sip_is_proxy_authenticate((h))


/**Duplicate a list of @ref sip_proxy_authenticate "Proxy-Authenticate header" header structures #sip_proxy_authenticate_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   proxy_authenticate = sip_proxy_authenticate_dup(home, sip->sip_proxy_authenticate);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_proxy_authenticate_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authenticate_t *sip_proxy_authenticate_dup(su_home_t *home, sip_proxy_authenticate_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_proxy_authenticate_t *sip_proxy_authenticate_dup(su_home_t *home, sip_proxy_authenticate_t const *hdr)
{
  return (sip_proxy_authenticate_t *)
    msg_header_dup_as(home, sip_proxy_authenticate_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_proxy_authenticate "Proxy-Authenticate header" header structures #sip_proxy_authenticate_t.
 *
 * The function sip_proxy_authenticate_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   proxy_authenticate = sip_proxy_authenticate_copy(home, sip->sip_proxy_authenticate);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authenticate_t *sip_proxy_authenticate_copy(su_home_t *home, sip_proxy_authenticate_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_proxy_authenticate_t *sip_proxy_authenticate_copy(su_home_t *home, sip_proxy_authenticate_t const *hdr)
{
  return (sip_proxy_authenticate_t *)
    msg_header_copy_as(home, sip_proxy_authenticate_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_proxy_authenticate "Proxy-Authenticate header" structure #sip_proxy_authenticate_t.
 *
 * The function sip_proxy_authenticate_make() makes a new
 * #sip_proxy_authenticate_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_proxy_authenticate_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authenticate_t *sip_proxy_authenticate_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_proxy_authenticate_t *sip_proxy_authenticate_make(su_home_t *home, char const *s)
{
  return (sip_proxy_authenticate_t *)sip_header_make(home, sip_proxy_authenticate_class, s);
}
#endif

/**Make a @ref sip_proxy_authenticate "Proxy-Authenticate header" from formatting result.
 *
 * Make a new #sip_proxy_authenticate_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_proxy_authenticate_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authenticate_t *sip_proxy_authenticate_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_proxy_authenticate_t *sip_proxy_authenticate_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_proxy_authenticate_class, fmt, ap);
  va_end(ap);

  return (sip_proxy_authenticate_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_proxy_authentication_info
 * @{
 */

/** Parse a SIP @ref sip_proxy_authentication_info "Proxy-Authentication-Info header". @internal */
SOFIAPUBFUN issize_t sip_proxy_authentication_info_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_proxy_authentication_info "Proxy-Authentication-Info header". @internal */
SOFIAPUBFUN issize_t sip_proxy_authentication_info_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_proxy_authentication_info "Proxy-Authentication-Info header"
 * structure #sip_proxy_authentication_info_t from #sip_t.
 *
 */
#define sip_proxy_authentication_info(sip) \
  ((sip_proxy_authentication_info_t *)msg_header_access((msg_pub_t*)(sip), sip_proxy_authentication_info_class))

/**Initializer for structure #sip_proxy_authentication_info_t.
 *
 * A static #sip_proxy_authentication_info_t structure for
 * @ref sip_proxy_authentication_info "Proxy-Authentication-Info header" must be initialized with
 * the SIP_PROXY_AUTHENTICATION_INFO_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_proxy_authentication_info_t sip_proxy_authentication_info = SIP_PROXY_AUTHENTICATION_INFO_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_PROXY_AUTHENTICATION_INFO_INIT() SIP_HDR_INIT(proxy_authentication_info)

/**Initialize a structure #sip_proxy_authentication_info_t.
 *
 * An #sip_proxy_authentication_info_t structure for
 * @ref sip_proxy_authentication_info "Proxy-Authentication-Info header" can be initialized with the
 * sip_proxy_authentication_info_init() function/macro. For instance,
 * @code
 *
 *  sip_proxy_authentication_info_t sip_proxy_authentication_info;
 *
 *  sip_proxy_authentication_info_init(&sip_proxy_authentication_info);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_proxy_authentication_info_t *sip_proxy_authentication_info_init(sip_proxy_authentication_info_t x[1])
{
  return SIP_HEADER_INIT(x, sip_proxy_authentication_info_class, sizeof(sip_proxy_authentication_info_t));
}
#else
#define sip_proxy_authentication_info_init(x) \
  SIP_HEADER_INIT(x, sip_proxy_authentication_info_class, sizeof(sip_proxy_authentication_info_t))
#endif

/**Test if header object is instance of #sip_proxy_authentication_info_t.
 *
 * Check if the header class is an instance of
 * @ref sip_proxy_authentication_info "Proxy-Authentication-Info header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header proxy_authentication_info
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_proxy_authentication_info(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_proxy_authentication_info_hash;
}
#else
int sip_is_proxy_authentication_info(sip_header_t const *header);
#endif

#define sip_proxy_authentication_info_p(h) sip_is_proxy_authentication_info((h))


/**Duplicate a list of @ref sip_proxy_authentication_info "Proxy-Authentication-Info header" header structures #sip_proxy_authentication_info_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   proxy_authentication_info = sip_proxy_authentication_info_dup(home, sip->sip_proxy_authentication_info);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_proxy_authentication_info_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authentication_info_t *sip_proxy_authentication_info_dup(su_home_t *home, sip_proxy_authentication_info_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_proxy_authentication_info_t *sip_proxy_authentication_info_dup(su_home_t *home, sip_proxy_authentication_info_t const *hdr)
{
  return (sip_proxy_authentication_info_t *)
    msg_header_dup_as(home, sip_proxy_authentication_info_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_proxy_authentication_info "Proxy-Authentication-Info header" header structures #sip_proxy_authentication_info_t.
 *
 * The function sip_proxy_authentication_info_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   proxy_authentication_info = sip_proxy_authentication_info_copy(home, sip->sip_proxy_authentication_info);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authentication_info_t *sip_proxy_authentication_info_copy(su_home_t *home, sip_proxy_authentication_info_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_proxy_authentication_info_t *sip_proxy_authentication_info_copy(su_home_t *home, sip_proxy_authentication_info_t const *hdr)
{
  return (sip_proxy_authentication_info_t *)
    msg_header_copy_as(home, sip_proxy_authentication_info_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_proxy_authentication_info "Proxy-Authentication-Info header" structure #sip_proxy_authentication_info_t.
 *
 * The function sip_proxy_authentication_info_make() makes a new
 * #sip_proxy_authentication_info_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_proxy_authentication_info_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authentication_info_t *sip_proxy_authentication_info_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_proxy_authentication_info_t *sip_proxy_authentication_info_make(su_home_t *home, char const *s)
{
  return (sip_proxy_authentication_info_t *)sip_header_make(home, sip_proxy_authentication_info_class, s);
}
#endif

/**Make a @ref sip_proxy_authentication_info "Proxy-Authentication-Info header" from formatting result.
 *
 * Make a new #sip_proxy_authentication_info_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_proxy_authentication_info_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authentication_info_t *sip_proxy_authentication_info_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_proxy_authentication_info_t *sip_proxy_authentication_info_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_proxy_authentication_info_class, fmt, ap);
  va_end(ap);

  return (sip_proxy_authentication_info_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_proxy_authorization
 * @{
 */

/** Parse a SIP @ref sip_proxy_authorization "Proxy-Authorization header". @internal */
SOFIAPUBFUN issize_t sip_proxy_authorization_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_proxy_authorization "Proxy-Authorization header". @internal */
SOFIAPUBFUN issize_t sip_proxy_authorization_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_proxy_authorization "Proxy-Authorization header"
 * structure #sip_proxy_authorization_t from #sip_t.
 *
 */
#define sip_proxy_authorization(sip) \
  ((sip_proxy_authorization_t *)msg_header_access((msg_pub_t*)(sip), sip_proxy_authorization_class))

/**Initializer for structure #sip_proxy_authorization_t.
 *
 * A static #sip_proxy_authorization_t structure for
 * @ref sip_proxy_authorization "Proxy-Authorization header" must be initialized with
 * the SIP_PROXY_AUTHORIZATION_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_proxy_authorization_t sip_proxy_authorization = SIP_PROXY_AUTHORIZATION_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_PROXY_AUTHORIZATION_INIT() SIP_HDR_INIT(proxy_authorization)

/**Initialize a structure #sip_proxy_authorization_t.
 *
 * An #sip_proxy_authorization_t structure for
 * @ref sip_proxy_authorization "Proxy-Authorization header" can be initialized with the
 * sip_proxy_authorization_init() function/macro. For instance,
 * @code
 *
 *  sip_proxy_authorization_t sip_proxy_authorization;
 *
 *  sip_proxy_authorization_init(&sip_proxy_authorization);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_proxy_authorization_t *sip_proxy_authorization_init(sip_proxy_authorization_t x[1])
{
  return SIP_HEADER_INIT(x, sip_proxy_authorization_class, sizeof(sip_proxy_authorization_t));
}
#else
#define sip_proxy_authorization_init(x) \
  SIP_HEADER_INIT(x, sip_proxy_authorization_class, sizeof(sip_proxy_authorization_t))
#endif

/**Test if header object is instance of #sip_proxy_authorization_t.
 *
 * Check if the header class is an instance of
 * @ref sip_proxy_authorization "Proxy-Authorization header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header proxy_authorization
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_proxy_authorization(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_proxy_authorization_hash;
}
#else
int sip_is_proxy_authorization(sip_header_t const *header);
#endif

#define sip_proxy_authorization_p(h) sip_is_proxy_authorization((h))


/**Duplicate a list of @ref sip_proxy_authorization "Proxy-Authorization header" header structures #sip_proxy_authorization_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   proxy_authorization = sip_proxy_authorization_dup(home, sip->sip_proxy_authorization);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_proxy_authorization_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authorization_t *sip_proxy_authorization_dup(su_home_t *home, sip_proxy_authorization_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_proxy_authorization_t *sip_proxy_authorization_dup(su_home_t *home, sip_proxy_authorization_t const *hdr)
{
  return (sip_proxy_authorization_t *)
    msg_header_dup_as(home, sip_proxy_authorization_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_proxy_authorization "Proxy-Authorization header" header structures #sip_proxy_authorization_t.
 *
 * The function sip_proxy_authorization_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   proxy_authorization = sip_proxy_authorization_copy(home, sip->sip_proxy_authorization);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authorization_t *sip_proxy_authorization_copy(su_home_t *home, sip_proxy_authorization_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_proxy_authorization_t *sip_proxy_authorization_copy(su_home_t *home, sip_proxy_authorization_t const *hdr)
{
  return (sip_proxy_authorization_t *)
    msg_header_copy_as(home, sip_proxy_authorization_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_proxy_authorization "Proxy-Authorization header" structure #sip_proxy_authorization_t.
 *
 * The function sip_proxy_authorization_make() makes a new
 * #sip_proxy_authorization_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_proxy_authorization_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authorization_t *sip_proxy_authorization_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_proxy_authorization_t *sip_proxy_authorization_make(su_home_t *home, char const *s)
{
  return (sip_proxy_authorization_t *)sip_header_make(home, sip_proxy_authorization_class, s);
}
#endif

/**Make a @ref sip_proxy_authorization "Proxy-Authorization header" from formatting result.
 *
 * Make a new #sip_proxy_authorization_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_proxy_authorization_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_proxy_authorization_t *sip_proxy_authorization_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_proxy_authorization_t *sip_proxy_authorization_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_proxy_authorization_class, fmt, ap);
  va_end(ap);

  return (sip_proxy_authorization_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_authorization
 * @{
 */

/** Parse a SIP @ref sip_authorization "Authorization header". @internal */
SOFIAPUBFUN issize_t sip_authorization_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_authorization "Authorization header". @internal */
SOFIAPUBFUN issize_t sip_authorization_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_authorization "Authorization header"
 * structure #sip_authorization_t from #sip_t.
 *
 */
#define sip_authorization(sip) \
  ((sip_authorization_t *)msg_header_access((msg_pub_t*)(sip), sip_authorization_class))

/**Initializer for structure #sip_authorization_t.
 *
 * A static #sip_authorization_t structure for
 * @ref sip_authorization "Authorization header" must be initialized with
 * the SIP_AUTHORIZATION_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_authorization_t sip_authorization = SIP_AUTHORIZATION_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_AUTHORIZATION_INIT() SIP_HDR_INIT(authorization)

/**Initialize a structure #sip_authorization_t.
 *
 * An #sip_authorization_t structure for
 * @ref sip_authorization "Authorization header" can be initialized with the
 * sip_authorization_init() function/macro. For instance,
 * @code
 *
 *  sip_authorization_t sip_authorization;
 *
 *  sip_authorization_init(&sip_authorization);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_authorization_t *sip_authorization_init(sip_authorization_t x[1])
{
  return SIP_HEADER_INIT(x, sip_authorization_class, sizeof(sip_authorization_t));
}
#else
#define sip_authorization_init(x) \
  SIP_HEADER_INIT(x, sip_authorization_class, sizeof(sip_authorization_t))
#endif

/**Test if header object is instance of #sip_authorization_t.
 *
 * Check if the header class is an instance of
 * @ref sip_authorization "Authorization header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header authorization
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_authorization(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_authorization_hash;
}
#else
int sip_is_authorization(sip_header_t const *header);
#endif

#define sip_authorization_p(h) sip_is_authorization((h))


/**Duplicate a list of @ref sip_authorization "Authorization header" header structures #sip_authorization_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   authorization = sip_authorization_dup(home, sip->sip_authorization);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_authorization_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_authorization_t *sip_authorization_dup(su_home_t *home, sip_authorization_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_authorization_t *sip_authorization_dup(su_home_t *home, sip_authorization_t const *hdr)
{
  return (sip_authorization_t *)
    msg_header_dup_as(home, sip_authorization_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_authorization "Authorization header" header structures #sip_authorization_t.
 *
 * The function sip_authorization_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   authorization = sip_authorization_copy(home, sip->sip_authorization);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_authorization_t *sip_authorization_copy(su_home_t *home, sip_authorization_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_authorization_t *sip_authorization_copy(su_home_t *home, sip_authorization_t const *hdr)
{
  return (sip_authorization_t *)
    msg_header_copy_as(home, sip_authorization_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_authorization "Authorization header" structure #sip_authorization_t.
 *
 * The function sip_authorization_make() makes a new
 * #sip_authorization_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_authorization_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_authorization_t *sip_authorization_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_authorization_t *sip_authorization_make(su_home_t *home, char const *s)
{
  return (sip_authorization_t *)sip_header_make(home, sip_authorization_class, s);
}
#endif

/**Make a @ref sip_authorization "Authorization header" from formatting result.
 *
 * Make a new #sip_authorization_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_authorization_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_authorization_t *sip_authorization_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_authorization_t *sip_authorization_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_authorization_class, fmt, ap);
  va_end(ap);

  return (sip_authorization_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_www_authenticate
 * @{
 */

/** Parse a SIP @ref sip_www_authenticate "WWW-Authenticate header". @internal */
SOFIAPUBFUN issize_t sip_www_authenticate_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_www_authenticate "WWW-Authenticate header". @internal */
SOFIAPUBFUN issize_t sip_www_authenticate_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_www_authenticate "WWW-Authenticate header"
 * structure #sip_www_authenticate_t from #sip_t.
 *
 */
#define sip_www_authenticate(sip) \
  ((sip_www_authenticate_t *)msg_header_access((msg_pub_t*)(sip), sip_www_authenticate_class))

/**Initializer for structure #sip_www_authenticate_t.
 *
 * A static #sip_www_authenticate_t structure for
 * @ref sip_www_authenticate "WWW-Authenticate header" must be initialized with
 * the SIP_WWW_AUTHENTICATE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_www_authenticate_t sip_www_authenticate = SIP_WWW_AUTHENTICATE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_WWW_AUTHENTICATE_INIT() SIP_HDR_INIT(www_authenticate)

/**Initialize a structure #sip_www_authenticate_t.
 *
 * An #sip_www_authenticate_t structure for
 * @ref sip_www_authenticate "WWW-Authenticate header" can be initialized with the
 * sip_www_authenticate_init() function/macro. For instance,
 * @code
 *
 *  sip_www_authenticate_t sip_www_authenticate;
 *
 *  sip_www_authenticate_init(&sip_www_authenticate);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_www_authenticate_t *sip_www_authenticate_init(sip_www_authenticate_t x[1])
{
  return SIP_HEADER_INIT(x, sip_www_authenticate_class, sizeof(sip_www_authenticate_t));
}
#else
#define sip_www_authenticate_init(x) \
  SIP_HEADER_INIT(x, sip_www_authenticate_class, sizeof(sip_www_authenticate_t))
#endif

/**Test if header object is instance of #sip_www_authenticate_t.
 *
 * Check if the header class is an instance of
 * @ref sip_www_authenticate "WWW-Authenticate header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header www_authenticate
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_www_authenticate(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_www_authenticate_hash;
}
#else
int sip_is_www_authenticate(sip_header_t const *header);
#endif

#define sip_www_authenticate_p(h) sip_is_www_authenticate((h))


/**Duplicate a list of @ref sip_www_authenticate "WWW-Authenticate header" header structures #sip_www_authenticate_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   www_authenticate = sip_www_authenticate_dup(home, sip->sip_www_authenticate);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_www_authenticate_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_www_authenticate_t *sip_www_authenticate_dup(su_home_t *home, sip_www_authenticate_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_www_authenticate_t *sip_www_authenticate_dup(su_home_t *home, sip_www_authenticate_t const *hdr)
{
  return (sip_www_authenticate_t *)
    msg_header_dup_as(home, sip_www_authenticate_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_www_authenticate "WWW-Authenticate header" header structures #sip_www_authenticate_t.
 *
 * The function sip_www_authenticate_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   www_authenticate = sip_www_authenticate_copy(home, sip->sip_www_authenticate);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_www_authenticate_t *sip_www_authenticate_copy(su_home_t *home, sip_www_authenticate_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_www_authenticate_t *sip_www_authenticate_copy(su_home_t *home, sip_www_authenticate_t const *hdr)
{
  return (sip_www_authenticate_t *)
    msg_header_copy_as(home, sip_www_authenticate_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_www_authenticate "WWW-Authenticate header" structure #sip_www_authenticate_t.
 *
 * The function sip_www_authenticate_make() makes a new
 * #sip_www_authenticate_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_www_authenticate_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_www_authenticate_t *sip_www_authenticate_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_www_authenticate_t *sip_www_authenticate_make(su_home_t *home, char const *s)
{
  return (sip_www_authenticate_t *)sip_header_make(home, sip_www_authenticate_class, s);
}
#endif

/**Make a @ref sip_www_authenticate "WWW-Authenticate header" from formatting result.
 *
 * Make a new #sip_www_authenticate_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_www_authenticate_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_www_authenticate_t *sip_www_authenticate_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_www_authenticate_t *sip_www_authenticate_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_www_authenticate_class, fmt, ap);
  va_end(ap);

  return (sip_www_authenticate_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_authentication_info
 * @{
 */

/** Parse a SIP @ref sip_authentication_info "Authentication-Info header". @internal */
SOFIAPUBFUN issize_t sip_authentication_info_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_authentication_info "Authentication-Info header". @internal */
SOFIAPUBFUN issize_t sip_authentication_info_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_authentication_info "Authentication-Info header"
 * structure #sip_authentication_info_t from #sip_t.
 *
 */
#define sip_authentication_info(sip) \
  ((sip_authentication_info_t *)msg_header_access((msg_pub_t*)(sip), sip_authentication_info_class))

/**Initializer for structure #sip_authentication_info_t.
 *
 * A static #sip_authentication_info_t structure for
 * @ref sip_authentication_info "Authentication-Info header" must be initialized with
 * the SIP_AUTHENTICATION_INFO_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_authentication_info_t sip_authentication_info = SIP_AUTHENTICATION_INFO_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_AUTHENTICATION_INFO_INIT() SIP_HDR_INIT(authentication_info)

/**Initialize a structure #sip_authentication_info_t.
 *
 * An #sip_authentication_info_t structure for
 * @ref sip_authentication_info "Authentication-Info header" can be initialized with the
 * sip_authentication_info_init() function/macro. For instance,
 * @code
 *
 *  sip_authentication_info_t sip_authentication_info;
 *
 *  sip_authentication_info_init(&sip_authentication_info);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_authentication_info_t *sip_authentication_info_init(sip_authentication_info_t x[1])
{
  return SIP_HEADER_INIT(x, sip_authentication_info_class, sizeof(sip_authentication_info_t));
}
#else
#define sip_authentication_info_init(x) \
  SIP_HEADER_INIT(x, sip_authentication_info_class, sizeof(sip_authentication_info_t))
#endif

/**Test if header object is instance of #sip_authentication_info_t.
 *
 * Check if the header class is an instance of
 * @ref sip_authentication_info "Authentication-Info header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header authentication_info
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_authentication_info(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_authentication_info_hash;
}
#else
int sip_is_authentication_info(sip_header_t const *header);
#endif

#define sip_authentication_info_p(h) sip_is_authentication_info((h))


/**Duplicate a list of @ref sip_authentication_info "Authentication-Info header" header structures #sip_authentication_info_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   authentication_info = sip_authentication_info_dup(home, sip->sip_authentication_info);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_authentication_info_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_authentication_info_t *sip_authentication_info_dup(su_home_t *home, sip_authentication_info_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_authentication_info_t *sip_authentication_info_dup(su_home_t *home, sip_authentication_info_t const *hdr)
{
  return (sip_authentication_info_t *)
    msg_header_dup_as(home, sip_authentication_info_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_authentication_info "Authentication-Info header" header structures #sip_authentication_info_t.
 *
 * The function sip_authentication_info_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   authentication_info = sip_authentication_info_copy(home, sip->sip_authentication_info);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_authentication_info_t *sip_authentication_info_copy(su_home_t *home, sip_authentication_info_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_authentication_info_t *sip_authentication_info_copy(su_home_t *home, sip_authentication_info_t const *hdr)
{
  return (sip_authentication_info_t *)
    msg_header_copy_as(home, sip_authentication_info_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_authentication_info "Authentication-Info header" structure #sip_authentication_info_t.
 *
 * The function sip_authentication_info_make() makes a new
 * #sip_authentication_info_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_authentication_info_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_authentication_info_t *sip_authentication_info_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_authentication_info_t *sip_authentication_info_make(su_home_t *home, char const *s)
{
  return (sip_authentication_info_t *)sip_header_make(home, sip_authentication_info_class, s);
}
#endif

/**Make a @ref sip_authentication_info "Authentication-Info header" from formatting result.
 *
 * Make a new #sip_authentication_info_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_authentication_info_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_authentication_info_t *sip_authentication_info_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_authentication_info_t *sip_authentication_info_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_authentication_info_class, fmt, ap);
  va_end(ap);

  return (sip_authentication_info_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_error_info
 * @{
 */

/** Parse a SIP @ref sip_error_info "Error-Info header". @internal */
SOFIAPUBFUN issize_t sip_error_info_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_error_info "Error-Info header". @internal */
SOFIAPUBFUN issize_t sip_error_info_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_error_info "Error-Info header"
 * structure #sip_error_info_t from #sip_t.
 *
 */
#define sip_error_info(sip) \
  ((sip_error_info_t *)msg_header_access((msg_pub_t*)(sip), sip_error_info_class))

/**Initializer for structure #sip_error_info_t.
 *
 * A static #sip_error_info_t structure for
 * @ref sip_error_info "Error-Info header" must be initialized with
 * the SIP_ERROR_INFO_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_error_info_t sip_error_info = SIP_ERROR_INFO_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ERROR_INFO_INIT() SIP_HDR_INIT(error_info)

/**Initialize a structure #sip_error_info_t.
 *
 * An #sip_error_info_t structure for
 * @ref sip_error_info "Error-Info header" can be initialized with the
 * sip_error_info_init() function/macro. For instance,
 * @code
 *
 *  sip_error_info_t sip_error_info;
 *
 *  sip_error_info_init(&sip_error_info);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_error_info_t *sip_error_info_init(sip_error_info_t x[1])
{
  return SIP_HEADER_INIT(x, sip_error_info_class, sizeof(sip_error_info_t));
}
#else
#define sip_error_info_init(x) \
  SIP_HEADER_INIT(x, sip_error_info_class, sizeof(sip_error_info_t))
#endif

/**Test if header object is instance of #sip_error_info_t.
 *
 * Check if the header class is an instance of
 * @ref sip_error_info "Error-Info header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header error_info
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_error_info(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_error_info_hash;
}
#else
int sip_is_error_info(sip_header_t const *header);
#endif

#define sip_error_info_p(h) sip_is_error_info((h))


/**Duplicate a list of @ref sip_error_info "Error-Info header" header structures #sip_error_info_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   error_info = sip_error_info_dup(home, sip->sip_error_info);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_error_info_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_error_info_t *sip_error_info_dup(su_home_t *home, sip_error_info_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_error_info_t *sip_error_info_dup(su_home_t *home, sip_error_info_t const *hdr)
{
  return (sip_error_info_t *)
    msg_header_dup_as(home, sip_error_info_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_error_info "Error-Info header" header structures #sip_error_info_t.
 *
 * The function sip_error_info_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   error_info = sip_error_info_copy(home, sip->sip_error_info);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_error_info_t *sip_error_info_copy(su_home_t *home, sip_error_info_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_error_info_t *sip_error_info_copy(su_home_t *home, sip_error_info_t const *hdr)
{
  return (sip_error_info_t *)
    msg_header_copy_as(home, sip_error_info_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_error_info "Error-Info header" structure #sip_error_info_t.
 *
 * The function sip_error_info_make() makes a new
 * #sip_error_info_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_error_info_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_error_info_t *sip_error_info_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_error_info_t *sip_error_info_make(su_home_t *home, char const *s)
{
  return (sip_error_info_t *)sip_header_make(home, sip_error_info_class, s);
}
#endif

/**Make a @ref sip_error_info "Error-Info header" from formatting result.
 *
 * Make a new #sip_error_info_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_error_info_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_error_info_t *sip_error_info_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_error_info_t *sip_error_info_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_error_info_class, fmt, ap);
  va_end(ap);

  return (sip_error_info_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_warning
 * @{
 */

/** Parse a SIP @ref sip_warning "Warning header". @internal */
SOFIAPUBFUN issize_t sip_warning_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_warning "Warning header". @internal */
SOFIAPUBFUN issize_t sip_warning_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_warning "Warning header"
 * structure #sip_warning_t from #sip_t.
 *
 */
#define sip_warning(sip) \
  ((sip_warning_t *)msg_header_access((msg_pub_t*)(sip), sip_warning_class))

/**Initializer for structure #sip_warning_t.
 *
 * A static #sip_warning_t structure for
 * @ref sip_warning "Warning header" must be initialized with
 * the SIP_WARNING_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_warning_t sip_warning = SIP_WARNING_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_WARNING_INIT() SIP_HDR_INIT(warning)

/**Initialize a structure #sip_warning_t.
 *
 * An #sip_warning_t structure for
 * @ref sip_warning "Warning header" can be initialized with the
 * sip_warning_init() function/macro. For instance,
 * @code
 *
 *  sip_warning_t sip_warning;
 *
 *  sip_warning_init(&sip_warning);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_warning_t *sip_warning_init(sip_warning_t x[1])
{
  return SIP_HEADER_INIT(x, sip_warning_class, sizeof(sip_warning_t));
}
#else
#define sip_warning_init(x) \
  SIP_HEADER_INIT(x, sip_warning_class, sizeof(sip_warning_t))
#endif

/**Test if header object is instance of #sip_warning_t.
 *
 * Check if the header class is an instance of
 * @ref sip_warning "Warning header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header warning
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_warning(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_warning_hash;
}
#else
int sip_is_warning(sip_header_t const *header);
#endif

#define sip_warning_p(h) sip_is_warning((h))


/**Duplicate a list of @ref sip_warning "Warning header" header structures #sip_warning_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   warning = sip_warning_dup(home, sip->sip_warning);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_warning_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_warning_t *sip_warning_dup(su_home_t *home, sip_warning_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_warning_t *sip_warning_dup(su_home_t *home, sip_warning_t const *hdr)
{
  return (sip_warning_t *)
    msg_header_dup_as(home, sip_warning_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_warning "Warning header" header structures #sip_warning_t.
 *
 * The function sip_warning_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   warning = sip_warning_copy(home, sip->sip_warning);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_warning_t *sip_warning_copy(su_home_t *home, sip_warning_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_warning_t *sip_warning_copy(su_home_t *home, sip_warning_t const *hdr)
{
  return (sip_warning_t *)
    msg_header_copy_as(home, sip_warning_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_warning "Warning header" structure #sip_warning_t.
 *
 * The function sip_warning_make() makes a new
 * #sip_warning_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_warning_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_warning_t *sip_warning_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_warning_t *sip_warning_make(su_home_t *home, char const *s)
{
  return (sip_warning_t *)sip_header_make(home, sip_warning_class, s);
}
#endif

/**Make a @ref sip_warning "Warning header" from formatting result.
 *
 * Make a new #sip_warning_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_warning_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_warning_t *sip_warning_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_warning_t *sip_warning_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_warning_class, fmt, ap);
  va_end(ap);

  return (sip_warning_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_refer_to
 * @{
 */

/** Parse a SIP @ref sip_refer_to "Refer-To header". @internal */
SOFIAPUBFUN issize_t sip_refer_to_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_refer_to "Refer-To header". @internal */
SOFIAPUBFUN issize_t sip_refer_to_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_refer_to "Refer-To header"
 * structure #sip_refer_to_t from #sip_t.
 *
 */
#define sip_refer_to(sip) \
  ((sip_refer_to_t *)msg_header_access((msg_pub_t*)(sip), sip_refer_to_class))

/**Initializer for structure #sip_refer_to_t.
 *
 * A static #sip_refer_to_t structure for
 * @ref sip_refer_to "Refer-To header" must be initialized with
 * the SIP_REFER_TO_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_refer_to_t sip_refer_to = SIP_REFER_TO_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_REFER_TO_INIT() SIP_HDR_INIT(refer_to)

/**Initialize a structure #sip_refer_to_t.
 *
 * An #sip_refer_to_t structure for
 * @ref sip_refer_to "Refer-To header" can be initialized with the
 * sip_refer_to_init() function/macro. For instance,
 * @code
 *
 *  sip_refer_to_t sip_refer_to;
 *
 *  sip_refer_to_init(&sip_refer_to);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_refer_to_t *sip_refer_to_init(sip_refer_to_t x[1])
{
  return SIP_HEADER_INIT(x, sip_refer_to_class, sizeof(sip_refer_to_t));
}
#else
#define sip_refer_to_init(x) \
  SIP_HEADER_INIT(x, sip_refer_to_class, sizeof(sip_refer_to_t))
#endif

/**Test if header object is instance of #sip_refer_to_t.
 *
 * Check if the header class is an instance of
 * @ref sip_refer_to "Refer-To header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header refer_to
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_refer_to(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_refer_to_hash;
}
#else
int sip_is_refer_to(sip_header_t const *header);
#endif

#define sip_refer_to_p(h) sip_is_refer_to((h))


/**Duplicate a list of @ref sip_refer_to "Refer-To header" header structures #sip_refer_to_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   refer_to = sip_refer_to_dup(home, sip->sip_refer_to);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_refer_to_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_refer_to_t *sip_refer_to_dup(su_home_t *home, sip_refer_to_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_refer_to_t *sip_refer_to_dup(su_home_t *home, sip_refer_to_t const *hdr)
{
  return (sip_refer_to_t *)
    msg_header_dup_as(home, sip_refer_to_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_refer_to "Refer-To header" header structures #sip_refer_to_t.
 *
 * The function sip_refer_to_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   refer_to = sip_refer_to_copy(home, sip->sip_refer_to);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_refer_to_t *sip_refer_to_copy(su_home_t *home, sip_refer_to_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_refer_to_t *sip_refer_to_copy(su_home_t *home, sip_refer_to_t const *hdr)
{
  return (sip_refer_to_t *)
    msg_header_copy_as(home, sip_refer_to_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_refer_to "Refer-To header" structure #sip_refer_to_t.
 *
 * The function sip_refer_to_make() makes a new
 * #sip_refer_to_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_refer_to_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_refer_to_t *sip_refer_to_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_refer_to_t *sip_refer_to_make(su_home_t *home, char const *s)
{
  return (sip_refer_to_t *)sip_header_make(home, sip_refer_to_class, s);
}
#endif

/**Make a @ref sip_refer_to "Refer-To header" from formatting result.
 *
 * Make a new #sip_refer_to_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_refer_to_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_refer_to_t *sip_refer_to_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_refer_to_t *sip_refer_to_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_refer_to_class, fmt, ap);
  va_end(ap);

  return (sip_refer_to_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_referred_by
 * @{
 */

/** Parse a SIP @ref sip_referred_by "Referred-By header". @internal */
SOFIAPUBFUN issize_t sip_referred_by_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_referred_by "Referred-By header". @internal */
SOFIAPUBFUN issize_t sip_referred_by_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_referred_by "Referred-By header"
 * structure #sip_referred_by_t from #sip_t.
 *
 */
#define sip_referred_by(sip) \
  ((sip_referred_by_t *)msg_header_access((msg_pub_t*)(sip), sip_referred_by_class))

/**Initializer for structure #sip_referred_by_t.
 *
 * A static #sip_referred_by_t structure for
 * @ref sip_referred_by "Referred-By header" must be initialized with
 * the SIP_REFERRED_BY_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_referred_by_t sip_referred_by = SIP_REFERRED_BY_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_REFERRED_BY_INIT() SIP_HDR_INIT(referred_by)

/**Initialize a structure #sip_referred_by_t.
 *
 * An #sip_referred_by_t structure for
 * @ref sip_referred_by "Referred-By header" can be initialized with the
 * sip_referred_by_init() function/macro. For instance,
 * @code
 *
 *  sip_referred_by_t sip_referred_by;
 *
 *  sip_referred_by_init(&sip_referred_by);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_referred_by_t *sip_referred_by_init(sip_referred_by_t x[1])
{
  return SIP_HEADER_INIT(x, sip_referred_by_class, sizeof(sip_referred_by_t));
}
#else
#define sip_referred_by_init(x) \
  SIP_HEADER_INIT(x, sip_referred_by_class, sizeof(sip_referred_by_t))
#endif

/**Test if header object is instance of #sip_referred_by_t.
 *
 * Check if the header class is an instance of
 * @ref sip_referred_by "Referred-By header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header referred_by
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_referred_by(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_referred_by_hash;
}
#else
int sip_is_referred_by(sip_header_t const *header);
#endif

#define sip_referred_by_p(h) sip_is_referred_by((h))


/**Duplicate a list of @ref sip_referred_by "Referred-By header" header structures #sip_referred_by_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   referred_by = sip_referred_by_dup(home, sip->sip_referred_by);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_referred_by_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_referred_by_t *sip_referred_by_dup(su_home_t *home, sip_referred_by_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_referred_by_t *sip_referred_by_dup(su_home_t *home, sip_referred_by_t const *hdr)
{
  return (sip_referred_by_t *)
    msg_header_dup_as(home, sip_referred_by_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_referred_by "Referred-By header" header structures #sip_referred_by_t.
 *
 * The function sip_referred_by_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   referred_by = sip_referred_by_copy(home, sip->sip_referred_by);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_referred_by_t *sip_referred_by_copy(su_home_t *home, sip_referred_by_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_referred_by_t *sip_referred_by_copy(su_home_t *home, sip_referred_by_t const *hdr)
{
  return (sip_referred_by_t *)
    msg_header_copy_as(home, sip_referred_by_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_referred_by "Referred-By header" structure #sip_referred_by_t.
 *
 * The function sip_referred_by_make() makes a new
 * #sip_referred_by_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_referred_by_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_referred_by_t *sip_referred_by_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_referred_by_t *sip_referred_by_make(su_home_t *home, char const *s)
{
  return (sip_referred_by_t *)sip_header_make(home, sip_referred_by_class, s);
}
#endif

/**Make a @ref sip_referred_by "Referred-By header" from formatting result.
 *
 * Make a new #sip_referred_by_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_referred_by_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_referred_by_t *sip_referred_by_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_referred_by_t *sip_referred_by_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_referred_by_class, fmt, ap);
  va_end(ap);

  return (sip_referred_by_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_replaces
 * @{
 */

/** Parse a SIP @ref sip_replaces "Replaces header". @internal */
SOFIAPUBFUN issize_t sip_replaces_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_replaces "Replaces header". @internal */
SOFIAPUBFUN issize_t sip_replaces_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_replaces "Replaces header"
 * structure #sip_replaces_t from #sip_t.
 *
 */
#define sip_replaces(sip) \
  ((sip_replaces_t *)msg_header_access((msg_pub_t*)(sip), sip_replaces_class))

/**Initializer for structure #sip_replaces_t.
 *
 * A static #sip_replaces_t structure for
 * @ref sip_replaces "Replaces header" must be initialized with
 * the SIP_REPLACES_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_replaces_t sip_replaces = SIP_REPLACES_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_REPLACES_INIT() SIP_HDR_INIT(replaces)

/**Initialize a structure #sip_replaces_t.
 *
 * An #sip_replaces_t structure for
 * @ref sip_replaces "Replaces header" can be initialized with the
 * sip_replaces_init() function/macro. For instance,
 * @code
 *
 *  sip_replaces_t sip_replaces;
 *
 *  sip_replaces_init(&sip_replaces);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_replaces_t *sip_replaces_init(sip_replaces_t x[1])
{
  return SIP_HEADER_INIT(x, sip_replaces_class, sizeof(sip_replaces_t));
}
#else
#define sip_replaces_init(x) \
  SIP_HEADER_INIT(x, sip_replaces_class, sizeof(sip_replaces_t))
#endif

/**Test if header object is instance of #sip_replaces_t.
 *
 * Check if the header class is an instance of
 * @ref sip_replaces "Replaces header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header replaces
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_replaces(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_replaces_hash;
}
#else
int sip_is_replaces(sip_header_t const *header);
#endif

#define sip_replaces_p(h) sip_is_replaces((h))


/**Duplicate a list of @ref sip_replaces "Replaces header" header structures #sip_replaces_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   replaces = sip_replaces_dup(home, sip->sip_replaces);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_replaces_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_replaces_t *sip_replaces_dup(su_home_t *home, sip_replaces_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_replaces_t *sip_replaces_dup(su_home_t *home, sip_replaces_t const *hdr)
{
  return (sip_replaces_t *)
    msg_header_dup_as(home, sip_replaces_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_replaces "Replaces header" header structures #sip_replaces_t.
 *
 * The function sip_replaces_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   replaces = sip_replaces_copy(home, sip->sip_replaces);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_replaces_t *sip_replaces_copy(su_home_t *home, sip_replaces_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_replaces_t *sip_replaces_copy(su_home_t *home, sip_replaces_t const *hdr)
{
  return (sip_replaces_t *)
    msg_header_copy_as(home, sip_replaces_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_replaces "Replaces header" structure #sip_replaces_t.
 *
 * The function sip_replaces_make() makes a new
 * #sip_replaces_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_replaces_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_replaces_t *sip_replaces_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_replaces_t *sip_replaces_make(su_home_t *home, char const *s)
{
  return (sip_replaces_t *)sip_header_make(home, sip_replaces_class, s);
}
#endif

/**Make a @ref sip_replaces "Replaces header" from formatting result.
 *
 * Make a new #sip_replaces_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_replaces_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_replaces_t *sip_replaces_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_replaces_t *sip_replaces_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_replaces_class, fmt, ap);
  va_end(ap);

  return (sip_replaces_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_session_expires
 * @{
 */

/** Parse a SIP @ref sip_session_expires "Session-Expires header". @internal */
SOFIAPUBFUN issize_t sip_session_expires_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_session_expires "Session-Expires header". @internal */
SOFIAPUBFUN issize_t sip_session_expires_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_session_expires "Session-Expires header"
 * structure #sip_session_expires_t from #sip_t.
 *
 */
#define sip_session_expires(sip) \
  ((sip_session_expires_t *)msg_header_access((msg_pub_t*)(sip), sip_session_expires_class))

/**Initializer for structure #sip_session_expires_t.
 *
 * A static #sip_session_expires_t structure for
 * @ref sip_session_expires "Session-Expires header" must be initialized with
 * the SIP_SESSION_EXPIRES_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_session_expires_t sip_session_expires = SIP_SESSION_EXPIRES_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SESSION_EXPIRES_INIT() SIP_HDR_INIT(session_expires)

/**Initialize a structure #sip_session_expires_t.
 *
 * An #sip_session_expires_t structure for
 * @ref sip_session_expires "Session-Expires header" can be initialized with the
 * sip_session_expires_init() function/macro. For instance,
 * @code
 *
 *  sip_session_expires_t sip_session_expires;
 *
 *  sip_session_expires_init(&sip_session_expires);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_session_expires_t *sip_session_expires_init(sip_session_expires_t x[1])
{
  return SIP_HEADER_INIT(x, sip_session_expires_class, sizeof(sip_session_expires_t));
}
#else
#define sip_session_expires_init(x) \
  SIP_HEADER_INIT(x, sip_session_expires_class, sizeof(sip_session_expires_t))
#endif

/**Test if header object is instance of #sip_session_expires_t.
 *
 * Check if the header class is an instance of
 * @ref sip_session_expires "Session-Expires header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header session_expires
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_session_expires(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_session_expires_hash;
}
#else
int sip_is_session_expires(sip_header_t const *header);
#endif

#define sip_session_expires_p(h) sip_is_session_expires((h))


/**Duplicate a list of @ref sip_session_expires "Session-Expires header" header structures #sip_session_expires_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   session_expires = sip_session_expires_dup(home, sip->sip_session_expires);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_session_expires_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_session_expires_t *sip_session_expires_dup(su_home_t *home, sip_session_expires_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_session_expires_t *sip_session_expires_dup(su_home_t *home, sip_session_expires_t const *hdr)
{
  return (sip_session_expires_t *)
    msg_header_dup_as(home, sip_session_expires_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_session_expires "Session-Expires header" header structures #sip_session_expires_t.
 *
 * The function sip_session_expires_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   session_expires = sip_session_expires_copy(home, sip->sip_session_expires);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_session_expires_t *sip_session_expires_copy(su_home_t *home, sip_session_expires_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_session_expires_t *sip_session_expires_copy(su_home_t *home, sip_session_expires_t const *hdr)
{
  return (sip_session_expires_t *)
    msg_header_copy_as(home, sip_session_expires_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_session_expires "Session-Expires header" structure #sip_session_expires_t.
 *
 * The function sip_session_expires_make() makes a new
 * #sip_session_expires_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_session_expires_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_session_expires_t *sip_session_expires_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_session_expires_t *sip_session_expires_make(su_home_t *home, char const *s)
{
  return (sip_session_expires_t *)sip_header_make(home, sip_session_expires_class, s);
}
#endif

/**Make a @ref sip_session_expires "Session-Expires header" from formatting result.
 *
 * Make a new #sip_session_expires_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_session_expires_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_session_expires_t *sip_session_expires_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_session_expires_t *sip_session_expires_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_session_expires_class, fmt, ap);
  va_end(ap);

  return (sip_session_expires_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_min_se
 * @{
 */

/** Parse a SIP @ref sip_min_se "Min-SE header". @internal */
SOFIAPUBFUN issize_t sip_min_se_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_min_se "Min-SE header". @internal */
SOFIAPUBFUN issize_t sip_min_se_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_min_se "Min-SE header"
 * structure #sip_min_se_t from #sip_t.
 *
 */
#define sip_min_se(sip) \
  ((sip_min_se_t *)msg_header_access((msg_pub_t*)(sip), sip_min_se_class))

/**Initializer for structure #sip_min_se_t.
 *
 * A static #sip_min_se_t structure for
 * @ref sip_min_se "Min-SE header" must be initialized with
 * the SIP_MIN_SE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_min_se_t sip_min_se = SIP_MIN_SE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_MIN_SE_INIT() SIP_HDR_INIT(min_se)

/**Initialize a structure #sip_min_se_t.
 *
 * An #sip_min_se_t structure for
 * @ref sip_min_se "Min-SE header" can be initialized with the
 * sip_min_se_init() function/macro. For instance,
 * @code
 *
 *  sip_min_se_t sip_min_se;
 *
 *  sip_min_se_init(&sip_min_se);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_min_se_t *sip_min_se_init(sip_min_se_t x[1])
{
  return SIP_HEADER_INIT(x, sip_min_se_class, sizeof(sip_min_se_t));
}
#else
#define sip_min_se_init(x) \
  SIP_HEADER_INIT(x, sip_min_se_class, sizeof(sip_min_se_t))
#endif

/**Test if header object is instance of #sip_min_se_t.
 *
 * Check if the header class is an instance of
 * @ref sip_min_se "Min-SE header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header min_se
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_min_se(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_min_se_hash;
}
#else
int sip_is_min_se(sip_header_t const *header);
#endif

#define sip_min_se_p(h) sip_is_min_se((h))


/**Duplicate a list of @ref sip_min_se "Min-SE header" header structures #sip_min_se_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   min_se = sip_min_se_dup(home, sip->sip_min_se);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_min_se_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_min_se_t *sip_min_se_dup(su_home_t *home, sip_min_se_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_min_se_t *sip_min_se_dup(su_home_t *home, sip_min_se_t const *hdr)
{
  return (sip_min_se_t *)
    msg_header_dup_as(home, sip_min_se_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_min_se "Min-SE header" header structures #sip_min_se_t.
 *
 * The function sip_min_se_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   min_se = sip_min_se_copy(home, sip->sip_min_se);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_min_se_t *sip_min_se_copy(su_home_t *home, sip_min_se_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_min_se_t *sip_min_se_copy(su_home_t *home, sip_min_se_t const *hdr)
{
  return (sip_min_se_t *)
    msg_header_copy_as(home, sip_min_se_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_min_se "Min-SE header" structure #sip_min_se_t.
 *
 * The function sip_min_se_make() makes a new
 * #sip_min_se_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_min_se_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_min_se_t *sip_min_se_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_min_se_t *sip_min_se_make(su_home_t *home, char const *s)
{
  return (sip_min_se_t *)sip_header_make(home, sip_min_se_class, s);
}
#endif

/**Make a @ref sip_min_se "Min-SE header" from formatting result.
 *
 * Make a new #sip_min_se_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_min_se_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_min_se_t *sip_min_se_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_min_se_t *sip_min_se_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_min_se_class, fmt, ap);
  va_end(ap);

  return (sip_min_se_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_path
 * @{
 */

/** Parse a SIP @ref sip_path "Path header". @internal */
SOFIAPUBFUN issize_t sip_path_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_path "Path header". @internal */
SOFIAPUBFUN issize_t sip_path_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_path "Path header"
 * structure #sip_path_t from #sip_t.
 *
 */
#define sip_path(sip) \
  ((sip_path_t *)msg_header_access((msg_pub_t*)(sip), sip_path_class))

/**Initializer for structure #sip_path_t.
 *
 * A static #sip_path_t structure for
 * @ref sip_path "Path header" must be initialized with
 * the SIP_PATH_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_path_t sip_path = SIP_PATH_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_PATH_INIT() SIP_HDR_INIT(path)

/**Initialize a structure #sip_path_t.
 *
 * An #sip_path_t structure for
 * @ref sip_path "Path header" can be initialized with the
 * sip_path_init() function/macro. For instance,
 * @code
 *
 *  sip_path_t sip_path;
 *
 *  sip_path_init(&sip_path);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_path_t *sip_path_init(sip_path_t x[1])
{
  return SIP_HEADER_INIT(x, sip_path_class, sizeof(sip_path_t));
}
#else
#define sip_path_init(x) \
  SIP_HEADER_INIT(x, sip_path_class, sizeof(sip_path_t))
#endif

/**Test if header object is instance of #sip_path_t.
 *
 * Check if the header class is an instance of
 * @ref sip_path "Path header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header path
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_path(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_path_hash;
}
#else
int sip_is_path(sip_header_t const *header);
#endif

#define sip_path_p(h) sip_is_path((h))


/**Duplicate a list of @ref sip_path "Path header" header structures #sip_path_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   path = sip_path_dup(home, sip->sip_path);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_path_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_path_t *sip_path_dup(su_home_t *home, sip_path_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_path_t *sip_path_dup(su_home_t *home, sip_path_t const *hdr)
{
  return (sip_path_t *)
    msg_header_dup_as(home, sip_path_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_path "Path header" header structures #sip_path_t.
 *
 * The function sip_path_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   path = sip_path_copy(home, sip->sip_path);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_path_t *sip_path_copy(su_home_t *home, sip_path_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_path_t *sip_path_copy(su_home_t *home, sip_path_t const *hdr)
{
  return (sip_path_t *)
    msg_header_copy_as(home, sip_path_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_path "Path header" structure #sip_path_t.
 *
 * The function sip_path_make() makes a new
 * #sip_path_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_path_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_path_t *sip_path_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_path_t *sip_path_make(su_home_t *home, char const *s)
{
  return (sip_path_t *)sip_header_make(home, sip_path_class, s);
}
#endif

/**Make a @ref sip_path "Path header" from formatting result.
 *
 * Make a new #sip_path_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_path_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_path_t *sip_path_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_path_t *sip_path_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_path_class, fmt, ap);
  va_end(ap);

  return (sip_path_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_service_route
 * @{
 */

/** Parse a SIP @ref sip_service_route "Service-Route header". @internal */
SOFIAPUBFUN issize_t sip_service_route_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_service_route "Service-Route header". @internal */
SOFIAPUBFUN issize_t sip_service_route_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_service_route "Service-Route header"
 * structure #sip_service_route_t from #sip_t.
 *
 */
#define sip_service_route(sip) \
  ((sip_service_route_t *)msg_header_access((msg_pub_t*)(sip), sip_service_route_class))

/**Initializer for structure #sip_service_route_t.
 *
 * A static #sip_service_route_t structure for
 * @ref sip_service_route "Service-Route header" must be initialized with
 * the SIP_SERVICE_ROUTE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_service_route_t sip_service_route = SIP_SERVICE_ROUTE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SERVICE_ROUTE_INIT() SIP_HDR_INIT(service_route)

/**Initialize a structure #sip_service_route_t.
 *
 * An #sip_service_route_t structure for
 * @ref sip_service_route "Service-Route header" can be initialized with the
 * sip_service_route_init() function/macro. For instance,
 * @code
 *
 *  sip_service_route_t sip_service_route;
 *
 *  sip_service_route_init(&sip_service_route);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_service_route_t *sip_service_route_init(sip_service_route_t x[1])
{
  return SIP_HEADER_INIT(x, sip_service_route_class, sizeof(sip_service_route_t));
}
#else
#define sip_service_route_init(x) \
  SIP_HEADER_INIT(x, sip_service_route_class, sizeof(sip_service_route_t))
#endif

/**Test if header object is instance of #sip_service_route_t.
 *
 * Check if the header class is an instance of
 * @ref sip_service_route "Service-Route header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header service_route
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_service_route(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_service_route_hash;
}
#else
int sip_is_service_route(sip_header_t const *header);
#endif

#define sip_service_route_p(h) sip_is_service_route((h))


/**Duplicate a list of @ref sip_service_route "Service-Route header" header structures #sip_service_route_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   service_route = sip_service_route_dup(home, sip->sip_service_route);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_service_route_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_service_route_t *sip_service_route_dup(su_home_t *home, sip_service_route_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_service_route_t *sip_service_route_dup(su_home_t *home, sip_service_route_t const *hdr)
{
  return (sip_service_route_t *)
    msg_header_dup_as(home, sip_service_route_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_service_route "Service-Route header" header structures #sip_service_route_t.
 *
 * The function sip_service_route_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   service_route = sip_service_route_copy(home, sip->sip_service_route);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_service_route_t *sip_service_route_copy(su_home_t *home, sip_service_route_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_service_route_t *sip_service_route_copy(su_home_t *home, sip_service_route_t const *hdr)
{
  return (sip_service_route_t *)
    msg_header_copy_as(home, sip_service_route_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_service_route "Service-Route header" structure #sip_service_route_t.
 *
 * The function sip_service_route_make() makes a new
 * #sip_service_route_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_service_route_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_service_route_t *sip_service_route_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_service_route_t *sip_service_route_make(su_home_t *home, char const *s)
{
  return (sip_service_route_t *)sip_header_make(home, sip_service_route_class, s);
}
#endif

/**Make a @ref sip_service_route "Service-Route header" from formatting result.
 *
 * Make a new #sip_service_route_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_service_route_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_service_route_t *sip_service_route_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_service_route_t *sip_service_route_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_service_route_class, fmt, ap);
  va_end(ap);

  return (sip_service_route_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_reason
 * @{
 */

/** Parse a SIP @ref sip_reason "Reason header". @internal */
SOFIAPUBFUN issize_t sip_reason_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_reason "Reason header". @internal */
SOFIAPUBFUN issize_t sip_reason_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_reason "Reason header"
 * structure #sip_reason_t from #sip_t.
 *
 */
#define sip_reason(sip) \
  ((sip_reason_t *)msg_header_access((msg_pub_t*)(sip), sip_reason_class))

/**Initializer for structure #sip_reason_t.
 *
 * A static #sip_reason_t structure for
 * @ref sip_reason "Reason header" must be initialized with
 * the SIP_REASON_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_reason_t sip_reason = SIP_REASON_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_REASON_INIT() SIP_HDR_INIT(reason)

/**Initialize a structure #sip_reason_t.
 *
 * An #sip_reason_t structure for
 * @ref sip_reason "Reason header" can be initialized with the
 * sip_reason_init() function/macro. For instance,
 * @code
 *
 *  sip_reason_t sip_reason;
 *
 *  sip_reason_init(&sip_reason);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_reason_t *sip_reason_init(sip_reason_t x[1])
{
  return SIP_HEADER_INIT(x, sip_reason_class, sizeof(sip_reason_t));
}
#else
#define sip_reason_init(x) \
  SIP_HEADER_INIT(x, sip_reason_class, sizeof(sip_reason_t))
#endif

/**Test if header object is instance of #sip_reason_t.
 *
 * Check if the header class is an instance of
 * @ref sip_reason "Reason header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header reason
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_reason(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_reason_hash;
}
#else
int sip_is_reason(sip_header_t const *header);
#endif

#define sip_reason_p(h) sip_is_reason((h))


/**Duplicate a list of @ref sip_reason "Reason header" header structures #sip_reason_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   reason = sip_reason_dup(home, sip->sip_reason);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_reason_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_reason_t *sip_reason_dup(su_home_t *home, sip_reason_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_reason_t *sip_reason_dup(su_home_t *home, sip_reason_t const *hdr)
{
  return (sip_reason_t *)
    msg_header_dup_as(home, sip_reason_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_reason "Reason header" header structures #sip_reason_t.
 *
 * The function sip_reason_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   reason = sip_reason_copy(home, sip->sip_reason);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_reason_t *sip_reason_copy(su_home_t *home, sip_reason_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_reason_t *sip_reason_copy(su_home_t *home, sip_reason_t const *hdr)
{
  return (sip_reason_t *)
    msg_header_copy_as(home, sip_reason_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_reason "Reason header" structure #sip_reason_t.
 *
 * The function sip_reason_make() makes a new
 * #sip_reason_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_reason_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_reason_t *sip_reason_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_reason_t *sip_reason_make(su_home_t *home, char const *s)
{
  return (sip_reason_t *)sip_header_make(home, sip_reason_class, s);
}
#endif

/**Make a @ref sip_reason "Reason header" from formatting result.
 *
 * Make a new #sip_reason_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_reason_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_reason_t *sip_reason_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_reason_t *sip_reason_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_reason_class, fmt, ap);
  va_end(ap);

  return (sip_reason_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_security_client
 * @{
 */

/** Parse a SIP @ref sip_security_client "Security-Client header". @internal */
SOFIAPUBFUN issize_t sip_security_client_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_security_client "Security-Client header". @internal */
SOFIAPUBFUN issize_t sip_security_client_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_security_client "Security-Client header"
 * structure #sip_security_client_t from #sip_t.
 *
 */
#define sip_security_client(sip) \
  ((sip_security_client_t *)msg_header_access((msg_pub_t*)(sip), sip_security_client_class))

/**Initializer for structure #sip_security_client_t.
 *
 * A static #sip_security_client_t structure for
 * @ref sip_security_client "Security-Client header" must be initialized with
 * the SIP_SECURITY_CLIENT_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_security_client_t sip_security_client = SIP_SECURITY_CLIENT_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SECURITY_CLIENT_INIT() SIP_HDR_INIT(security_client)

/**Initialize a structure #sip_security_client_t.
 *
 * An #sip_security_client_t structure for
 * @ref sip_security_client "Security-Client header" can be initialized with the
 * sip_security_client_init() function/macro. For instance,
 * @code
 *
 *  sip_security_client_t sip_security_client;
 *
 *  sip_security_client_init(&sip_security_client);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_security_client_t *sip_security_client_init(sip_security_client_t x[1])
{
  return SIP_HEADER_INIT(x, sip_security_client_class, sizeof(sip_security_client_t));
}
#else
#define sip_security_client_init(x) \
  SIP_HEADER_INIT(x, sip_security_client_class, sizeof(sip_security_client_t))
#endif

/**Test if header object is instance of #sip_security_client_t.
 *
 * Check if the header class is an instance of
 * @ref sip_security_client "Security-Client header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header security_client
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_security_client(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_security_client_hash;
}
#else
int sip_is_security_client(sip_header_t const *header);
#endif

#define sip_security_client_p(h) sip_is_security_client((h))


/**Duplicate a list of @ref sip_security_client "Security-Client header" header structures #sip_security_client_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   security_client = sip_security_client_dup(home, sip->sip_security_client);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_security_client_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_client_t *sip_security_client_dup(su_home_t *home, sip_security_client_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_security_client_t *sip_security_client_dup(su_home_t *home, sip_security_client_t const *hdr)
{
  return (sip_security_client_t *)
    msg_header_dup_as(home, sip_security_client_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_security_client "Security-Client header" header structures #sip_security_client_t.
 *
 * The function sip_security_client_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   security_client = sip_security_client_copy(home, sip->sip_security_client);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_client_t *sip_security_client_copy(su_home_t *home, sip_security_client_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_security_client_t *sip_security_client_copy(su_home_t *home, sip_security_client_t const *hdr)
{
  return (sip_security_client_t *)
    msg_header_copy_as(home, sip_security_client_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_security_client "Security-Client header" structure #sip_security_client_t.
 *
 * The function sip_security_client_make() makes a new
 * #sip_security_client_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_security_client_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_client_t *sip_security_client_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_security_client_t *sip_security_client_make(su_home_t *home, char const *s)
{
  return (sip_security_client_t *)sip_header_make(home, sip_security_client_class, s);
}
#endif

/**Make a @ref sip_security_client "Security-Client header" from formatting result.
 *
 * Make a new #sip_security_client_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_security_client_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_client_t *sip_security_client_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_security_client_t *sip_security_client_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_security_client_class, fmt, ap);
  va_end(ap);

  return (sip_security_client_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_security_server
 * @{
 */

/** Parse a SIP @ref sip_security_server "Security-Server header". @internal */
SOFIAPUBFUN issize_t sip_security_server_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_security_server "Security-Server header". @internal */
SOFIAPUBFUN issize_t sip_security_server_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_security_server "Security-Server header"
 * structure #sip_security_server_t from #sip_t.
 *
 */
#define sip_security_server(sip) \
  ((sip_security_server_t *)msg_header_access((msg_pub_t*)(sip), sip_security_server_class))

/**Initializer for structure #sip_security_server_t.
 *
 * A static #sip_security_server_t structure for
 * @ref sip_security_server "Security-Server header" must be initialized with
 * the SIP_SECURITY_SERVER_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_security_server_t sip_security_server = SIP_SECURITY_SERVER_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SECURITY_SERVER_INIT() SIP_HDR_INIT(security_server)

/**Initialize a structure #sip_security_server_t.
 *
 * An #sip_security_server_t structure for
 * @ref sip_security_server "Security-Server header" can be initialized with the
 * sip_security_server_init() function/macro. For instance,
 * @code
 *
 *  sip_security_server_t sip_security_server;
 *
 *  sip_security_server_init(&sip_security_server);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_security_server_t *sip_security_server_init(sip_security_server_t x[1])
{
  return SIP_HEADER_INIT(x, sip_security_server_class, sizeof(sip_security_server_t));
}
#else
#define sip_security_server_init(x) \
  SIP_HEADER_INIT(x, sip_security_server_class, sizeof(sip_security_server_t))
#endif

/**Test if header object is instance of #sip_security_server_t.
 *
 * Check if the header class is an instance of
 * @ref sip_security_server "Security-Server header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header security_server
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_security_server(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_security_server_hash;
}
#else
int sip_is_security_server(sip_header_t const *header);
#endif

#define sip_security_server_p(h) sip_is_security_server((h))


/**Duplicate a list of @ref sip_security_server "Security-Server header" header structures #sip_security_server_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   security_server = sip_security_server_dup(home, sip->sip_security_server);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_security_server_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_server_t *sip_security_server_dup(su_home_t *home, sip_security_server_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_security_server_t *sip_security_server_dup(su_home_t *home, sip_security_server_t const *hdr)
{
  return (sip_security_server_t *)
    msg_header_dup_as(home, sip_security_server_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_security_server "Security-Server header" header structures #sip_security_server_t.
 *
 * The function sip_security_server_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   security_server = sip_security_server_copy(home, sip->sip_security_server);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_server_t *sip_security_server_copy(su_home_t *home, sip_security_server_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_security_server_t *sip_security_server_copy(su_home_t *home, sip_security_server_t const *hdr)
{
  return (sip_security_server_t *)
    msg_header_copy_as(home, sip_security_server_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_security_server "Security-Server header" structure #sip_security_server_t.
 *
 * The function sip_security_server_make() makes a new
 * #sip_security_server_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_security_server_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_server_t *sip_security_server_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_security_server_t *sip_security_server_make(su_home_t *home, char const *s)
{
  return (sip_security_server_t *)sip_header_make(home, sip_security_server_class, s);
}
#endif

/**Make a @ref sip_security_server "Security-Server header" from formatting result.
 *
 * Make a new #sip_security_server_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_security_server_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_server_t *sip_security_server_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_security_server_t *sip_security_server_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_security_server_class, fmt, ap);
  va_end(ap);

  return (sip_security_server_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_security_verify
 * @{
 */

/** Parse a SIP @ref sip_security_verify "Security-Verify header". @internal */
SOFIAPUBFUN issize_t sip_security_verify_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_security_verify "Security-Verify header". @internal */
SOFIAPUBFUN issize_t sip_security_verify_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_security_verify "Security-Verify header"
 * structure #sip_security_verify_t from #sip_t.
 *
 */
#define sip_security_verify(sip) \
  ((sip_security_verify_t *)msg_header_access((msg_pub_t*)(sip), sip_security_verify_class))

/**Initializer for structure #sip_security_verify_t.
 *
 * A static #sip_security_verify_t structure for
 * @ref sip_security_verify "Security-Verify header" must be initialized with
 * the SIP_SECURITY_VERIFY_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_security_verify_t sip_security_verify = SIP_SECURITY_VERIFY_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SECURITY_VERIFY_INIT() SIP_HDR_INIT(security_verify)

/**Initialize a structure #sip_security_verify_t.
 *
 * An #sip_security_verify_t structure for
 * @ref sip_security_verify "Security-Verify header" can be initialized with the
 * sip_security_verify_init() function/macro. For instance,
 * @code
 *
 *  sip_security_verify_t sip_security_verify;
 *
 *  sip_security_verify_init(&sip_security_verify);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_security_verify_t *sip_security_verify_init(sip_security_verify_t x[1])
{
  return SIP_HEADER_INIT(x, sip_security_verify_class, sizeof(sip_security_verify_t));
}
#else
#define sip_security_verify_init(x) \
  SIP_HEADER_INIT(x, sip_security_verify_class, sizeof(sip_security_verify_t))
#endif

/**Test if header object is instance of #sip_security_verify_t.
 *
 * Check if the header class is an instance of
 * @ref sip_security_verify "Security-Verify header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header security_verify
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_security_verify(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_security_verify_hash;
}
#else
int sip_is_security_verify(sip_header_t const *header);
#endif

#define sip_security_verify_p(h) sip_is_security_verify((h))


/**Duplicate a list of @ref sip_security_verify "Security-Verify header" header structures #sip_security_verify_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   security_verify = sip_security_verify_dup(home, sip->sip_security_verify);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_security_verify_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_verify_t *sip_security_verify_dup(su_home_t *home, sip_security_verify_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_security_verify_t *sip_security_verify_dup(su_home_t *home, sip_security_verify_t const *hdr)
{
  return (sip_security_verify_t *)
    msg_header_dup_as(home, sip_security_verify_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_security_verify "Security-Verify header" header structures #sip_security_verify_t.
 *
 * The function sip_security_verify_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   security_verify = sip_security_verify_copy(home, sip->sip_security_verify);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_verify_t *sip_security_verify_copy(su_home_t *home, sip_security_verify_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_security_verify_t *sip_security_verify_copy(su_home_t *home, sip_security_verify_t const *hdr)
{
  return (sip_security_verify_t *)
    msg_header_copy_as(home, sip_security_verify_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_security_verify "Security-Verify header" structure #sip_security_verify_t.
 *
 * The function sip_security_verify_make() makes a new
 * #sip_security_verify_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_security_verify_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_verify_t *sip_security_verify_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_security_verify_t *sip_security_verify_make(su_home_t *home, char const *s)
{
  return (sip_security_verify_t *)sip_header_make(home, sip_security_verify_class, s);
}
#endif

/**Make a @ref sip_security_verify "Security-Verify header" from formatting result.
 *
 * Make a new #sip_security_verify_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_security_verify_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_security_verify_t *sip_security_verify_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_security_verify_t *sip_security_verify_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_security_verify_class, fmt, ap);
  va_end(ap);

  return (sip_security_verify_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_privacy
 * @{
 */

/** Parse a SIP @ref sip_privacy "Privacy header". @internal */
SOFIAPUBFUN issize_t sip_privacy_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_privacy "Privacy header". @internal */
SOFIAPUBFUN issize_t sip_privacy_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_privacy "Privacy header"
 * structure #sip_privacy_t from #sip_t.
 *
 */
#define sip_privacy(sip) \
  ((sip_privacy_t *)msg_header_access((msg_pub_t*)(sip), sip_privacy_class))

/**Initializer for structure #sip_privacy_t.
 *
 * A static #sip_privacy_t structure for
 * @ref sip_privacy "Privacy header" must be initialized with
 * the SIP_PRIVACY_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_privacy_t sip_privacy = SIP_PRIVACY_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_PRIVACY_INIT() SIP_HDR_INIT(privacy)

/**Initialize a structure #sip_privacy_t.
 *
 * An #sip_privacy_t structure for
 * @ref sip_privacy "Privacy header" can be initialized with the
 * sip_privacy_init() function/macro. For instance,
 * @code
 *
 *  sip_privacy_t sip_privacy;
 *
 *  sip_privacy_init(&sip_privacy);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_privacy_t *sip_privacy_init(sip_privacy_t x[1])
{
  return SIP_HEADER_INIT(x, sip_privacy_class, sizeof(sip_privacy_t));
}
#else
#define sip_privacy_init(x) \
  SIP_HEADER_INIT(x, sip_privacy_class, sizeof(sip_privacy_t))
#endif

/**Test if header object is instance of #sip_privacy_t.
 *
 * Check if the header class is an instance of
 * @ref sip_privacy "Privacy header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header privacy
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_privacy(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_privacy_hash;
}
#else
int sip_is_privacy(sip_header_t const *header);
#endif

#define sip_privacy_p(h) sip_is_privacy((h))


/**Duplicate a list of @ref sip_privacy "Privacy header" header structures #sip_privacy_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   privacy = sip_privacy_dup(home, sip->sip_privacy);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_privacy_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_privacy_t *sip_privacy_dup(su_home_t *home, sip_privacy_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_privacy_t *sip_privacy_dup(su_home_t *home, sip_privacy_t const *hdr)
{
  return (sip_privacy_t *)
    msg_header_dup_as(home, sip_privacy_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_privacy "Privacy header" header structures #sip_privacy_t.
 *
 * The function sip_privacy_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   privacy = sip_privacy_copy(home, sip->sip_privacy);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_privacy_t *sip_privacy_copy(su_home_t *home, sip_privacy_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_privacy_t *sip_privacy_copy(su_home_t *home, sip_privacy_t const *hdr)
{
  return (sip_privacy_t *)
    msg_header_copy_as(home, sip_privacy_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_privacy "Privacy header" structure #sip_privacy_t.
 *
 * The function sip_privacy_make() makes a new
 * #sip_privacy_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_privacy_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_privacy_t *sip_privacy_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_privacy_t *sip_privacy_make(su_home_t *home, char const *s)
{
  return (sip_privacy_t *)sip_header_make(home, sip_privacy_class, s);
}
#endif

/**Make a @ref sip_privacy "Privacy header" from formatting result.
 *
 * Make a new #sip_privacy_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_privacy_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_privacy_t *sip_privacy_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_privacy_t *sip_privacy_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_privacy_class, fmt, ap);
  va_end(ap);

  return (sip_privacy_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_etag
 * @{
 */

/** Parse a SIP @ref sip_etag "SIP-ETag header". @internal */
SOFIAPUBFUN issize_t sip_etag_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_etag "SIP-ETag header". @internal */
SOFIAPUBFUN issize_t sip_etag_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_etag "SIP-ETag header"
 * structure #sip_etag_t from #sip_t.
 *
 */
#define sip_etag(sip) \
  ((sip_etag_t *)msg_header_access((msg_pub_t*)(sip), sip_etag_class))

/**Initializer for structure #sip_etag_t.
 *
 * A static #sip_etag_t structure for
 * @ref sip_etag "SIP-ETag header" must be initialized with
 * the SIP_ETAG_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_etag_t sip_etag = SIP_ETAG_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ETAG_INIT() SIP_HDR_INIT(etag)

/**Initialize a structure #sip_etag_t.
 *
 * An #sip_etag_t structure for
 * @ref sip_etag "SIP-ETag header" can be initialized with the
 * sip_etag_init() function/macro. For instance,
 * @code
 *
 *  sip_etag_t sip_etag;
 *
 *  sip_etag_init(&sip_etag);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_etag_t *sip_etag_init(sip_etag_t x[1])
{
  return SIP_HEADER_INIT(x, sip_etag_class, sizeof(sip_etag_t));
}
#else
#define sip_etag_init(x) \
  SIP_HEADER_INIT(x, sip_etag_class, sizeof(sip_etag_t))
#endif

/**Test if header object is instance of #sip_etag_t.
 *
 * Check if the header class is an instance of
 * @ref sip_etag "SIP-ETag header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header etag
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_etag(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_etag_hash;
}
#else
int sip_is_etag(sip_header_t const *header);
#endif

#define sip_etag_p(h) sip_is_etag((h))


/**Duplicate a list of @ref sip_etag "SIP-ETag header" header structures #sip_etag_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   etag = sip_etag_dup(home, sip->sip_etag);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_etag_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_etag_t *sip_etag_dup(su_home_t *home, sip_etag_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_etag_t *sip_etag_dup(su_home_t *home, sip_etag_t const *hdr)
{
  return (sip_etag_t *)
    msg_header_dup_as(home, sip_etag_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_etag "SIP-ETag header" header structures #sip_etag_t.
 *
 * The function sip_etag_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   etag = sip_etag_copy(home, sip->sip_etag);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_etag_t *sip_etag_copy(su_home_t *home, sip_etag_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_etag_t *sip_etag_copy(su_home_t *home, sip_etag_t const *hdr)
{
  return (sip_etag_t *)
    msg_header_copy_as(home, sip_etag_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_etag "SIP-ETag header" structure #sip_etag_t.
 *
 * The function sip_etag_make() makes a new
 * #sip_etag_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_etag_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_etag_t *sip_etag_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_etag_t *sip_etag_make(su_home_t *home, char const *s)
{
  return (sip_etag_t *)sip_header_make(home, sip_etag_class, s);
}
#endif

/**Make a @ref sip_etag "SIP-ETag header" from formatting result.
 *
 * Make a new #sip_etag_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_etag_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_etag_t *sip_etag_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_etag_t *sip_etag_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_etag_class, fmt, ap);
  va_end(ap);

  return (sip_etag_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_if_match
 * @{
 */

/** Parse a SIP @ref sip_if_match "SIP-If-Match header". @internal */
SOFIAPUBFUN issize_t sip_if_match_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_if_match "SIP-If-Match header". @internal */
SOFIAPUBFUN issize_t sip_if_match_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_if_match "SIP-If-Match header"
 * structure #sip_if_match_t from #sip_t.
 *
 */
#define sip_if_match(sip) \
  ((sip_if_match_t *)msg_header_access((msg_pub_t*)(sip), sip_if_match_class))

/**Initializer for structure #sip_if_match_t.
 *
 * A static #sip_if_match_t structure for
 * @ref sip_if_match "SIP-If-Match header" must be initialized with
 * the SIP_IF_MATCH_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_if_match_t sip_if_match = SIP_IF_MATCH_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_IF_MATCH_INIT() SIP_HDR_INIT(if_match)

/**Initialize a structure #sip_if_match_t.
 *
 * An #sip_if_match_t structure for
 * @ref sip_if_match "SIP-If-Match header" can be initialized with the
 * sip_if_match_init() function/macro. For instance,
 * @code
 *
 *  sip_if_match_t sip_if_match;
 *
 *  sip_if_match_init(&sip_if_match);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_if_match_t *sip_if_match_init(sip_if_match_t x[1])
{
  return SIP_HEADER_INIT(x, sip_if_match_class, sizeof(sip_if_match_t));
}
#else
#define sip_if_match_init(x) \
  SIP_HEADER_INIT(x, sip_if_match_class, sizeof(sip_if_match_t))
#endif

/**Test if header object is instance of #sip_if_match_t.
 *
 * Check if the header class is an instance of
 * @ref sip_if_match "SIP-If-Match header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header if_match
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_if_match(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_if_match_hash;
}
#else
int sip_is_if_match(sip_header_t const *header);
#endif

#define sip_if_match_p(h) sip_is_if_match((h))


/**Duplicate a list of @ref sip_if_match "SIP-If-Match header" header structures #sip_if_match_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   if_match = sip_if_match_dup(home, sip->sip_if_match);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_if_match_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_if_match_t *sip_if_match_dup(su_home_t *home, sip_if_match_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_if_match_t *sip_if_match_dup(su_home_t *home, sip_if_match_t const *hdr)
{
  return (sip_if_match_t *)
    msg_header_dup_as(home, sip_if_match_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_if_match "SIP-If-Match header" header structures #sip_if_match_t.
 *
 * The function sip_if_match_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   if_match = sip_if_match_copy(home, sip->sip_if_match);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_if_match_t *sip_if_match_copy(su_home_t *home, sip_if_match_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_if_match_t *sip_if_match_copy(su_home_t *home, sip_if_match_t const *hdr)
{
  return (sip_if_match_t *)
    msg_header_copy_as(home, sip_if_match_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_if_match "SIP-If-Match header" structure #sip_if_match_t.
 *
 * The function sip_if_match_make() makes a new
 * #sip_if_match_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_if_match_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_if_match_t *sip_if_match_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_if_match_t *sip_if_match_make(su_home_t *home, char const *s)
{
  return (sip_if_match_t *)sip_header_make(home, sip_if_match_class, s);
}
#endif

/**Make a @ref sip_if_match "SIP-If-Match header" from formatting result.
 *
 * Make a new #sip_if_match_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_if_match_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_if_match_t *sip_if_match_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_if_match_t *sip_if_match_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_if_match_class, fmt, ap);
  va_end(ap);

  return (sip_if_match_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_mime_version
 * @{
 */

/** Parse a SIP @ref sip_mime_version "MIME-Version header". @internal */
SOFIAPUBFUN issize_t sip_mime_version_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_mime_version "MIME-Version header". @internal */
SOFIAPUBFUN issize_t sip_mime_version_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_mime_version "MIME-Version header"
 * structure #sip_mime_version_t from #sip_t.
 *
 */
#define sip_mime_version(sip) \
  ((sip_mime_version_t *)msg_header_access((msg_pub_t*)(sip), sip_mime_version_class))

/**Initializer for structure #sip_mime_version_t.
 *
 * A static #sip_mime_version_t structure for
 * @ref sip_mime_version "MIME-Version header" must be initialized with
 * the SIP_MIME_VERSION_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_mime_version_t sip_mime_version = SIP_MIME_VERSION_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_MIME_VERSION_INIT() SIP_HDR_INIT(mime_version)

/**Initialize a structure #sip_mime_version_t.
 *
 * An #sip_mime_version_t structure for
 * @ref sip_mime_version "MIME-Version header" can be initialized with the
 * sip_mime_version_init() function/macro. For instance,
 * @code
 *
 *  sip_mime_version_t sip_mime_version;
 *
 *  sip_mime_version_init(&sip_mime_version);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_mime_version_t *sip_mime_version_init(sip_mime_version_t x[1])
{
  return SIP_HEADER_INIT(x, sip_mime_version_class, sizeof(sip_mime_version_t));
}
#else
#define sip_mime_version_init(x) \
  SIP_HEADER_INIT(x, sip_mime_version_class, sizeof(sip_mime_version_t))
#endif

/**Test if header object is instance of #sip_mime_version_t.
 *
 * Check if the header class is an instance of
 * @ref sip_mime_version "MIME-Version header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header mime_version
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_mime_version(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_mime_version_hash;
}
#else
int sip_is_mime_version(sip_header_t const *header);
#endif

#define sip_mime_version_p(h) sip_is_mime_version((h))


/**Duplicate a list of @ref sip_mime_version "MIME-Version header" header structures #sip_mime_version_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   mime_version = sip_mime_version_dup(home, sip->sip_mime_version);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_mime_version_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_mime_version_t *sip_mime_version_dup(su_home_t *home, sip_mime_version_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_mime_version_t *sip_mime_version_dup(su_home_t *home, sip_mime_version_t const *hdr)
{
  return (sip_mime_version_t *)
    msg_header_dup_as(home, sip_mime_version_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_mime_version "MIME-Version header" header structures #sip_mime_version_t.
 *
 * The function sip_mime_version_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   mime_version = sip_mime_version_copy(home, sip->sip_mime_version);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_mime_version_t *sip_mime_version_copy(su_home_t *home, sip_mime_version_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_mime_version_t *sip_mime_version_copy(su_home_t *home, sip_mime_version_t const *hdr)
{
  return (sip_mime_version_t *)
    msg_header_copy_as(home, sip_mime_version_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_mime_version "MIME-Version header" structure #sip_mime_version_t.
 *
 * The function sip_mime_version_make() makes a new
 * #sip_mime_version_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_mime_version_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_mime_version_t *sip_mime_version_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_mime_version_t *sip_mime_version_make(su_home_t *home, char const *s)
{
  return (sip_mime_version_t *)sip_header_make(home, sip_mime_version_class, s);
}
#endif

/**Make a @ref sip_mime_version "MIME-Version header" from formatting result.
 *
 * Make a new #sip_mime_version_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_mime_version_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_mime_version_t *sip_mime_version_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_mime_version_t *sip_mime_version_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_mime_version_class, fmt, ap);
  va_end(ap);

  return (sip_mime_version_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_content_type
 * @{
 */

/** Parse a SIP @ref sip_content_type "Content-Type header". @internal */
SOFIAPUBFUN issize_t sip_content_type_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_content_type "Content-Type header". @internal */
SOFIAPUBFUN issize_t sip_content_type_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_content_type "Content-Type header"
 * structure #sip_content_type_t from #sip_t.
 *
 */
#define sip_content_type(sip) \
  ((sip_content_type_t *)msg_header_access((msg_pub_t*)(sip), sip_content_type_class))

/**Initializer for structure #sip_content_type_t.
 *
 * A static #sip_content_type_t structure for
 * @ref sip_content_type "Content-Type header" must be initialized with
 * the SIP_CONTENT_TYPE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_content_type_t sip_content_type = SIP_CONTENT_TYPE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CONTENT_TYPE_INIT() SIP_HDR_INIT(content_type)

/**Initialize a structure #sip_content_type_t.
 *
 * An #sip_content_type_t structure for
 * @ref sip_content_type "Content-Type header" can be initialized with the
 * sip_content_type_init() function/macro. For instance,
 * @code
 *
 *  sip_content_type_t sip_content_type;
 *
 *  sip_content_type_init(&sip_content_type);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_content_type_t *sip_content_type_init(sip_content_type_t x[1])
{
  return SIP_HEADER_INIT(x, sip_content_type_class, sizeof(sip_content_type_t));
}
#else
#define sip_content_type_init(x) \
  SIP_HEADER_INIT(x, sip_content_type_class, sizeof(sip_content_type_t))
#endif

/**Test if header object is instance of #sip_content_type_t.
 *
 * Check if the header class is an instance of
 * @ref sip_content_type "Content-Type header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header content_type
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_content_type(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_content_type_hash;
}
#else
int sip_is_content_type(sip_header_t const *header);
#endif

#define sip_content_type_p(h) sip_is_content_type((h))


/**Duplicate a list of @ref sip_content_type "Content-Type header" header structures #sip_content_type_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   content_type = sip_content_type_dup(home, sip->sip_content_type);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_content_type_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_type_t *sip_content_type_dup(su_home_t *home, sip_content_type_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_type_t *sip_content_type_dup(su_home_t *home, sip_content_type_t const *hdr)
{
  return (sip_content_type_t *)
    msg_header_dup_as(home, sip_content_type_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_content_type "Content-Type header" header structures #sip_content_type_t.
 *
 * The function sip_content_type_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   content_type = sip_content_type_copy(home, sip->sip_content_type);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_type_t *sip_content_type_copy(su_home_t *home, sip_content_type_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_type_t *sip_content_type_copy(su_home_t *home, sip_content_type_t const *hdr)
{
  return (sip_content_type_t *)
    msg_header_copy_as(home, sip_content_type_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_content_type "Content-Type header" structure #sip_content_type_t.
 *
 * The function sip_content_type_make() makes a new
 * #sip_content_type_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_content_type_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_type_t *sip_content_type_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_content_type_t *sip_content_type_make(su_home_t *home, char const *s)
{
  return (sip_content_type_t *)sip_header_make(home, sip_content_type_class, s);
}
#endif

/**Make a @ref sip_content_type "Content-Type header" from formatting result.
 *
 * Make a new #sip_content_type_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_content_type_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_type_t *sip_content_type_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_content_type_t *sip_content_type_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_content_type_class, fmt, ap);
  va_end(ap);

  return (sip_content_type_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_content_encoding
 * @{
 */

/** Parse a SIP @ref sip_content_encoding "Content-Encoding header". @internal */
SOFIAPUBFUN issize_t sip_content_encoding_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_content_encoding "Content-Encoding header". @internal */
SOFIAPUBFUN issize_t sip_content_encoding_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_content_encoding "Content-Encoding header"
 * structure #sip_content_encoding_t from #sip_t.
 *
 */
#define sip_content_encoding(sip) \
  ((sip_content_encoding_t *)msg_header_access((msg_pub_t*)(sip), sip_content_encoding_class))

/**Initializer for structure #sip_content_encoding_t.
 *
 * A static #sip_content_encoding_t structure for
 * @ref sip_content_encoding "Content-Encoding header" must be initialized with
 * the SIP_CONTENT_ENCODING_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_content_encoding_t sip_content_encoding = SIP_CONTENT_ENCODING_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CONTENT_ENCODING_INIT() SIP_HDR_INIT(content_encoding)

/**Initialize a structure #sip_content_encoding_t.
 *
 * An #sip_content_encoding_t structure for
 * @ref sip_content_encoding "Content-Encoding header" can be initialized with the
 * sip_content_encoding_init() function/macro. For instance,
 * @code
 *
 *  sip_content_encoding_t sip_content_encoding;
 *
 *  sip_content_encoding_init(&sip_content_encoding);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_content_encoding_t *sip_content_encoding_init(sip_content_encoding_t x[1])
{
  return SIP_HEADER_INIT(x, sip_content_encoding_class, sizeof(sip_content_encoding_t));
}
#else
#define sip_content_encoding_init(x) \
  SIP_HEADER_INIT(x, sip_content_encoding_class, sizeof(sip_content_encoding_t))
#endif

/**Test if header object is instance of #sip_content_encoding_t.
 *
 * Check if the header class is an instance of
 * @ref sip_content_encoding "Content-Encoding header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header content_encoding
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_content_encoding(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_content_encoding_hash;
}
#else
int sip_is_content_encoding(sip_header_t const *header);
#endif

#define sip_content_encoding_p(h) sip_is_content_encoding((h))


/**Duplicate a list of @ref sip_content_encoding "Content-Encoding header" header structures #sip_content_encoding_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   content_encoding = sip_content_encoding_dup(home, sip->sip_content_encoding);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_content_encoding_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_encoding_t *sip_content_encoding_dup(su_home_t *home, sip_content_encoding_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_encoding_t *sip_content_encoding_dup(su_home_t *home, sip_content_encoding_t const *hdr)
{
  return (sip_content_encoding_t *)
    msg_header_dup_as(home, sip_content_encoding_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_content_encoding "Content-Encoding header" header structures #sip_content_encoding_t.
 *
 * The function sip_content_encoding_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   content_encoding = sip_content_encoding_copy(home, sip->sip_content_encoding);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_encoding_t *sip_content_encoding_copy(su_home_t *home, sip_content_encoding_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_encoding_t *sip_content_encoding_copy(su_home_t *home, sip_content_encoding_t const *hdr)
{
  return (sip_content_encoding_t *)
    msg_header_copy_as(home, sip_content_encoding_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_content_encoding "Content-Encoding header" structure #sip_content_encoding_t.
 *
 * The function sip_content_encoding_make() makes a new
 * #sip_content_encoding_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_content_encoding_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_encoding_t *sip_content_encoding_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_content_encoding_t *sip_content_encoding_make(su_home_t *home, char const *s)
{
  return (sip_content_encoding_t *)sip_header_make(home, sip_content_encoding_class, s);
}
#endif

/**Make a @ref sip_content_encoding "Content-Encoding header" from formatting result.
 *
 * Make a new #sip_content_encoding_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_content_encoding_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_encoding_t *sip_content_encoding_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_content_encoding_t *sip_content_encoding_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_content_encoding_class, fmt, ap);
  va_end(ap);

  return (sip_content_encoding_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_content_language
 * @{
 */

/** Parse a SIP @ref sip_content_language "Content-Language header". @internal */
SOFIAPUBFUN issize_t sip_content_language_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_content_language "Content-Language header". @internal */
SOFIAPUBFUN issize_t sip_content_language_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_content_language "Content-Language header"
 * structure #sip_content_language_t from #sip_t.
 *
 */
#define sip_content_language(sip) \
  ((sip_content_language_t *)msg_header_access((msg_pub_t*)(sip), sip_content_language_class))

/**Initializer for structure #sip_content_language_t.
 *
 * A static #sip_content_language_t structure for
 * @ref sip_content_language "Content-Language header" must be initialized with
 * the SIP_CONTENT_LANGUAGE_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_content_language_t sip_content_language = SIP_CONTENT_LANGUAGE_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CONTENT_LANGUAGE_INIT() SIP_HDR_INIT(content_language)

/**Initialize a structure #sip_content_language_t.
 *
 * An #sip_content_language_t structure for
 * @ref sip_content_language "Content-Language header" can be initialized with the
 * sip_content_language_init() function/macro. For instance,
 * @code
 *
 *  sip_content_language_t sip_content_language;
 *
 *  sip_content_language_init(&sip_content_language);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_content_language_t *sip_content_language_init(sip_content_language_t x[1])
{
  return SIP_HEADER_INIT(x, sip_content_language_class, sizeof(sip_content_language_t));
}
#else
#define sip_content_language_init(x) \
  SIP_HEADER_INIT(x, sip_content_language_class, sizeof(sip_content_language_t))
#endif

/**Test if header object is instance of #sip_content_language_t.
 *
 * Check if the header class is an instance of
 * @ref sip_content_language "Content-Language header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header content_language
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_content_language(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_content_language_hash;
}
#else
int sip_is_content_language(sip_header_t const *header);
#endif

#define sip_content_language_p(h) sip_is_content_language((h))


/**Duplicate a list of @ref sip_content_language "Content-Language header" header structures #sip_content_language_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   content_language = sip_content_language_dup(home, sip->sip_content_language);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_content_language_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_language_t *sip_content_language_dup(su_home_t *home, sip_content_language_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_language_t *sip_content_language_dup(su_home_t *home, sip_content_language_t const *hdr)
{
  return (sip_content_language_t *)
    msg_header_dup_as(home, sip_content_language_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_content_language "Content-Language header" header structures #sip_content_language_t.
 *
 * The function sip_content_language_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   content_language = sip_content_language_copy(home, sip->sip_content_language);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_language_t *sip_content_language_copy(su_home_t *home, sip_content_language_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_language_t *sip_content_language_copy(su_home_t *home, sip_content_language_t const *hdr)
{
  return (sip_content_language_t *)
    msg_header_copy_as(home, sip_content_language_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_content_language "Content-Language header" structure #sip_content_language_t.
 *
 * The function sip_content_language_make() makes a new
 * #sip_content_language_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_content_language_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_language_t *sip_content_language_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_content_language_t *sip_content_language_make(su_home_t *home, char const *s)
{
  return (sip_content_language_t *)sip_header_make(home, sip_content_language_class, s);
}
#endif

/**Make a @ref sip_content_language "Content-Language header" from formatting result.
 *
 * Make a new #sip_content_language_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_content_language_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_language_t *sip_content_language_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_content_language_t *sip_content_language_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_content_language_class, fmt, ap);
  va_end(ap);

  return (sip_content_language_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_content_disposition
 * @{
 */

/** Parse a SIP @ref sip_content_disposition "Content-Disposition header". @internal */
SOFIAPUBFUN issize_t sip_content_disposition_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_content_disposition "Content-Disposition header". @internal */
SOFIAPUBFUN issize_t sip_content_disposition_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_content_disposition "Content-Disposition header"
 * structure #sip_content_disposition_t from #sip_t.
 *
 */
#define sip_content_disposition(sip) \
  ((sip_content_disposition_t *)msg_header_access((msg_pub_t*)(sip), sip_content_disposition_class))

/**Initializer for structure #sip_content_disposition_t.
 *
 * A static #sip_content_disposition_t structure for
 * @ref sip_content_disposition "Content-Disposition header" must be initialized with
 * the SIP_CONTENT_DISPOSITION_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_content_disposition_t sip_content_disposition = SIP_CONTENT_DISPOSITION_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CONTENT_DISPOSITION_INIT() SIP_HDR_INIT(content_disposition)

/**Initialize a structure #sip_content_disposition_t.
 *
 * An #sip_content_disposition_t structure for
 * @ref sip_content_disposition "Content-Disposition header" can be initialized with the
 * sip_content_disposition_init() function/macro. For instance,
 * @code
 *
 *  sip_content_disposition_t sip_content_disposition;
 *
 *  sip_content_disposition_init(&sip_content_disposition);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_content_disposition_t *sip_content_disposition_init(sip_content_disposition_t x[1])
{
  return SIP_HEADER_INIT(x, sip_content_disposition_class, sizeof(sip_content_disposition_t));
}
#else
#define sip_content_disposition_init(x) \
  SIP_HEADER_INIT(x, sip_content_disposition_class, sizeof(sip_content_disposition_t))
#endif

/**Test if header object is instance of #sip_content_disposition_t.
 *
 * Check if the header class is an instance of
 * @ref sip_content_disposition "Content-Disposition header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header content_disposition
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_content_disposition(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_content_disposition_hash;
}
#else
int sip_is_content_disposition(sip_header_t const *header);
#endif

#define sip_content_disposition_p(h) sip_is_content_disposition((h))


/**Duplicate a list of @ref sip_content_disposition "Content-Disposition header" header structures #sip_content_disposition_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   content_disposition = sip_content_disposition_dup(home, sip->sip_content_disposition);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_content_disposition_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_disposition_t *sip_content_disposition_dup(su_home_t *home, sip_content_disposition_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_disposition_t *sip_content_disposition_dup(su_home_t *home, sip_content_disposition_t const *hdr)
{
  return (sip_content_disposition_t *)
    msg_header_dup_as(home, sip_content_disposition_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_content_disposition "Content-Disposition header" header structures #sip_content_disposition_t.
 *
 * The function sip_content_disposition_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   content_disposition = sip_content_disposition_copy(home, sip->sip_content_disposition);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_disposition_t *sip_content_disposition_copy(su_home_t *home, sip_content_disposition_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_disposition_t *sip_content_disposition_copy(su_home_t *home, sip_content_disposition_t const *hdr)
{
  return (sip_content_disposition_t *)
    msg_header_copy_as(home, sip_content_disposition_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_content_disposition "Content-Disposition header" structure #sip_content_disposition_t.
 *
 * The function sip_content_disposition_make() makes a new
 * #sip_content_disposition_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_content_disposition_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_disposition_t *sip_content_disposition_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_content_disposition_t *sip_content_disposition_make(su_home_t *home, char const *s)
{
  return (sip_content_disposition_t *)sip_header_make(home, sip_content_disposition_class, s);
}
#endif

/**Make a @ref sip_content_disposition "Content-Disposition header" from formatting result.
 *
 * Make a new #sip_content_disposition_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_content_disposition_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_disposition_t *sip_content_disposition_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_content_disposition_t *sip_content_disposition_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_content_disposition_class, fmt, ap);
  va_end(ap);

  return (sip_content_disposition_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_content_length
 * @{
 */

/** Parse a SIP @ref sip_content_length "Content-Length header". @internal */
SOFIAPUBFUN issize_t sip_content_length_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_content_length "Content-Length header". @internal */
SOFIAPUBFUN issize_t sip_content_length_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_content_length "Content-Length header"
 * structure #sip_content_length_t from #sip_t.
 *
 */
#define sip_content_length(sip) \
  ((sip_content_length_t *)msg_header_access((msg_pub_t*)(sip), sip_content_length_class))

/**Initializer for structure #sip_content_length_t.
 *
 * A static #sip_content_length_t structure for
 * @ref sip_content_length "Content-Length header" must be initialized with
 * the SIP_CONTENT_LENGTH_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_content_length_t sip_content_length = SIP_CONTENT_LENGTH_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_CONTENT_LENGTH_INIT() SIP_HDR_INIT(content_length)

/**Initialize a structure #sip_content_length_t.
 *
 * An #sip_content_length_t structure for
 * @ref sip_content_length "Content-Length header" can be initialized with the
 * sip_content_length_init() function/macro. For instance,
 * @code
 *
 *  sip_content_length_t sip_content_length;
 *
 *  sip_content_length_init(&sip_content_length);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_content_length_t *sip_content_length_init(sip_content_length_t x[1])
{
  return SIP_HEADER_INIT(x, sip_content_length_class, sizeof(sip_content_length_t));
}
#else
#define sip_content_length_init(x) \
  SIP_HEADER_INIT(x, sip_content_length_class, sizeof(sip_content_length_t))
#endif

/**Test if header object is instance of #sip_content_length_t.
 *
 * Check if the header class is an instance of
 * @ref sip_content_length "Content-Length header" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header content_length
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_content_length(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_content_length_hash;
}
#else
int sip_is_content_length(sip_header_t const *header);
#endif

#define sip_content_length_p(h) sip_is_content_length((h))


/**Duplicate a list of @ref sip_content_length "Content-Length header" header structures #sip_content_length_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   content_length = sip_content_length_dup(home, sip->sip_content_length);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_content_length_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_length_t *sip_content_length_dup(su_home_t *home, sip_content_length_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_length_t *sip_content_length_dup(su_home_t *home, sip_content_length_t const *hdr)
{
  return (sip_content_length_t *)
    msg_header_dup_as(home, sip_content_length_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_content_length "Content-Length header" header structures #sip_content_length_t.
 *
 * The function sip_content_length_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   content_length = sip_content_length_copy(home, sip->sip_content_length);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_length_t *sip_content_length_copy(su_home_t *home, sip_content_length_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_content_length_t *sip_content_length_copy(su_home_t *home, sip_content_length_t const *hdr)
{
  return (sip_content_length_t *)
    msg_header_copy_as(home, sip_content_length_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_content_length "Content-Length header" structure #sip_content_length_t.
 *
 * The function sip_content_length_make() makes a new
 * #sip_content_length_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_content_length_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_length_t *sip_content_length_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_content_length_t *sip_content_length_make(su_home_t *home, char const *s)
{
  return (sip_content_length_t *)sip_header_make(home, sip_content_length_class, s);
}
#endif

/**Make a @ref sip_content_length "Content-Length header" from formatting result.
 *
 * Make a new #sip_content_length_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_content_length_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_content_length_t *sip_content_length_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_content_length_t *sip_content_length_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_content_length_class, fmt, ap);
  va_end(ap);

  return (sip_content_length_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_unknown
 * @{
 */

/** Parse a SIP @ref sip_unknown "unknown headers". @internal */
SOFIAPUBFUN issize_t sip_unknown_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_unknown "unknown headers". @internal */
SOFIAPUBFUN issize_t sip_unknown_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_unknown "unknown headers"
 * structure #sip_unknown_t from #sip_t.
 *
 */
#define sip_unknown(sip) \
  ((sip_unknown_t *)msg_header_access((msg_pub_t*)(sip), sip_unknown_class))

/**Initializer for structure #sip_unknown_t.
 *
 * A static #sip_unknown_t structure for
 * @ref sip_unknown "unknown headers" must be initialized with
 * the SIP_UNKNOWN_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_unknown_t sip_unknown = SIP_UNKNOWN_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_UNKNOWN_INIT() SIP_HDR_INIT(unknown)

/**Initialize a structure #sip_unknown_t.
 *
 * An #sip_unknown_t structure for
 * @ref sip_unknown "unknown headers" can be initialized with the
 * sip_unknown_init() function/macro. For instance,
 * @code
 *
 *  sip_unknown_t sip_unknown;
 *
 *  sip_unknown_init(&sip_unknown);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_unknown_t *sip_unknown_init(sip_unknown_t x[1])
{
  return SIP_HEADER_INIT(x, sip_unknown_class, sizeof(sip_unknown_t));
}
#else
#define sip_unknown_init(x) \
  SIP_HEADER_INIT(x, sip_unknown_class, sizeof(sip_unknown_t))
#endif

/**Test if header object is instance of #sip_unknown_t.
 *
 * Check if the header class is an instance of
 * @ref sip_unknown "unknown headers" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header unknown
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_unknown(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_unknown_hash;
}
#else
int sip_is_unknown(sip_header_t const *header);
#endif

#define sip_unknown_p(h) sip_is_unknown((h))


/**Duplicate a list of @ref sip_unknown "unknown headers" header structures #sip_unknown_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   unknown = sip_unknown_dup(home, sip->sip_unknown);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_unknown_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_unknown_t *sip_unknown_dup(su_home_t *home, sip_unknown_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_unknown_t *sip_unknown_dup(su_home_t *home, sip_unknown_t const *hdr)
{
  return (sip_unknown_t *)
    msg_header_dup_as(home, sip_unknown_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_unknown "unknown headers" header structures #sip_unknown_t.
 *
 * The function sip_unknown_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   unknown = sip_unknown_copy(home, sip->sip_unknown);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_unknown_t *sip_unknown_copy(su_home_t *home, sip_unknown_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_unknown_t *sip_unknown_copy(su_home_t *home, sip_unknown_t const *hdr)
{
  return (sip_unknown_t *)
    msg_header_copy_as(home, sip_unknown_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_unknown "unknown headers" structure #sip_unknown_t.
 *
 * The function sip_unknown_make() makes a new
 * #sip_unknown_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_unknown_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_unknown_t *sip_unknown_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_unknown_t *sip_unknown_make(su_home_t *home, char const *s)
{
  return (sip_unknown_t *)sip_header_make(home, sip_unknown_class, s);
}
#endif

/**Make a @ref sip_unknown "unknown headers" from formatting result.
 *
 * Make a new #sip_unknown_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_unknown_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_unknown_t *sip_unknown_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_unknown_t *sip_unknown_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_unknown_class, fmt, ap);
  va_end(ap);

  return (sip_unknown_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_error
 * @{
 */

/** Parse a SIP @ref sip_error "erroneous headers". @internal */
SOFIAPUBFUN issize_t sip_error_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_error "erroneous headers". @internal */
SOFIAPUBFUN issize_t sip_error_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_error "erroneous headers"
 * structure #sip_error_t from #sip_t.
 *
 */
#define sip_error(sip) \
  ((sip_error_t *)msg_header_access((msg_pub_t*)(sip), sip_error_class))

/**Initializer for structure #sip_error_t.
 *
 * A static #sip_error_t structure for
 * @ref sip_error "erroneous headers" must be initialized with
 * the SIP_ERROR_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_error_t sip_error = SIP_ERROR_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_ERROR_INIT() SIP_HDR_INIT(error)

/**Initialize a structure #sip_error_t.
 *
 * An #sip_error_t structure for
 * @ref sip_error "erroneous headers" can be initialized with the
 * sip_error_init() function/macro. For instance,
 * @code
 *
 *  sip_error_t sip_error;
 *
 *  sip_error_init(&sip_error);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_error_t *sip_error_init(sip_error_t x[1])
{
  return SIP_HEADER_INIT(x, sip_error_class, sizeof(sip_error_t));
}
#else
#define sip_error_init(x) \
  SIP_HEADER_INIT(x, sip_error_class, sizeof(sip_error_t))
#endif

/**Test if header object is instance of #sip_error_t.
 *
 * Check if the header class is an instance of
 * @ref sip_error "erroneous headers" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header error
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_error(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_error_hash;
}
#else
int sip_is_error(sip_header_t const *header);
#endif

#define sip_error_p(h) sip_is_error((h))


/**Duplicate a list of @ref sip_error "erroneous headers" header structures #sip_error_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   error = sip_error_dup(home, sip->sip_error);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_error_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_error_t *sip_error_dup(su_home_t *home, sip_error_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_error_t *sip_error_dup(su_home_t *home, sip_error_t const *hdr)
{
  return (sip_error_t *)
    msg_header_dup_as(home, sip_error_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_error "erroneous headers" header structures #sip_error_t.
 *
 * The function sip_error_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   error = sip_error_copy(home, sip->sip_error);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_error_t *sip_error_copy(su_home_t *home, sip_error_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_error_t *sip_error_copy(su_home_t *home, sip_error_t const *hdr)
{
  return (sip_error_t *)
    msg_header_copy_as(home, sip_error_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_error "erroneous headers" structure #sip_error_t.
 *
 * The function sip_error_make() makes a new
 * #sip_error_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_error_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_error_t *sip_error_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_error_t *sip_error_make(su_home_t *home, char const *s)
{
  return (sip_error_t *)sip_header_make(home, sip_error_class, s);
}
#endif

/**Make a @ref sip_error "erroneous headers" from formatting result.
 *
 * Make a new #sip_error_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_error_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_error_t *sip_error_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_error_t *sip_error_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_error_class, fmt, ap);
  va_end(ap);

  return (sip_error_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_separator
 * @{
 */

/** Parse a SIP @ref sip_separator "separator line between headers and body". @internal */
SOFIAPUBFUN issize_t sip_separator_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_separator "separator line between headers and body". @internal */
SOFIAPUBFUN issize_t sip_separator_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_separator "separator line between headers and body"
 * structure #sip_separator_t from #sip_t.
 *
 */
#define sip_separator(sip) \
  ((sip_separator_t *)msg_header_access((msg_pub_t*)(sip), sip_separator_class))

/**Initializer for structure #sip_separator_t.
 *
 * A static #sip_separator_t structure for
 * @ref sip_separator "separator line between headers and body" must be initialized with
 * the SIP_SEPARATOR_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_separator_t sip_separator = SIP_SEPARATOR_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_SEPARATOR_INIT() SIP_HDR_INIT(separator)

/**Initialize a structure #sip_separator_t.
 *
 * An #sip_separator_t structure for
 * @ref sip_separator "separator line between headers and body" can be initialized with the
 * sip_separator_init() function/macro. For instance,
 * @code
 *
 *  sip_separator_t sip_separator;
 *
 *  sip_separator_init(&sip_separator);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_separator_t *sip_separator_init(sip_separator_t x[1])
{
  return SIP_HEADER_INIT(x, sip_separator_class, sizeof(sip_separator_t));
}
#else
#define sip_separator_init(x) \
  SIP_HEADER_INIT(x, sip_separator_class, sizeof(sip_separator_t))
#endif

/**Test if header object is instance of #sip_separator_t.
 *
 * Check if the header class is an instance of
 * @ref sip_separator "separator line between headers and body" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header separator
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_separator(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_separator_hash;
}
#else
int sip_is_separator(sip_header_t const *header);
#endif

#define sip_separator_p(h) sip_is_separator((h))


/**Duplicate a list of @ref sip_separator "separator line between headers and body" header structures #sip_separator_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   separator = sip_separator_dup(home, sip->sip_separator);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_separator_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_separator_t *sip_separator_dup(su_home_t *home, sip_separator_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_separator_t *sip_separator_dup(su_home_t *home, sip_separator_t const *hdr)
{
  return (sip_separator_t *)
    msg_header_dup_as(home, sip_separator_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_separator "separator line between headers and body" header structures #sip_separator_t.
 *
 * The function sip_separator_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   separator = sip_separator_copy(home, sip->sip_separator);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_separator_t *sip_separator_copy(su_home_t *home, sip_separator_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_separator_t *sip_separator_copy(su_home_t *home, sip_separator_t const *hdr)
{
  return (sip_separator_t *)
    msg_header_copy_as(home, sip_separator_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_separator "separator line between headers and body" structure #sip_separator_t.
 *
 * The function sip_separator_make() makes a new
 * #sip_separator_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_separator_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_separator_t *sip_separator_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_separator_t *sip_separator_make(su_home_t *home, char const *s)
{
  return (sip_separator_t *)sip_header_make(home, sip_separator_class, s);
}
#endif

/**Make a @ref sip_separator "separator line between headers and body" from formatting result.
 *
 * Make a new #sip_separator_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_separator_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_separator_t *sip_separator_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_separator_t *sip_separator_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_separator_class, fmt, ap);
  va_end(ap);

  return (sip_separator_t *)h;
}
#endif

/** @} */

/**@addtogroup sip_payload
 * @{
 */

/** Parse a SIP @ref sip_payload "message payload". @internal */
SOFIAPUBFUN issize_t sip_payload_d(su_home_t *, msg_header_t *,
				       char *s, isize_t slen);

/** Print a SIP @ref sip_payload "message payload". @internal */
SOFIAPUBFUN issize_t sip_payload_e(char b[], isize_t bsiz,
                    	            msg_header_t const *h, int flags);

/**Access a SIP @ref sip_payload "message payload"
 * structure #sip_payload_t from #sip_t.
 *
 */
#define sip_payload(sip) \
  ((sip_payload_t *)msg_header_access((msg_pub_t*)(sip), sip_payload_class))

/**Initializer for structure #sip_payload_t.
 *
 * A static #sip_payload_t structure for
 * @ref sip_payload "message payload" must be initialized with
 * the SIP_PAYLOAD_INIT() macro.
 * For instance,
 * @code
 *
 *  sip_payload_t sip_payload = SIP_PAYLOAD_INIT;
 *
 * @endcode
 * @HI
 *
 */
#define SIP_PAYLOAD_INIT() SIP_HDR_INIT(payload)

/**Initialize a structure #sip_payload_t.
 *
 * An #sip_payload_t structure for
 * @ref sip_payload "message payload" can be initialized with the
 * sip_payload_init() function/macro. For instance,
 * @code
 *
 *  sip_payload_t sip_payload;
 *
 *  sip_payload_init(&sip_payload);
 *
 * @endcode
 * @HI
 *
 */
#if SU_HAVE_INLINE
su_inline sip_payload_t *sip_payload_init(sip_payload_t x[1])
{
  return SIP_HEADER_INIT(x, sip_payload_class, sizeof(sip_payload_t));
}
#else
#define sip_payload_init(x) \
  SIP_HEADER_INIT(x, sip_payload_class, sizeof(sip_payload_t))
#endif

/**Test if header object is instance of #sip_payload_t.
 *
 * Check if the header class is an instance of
 * @ref sip_payload "message payload" object and return true (nonzero),
 * otherwise return false (zero).
 *
 * @param header pointer to the header structure to be tested
 *
 * @retval 1 (true) if the @a header is an instance of header payload
 * @retval 0 (false) otherwise
 *
 */
#if SU_HAVE_INLINE
su_inline int sip_is_payload(sip_header_t const *header)
{
  return header && header->sh_class->hc_hash == sip_payload_hash;
}
#else
int sip_is_payload(sip_header_t const *header);
#endif

#define sip_payload_p(h) sip_is_payload((h))


/**Duplicate a list of @ref sip_payload "message payload" header structures #sip_payload_t.
 *
 * Duplicate a header
 * structure @a hdr.  If the header structure @a hdr
 * contains a reference (@c hdr->x_next) to a list of
 * headers, all the headers in the list are duplicated, too.
 *
 * @param home  memory home used to allocate new structure
 * @param hdr   header structure to be duplicated
 *
 * When duplicating, all parameter lists and non-constant
 * strings attached to the header are copied, too.  The
 * function uses given memory @a home to allocate all the
 * memory areas used to copy the header.
 *
 * @par Example
 * @code
 *
 *   payload = sip_payload_dup(home, sip->sip_payload);
 *
 * @endcode
 *
 * @return
 * A pointer to the
 * newly duplicated #sip_payload_t header structure, or NULL
 * upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_payload_t *sip_payload_dup(su_home_t *home, sip_payload_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_payload_t *sip_payload_dup(su_home_t *home, sip_payload_t const *hdr)
{
  return (sip_payload_t *)
    msg_header_dup_as(home, sip_payload_class, (msg_header_t const *)hdr);
}
#endif

/**Copy a list of @ref sip_payload "message payload" header structures #sip_payload_t.
 *
 * The function sip_payload_copy() copies a header structure @a
 * hdr.  If the header structure @a hdr contains a reference (@c
 * hdr->h_next) to a list of headers, all the headers in that
 * list are copied, too. The function uses given memory @a home
 * to allocate all the memory areas used to copy the list of header
 * structure @a hdr.
 *
 * @param home    memory home used to allocate new structure
 * @param hdr     pointer to the header structure to be copied
 *
 * When copying, only the header structure and parameter lists attached to
 * it are duplicated. The new header structure retains all the references to
 * the strings within the old @a hdr header, including the encoding of the
 * old header, if present.
 *
 * @par Example
 * @code
 *
 *   payload = sip_payload_copy(home, sip->sip_payload);
 *
 * @endcode
 *
 * @return
 * A pointer to newly copied header structure, or NULL upon an error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_payload_t *sip_payload_copy(su_home_t *home, sip_payload_t const *hdr)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline
sip_payload_t *sip_payload_copy(su_home_t *home, sip_payload_t const *hdr)
{
  return (sip_payload_t *)
    msg_header_copy_as(home, sip_payload_class, (msg_header_t const *)hdr);
}
#endif

/**Make a @ref sip_payload "message payload" structure #sip_payload_t.
 *
 * The function sip_payload_make() makes a new
 * #sip_payload_t header structure.  It allocates a new
 * header structure, and decodes the string @a s as the
 * value of the structure.
 *
 * @param home memory home used to allocate new header structure.
 * @param s    string to be decoded as value of the new header structure
 *
 * @return
 * A pointer to newly maked #sip_payload_t header structure, or NULL upon an
 * error.
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_payload_t *sip_payload_make(su_home_t *home, char const *s)
     __attribute__((__malloc__));

#if SU_HAVE_INLINE
su_inline sip_payload_t *sip_payload_make(su_home_t *home, char const *s)
{
  return (sip_payload_t *)sip_header_make(home, sip_payload_class, s);
}
#endif

/**Make a @ref sip_payload "message payload" from formatting result.
 *
 * Make a new #sip_payload_t object using formatting result as its value.
 * The function first prints the arguments according to the format @a fmt
 * specified. Then it allocates a new header structure, and parses the
 * formatting result to the structure #sip_payload_t.
 *
 * @param home   memory home used to allocate new header structure.
 * @param fmt    string used as a printf()-style format
 * @param ...    argument list for format
 *
 * @return
 * A pointer to newly
 * makes header structure, or NULL upon an error.
 *
 * @HIDE
 *
 */
#if SU_HAVE_INLINE
su_inline
#endif
sip_payload_t *sip_payload_format(su_home_t *home, char const *fmt, ...)
     __attribute__((__malloc__, __format__ (printf, 2, 3)));

#if SU_HAVE_INLINE
su_inline sip_payload_t *sip_payload_format(su_home_t *home, char const *fmt, ...)
{
  sip_header_t *h;
  va_list ap;

  va_start(ap, fmt);
  h = sip_header_vformat(home, sip_payload_class, fmt, ap);
  va_end(ap);

  return (sip_payload_t *)h;
}
#endif

/** @} */

SOFIA_END_DECLS
#endif /* !defined(SIP_PROTOS_H) */
