/*
 * Decompiled with CFR 0.152.
 */
package jadx.gui.jobs;

import jadx.api.utils.tasks.ITaskExecutor;
import jadx.core.utils.Utils;
import jadx.core.utils.exceptions.JadxRuntimeException;
import jadx.gui.jobs.IBackgroundTask;
import jadx.gui.jobs.InternalTask;
import jadx.gui.jobs.ProgressUpdater;
import jadx.gui.jobs.SimpleTask;
import jadx.gui.jobs.TaskStatus;
import jadx.gui.settings.JadxSettings;
import jadx.gui.ui.panel.ProgressPanel;
import jadx.gui.utils.NLS;
import jadx.gui.utils.UiUtils;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackgroundExecutor {
    private static final Logger LOG = LoggerFactory.getLogger(BackgroundExecutor.class);
    private final JadxSettings settings;
    private final ProgressUpdater progressUpdater;
    private ThreadPoolExecutor taskQueueExecutor;
    private final Map<Long, InternalTask> taskRunning = new ConcurrentHashMap<Long, InternalTask>();
    private final AtomicLong idSupplier = new AtomicLong(0L);

    public BackgroundExecutor(JadxSettings settings, ProgressPanel progressPane) {
        this.settings = Objects.requireNonNull(settings);
        this.progressUpdater = new ProgressUpdater(progressPane, this::taskCanceled);
        this.reset();
    }

    public synchronized void execute(IBackgroundTask task) {
        InternalTask internalTask = this.buildTask(task);
        this.taskQueueExecutor.execute(() -> this.runTask(internalTask));
    }

    public synchronized Future<TaskStatus> executeWithFuture(IBackgroundTask task) {
        InternalTask internalTask = this.buildTask(task);
        return this.taskQueueExecutor.submit(() -> {
            this.runTask(internalTask);
            return internalTask.getStatus();
        });
    }

    public synchronized void cancelAll() {
        try {
            this.taskRunning.values().forEach(this::cancelTask);
            this.taskQueueExecutor.shutdownNow();
            boolean complete = this.taskQueueExecutor.awaitTermination(3L, TimeUnit.SECONDS);
            if (complete) {
                LOG.debug("Background task executor canceled successfully");
            } else {
                String taskNames = this.taskRunning.values().stream().map(t15 -> t15.getBgTask().getTitle()).collect(Collectors.joining(", "));
                LOG.debug("Background task executor cancel failed. Running tasks: {}", (Object)taskNames);
            }
        }
        catch (Exception e15) {
            LOG.error("Error terminating task executor", e15);
        }
        finally {
            this.reset();
        }
    }

    public synchronized void waitForComplete() {
        try {
            this.taskQueueExecutor.submit(UiUtils.EMPTY_RUNNABLE).get();
        }
        catch (Exception e15) {
            throw new JadxRuntimeException("Failed to wait tasks completion", e15);
        }
    }

    public void execute(String title, List<Runnable> backgroundJobs, Consumer<TaskStatus> onFinishUiRunnable) {
        this.execute(new SimpleTask(title, backgroundJobs, onFinishUiRunnable));
    }

    public void execute(String title, Runnable backgroundRunnable, Consumer<TaskStatus> onFinishUiRunnable) {
        this.execute(new SimpleTask(title, Collections.singletonList(backgroundRunnable), onFinishUiRunnable));
    }

    public void execute(String title, Runnable backgroundRunnable) {
        this.execute(new SimpleTask(title, Collections.singletonList(backgroundRunnable)));
    }

    public void startLoading(Runnable backgroundRunnable, Runnable onFinishUiRunnable) {
        this.execute(new SimpleTask(NLS.str("progress.load"), backgroundRunnable, onFinishUiRunnable));
    }

    public void startLoading(Runnable backgroundRunnable) {
        this.execute(new SimpleTask(NLS.str("progress.load"), backgroundRunnable));
    }

    private synchronized void reset() {
        this.taskQueueExecutor = (ThreadPoolExecutor)Executors.newFixedThreadPool(1, Utils.simpleThreadFactory("bg"));
        this.taskRunning.clear();
        this.idSupplier.set(0L);
    }

    private InternalTask buildTask(IBackgroundTask task) {
        long id5 = this.idSupplier.incrementAndGet();
        InternalTask internalTask = new InternalTask(id5, task);
        this.taskRunning.put(id5, internalTask);
        return internalTask;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runTask(InternalTask internalTask) {
        try {
            IBackgroundTask task = internalTask.getBgTask();
            ITaskExecutor taskExecutor = task.scheduleTasks();
            taskExecutor.setThreadsCount(this.settings.getThreadsCount());
            int tasksCount = taskExecutor.getTasksCount();
            internalTask.setTaskExecutor(taskExecutor);
            internalTask.setJobsCount(tasksCount);
            if (UiUtils.JADX_GUI_DEBUG) {
                LOG.debug("Starting background task '{}', jobs count: {}, time limit: {} ms, memory check: {}", task.getTitle(), tasksCount, task.timeLimit(), task.checkMemoryUsage());
            }
            long startTime = System.currentTimeMillis();
            Supplier<TaskStatus> cancelCheck = this.buildCancelCheck(internalTask, startTime);
            internalTask.taskStart(startTime, cancelCheck);
            this.progressUpdater.addTask(internalTask);
            taskExecutor.execute();
            taskExecutor.awaitTermination();
        }
        catch (Exception e15) {
            LOG.error("Task failed", e15);
            internalTask.setStatus(TaskStatus.ERROR);
        }
        finally {
            this.taskComplete(internalTask);
        }
    }

    private void taskComplete(InternalTask internalTask) {
        try {
            IBackgroundTask task = internalTask.getBgTask();
            internalTask.setJobsComplete(internalTask.getTaskExecutor().getProgress());
            internalTask.setStatus(TaskStatus.COMPLETE);
            internalTask.updateExecTime();
            task.onDone(internalTask);
            UiUtils.uiRunAndWait(() -> {
                try {
                    task.onFinish(internalTask);
                }
                catch (Exception e15) {
                    LOG.error("Task onFinish failed", e15);
                    internalTask.setStatus(TaskStatus.ERROR);
                }
            });
        }
        catch (Exception e15) {
            LOG.error("Task complete failed", e15);
            internalTask.setStatus(TaskStatus.ERROR);
        }
        finally {
            internalTask.taskComplete();
            this.progressUpdater.taskComplete(internalTask);
            this.removeTask(internalTask);
        }
    }

    private void removeTask(InternalTask internalTask) {
        this.taskRunning.remove(internalTask.getId());
    }

    private void cancelTask(InternalTask internalTask) {
        try {
            IBackgroundTask task = internalTask.getBgTask();
            if (!internalTask.isRunning()) {
                task.cancel();
                this.removeTask(internalTask);
                return;
            }
            ITaskExecutor taskExecutor = internalTask.getTaskExecutor();
            task.cancel();
            taskExecutor.terminate();
            ExecutorService executor = taskExecutor.getInternalExecutor();
            if (executor == null) {
                return;
            }
            int cancelTimeout = task.getCancelTimeoutMS();
            if (cancelTimeout != 0 && executor.awaitTermination(cancelTimeout, TimeUnit.MILLISECONDS)) {
                LOG.debug("Task cancel complete");
                return;
            }
            LOG.debug("Forcing tasks cancel");
            executor.shutdownNow();
            boolean complete = executor.awaitTermination(task.getShutdownTimeoutMS(), TimeUnit.MILLISECONDS);
            LOG.debug("Forced task cancel status: {}", complete ? "success" : "fail, still active: " + (taskExecutor.getTasksCount() - taskExecutor.getProgress()));
        }
        catch (Exception e15) {
            LOG.error("Failed to cancel task: {}", (Object)internalTask, (Object)e15);
        }
    }

    private void taskCanceled(InternalTask task) {
        this.cancelTask(task);
    }

    private Supplier<TaskStatus> buildCancelCheck(InternalTask internalTask, long startTime) {
        IBackgroundTask task = internalTask.getBgTask();
        int timeLimit = task.timeLimit();
        long waitUntilTime = timeLimit == 0 ? 0L : startTime + (long)timeLimit;
        boolean checkMemoryUsage = task.checkMemoryUsage();
        return () -> {
            if (task.isCanceled() || Thread.currentThread().isInterrupted()) {
                return TaskStatus.CANCEL_BY_USER;
            }
            if (waitUntilTime != 0L && waitUntilTime < System.currentTimeMillis()) {
                LOG.warn("Task '{}' execution timeout, force cancel", (Object)task.getTitle());
                return TaskStatus.CANCEL_BY_TIMEOUT;
            }
            if (checkMemoryUsage && !UiUtils.isFreeMemoryAvailable()) {
                LOG.warn("High memory usage: {}", (Object)UiUtils.memoryInfo());
                if (internalTask.getTaskExecutor().getThreadsCount() == 1) {
                    LOG.warn("Task '{}' memory limit reached, force cancel", (Object)task.getTitle());
                    return TaskStatus.CANCEL_BY_MEMORY;
                }
                LOG.warn("Low free memory, reduce processing threads count to 1");
                internalTask.getTaskExecutor().setThreadsCount(1);
                System.gc();
                UiUtils.sleep(1000);
                if (!UiUtils.isFreeMemoryAvailable()) {
                    LOG.error("Task '{}' memory limit reached (after GC), force cancel", (Object)task.getTitle());
                    return TaskStatus.CANCEL_BY_MEMORY;
                }
            }
            return null;
        };
    }
}

