/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package tlsconfig

import (
	"testing"

	"github.com/google/go-cmp/cmp"
)

func TestCreateDetails(t *testing.T) {
	t.Parallel()

	type args struct {
		session   string
		dbConnect string
		caFile    string
		certFile  string
		keyFile   string
		uri       string
	}

	tests := []struct {
		name    string
		args    args
		want    Details
		wantErr bool
	}{
		{
			name: "+verifyFullAllFieldsSet",
			args: args{
				session:   "s1",
				dbConnect: "verify_full",
				caFile:    "ca",
				certFile:  "cert",
				keyFile:   "key",
				uri:       "tcp://localhost:5432",
			},
			want: Details{
				SessionName: "s1",
				TLSConnect:  "verify_full",
				TLSCaFile:   "ca",
				TLSCertFile: "cert",
				TLSKeyFile:  "key",
				RawURI:      "tcp://localhost:5432",
			},
			wantErr: false,
		},
		{
			name: "+unencryptedConnectionNoFilesSet",
			args: args{
				session:   "s2",
				dbConnect: "",
				caFile:    "",
				certFile:  "",
				keyFile:   "",
				uri:       "tcp://localhost:5432",
			},
			want: Details{
				SessionName: "s2",
				TLSConnect:  "",
				TLSCaFile:   "",
				TLSCertFile: "",
				TLSKeyFile:  "",
				RawURI:      "tcp://localhost:5432",
			},
			wantErr: false,
		},
		{
			name: "+requiredNoFilesSet",
			args: args{
				session:   "s3",
				dbConnect: "required",
				caFile:    "",
				certFile:  "",
				keyFile:   "",
				uri:       "tcp://localhost:5432",
			},
			want: Details{
				SessionName: "s3",
				TLSConnect:  "required",
				TLSCaFile:   "",
				TLSCertFile: "",
				TLSKeyFile:  "",
				RawURI:      "tcp://localhost:5432",
			},
			wantErr: false,
		},
		{
			name: "-verify_fullMissingFile",
			args: args{
				session:   "s1",
				dbConnect: "verify_full",
				caFile:    "",
				certFile:  "cert",
				keyFile:   "key",
				uri:       "tcp://localhost:5432",
			},
			want:    Details{},
			wantErr: true,
		},
		{
			name: "-unencryptedConnectionFileSet",
			args: args{
				session:   "s2",
				dbConnect: "",
				caFile:    "ca",
				certFile:  "",
				keyFile:   "",
				uri:       "tcp://localhost:5432",
			},
			want:    Details{},
			wantErr: true,
		},
		{
			name: "-requiredFilesSet",
			args: args{
				session:   "s3",
				dbConnect: "required",
				caFile:    "ca",
				certFile:  "cert",
				keyFile:   "key",
				uri:       "tcp://localhost:5432",
			},
			want:    Details{},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, err := CreateDetails(
				tt.args.session,
				tt.args.dbConnect,
				tt.args.caFile,
				tt.args.certFile,
				tt.args.keyFile,
				tt.args.uri,
			)
			if (err != nil) != tt.wantErr {
				t.Fatalf("CreateDetails() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("CreateDetails() = %s", diff)
			}
		})
	}
}

func Test_validateSetTLSFiles(t *testing.T) {
	t.Parallel()

	type args struct {
		caFile   string
		certFile string
		keyFile  string
	}

	tests := []struct {
		name    string
		args    args
		wantErr bool
	}{
		{
			name: "+allFilesProvided",
			args: args{
				caFile:   "ca.pem",
				certFile: "cert.pem",
				keyFile:  "key.pem",
			},
			wantErr: false,
		},
		{
			name: "-missingCAFile",
			args: args{
				caFile:   "",
				certFile: "cert.pem",
				keyFile:  "key.pem",
			},
			wantErr: true,
		},
		{
			name: "-missingCertificateFile",
			args: args{
				caFile:   "ca.pem",
				certFile: "",
				keyFile:  "key.pem",
			},
			wantErr: true,
		},
		{
			name: "-missingKeyFile",
			args: args{
				caFile:   "ca.pem",
				certFile: "cert.pem",
				keyFile:  "",
			},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			err := validateSetTLSFiles(tt.args.caFile, tt.args.certFile, tt.args.keyFile)
			if (err != nil) != tt.wantErr {
				t.Fatalf("validateSetTLSFiles() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func Test_validateUnsetTLSFiles(t *testing.T) {
	t.Parallel()

	type args struct {
		caFile   string
		certFile string
		keyFile  string
	}

	tests := []struct {
		name    string
		args    args
		wantErr bool
	}{
		{
			name: "+allFilesUnset",
			args: args{
				caFile:   "",
				certFile: "",
				keyFile:  "",
			},
			wantErr: false,
		},
		{
			name: "-caFileSet",
			args: args{
				caFile:   "ca.pem",
				certFile: "",
				keyFile:  "",
			},
			wantErr: true,
		},
		{
			name: "-certificateFileSet",
			args: args{
				caFile:   "",
				certFile: "cert.pem",
				keyFile:  "",
			},
			wantErr: true,
		},
		{
			name: "-keyFileSet",
			args: args{
				caFile:   "",
				certFile: "",
				keyFile:  "key.pem",
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			err := validateUnsetTLSFiles(tt.args.caFile, tt.args.certFile, tt.args.keyFile)
			if (err != nil) != tt.wantErr {
				t.Fatalf("validateUnsetTLSFiles() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}
