/**********************************************************************
  Copyright(c) 2011-2016 Intel Corporation All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**********************************************************************/

#ifndef AES_CBC_STD_VECTORS_H_
#define AES_CBC_STD_VECTORS_H_
#include <aes_cbc.h>

// struct to hold pointers to the cbc data vectors
struct cbc_vector {
        uint8_t *K;              // AES Key
        isal_cbc_key_size K_LEN; // length of key in bits
        uint8_t *IV;             // initial value used by GCM
        uint64_t P_LEN;          // length of our plaintext
        uint8_t *P;              // Plain text
        // outputs of encryption
        uint8_t *EXP_C; // same length as P
        // used in vector checks, not populated in std vector array
        uint8_t *C;
        struct isal_cbc_key_data *KEYS;
};

///////////////////////////////////////////
// Test vectors from:
// Intel IPSec library 1..3
//
///////////////////////////////////////////
static unsigned char K1[] = { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                              0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c };
static unsigned char IV1[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f };
static unsigned char P1[] = { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
                              0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
                              0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30,
                              0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19,
                              0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b,
                              0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 };
static unsigned char C1[] = { 0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46, 0xce, 0xe9, 0x8e,
                              0x9b, 0x12, 0xe9, 0x19, 0x7d, 0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72,
                              0x19, 0xee, 0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2, 0x73,
                              0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b, 0x71, 0x16, 0xe6, 0x9e,
                              0x22, 0x22, 0x95, 0x16, 0x3f, 0xf1, 0xca, 0xa1, 0x68, 0x1f, 0xac,
                              0x09, 0x12, 0x0e, 0xca, 0x30, 0x75, 0x86, 0xe1, 0xa7 };

static unsigned char K2[] = { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae,
                              0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61,
                              0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 };
static unsigned char IV2[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f };
static unsigned char P2[] = { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
                              0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
                              0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30,
                              0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19,
                              0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b,
                              0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 };
static unsigned char C2[] = { 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e, 0xab,
                              0xfb, 0x5f, 0x7b, 0xfb, 0xd6, 0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb,
                              0x80, 0x8d, 0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d, 0x39,
                              0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf, 0xa5, 0x30, 0xe2, 0x63,
                              0x04, 0x23, 0x14, 0x61, 0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9,
                              0xfc, 0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b };

static unsigned char K3[] = { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
                              0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
                              0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7 };
static unsigned char IV3[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f };
static unsigned char P3[] = { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
                              0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
                              0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30,
                              0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19,
                              0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b,
                              0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 };
static unsigned char C3[] = { 0x17, 0x70, 0x1a, 0x9d, 0x29, 0xc9, 0x1a, 0x94, 0xce, 0xed, 0x72,
                              0x3c, 0x34, 0xe8, 0x7a, 0xbe, 0x1c, 0x96, 0x84, 0x5c, 0xa8, 0xb7,
                              0xe8, 0x58, 0x6d, 0xfe, 0xf2, 0xfa, 0x6b, 0xed, 0x24, 0x09, 0x8a,
                              0x52, 0xce, 0xe8, 0xd7, 0x6d, 0xb6, 0x7b, 0xfd, 0xe2, 0x15, 0x53,
                              0xd3, 0x1c, 0x28, 0x33, 0xf7, 0x7e, 0xb5, 0x95, 0x00, 0xac, 0x49,
                              0x03, 0xbc, 0x70, 0x76, 0xb1, 0x84, 0x65, 0xd0, 0xea };

///////////////////////////////////////////
// Test vectors from:
// 'https://tools.ietf.org/html/rfc3602#section-3.2'
//  The AES-CBC Cipher Algorithm and Its Use with IPsec
//
///////////////////////////////////////////
/*
Case #1: Encrypting 16 bytes (1 block) using AES-CBC with 128-bit key
Key       : 0x06a9214036b8a15b512e03d534120006
IV        : 0x3dafba429d9eb430b422da802c9fac41
Plaintext : "Single block msg"
Ciphertext: 0xe353779c1079aeb82708942dbe77181a
 *
 */
static unsigned char K4[] = { 0x06, 0xa9, 0x21, 0x40, 0x36, 0xb8, 0xa1, 0x5b,
                              0x51, 0x2e, 0x03, 0xd5, 0x34, 0x12, 0x00, 0x06 };
static unsigned char IV4[] = { 0x3d, 0xaf, 0xba, 0x42, 0x9d, 0x9e, 0xb4, 0x30,
                               0xb4, 0x22, 0xda, 0x80, 0x2c, 0x9f, 0xac, 0x41 };
static unsigned char P4[] = { "Single block msg" };
static unsigned char C4[] = { 0xe3, 0x53, 0x77, 0x9c, 0x10, 0x79, 0xae, 0xb8,
                              0x27, 0x08, 0x94, 0x2d, 0xbe, 0x77, 0x18, 0x1a };

/*
Case #2: Encrypting 32 bytes (2 blocks) using AES-CBC with 128-bit key
Key       : 0xc286696d887c9aa0611bbb3e2025a45a
IV        : 0x562e17996d093d28ddb3ba695a2e6f58
Plaintext : 0x000102030405060708090a0b0c0d0e0f
              101112131415161718191a1b1c1d1e1f
Ciphertext: 0xd296cd94c2cccf8a3a863028b5e1dc0a
              7586602d253cfff91b8266bea6d61ab1
*/
static unsigned char K5[] = { 0xc2, 0x86, 0x69, 0x6d, 0x88, 0x7c, 0x9a, 0xa0,
                              0x61, 0x1b, 0xbb, 0x3e, 0x20, 0x25, 0xa4, 0x5a };
static unsigned char IV5[] = { 0x56, 0x2e, 0x17, 0x99, 0x6d, 0x09, 0x3d, 0x28,
                               0xdd, 0xb3, 0xba, 0x69, 0x5a, 0x2e, 0x6f, 0x58 };
static unsigned char P5[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
                              0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
                              0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f };
static unsigned char C5[] = { 0xd2, 0x96, 0xcd, 0x94, 0xc2, 0xcc, 0xcf, 0x8a, 0x3a, 0x86, 0x30,
                              0x28, 0xb5, 0xe1, 0xdc, 0x0a, 0x75, 0x86, 0x60, 0x2d, 0x25, 0x3c,
                              0xff, 0xf9, 0x1b, 0x82, 0x66, 0xbe, 0xa6, 0xd6, 0x1a, 0xb1 };

/*
Case #3: Encrypting 48 bytes (3 blocks) using AES-CBC with 128-bit key
Key       : 0x6c3ea0477630ce21a2ce334aa746c2cd
IV        : 0xc782dc4c098c66cbd9cd27d825682c81
Plaintext : "This is a 48-byte message (exactly 3 AES blocks)"
Ciphertext: 0xd0a02b3836451753d493665d33f0e886
              2dea54cdb293abc7506939276772f8d5
              021c19216bad525c8579695d83ba2684

 */
static unsigned char K6[] = { 0x6c, 0x3e, 0xa0, 0x47, 0x76, 0x30, 0xce, 0x21,
                              0xa2, 0xce, 0x33, 0x4a, 0xa7, 0x46, 0xc2, 0xcd };
static unsigned char IV6[] = { 0xc7, 0x82, 0xdc, 0x4c, 0x09, 0x8c, 0x66, 0xcb,
                               0xd9, 0xcd, 0x27, 0xd8, 0x25, 0x68, 0x2c, 0x81 };
static unsigned char P6[] = { "This is a 48-byte message (exactly 3 AES blocks)" };
static unsigned char C6[] = { 0xd0, 0xa0, 0x2b, 0x38, 0x36, 0x45, 0x17, 0x53, 0xd4, 0x93,
                              0x66, 0x5d, 0x33, 0xf0, 0xe8, 0x86, 0x2d, 0xea, 0x54, 0xcd,
                              0xb2, 0x93, 0xab, 0xc7, 0x50, 0x69, 0x39, 0x27, 0x67, 0x72,
                              0xf8, 0xd5, 0x02, 0x1c, 0x19, 0x21, 0x6b, 0xad, 0x52, 0x5c,
                              0x85, 0x79, 0x69, 0x5d, 0x83, 0xba, 0x26, 0x84 };

/*
Case #4: Encrypting 64 bytes (4 blocks) using AES-CBC with 128-bit key
Key       : 0x56e47a38c5598974bc46903dba290349
IV        : 0x8ce82eefbea0da3c44699ed7db51b7d9
Plaintext : 0xa0a1a2a3a4a5a6a7a8a9aaabacadaeaf
              b0b1b2b3b4b5b6b7b8b9babbbcbdbebf
              c0c1c2c3c4c5c6c7c8c9cacbcccdcecf
              d0d1d2d3d4d5d6d7d8d9dadbdcdddedf
Ciphertext: 0xc30e32ffedc0774e6aff6af0869f71aa
              0f3af07a9a31a9c684db207eb0ef8e4e
              35907aa632c3ffdf868bb7b29d3d46ad
              83ce9f9a102ee99d49a53e87f4c3da55
 */
static unsigned char K7[] = { 0x56, 0xe4, 0x7a, 0x38, 0xc5, 0x59, 0x89, 0x74,
                              0xbc, 0x46, 0x90, 0x3d, 0xba, 0x29, 0x03, 0x49 };
static unsigned char IV7[] = { 0x8c, 0xe8, 0x2e, 0xef, 0xbe, 0xa0, 0xda, 0x3c,
                               0x44, 0x69, 0x9e, 0xd7, 0xdb, 0x51, 0xb7, 0xd9 };
static unsigned char P7[] = { 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa,
                              0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5,
                              0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0,
                              0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
                              0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
                              0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf };
static unsigned char C7[] = { 0xc3, 0x0e, 0x32, 0xff, 0xed, 0xc0, 0x77, 0x4e, 0x6a, 0xff, 0x6a,
                              0xf0, 0x86, 0x9f, 0x71, 0xaa, 0x0f, 0x3a, 0xf0, 0x7a, 0x9a, 0x31,
                              0xa9, 0xc6, 0x84, 0xdb, 0x20, 0x7e, 0xb0, 0xef, 0x8e, 0x4e, 0x35,
                              0x90, 0x7a, 0xa6, 0x32, 0xc3, 0xff, 0xdf, 0x86, 0x8b, 0xb7, 0xb2,
                              0x9d, 0x3d, 0x46, 0xad, 0x83, 0xce, 0x9f, 0x9a, 0x10, 0x2e, 0xe9,
                              0x9d, 0x49, 0xa5, 0x3e, 0x87, 0xf4, 0xc3, 0xda, 0x55 };

/*
Case #5: Sample transport-mode ESP packet (ping 192.168.123.100)
Key: 90d382b4 10eeba7a d938c46c ec1a82bf
SPI: 4321
Source address: 192.168.123.3
Destination address: 192.168.123.100
Sequence number: 1
IV: e96e8c08 ab465763 fd098d45 dd3ff893

Original packet:
IP header (20 bytes): 45000054 08f20000 4001f9fe c0a87b03 c0a87b64
Data (64 bytes):
08000ebd a70a0000 8e9c083d b95b0700 08090a0b 0c0d0e0f 10111213 14151617
18191a1b 1c1d1e1f 20212223 24252627 28292a2b 2c2d2e2f 30313233 34353637

Augment data with:
Padding: 01020304 05060708 090a0b0c 0d0e
Pad length: 0e
Next header: 01 (ICMP)

Pre-encryption Data with padding, pad length and next header (80 bytes):
08000ebd a70a0000 8e9c083d b95b0700 08090a0b 0c0d0e0f 10111213 14151617
18191a1b 1c1d1e1f 20212223 24252627 28292a2b 2c2d2e2f 30313233 34353637
01020304 05060708 090a0b0c 0d0e0e01

Post-encryption packet with SPI, Sequence number, IV:
IP header: 4500007c 08f20000 4032f9a5 c0a87b03 c0a87b64
SPI/Seq #: 00004321 00000001
IV: e96e8c08 ab465763 fd098d45 dd3ff893
Encrypted Data (80 bytes):
f663c25d 325c18c6 a9453e19 4e120849 a4870b66 cc6b9965 330013b4 898dc856
a4699e52 3a55db08 0b59ec3a 8e4b7e52 775b07d1 db34ed9c 538ab50c 551b874a
a269add0 47ad2d59 13ac19b7 cfbad4a6
*/
static unsigned char K8[] = { 0x90, 0xd3, 0x82, 0xb4, 0x10, 0xee, 0xba, 0x7a,
                              0xd9, 0x38, 0xc4, 0x6c, 0xec, 0x1a, 0x82, 0xbf };
static unsigned char IV8[] = { 0xe9, 0x6e, 0x8c, 0x08, 0xab, 0x46, 0x57, 0x63,
                               0xfd, 0x09, 0x8d, 0x45, 0xdd, 0x3f, 0xf8, 0x93 };
static unsigned char P8[] = { 0x08, 0x00, 0x0e, 0xbd, 0xa7, 0x0a, 0x00, 0x00, 0x8e, 0x9c,
                              0x08, 0x3d, 0xb9, 0x5b, 0x07, 0x00, 0x08, 0x09, 0x0a, 0x0b,
                              0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
                              0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                              0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
                              0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                              0x34, 0x35, 0x36, 0x37, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
                              0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0e, 0x01 };
static unsigned char C8[] = { 0xf6, 0x63, 0xc2, 0x5d, 0x32, 0x5c, 0x18, 0xc6, 0xa9, 0x45,
                              0x3e, 0x19, 0x4e, 0x12, 0x08, 0x49, 0xa4, 0x87, 0x0b, 0x66,
                              0xcc, 0x6b, 0x99, 0x65, 0x33, 0x00, 0x13, 0xb4, 0x89, 0x8d,
                              0xc8, 0x56, 0xa4, 0x69, 0x9e, 0x52, 0x3a, 0x55, 0xdb, 0x08,
                              0x0b, 0x59, 0xec, 0x3a, 0x8e, 0x4b, 0x7e, 0x52, 0x77, 0x5b,
                              0x07, 0xd1, 0xdb, 0x34, 0xed, 0x9c, 0x53, 0x8a, 0xb5, 0x0c,
                              0x55, 0x1b, 0x87, 0x4a, 0xa2, 0x69, 0xad, 0xd0, 0x47, 0xad,
                              0x2d, 0x59, 0x13, 0xac, 0x19, 0xb7, 0xcf, 0xba, 0xd4, 0xa6 };

/*
Case #6: Sample transport-mode ESP packet
         (ping -p 77 -s 20 192.168.123.100)
Key: 90d382b4 10eeba7a d938c46c ec1a82bf
SPI: 4321
Source address: 192.168.123.3
Destination address: 192.168.123.100
Sequence number: 8
IV: 69d08df7 d203329d b093fc49 24e5bd80

Original packet:
IP header (20 bytes): 45000030 08fe0000 4001fa16 c0a87b03 c0a87b64
Data (28 bytes):
0800b5e8 a80a0500 a69c083d 0b660e00 77777777 77777777 77777777

Augment data with:
Padding: 0102
Pad length: 02
Next header: 01 (ICMP)

Pre-encryption Data with padding, pad length and next header (32 bytes):
0800b5e8 a80a0500 a69c083d 0b660e00 77777777 77777777 77777777 01020201

Post-encryption packet with SPI, Sequence number, IV:
IP header: 4500004c 08fe0000 4032f9c9 c0a87b03 c0a87b64
SPI/Seq #: 00004321 00000008
IV: 69d08df7 d203329d b093fc49 24e5bd80
Encrypted Data (32 bytes):
f5199588 1ec4e0c4 488987ce 742e8109 689bb379 d2d750c0 d915dca3 46a89f75
 */
static unsigned char K9[] = { 0x90, 0xd3, 0x82, 0xb4, 0x10, 0xee, 0xba, 0x7a,
                              0xd9, 0x38, 0xc4, 0x6c, 0xec, 0x1a, 0x82, 0xbf };
static unsigned char IV9[] = { 0x69, 0xd0, 0x8d, 0xf7, 0xd2, 0x03, 0x32, 0x9d,
                               0xb0, 0x93, 0xfc, 0x49, 0x24, 0xe5, 0xbd, 0x80 };
static unsigned char P9[] = { 0x08, 0x00, 0xb5, 0xe8, 0xa8, 0x0a, 0x05, 0x00, 0xa6, 0x9c, 0x08,
                              0x3d, 0x0b, 0x66, 0x0e, 0x00, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
                              0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x01, 0x02, 0x02, 0x01 };
static unsigned char C9[] = { 0xf5, 0x19, 0x95, 0x88, 0x1e, 0xc4, 0xe0, 0xc4, 0x48, 0x89, 0x87,
                              0xce, 0x74, 0x2e, 0x81, 0x09, 0x68, 0x9b, 0xb3, 0x79, 0xd2, 0xd7,
                              0x50, 0xc0, 0xd9, 0x15, 0xdc, 0xa3, 0x46, 0xa8, 0x9f, 0x75 };

/*
Case #7: Sample tunnel-mode ESP packet (ping 192.168.123.200)
Key: 01234567 89abcdef 01234567 89abcdef
SPI: 8765
Source address: 192.168.123.3
Destination address: 192.168.123.200
Sequence number: 2
IV: f4e76524 4f6407ad f13dc138 0f673f37

Original packet:
IP header (20 bytes): 45000054 09040000 4001f988 c0a87b03 c0a87bc8
Data (64 bytes):
08009f76 a90a0100 b49c083d 02a20400 08090a0b 0c0d0e0f 10111213 14151617
18191a1b 1c1d1e1f 20212223 24252627 28292a2b 2c2d2e2f 30313233 34353637

Augment data with:
Padding: 01020304 05060708 090a
Pad length: 0a
Next header: 04 (IP-in-IP)

Pre-encryption Data with original IP header, padding, pad length and
                         next header (96 bytes):
45000054 09040000 4001f988 c0a87b03 c0a87bc8 08009f76 a90a0100 b49c083d
02a20400 08090a0b 0c0d0e0f 10111213 14151617 18191a1b 1c1d1e1f 20212223
24252627 28292a2b 2c2d2e2f 30313233 34353637 01020304 05060708 090a0a04


Post-encryption packet with SPI, Sequence number, IV:
IP header: 4500008c 09050000 4032f91e c0a87b03 c0a87bc8
SPI/Seq #: 00008765 00000002
IV: f4e76524 4f6407ad f13dc138 0f673f37
Encrypted Data (96 bytes):
773b5241 a4c44922 5e4f3ce5 ed611b0c 237ca96c f74a9301 3c1b0ea1 a0cf70f8
e4ecaec7 8ac53aad 7a0f022b 859243c6 47752e94 a859352b 8a4d4d2d ecd136e5
c177f132 ad3fbfb2 201ac990 4c74ee0a 109e0ca1 e4dfe9d5 a100b842 f1c22f0d
 */
static unsigned char K10[] = { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
                               0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef };
static unsigned char IV10[] = { 0xf4, 0xe7, 0x65, 0x24, 0x4f, 0x64, 0x07, 0xad,
                                0xf1, 0x3d, 0xc1, 0x38, 0x0f, 0x67, 0x3f, 0x37 };
static unsigned char P10[] = {
        0x45, 0x00, 0x00, 0x54, 0x09, 0x04, 0x00, 0x00, 0x40, 0x01, 0xf9, 0x88, 0xc0, 0xa8,
        0x7b, 0x03, 0xc0, 0xa8, 0x7b, 0xc8, 0x08, 0x00, 0x9f, 0x76, 0xa9, 0x0a, 0x01, 0x00,
        0xb4, 0x9c, 0x08, 0x3d, 0x02, 0xa2, 0x04, 0x00, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
        0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
        0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
        0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0a, 0x04

};
static unsigned char C10[] = { 0x77, 0x3b, 0x52, 0x41, 0xa4, 0xc4, 0x49, 0x22, 0x5e, 0x4f, 0x3c,
                               0xe5, 0xed, 0x61, 0x1b, 0x0c, 0x23, 0x7c, 0xa9, 0x6c, 0xf7, 0x4a,
                               0x93, 0x01, 0x3c, 0x1b, 0x0e, 0xa1, 0xa0, 0xcf, 0x70, 0xf8, 0xe4,
                               0xec, 0xae, 0xc7, 0x8a, 0xc5, 0x3a, 0xad, 0x7a, 0x0f, 0x02, 0x2b,
                               0x85, 0x92, 0x43, 0xc6, 0x47, 0x75, 0x2e, 0x94, 0xa8, 0x59, 0x35,
                               0x2b, 0x8a, 0x4d, 0x4d, 0x2d, 0xec, 0xd1, 0x36, 0xe5, 0xc1, 0x77,
                               0xf1, 0x32, 0xad, 0x3f, 0xbf, 0xb2, 0x20, 0x1a, 0xc9, 0x90, 0x4c,
                               0x74, 0xee, 0x0a, 0x10, 0x9e, 0x0c, 0xa1, 0xe4, 0xdf, 0xe9, 0xd5,
                               0xa1, 0x00, 0xb8, 0x42, 0xf1, 0xc2, 0x2f, 0x0d };

/*
Case #8: Sample tunnel-mode ESP packet
         (ping -p ff -s 40 192.168.123.200)
Key: 01234567 89abcdef 01234567 89abcdef
SPI: 8765
Source address: 192.168.123.3
Destination address: 192.168.123.200
Sequence number: 5
IV: 85d47224 b5f3dd5d 2101d4ea 8dffab22

Original packet:
IP header (20 bytes): 45000044 090c0000 4001f990 c0a87b03 c0a87bc8
Data (48 bytes):
0800d63c aa0a0200 c69c083d a3de0300 ffffffff ffffffff ffffffff ffffffff
ffffffff ffffffff ffffffff ffffffff

Augment data with:
Padding: 01020304 05060708 090a
Pad length: 0a
Next header: 04 (IP-in-IP)

Pre-encryption Data with original IP header, padding, pad length and
                         next header (80 bytes):
45000044 090c0000 4001f990 c0a87b03 c0a87bc8 0800d63c aa0a0200 c69c083d
a3de0300 ffffffff ffffffff ffffffff ffffffff ffffffff ffffffff ffffffff
ffffffff 01020304 05060708 090a0a04

Post-encryption packet with SPI, Sequence number, IV:
IP header: 4500007c 090d0000 4032f926 c0a87b03 c0a87bc8
SPI/Seq #: 00008765 00000005
IV: 85d47224 b5f3dd5d 2101d4ea 8dffab22
Encrypted Data (80 bytes):
15b92683 819596a8 047232cc 00f7048f e45318e1 1f8a0f62 ede3c3fc 61203bb5
0f980a08 c9843fd3 a1b06d5c 07ff9639 b7eb7dfb 3512e5de 435e7207 ed971ef3
d2726d9b 5ef6affc 6d17a0de cbb13892
 */
static unsigned char K11[] = { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
                               0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef };
static unsigned char IV11[] = { 0x85, 0xd4, 0x72, 0x24, 0xb5, 0xf3, 0xdd, 0x5d,
                                0x21, 0x01, 0xd4, 0xea, 0x8d, 0xff, 0xab, 0x22 };
static unsigned char P11[] = { 0x45, 0x00, 0x00, 0x44, 0x09, 0x0c, 0x00, 0x00, 0x40, 0x01,
                               0xf9, 0x90, 0xc0, 0xa8, 0x7b, 0x03, 0xc0, 0xa8, 0x7b, 0xc8,
                               0x08, 0x00, 0xd6, 0x3c, 0xaa, 0x0a, 0x02, 0x00, 0xc6, 0x9c,
                               0x08, 0x3d, 0xa3, 0xde, 0x03, 0x00, 0xff, 0xff, 0xff, 0xff,
                               0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                               0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                               0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x02,
                               0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0a, 0x04 };
static unsigned char C11[] = { 0x15, 0xb9, 0x26, 0x83, 0x81, 0x95, 0x96, 0xa8, 0x04, 0x72,
                               0x32, 0xcc, 0x00, 0xf7, 0x04, 0x8f, 0xe4, 0x53, 0x18, 0xe1,
                               0x1f, 0x8a, 0x0f, 0x62, 0xed, 0xe3, 0xc3, 0xfc, 0x61, 0x20,
                               0x3b, 0xb5, 0x0f, 0x98, 0x0a, 0x08, 0xc9, 0x84, 0x3f, 0xd3,
                               0xa1, 0xb0, 0x6d, 0x5c, 0x07, 0xff, 0x96, 0x39, 0xb7, 0xeb,
                               0x7d, 0xfb, 0x35, 0x12, 0xe5, 0xde, 0x43, 0x5e, 0x72, 0x07,
                               0xed, 0x97, 0x1e, 0xf3, 0xd2, 0x72, 0x6d, 0x9b, 0x5e, 0xf6,
                               0xaf, 0xfc, 0x6d, 0x17, 0xa0, 0xde, 0xcb, 0xb1, 0x38, 0x92 };

#define min_size(a, b) (((a) < (b)) ? (a) : (b))
// Plain and cypher text will be the same size
// Those vectors using strings for plain text have an extra null terminator that needs
// to be ignored
#define vect_size(P, C)    (min_size((sizeof(P)), (sizeof(C))))
#define CBC_KEY_LEN(kdata) (sizeof(kdata))

// field order {K, Klen, IV, Plen, P, C};
#define vector(N)                                                                                  \
        {                                                                                          \
                K##N, (CBC_KEY_LEN(K##N)), IV##N, vect_size(P##N, C##N), P##N, C##N, NULL,         \
                NULL, /*NULL, NULL*/                                                               \
        }
struct cbc_vector const cbc_vectors[] = {
        vector(1), vector(2), vector(3), vector(4),  vector(5),  vector(6),
        vector(7), vector(8), vector(9), vector(10), vector(11),
};

#endif /* AES_CBC_STD_VECTORS_H_ */
