﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/cloud9/model/ConnectionType.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace Cloud9 {
namespace Model {
namespace ConnectionTypeMapper {

static const int CONNECT_SSH_HASH = HashingUtils::HashString("CONNECT_SSH");
static const int CONNECT_SSM_HASH = HashingUtils::HashString("CONNECT_SSM");

ConnectionType GetConnectionTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == CONNECT_SSH_HASH) {
    return ConnectionType::CONNECT_SSH;
  } else if (hashCode == CONNECT_SSM_HASH) {
    return ConnectionType::CONNECT_SSM;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<ConnectionType>(hashCode);
  }

  return ConnectionType::NOT_SET;
}

Aws::String GetNameForConnectionType(ConnectionType enumValue) {
  switch (enumValue) {
    case ConnectionType::NOT_SET:
      return {};
    case ConnectionType::CONNECT_SSH:
      return "CONNECT_SSH";
    case ConnectionType::CONNECT_SSM:
      return "CONNECT_SSM";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ConnectionTypeMapper
}  // namespace Model
}  // namespace Cloud9
}  // namespace Aws
