﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/acm-pca/ACMPCARequest.h>
#include <aws/acm-pca/ACMPCA_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ACMPCA {
namespace Model {

/**
 */
class DescribeCertificateAuthorityAuditReportRequest : public ACMPCARequest {
 public:
  AWS_ACMPCA_API DescribeCertificateAuthorityAuditReportRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeCertificateAuthorityAuditReport"; }

  AWS_ACMPCA_API Aws::String SerializePayload() const override;

  AWS_ACMPCA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the private CA. This must be of the
   * form:</p> <p>
   * <code>arn:aws:acm-pca:<i>region</i>:<i>account</i>:certificate-authority/<i>12345678-1234-1234-1234-123456789012</i>
   * </code>. </p>
   */
  inline const Aws::String& GetCertificateAuthorityArn() const { return m_certificateAuthorityArn; }
  inline bool CertificateAuthorityArnHasBeenSet() const { return m_certificateAuthorityArnHasBeenSet; }
  template <typename CertificateAuthorityArnT = Aws::String>
  void SetCertificateAuthorityArn(CertificateAuthorityArnT&& value) {
    m_certificateAuthorityArnHasBeenSet = true;
    m_certificateAuthorityArn = std::forward<CertificateAuthorityArnT>(value);
  }
  template <typename CertificateAuthorityArnT = Aws::String>
  DescribeCertificateAuthorityAuditReportRequest& WithCertificateAuthorityArn(CertificateAuthorityArnT&& value) {
    SetCertificateAuthorityArn(std::forward<CertificateAuthorityArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The report ID returned by calling the <a
   * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html">CreateCertificateAuthorityAuditReport</a>
   * action.</p>
   */
  inline const Aws::String& GetAuditReportId() const { return m_auditReportId; }
  inline bool AuditReportIdHasBeenSet() const { return m_auditReportIdHasBeenSet; }
  template <typename AuditReportIdT = Aws::String>
  void SetAuditReportId(AuditReportIdT&& value) {
    m_auditReportIdHasBeenSet = true;
    m_auditReportId = std::forward<AuditReportIdT>(value);
  }
  template <typename AuditReportIdT = Aws::String>
  DescribeCertificateAuthorityAuditReportRequest& WithAuditReportId(AuditReportIdT&& value) {
    SetAuditReportId(std::forward<AuditReportIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_certificateAuthorityArn;
  bool m_certificateAuthorityArnHasBeenSet = false;

  Aws::String m_auditReportId;
  bool m_auditReportIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace ACMPCA
}  // namespace Aws
