\page ExampleBlineTracer Example BlineTracer

\author Laurent Desorgher (desorgher@phim.unibe.ch) - 04/10/2003

The BlineTracer module allows to trace and visualise magnetic field
lines in a Geant4 application where particle are tracked through the
magnetic field (in future, these functionalities may be integrated in
the Geant4 kernel).

To use the tracer, the user should copy the classes provided here in
his/her own application/example and create somewhere in his/her code
an instance of a G4BlineTracer object.
It can be anywhere (for example in the main code or in the user class
defining the magnetic field):

```cpp
#include "G4BlineTracer.hh"
G4BlineTracer* theBlineTool = new G4BlineTracer();
```

## Design principles

The core of the tool is the method G4BlineTracer::ComputeBlines() of G4BlineTracer class.
In this method a bline is computed by tracking a ChargedGeantino in the user
defined magnetic field and by using a Bline equation of motion (class
G4BlineEquation, motion along the field) instead of a Lorentz equation.

During the execution of this method :

 - The user defined  equations of motion associated to the different
  global  and local fields are replaced by instances of G4BlineEquation
  associated to the same fields.
  The G4BlineEquation class defines the differential equation of a
  magnetic field line.

 - User defined ChordFinders are replaced by new ChordFinders
  associated to the G4BlineEquation object.

 - The user primary generator action, run action, event action and
  stepping action are replaced by instances of the classes
  G4BlinePrimaryGeneratorAction, G4BlineTracer, G4BlineEventAction
  and G4BlineSteppingAction respectively. Other actions are set to
  a NULL pointer.

After the execution of the method, the original user defined actions, equation
of motions and chord-finders are restored.

In other words, the deafult run-action is temporarly replaced by a
G4BlineTracer run-action using the same Geometry, physics and magnetic fields
but with a different equation of motion and user actions.

The  G4BlinePrimaryGeneratorAction::GeneratePrimaries() method of the G4BlinePrimaryGeneratorAction class
call the GeneratePrimaries() method of the user defined PrimaryGeneratorAction
for defining the start position and start time of tracking.
Start position for user application and for Bline tracking are therefore
controlled by the same UI commands.
The type of particles to be tracked when tracing Blines is always set
to Charged-Geantino. This allows to switch off the effect of electromagnetic
and hadronics physics when tracing field lines.


The G4BlineEventAction class is responsible to store computed magnetic field
lines as a vector of Polylines and Polymarkers for later visualisation.
These vectors can be drawn and reset at any time.

The stepping action does nothing in this implementation but it can be used
in future versions to limit field line tracing to physical volumes defined
by the user. For this purpose a G4BlineStackingAction could also be
implemented.

## User Manual

### General description

The BlineTracer is controlled by the UI commands contained in the directory
/vis/blineTracer. By calling the command 'computeBline', several magnetic field
lines passing through user defined start positions are computed.
Start positions are generated by the user primary generator action.
By doing so, the definition of start positions is the same for usual particles
tracking and magnetic field line tracking.

A magnetic field line is computed as a track of a charged geantino that moves
along the field line. The user can define the maximum length of a tracking step
(only valid for Bline tracing purposes) by the use of the 'setMaxStepLength'
command.

By using small enough maximum step length, smooth magnetic field lines are
obtained. By using the command 'stockLines' and 'stockPoints' the user
can decide to store the series of tracking step positions defining
a magnetic field line as a Polyline object and/or a PolyMarker object
(circles) respectively.

These objects are stored in vectors of PolyLines and PolyMarkers.
By using the command 'draw', these vectors are added to the scene
of the visualisation manager, provided that a scene handler and
visualisation driver have been properly created.

The scene is visualised by invoking the vis command '/vis/show'.
Polyline objects are visualised as line segments joining the different
step positions defining a line, while for a Polymarker object markers
(here circles), are  drawn at each step positions.

By using the 'setColour' the user defines the visualisation colour
that will be associated to the next computed magnetic field lines.
By calling 'setPointSize' the user defines the size of visualisation markers
that will be  associated to the next computed magnetic field lines.
The user can remove the vector of Polymarker and Polyline from the memory
by invoking 'resetMaterialToBeDrawn'.

When using small max step size and polymarkers for visualisation purposes, the
thickness of a smooth magnetic field line is obtained. It is controlled by the
Marker size parameter ('setPointSize')

### Command description

```
/vis/blineTracer/computeBline  nb_of_lines
```
    - Parameters: integer nb_of_lines
    - Description: Compute nb_of_lines different magnetic field lines

```
/vis/blineTracer/setMaxStepLength max_step_length
```
    - Parameters: double max_step_length
    - Description: Set the maximum tracking step length for computing
		   magnetic field lines

```
/vis/blineTracer/setColour  red green blue
```
    - Parameters: double red, green, blue
    - Description: Define  the colour for visualisation of the
		   next computed magnetic field lines. The color is
		   defined by a  RGB code (red,green,blue)  with all
		   parameters smaller than 1.

```
 /vis/blineTracer/stockLines aBool
```
    - Parameters: boolean aBool
    - Description: If true the next computed field lines are stored
		   as Polylines for further visualisation

```
/vis/blineTracer/stockLines aBool
```
    - Parameters: boolean aBool
    - Description: If true the next computed field lines are stored
	           as Polymarkers for further visualisation

```
/vis/blineTracer/setPointSize point_size
```
    - Parameters: double point_size
    - Description: set the size of the visualisation  markers
		   that will be associated with the next computed
		   magnetic field lines

```
/vis/blineTracer/resetMaterialToBeDrawn
```
    - Parameters: none
    - Description: The vector of Polyline and Polymarker representing
		   magnetic field lines to be visualised are removed
		   from memory

```
/tracking/storeTrajectory 1
```
    - If the storeTrajectory parameter is not set no field lines are
      stored.

  An example for Bline visualisation is provided in the bline_vis.mac macro.

## Current limitations & known problems

 The tool is working properly only for detectors parts where magnetic
 field are defined. It is planned in the future to stop the tracking of
 field lines in regions where no fields are existing.
