"use strict";
/**
 * The plugin is intended to override and/or extend some behaviour of ts service,
 * please before implementing custom logic here, first consider to implement logic based on tsserver standard api.
 */
const get_element_type_ts_server_1 = require("./get-element-type-ts-server");
const ide_project_service_1 = require("./ide-project-service");
const ide_commands_1 = require("./ide-commands");
const openExternalProjectExCommand = "ideOpenExternalProject";
const projectInfoExCommand = "ideProjectInfo";
const emitFileExCommand = "ideCompileOnSaveEmitFile";
let addedCommands = false;
let nextProjectId = 0;
const init = ((modules) => {
    const ts = modules.typescript;
    let tsVersion = parseNumbersInVersion(ts.version);
    function addNewHandlers(info) {
        let projectService = info.project.projectService;
        projectService.logger.info("Called handler processing");
        patchProjectService(projectService);
        const session = info.session;
        if (session == undefined) {
            projectService.logger.info("There is no session in info.");
            return;
        }
        if (session.addProtocolHandler == undefined) {
            // addProtocolHandler was introduced in TS 4.4 or 4.5 in 2021, see https://github.com/microsoft/TypeScript/issues/43893
            projectService.logger.info("There is no addProtocolHandler method.");
            return;
        }
        if (!isVersionMoreOrEqual(tsVersion, 5)) {
            projectService.logger.info("Skip adding handlers for old version: " + ts.version);
            return;
        }
        session.addProtocolHandler(openExternalProjectExCommand, openExternalProjectHandler.bind(null, ts, projectService));
        session.addProtocolHandler(projectInfoExCommand, projectInfoHandler.bind(null, ts, projectService));
        session.addProtocolHandler(emitFileExCommand, emitFileHandler.bind(null, ts, projectService));
        (0, ide_commands_1.registerProtocolHandlers)(session, ts, projectService);
        projectService.logger.info("IDE specific commands are successfully added.");
    }
    function patchProjectService(projectService) {
        if (projectService.ideProjectService)
            return;
        projectService.ideProjectService = new ide_project_service_1.IdeProjectService(ts, projectService);
    }
    function create(info) {
        if (!addedCommands) {
            addNewHandlers(info);
            addedCommands = true;
        }
        if (info.project.ideProjectId === undefined) {
            info.project.ideProjectId = nextProjectId++;
        }
        info.languageService.ideProjectId = info.project.ideProjectId;
        (0, get_element_type_ts_server_1.decorateLanguageService)(info.languageService);
        return info.languageService;
    }
    return { create };
});
function openExternalProjectEx(ts, projectService, externalProject) {
    let fileName = externalProject.projectFileName;
    if (fileName == null)
        return; //nothing to open
    const tsConfigPath = ts.server.toNormalizedPath(fileName);
    // Normalize file names
    externalProject = Object.assign(Object.assign({}, externalProject), { projectFileName: tsConfigPath, rootFiles: externalProject.rootFiles.map(it => (Object.assign(Object.assign({}, it), { fileName: ts.server.toNormalizedPath(it.fileName) }))) });
    /*
     * There's no public API method to open a project with the given config file.
     * The projectService.openExternalProject() method doesn't take the config file as a
     * parameter; instead, it tries to infer the config file, allowing only tsconfig.json
     * and jsconfig.json names. Therefore, we use this method only when the name matches.
     * Otherwise, we have to use private methods.
     */
    if (tsConfigPath.endsWith("/tsconfig.json") ||
        tsConfigPath.endsWith("/jsconfig.json") ||
        !tsConfigPath.endsWith(".json")) {
        projectService.openExternalProject(externalProject);
        return;
    }
    if (!projectService.ideProjectService.tryOpenProject(tsConfigPath)) {
        // TypeScript will anyway try to infer the project on the first 'open'. However, we throw to appear in logs.
        throw new Error(`Could not create configured project with tsConfigPath=${tsConfigPath}`);
    }
}
const openExternalProjectHandler = (ts, projectService, request) => {
    let openRequest = request;
    let externalProject = openRequest.arguments;
    openExternalProjectEx(ts, projectService, externalProject);
    return {
        response: true,
        responseRequired: true
    };
};
const emitFileHandler = (ts, projectService, request) => {
    let host = projectService.host;
    let compileRequest = request;
    let args = compileRequest.arguments;
    let file = args.file ? ts.server.toNormalizedPath(args.file) : undefined;
    let projectFileName = args.projectFileName;
    if (file == null) {
        projectService.logger.info("There is no file to compile: " + projectFileName);
        return {
            response: {
                emitSkipped: true,
                diagnostics: []
            },
            responseRequired: true
        };
    }
    let diagnostics = [];
    let generatedFiles = [];
    let processedFiles = [];
    let result = {
        emitSkipped: false,
        generatedFiles,
        processedFiles,
        diagnostics: [
            {
                file,
                diagnostics: diagnostics
            }
        ]
    };
    let project = projectFileName ?
        projectService.findProject(projectFileName) :
        projectService.getDefaultProjectForFile(file, false);
    let openedProject = false;
    if (project == null) {
        let fileToOpen = projectFileName ? projectFileName : file;
        openExternalProjectEx(ts, projectService, {
            options: {},
            projectFileName: fileToOpen,
            rootFiles: [{ fileName: fileToOpen }]
        });
        openedProject = true;
        project = projectService.findProject(fileToOpen);
    }
    if (project != null) {
        let scriptInfo = project.getScriptInfo(file);
        if (scriptInfo) {
            let result = project.emitFile(scriptInfo, (path, data, writeByteOrderMark) => {
                generatedFiles.push(path);
                return host.writeFile(path, data, writeByteOrderMark);
            });
            result.emitSkipped = result.emitSkipped;
            if (args.richResponse) {
                let resultDiagnostics = [];
                let languageService = project.getLanguageService();
                resultDiagnostics.push(...languageService.getSemanticDiagnostics(file));
                resultDiagnostics.push(...languageService.getSyntacticDiagnostics(file));
                resultDiagnostics.push(...result.diagnostics);
                diagnostics.push(...resultDiagnostics.map(el => formatDiagnosticToProtocol(ts, el, true)));
            }
            if (result.emitSkipped) {
                //no emitted -> reset info
                generatedFiles.length = 0;
            }
        }
        else {
            projectService.logger.info(`There is no script info. File ${file}, project ${projectFileName}`);
        }
    }
    else {
        projectService.logger.info(`There is no project. File ${file}, project ${projectFileName}`);
    }
    if (openedProject) {
        //do not close for perf reason
        // projectService.closeExternalProject(projectFileName);
    }
    return {
        response: result,
        responseRequired: true
    };
};
const projectInfoHandler = (_ts, projectService, _request) => {
    let infos = [];
    let configuredProjects = projectService.configuredProjects;
    configuredProjects.forEach((configuredProject) => {
        addProjectInfo(configuredProject, infos);
    });
    for (let inferredProject of projectService.inferredProjects) {
        addProjectInfo(inferredProject, infos);
    }
    for (let externalProject of projectService.externalProjects) {
        addProjectInfo(externalProject, infos);
    }
    return {
        responseRequired: true,
        response: infos
    };
};
function addProjectInfo(project, infos) {
    let name = project.getProjectName();
    let regularFileInfos = project.getFileNames(false).map(el => {
        let info = project.getScriptInfo(el);
        return {
            fileName: el,
            isOpen: !!info && info.isScriptOpen(),
            isExternal: false
        };
    });
    let externalFileInfos = [];
    if (project.getExternalFiles) {
        externalFileInfos = project.getExternalFiles().map(el => {
            let info = project.getScriptInfo(el);
            return {
                fileName: el,
                isOpen: !!info && info.isScriptOpen(),
                isExternal: true
            };
        });
    }
    infos.push({
        projectName: name,
        fileInfos: regularFileInfos.concat(externalFileInfos)
    });
}
/** also update tsLanguageService/typescript/util.ts */
function parseNumbersInVersion(version) {
    let result = [];
    let versions = version.split(/[.-]/);
    for (version of versions) {
        if (!version)
            break;
        let currentNumber = Number(version);
        if (currentNumber == null || isNaN(currentNumber))
            break;
        result = result.concat(currentNumber);
    }
    return result;
}
function diagnosticCategoryName(ts, d, lowerCase = true) {
    const name = ts.DiagnosticCategory[d.category];
    return lowerCase ? name.toLowerCase() : name;
}
function isVersionMoreOrEqual(version, ...expected) {
    for (let i = 0; i < expected.length; i++) {
        let expectedNumber = expected[i];
        let currentNumber = version.length > i ? version[i] : 0;
        if (currentNumber < expectedNumber)
            return false;
        if (currentNumber > expectedNumber)
            return true;
    }
    return version.length >= expected.length;
}
function formatDiagnosticToProtocol(ts, diag, includeFileName) {
    const start = (diag.file && convertToLocation(ts.getLineAndCharacterOfPosition(diag.file, diag.start)));
    const end = (diag.file && convertToLocation(ts.getLineAndCharacterOfPosition(diag.file, diag.start + diag.length)));
    const text = ts.flattenDiagnosticMessageText(diag.messageText, "\n");
    const { code, source } = diag;
    const category = diagnosticCategoryName(ts, diag);
    const common = {
        start,
        end,
        text,
        code,
        category,
        reportsUnnecessary: diag.reportsUnnecessary,
        reportsDeprecated: diag.reportsDeprecated,
        source
    };
    return includeFileName
        ? Object.assign(Object.assign({}, common), { fileName: diag.file && diag.file.fileName }) : common;
}
function convertToLocation(lineAndCharacter) {
    return {
        line: lineAndCharacter.line + 1,
        offset: lineAndCharacter.character + 1
    };
}
module.exports = init;
//# sourceMappingURL=index.js.map