/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.textstructure.structurefinder;

import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.time.format.ResolverStyle;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.grok.Grok;
import org.elasticsearch.grok.GrokBuiltinPatterns;
import org.elasticsearch.xpack.textstructure.structurefinder.TimeoutChecker;

public final class TimestampFormatFinder {
    private static final boolean ECS_COMPATIBILITY = false;
    private static final String PREFACE = "preface";
    private static final String EPILOGUE = "epilogue";
    private static final Logger logger = LogManager.getLogger(TimestampFormatFinder.class);
    private static final String PUNCTUATION_THAT_NEEDS_ESCAPING_IN_REGEX = "\\|()[]{}^$.*?";
    private static final String FRACTIONAL_SECOND_SEPARATORS = ":.,";
    private static final Pattern FRACTIONAL_SECOND_INTERPRETER = Pattern.compile("([:.,])(\\d{3,9})($|[Z+-])");
    private static final char INDETERMINATE_FIELD_PLACEHOLDER = '?';
    private static final Pattern INDETERMINATE_FORMAT_INTERPRETER = Pattern.compile("([^?]*)(\\?{1,2})(?:([^?]*)(\\?{1,2})([^?]*))?");
    private static final Map<String, Tuple<String, String>> VALID_LETTER_GROUPS;
    static final String CUSTOM_TIMESTAMP_GROK_NAME = "CUSTOM_TIMESTAMP";
    static final CandidateTimestampFormat ISO8601_CANDIDATE_FORMAT;
    static final CandidateTimestampFormat UNIX_MS_CANDIDATE_FORMAT;
    static final CandidateTimestampFormat UNIX_CANDIDATE_FORMAT;
    static final CandidateTimestampFormat TAI64N_CANDIDATE_FORMAT;
    static final List<CandidateTimestampFormat> ORDERED_CANDIDATE_FORMATS;
    static final List<CandidateTimestampFormat> ORDERED_CANDIDATE_FORMATS_ECS_V1;
    private final List<String> explanation;
    private final boolean requireFullMatch;
    private final boolean errorOnNoTimestamp;
    private final boolean errorOnMultiplePatterns;
    private final List<CandidateTimestampFormat> orderedCandidateFormats;
    private final TimeoutChecker timeoutChecker;
    private final List<TimestampMatch> matches;
    private List<TimestampFormat> matchedFormats;
    private List<String> cachedJavaTimestampFormats;

    public TimestampFormatFinder(List<String> explanation, boolean requireFullMatch, boolean errorOnNoTimestamp, boolean errorOnMultiplePatterns, TimeoutChecker timeoutChecker, boolean ecsCompatibility) {
        this(explanation, null, requireFullMatch, errorOnNoTimestamp, errorOnMultiplePatterns, timeoutChecker, ecsCompatibility);
    }

    public TimestampFormatFinder(List<String> explanation, boolean requireFullMatch, boolean errorOnNoTimestamp, boolean errorOnMultiplePatterns, TimeoutChecker timeoutChecker) {
        this(explanation, null, requireFullMatch, errorOnNoTimestamp, errorOnMultiplePatterns, timeoutChecker, false);
    }

    public TimestampFormatFinder(List<String> explanation, @Nullable String overrideFormat, boolean requireFullMatch, boolean errorOnNoTimestamp, boolean errorOnMultiplePatterns, TimeoutChecker timeoutChecker, boolean ecsCompatibility) {
        this.explanation = Objects.requireNonNull(explanation);
        this.requireFullMatch = requireFullMatch;
        this.errorOnNoTimestamp = errorOnNoTimestamp;
        this.errorOnMultiplePatterns = errorOnMultiplePatterns;
        this.orderedCandidateFormats = overrideFormat != null ? Collections.singletonList(TimestampFormatFinder.makeCandidateFromOverrideFormat(overrideFormat, timeoutChecker, ecsCompatibility)) : (ecsCompatibility ? ORDERED_CANDIDATE_FORMATS_ECS_V1 : ORDERED_CANDIDATE_FORMATS);
        this.timeoutChecker = Objects.requireNonNull(timeoutChecker);
        this.matches = new ArrayList<TimestampMatch>();
        this.matchedFormats = new ArrayList<TimestampFormat>();
    }

    static Tuple<String, String> overrideFormatToGrokAndRegex(String overrideFormat) {
        if (overrideFormat.indexOf(10) >= 0 || overrideFormat.indexOf(13) >= 0) {
            throw new IllegalArgumentException("Multi-line timestamp formats [" + overrideFormat + "] not supported");
        }
        if (overrideFormat.indexOf(63) >= 0) {
            throw new IllegalArgumentException("Timestamp format [" + overrideFormat + "] not supported because it contains [?]");
        }
        StringBuilder grokPatternBuilder = new StringBuilder();
        StringBuilder regexBuilder = new StringBuilder();
        boolean notQuoted = true;
        char prevChar = '\u0000';
        String prevLetterGroup = null;
        for (int pos = 0; pos < overrideFormat.length(); ++pos) {
            char curChar = overrideFormat.charAt(pos);
            if (curChar == '\'') {
                notQuoted = !notQuoted;
            } else if (notQuoted && Character.isLetter(curChar)) {
                int startPos = pos;
                int endPos = startPos + 1;
                while (endPos < overrideFormat.length() && overrideFormat.charAt(endPos) == curChar) {
                    ++endPos;
                    ++pos;
                }
                String letterGroup = overrideFormat.substring(startPos, endPos);
                Tuple<String, String> grokPatternAndRegexForGroup = VALID_LETTER_GROUPS.get(letterGroup);
                if (grokPatternAndRegexForGroup == null) {
                    if (curChar != 'S' || FRACTIONAL_SECOND_SEPARATORS.indexOf(prevChar) == -1 || !"ss".equals(prevLetterGroup) || endPos - startPos > 9) {
                        String msg = "Letter group [" + letterGroup + "] in [" + overrideFormat + "] is not supported";
                        if (curChar == 'S') {
                            msg = msg + " because it is not preceded by [ss] and a separator from [:.,]";
                        }
                        throw new IllegalArgumentException(msg);
                    }
                    int numCharsToDelete = PUNCTUATION_THAT_NEEDS_ESCAPING_IN_REGEX.indexOf(prevChar) >= 0 ? 2 : 1;
                    grokPatternBuilder.delete(grokPatternBuilder.length() - numCharsToDelete, grokPatternBuilder.length());
                    regexBuilder.append("\\d{").append(endPos - startPos).append('}');
                } else {
                    grokPatternBuilder.append((String)grokPatternAndRegexForGroup.v1());
                    if (regexBuilder.length() == 0) {
                        regexBuilder.append("\\b");
                    }
                    regexBuilder.append((String)grokPatternAndRegexForGroup.v2());
                }
                if (pos + 1 == overrideFormat.length()) {
                    regexBuilder.append("\\b");
                }
                prevLetterGroup = letterGroup;
            } else {
                if (PUNCTUATION_THAT_NEEDS_ESCAPING_IN_REGEX.indexOf(curChar) >= 0) {
                    grokPatternBuilder.append('\\');
                    regexBuilder.append('\\');
                }
                grokPatternBuilder.append(curChar);
                regexBuilder.append(curChar);
            }
            prevChar = curChar;
        }
        if (prevLetterGroup == null) {
            throw new IllegalArgumentException("No time format letter groups in override format [" + overrideFormat + "]");
        }
        return new Tuple((Object)grokPatternBuilder.toString(), (Object)regexBuilder.toString());
    }

    static CandidateTimestampFormat makeCandidateFromOverrideFormat(String overrideFormat, TimeoutChecker timeoutChecker, boolean ecsCompatibility) {
        switch (overrideFormat.toUpperCase(Locale.ROOT)) {
            case "ISO8601": {
                return ISO8601_CANDIDATE_FORMAT;
            }
            case "UNIX_MS": {
                return UNIX_MS_CANDIDATE_FORMAT;
            }
            case "UNIX": {
                return UNIX_CANDIDATE_FORMAT;
            }
            case "TAI64N": {
                return TAI64N_CANDIDATE_FORMAT;
            }
        }
        Tuple<String, String> grokPatternAndRegex = TimestampFormatFinder.overrideFormatToGrokAndRegex(overrideFormat);
        DateTimeFormatter javaTimeFormatter = DateTimeFormatter.ofPattern(overrideFormat, Locale.ROOT);
        String generatedTimestamp = javaTimeFormatter.withZone(ZoneOffset.ofHoursMinutesSeconds(5, 45, 0)).format(Instant.ofEpochMilli(981173106123L).plusNanos(456789L));
        BitSet numberPosBitSet = TimestampFormatFinder.stringToNumberPosBitSet(generatedTimestamp);
        for (CandidateTimestampFormat candidate : ecsCompatibility ? ORDERED_CANDIDATE_FORMATS_ECS_V1 : ORDERED_CANDIDATE_FORMATS) {
            TimestampMatch match = TimestampFormatFinder.checkCandidate(candidate, generatedTimestamp, numberPosBitSet, true, timeoutChecker);
            if (match == null) continue;
            return new CandidateTimestampFormat(example -> {
                try {
                    javaTimeFormatter.parse((CharSequence)example);
                    return Collections.singletonList(overrideFormat);
                }
                catch (DateTimeException e) {
                    return candidate.javaTimestampFormatSupplier.apply((String)example);
                }
            }, candidate.simplePattern.pattern(), candidate.strictGrokPattern, candidate.outputGrokPatternName);
        }
        return new CandidateTimestampFormat(example -> Collections.singletonList(overrideFormat), (String)grokPatternAndRegex.v2(), (String)grokPatternAndRegex.v1(), CUSTOM_TIMESTAMP_GROK_NAME);
    }

    private static TimestampMatch checkCandidate(CandidateTimestampFormat candidate, String text, @Nullable BitSet numberPosBitSet, boolean requireFullMatch, TimeoutChecker timeoutChecker) {
        Tuple<Integer, Integer> boundsForCandidate = TimestampFormatFinder.findBoundsForCandidate(candidate, numberPosBitSet);
        if (requireFullMatch) {
            Map<String, Object> captures;
            if ((Integer)boundsForCandidate.v1() == 0 && (captures = timeoutChecker.grokCaptures(candidate.strictFullMatchGrok, text, "timestamp format determination")) != null) {
                return new TimestampMatch(candidate, "", text, "");
            }
        } else if ((Integer)boundsForCandidate.v1() >= 0) {
            assert ((Integer)boundsForCandidate.v2() > (Integer)boundsForCandidate.v1());
            String matchIn = text.substring((Integer)boundsForCandidate.v1(), Math.min((Integer)boundsForCandidate.v2(), text.length()));
            Map<String, Object> captures = timeoutChecker.grokCaptures(candidate.strictSearchGrok, matchIn, "timestamp format determination");
            if (captures != null) {
                StringBuilder prefaceBuilder = new StringBuilder();
                if ((Integer)boundsForCandidate.v1() > 0) {
                    prefaceBuilder.append(text.subSequence(0, (Integer)boundsForCandidate.v1()));
                }
                prefaceBuilder.append(captures.getOrDefault(PREFACE, ""));
                StringBuilder epilogueBuilder = new StringBuilder();
                epilogueBuilder.append(captures.getOrDefault(EPILOGUE, ""));
                if ((Integer)boundsForCandidate.v2() < text.length()) {
                    epilogueBuilder.append(text.subSequence((Integer)boundsForCandidate.v2(), text.length()));
                }
                return new TimestampMatch(candidate, prefaceBuilder.toString(), text.substring(prefaceBuilder.length(), text.length() - epilogueBuilder.length()), epilogueBuilder.toString());
            }
        } else {
            timeoutChecker.check("timestamp format determination");
        }
        return null;
    }

    public void addSample(String text) {
        BitSet numberPosBitSet = TimestampFormatFinder.stringToNumberPosBitSet(text);
        for (CandidateTimestampFormat candidate : this.orderedCandidateFormats) {
            TimestampMatch match = TimestampFormatFinder.checkCandidate(candidate, text, numberPosBitSet, this.requireFullMatch, this.timeoutChecker);
            if (match == null) continue;
            TimestampFormat newFormat = match.timestampFormat;
            boolean mustAdd = true;
            for (int i = 0; i < this.matchedFormats.size(); ++i) {
                TimestampFormat existingFormat = this.matchedFormats.get(i);
                if (!existingFormat.canMergeWith(newFormat)) continue;
                this.matchedFormats.set(i, existingFormat.mergeWith(newFormat));
                mustAdd = false;
                match = new TimestampMatch(match, this.matchedFormats.get(i));
                break;
            }
            if (mustAdd) {
                if (this.errorOnMultiplePatterns && !this.matchedFormats.isEmpty()) {
                    throw new IllegalArgumentException("Multiple timestamp formats found [" + String.valueOf(this.matchedFormats.get(0)) + "] and [" + String.valueOf(newFormat) + "]");
                }
                this.matchedFormats.add(newFormat);
            }
            this.matches.add(match);
            this.cachedJavaTimestampFormats = null;
            return;
        }
        if (this.errorOnNoTimestamp) {
            throw new IllegalArgumentException("No timestamp found in [" + text + "]");
        }
    }

    public void selectBestMatch() {
        if (this.matchedFormats.size() < 2) {
            return;
        }
        double[] weights = this.calculateMatchWeights();
        this.timeoutChecker.check("timestamp format determination");
        int highestWeightFormatIndex = TimestampFormatFinder.findHighestWeightIndex(weights);
        this.timeoutChecker.check("timestamp format determination");
        this.selectHighestWeightFormat(highestWeightFormatIndex);
    }

    private double[] calculateMatchWeights() {
        int remainingMatches = this.matches.size();
        double[] weights = new double[this.matchedFormats.size()];
        for (TimestampMatch match : this.matches) {
            for (int matchedFormatIndex = 0; matchedFormatIndex < this.matchedFormats.size(); ++matchedFormatIndex) {
                if (this.matchedFormats.get(matchedFormatIndex).canMergeWith(match.timestampFormat)) {
                    int n = matchedFormatIndex;
                    weights[n] = weights[n] + TimestampFormatFinder.weightForMatch(match.preface);
                    break;
                }
                ++matchedFormatIndex;
            }
            --remainingMatches;
            if (!(TimestampFormatFinder.findDifferenceBetweenTwoHighestWeights(weights) > (double)remainingMatches)) continue;
            break;
        }
        return weights;
    }

    private static double weightForMatch(String preface) {
        return Math.pow(1.0 + (double)preface.length() / 15.0, -1.1);
    }

    private static double findDifferenceBetweenTwoHighestWeights(double[] weights) {
        assert (weights.length >= 2);
        double highestWeight = 0.0;
        double secondHighestWeight = 0.0;
        for (double weight : weights) {
            if (weight > highestWeight) {
                secondHighestWeight = highestWeight;
                highestWeight = weight;
                continue;
            }
            if (!(weight > secondHighestWeight)) continue;
            secondHighestWeight = weight;
        }
        return highestWeight - secondHighestWeight;
    }

    private static int findHighestWeightIndex(double[] weights) {
        double highestWeight = Double.NEGATIVE_INFINITY;
        int highestWeightFormatIndex = -1;
        for (int index = 0; index < weights.length; ++index) {
            double weight = weights[index];
            if (!(weight > highestWeight)) continue;
            highestWeight = weight;
            highestWeightFormatIndex = index;
        }
        return highestWeightFormatIndex;
    }

    private void selectHighestWeightFormat(int highestWeightFormatIndex) {
        assert (highestWeightFormatIndex >= 0);
        if (highestWeightFormatIndex == 0) {
            return;
        }
        this.cachedJavaTimestampFormats = null;
        ArrayList<TimestampFormat> newMatchedFormats = new ArrayList<TimestampFormat>(this.matchedFormats);
        newMatchedFormats.set(0, this.matchedFormats.get(highestWeightFormatIndex));
        newMatchedFormats.set(highestWeightFormatIndex, this.matchedFormats.get(0));
        this.matchedFormats = newMatchedFormats;
    }

    public int getNumMatchedFormats() {
        return this.matchedFormats.size();
    }

    public String getGrokPatternName() {
        if (this.matchedFormats.isEmpty()) {
            assert (!this.errorOnNoTimestamp);
            return null;
        }
        return this.matchedFormats.get((int)0).grokPatternName;
    }

    public Map<String, String> getCustomGrokPatternDefinitions() {
        if (this.matchedFormats.isEmpty()) {
            assert (!this.errorOnNoTimestamp);
            return Collections.emptyMap();
        }
        return this.matchedFormats.get((int)0).customGrokPatternDefinitions;
    }

    public List<String> getPrefaces() {
        if (this.matchedFormats.isEmpty()) {
            assert (!this.errorOnNoTimestamp);
            return Collections.emptyList();
        }
        return this.matches.stream().filter(match -> this.matchedFormats.size() < 2 || this.matchedFormats.get(0).canMergeWith(match.timestampFormat)).map(match -> match.preface).collect(Collectors.toList());
    }

    public Pattern getSimplePattern() {
        if (this.matchedFormats.isEmpty()) {
            assert (!this.errorOnNoTimestamp);
            return null;
        }
        return this.matchedFormats.get((int)0).simplePattern;
    }

    public List<String> getRawJavaTimestampFormats() {
        if (this.matchedFormats.isEmpty()) {
            assert (!this.errorOnNoTimestamp);
            return Collections.emptyList();
        }
        return this.matchedFormats.get((int)0).rawJavaTimestampFormats;
    }

    public List<String> getJavaTimestampFormats() {
        if (this.cachedJavaTimestampFormats != null) {
            return this.cachedJavaTimestampFormats;
        }
        return this.determiniseJavaTimestampFormats(this.getRawJavaTimestampFormats(), this.matchedFormats.size() > 1 ? this.matchedFormats.get(0) : null);
    }

    public boolean needNanosecondPrecision() {
        if (this.matchedFormats.isEmpty()) {
            assert (!this.errorOnNoTimestamp);
            return false;
        }
        return this.matches.stream().filter(match -> this.matchedFormats.size() < 2 || this.matchedFormats.get(0).canMergeWith(match.timestampFormat)).anyMatch(match -> match.hasNanosecondPrecision);
    }

    private List<String> determiniseJavaTimestampFormats(List<String> rawJavaTimestampFormats, @Nullable TimestampFormat onlyConsiderFormat) {
        if (rawJavaTimestampFormats.stream().anyMatch(format -> format.indexOf(63) >= 0)) {
            boolean isDayFirst = this.guessIsDayFirst(rawJavaTimestampFormats, onlyConsiderFormat, Locale.getDefault());
            this.cachedJavaTimestampFormats = rawJavaTimestampFormats.stream().map(format -> TimestampFormatFinder.determiniseJavaTimestampFormat(format, isDayFirst)).collect(Collectors.toList());
        } else {
            this.cachedJavaTimestampFormats = rawJavaTimestampFormats;
        }
        return this.cachedJavaTimestampFormats;
    }

    private boolean guessIsDayFirst(List<String> rawJavaTimestampFormats, @Nullable TimestampFormat onlyConsiderFormat, Locale localeForFallback) {
        Boolean isDayFirst = this.guessIsDayFirstFromFormats(rawJavaTimestampFormats);
        if (isDayFirst != null) {
            return isDayFirst;
        }
        isDayFirst = this.guessIsDayFirstFromMatches(onlyConsiderFormat);
        if (isDayFirst != null) {
            return isDayFirst;
        }
        return this.guessIsDayFirstFromLocale(localeForFallback);
    }

    Boolean guessIsDayFirstFromFormats(List<String> rawJavaTimestampFormats) {
        Boolean isDayFirst = null;
        for (String rawJavaTimestampFormat : rawJavaTimestampFormats) {
            Matcher matcher = INDETERMINATE_FORMAT_INTERPRETER.matcher(rawJavaTimestampFormat);
            if (!matcher.matches()) continue;
            String firstNumber = matcher.group(2);
            assert (firstNumber != null);
            String secondNumber = matcher.group(4);
            if (secondNumber == null) {
                return null;
            }
            if (firstNumber.length() == 2 && secondNumber.length() == 1) {
                if (Boolean.FALSE.equals(isDayFirst)) {
                    return null;
                }
                isDayFirst = Boolean.TRUE;
            }
            if (firstNumber.length() != 1 || secondNumber.length() != 2) continue;
            if (Boolean.TRUE.equals(isDayFirst)) {
                return null;
            }
            isDayFirst = Boolean.FALSE;
        }
        if (isDayFirst != null) {
            if (isDayFirst.booleanValue()) {
                this.explanation.add("Guessing day precedes month in timestamps as all detected formats have a two digits in the first number and a single digit in the second number which is what the %{MONTHDAY} and %{MONTHNUM} Grok patterns permit");
            } else {
                this.explanation.add("Guessing month precedes day in timestamps as all detected formats have a single digit in the first number and two digits in the second number which is what the %{MONTHNUM} and %{MONTHDAY} Grok patterns permit");
            }
        }
        return isDayFirst;
    }

    Boolean guessIsDayFirstFromMatches(@Nullable TimestampFormat onlyConsiderFormat) {
        BitSet firstIndeterminateNumbers = new BitSet();
        BitSet secondIndeterminateNumbers = new BitSet();
        for (TimestampMatch match : this.matches) {
            if (onlyConsiderFormat != null && !onlyConsiderFormat.canMergeWith(match.timestampFormat)) continue;
            if (match.firstIndeterminateDateNumber > 0) {
                assert (match.firstIndeterminateDateNumber <= 31);
                if (match.firstIndeterminateDateNumber > 12) {
                    this.explanation.add("Guessing day precedes month in timestamps as one sample had first number [" + match.firstIndeterminateDateNumber + "]");
                    return Boolean.TRUE;
                }
                firstIndeterminateNumbers.set(match.firstIndeterminateDateNumber);
            }
            if (match.secondIndeterminateDateNumber <= 0) continue;
            assert (match.secondIndeterminateDateNumber <= 31);
            if (match.secondIndeterminateDateNumber > 12) {
                this.explanation.add("Guessing month precedes day in timestamps as one sample had second number [" + match.secondIndeterminateDateNumber + "]");
                return Boolean.FALSE;
            }
            secondIndeterminateNumbers.set(match.secondIndeterminateDateNumber);
        }
        int ratioForResult = 3;
        int firstCardinality = firstIndeterminateNumbers.cardinality();
        int secondCardinality = secondIndeterminateNumbers.cardinality();
        if (secondCardinality == 0) {
            return Boolean.FALSE;
        }
        assert (firstCardinality > 0);
        if (firstCardinality >= 3 * secondCardinality) {
            this.explanation.add("Guessing day precedes month in timestamps as there were [" + firstCardinality + "] distinct values of the first number but only [" + secondCardinality + "] for the second");
            return Boolean.TRUE;
        }
        if (secondCardinality >= 3 * firstCardinality) {
            this.explanation.add("Guessing month precedes day in timestamps as there " + (firstCardinality == 1 ? "was" : "were") + " only [" + firstCardinality + "] distinct " + (firstCardinality == 1 ? "value" : "values") + " of the first number but [" + secondCardinality + "] for the second");
            return Boolean.FALSE;
        }
        return null;
    }

    boolean guessIsDayFirstFromLocale(Locale locale) {
        String feb3rd1970 = TimestampFormatFinder.makeShortLocalizedDateTimeFormatterForLocale(locale).format(LocalDate.ofEpochDay(33L));
        if (feb3rd1970.indexOf(51) < feb3rd1970.indexOf(50)) {
            this.explanation.add("Guessing day precedes month in timestamps based on server locale [" + locale.getDisplayName(Locale.ROOT) + "]");
            return true;
        }
        this.explanation.add("Guessing month precedes day in timestamps based on server locale [" + locale.getDisplayName(Locale.ROOT) + "]");
        return false;
    }

    @SuppressForbidden(reason="DateTimeFormatter.ofLocalizedDate() is forbidden because it uses the default locale, but here we are explicitly setting the locale on the formatter in a subsequent call")
    private static DateTimeFormatter makeShortLocalizedDateTimeFormatterForLocale(Locale locale) {
        return DateTimeFormatter.ofLocalizedDate(FormatStyle.SHORT).withLocale(locale).withZone(ZoneOffset.UTC);
    }

    static String determiniseJavaTimestampFormat(String rawJavaTimestampFormat, boolean isDayFirst) {
        Matcher matcher = INDETERMINATE_FORMAT_INTERPRETER.matcher(rawJavaTimestampFormat);
        if (matcher.matches()) {
            StringBuilder builder = new StringBuilder();
            block4: for (int groupNum = 1; groupNum <= matcher.groupCount(); ++groupNum) {
                switch (groupNum) {
                    case 2: {
                        int count;
                        char formatChar = isDayFirst ? (char)'d' : 'M';
                        for (count = matcher.group(groupNum).length(); count > 0; --count) {
                            builder.append(formatChar);
                        }
                        continue block4;
                    }
                    case 4: {
                        int count;
                        char formatChar = isDayFirst ? (char)'M' : 'd';
                        for (count = matcher.group(groupNum).length(); count > 0; --count) {
                            builder.append(formatChar);
                        }
                        continue block4;
                    }
                    default: {
                        builder.append(matcher.group(groupNum));
                    }
                }
            }
            return builder.toString();
        }
        return rawJavaTimestampFormat;
    }

    public List<String> getJodaTimestampFormats() {
        List<String> javaTimestampFormats = this.getJavaTimestampFormats();
        return javaTimestampFormats == null ? null : javaTimestampFormats.stream().map(format -> format.replace("yy", "YY").replace("XXX", "ZZ").replace("XX", "Z")).collect(Collectors.toList());
    }

    public boolean hasTimezoneDependentParsing() {
        if (this.matchedFormats.isEmpty()) {
            assert (!this.errorOnNoTimestamp);
            return false;
        }
        return this.matches.stream().filter(match -> this.matchedFormats.size() < 2 || this.matchedFormats.get(0).canMergeWith(match.timestampFormat)).anyMatch(match -> match.hasTimezoneDependentParsing);
    }

    public Map<String, String> getEsDateMappingTypeWithoutFormat() {
        return Collections.singletonMap("type", this.needNanosecondPrecision() ? "date_nanos" : "date");
    }

    public Map<String, String> getEsDateMappingTypeWithFormat() {
        List<String> javaTimestampFormats = this.getJavaTimestampFormats();
        if (javaTimestampFormats.contains("TAI64N")) {
            return Collections.singletonMap("type", "keyword");
        }
        LinkedHashMap<String, String> mapping = new LinkedHashMap<String, String>();
        mapping.put("type", this.needNanosecondPrecision() ? "date_nanos" : "date");
        String formats = javaTimestampFormats.stream().map(format -> switch (format) {
            case "ISO8601" -> "iso8601";
            case "UNIX_MS" -> "epoch_millis";
            case "UNIX" -> "epoch_second";
            default -> format;
        }).collect(Collectors.joining("||"));
        if (!formats.isEmpty()) {
            mapping.put("format", formats);
        }
        return mapping;
    }

    static Tuple<Integer, Integer> findBoundsForCandidate(CandidateTimestampFormat candidate, BitSet numberPosBitSet) {
        if (numberPosBitSet == null || candidate.quickRuleOutBitSets.isEmpty()) {
            return new Tuple((Object)0, (Object)Integer.MAX_VALUE);
        }
        int minFirstMatchStart = -1;
        int maxLastMatchEnd = -1;
        for (BitSet quickRuleOutBitSet : candidate.quickRuleOutBitSets) {
            int currentMatch = TimestampFormatFinder.findBitPattern(numberPosBitSet, 0, quickRuleOutBitSet);
            if (currentMatch < 0) continue;
            if (minFirstMatchStart == -1 || currentMatch < minFirstMatchStart) {
                minFirstMatchStart = currentMatch;
            }
            do {
                int currentMatchEnd;
                if ((currentMatchEnd = currentMatch + quickRuleOutBitSet.length()) <= maxLastMatchEnd) continue;
                maxLastMatchEnd = currentMatchEnd;
            } while ((currentMatch = TimestampFormatFinder.findBitPattern(numberPosBitSet, currentMatch + 1, quickRuleOutBitSet)) > 0);
        }
        if (minFirstMatchStart == -1) {
            assert (maxLastMatchEnd == -1);
            return new Tuple((Object)-1, (Object)-1);
        }
        int lowerBound = Math.max(0, minFirstMatchStart - candidate.maxCharsBeforeQuickRuleOutMatch);
        int upperBound = Integer.MAX_VALUE - candidate.maxCharsAfterQuickRuleOutMatch - maxLastMatchEnd < 0 ? Integer.MAX_VALUE : maxLastMatchEnd + candidate.maxCharsAfterQuickRuleOutMatch;
        return new Tuple((Object)lowerBound, (Object)upperBound);
    }

    static int findBitPattern(BitSet findIn, int beginIndex, BitSet toFind) {
        int i;
        assert (beginIndex >= 0);
        int toFindLength = toFind.length();
        int findInLength = findIn.length();
        if (toFindLength == 0) {
            return beginIndex;
        }
        if (toFindLength > Math.min(63, findInLength)) {
            assert (toFindLength <= 63) : "Length to find was [" + toFindLength + "] - cannot be greater than 63";
            return -1;
        }
        long state = -2L;
        long[] toFindMask = new long[]{-1L, -1L};
        for (i = 0; i < toFindLength; ++i) {
            int n = toFind.get(i) ? 1 : 0;
            toFindMask[n] = toFindMask[n] & (1L << i ^ 0xFFFFFFFFFFFFFFFFL);
        }
        for (i = beginIndex; i < findInLength; ++i) {
            state |= toFindMask[findIn.get(i) ? 1 : 0];
            if (((state <<= 1) & 1L << toFindLength) != 0L) continue;
            return i - toFindLength + 1;
        }
        return -1;
    }

    static BitSet stringToNumberPosBitSet(String str) {
        BitSet result = new BitSet();
        for (int index = 0; index < str.length(); ++index) {
            if (!Character.isDigit(str.charAt(index))) continue;
            result.set(index);
        }
        return result;
    }

    static {
        HashMap<String, Tuple> validLetterGroups = new HashMap<String, Tuple>();
        validLetterGroups.put("yyyy", new Tuple((Object)"%{YEAR}", (Object)"\\d{4}"));
        validLetterGroups.put("yy", new Tuple((Object)"%{YEAR}", (Object)"\\d{2}"));
        validLetterGroups.put("M", new Tuple((Object)"%{MONTHNUM}", (Object)"\\d{1,2}"));
        validLetterGroups.put("MM", new Tuple((Object)"%{MONTHNUM2}", (Object)"\\d{2}"));
        validLetterGroups.put("MMM", new Tuple((Object)"%{MONTH}", (Object)"[A-Z]\\S{2}"));
        validLetterGroups.put("MMMM", new Tuple((Object)"%{MONTH}", (Object)"[A-Z]\\S{2,8}"));
        validLetterGroups.put("d", new Tuple((Object)"%{MONTHDAY}", (Object)"\\d{1,2}"));
        validLetterGroups.put("dd", new Tuple((Object)"%{MONTHDAY}", (Object)"\\d{2}"));
        validLetterGroups.put("EEE", new Tuple((Object)"%{DAY}", (Object)"[A-Z]\\S{2}"));
        validLetterGroups.put("EEEE", new Tuple((Object)"%{DAY}", (Object)"[A-Z]\\S{2,8}"));
        validLetterGroups.put("H", new Tuple((Object)"%{HOUR}", (Object)"\\d{1,2}"));
        validLetterGroups.put("HH", new Tuple((Object)"%{HOUR}", (Object)"\\d{2}"));
        validLetterGroups.put("h", new Tuple((Object)"%{HOUR}", (Object)"\\d{1,2}"));
        validLetterGroups.put("mm", new Tuple((Object)"%{MINUTE}", (Object)"\\d{2}"));
        validLetterGroups.put("ss", new Tuple((Object)"%{SECOND}", (Object)"\\d{2}"));
        validLetterGroups.put("a", new Tuple((Object)"(?:AM|PM)", (Object)"[AP]M"));
        validLetterGroups.put("XX", new Tuple((Object)"%{ISO8601_TIMEZONE}", (Object)"(?:Z|[+-]\\d{4})"));
        validLetterGroups.put("XXX", new Tuple((Object)"%{ISO8601_TIMEZONE}", (Object)"(?:Z|[+-]\\d{2}:\\d{2})"));
        validLetterGroups.put("zzz", new Tuple((Object)"%{TZ}", (Object)"[A-Z]{3}"));
        VALID_LETTER_GROUPS = Collections.unmodifiableMap(validLetterGroups);
        ISO8601_CANDIDATE_FORMAT = new CandidateTimestampFormat(CandidateTimestampFormat::iso8601FormatFromExample, "\\b\\d{4}-\\d{2}-\\d{2}[T ]\\d{2}:\\d{2}", "\\b%{TIMESTAMP_ISO8601}\\b", "TIMESTAMP_ISO8601", "1111 11 11 11 11", 0, 19);
        UNIX_MS_CANDIDATE_FORMAT = new CandidateTimestampFormat(example -> Collections.singletonList("UNIX_MS"), "\\b\\d{13}\\b", "\\b[12]\\d{12}\\b", "POSINT", "1111111111111", 0, 0);
        UNIX_CANDIDATE_FORMAT = new CandidateTimestampFormat(example -> Collections.singletonList("UNIX"), "\\b\\d{10}\\b", "\\b[12]\\d{9}(?:\\.\\d{3,9})?\\b", "NUMBER", "1111111111", 0, 10);
        TAI64N_CANDIDATE_FORMAT = new CandidateTimestampFormat(example -> Collections.singletonList("TAI64N"), "\\b[0-9A-Fa-f]{24}\\b", "\\b[0-9A-Fa-f]{24}\\b", "BASE16NUM");
        ORDERED_CANDIDATE_FORMATS = Arrays.asList(new CandidateTimestampFormat(example -> CandidateTimestampFormat.iso8601LikeFormatFromExample(example, " ", " "), "\\b\\d{4}-\\d{2}-\\d{2} \\d{2}:\\d{2}:\\d{2}[:.,]\\d{3}", "\\b20\\d{2}-%{MONTHNUM}-%{MONTHDAY} %{HOUR}:?%{MINUTE}:(?:[0-5][0-9]|60)[:.,][0-9]{3,9} (?:Z|[+-]%{HOUR}%{MINUTE})\\b", "TOMCAT_DATESTAMP", "1111 11 11 11 11 11 111", 0, 13), ISO8601_CANDIDATE_FORMAT, new CandidateTimestampFormat(example -> Arrays.asList("EEE MMM dd yy HH:mm:ss zzz", "EEE MMM d yy HH:mm:ss zzz"), "\\b[A-Z]\\S{2} [A-Z]\\S{2} \\d{1,2} \\d{2} \\d{2}:\\d{2}:\\d{2}\\b", "\\b%{DAY} %{MONTH} %{MONTHDAY} %{YEAR} %{HOUR}:%{MINUTE}(?::(?:[0-5][0-9]|60)) %{TZ}\\b", "DATESTAMP_RFC822", Arrays.asList("        11 11 11 11 11", "        1 11 11 11 11"), 0, 5), new CandidateTimestampFormat(example -> CandidateTimestampFormat.adjustTrailingTimezoneFromExample(example, "EEE, dd MMM yyyy HH:mm:ss XX"), "\\b[A-Z]\\S{2}, \\d{1,2} [A-Z]\\S{2} \\d{4} \\d{2}:\\d{2}:\\d{2}\\b", "\\b%{DAY}, %{MONTHDAY} %{MONTH} %{YEAR} %{HOUR}:%{MINUTE}(?::(?:[0-5][0-9]|60)) (?:Z|[+-]%{HOUR}:?%{MINUTE})\\b", "DATESTAMP_RFC2822", Arrays.asList("     11     1111 11 11 11", "     1     1111 11 11 11"), 0, 7), new CandidateTimestampFormat(example -> Arrays.asList("EEE MMM dd HH:mm:ss zzz yyyy", "EEE MMM d HH:mm:ss zzz yyyy"), "\\b[A-Z]\\S{2,8} [A-Z]\\S{2,8} \\d{1,2} \\d{2}:\\d{2}:\\d{2}\\b", "\\b%{DAY} %{MONTH} %{MONTHDAY} %{HOUR}:%{MINUTE}(?::(?:[0-5][0-9]|60)) %{TZ} %{YEAR}\\b", "DATESTAMP_OTHER", Arrays.asList("        11 11 11 11", "        1 11 11 11"), 12, 10), new CandidateTimestampFormat(example -> Collections.singletonList("yyyyMMddHHmmss"), "\\b\\d{14}\\b", "\\b20\\d{2}%{MONTHNUM2}(?:(?:0[1-9])|(?:[12][0-9])|(?:3[01]))(?:2[0123]|[01][0-9])%{MINUTE}(?:[0-5][0-9]|60)\\b", "DATESTAMP_EVENTLOG", "11111111111111", 0, 0), new CandidateTimestampFormat(example -> Collections.singletonList("EEE MMM dd HH:mm:ss yyyy"), "\\b[A-Z]\\S{2} [A-Z]\\S{2} \\d{2} \\d{2}:\\d{2}:\\d{2} \\d{4}\\b", "\\b%{DAY} %{MONTH} %{MONTHDAY} %{HOUR}:%{MINUTE}:(?:[0-5][0-9]|60) %{YEAR}\\b", "HTTPDERROR_DATE", "        11 11 11 11 1111", 0, 0), new CandidateTimestampFormat(example -> CandidateTimestampFormat.expandDayAndAdjustFractionalSecondsFromExample(example, "MMM dd HH:mm:ss"), "\\b[A-Z]\\S{2,8} {1,2}\\d{1,2} \\d{2}:\\d{2}:\\d{2}\\b", "%{MONTH} +%{MONTHDAY} %{HOUR}:%{MINUTE}:(?:[0-5][0-9]|60)(?:[:.,][0-9]{3,9})?\\b", "SYSLOGTIMESTAMP", Arrays.asList("    11 11 11 11", "    1 11 11 11"), 6, 10), new CandidateTimestampFormat(example -> Collections.singletonList("dd/MMM/yyyy:HH:mm:ss XX"), "\\b\\d{2}/[A-Z]\\S{2}/\\d{4}:\\d{2}:\\d{2}:\\d{2} ", "\\b%{MONTHDAY}/%{MONTH}/%{YEAR}:%{HOUR}:%{MINUTE}:(?:[0-5][0-9]|60) [+-]?%{HOUR}%{MINUTE}\\b", "HTTPDATE", "11     1111 11 11 11", 0, 6), new CandidateTimestampFormat(example -> Collections.singletonList("MMM dd, yyyy h:mm:ss a"), "\\b[A-Z]\\S{2} \\d{2}, \\d{4} \\d{1,2}:\\d{2}:\\d{2} [AP]M\\b", "%{MONTH} %{MONTHDAY}, 20\\d{2} %{HOUR}:%{MINUTE}:(?:[0-5][0-9]|60) (?:AM|PM)\\b", "CATALINA_DATESTAMP", Arrays.asList("    11  1111 1 11 11", "    11  1111 11 11 11"), 0, 3), new CandidateTimestampFormat(example -> Arrays.asList("MMM dd yyyy HH:mm:ss", "MMM  d yyyy HH:mm:ss", "MMM d yyyy HH:mm:ss"), "\\b[A-Z]\\S{2} {1,2}\\d{1,2} \\d{4} \\d{2}:\\d{2}:\\d{2}\\b", "%{MONTH} +%{MONTHDAY} %{YEAR} %{HOUR}:%{MINUTE}:(?:[0-5][0-9]|60)\\b", "CISCOTIMESTAMP", Arrays.asList("    11 1111 11 11 11", "    1 1111 11 11 11"), 1, 0), new CandidateTimestampFormat(CandidateTimestampFormat::indeterminateDayMonthFormatFromExample, "\\b\\d{1,2}[/.-]\\d{1,2}[/.-](?:\\d{2}){1,2}[- ]\\d{2}:\\d{2}:\\d{2}\\b", "\\b%{DATESTAMP}\\b", "DATESTAMP", Arrays.asList("11 11 1111 11 11 11", "1 11 1111 11 11 11", "11 1 1111 11 11 11", "11 11 11 11 11 11", "1 11 11 11 11 11", "11 1 11 11 11 11"), 1, 10), new CandidateTimestampFormat(CandidateTimestampFormat::indeterminateDayMonthFormatFromExample, "\\b\\d{1,2}[/.-]\\d{1,2}[/.-](?:\\d{2}){1,2}\\b", "\\b%{DATE}\\b", "DATE", Arrays.asList("11 11 1111", "11 1 1111", "1 11 1111", "11 11 11", "11 1 11", "1 11 11"), 1, 0), UNIX_MS_CANDIDATE_FORMAT, UNIX_CANDIDATE_FORMAT, TAI64N_CANDIDATE_FORMAT, new CandidateTimestampFormat(example -> Collections.singletonList("ISO8601"), "\\b\\d{4}-\\d{2}-\\d{2}\\b", "\\b%{YEAR}-%{MONTHNUM2}-%{MONTHDAY}\\b", CUSTOM_TIMESTAMP_GROK_NAME, "1111 11 11", 0, 0), new CandidateTimestampFormat(example -> Collections.singletonList("MMM d, yyyy @ HH:mm:ss.SSS"), "\\b[A-Z]\\S{2} \\d{1,2}, \\d{4} @ \\d{2}:\\d{2}:\\d{2}\\.\\d{3}\\b", "\\b%{MONTH} %{MONTHDAY}, %{YEAR} @ %{HOUR}:%{MINUTE}:%{SECOND}\\b", CUSTOM_TIMESTAMP_GROK_NAME, Arrays.asList("    11  1111   11 11 11 111", "    1  1111   11 11 11 111"), 0, 0));
        ArrayList<CandidateTimestampFormat> items = new ArrayList<CandidateTimestampFormat>();
        items.add(new CandidateTimestampFormat(example -> Collections.singletonList(CandidateTimestampFormat.adjustFractionalSecondsFromEndOfExample(example, "dd-MMM-yyyy hh:mm:ss")), "\\b\\d{2}-[A-Z]\\S{2}-\\d{4} \\d{2}:\\d{2}:\\d{2}[:.,]\\d{3}", "\\b%{MONTHDAY}-%{MONTH}-%{YEAR} %{HOUR}:%{MINUTE}:%{SECOND}\\b", "CATALINA8_DATESTAMP", "11     1111 11 11 11 111", 0, 0));
        items.add(new CandidateTimestampFormat(example -> Collections.singletonList("MMM dd, yyyy h:mm:ss a"), "\\b[A-Z]\\S{2} \\d{2}, \\d{4} \\d{1,2}:\\d{2}:\\d{2} [AP]M\\b", "\\b%{MONTH} %{MONTHDAY}, %{YEAR} %{HOUR}:%{MINUTE}:%{SECOND} (?:AM|PM)\\b", "CATALINA7_DATESTAMP", Arrays.asList("    11  1111 1 11 11", "    11  1111 11 11 11"), 0, 3));
        items.add(new CandidateTimestampFormat(example -> CandidateTimestampFormat.iso8601LikeFormatFromExample(example, " ", " "), "\\b\\d{4}-\\d{2}-\\d{2} \\d{2}:\\d{2}:\\d{2}[:.,]\\d{3}", "\\b20\\d{2}-%{MONTHNUM}-%{MONTHDAY} %{HOUR}:?%{MINUTE}:(?:[0-5][0-9]|60)[:.,][0-9]{3,9} (?:Z|[+-]%{HOUR}%{MINUTE})\\b", "TOMCATLEGACY_DATESTAMP", "1111 11 11 11 11 11 111", 0, 13));
        items.addAll(ORDERED_CANDIDATE_FORMATS.stream().filter(p -> !"CATALINA_DATESTAMP".equals(p.outputGrokPatternName) && !"TOMCAT_DATESTAMP".equals(p.outputGrokPatternName)).toList());
        ORDERED_CANDIDATE_FORMATS_ECS_V1 = Collections.unmodifiableList(items);
    }

    static final class CandidateTimestampFormat {
        private static final Pattern TRAILING_OFFSET_WITHOUT_COLON_FINDER = Pattern.compile("[+-]\\d{4}$");
        final Function<String, List<String>> javaTimestampFormatSupplier;
        final Pattern simplePattern;
        final String strictGrokPattern;
        final Grok strictSearchGrok;
        final Grok strictFullMatchGrok;
        final String outputGrokPatternName;
        final List<BitSet> quickRuleOutBitSets;
        final int maxCharsBeforeQuickRuleOutMatch;
        final int maxCharsAfterQuickRuleOutMatch;

        CandidateTimestampFormat(Function<String, List<String>> javaTimestampFormatSupplier, String simpleRegex, String strictGrokPattern, String outputGrokPatternName) {
            this(javaTimestampFormatSupplier, simpleRegex, strictGrokPattern, outputGrokPatternName, Collections.emptyList(), Integer.MAX_VALUE, Integer.MAX_VALUE);
        }

        CandidateTimestampFormat(Function<String, List<String>> javaTimestampFormatSupplier, String simpleRegex, String strictGrokPattern, String outputGrokPatternName, String quickRuleOutPattern, int maxCharsBeforeQuickRuleOutMatch, int maxCharsAfterQuickRuleOutMatch) {
            this(javaTimestampFormatSupplier, simpleRegex, strictGrokPattern, outputGrokPatternName, Collections.singletonList(quickRuleOutPattern), maxCharsBeforeQuickRuleOutMatch, maxCharsAfterQuickRuleOutMatch);
        }

        CandidateTimestampFormat(Function<String, List<String>> javaTimestampFormatSupplier, String simpleRegex, String strictGrokPattern, String outputGrokPatternName, List<String> quickRuleOutPatterns, int maxCharsBeforeQuickRuleOutMatch, int maxCharsAfterQuickRuleOutMatch) {
            this.javaTimestampFormatSupplier = Objects.requireNonNull(javaTimestampFormatSupplier);
            this.simplePattern = Pattern.compile(simpleRegex, 8);
            this.strictGrokPattern = Objects.requireNonNull(strictGrokPattern);
            this.strictSearchGrok = new Grok(GrokBuiltinPatterns.legacyPatterns(), "(?m)%{DATA:preface}" + strictGrokPattern + "%{GREEDYDATA:epilogue}", TimeoutChecker.watchdog, arg_0 -> ((Logger)logger).warn(arg_0));
            this.strictFullMatchGrok = new Grok(GrokBuiltinPatterns.legacyPatterns(), "^" + strictGrokPattern + "$", TimeoutChecker.watchdog, arg_0 -> ((Logger)logger).warn(arg_0));
            this.outputGrokPatternName = Objects.requireNonNull(outputGrokPatternName);
            this.quickRuleOutBitSets = quickRuleOutPatterns.stream().map(TimestampFormatFinder::stringToNumberPosBitSet).collect(Collectors.toList());
            assert (maxCharsBeforeQuickRuleOutMatch >= 0);
            this.maxCharsBeforeQuickRuleOutMatch = maxCharsBeforeQuickRuleOutMatch;
            assert (maxCharsAfterQuickRuleOutMatch >= 0);
            this.maxCharsAfterQuickRuleOutMatch = maxCharsAfterQuickRuleOutMatch;
        }

        Map<String, String> customGrokPatternDefinitions() {
            return TimestampFormatFinder.CUSTOM_TIMESTAMP_GROK_NAME.equals(this.outputGrokPatternName) ? Collections.singletonMap(TimestampFormatFinder.CUSTOM_TIMESTAMP_GROK_NAME, this.strictGrokPattern) : Collections.emptyMap();
        }

        static List<String> iso8601FormatFromExample(String example) {
            return example.indexOf(84) >= 0 ? Collections.singletonList("ISO8601") : CandidateTimestampFormat.iso8601LikeFormatFromExample(example, " ", "");
        }

        static List<String> iso8601LikeFormatFromExample(String example, String timeSeparator, String timezoneSeparator) {
            StringBuilder builder = new StringBuilder("yyyy-MM-dd");
            builder.append(timeSeparator).append("HH:mm");
            if (example.length() > builder.length() && example.charAt(builder.length()) == ':') {
                builder.append(":ss");
            }
            if (example.length() > builder.length()) {
                char nextChar = example.charAt(builder.length());
                if (TimestampFormatFinder.FRACTIONAL_SECOND_SEPARATORS.indexOf(nextChar) >= 0) {
                    builder.append(nextChar);
                    for (int pos = builder.length(); pos < example.length() && Character.isDigit(example.charAt(pos)); ++pos) {
                        builder.append('S');
                    }
                }
                if (example.length() > builder.length()) {
                    builder.append(timezoneSeparator).append(example.indexOf(58, builder.length()) > 0 ? "XXX" : "XX");
                }
            } else assert (example.length() == builder.length()) : "Expected [" + example + "] and [" + String.valueOf(builder) + "] to be the same length";
            return Collections.singletonList(builder.toString());
        }

        static List<String> adjustTrailingTimezoneFromExample(String example, String formatWithSecondsAndXX) {
            return Collections.singletonList(TRAILING_OFFSET_WITHOUT_COLON_FINDER.matcher(example).find() ? formatWithSecondsAndXX : formatWithSecondsAndXX + "X");
        }

        private static String adjustFractionalSecondsFromEndOfExample(String example, String formatNoFraction) {
            Matcher matcher = FRACTIONAL_SECOND_INTERPRETER.matcher(example);
            return matcher.find() ? formatNoFraction + matcher.group(1).charAt(0) + "SSSSSSSSS".substring(0, matcher.group(2).length()) : formatNoFraction;
        }

        static List<String> expandDayAndAdjustFractionalSecondsFromExample(String example, String formatWithddAndNoFraction) {
            String formatWithdd = CandidateTimestampFormat.adjustFractionalSecondsFromEndOfExample(example, formatWithddAndNoFraction);
            return Arrays.asList(formatWithdd, formatWithdd.replace(" dd", "  d"), formatWithdd.replace(" dd", " d"));
        }

        static List<String> indeterminateDayMonthFormatFromExample(String example) {
            StringBuilder builder = new StringBuilder();
            int examplePos = 0;
            for (Character patternChar : Arrays.asList(Character.valueOf('?'), Character.valueOf('?'), Character.valueOf('y'), Character.valueOf('H'), Character.valueOf('m'), Character.valueOf('s'))) {
                boolean foundDigit = false;
                while (examplePos < example.length() && Character.isDigit(example.charAt(examplePos))) {
                    foundDigit = true;
                    builder.append(patternChar);
                    ++examplePos;
                }
                if (patternChar.charValue() == 's' || examplePos >= example.length() || !foundDigit) break;
                builder.append(example.charAt(examplePos));
                ++examplePos;
            }
            String format = builder.toString();
            assert (format.contains("yy")) : "Unexpected format [" + format + "] from example [" + example + "]";
            if (examplePos < example.length()) {
                assert (builder.toString().endsWith("ss")) : "Unexpected format [" + format + "] from example [" + example + "]";
                format = CandidateTimestampFormat.adjustFractionalSecondsFromEndOfExample(example, format);
            }
            assert (Character.isLetter(format.charAt(format.length() - 1))) : "Unexpected format [" + format + "] from example [" + example + "]";
            assert (format.length() == example.length()) : "Unexpected format [" + format + "] from example [" + example + "]";
            return Collections.singletonList(format);
        }
    }

    static final class TimestampMatch {
        private static final Pattern NON_PUNCTUATION_PATTERN = Pattern.compile("[^\\\\/|~:;,<>()\\[\\]{}\u00ab\u00bb\t]+");
        private static final Pattern ISO8601_TIMEZONE_PATTERN = Pattern.compile("(Z|[+-]\\d{2}:?\\d{2})$");
        final String preface;
        final TimestampFormat timestampFormat;
        final int firstIndeterminateDateNumber;
        final int secondIndeterminateDateNumber;
        final boolean hasTimezoneDependentParsing;
        final boolean hasNanosecondPrecision;
        final String epilogue;

        TimestampMatch(CandidateTimestampFormat chosenTimestampFormat, String preface, String matchedDate, String epilogue) {
            this.preface = Objects.requireNonNull(preface);
            this.timestampFormat = new TimestampFormat(chosenTimestampFormat.javaTimestampFormatSupplier.apply(matchedDate), chosenTimestampFormat.simplePattern, chosenTimestampFormat.outputGrokPatternName, chosenTimestampFormat.customGrokPatternDefinitions(), preface.isEmpty() ? preface : NON_PUNCTUATION_PATTERN.matcher(preface).replaceAll(""));
            int[] indeterminateDateNumbers = TimestampMatch.parseIndeterminateDateNumbers(matchedDate, this.timestampFormat.rawJavaTimestampFormats);
            this.firstIndeterminateDateNumber = indeterminateDateNumbers[0];
            this.secondIndeterminateDateNumber = indeterminateDateNumbers[1];
            this.hasTimezoneDependentParsing = TimestampMatch.requiresTimezoneDependentParsing(this.timestampFormat.rawJavaTimestampFormats.get(0), matchedDate);
            this.hasNanosecondPrecision = TimestampMatch.matchHasNanosecondPrecision(this.timestampFormat.rawJavaTimestampFormats.get(0), matchedDate);
            this.epilogue = Objects.requireNonNull(epilogue);
        }

        TimestampMatch(TimestampMatch toCopyExceptFormat, TimestampFormat timestampFormat) {
            this.preface = toCopyExceptFormat.preface;
            this.timestampFormat = Objects.requireNonNull(timestampFormat);
            this.firstIndeterminateDateNumber = toCopyExceptFormat.firstIndeterminateDateNumber;
            this.secondIndeterminateDateNumber = toCopyExceptFormat.secondIndeterminateDateNumber;
            this.hasTimezoneDependentParsing = toCopyExceptFormat.hasTimezoneDependentParsing;
            this.hasNanosecondPrecision = toCopyExceptFormat.hasNanosecondPrecision;
            this.epilogue = toCopyExceptFormat.epilogue;
        }

        static boolean requiresTimezoneDependentParsing(String format, String matchedDate) {
            switch (format) {
                case "ISO8601": {
                    assert (matchedDate.length() > 6);
                    return !ISO8601_TIMEZONE_PATTERN.matcher(matchedDate).find(matchedDate.length() - 6);
                }
                case "UNIX_MS": 
                case "UNIX": 
                case "TAI64N": {
                    return false;
                }
            }
            boolean notQuoted = true;
            for (int pos = 0; pos < format.length(); ++pos) {
                char curChar = format.charAt(pos);
                if (curChar == '\'') {
                    notQuoted = !notQuoted;
                    continue;
                }
                if (!notQuoted || curChar != 'X' && curChar != 'z') continue;
                return false;
            }
            return true;
        }

        static boolean matchHasNanosecondPrecision(String format, String matchedDate) {
            switch (format) {
                case "ISO8601": {
                    Matcher matcher = FRACTIONAL_SECOND_INTERPRETER.matcher(matchedDate);
                    return matcher.find() && matcher.group(2).length() > 3;
                }
                case "UNIX_MS": 
                case "UNIX": {
                    return false;
                }
                case "TAI64N": {
                    return true;
                }
            }
            boolean notQuoted = true;
            int consecutiveSs = 0;
            for (int pos = 0; pos < format.length(); ++pos) {
                char curChar = format.charAt(pos);
                if (curChar == '\'') {
                    notQuoted = !notQuoted;
                    consecutiveSs = 0;
                    continue;
                }
                if (!notQuoted) continue;
                if (curChar == 'S') {
                    if (++consecutiveSs <= 3) continue;
                    return true;
                }
                consecutiveSs = 0;
            }
            return false;
        }

        static int[] parseIndeterminateDateNumbers(String matchedDate, List<String> rawJavaTimestampFormats) {
            int[] indeterminateDateNumbers = new int[]{-1, -1};
            for (String rawJavaTimestampFormat : rawJavaTimestampFormats) {
                if (rawJavaTimestampFormat.indexOf(63) < 0) continue;
                try {
                    String javaTimestampFormat = TimestampFormatFinder.determiniseJavaTimestampFormat(rawJavaTimestampFormat, true);
                    DateTimeFormatter javaTimeFormatter = DateTimeFormatter.ofPattern(javaTimestampFormat, Locale.ROOT).withResolverStyle(ResolverStyle.LENIENT);
                    TemporalAccessor accessor = javaTimeFormatter.parse(matchedDate);
                    indeterminateDateNumbers[0] = accessor.get(ChronoField.DAY_OF_MONTH);
                    javaTimestampFormat = TimestampFormatFinder.determiniseJavaTimestampFormat(rawJavaTimestampFormat, false);
                    javaTimeFormatter = DateTimeFormatter.ofPattern(javaTimestampFormat, Locale.ROOT).withResolverStyle(ResolverStyle.LENIENT);
                    accessor = javaTimeFormatter.parse(matchedDate);
                    indeterminateDateNumbers[1] = accessor.get(ChronoField.DAY_OF_MONTH);
                    if (indeterminateDateNumbers[0] <= 0 || indeterminateDateNumbers[1] <= 0) continue;
                    break;
                }
                catch (DateTimeException dateTimeException) {
                }
            }
            return indeterminateDateNumbers;
        }

        public int hashCode() {
            return Objects.hash(this.preface, this.timestampFormat, this.firstIndeterminateDateNumber, this.secondIndeterminateDateNumber, this.hasTimezoneDependentParsing, this.epilogue);
        }

        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other == null || this.getClass() != other.getClass()) {
                return false;
            }
            TimestampMatch that = (TimestampMatch)other;
            return Objects.equals(this.preface, that.preface) && Objects.equals(this.timestampFormat, that.timestampFormat) && this.firstIndeterminateDateNumber == that.firstIndeterminateDateNumber && this.secondIndeterminateDateNumber == that.secondIndeterminateDateNumber && this.hasTimezoneDependentParsing == that.hasTimezoneDependentParsing && Objects.equals(this.epilogue, that.epilogue);
        }

        public String toString() {
            return (String)(this.preface.isEmpty() ? "" : "preface = '" + this.preface + "', ") + String.valueOf(this.timestampFormat) + (String)(this.firstIndeterminateDateNumber > 0 || this.secondIndeterminateDateNumber > 0 ? ", indeterminate date numbers = (" + this.firstIndeterminateDateNumber + "," + this.secondIndeterminateDateNumber + ")" : "") + ", has timezone-dependent parsing = " + this.hasTimezoneDependentParsing + (String)(this.epilogue.isEmpty() ? "" : ", epilogue = '" + this.epilogue + "'");
        }
    }

    static final class TimestampFormat {
        final List<String> rawJavaTimestampFormats;
        final Pattern simplePattern;
        final String grokPatternName;
        final Map<String, String> customGrokPatternDefinitions;
        final String prefacePunctuation;

        TimestampFormat(List<String> rawJavaTimestampFormats, Pattern simplePattern, String grokPatternName, Map<String, String> customGrokPatternDefinitions, String prefacePunctuation) {
            this.rawJavaTimestampFormats = Collections.unmodifiableList(rawJavaTimestampFormats);
            this.simplePattern = Objects.requireNonNull(simplePattern);
            this.grokPatternName = Objects.requireNonNull(grokPatternName);
            this.customGrokPatternDefinitions = Objects.requireNonNull(customGrokPatternDefinitions);
            this.prefacePunctuation = prefacePunctuation;
        }

        boolean canMergeWith(TimestampFormat other) {
            if (this == other) {
                return true;
            }
            return other != null && this.simplePattern.pattern().equals(other.simplePattern.pattern()) && this.grokPatternName.equals(other.grokPatternName) && Objects.equals(this.customGrokPatternDefinitions, other.customGrokPatternDefinitions) && this.prefacePunctuation.equals(other.prefacePunctuation);
        }

        TimestampFormat mergeWith(TimestampFormat other) {
            if (this.canMergeWith(other)) {
                LinkedHashSet<String> mergedJavaTimestampFormats;
                if (!this.rawJavaTimestampFormats.equals(other.rawJavaTimestampFormats) && (mergedJavaTimestampFormats = new LinkedHashSet<String>(this.rawJavaTimestampFormats)).addAll(other.rawJavaTimestampFormats)) {
                    return new TimestampFormat(new ArrayList<String>(mergedJavaTimestampFormats), this.simplePattern, this.grokPatternName, this.customGrokPatternDefinitions, this.prefacePunctuation);
                }
                return this;
            }
            throw new IllegalArgumentException("Cannot merge timestamp format [" + String.valueOf(this) + "] with [" + String.valueOf(other) + "]");
        }

        public int hashCode() {
            return Objects.hash(this.rawJavaTimestampFormats, this.simplePattern.pattern(), this.grokPatternName, this.customGrokPatternDefinitions, this.prefacePunctuation);
        }

        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other == null || this.getClass() != other.getClass()) {
                return false;
            }
            TimestampFormat that = (TimestampFormat)other;
            return Objects.equals(this.rawJavaTimestampFormats, that.rawJavaTimestampFormats) && Objects.equals(this.simplePattern.pattern(), that.simplePattern.pattern()) && Objects.equals(this.grokPatternName, that.grokPatternName) && Objects.equals(this.customGrokPatternDefinitions, that.customGrokPatternDefinitions) && Objects.equals(this.prefacePunctuation, that.prefacePunctuation);
        }

        public String toString() {
            return "Java timestamp formats = " + this.rawJavaTimestampFormats.stream().collect(Collectors.joining("', '", "[ '", "' ]")) + ", simple pattern = '" + this.simplePattern.pattern() + "', grok pattern = '" + this.grokPatternName + "'" + (String)(this.customGrokPatternDefinitions.isEmpty() ? "" : ", custom grok pattern definitions = " + String.valueOf(this.customGrokPatternDefinitions)) + ", preface punctuation = '" + this.prefacePunctuation + "'";
        }
    }
}

