/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.transport;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.transport.Compression;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportSettings;

public final class ConnectionProfile {
    private final List<ConnectionTypeHandle> handles;
    private final int numConnections;
    private final TimeValue connectTimeout;
    private final TimeValue handshakeTimeout;
    private final TimeValue pingInterval;
    private final Compression.Enabled compressionEnabled;
    private final Compression.Scheme compressionScheme;
    private final String transportProfile;

    public static ConnectionProfile resolveConnectionProfile(@Nullable ConnectionProfile profile, ConnectionProfile fallbackProfile) {
        Objects.requireNonNull(fallbackProfile);
        if (profile == null) {
            return fallbackProfile;
        }
        if (profile.getConnectTimeout() != null && profile.getHandshakeTimeout() != null && profile.getPingInterval() != null && profile.getCompressionEnabled() != null && profile.getCompressionScheme() != null) {
            return profile;
        }
        Builder builder = new Builder(profile);
        if (profile.getConnectTimeout() == null) {
            builder.setConnectTimeout(fallbackProfile.getConnectTimeout());
        }
        if (profile.getHandshakeTimeout() == null) {
            builder.setHandshakeTimeout(fallbackProfile.getHandshakeTimeout());
        }
        if (profile.getPingInterval() == null) {
            builder.setPingInterval(fallbackProfile.getPingInterval());
        }
        if (profile.getCompressionEnabled() == null) {
            builder.setCompressionEnabled(fallbackProfile.getCompressionEnabled());
        }
        if (profile.getCompressionScheme() == null) {
            builder.setCompressionScheme(fallbackProfile.getCompressionScheme());
        }
        return builder.build();
    }

    public static ConnectionProfile buildDefaultConnectionProfile(Settings settings) {
        int connectionsPerNodeRecovery = TransportSettings.CONNECTIONS_PER_NODE_RECOVERY.get(settings);
        int connectionsPerNodeBulk = TransportSettings.CONNECTIONS_PER_NODE_BULK.get(settings);
        int connectionsPerNodeReg = TransportSettings.CONNECTIONS_PER_NODE_REG.get(settings);
        int connectionsPerNodeState = TransportSettings.CONNECTIONS_PER_NODE_STATE.get(settings);
        int connectionsPerNodePing = TransportSettings.CONNECTIONS_PER_NODE_PING.get(settings);
        Builder builder = new Builder();
        builder.setConnectTimeout(TransportSettings.CONNECT_TIMEOUT.get(settings));
        builder.setHandshakeTimeout(TransportSettings.CONNECT_TIMEOUT.get(settings));
        builder.setPingInterval(TransportSettings.PING_SCHEDULE.get(settings));
        builder.setCompressionEnabled(TransportSettings.TRANSPORT_COMPRESS.get(settings));
        builder.setCompressionScheme(TransportSettings.TRANSPORT_COMPRESSION_SCHEME.get(settings));
        builder.addConnections(connectionsPerNodeBulk, TransportRequestOptions.Type.BULK);
        builder.addConnections(connectionsPerNodePing, TransportRequestOptions.Type.PING);
        builder.addConnections(DiscoveryNode.isMasterNode(settings) ? connectionsPerNodeState : 0, TransportRequestOptions.Type.STATE);
        builder.addConnections(DiscoveryNode.canContainData(settings) ? connectionsPerNodeRecovery : 0, TransportRequestOptions.Type.RECOVERY);
        builder.addConnections(connectionsPerNodeReg, TransportRequestOptions.Type.REG);
        return builder.build();
    }

    public static ConnectionProfile buildSingleChannelProfile(TransportRequestOptions.Type channelType, @Nullable TimeValue connectTimeout, @Nullable TimeValue handshakeTimeout, @Nullable TimeValue pingInterval, @Nullable Compression.Enabled compressionEnabled, @Nullable Compression.Scheme compressionScheme) {
        Builder builder = new Builder();
        builder.addConnections(1, channelType);
        EnumSet<TransportRequestOptions.Type> otherTypes = EnumSet.allOf(TransportRequestOptions.Type.class);
        otherTypes.remove((Object)channelType);
        builder.addConnections(0, otherTypes.toArray(new TransportRequestOptions.Type[0]));
        if (connectTimeout != null) {
            builder.setConnectTimeout(connectTimeout);
        }
        if (handshakeTimeout != null) {
            builder.setHandshakeTimeout(handshakeTimeout);
        }
        if (pingInterval != null) {
            builder.setPingInterval(pingInterval);
        }
        if (compressionEnabled != null) {
            builder.setCompressionEnabled(compressionEnabled);
        }
        if (compressionScheme != null) {
            builder.setCompressionScheme(compressionScheme);
        }
        return builder.build();
    }

    private ConnectionProfile(List<ConnectionTypeHandle> handles, int numConnections, TimeValue connectTimeout, TimeValue handshakeTimeout, TimeValue pingInterval, Compression.Enabled compressionEnabled, Compression.Scheme compressionScheme, String transportProfile) {
        this.handles = handles;
        this.numConnections = numConnections;
        this.connectTimeout = connectTimeout;
        this.handshakeTimeout = handshakeTimeout;
        this.pingInterval = pingInterval;
        this.compressionEnabled = compressionEnabled;
        this.compressionScheme = compressionScheme;
        this.transportProfile = Objects.requireNonNull(transportProfile, "transport profile name must not be null");
    }

    public TimeValue getConnectTimeout() {
        return this.connectTimeout;
    }

    public TimeValue getHandshakeTimeout() {
        return this.handshakeTimeout;
    }

    public TimeValue getPingInterval() {
        return this.pingInterval;
    }

    public Compression.Enabled getCompressionEnabled() {
        return this.compressionEnabled;
    }

    public Compression.Scheme getCompressionScheme() {
        return this.compressionScheme;
    }

    public int getNumConnections() {
        return this.numConnections;
    }

    public String getTransportProfile() {
        return this.transportProfile;
    }

    public int getNumConnectionsPerType(TransportRequestOptions.Type type) {
        for (ConnectionTypeHandle handle : this.handles) {
            if (!handle.getTypes().contains((Object)type)) continue;
            return handle.length;
        }
        throw new AssertionError((Object)("no handle found for type: " + String.valueOf((Object)type)));
    }

    List<ConnectionTypeHandle> getHandles() {
        return Collections.unmodifiableList(this.handles);
    }

    public static class Builder {
        private final List<ConnectionTypeHandle> handles = new ArrayList<ConnectionTypeHandle>();
        private final Set<TransportRequestOptions.Type> addedTypes = EnumSet.noneOf(TransportRequestOptions.Type.class);
        private int numConnections = 0;
        private TimeValue connectTimeout;
        private TimeValue handshakeTimeout;
        private Compression.Enabled compressionEnabled;
        private Compression.Scheme compressionScheme;
        private TimeValue pingInterval;
        private String transportProfile = "default";

        public Builder() {
        }

        public Builder(ConnectionProfile source) {
            this.handles.addAll(source.getHandles());
            this.numConnections = source.getNumConnections();
            this.handles.forEach(th -> this.addedTypes.addAll(th.types));
            this.connectTimeout = source.getConnectTimeout();
            this.handshakeTimeout = source.getHandshakeTimeout();
            this.compressionEnabled = source.getCompressionEnabled();
            this.compressionScheme = source.getCompressionScheme();
            this.pingInterval = source.getPingInterval();
            this.transportProfile = source.getTransportProfile();
        }

        public Builder setConnectTimeout(TimeValue connectTimeout) {
            if (connectTimeout.millis() < 0L) {
                throw new IllegalArgumentException("connectTimeout must be non-negative but was: " + String.valueOf(connectTimeout));
            }
            this.connectTimeout = connectTimeout;
            return this;
        }

        public Builder setHandshakeTimeout(TimeValue handshakeTimeout) {
            if (handshakeTimeout.millis() < 0L) {
                throw new IllegalArgumentException("handshakeTimeout must be non-negative but was: " + String.valueOf(handshakeTimeout));
            }
            this.handshakeTimeout = handshakeTimeout;
            return this;
        }

        public Builder setPingInterval(TimeValue pingInterval) {
            this.pingInterval = pingInterval;
            return this;
        }

        public Builder setCompressionEnabled(Compression.Enabled compressionEnabled) {
            this.compressionEnabled = compressionEnabled;
            return this;
        }

        public Builder setCompressionScheme(Compression.Scheme compressionScheme) {
            this.compressionScheme = compressionScheme;
            return this;
        }

        public Builder addConnections(int numConnections, TransportRequestOptions.Type ... types) {
            if (types == null || types.length == 0) {
                throw new IllegalArgumentException("types must not be null");
            }
            for (TransportRequestOptions.Type type : types) {
                if (!this.addedTypes.contains((Object)type)) continue;
                throw new IllegalArgumentException("type [" + String.valueOf((Object)type) + "] is already registered");
            }
            this.addedTypes.addAll(Arrays.asList(types));
            this.handles.add(new ConnectionTypeHandle(this.numConnections, numConnections, EnumSet.copyOf(Arrays.asList(types))));
            this.numConnections += numConnections;
            return this;
        }

        public Builder setTransportProfile(String transportProfile) {
            this.transportProfile = transportProfile;
            return this;
        }

        public ConnectionProfile build() {
            EnumSet<TransportRequestOptions.Type> types = EnumSet.allOf(TransportRequestOptions.Type.class);
            types.removeAll(this.addedTypes);
            if (!types.isEmpty()) {
                throw new IllegalStateException("not all types are added for this connection profile - missing types: " + String.valueOf(types));
            }
            return new ConnectionProfile(Collections.unmodifiableList(this.handles), this.numConnections, this.connectTimeout, this.handshakeTimeout, this.pingInterval, this.compressionEnabled, this.compressionScheme, this.transportProfile);
        }
    }

    static final class ConnectionTypeHandle {
        public final int length;
        public final int offset;
        private final Set<TransportRequestOptions.Type> types;
        private final AtomicInteger counter = new AtomicInteger();

        private ConnectionTypeHandle(int offset, int length, Set<TransportRequestOptions.Type> types) {
            this.length = length;
            this.offset = offset;
            this.types = types;
        }

        <T> T getChannel(List<T> channels) {
            if (this.length == 0) {
                throw new IllegalStateException("can't select channel size is 0 for types: " + String.valueOf(this.types));
            }
            assert (channels.size() >= this.offset + this.length) : "illegal size: " + channels.size() + " expected >= " + (this.offset + this.length);
            return channels.get(this.offset + Math.floorMod(this.counter.incrementAndGet(), this.length));
        }

        Set<TransportRequestOptions.Type> getTypes() {
            return this.types;
        }
    }
}

