package timestamp

import (
	"bytes"
	"crypto"
	"crypto/rsa"
	_ "crypto/sha1"
	"crypto/sha256"
	_ "crypto/sha256"
	_ "crypto/sha512"
	"crypto/x509"
	"encoding/asn1"
	"encoding/pem"
	"fmt"
	"math/big"
	"strings"
	"testing"
	"time"

	"github.com/digitorus/pkcs7"
)

// Random data, to create with OpenSSL:
//
//	$ openssl rand -base64 32 > data.txt
//
// Contents of our random date in tests below "RhT49MYCgJzWssTF+LXtFZbLD4pe94q4uezLWoTLzyM="
var hashedMessage = []byte{0xf4, 0x87, 0xd8, 0x81, 0x64, 0xbd, 0xe2, 0x23, 0xb7, 0xfd, 0x3c, 0x71, 0xb3, 0xe9, 0xc6, 0x24, 0xc8, 0xa0, 0xcb, 0x55, 0x9c, 0x85, 0x3b, 0x49, 0x55, 0x8e, 0x7e, 0x5e, 0xf4, 0xce, 0x55, 0xf0}

// Time-Stamp request with nonce, to create with OpenSSL:
//
//	$ openssl ts -query -data data.txt -cert -sha256 -out reqnonoce.tsq
var reqNonce = []byte{0x30, 0x43, 0x2, 0x1, 0x1, 0x30, 0x31, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x4, 0x20, 0xf4, 0x87, 0xd8, 0x81, 0x64, 0xbd, 0xe2, 0x23, 0xb7, 0xfd, 0x3c, 0x71, 0xb3, 0xe9, 0xc6, 0x24, 0xc8, 0xa0, 0xcb, 0x55, 0x9c, 0x85, 0x3b, 0x49, 0x55, 0x8e, 0x7e, 0x5e, 0xf4, 0xce, 0x55, 0xf0, 0x2, 0x8, 0x9, 0x2e, 0xf1, 0x9f, 0xfb, 0x5d, 0x2a, 0xe8, 0x1, 0x1, 0xff}

var respNonce = []byte{0x30, 0x82, 0xe, 0x35, 0x30, 0x3, 0x2, 0x1, 0x0, 0x30, 0x82, 0xe, 0x2c, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x7, 0x2, 0xa0, 0x82, 0xe, 0x1d, 0x30, 0x82, 0xe, 0x19, 0x2, 0x1, 0x3, 0x31, 0xf, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x30, 0x81, 0xed, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1, 0x4, 0xa0, 0x81, 0xdd, 0x4, 0x81, 0xda, 0x30, 0x81, 0xd7, 0x2, 0x1, 0x1, 0x6, 0x9, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xa0, 0x32, 0x2, 0x1, 0x30, 0x31, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x4, 0x20, 0xf4, 0x87, 0xd8, 0x81, 0x64, 0xbd, 0xe2, 0x23, 0xb7, 0xfd, 0x3c, 0x71, 0xb3, 0xe9, 0xc6, 0x24, 0xc8, 0xa0, 0xcb, 0x55, 0x9c, 0x85, 0x3b, 0x49, 0x55, 0x8e, 0x7e, 0x5e, 0xf4, 0xce, 0x55, 0xf0, 0x2, 0x14, 0x5, 0xee, 0xb0, 0xc5, 0x78, 0x43, 0x5b, 0xa7, 0x17, 0x90, 0x39, 0x4f, 0xeb, 0xb8, 0x21, 0x67, 0xb3, 0x64, 0x4c, 0xcc, 0x18, 0xf, 0x32, 0x30, 0x31, 0x37, 0x30, 0x34, 0x31, 0x39, 0x30, 0x36, 0x32, 0x39, 0x35, 0x33, 0x5a, 0x30, 0x3, 0x2, 0x1, 0x1, 0x2, 0x8, 0x9, 0x2e, 0xf1, 0x9f, 0xfb, 0x5d, 0x2a, 0xe8, 0xa0, 0x5e, 0xa4, 0x5c, 0x30, 0x5a, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x53, 0x47, 0x31, 0x1f, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x16, 0x47, 0x4d, 0x4f, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x74, 0x65, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x2a, 0x30, 0x28, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x21, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x53, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x43, 0x44, 0x53, 0x20, 0x2d, 0x20, 0x47, 0x32, 0xa0, 0x82, 0xa, 0x47, 0x30, 0x82, 0x5, 0x47, 0x30, 0x82, 0x4, 0x2f, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x12, 0x11, 0x21, 0xd2, 0xd1, 0x82, 0xa, 0xf0, 0xa5, 0xc5, 0x48, 0x14, 0x9a, 0x32, 0x74, 0xdc, 0xc3, 0x43, 0x57, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x36, 0x30, 0x35, 0x32, 0x34, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0xd, 0x32, 0x32, 0x30, 0x35, 0x32, 0x34, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x5a, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x53, 0x47, 0x31, 0x1f, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x16, 0x47, 0x4d, 0x4f, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x74, 0x65, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x2a, 0x30, 0x28, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x21, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x53, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x43, 0x44, 0x53, 0x20, 0x2d, 0x20, 0x47, 0x32, 0x30, 0x82, 0x1, 0x22, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x3, 0x82, 0x1, 0xf, 0x0, 0x30, 0x82, 0x1, 0xa, 0x2, 0x82, 0x1, 0x1, 0x0, 0xd7, 0xbc, 0xf9, 0x4f, 0xea, 0xab, 0x40, 0xdf, 0x15, 0x5c, 0xd4, 0xe8, 0xa5, 0x8e, 0x7a, 0x8, 0x2e, 0xc6, 0x15, 0xea, 0xa, 0xc, 0x69, 0x94, 0xae, 0x9, 0xc2, 0xe2, 0x86, 0xc, 0xf4, 0xed, 0x18, 0x60, 0x5b, 0x9f, 0x28, 0x9b, 0x6c, 0xdb, 0xf3, 0xa7, 0x55, 0x2b, 0xd7, 0x5, 0x5e, 0x81, 0xd0, 0xed, 0xe6, 0xf4, 0xe2, 0xb8, 0x28, 0x12, 0x46, 0xa7, 0x27, 0xfd, 0x8a, 0x89, 0x65, 0x5e, 0x1a, 0x4b, 0x58, 0x67, 0xbc, 0x10, 0xa9, 0x2b, 0x4c, 0x5e, 0xb0, 0x85, 0xdf, 0x7f, 0x59, 0x6c, 0x9e, 0x8c, 0xe1, 0xb9, 0x5f, 0x8f, 0x70, 0x4f, 0xf9, 0x58, 0x0, 0x70, 0x78, 0x2a, 0xcb, 0xd5, 0x36, 0x15, 0xb1, 0xe8, 0xc0, 0x62, 0xa5, 0x7a, 0xe2, 0x6b, 0xa7, 0x90, 0xa5, 0x1, 0x95, 0x6c, 0xa6, 0x8c, 0xea, 0x2a, 0x8a, 0xe5, 0x2a, 0x3e, 0xd3, 0x42, 0xbc, 0x8e, 0x5a, 0xb1, 0x18, 0x18, 0x71, 0x4c, 0x84, 0x7c, 0xaf, 0xf5, 0xb, 0x49, 0x1c, 0x16, 0xf6, 0x50, 0xf9, 0xaf, 0xb0, 0x20, 0x4f, 0x40, 0x49, 0x35, 0xe5, 0x5, 0x9b, 0xaf, 0x6, 0x6f, 0xca, 0x89, 0xb8, 0xe8, 0xae, 0x92, 0x18, 0xb1, 0xb2, 0x56, 0xf1, 0xad, 0x70, 0x94, 0x96, 0x47, 0xea, 0xeb, 0x4f, 0x26, 0x1, 0x5a, 0xd2, 0x65, 0xeb, 0x25, 0xd0, 0x19, 0xb, 0x22, 0x3c, 0xb0, 0x2e, 0x7b, 0x88, 0x92, 0x9, 0x1a, 0xb3, 0xf9, 0x80, 0xa2, 0x7e, 0x90, 0x78, 0xca, 0x4a, 0x20, 0x8b, 0xc4, 0xd, 0xe9, 0x5f, 0x66, 0xb, 0x5c, 0xac, 0xb2, 0xc4, 0x9c, 0x1b, 0xf7, 0x95, 0x24, 0xdc, 0x13, 0x83, 0x8c, 0x94, 0xc8, 0x57, 0xd8, 0x4f, 0x33, 0xf2, 0xda, 0x5d, 0x73, 0x70, 0x4f, 0x3a, 0xc9, 0xa2, 0x5f, 0x2a, 0x7c, 0x71, 0xc9, 0xc1, 0x4a, 0x5d, 0xe1, 0x98, 0x34, 0x6b, 0xd3, 0xc1, 0x74, 0x15, 0x33, 0x89, 0x93, 0x2, 0x3, 0x1, 0x0, 0x1, 0xa3, 0x82, 0x1, 0xed, 0x30, 0x82, 0x1, 0xe9, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x7, 0x80, 0x30, 0x81, 0xdd, 0x6, 0x3, 0x55, 0x1d, 0x20, 0x4, 0x81, 0xd5, 0x30, 0x81, 0xd2, 0x30, 0x81, 0xcf, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x2f, 0x1, 0x2, 0x1, 0x30, 0x81, 0xc1, 0x30, 0x81, 0x8a, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x2, 0x30, 0x7e, 0xc, 0x7c, 0x54, 0x68, 0x69, 0x73, 0x20, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x68, 0x61, 0x73, 0x20, 0x62, 0x65, 0x65, 0x6e, 0x20, 0x69, 0x73, 0x73, 0x75, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x63, 0x63, 0x6f, 0x72, 0x64, 0x61, 0x6e, 0x63, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x20, 0x43, 0x50, 0x53, 0x20, 0x6c, 0x6f, 0x63, 0x61, 0x74, 0x65, 0x64, 0x20, 0x61, 0x74, 0x20, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79, 0x2f, 0x30, 0x32, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x1, 0x16, 0x26, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79, 0x2f, 0x30, 0x9, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x4, 0x2, 0x30, 0x0, 0x30, 0x16, 0x6, 0x3, 0x55, 0x1d, 0x25, 0x1, 0x1, 0xff, 0x4, 0xc, 0x30, 0xa, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x3, 0x8, 0x30, 0x40, 0x6, 0x3, 0x55, 0x1d, 0x1f, 0x4, 0x39, 0x30, 0x37, 0x30, 0x35, 0xa0, 0x33, 0xa0, 0x31, 0x86, 0x2f, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x73, 0x2f, 0x67, 0x73, 0x70, 0x72, 0x6d, 0x73, 0x68, 0x61, 0x32, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x52, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x1, 0x1, 0x4, 0x46, 0x30, 0x44, 0x30, 0x42, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x30, 0x2, 0x86, 0x36, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x73, 0x65, 0x63, 0x75, 0x72, 0x65, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x63, 0x61, 0x63, 0x65, 0x72, 0x74, 0x2f, 0x67, 0x73, 0x70, 0x72, 0x6d, 0x73, 0x68, 0x61, 0x32, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x64, 0x65, 0x72, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0x48, 0x78, 0xeb, 0x96, 0x27, 0x1e, 0x4e, 0xbd, 0xe4, 0x5b, 0x2a, 0x65, 0x11, 0xbb, 0x92, 0x1c, 0xc6, 0x1, 0x96, 0x38, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x59, 0xd8, 0x24, 0xc2, 0xcf, 0x6b, 0x6, 0x42, 0xd4, 0x95, 0x76, 0xb5, 0x29, 0x5c, 0xf5, 0xd8, 0x41, 0x2b, 0x24, 0x5f, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x3, 0x82, 0x1, 0x1, 0x0, 0x1b, 0xfa, 0xb9, 0xab, 0x45, 0xab, 0x43, 0x92, 0xf3, 0x67, 0xb2, 0x25, 0xb7, 0x20, 0x94, 0xb2, 0xee, 0x9a, 0xa5, 0xba, 0x78, 0xa4, 0x46, 0xbe, 0x8c, 0x9e, 0x9e, 0xfd, 0xb7, 0xb7, 0x5c, 0x28, 0x65, 0x13, 0xc7, 0x12, 0x38, 0xbf, 0xf6, 0x8c, 0xee, 0xba, 0x21, 0x62, 0x30, 0xa6, 0xbe, 0xfe, 0xb8, 0x16, 0x1f, 0x86, 0x0, 0x1e, 0xa2, 0x65, 0x2c, 0x80, 0xb, 0x15, 0xe2, 0x16, 0x72, 0x5, 0xad, 0x61, 0x9f, 0xd8, 0xca, 0x1d, 0x12, 0xa8, 0x41, 0x5a, 0xc5, 0x3b, 0x3, 0xb9, 0x65, 0xd4, 0xa3, 0x8, 0xcc, 0x4, 0xc0, 0xa8, 0x3, 0x35, 0xe1, 0xfe, 0xfd, 0x3a, 0x98, 0x13, 0x3d, 0x8d, 0xb5, 0x81, 0x7b, 0x36, 0x38, 0xc7, 0xc5, 0x7, 0x29, 0x6d, 0x4, 0x6b, 0x6e, 0x9a, 0xdf, 0x1c, 0x6c, 0x16, 0x73, 0xed, 0x52, 0xfd, 0xf6, 0x24, 0xdf, 0x1, 0x28, 0xae, 0x3, 0xc, 0x66, 0xc5, 0x54, 0x4d, 0xf8, 0x9f, 0xdb, 0x0, 0xa6, 0x94, 0xc4, 0xc7, 0x20, 0x36, 0x49, 0x35, 0x23, 0xbe, 0x1f, 0xbb, 0xa7, 0x17, 0x28, 0xba, 0x1a, 0x27, 0xdf, 0x5f, 0xf4, 0x8e, 0xac, 0x6b, 0xf4, 0x7, 0xa7, 0x8b, 0xf7, 0x1e, 0x90, 0xd7, 0xcc, 0xd, 0xf1, 0x48, 0xec, 0xd5, 0x23, 0xed, 0xcb, 0x5f, 0x11, 0xbe, 0x7, 0x64, 0x67, 0x40, 0x7e, 0xf1, 0x9c, 0x6f, 0xc0, 0xd6, 0xd2, 0xd6, 0x82, 0xb7, 0x39, 0xa1, 0xd3, 0xf7, 0x14, 0xf1, 0x38, 0xdb, 0x86, 0x0, 0xcb, 0x43, 0xdf, 0x3, 0x42, 0x9, 0xd7, 0xaf, 0xed, 0x6, 0xf1, 0xc6, 0x47, 0xe7, 0xbc, 0xae, 0x29, 0xdb, 0x43, 0x7, 0xe2, 0xf5, 0xd1, 0xb3, 0xb0, 0x47, 0xbb, 0x47, 0x6b, 0x3, 0x8f, 0x49, 0x6, 0xc6, 0x8b, 0x92, 0xf5, 0x56, 0x7b, 0xdc, 0xb8, 0x21, 0x5c, 0x52, 0x2, 0x51, 0x1c, 0x7b, 0x9, 0x58, 0x43, 0x59, 0xa9, 0x37, 0x30, 0x82, 0x4, 0xf8, 0x30, 0x82, 0x3, 0xe0, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x10, 0x35, 0xfb, 0xe4, 0xfa, 0xdf, 0xe4, 0xb0, 0x92, 0x27, 0x6c, 0x31, 0x9b, 0x99, 0xf8, 0xce, 0xb3, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x30, 0x69, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x55, 0x53, 0x31, 0x23, 0x30, 0x21, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x1a, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x73, 0x20, 0x49, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65, 0x64, 0x31, 0x1d, 0x30, 0x1b, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0x14, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x73, 0x31, 0x16, 0x30, 0x14, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0xd, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x31, 0x30, 0x35, 0x32, 0x35, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0xd, 0x32, 0x32, 0x30, 0x35, 0x32, 0x34, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x30, 0x82, 0x1, 0x22, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x3, 0x82, 0x1, 0xf, 0x0, 0x30, 0x82, 0x1, 0xa, 0x2, 0x82, 0x1, 0x1, 0x0, 0xc2, 0xc6, 0xb, 0x84, 0x55, 0x5a, 0x96, 0x36, 0x1d, 0x95, 0xe7, 0x88, 0xed, 0xd0, 0x2d, 0x23, 0xb1, 0x2a, 0x37, 0xc4, 0xb, 0x35, 0x7f, 0xc7, 0x4, 0x69, 0x36, 0x90, 0x53, 0xd3, 0x56, 0x5a, 0x13, 0x8, 0x1f, 0xc0, 0xc6, 0x46, 0x38, 0x81, 0x72, 0x76, 0xa7, 0xff, 0x9e, 0x5d, 0xc5, 0x83, 0x52, 0x8a, 0x11, 0x7, 0x5a, 0x8b, 0xb5, 0xb3, 0x1f, 0x37, 0xf4, 0x2f, 0xad, 0x6d, 0x45, 0xef, 0x0, 0x35, 0xad, 0x30, 0xf, 0x3b, 0xda, 0x5c, 0xe2, 0xe6, 0x9f, 0xa9, 0xce, 0xc6, 0x52, 0xc8, 0xf7, 0x56, 0x8b, 0xe1, 0xe1, 0x97, 0x39, 0x66, 0x80, 0x2f, 0x3d, 0x57, 0x2e, 0x6b, 0x89, 0x2d, 0x15, 0x8f, 0x3d, 0x67, 0xe5, 0x51, 0xac, 0x6c, 0xd1, 0x9f, 0xb6, 0xb1, 0x17, 0x84, 0xdb, 0x6c, 0xa0, 0xb2, 0xf4, 0x35, 0x83, 0x36, 0xe6, 0x77, 0x1f, 0x1, 0x67, 0xde, 0x74, 0x37, 0xb9, 0xba, 0x36, 0x40, 0xe5, 0x49, 0x13, 0x4e, 0x33, 0x1a, 0x8c, 0xb2, 0x1, 0x80, 0x8, 0xc0, 0xbc, 0x72, 0xfb, 0x3e, 0x6f, 0x6d, 0x42, 0xa8, 0xe6, 0x64, 0x69, 0xce, 0xbf, 0x15, 0xed, 0xd4, 0x92, 0x89, 0x2d, 0xc0, 0x6e, 0x1b, 0xb, 0x6e, 0x1d, 0xf5, 0xf9, 0x6, 0x9b, 0x8, 0xf2, 0xbf, 0x87, 0xb9, 0xf9, 0xe9, 0xb4, 0x68, 0x55, 0xda, 0x92, 0x24, 0xcc, 0xe5, 0x61, 0x91, 0xee, 0xc6, 0x9e, 0xfc, 0x72, 0x99, 0x72, 0x9, 0x32, 0xf4, 0x3a, 0x63, 0x3d, 0x85, 0xcd, 0x5c, 0x70, 0x57, 0x86, 0x2, 0x4d, 0xa6, 0xf8, 0x63, 0x84, 0xbb, 0x0, 0xc3, 0xe9, 0x7d, 0xda, 0x28, 0xe5, 0xd5, 0x1b, 0x13, 0xf2, 0x96, 0xea, 0x8, 0x91, 0xda, 0x6b, 0xd7, 0x3, 0x40, 0xce, 0x2a, 0x88, 0xfc, 0x9b, 0x9b, 0xf3, 0x2e, 0xfa, 0xab, 0x74, 0x6e, 0x38, 0x55, 0x18, 0x78, 0x9f, 0xf8, 0xa6, 0x64, 0xd2, 0x6d, 0x2, 0x3, 0x1, 0x0, 0x1, 0xa3, 0x82, 0x1, 0x91, 0x30, 0x82, 0x1, 0x8d, 0x30, 0x12, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x8, 0x30, 0x6, 0x1, 0x1, 0xff, 0x2, 0x1, 0x1, 0x30, 0x81, 0xe4, 0x6, 0x3, 0x55, 0x1d, 0x20, 0x4, 0x81, 0xdc, 0x30, 0x81, 0xd9, 0x30, 0x81, 0xd6, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x2f, 0x1, 0x2, 0x1, 0x30, 0x81, 0xc8, 0x30, 0x36, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x1, 0x16, 0x2a, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6d, 0x69, 0x73, 0x63, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x63, 0x64, 0x73, 0x5f, 0x63, 0x70, 0x2e, 0x68, 0x74, 0x6d, 0x6c, 0x30, 0x81, 0x8d, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x2, 0x30, 0x81, 0x80, 0x1a, 0x7e, 0x54, 0x68, 0x65, 0x20, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x68, 0x61, 0x73, 0x20, 0x62, 0x65, 0x65, 0x6e, 0x20, 0x69, 0x73, 0x73, 0x75, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x63, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x50, 0x6f, 0x6c, 0x69, 0x63, 0x79, 0x20, 0x66, 0x6f, 0x75, 0x6e, 0x64, 0x20, 0x61, 0x74, 0x20, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6d, 0x69, 0x73, 0x63, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x63, 0x64, 0x73, 0x5f, 0x63, 0x70, 0x2e, 0x68, 0x74, 0x6d, 0x6c, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x1d, 0x1f, 0x4, 0x26, 0x30, 0x24, 0x30, 0x22, 0xa0, 0x20, 0xa0, 0x1e, 0x86, 0x1c, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x63, 0x64, 0x73, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x14, 0x6, 0x3, 0x55, 0x1d, 0x25, 0x4, 0xd, 0x30, 0xb, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x2f, 0x1, 0x1, 0x5, 0x30, 0xb, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x4, 0x4, 0x3, 0x2, 0x1, 0x6, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0x59, 0xd8, 0x24, 0xc2, 0xcf, 0x6b, 0x6, 0x42, 0xd4, 0x95, 0x76, 0xb5, 0x29, 0x5c, 0xf5, 0xd8, 0x41, 0x2b, 0x24, 0x5f, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x82, 0xb7, 0x38, 0x4a, 0x93, 0xaa, 0x9b, 0x10, 0xef, 0x80, 0xbb, 0xd9, 0x54, 0xe2, 0xf1, 0xf, 0xfb, 0x80, 0x9c, 0xde, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x3, 0x82, 0x1, 0x1, 0x0, 0x3, 0x80, 0xd0, 0xb, 0x7e, 0x3e, 0x47, 0x44, 0x8a, 0xc5, 0x38, 0xc5, 0x7e, 0x36, 0x5e, 0xcd, 0x60, 0x82, 0x65, 0x62, 0xf3, 0x86, 0xcc, 0x23, 0xbe, 0x47, 0xd7, 0x75, 0x4, 0xc9, 0xdc, 0x5b, 0x3f, 0x12, 0xd3, 0xdb, 0x26, 0x75, 0x59, 0x1d, 0xb6, 0xd4, 0xa, 0x58, 0x3f, 0x5a, 0xd7, 0x28, 0x14, 0xbf, 0x56, 0x86, 0xbd, 0xba, 0xc2, 0x64, 0x20, 0xc8, 0x6e, 0x7d, 0xa9, 0xe8, 0xde, 0xbd, 0x9f, 0x6, 0xff, 0x18, 0xea, 0x57, 0xa1, 0xd7, 0x80, 0x29, 0x87, 0xe8, 0xd0, 0x7b, 0x57, 0x80, 0x45, 0x68, 0x4f, 0x9, 0xa4, 0x83, 0x4c, 0x2d, 0x73, 0x2a, 0x60, 0xee, 0x55, 0x21, 0x9, 0x2c, 0x95, 0x3f, 0xc3, 0x71, 0xe1, 0xb8, 0xad, 0x9c, 0xab, 0x94, 0x37, 0xec, 0x71, 0x4e, 0x66, 0x4d, 0xe8, 0x20, 0x49, 0x4a, 0xea, 0xc7, 0xd7, 0x3c, 0x96, 0xeb, 0xf, 0xc5, 0xf9, 0xd4, 0x7d, 0x5c, 0x13, 0x81, 0x3a, 0xdf, 0xc3, 0x18, 0xc0, 0x47, 0xae, 0x84, 0xff, 0x12, 0xfd, 0xb8, 0x4, 0xc9, 0x9a, 0x5b, 0x33, 0x83, 0x6e, 0xec, 0x11, 0x2c, 0x35, 0x3, 0x71, 0xac, 0xa5, 0xd1, 0xd3, 0x52, 0xd9, 0x9a, 0x70, 0xcf, 0xfe, 0x9d, 0xc0, 0x7e, 0x3, 0x2d, 0x86, 0xe7, 0x7d, 0x66, 0x10, 0x2c, 0x45, 0x86, 0x37, 0x88, 0x9c, 0xe3, 0xb5, 0x39, 0x9f, 0x1c, 0xb2, 0x67, 0x96, 0x43, 0x3a, 0xc2, 0x2c, 0xcd, 0xfa, 0x3d, 0x9a, 0x4, 0x32, 0x5c, 0xb1, 0x3d, 0xf3, 0x2, 0xb4, 0x31, 0x1, 0x4a, 0xdf, 0x92, 0x5b, 0x7d, 0x5f, 0xb6, 0xd0, 0x7, 0xc, 0x93, 0xae, 0xe5, 0xa9, 0xe2, 0x19, 0xdb, 0xc, 0xa3, 0xf3, 0xc8, 0xb1, 0x14, 0xd, 0x61, 0xb9, 0x3, 0x9e, 0xd3, 0xf4, 0x7, 0x17, 0x2b, 0xd9, 0x4c, 0x70, 0x5e, 0x48, 0x6f, 0xc2, 0xc5, 0x3a, 0x9, 0xb0, 0xed, 0x27, 0x5a, 0x11, 0xcb, 0x31, 0x82, 0x2, 0xc6, 0x30, 0x82, 0x2, 0xc2, 0x2, 0x1, 0x1, 0x30, 0x81, 0x88, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x2, 0x12, 0x11, 0x21, 0xd2, 0xd1, 0x82, 0xa, 0xf0, 0xa5, 0xc5, 0x48, 0x14, 0x9a, 0x32, 0x74, 0xdc, 0xc3, 0x43, 0x57, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0xa0, 0x82, 0x1, 0xe, 0x30, 0x1a, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x3, 0x31, 0xd, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1, 0x4, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x4, 0x31, 0x22, 0x4, 0x20, 0x98, 0x7a, 0x55, 0x9c, 0x19, 0x83, 0x73, 0xd0, 0xcb, 0xfc, 0xe1, 0x4, 0x40, 0xf, 0x78, 0x77, 0xe, 0xbe, 0x86, 0xe5, 0xe0, 0x67, 0xa, 0x2b, 0xc8, 0x3c, 0x68, 0x75, 0xef, 0x7, 0x6d, 0x99, 0x30, 0x81, 0xbe, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x2, 0xc, 0x31, 0x81, 0xae, 0x30, 0x81, 0xab, 0x30, 0x81, 0xa8, 0x30, 0x81, 0xa5, 0x4, 0x14, 0xba, 0x64, 0xa3, 0x69, 0x93, 0xb8, 0xb6, 0x4, 0xf7, 0xfe, 0x65, 0xd3, 0x7d, 0x8f, 0x8e, 0x68, 0x96, 0xfc, 0x96, 0xb, 0x30, 0x81, 0x8c, 0x30, 0x76, 0xa4, 0x74, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x2, 0x12, 0x11, 0x21, 0xd2, 0xd1, 0x82, 0xa, 0xf0, 0xa5, 0xc5, 0x48, 0x14, 0x9a, 0x32, 0x74, 0xdc, 0xc3, 0x43, 0x57, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x4, 0x82, 0x1, 0x0, 0x92, 0x89, 0xaa, 0x92, 0xf0, 0x29, 0x62, 0x1c, 0xef, 0x19, 0x23, 0xb9, 0x46, 0xc4, 0x7d, 0xe1, 0x89, 0x4e, 0xa3, 0x72, 0xa9, 0xe1, 0xf6, 0xc9, 0x54, 0x8b, 0xc5, 0xab, 0x5a, 0xbb, 0x2c, 0xe, 0x41, 0x63, 0x3, 0x8f, 0xc3, 0xae, 0xf4, 0x46, 0x83, 0x72, 0x96, 0xa0, 0xba, 0xf0, 0xbf, 0x2f, 0xed, 0x47, 0xad, 0xd7, 0x14, 0x6f, 0x57, 0x7d, 0xd3, 0x12, 0xd1, 0xe7, 0x9b, 0xac, 0x48, 0x72, 0xc4, 0x55, 0x3f, 0x55, 0xda, 0xe, 0x91, 0x1d, 0x5b, 0xb3, 0x14, 0x30, 0x2c, 0x29, 0xd3, 0x41, 0x72, 0x17, 0x5e, 0x37, 0x1b, 0xec, 0xfe, 0x6b, 0x4a, 0x98, 0x6b, 0x63, 0x7d, 0x1a, 0x7f, 0x3b, 0x84, 0xc1, 0x55, 0x50, 0x26, 0xa7, 0x1e, 0x5d, 0x9b, 0xea, 0x17, 0x5b, 0xfd, 0x35, 0x41, 0xa1, 0x6a, 0x35, 0xc6, 0xdd, 0x11, 0xc5, 0x4e, 0xae, 0x3c, 0x85, 0x5d, 0x10, 0xb8, 0xe8, 0x8d, 0xf, 0x3d, 0x3c, 0xe8, 0xef, 0x2c, 0xff, 0xac, 0x44, 0x38, 0x85, 0xcc, 0xee, 0xb6, 0xe8, 0xf7, 0xae, 0xaa, 0x97, 0xbd, 0x68, 0x4a, 0xbe, 0x3e, 0xcf, 0xe, 0x71, 0xcd, 0xf3, 0x3e, 0xef, 0xea, 0x20, 0xd5, 0x3c, 0x40, 0xcd, 0x5a, 0x88, 0x7c, 0x16, 0xd4, 0xd8, 0x8, 0xf4, 0x35, 0x6b, 0xca, 0xa8, 0xce, 0x2d, 0x53, 0x99, 0x87, 0xa3, 0x31, 0xd, 0x16, 0xa5, 0x43, 0x1c, 0xff, 0x17, 0x82, 0xd4, 0x39, 0x7, 0xd6, 0xa6, 0xad, 0x2b, 0x1a, 0x48, 0x32, 0x70, 0x5d, 0x8e, 0xfe, 0xe7, 0x54, 0x8b, 0x15, 0xa5, 0x76, 0x2e, 0xa0, 0x2a, 0x9c, 0x4, 0x2e, 0x3e, 0x0, 0x9f, 0xbc, 0x7d, 0xce, 0x71, 0x94, 0x52, 0x5c, 0x6d, 0x78, 0xc5, 0xd2, 0xbc, 0x2e, 0x78, 0x29, 0xb5, 0x95, 0x2a, 0x34, 0x9a, 0x55, 0x60, 0x9a, 0x18, 0x1b, 0xc1, 0x4d, 0x47, 0xc1, 0xa1, 0xe9, 0x16, 0x3b, 0xee, 0x6b, 0xd2}

// Time-Stamp request without nonce, to create with OpenSSL:
//
//	$ openssl ts -query -data data.txt -cert -sha256 -no_nonce -out reqnonoce.tsq
var reqNoNonce = []byte{0x30, 0x39, 0x2, 0x1, 0x1, 0x30, 0x31, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x4, 0x20, 0xf4, 0x87, 0xd8, 0x81, 0x64, 0xbd, 0xe2, 0x23, 0xb7, 0xfd, 0x3c, 0x71, 0xb3, 0xe9, 0xc6, 0x24, 0xc8, 0xa0, 0xcb, 0x55, 0x9c, 0x85, 0x3b, 0x49, 0x55, 0x8e, 0x7e, 0x5e, 0xf4, 0xce, 0x55, 0xf0, 0x1, 0x1, 0xff}

var respNoNonce = []byte{0x30, 0x82, 0xe, 0x2b, 0x30, 0x3, 0x2, 0x1, 0x0, 0x30, 0x82, 0xe, 0x22, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x7, 0x2, 0xa0, 0x82, 0xe, 0x13, 0x30, 0x82, 0xe, 0xf, 0x2, 0x1, 0x3, 0x31, 0xf, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x30, 0x81, 0xe3, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1, 0x4, 0xa0, 0x81, 0xd3, 0x4, 0x81, 0xd0, 0x30, 0x81, 0xcd, 0x2, 0x1, 0x1, 0x6, 0x9, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xa0, 0x32, 0x2, 0x1, 0x30, 0x31, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x4, 0x20, 0xf4, 0x87, 0xd8, 0x81, 0x64, 0xbd, 0xe2, 0x23, 0xb7, 0xfd, 0x3c, 0x71, 0xb3, 0xe9, 0xc6, 0x24, 0xc8, 0xa0, 0xcb, 0x55, 0x9c, 0x85, 0x3b, 0x49, 0x55, 0x8e, 0x7e, 0x5e, 0xf4, 0xce, 0x55, 0xf0, 0x2, 0x14, 0x25, 0x92, 0xb9, 0xa, 0x43, 0x4b, 0x2c, 0x7, 0x99, 0xa6, 0xb0, 0xbc, 0xc4, 0x25, 0x7, 0x9e, 0xa5, 0x6e, 0xe5, 0xf6, 0x18, 0xf, 0x32, 0x30, 0x31, 0x37, 0x30, 0x34, 0x31, 0x39, 0x30, 0x36, 0x32, 0x38, 0x31, 0x33, 0x5a, 0x30, 0x3, 0x2, 0x1, 0x1, 0xa0, 0x5e, 0xa4, 0x5c, 0x30, 0x5a, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x53, 0x47, 0x31, 0x1f, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x16, 0x47, 0x4d, 0x4f, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x74, 0x65, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x2a, 0x30, 0x28, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x21, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x53, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x43, 0x44, 0x53, 0x20, 0x2d, 0x20, 0x47, 0x32, 0xa0, 0x82, 0xa, 0x47, 0x30, 0x82, 0x5, 0x47, 0x30, 0x82, 0x4, 0x2f, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x12, 0x11, 0x21, 0xd2, 0xd1, 0x82, 0xa, 0xf0, 0xa5, 0xc5, 0x48, 0x14, 0x9a, 0x32, 0x74, 0xdc, 0xc3, 0x43, 0x57, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x36, 0x30, 0x35, 0x32, 0x34, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0xd, 0x32, 0x32, 0x30, 0x35, 0x32, 0x34, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x5a, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x53, 0x47, 0x31, 0x1f, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x16, 0x47, 0x4d, 0x4f, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x74, 0x65, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x2a, 0x30, 0x28, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x21, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x53, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x43, 0x44, 0x53, 0x20, 0x2d, 0x20, 0x47, 0x32, 0x30, 0x82, 0x1, 0x22, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x3, 0x82, 0x1, 0xf, 0x0, 0x30, 0x82, 0x1, 0xa, 0x2, 0x82, 0x1, 0x1, 0x0, 0xd7, 0xbc, 0xf9, 0x4f, 0xea, 0xab, 0x40, 0xdf, 0x15, 0x5c, 0xd4, 0xe8, 0xa5, 0x8e, 0x7a, 0x8, 0x2e, 0xc6, 0x15, 0xea, 0xa, 0xc, 0x69, 0x94, 0xae, 0x9, 0xc2, 0xe2, 0x86, 0xc, 0xf4, 0xed, 0x18, 0x60, 0x5b, 0x9f, 0x28, 0x9b, 0x6c, 0xdb, 0xf3, 0xa7, 0x55, 0x2b, 0xd7, 0x5, 0x5e, 0x81, 0xd0, 0xed, 0xe6, 0xf4, 0xe2, 0xb8, 0x28, 0x12, 0x46, 0xa7, 0x27, 0xfd, 0x8a, 0x89, 0x65, 0x5e, 0x1a, 0x4b, 0x58, 0x67, 0xbc, 0x10, 0xa9, 0x2b, 0x4c, 0x5e, 0xb0, 0x85, 0xdf, 0x7f, 0x59, 0x6c, 0x9e, 0x8c, 0xe1, 0xb9, 0x5f, 0x8f, 0x70, 0x4f, 0xf9, 0x58, 0x0, 0x70, 0x78, 0x2a, 0xcb, 0xd5, 0x36, 0x15, 0xb1, 0xe8, 0xc0, 0x62, 0xa5, 0x7a, 0xe2, 0x6b, 0xa7, 0x90, 0xa5, 0x1, 0x95, 0x6c, 0xa6, 0x8c, 0xea, 0x2a, 0x8a, 0xe5, 0x2a, 0x3e, 0xd3, 0x42, 0xbc, 0x8e, 0x5a, 0xb1, 0x18, 0x18, 0x71, 0x4c, 0x84, 0x7c, 0xaf, 0xf5, 0xb, 0x49, 0x1c, 0x16, 0xf6, 0x50, 0xf9, 0xaf, 0xb0, 0x20, 0x4f, 0x40, 0x49, 0x35, 0xe5, 0x5, 0x9b, 0xaf, 0x6, 0x6f, 0xca, 0x89, 0xb8, 0xe8, 0xae, 0x92, 0x18, 0xb1, 0xb2, 0x56, 0xf1, 0xad, 0x70, 0x94, 0x96, 0x47, 0xea, 0xeb, 0x4f, 0x26, 0x1, 0x5a, 0xd2, 0x65, 0xeb, 0x25, 0xd0, 0x19, 0xb, 0x22, 0x3c, 0xb0, 0x2e, 0x7b, 0x88, 0x92, 0x9, 0x1a, 0xb3, 0xf9, 0x80, 0xa2, 0x7e, 0x90, 0x78, 0xca, 0x4a, 0x20, 0x8b, 0xc4, 0xd, 0xe9, 0x5f, 0x66, 0xb, 0x5c, 0xac, 0xb2, 0xc4, 0x9c, 0x1b, 0xf7, 0x95, 0x24, 0xdc, 0x13, 0x83, 0x8c, 0x94, 0xc8, 0x57, 0xd8, 0x4f, 0x33, 0xf2, 0xda, 0x5d, 0x73, 0x70, 0x4f, 0x3a, 0xc9, 0xa2, 0x5f, 0x2a, 0x7c, 0x71, 0xc9, 0xc1, 0x4a, 0x5d, 0xe1, 0x98, 0x34, 0x6b, 0xd3, 0xc1, 0x74, 0x15, 0x33, 0x89, 0x93, 0x2, 0x3, 0x1, 0x0, 0x1, 0xa3, 0x82, 0x1, 0xed, 0x30, 0x82, 0x1, 0xe9, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x7, 0x80, 0x30, 0x81, 0xdd, 0x6, 0x3, 0x55, 0x1d, 0x20, 0x4, 0x81, 0xd5, 0x30, 0x81, 0xd2, 0x30, 0x81, 0xcf, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x2f, 0x1, 0x2, 0x1, 0x30, 0x81, 0xc1, 0x30, 0x81, 0x8a, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x2, 0x30, 0x7e, 0xc, 0x7c, 0x54, 0x68, 0x69, 0x73, 0x20, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x68, 0x61, 0x73, 0x20, 0x62, 0x65, 0x65, 0x6e, 0x20, 0x69, 0x73, 0x73, 0x75, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x63, 0x63, 0x6f, 0x72, 0x64, 0x61, 0x6e, 0x63, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x20, 0x43, 0x50, 0x53, 0x20, 0x6c, 0x6f, 0x63, 0x61, 0x74, 0x65, 0x64, 0x20, 0x61, 0x74, 0x20, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79, 0x2f, 0x30, 0x32, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x1, 0x16, 0x26, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79, 0x2f, 0x30, 0x9, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x4, 0x2, 0x30, 0x0, 0x30, 0x16, 0x6, 0x3, 0x55, 0x1d, 0x25, 0x1, 0x1, 0xff, 0x4, 0xc, 0x30, 0xa, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x3, 0x8, 0x30, 0x40, 0x6, 0x3, 0x55, 0x1d, 0x1f, 0x4, 0x39, 0x30, 0x37, 0x30, 0x35, 0xa0, 0x33, 0xa0, 0x31, 0x86, 0x2f, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x73, 0x2f, 0x67, 0x73, 0x70, 0x72, 0x6d, 0x73, 0x68, 0x61, 0x32, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x52, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x1, 0x1, 0x4, 0x46, 0x30, 0x44, 0x30, 0x42, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x30, 0x2, 0x86, 0x36, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x73, 0x65, 0x63, 0x75, 0x72, 0x65, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x63, 0x61, 0x63, 0x65, 0x72, 0x74, 0x2f, 0x67, 0x73, 0x70, 0x72, 0x6d, 0x73, 0x68, 0x61, 0x32, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x64, 0x65, 0x72, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0x48, 0x78, 0xeb, 0x96, 0x27, 0x1e, 0x4e, 0xbd, 0xe4, 0x5b, 0x2a, 0x65, 0x11, 0xbb, 0x92, 0x1c, 0xc6, 0x1, 0x96, 0x38, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x59, 0xd8, 0x24, 0xc2, 0xcf, 0x6b, 0x6, 0x42, 0xd4, 0x95, 0x76, 0xb5, 0x29, 0x5c, 0xf5, 0xd8, 0x41, 0x2b, 0x24, 0x5f, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x3, 0x82, 0x1, 0x1, 0x0, 0x1b, 0xfa, 0xb9, 0xab, 0x45, 0xab, 0x43, 0x92, 0xf3, 0x67, 0xb2, 0x25, 0xb7, 0x20, 0x94, 0xb2, 0xee, 0x9a, 0xa5, 0xba, 0x78, 0xa4, 0x46, 0xbe, 0x8c, 0x9e, 0x9e, 0xfd, 0xb7, 0xb7, 0x5c, 0x28, 0x65, 0x13, 0xc7, 0x12, 0x38, 0xbf, 0xf6, 0x8c, 0xee, 0xba, 0x21, 0x62, 0x30, 0xa6, 0xbe, 0xfe, 0xb8, 0x16, 0x1f, 0x86, 0x0, 0x1e, 0xa2, 0x65, 0x2c, 0x80, 0xb, 0x15, 0xe2, 0x16, 0x72, 0x5, 0xad, 0x61, 0x9f, 0xd8, 0xca, 0x1d, 0x12, 0xa8, 0x41, 0x5a, 0xc5, 0x3b, 0x3, 0xb9, 0x65, 0xd4, 0xa3, 0x8, 0xcc, 0x4, 0xc0, 0xa8, 0x3, 0x35, 0xe1, 0xfe, 0xfd, 0x3a, 0x98, 0x13, 0x3d, 0x8d, 0xb5, 0x81, 0x7b, 0x36, 0x38, 0xc7, 0xc5, 0x7, 0x29, 0x6d, 0x4, 0x6b, 0x6e, 0x9a, 0xdf, 0x1c, 0x6c, 0x16, 0x73, 0xed, 0x52, 0xfd, 0xf6, 0x24, 0xdf, 0x1, 0x28, 0xae, 0x3, 0xc, 0x66, 0xc5, 0x54, 0x4d, 0xf8, 0x9f, 0xdb, 0x0, 0xa6, 0x94, 0xc4, 0xc7, 0x20, 0x36, 0x49, 0x35, 0x23, 0xbe, 0x1f, 0xbb, 0xa7, 0x17, 0x28, 0xba, 0x1a, 0x27, 0xdf, 0x5f, 0xf4, 0x8e, 0xac, 0x6b, 0xf4, 0x7, 0xa7, 0x8b, 0xf7, 0x1e, 0x90, 0xd7, 0xcc, 0xd, 0xf1, 0x48, 0xec, 0xd5, 0x23, 0xed, 0xcb, 0x5f, 0x11, 0xbe, 0x7, 0x64, 0x67, 0x40, 0x7e, 0xf1, 0x9c, 0x6f, 0xc0, 0xd6, 0xd2, 0xd6, 0x82, 0xb7, 0x39, 0xa1, 0xd3, 0xf7, 0x14, 0xf1, 0x38, 0xdb, 0x86, 0x0, 0xcb, 0x43, 0xdf, 0x3, 0x42, 0x9, 0xd7, 0xaf, 0xed, 0x6, 0xf1, 0xc6, 0x47, 0xe7, 0xbc, 0xae, 0x29, 0xdb, 0x43, 0x7, 0xe2, 0xf5, 0xd1, 0xb3, 0xb0, 0x47, 0xbb, 0x47, 0x6b, 0x3, 0x8f, 0x49, 0x6, 0xc6, 0x8b, 0x92, 0xf5, 0x56, 0x7b, 0xdc, 0xb8, 0x21, 0x5c, 0x52, 0x2, 0x51, 0x1c, 0x7b, 0x9, 0x58, 0x43, 0x59, 0xa9, 0x37, 0x30, 0x82, 0x4, 0xf8, 0x30, 0x82, 0x3, 0xe0, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x10, 0x35, 0xfb, 0xe4, 0xfa, 0xdf, 0xe4, 0xb0, 0x92, 0x27, 0x6c, 0x31, 0x9b, 0x99, 0xf8, 0xce, 0xb3, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x30, 0x69, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x55, 0x53, 0x31, 0x23, 0x30, 0x21, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x1a, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x73, 0x20, 0x49, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65, 0x64, 0x31, 0x1d, 0x30, 0x1b, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0x14, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x73, 0x31, 0x16, 0x30, 0x14, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0xd, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x31, 0x30, 0x35, 0x32, 0x35, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0xd, 0x32, 0x32, 0x30, 0x35, 0x32, 0x34, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x30, 0x82, 0x1, 0x22, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x3, 0x82, 0x1, 0xf, 0x0, 0x30, 0x82, 0x1, 0xa, 0x2, 0x82, 0x1, 0x1, 0x0, 0xc2, 0xc6, 0xb, 0x84, 0x55, 0x5a, 0x96, 0x36, 0x1d, 0x95, 0xe7, 0x88, 0xed, 0xd0, 0x2d, 0x23, 0xb1, 0x2a, 0x37, 0xc4, 0xb, 0x35, 0x7f, 0xc7, 0x4, 0x69, 0x36, 0x90, 0x53, 0xd3, 0x56, 0x5a, 0x13, 0x8, 0x1f, 0xc0, 0xc6, 0x46, 0x38, 0x81, 0x72, 0x76, 0xa7, 0xff, 0x9e, 0x5d, 0xc5, 0x83, 0x52, 0x8a, 0x11, 0x7, 0x5a, 0x8b, 0xb5, 0xb3, 0x1f, 0x37, 0xf4, 0x2f, 0xad, 0x6d, 0x45, 0xef, 0x0, 0x35, 0xad, 0x30, 0xf, 0x3b, 0xda, 0x5c, 0xe2, 0xe6, 0x9f, 0xa9, 0xce, 0xc6, 0x52, 0xc8, 0xf7, 0x56, 0x8b, 0xe1, 0xe1, 0x97, 0x39, 0x66, 0x80, 0x2f, 0x3d, 0x57, 0x2e, 0x6b, 0x89, 0x2d, 0x15, 0x8f, 0x3d, 0x67, 0xe5, 0x51, 0xac, 0x6c, 0xd1, 0x9f, 0xb6, 0xb1, 0x17, 0x84, 0xdb, 0x6c, 0xa0, 0xb2, 0xf4, 0x35, 0x83, 0x36, 0xe6, 0x77, 0x1f, 0x1, 0x67, 0xde, 0x74, 0x37, 0xb9, 0xba, 0x36, 0x40, 0xe5, 0x49, 0x13, 0x4e, 0x33, 0x1a, 0x8c, 0xb2, 0x1, 0x80, 0x8, 0xc0, 0xbc, 0x72, 0xfb, 0x3e, 0x6f, 0x6d, 0x42, 0xa8, 0xe6, 0x64, 0x69, 0xce, 0xbf, 0x15, 0xed, 0xd4, 0x92, 0x89, 0x2d, 0xc0, 0x6e, 0x1b, 0xb, 0x6e, 0x1d, 0xf5, 0xf9, 0x6, 0x9b, 0x8, 0xf2, 0xbf, 0x87, 0xb9, 0xf9, 0xe9, 0xb4, 0x68, 0x55, 0xda, 0x92, 0x24, 0xcc, 0xe5, 0x61, 0x91, 0xee, 0xc6, 0x9e, 0xfc, 0x72, 0x99, 0x72, 0x9, 0x32, 0xf4, 0x3a, 0x63, 0x3d, 0x85, 0xcd, 0x5c, 0x70, 0x57, 0x86, 0x2, 0x4d, 0xa6, 0xf8, 0x63, 0x84, 0xbb, 0x0, 0xc3, 0xe9, 0x7d, 0xda, 0x28, 0xe5, 0xd5, 0x1b, 0x13, 0xf2, 0x96, 0xea, 0x8, 0x91, 0xda, 0x6b, 0xd7, 0x3, 0x40, 0xce, 0x2a, 0x88, 0xfc, 0x9b, 0x9b, 0xf3, 0x2e, 0xfa, 0xab, 0x74, 0x6e, 0x38, 0x55, 0x18, 0x78, 0x9f, 0xf8, 0xa6, 0x64, 0xd2, 0x6d, 0x2, 0x3, 0x1, 0x0, 0x1, 0xa3, 0x82, 0x1, 0x91, 0x30, 0x82, 0x1, 0x8d, 0x30, 0x12, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x8, 0x30, 0x6, 0x1, 0x1, 0xff, 0x2, 0x1, 0x1, 0x30, 0x81, 0xe4, 0x6, 0x3, 0x55, 0x1d, 0x20, 0x4, 0x81, 0xdc, 0x30, 0x81, 0xd9, 0x30, 0x81, 0xd6, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x2f, 0x1, 0x2, 0x1, 0x30, 0x81, 0xc8, 0x30, 0x36, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x1, 0x16, 0x2a, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6d, 0x69, 0x73, 0x63, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x63, 0x64, 0x73, 0x5f, 0x63, 0x70, 0x2e, 0x68, 0x74, 0x6d, 0x6c, 0x30, 0x81, 0x8d, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x2, 0x30, 0x81, 0x80, 0x1a, 0x7e, 0x54, 0x68, 0x65, 0x20, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x68, 0x61, 0x73, 0x20, 0x62, 0x65, 0x65, 0x6e, 0x20, 0x69, 0x73, 0x73, 0x75, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x63, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x50, 0x6f, 0x6c, 0x69, 0x63, 0x79, 0x20, 0x66, 0x6f, 0x75, 0x6e, 0x64, 0x20, 0x61, 0x74, 0x20, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6d, 0x69, 0x73, 0x63, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x63, 0x64, 0x73, 0x5f, 0x63, 0x70, 0x2e, 0x68, 0x74, 0x6d, 0x6c, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x1d, 0x1f, 0x4, 0x26, 0x30, 0x24, 0x30, 0x22, 0xa0, 0x20, 0xa0, 0x1e, 0x86, 0x1c, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x63, 0x64, 0x73, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x14, 0x6, 0x3, 0x55, 0x1d, 0x25, 0x4, 0xd, 0x30, 0xb, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x2f, 0x1, 0x1, 0x5, 0x30, 0xb, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x4, 0x4, 0x3, 0x2, 0x1, 0x6, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0x59, 0xd8, 0x24, 0xc2, 0xcf, 0x6b, 0x6, 0x42, 0xd4, 0x95, 0x76, 0xb5, 0x29, 0x5c, 0xf5, 0xd8, 0x41, 0x2b, 0x24, 0x5f, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x82, 0xb7, 0x38, 0x4a, 0x93, 0xaa, 0x9b, 0x10, 0xef, 0x80, 0xbb, 0xd9, 0x54, 0xe2, 0xf1, 0xf, 0xfb, 0x80, 0x9c, 0xde, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x3, 0x82, 0x1, 0x1, 0x0, 0x3, 0x80, 0xd0, 0xb, 0x7e, 0x3e, 0x47, 0x44, 0x8a, 0xc5, 0x38, 0xc5, 0x7e, 0x36, 0x5e, 0xcd, 0x60, 0x82, 0x65, 0x62, 0xf3, 0x86, 0xcc, 0x23, 0xbe, 0x47, 0xd7, 0x75, 0x4, 0xc9, 0xdc, 0x5b, 0x3f, 0x12, 0xd3, 0xdb, 0x26, 0x75, 0x59, 0x1d, 0xb6, 0xd4, 0xa, 0x58, 0x3f, 0x5a, 0xd7, 0x28, 0x14, 0xbf, 0x56, 0x86, 0xbd, 0xba, 0xc2, 0x64, 0x20, 0xc8, 0x6e, 0x7d, 0xa9, 0xe8, 0xde, 0xbd, 0x9f, 0x6, 0xff, 0x18, 0xea, 0x57, 0xa1, 0xd7, 0x80, 0x29, 0x87, 0xe8, 0xd0, 0x7b, 0x57, 0x80, 0x45, 0x68, 0x4f, 0x9, 0xa4, 0x83, 0x4c, 0x2d, 0x73, 0x2a, 0x60, 0xee, 0x55, 0x21, 0x9, 0x2c, 0x95, 0x3f, 0xc3, 0x71, 0xe1, 0xb8, 0xad, 0x9c, 0xab, 0x94, 0x37, 0xec, 0x71, 0x4e, 0x66, 0x4d, 0xe8, 0x20, 0x49, 0x4a, 0xea, 0xc7, 0xd7, 0x3c, 0x96, 0xeb, 0xf, 0xc5, 0xf9, 0xd4, 0x7d, 0x5c, 0x13, 0x81, 0x3a, 0xdf, 0xc3, 0x18, 0xc0, 0x47, 0xae, 0x84, 0xff, 0x12, 0xfd, 0xb8, 0x4, 0xc9, 0x9a, 0x5b, 0x33, 0x83, 0x6e, 0xec, 0x11, 0x2c, 0x35, 0x3, 0x71, 0xac, 0xa5, 0xd1, 0xd3, 0x52, 0xd9, 0x9a, 0x70, 0xcf, 0xfe, 0x9d, 0xc0, 0x7e, 0x3, 0x2d, 0x86, 0xe7, 0x7d, 0x66, 0x10, 0x2c, 0x45, 0x86, 0x37, 0x88, 0x9c, 0xe3, 0xb5, 0x39, 0x9f, 0x1c, 0xb2, 0x67, 0x96, 0x43, 0x3a, 0xc2, 0x2c, 0xcd, 0xfa, 0x3d, 0x9a, 0x4, 0x32, 0x5c, 0xb1, 0x3d, 0xf3, 0x2, 0xb4, 0x31, 0x1, 0x4a, 0xdf, 0x92, 0x5b, 0x7d, 0x5f, 0xb6, 0xd0, 0x7, 0xc, 0x93, 0xae, 0xe5, 0xa9, 0xe2, 0x19, 0xdb, 0xc, 0xa3, 0xf3, 0xc8, 0xb1, 0x14, 0xd, 0x61, 0xb9, 0x3, 0x9e, 0xd3, 0xf4, 0x7, 0x17, 0x2b, 0xd9, 0x4c, 0x70, 0x5e, 0x48, 0x6f, 0xc2, 0xc5, 0x3a, 0x9, 0xb0, 0xed, 0x27, 0x5a, 0x11, 0xcb, 0x31, 0x82, 0x2, 0xc6, 0x30, 0x82, 0x2, 0xc2, 0x2, 0x1, 0x1, 0x30, 0x81, 0x88, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x2, 0x12, 0x11, 0x21, 0xd2, 0xd1, 0x82, 0xa, 0xf0, 0xa5, 0xc5, 0x48, 0x14, 0x9a, 0x32, 0x74, 0xdc, 0xc3, 0x43, 0x57, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0xa0, 0x82, 0x1, 0xe, 0x30, 0x1a, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x3, 0x31, 0xd, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1, 0x4, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x4, 0x31, 0x22, 0x4, 0x20, 0xb1, 0xcd, 0x2b, 0x8d, 0x14, 0x62, 0xe9, 0x8c, 0x98, 0xa6, 0x47, 0x4c, 0x2d, 0x2d, 0xac, 0x61, 0xbf, 0x32, 0x6a, 0xfb, 0xf5, 0x3b, 0xf8, 0xf1, 0xe7, 0x4a, 0x9a, 0xb7, 0xc3, 0x5b, 0xaa, 0xb, 0x30, 0x81, 0xbe, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x2, 0xc, 0x31, 0x81, 0xae, 0x30, 0x81, 0xab, 0x30, 0x81, 0xa8, 0x30, 0x81, 0xa5, 0x4, 0x14, 0xba, 0x64, 0xa3, 0x69, 0x93, 0xb8, 0xb6, 0x4, 0xf7, 0xfe, 0x65, 0xd3, 0x7d, 0x8f, 0x8e, 0x68, 0x96, 0xfc, 0x96, 0xb, 0x30, 0x81, 0x8c, 0x30, 0x76, 0xa4, 0x74, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x2, 0x12, 0x11, 0x21, 0xd2, 0xd1, 0x82, 0xa, 0xf0, 0xa5, 0xc5, 0x48, 0x14, 0x9a, 0x32, 0x74, 0xdc, 0xc3, 0x43, 0x57, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x4, 0x82, 0x1, 0x0, 0x21, 0xd7, 0xe8, 0x74, 0xba, 0x1, 0xb0, 0xb, 0xc2, 0x57, 0x24, 0x62, 0x7b, 0xbd, 0xfb, 0xf2, 0xaa, 0xeb, 0x30, 0x3b, 0x89, 0x51, 0xb7, 0x7f, 0x96, 0x80, 0x84, 0xff, 0x55, 0xbf, 0x0, 0x36, 0x4c, 0xaf, 0xa2, 0x65, 0x36, 0x26, 0x53, 0xda, 0x4c, 0x19, 0xa4, 0x61, 0x97, 0x6a, 0x8, 0xd1, 0xd9, 0x6d, 0x61, 0x8b, 0x67, 0x6b, 0xc7, 0xc5, 0xbe, 0x46, 0x10, 0x8a, 0xf, 0x53, 0x5e, 0xa8, 0x7, 0x67, 0xe8, 0xf3, 0x3c, 0x3f, 0x95, 0x47, 0xf8, 0xca, 0x50, 0xbe, 0x9e, 0xf7, 0x2f, 0xc2, 0x4d, 0x50, 0xf8, 0xe6, 0xde, 0xe2, 0x9b, 0x61, 0xe0, 0x52, 0xc0, 0x2e, 0x45, 0x4a, 0x29, 0x4f, 0xe2, 0xbb, 0x7, 0xa6, 0xd5, 0xd4, 0xa8, 0x17, 0x90, 0xad, 0xf3, 0xf7, 0xfd, 0x31, 0x2d, 0x29, 0x26, 0x2, 0xd2, 0x9e, 0xdd, 0x9e, 0xd1, 0x8a, 0xa1, 0x25, 0x9b, 0x36, 0x0, 0xbf, 0xe1, 0x96, 0x96, 0xe8, 0x8f, 0x89, 0x57, 0xa3, 0x6f, 0x7f, 0xb0, 0x84, 0x7a, 0x63, 0x63, 0x9c, 0xf1, 0xc6, 0x26, 0x60, 0x97, 0x64, 0xf7, 0xbc, 0x70, 0x69, 0x1b, 0xa5, 0x9d, 0x6d, 0x33, 0x50, 0x58, 0x9e, 0x41, 0xa4, 0x21, 0x5c, 0x2a, 0x53, 0x9c, 0x4c, 0xae, 0x27, 0xfc, 0x8b, 0x92, 0xaf, 0x68, 0xb, 0x58, 0xab, 0x3b, 0x51, 0x9, 0xc, 0xc3, 0x7f, 0xcc, 0x6b, 0x47, 0x7e, 0xb5, 0xe, 0xf8, 0x6e, 0x11, 0xb4, 0x38, 0xdf, 0xeb, 0xfa, 0xaa, 0x33, 0x1e, 0xe2, 0xd6, 0xba, 0x93, 0x2e, 0xf2, 0x2a, 0x8f, 0xaf, 0x5, 0xb7, 0x21, 0x57, 0xda, 0x47, 0x93, 0xc1, 0xc4, 0x52, 0x4e, 0xce, 0xb1, 0x87, 0xc5, 0x56, 0x16, 0x4d, 0x18, 0x6d, 0xcb, 0xde, 0x82, 0xd1, 0x81, 0x80, 0x2f, 0xa8, 0x45, 0x9a, 0x41, 0xe3, 0x29, 0x49, 0x50, 0xbf, 0x9e, 0x32, 0xc0, 0xb7, 0x7e, 0x35, 0x32, 0x8, 0xd5, 0x2a}

// Time-Stamp request without certificate, to create with OpenSSL:
//
//	$ openssl ts -query -data data.txt -sha256 -out reqnonoce.tsq
var reqNoCert = []byte{0x30, 0x41, 0x2, 0x1, 0x1, 0x30, 0x31, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x4, 0x20, 0xf4, 0x87, 0xd8, 0x81, 0x64, 0xbd, 0xe2, 0x23, 0xb7, 0xfd, 0x3c, 0x71, 0xb3, 0xe9, 0xc6, 0x24, 0xc8, 0xa0, 0xcb, 0x55, 0x9c, 0x85, 0x3b, 0x49, 0x55, 0x8e, 0x7e, 0x5e, 0xf4, 0xce, 0x55, 0xf0, 0x2, 0x9, 0x0, 0xb1, 0xfc, 0x81, 0xde, 0xc9, 0x57, 0x49, 0xd9}

var respNoCert = []byte{0x30, 0x82, 0x3, 0xeb, 0x30, 0x3, 0x2, 0x1, 0x0, 0x30, 0x82, 0x3, 0xe2, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x7, 0x2, 0xa0, 0x82, 0x3, 0xd3, 0x30, 0x82, 0x3, 0xcf, 0x2, 0x1, 0x3, 0x31, 0xf, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x30, 0x81, 0xee, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1, 0x4, 0xa0, 0x81, 0xde, 0x4, 0x81, 0xdb, 0x30, 0x81, 0xd8, 0x2, 0x1, 0x1, 0x6, 0x9, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xa0, 0x32, 0x2, 0x1, 0x30, 0x31, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x4, 0x20, 0xf4, 0x87, 0xd8, 0x81, 0x64, 0xbd, 0xe2, 0x23, 0xb7, 0xfd, 0x3c, 0x71, 0xb3, 0xe9, 0xc6, 0x24, 0xc8, 0xa0, 0xcb, 0x55, 0x9c, 0x85, 0x3b, 0x49, 0x55, 0x8e, 0x7e, 0x5e, 0xf4, 0xce, 0x55, 0xf0, 0x2, 0x14, 0x48, 0xf7, 0x9a, 0x1f, 0xa2, 0x58, 0x61, 0xe1, 0xc4, 0xea, 0x9d, 0x48, 0x9e, 0x6, 0xb9, 0xce, 0xf9, 0xaa, 0xcc, 0x1d, 0x18, 0xf, 0x32, 0x30, 0x31, 0x37, 0x30, 0x34, 0x31, 0x39, 0x30, 0x36, 0x33, 0x32, 0x34, 0x33, 0x5a, 0x30, 0x3, 0x2, 0x1, 0x1, 0x2, 0x9, 0x0, 0xb1, 0xfc, 0x81, 0xde, 0xc9, 0x57, 0x49, 0xd9, 0xa0, 0x5e, 0xa4, 0x5c, 0x30, 0x5a, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x53, 0x47, 0x31, 0x1f, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x16, 0x47, 0x4d, 0x4f, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x74, 0x65, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x2a, 0x30, 0x28, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x21, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x53, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x43, 0x44, 0x53, 0x20, 0x2d, 0x20, 0x47, 0x32, 0x31, 0x82, 0x2, 0xc6, 0x30, 0x82, 0x2, 0xc2, 0x2, 0x1, 0x1, 0x30, 0x81, 0x88, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x2, 0x12, 0x11, 0x21, 0xd2, 0xd1, 0x82, 0xa, 0xf0, 0xa5, 0xc5, 0x48, 0x14, 0x9a, 0x32, 0x74, 0xdc, 0xc3, 0x43, 0x57, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0xa0, 0x82, 0x1, 0xe, 0x30, 0x1a, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x3, 0x31, 0xd, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1, 0x4, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x4, 0x31, 0x22, 0x4, 0x20, 0x6e, 0x24, 0x6d, 0x44, 0xcb, 0xa4, 0x33, 0x6c, 0xb7, 0xe5, 0x58, 0x1a, 0x54, 0x5f, 0xcc, 0x50, 0x7c, 0x5, 0x3, 0x56, 0x20, 0x31, 0xb4, 0x2d, 0x3c, 0xbf, 0x9c, 0xc8, 0x1a, 0xf4, 0xb8, 0x39, 0x30, 0x81, 0xbe, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x2, 0xc, 0x31, 0x81, 0xae, 0x30, 0x81, 0xab, 0x30, 0x81, 0xa8, 0x30, 0x81, 0xa5, 0x4, 0x14, 0xba, 0x64, 0xa3, 0x69, 0x93, 0xb8, 0xb6, 0x4, 0xf7, 0xfe, 0x65, 0xd3, 0x7d, 0x8f, 0x8e, 0x68, 0x96, 0xfc, 0x96, 0xb, 0x30, 0x81, 0x8c, 0x30, 0x76, 0xa4, 0x74, 0x30, 0x72, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x17, 0x30, 0x15, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0xe, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x44, 0x53, 0x31, 0x2f, 0x30, 0x2d, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x26, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x2, 0x12, 0x11, 0x21, 0xd2, 0xd1, 0x82, 0xa, 0xf0, 0xa5, 0xc5, 0x48, 0x14, 0x9a, 0x32, 0x74, 0xdc, 0xc3, 0x43, 0x57, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x4, 0x82, 0x1, 0x0, 0xb5, 0xa5, 0x55, 0x83, 0xf, 0x16, 0xe4, 0x78, 0x99, 0xfe, 0xc7, 0x52, 0x68, 0x90, 0xe4, 0x1e, 0xc8, 0xc7, 0xb0, 0x45, 0xf3, 0xa5, 0x2d, 0xc, 0xd1, 0x5f, 0xad, 0x71, 0x7c, 0x6, 0x7d, 0x65, 0xc2, 0x8f, 0x47, 0x12, 0x43, 0x4e, 0x3d, 0x5d, 0xa3, 0x52, 0x4e, 0x18, 0x98, 0x6d, 0x60, 0x47, 0xda, 0x60, 0x5c, 0x33, 0xfc, 0xda, 0x5b, 0x27, 0x5d, 0x1a, 0x8, 0xec, 0xc4, 0xbe, 0xb8, 0x4e, 0x31, 0x8e, 0xf9, 0x84, 0x8b, 0x7c, 0x57, 0xe4, 0x5b, 0x13, 0xca, 0xed, 0x44, 0x2f, 0x24, 0x29, 0x5d, 0x76, 0x75, 0x34, 0x6, 0x36, 0x1c, 0x0, 0x8, 0x4, 0x8d, 0x3d, 0x2d, 0x14, 0xfb, 0xc8, 0xb3, 0x92, 0x9b, 0x88, 0x14, 0x61, 0x44, 0xf4, 0x82, 0xbd, 0xb, 0x61, 0x36, 0xd2, 0x4e, 0x2b, 0xc8, 0xae, 0x75, 0x77, 0x2e, 0x54, 0xe7, 0x2d, 0xd5, 0x42, 0xe5, 0x23, 0xc, 0x1e, 0xb3, 0x80, 0xc7, 0x13, 0xe3, 0x13, 0xb5, 0x63, 0x59, 0x5a, 0x95, 0x4e, 0x33, 0x3, 0xe6, 0x31, 0x82, 0xbc, 0x85, 0xa6, 0x66, 0x48, 0xfa, 0x42, 0x47, 0x6a, 0x76, 0x9a, 0x97, 0x56, 0x6e, 0x1a, 0x33, 0xaa, 0x7d, 0xe5, 0x5e, 0xf9, 0xd, 0x24, 0x19, 0xed, 0xf8, 0x37, 0xe, 0xd0, 0x91, 0xa5, 0x9, 0x72, 0x58, 0x66, 0xf8, 0x13, 0x6c, 0x77, 0x99, 0x8f, 0x1c, 0xd9, 0xad, 0x17, 0x3f, 0xa3, 0xd7, 0x2a, 0x91, 0x93, 0xf9, 0x36, 0x36, 0x6f, 0x6, 0x53, 0x0, 0x23, 0xce, 0x18, 0x5d, 0xd, 0x98, 0x8a, 0x24, 0x94, 0x7f, 0x6d, 0xb9, 0x13, 0xcc, 0xa7, 0xcb, 0x89, 0x5f, 0x3, 0x23, 0x1a, 0x49, 0x70, 0x9b, 0x75, 0x9f, 0xf9, 0x64, 0x42, 0x75, 0xcb, 0x76, 0x2f, 0xb9, 0x82, 0x5e, 0x1c, 0xa9, 0x41, 0xd4, 0x6d, 0x84, 0x4e, 0xf8, 0x2d, 0x93, 0x2c, 0x4c, 0x4f, 0x21, 0x82, 0x6b, 0x5b, 0x95, 0xe3}

// TimeStampToken excluding response struct
var timeStampToken = []byte{0x30, 0x82, 0xc, 0xdc, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x7, 0x2, 0xa0, 0x82, 0xc, 0xcd, 0x30, 0x82, 0xc, 0xc9, 0x2, 0x1, 0x3, 0x31, 0xf, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x30, 0x81, 0xf1, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1, 0x4, 0xa0, 0x81, 0xe1, 0x4, 0x81, 0xde, 0x30, 0x81, 0xdb, 0x2, 0x1, 0x1, 0x6, 0x9, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xa0, 0x32, 0x1, 0x1f, 0x30, 0x31, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0x4, 0x20, 0x2d, 0xa8, 0x2c, 0x15, 0x72, 0xff, 0x2, 0x5, 0x24, 0xe0, 0x50, 0x69, 0x21, 0x99, 0x75, 0xc9, 0xba, 0x90, 0x72, 0x3f, 0x1a, 0x4d, 0xd5, 0xb9, 0x72, 0x2a, 0xee, 0x8e, 0xc5, 0x47, 0xa2, 0xff, 0x2, 0x14, 0x6d, 0x43, 0xbe, 0x1c, 0xa9, 0xbc, 0x92, 0x5, 0x4c, 0x31, 0xc9, 0x7, 0x2c, 0xd2, 0x42, 0x49, 0x74, 0x1d, 0x1e, 0x7e, 0x18, 0xf, 0x32, 0x30, 0x31, 0x37, 0x30, 0x32, 0x30, 0x31, 0x31, 0x35, 0x33, 0x39, 0x35, 0x30, 0x5a, 0x30, 0x3, 0x2, 0x1, 0x1, 0x2, 0x8, 0x3f, 0x2b, 0x95, 0xd0, 0x72, 0x8b, 0xed, 0x57, 0xa0, 0x62, 0xa4, 0x60, 0x30, 0x5e, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x53, 0x47, 0x31, 0x1f, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x16, 0x47, 0x4d, 0x4f, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x74, 0x65, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x2e, 0x30, 0x2c, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x25, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x53, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x41, 0x54, 0x4c, 0x20, 0x2d, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x2d, 0x20, 0x47, 0x32, 0xa0, 0x82, 0x9, 0x2c, 0x30, 0x82, 0x4, 0xd1, 0x30, 0x82, 0x3, 0xb9, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x12, 0x11, 0x21, 0x12, 0x42, 0x28, 0x72, 0xda, 0x4f, 0xa5, 0x6c, 0x1, 0x3b, 0x29, 0x7c, 0xaf, 0x65, 0xf1, 0xfb, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x30, 0x57, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x2d, 0x30, 0x2b, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x24, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x41, 0x54, 0x4c, 0x20, 0x2d, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x2d, 0x20, 0x47, 0x32, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x36, 0x30, 0x35, 0x32, 0x34, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0xd, 0x32, 0x37, 0x30, 0x36, 0x32, 0x34, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x5e, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x53, 0x47, 0x31, 0x1f, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x16, 0x47, 0x4d, 0x4f, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x74, 0x65, 0x20, 0x4c, 0x74, 0x64, 0x31, 0x2e, 0x30, 0x2c, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x25, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x53, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x41, 0x54, 0x4c, 0x20, 0x2d, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x2d, 0x20, 0x47, 0x32, 0x30, 0x82, 0x1, 0x22, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x3, 0x82, 0x1, 0xf, 0x0, 0x30, 0x82, 0x1, 0xa, 0x2, 0x82, 0x1, 0x1, 0x0, 0xc3, 0xcf, 0xdd, 0x35, 0xb0, 0x13, 0xc1, 0xdb, 0xae, 0xdd, 0x96, 0xef, 0x26, 0xfb, 0x6a, 0x1e, 0x78, 0xf, 0x1a, 0x44, 0x98, 0x0, 0x70, 0x60, 0x31, 0x8, 0xde, 0x32, 0x7c, 0x1b, 0xad, 0x6a, 0xa0, 0xa9, 0xf7, 0xd6, 0x60, 0x0, 0x50, 0x69, 0x33, 0x59, 0x9c, 0x7c, 0x14, 0xa6, 0x93, 0x2a, 0xba, 0xaf, 0xfe, 0xec, 0x14, 0x82, 0x1c, 0x9b, 0x7c, 0xd2, 0x8e, 0xb8, 0xc1, 0x36, 0x1f, 0xcc, 0xfa, 0x62, 0xef, 0x59, 0xc0, 0xd2, 0x38, 0x8b, 0x33, 0x4b, 0x35, 0x3f, 0x56, 0xfc, 0x2c, 0xd5, 0x27, 0x8d, 0x18, 0xfd, 0x7d, 0x35, 0x9e, 0xf1, 0x64, 0x7e, 0x90, 0xd2, 0x92, 0x78, 0xcc, 0x23, 0xc6, 0xa8, 0x23, 0x25, 0x40, 0x17, 0x18, 0x22, 0x24, 0xd0, 0x44, 0x61, 0x48, 0x9d, 0x18, 0xba, 0xd8, 0x20, 0x2c, 0x21, 0x18, 0x2c, 0xcf, 0xcf, 0x1c, 0x28, 0x7b, 0xb2, 0xf3, 0xde, 0x8e, 0xa8, 0xd8, 0xc4, 0xd9, 0xe1, 0x63, 0x6e, 0x3e, 0xa, 0x0, 0x4e, 0x1d, 0x89, 0x5d, 0xa1, 0x5a, 0x64, 0x0, 0xb4, 0x16, 0x63, 0x30, 0xf7, 0xa6, 0x42, 0xfc, 0x27, 0x7b, 0xae, 0xea, 0x2b, 0x3d, 0x78, 0x3, 0x3, 0xd3, 0x8b, 0x69, 0xd5, 0x8a, 0xf4, 0xff, 0x55, 0x0, 0xfa, 0xfe, 0xd3, 0xff, 0xc6, 0x58, 0xca, 0x6a, 0x1d, 0x60, 0x78, 0x15, 0xdf, 0x1c, 0xb0, 0xff, 0xe2, 0xd8, 0x21, 0x17, 0x45, 0xce, 0xb9, 0xc9, 0x30, 0xc7, 0x0, 0x46, 0xde, 0x35, 0xe1, 0x1f, 0x74, 0x1b, 0xe4, 0x3d, 0x25, 0xf3, 0xdd, 0xfb, 0xb2, 0xd0, 0xeb, 0x5b, 0xe3, 0x51, 0x13, 0x38, 0x1b, 0x4, 0x8b, 0xc8, 0x0, 0x83, 0x83, 0x58, 0xb8, 0x64, 0xbe, 0xd3, 0xc1, 0xb5, 0xc4, 0xce, 0xeb, 0x70, 0xcd, 0x9a, 0x91, 0x5, 0x6b, 0x61, 0xbd, 0xd2, 0xd7, 0xbc, 0x29, 0x5d, 0xb7, 0xc5, 0x56, 0xf0, 0x6, 0x1, 0xaf, 0x2, 0x3, 0x1, 0x0, 0x1, 0xa3, 0x82, 0x1, 0x8e, 0x30, 0x82, 0x1, 0x8a, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x7, 0x80, 0x30, 0x4c, 0x6, 0x3, 0x55, 0x1d, 0x20, 0x4, 0x45, 0x30, 0x43, 0x30, 0x41, 0x6, 0x9, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xa0, 0x32, 0x1, 0x1f, 0x30, 0x34, 0x30, 0x32, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x1, 0x16, 0x26, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79, 0x2f, 0x30, 0x9, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x4, 0x2, 0x30, 0x0, 0x30, 0x16, 0x6, 0x3, 0x55, 0x1d, 0x25, 0x1, 0x1, 0xff, 0x4, 0xc, 0x30, 0xa, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x3, 0x8, 0x30, 0x3e, 0x6, 0x3, 0x55, 0x1d, 0x1f, 0x4, 0x37, 0x30, 0x35, 0x30, 0x33, 0xa0, 0x31, 0xa0, 0x2f, 0x86, 0x2d, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x73, 0x2f, 0x67, 0x73, 0x61, 0x61, 0x74, 0x6c, 0x73, 0x68, 0x61, 0x32, 0x67, 0x32, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x81, 0x86, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x1, 0x1, 0x4, 0x7a, 0x30, 0x78, 0x30, 0x40, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x30, 0x2, 0x86, 0x34, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x73, 0x65, 0x63, 0x75, 0x72, 0x65, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x63, 0x61, 0x63, 0x65, 0x72, 0x74, 0x2f, 0x67, 0x73, 0x61, 0x61, 0x74, 0x6c, 0x73, 0x68, 0x61, 0x32, 0x67, 0x32, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x34, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x30, 0x1, 0x86, 0x28, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6f, 0x63, 0x73, 0x70, 0x32, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x73, 0x61, 0x61, 0x74, 0x6c, 0x73, 0x68, 0x61, 0x32, 0x67, 0x32, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0xf4, 0xd4, 0xc1, 0x17, 0x61, 0x17, 0x11, 0x1e, 0xc, 0x4e, 0x6e, 0xa1, 0x3b, 0x40, 0xb8, 0x87, 0xb6, 0x94, 0x87, 0x3a, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x60, 0xc2, 0xf1, 0x52, 0x3e, 0xad, 0x8c, 0x13, 0xdc, 0xdb, 0xca, 0xe, 0xfa, 0xa, 0xe6, 0x2a, 0x2c, 0x99, 0x4b, 0xd8, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x3, 0x82, 0x1, 0x1, 0x0, 0xea, 0x88, 0x3f, 0xd2, 0xef, 0x94, 0xf, 0x5d, 0x54, 0xc2, 0x52, 0x68, 0x91, 0x23, 0xe7, 0x12, 0xe2, 0x72, 0xc3, 0x4, 0xe7, 0xea, 0x88, 0x68, 0xa1, 0xba, 0x20, 0xde, 0x25, 0x77, 0x49, 0x3b, 0x7b, 0x10, 0xf7, 0xa3, 0x60, 0x24, 0xc9, 0xe5, 0xcd, 0x38, 0x9e, 0xc7, 0x1a, 0xf6, 0xa, 0xe6, 0x14, 0x83, 0xfc, 0x7c, 0x38, 0x73, 0x12, 0x40, 0xfc, 0xbe, 0x2e, 0xb8, 0x39, 0xb4, 0x47, 0xaa, 0x75, 0x2c, 0x6a, 0x71, 0x53, 0x6e, 0x4c, 0x3c, 0xc5, 0xda, 0xbe, 0x90, 0xca, 0x66, 0x83, 0x1f, 0x44, 0x1a, 0xf1, 0xf5, 0x80, 0xa0, 0x70, 0x1d, 0xe2, 0xb, 0xb6, 0x6f, 0x4a, 0x34, 0xf9, 0x66, 0xd6, 0x3a, 0xb, 0x9d, 0x57, 0xb7, 0xef, 0x9b, 0x0, 0xb7, 0x5f, 0x2, 0x2e, 0xc4, 0xd9, 0x38, 0x45, 0xa2, 0x45, 0xcb, 0xe8, 0x1, 0x23, 0xaf, 0x64, 0xf, 0x3b, 0x4a, 0x33, 0x24, 0xa7, 0x19, 0x94, 0xb4, 0x52, 0x30, 0x31, 0x89, 0x21, 0x35, 0x3c, 0xe6, 0xee, 0xa0, 0xe7, 0xdd, 0xcf, 0x1c, 0xaa, 0x3, 0x87, 0xe8, 0x3b, 0xb0, 0x54, 0xae, 0x9a, 0x37, 0x3c, 0xd4, 0x23, 0xb4, 0x94, 0xce, 0x7e, 0xc0, 0xa1, 0xc, 0x6c, 0x63, 0x70, 0x33, 0x45, 0xec, 0xc7, 0xe8, 0xf2, 0xfc, 0x2b, 0x64, 0xdd, 0xd6, 0xe1, 0xd7, 0x69, 0xe8, 0x2, 0xb0, 0x96, 0xac, 0xb7, 0x13, 0x34, 0x78, 0xb8, 0x94, 0x90, 0x6c, 0x55, 0x8e, 0x4e, 0x55, 0xe3, 0xb6, 0xc5, 0x6, 0xee, 0x15, 0xd7, 0x1, 0xc2, 0xca, 0xae, 0x42, 0xf6, 0x59, 0x7b, 0xd1, 0x59, 0xa4, 0x2f, 0x73, 0xb0, 0x16, 0x9a, 0x5e, 0xd7, 0xb, 0x8a, 0xfb, 0x74, 0x3c, 0x89, 0x33, 0x77, 0x57, 0x46, 0xc4, 0x63, 0x3b, 0x31, 0xa4, 0x5, 0xb, 0x67, 0x5b, 0xca, 0x10, 0x5e, 0x51, 0xd7, 0xea, 0x4b, 0xbb, 0xcd, 0x73, 0x9f, 0x81, 0xb2, 0x38, 0x30, 0x82, 0x4, 0x53, 0x30, 0x82, 0x3, 0x3b, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0xb, 0x4, 0x0, 0x0, 0x0, 0x0, 0x1, 0x44, 0x4f, 0x2, 0x36, 0xb6, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x30, 0x4c, 0x31, 0x20, 0x30, 0x1e, 0x6, 0x3, 0x55, 0x4, 0xb, 0x13, 0x17, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x20, 0x2d, 0x20, 0x52, 0x33, 0x31, 0x13, 0x30, 0x11, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0xa, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x31, 0x13, 0x30, 0x11, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0xa, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x34, 0x30, 0x32, 0x32, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0xd, 0x32, 0x39, 0x30, 0x33, 0x31, 0x38, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x57, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x2d, 0x30, 0x2b, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x24, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x41, 0x54, 0x4c, 0x20, 0x2d, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x2d, 0x20, 0x47, 0x32, 0x30, 0x82, 0x1, 0x22, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x3, 0x82, 0x1, 0xf, 0x0, 0x30, 0x82, 0x1, 0xa, 0x2, 0x82, 0x1, 0x1, 0x0, 0xf9, 0xfc, 0x9e, 0x98, 0x63, 0xd3, 0xe, 0xca, 0x30, 0xe8, 0xfa, 0xab, 0xd0, 0x83, 0x18, 0x53, 0x51, 0x3f, 0x92, 0x8d, 0xc3, 0x1b, 0x75, 0xf, 0x5d, 0x3, 0xa5, 0x14, 0xc7, 0x25, 0x31, 0xbb, 0x1d, 0x33, 0xd0, 0xa9, 0xb9, 0xe1, 0x41, 0xeb, 0x38, 0x3e, 0xa1, 0x45, 0xef, 0x35, 0x7c, 0x2d, 0x5b, 0xf0, 0xad, 0xa8, 0xe4, 0x8b, 0xe3, 0xe4, 0x5f, 0x2b, 0x2c, 0xe5, 0x78, 0x4b, 0x87, 0x32, 0x81, 0xd5, 0x16, 0x50, 0x93, 0xb8, 0x96, 0xc2, 0x6a, 0xc, 0x81, 0xed, 0x57, 0x1, 0x84, 0x86, 0xb9, 0x0, 0xa3, 0x38, 0x7, 0x9a, 0xc8, 0x45, 0x5c, 0xc0, 0xca, 0x56, 0xdc, 0x14, 0x8e, 0xb3, 0xda, 0x97, 0x61, 0xaa, 0xb5, 0x99, 0x23, 0x22, 0x95, 0xbe, 0x70, 0xd6, 0x11, 0xd9, 0x83, 0x82, 0xae, 0xaf, 0xc9, 0xc5, 0x2b, 0x4b, 0xc, 0x6a, 0xcd, 0x62, 0x56, 0x2a, 0x28, 0x61, 0xcf, 0x64, 0x8b, 0xc5, 0xee, 0x57, 0xda, 0x3f, 0xc, 0x6e, 0x2d, 0x6e, 0x31, 0x31, 0x97, 0x7c, 0x54, 0x9d, 0xf6, 0x12, 0x61, 0x4d, 0xe8, 0x77, 0xa8, 0xe8, 0x2, 0x73, 0x20, 0xa6, 0x4c, 0x15, 0xe8, 0xf1, 0x23, 0xee, 0x86, 0x86, 0x52, 0xab, 0x83, 0xb9, 0x52, 0x3a, 0xc7, 0xe7, 0xad, 0x9b, 0x40, 0x1f, 0xa2, 0x18, 0x17, 0x53, 0xff, 0xff, 0x6f, 0xd, 0x74, 0xfb, 0xc5, 0xdc, 0xd1, 0x6a, 0x57, 0x8, 0xdc, 0x67, 0x81, 0xd7, 0xa8, 0x2c, 0xb8, 0xa3, 0xb6, 0xe1, 0x54, 0xab, 0xf3, 0xc5, 0xfb, 0xe4, 0x41, 0x21, 0x4, 0x6, 0xcb, 0xb4, 0x6d, 0xf9, 0x43, 0xc2, 0xd9, 0x45, 0x4c, 0x94, 0x53, 0xc6, 0xa5, 0x28, 0xef, 0xc3, 0xde, 0xf, 0xda, 0xf7, 0xea, 0x38, 0x5c, 0x26, 0xbd, 0x54, 0x2a, 0xea, 0xe0, 0x60, 0xbf, 0x58, 0x9f, 0x2c, 0x2c, 0x60, 0x6c, 0xa4, 0x40, 0x7e, 0x3a, 0x4d, 0x19, 0x41, 0x2, 0x3, 0x1, 0x0, 0x1, 0xa3, 0x82, 0x1, 0x29, 0x30, 0x82, 0x1, 0x25, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x1, 0x6, 0x30, 0x12, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x8, 0x30, 0x6, 0x1, 0x1, 0xff, 0x2, 0x1, 0x1, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0x60, 0xc2, 0xf1, 0x52, 0x3e, 0xad, 0x8c, 0x13, 0xdc, 0xdb, 0xca, 0xe, 0xfa, 0xa, 0xe6, 0x2a, 0x2c, 0x99, 0x4b, 0xd8, 0x30, 0x47, 0x6, 0x3, 0x55, 0x1d, 0x20, 0x4, 0x40, 0x30, 0x3e, 0x30, 0x3c, 0x6, 0x4, 0x55, 0x1d, 0x20, 0x0, 0x30, 0x34, 0x30, 0x32, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x2, 0x1, 0x16, 0x26, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79, 0x2f, 0x30, 0x36, 0x6, 0x3, 0x55, 0x1d, 0x1f, 0x4, 0x2f, 0x30, 0x2d, 0x30, 0x2b, 0xa0, 0x29, 0xa0, 0x27, 0x86, 0x25, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x6e, 0x65, 0x74, 0x2f, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x72, 0x33, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x3e, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x1, 0x1, 0x4, 0x32, 0x30, 0x30, 0x30, 0x2e, 0x6, 0x8, 0x2b, 0x6, 0x1, 0x5, 0x5, 0x7, 0x30, 0x1, 0x86, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6f, 0x63, 0x73, 0x70, 0x32, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x6f, 0x6f, 0x74, 0x72, 0x33, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x8f, 0xf0, 0x4b, 0x7f, 0xa8, 0x2e, 0x45, 0x24, 0xae, 0x4d, 0x50, 0xfa, 0x63, 0x9a, 0x8b, 0xde, 0xe2, 0xdd, 0x1b, 0xbc, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x3, 0x82, 0x1, 0x1, 0x0, 0x13, 0xa2, 0xa9, 0xae, 0x72, 0x73, 0x5b, 0x11, 0x5f, 0x37, 0x47, 0xd0, 0xff, 0xf4, 0x4, 0x82, 0xe4, 0xfc, 0xf1, 0x36, 0x5d, 0xb2, 0x4d, 0xa7, 0x13, 0x87, 0x94, 0xe4, 0x3e, 0xda, 0x73, 0xec, 0x27, 0x7e, 0xa8, 0xa1, 0xbb, 0x7a, 0x12, 0x41, 0x23, 0x9e, 0x36, 0x18, 0xdd, 0x22, 0x62, 0x56, 0x52, 0x9d, 0x1c, 0x7d, 0xf3, 0x96, 0x18, 0x25, 0x97, 0x12, 0xe0, 0x2e, 0xf3, 0xf8, 0x4a, 0x42, 0x3b, 0xa1, 0x1e, 0x82, 0x1c, 0xe7, 0x73, 0xe4, 0x73, 0x77, 0xae, 0x40, 0x23, 0x46, 0x20, 0x8d, 0x10, 0xfb, 0x8d, 0x3a, 0xd7, 0xcd, 0x3d, 0x10, 0x2d, 0x5f, 0xf3, 0x22, 0xda, 0x50, 0xfb, 0x31, 0xef, 0xd7, 0x72, 0xc4, 0x50, 0xd3, 0x94, 0x89, 0x0, 0x60, 0xda, 0xde, 0x2, 0x9d, 0xa8, 0x5e, 0x9c, 0x39, 0x79, 0xeb, 0xb2, 0x8, 0x3c, 0xc4, 0xdb, 0xac, 0x2b, 0x7a, 0xcd, 0x61, 0xf8, 0xee, 0xfc, 0xbd, 0x20, 0xbf, 0x71, 0x94, 0x5d, 0xa, 0x7a, 0x99, 0xda, 0x79, 0xa8, 0x3d, 0xe, 0x97, 0x1, 0x3d, 0x0, 0x3c, 0x3f, 0x39, 0xa7, 0x63, 0x1f, 0xa3, 0xaf, 0x4f, 0xed, 0x6f, 0x7, 0x1d, 0x7f, 0x3b, 0x77, 0xca, 0xd0, 0x43, 0xac, 0xa0, 0x29, 0x49, 0x2d, 0x4, 0x29, 0x36, 0xc2, 0xb9, 0x44, 0x7f, 0xe7, 0xb3, 0xa9, 0xf7, 0x84, 0xd6, 0x9, 0x3f, 0x7b, 0x2e, 0x87, 0x95, 0xe3, 0xbd, 0xb7, 0xfe, 0xf7, 0x96, 0x8b, 0xc5, 0x1c, 0x45, 0xae, 0xc, 0x2e, 0x29, 0x85, 0x83, 0x30, 0x50, 0xf, 0x3a, 0x41, 0xdb, 0x45, 0x7d, 0x4f, 0x34, 0x40, 0x7e, 0x50, 0xcf, 0x12, 0xa2, 0x78, 0x61, 0x16, 0x1f, 0x37, 0x3c, 0xfe, 0x51, 0x67, 0xe3, 0xd, 0xd6, 0xbb, 0x14, 0x50, 0xf3, 0x9c, 0x7, 0x42, 0x2c, 0xbd, 0x7f, 0x7d, 0x90, 0xc, 0x70, 0xe8, 0x5c, 0x13, 0x5c, 0x5b, 0x23, 0x83, 0x6, 0x31, 0x82, 0x2, 0x8d, 0x30, 0x82, 0x2, 0x89, 0x2, 0x1, 0x1, 0x30, 0x6d, 0x30, 0x57, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x2d, 0x30, 0x2b, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x24, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x41, 0x54, 0x4c, 0x20, 0x2d, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x2d, 0x20, 0x47, 0x32, 0x2, 0x12, 0x11, 0x21, 0x12, 0x42, 0x28, 0x72, 0xda, 0x4f, 0xa5, 0x6c, 0x1, 0x3b, 0x29, 0x7c, 0xaf, 0x65, 0xf1, 0xfb, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2, 0x1, 0x5, 0x0, 0xa0, 0x81, 0xf2, 0x30, 0x1a, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x3, 0x31, 0xd, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1, 0x4, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x4, 0x31, 0x22, 0x4, 0x20, 0xc5, 0x8f, 0x24, 0x84, 0xc4, 0x8c, 0xca, 0x82, 0x2b, 0xd3, 0x26, 0xf5, 0xef, 0xf6, 0xa0, 0x3f, 0x15, 0xef, 0xd8, 0x23, 0xd7, 0x74, 0xf9, 0x4d, 0x2, 0x48, 0x59, 0x9f, 0x73, 0x2, 0xe5, 0xe0, 0x30, 0x81, 0xa2, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x2, 0xc, 0x31, 0x81, 0x92, 0x30, 0x81, 0x8f, 0x30, 0x81, 0x8c, 0x30, 0x81, 0x89, 0x4, 0x14, 0x97, 0xc4, 0xb7, 0x95, 0x36, 0x8d, 0xb5, 0xae, 0x1a, 0x6, 0xc5, 0x44, 0x83, 0x80, 0xaa, 0xeb, 0x1a, 0xc0, 0x9b, 0x8b, 0x30, 0x71, 0x30, 0x5b, 0xa4, 0x59, 0x30, 0x57, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61, 0x31, 0x2d, 0x30, 0x2b, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x24, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x41, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x41, 0x54, 0x4c, 0x20, 0x2d, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x2d, 0x20, 0x47, 0x32, 0x2, 0x12, 0x11, 0x21, 0x12, 0x42, 0x28, 0x72, 0xda, 0x4f, 0xa5, 0x6c, 0x1, 0x3b, 0x29, 0x7c, 0xaf, 0x65, 0xf1, 0xfb, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x4, 0x82, 0x1, 0x0, 0xb6, 0x24, 0xfe, 0xeb, 0x50, 0x44, 0xb4, 0xc, 0xf1, 0xc9, 0x55, 0x46, 0x90, 0xa4, 0xad, 0x91, 0xfa, 0xb0, 0x79, 0x60, 0x54, 0xb2, 0x57, 0x31, 0xcf, 0xee, 0xae, 0xf7, 0x5d, 0xb0, 0xa9, 0x8e, 0x4d, 0x58, 0x25, 0x22, 0x6d, 0x82, 0x2a, 0xe2, 0xf6, 0x72, 0xde, 0x96, 0xc2, 0xdc, 0x42, 0x62, 0x1a, 0x0, 0x69, 0x5b, 0x6e, 0x51, 0x5b, 0x59, 0x39, 0xba, 0x3f, 0xad, 0xa2, 0xfd, 0x7f, 0x87, 0x28, 0x6e, 0x7f, 0x5b, 0xfd, 0x26, 0x98, 0x37, 0xaa, 0xfd, 0x98, 0x47, 0x8c, 0xb1, 0xab, 0x8c, 0xd2, 0xe0, 0x49, 0x48, 0x68, 0x35, 0x86, 0xbf, 0x7, 0x22, 0x7c, 0xf4, 0x56, 0xe8, 0x71, 0xe5, 0x8a, 0x19, 0x68, 0x99, 0xb7, 0x4a, 0x10, 0xb8, 0x5a, 0x49, 0xee, 0x10, 0x45, 0x76, 0xd1, 0x7f, 0xaa, 0x29, 0x6d, 0x4a, 0x31, 0x6d, 0x3e, 0xfb, 0xf, 0xfc, 0x5e, 0x12, 0xc6, 0x1b, 0x57, 0x16, 0xd8, 0x72, 0xbd, 0x6e, 0xdd, 0x5e, 0xd, 0xaf, 0x7f, 0x35, 0x4b, 0xf0, 0xaa, 0x14, 0xf9, 0x37, 0xc, 0x39, 0x50, 0x81, 0x46, 0xdc, 0x1f, 0x81, 0xb, 0xc0, 0x15, 0xef, 0xbe, 0xa0, 0x48, 0x84, 0x19, 0xb0, 0xf7, 0x96, 0xd4, 0x8a, 0x22, 0x5d, 0x9, 0xc2, 0x56, 0xc5, 0x42, 0xe6, 0x93, 0xa, 0xd, 0x88, 0x21, 0x87, 0x4f, 0x48, 0x7b, 0xd5, 0x76, 0xb9, 0xfa, 0x2a, 0x37, 0x94, 0xd, 0xa7, 0x2e, 0x23, 0x8c, 0xd1, 0xd7, 0xef, 0x7c, 0x22, 0xab, 0x22, 0x94, 0x6f, 0x2, 0x48, 0xa, 0xc0, 0x8d, 0x4f, 0xfc, 0x1d, 0x6a, 0x11, 0x19, 0xb3, 0x96, 0xd7, 0x50, 0x8a, 0xe, 0x3f, 0xe2, 0xfd, 0x78, 0x5d, 0xa7, 0x3a, 0xa5, 0x57, 0xab, 0x4f, 0x62, 0xf4, 0x2b, 0x3f, 0xb1, 0xd8, 0x77, 0xf2, 0xfe, 0x67, 0x95, 0xfd, 0x6, 0x10, 0x7, 0x4b, 0xd9, 0x1, 0x2a, 0xb4, 0x2b, 0x6f, 0xd3, 0x9c}

var respRejection = []byte{0x30, 0x35, 0x30, 0x33, 0x2, 0x1, 0x2, 0x30, 0x28, 0xc, 0x26, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x75, 0x72, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x3, 0x4, 0x6, 0x0, 0x0, 0x40}

type testData struct {
	name          string
	request       []byte
	response      []byte
	certificates  bool
	nonce         *big.Int
	time          time.Time
	accuracy      time.Duration
	hashedMessage []byte
	hashAlgorithm crypto.Hash
}

var testCases = []testData{
	{
		name:          "Including nonce",
		request:       reqNonce,
		response:      respNonce,
		certificates:  true,
		nonce:         big.NewInt(0).SetBytes([]byte{0x9, 0x2e, 0xf1, 0x9f, 0xfb, 0x5d, 0x2a, 0xe8}),
		time:          time.Date(2017, 4, 19, 6, 29, 53, 0, time.UTC),
		accuracy:      time.Second,
		hashedMessage: hashedMessage,
		hashAlgorithm: crypto.SHA256,
	},
	{
		name:          "Containing no nonce",
		request:       reqNoNonce,
		response:      respNoNonce,
		certificates:  true,
		nonce:         nil,
		time:          time.Date(2017, 4, 19, 6, 28, 13, 0, time.UTC),
		accuracy:      time.Second,
		hashedMessage: hashedMessage,
		hashAlgorithm: crypto.SHA256,
	},
	{
		name:          "Containing no certificates",
		request:       reqNoCert,
		response:      respNoCert,
		certificates:  false,
		nonce:         big.NewInt(0).SetBytes([]byte{0xb1, 0xfc, 0x81, 0xde, 0xc9, 0x57, 0x49, 0xd9}),
		time:          time.Date(2017, 4, 19, 6, 32, 43, 0, time.UTC),
		accuracy:      time.Second,
		hashedMessage: hashedMessage,
		hashAlgorithm: crypto.SHA256,
	},
}

// Send the timestamp request to our timestamp server and save the response
//  $ curl --globoff -s -S -H Content-Type:application/timestamp-query -H Host:${HOST} --data-binary @request-sha256.tsq -o ts-output.tsr ${URL}

func TestParseRequest(t *testing.T) {
	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			if tc.request == nil {
				return
			}

			req, err := ParseRequest(tc.request)
			if err != nil {
				t.Errorf("failed to parse request: %s", err.Error())
				return
			}

			if !bytes.Equal(req.HashedMessage, tc.hashedMessage) {
				t.Errorf("req.HashedMessage: got %x, want %x", req.HashedMessage, tc.hashedMessage)
			}

			if (tc.nonce != nil && tc.nonce.CmpAbs(req.Nonce) != 0) || (req.Nonce != nil && tc.nonce == nil) {
				t.Errorf("req.Nonce: got %v, want %v", req.Nonce, tc.nonce)
			}

			if req.HashAlgorithm != tc.hashAlgorithm {
				t.Errorf("req.HashAlgorithm: got %v, want %v", req.HashAlgorithm, tc.hashAlgorithm)
			}

			if req.Certificates != tc.certificates {
				t.Errorf("req.Certificates: got %v, want %v", req.Certificates, tc.certificates)
			}
		})
	}
}

func TestParseResponse(t *testing.T) {
	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			if tc.response == nil {
				return
			}

			resp, err := ParseResponse(tc.response)
			if err != nil {
				t.Errorf("failed to parse response: %s", err.Error())
				return
			}

			if !bytes.Equal(resp.HashedMessage, tc.hashedMessage) {
				t.Errorf("resp.HashedMessage: got %x, want %x", resp.HashedMessage, tc.hashedMessage)
			}

			if resp.HashAlgorithm != tc.hashAlgorithm {
				t.Errorf("resp.HashAlgorithm: got %v, want %v", resp.HashAlgorithm, tc.hashAlgorithm)
			}

			if !resp.Time.Equal(tc.time) {
				t.Errorf("resp.Time: got %v, want %v", resp.Time, tc.time)
			}

			if resp.Accuracy != tc.accuracy {
				t.Errorf("resp.Accuracy: got %v, want %v", resp.Accuracy, tc.accuracy)
			}

			if tc.certificates && len(resp.Certificates) == 0 {
				t.Errorf("resp.Certificates: got %v, want %v", len(resp.Certificates), tc.certificates)
			}

			/*
				if !(resp.Nonce == nil && tc.nonce == nil && resp.Nonce.Cmp(tc.nonce) != 0 {
					t.Errorf("resp.Nonce: got %v, want %v", resp.Nonce, tc.nonce)
				}
			*/
		})
	}
}

func TestParse(t *testing.T) {
	ts, err := Parse(timeStampToken)
	if err != nil {
		t.Errorf("failed to parse timeStampToken: %s", err.Error())
	}

	hashedMessage := []byte{0x2d, 0xa8, 0x2c, 0x15, 0x72, 0xff, 0x2, 0x5, 0x24, 0xe0, 0x50, 0x69, 0x21, 0x99, 0x75, 0xc9, 0xba, 0x90, 0x72, 0x3f, 0x1a, 0x4d, 0xd5, 0xb9, 0x72, 0x2a, 0xee, 0x8e, 0xc5, 0x47, 0xa2, 0xff}
	if !bytes.Equal(ts.HashedMessage, hashedMessage) {
		t.Errorf("ts.HashedMessage: got %x, want %x", ts.HashedMessage, hashedMessage)
	}

	tsTime := time.Date(2017, 2, 1, 15, 39, 50, 0, time.UTC)
	if !ts.Time.Equal(tsTime) {
		t.Errorf("ts.Time: got %v, want %v", ts.Time, tsTime)
	}

	if ts.Accuracy != time.Second {
		t.Errorf("ts.Accuracy: got %v, want %v", ts.Accuracy, time.Second)
	}
}

func TestParseResponseRejection(t *testing.T) {
	_, err := ParseResponse(respRejection)
	if err == nil {
		t.Errorf("failed to parse response with rejection: %s", err.Error())
	}
	expected := "the request is rejected: Error during serial number generation. (the additional information requested could not be understood or is not available)"
	if err.Error() != expected {
		t.Errorf("unexpected error message:\n\t%s\nexpected:\n\t%s\n", err.Error(), expected)
	}
}

func TestCreateErrorResponse(t *testing.T) {
	resp, err := CreateErrorResponse(Rejection, TimeNotAvailable)
	if err != nil {
		t.Errorf("failed to create error: %s", err.Error())
	}

	expected := "the request is rejected:  (the TSA's time source is not available)"
	_, err = ParseResponse(resp)
	if err.Error() != expected {
		t.Errorf("unexpected error message:\n\t%s\nexpected:\n\t%s\n", err.Error(), expected)
	}
}

func TestMarshalRequest(t *testing.T) {
	req, err := ParseRequest(reqNoNonce)
	if err != nil {
		t.Fatal(err)
	}

	reqByes, err := req.Marshal()
	if err != nil {
		t.Fatal(err)
	}

	if !bytes.Equal(reqByes, reqNoNonce) {
		t.Error("Marshalled response bytes are not the same as parsed")
	}
}

func TestCreateRequest(t *testing.T) {
	var testHashes = []crypto.Hash{crypto.SHA1, crypto.SHA256, crypto.SHA384, crypto.SHA512}

	nonce := big.NewInt(0)
	nonce = nonce.SetBytes([]byte{0x1, 0x2, 0x3})

	for _, th := range testHashes {
		t.Run(fmt.Sprintf("%d", th), func(t *testing.T) {
			msg := "Content to by timestamped"

			h := th.New()
			_, err := h.Write([]byte(msg))
			if err != nil {
				t.Fatal(err)
			}
			hashedMsg := h.Sum(nil)

			req, err := CreateRequest(strings.NewReader(msg), &RequestOptions{
				Hash:         th,
				Nonce:        nonce,
				TSAPolicyOID: asn1.ObjectIdentifier{2, 5, 6, 7},
				Certificates: true,
			})
			if err != nil {
				t.Fatal(err)
			}

			if len(req) == 0 {
				t.Error("request contains no bytes")
			}

			reqCheck, err := ParseRequest(req)
			if err != nil {
				t.Fatal(err)
			}

			if !bytes.Equal(reqCheck.HashedMessage, hashedMsg) {
				t.Errorf("reqCheck.HashedMessage: got %x, want %x", reqCheck.HashedMessage, hashedMsg)
			}

			if reqCheck.Nonce.Cmp(nonce) != 0 {
				t.Errorf("reqCheck.Nonce: got %x, want %x", reqCheck.Nonce, nonce)
			}

			if reqCheck.Certificates != true {
				t.Errorf("reqCheck.Certificates: got %t, want %t", reqCheck.Certificates, true)
			}

			if !reqCheck.TSAPolicyOID.Equal(asn1.ObjectIdentifier{2, 5, 6, 7}) {
				t.Errorf("reqCheck.TSAPolicyOID: got %x, want %x", reqCheck.TSAPolicyOID, asn1.ObjectIdentifier{2, 5, 6, 7})
			}
		})
	}
}

func BenchmarkCreateRequest(b *testing.B) {
	reader := strings.NewReader("Content to be time-stamped")

	for n := 0; n < b.N; n++ {
		_, _ = CreateRequest(reader, nil)
	}
}

func BenchmarkParseRequest(b *testing.B) {
	for n := 0; n < b.N; n++ {
		_, _ = ParseRequest(reqNonce)
	}
}

func BenchmarkParseResponse(b *testing.B) {
	for n := 0; n < b.N; n++ {
		_, _ = ParseResponse(respNonce)
	}
}

// ExampleCreateRequest demonstrates how to create a new time-stamping request
// for an io.Reader.
func ExampleCreateRequest() {
	_, err := CreateRequest(strings.NewReader("Content to be time-stamped"), nil)
	if err != nil {
		panic(err)
	}
}

// ExampleCreateRequest_customHashingAlgorithm demonstrates how to create a new
// time-stamping request with options
func ExampleCreateRequest_customHashingAlgorithm() {
	_, err := CreateRequest(
		strings.NewReader("Content to be time-stamped"),
		&RequestOptions{
			Hash: crypto.SHA512,
		})
	if err != nil {
		panic(err)
	}
}

// ExampleParseRequest demonstrates how to parse a raw der time-stamping request
func ExampleParseRequest() {
	// CreateRequest returns the request in der bytes
	createdRequest, err := CreateRequest(strings.NewReader("Content to be time-stamped"), nil)
	if err != nil {
		panic(err)
	}

	// ParseRequest parses a request in der bytes
	parsedRequest, err := ParseRequest(createdRequest)
	if err != nil {
		panic(err)
	}

	fmt.Printf("%x\n", parsedRequest.HashedMessage)
	// Output: 51a3620a3b62ffaff41a434e932223b31bc69e86490c365fa1186033904f1132
}

func TestCreateResponseWithNoTSACertificate(t *testing.T) {
	tsakey := getTSARSAKey()
	tsaCert := getTSACert()

	h := sha256.New()
	_, err := h.Write([]byte("Hello World"))
	if err != nil {
		t.Fatal(err)
	}

	genTime := time.Now().UTC()

	nonce := big.NewInt(0)
	nonce = nonce.SetBytes([]byte{0x1, 0x2, 0x3})

	duration, _ := time.ParseDuration("1s")

	timestamp := Timestamp{
		HashAlgorithm:     crypto.SHA256,
		HashedMessage:     h.Sum(nil),
		Time:              genTime,
		Nonce:             nonce,
		Policy:            asn1.ObjectIdentifier{2, 4, 5, 6},
		Ordering:          true,
		Accuracy:          duration,
		Qualified:         false,
		AddTSACertificate: false,
	}
	timestampBytes, err := timestamp.CreateResponse(tsaCert, tsakey)
	if err != nil {
		t.Errorf("unable to generate time stamp response: %s", err.Error())
	}
	timestampRes, err := ParseResponse(timestampBytes)
	if err != nil {
		t.Fatalf("unable to parse time stamp response: %s", err.Error())
	}

	if timestampRes.HashAlgorithm.HashFunc() != crypto.SHA256 {
		t.Errorf("expected hash algorithm is SHA256")
	}
	if len(timestampRes.HashedMessage) != 32 {
		t.Errorf("got %d: expected: %d", len(timestampRes.HashedMessage), 32)
	}

	if timestampRes.Qualified {
		t.Errorf("got %t: expected: %t", timestampRes.Qualified, true)
	}

	if timestampRes.AddTSACertificate {
		t.Error("TSA certificate must not be included in timestamp response")
	}
}

func TestCreateResponseWithIncludeTSACertificate(t *testing.T) {
	tsakey := getTSARSAKey()
	tsaCert := getTSACert()

	h := sha256.New()
	_, err := h.Write([]byte("Hello World"))
	if err != nil {
		t.Fatal(err)
	}

	genTime := time.Now().UTC()

	nonce := big.NewInt(0)
	nonce = nonce.SetBytes([]byte{0x1, 0x2, 0x3})

	duration, _ := time.ParseDuration("1s")

	timestamp := Timestamp{
		HashAlgorithm:     crypto.SHA256,
		HashedMessage:     h.Sum(nil),
		Time:              genTime,
		Nonce:             nonce,
		Policy:            asn1.ObjectIdentifier{2, 4, 5, 6},
		Ordering:          true,
		Accuracy:          duration,
		Qualified:         true,
		AddTSACertificate: true,
	}
	timestampBytes, err := timestamp.CreateResponse(tsaCert, tsakey)
	if err != nil {
		t.Errorf("unable to generate time stamp response: %s", err.Error())
	}

	// To verify the reponse using OpenSSL
	// openssl ts -reply -in timestamp.tsr -text
	// _ = os.WriteFile("timestamp.tsr", timestampBytes, 0644)

	timestampRes, err := ParseResponse(timestampBytes)
	if err != nil {
		t.Errorf("unable to parse time stamp response: %s", err.Error())
	}

	if timestampRes.HashAlgorithm.HashFunc() != crypto.SHA256 {
		t.Errorf("expected hash algorithm is SHA256")
	}
	if len(timestampRes.HashedMessage) != 32 {
		t.Errorf("got %d: expected: %d", len(timestampRes.HashedMessage), 32)
	}

	if timestampRes.Accuracy != duration {
		t.Errorf("got accuracy %s: expected: %s", timestampRes.Accuracy, duration)
	}

	if !timestampRes.Qualified {
		t.Errorf("got %t: expected: %t", timestampRes.Qualified, true)
	}

	if !timestampRes.AddTSACertificate {
		t.Error("TSA certificate must be included in timestamp response")
	}
}

// Sign with TSU and do not embed certificates
func TestSignWithTSUNoCertificate(t *testing.T) {
	tsukey := getTSURSAKey()
	tsuCert := getTSUCert()

	h := sha256.New()
	_, err := h.Write([]byte("Hello World"))
	if err != nil {
		t.Fatal(err)
	}

	genTime := time.Now().UTC()

	nonce := big.NewInt(0)
	nonce = nonce.SetBytes([]byte{0x1, 0x2, 0x3})

	duration, _ := time.ParseDuration("1s")

	timestamp := Timestamp{
		HashAlgorithm:     crypto.SHA256,
		HashedMessage:     h.Sum(nil),
		Time:              genTime,
		Nonce:             nonce,
		Policy:            asn1.ObjectIdentifier{2, 4, 5, 6},
		Ordering:          true,
		Accuracy:          duration,
		Qualified:         false,
		AddTSACertificate: false,
	}
	timestampBytes, err := timestamp.CreateResponse(tsuCert, tsukey)
	if err != nil {
		t.Errorf("unable to generate time stamp response: %s", err.Error())
	}
	timestampRes, err := ParseResponse(timestampBytes)
	if err != nil {
		t.Fatalf("unable to parse time stamp response: %s", err.Error())
	}

	if timestampRes.HashAlgorithm.HashFunc() != crypto.SHA256 {
		t.Errorf("expected hash algorithm is SHA256")
	}
	if len(timestampRes.HashedMessage) != 32 {
		t.Errorf("got %d: expected: %d", len(timestampRes.HashedMessage), 32)
	}

	if timestampRes.Qualified {
		t.Errorf("got %t: expected: %t", timestampRes.Qualified, true)
	}

	if timestampRes.AddTSACertificate {
		t.Error("TSU certificate must not be included in timestamp response")
	}
}

// Sign with TSU and only embed TSU certificate
func TestSignWithTSUEmbedTSUCertificate(t *testing.T) {
	tsukey := getTSURSAKey()
	tsuCert := getTSUCert()

	h := sha256.New()
	_, err := h.Write([]byte("Hello World"))
	if err != nil {
		t.Fatal(err)
	}

	genTime := time.Now().UTC()

	nonce := big.NewInt(0)
	nonce = nonce.SetBytes([]byte{0x1, 0x2, 0x3})

	duration, _ := time.ParseDuration("1s")

	timestamp := Timestamp{
		HashAlgorithm:     crypto.SHA256,
		HashedMessage:     h.Sum(nil),
		Time:              genTime,
		Nonce:             nonce,
		Policy:            asn1.ObjectIdentifier{2, 4, 5, 6},
		Ordering:          true,
		Accuracy:          duration,
		Qualified:         true,
		AddTSACertificate: true,
	}
	timestampBytes, err := timestamp.CreateResponse(tsuCert, tsukey)
	if err != nil {
		t.Errorf("unable to generate time stamp response: %s", err.Error())
	}

	timestampRes, err := ParseResponse(timestampBytes)
	if err != nil {
		t.Fatalf("unable to parse time stamp response: %s", err.Error())
	}

	if timestampRes.HashAlgorithm.HashFunc() != crypto.SHA256 {
		t.Errorf("expected hash algorithm is SHA256")
	}
	if len(timestampRes.HashedMessage) != 32 {
		t.Errorf("got %d: expected: %d", len(timestampRes.HashedMessage), 32)
	}

	if timestampRes.Accuracy != duration {
		t.Errorf("got accuracy %s: expected: %s", timestampRes.Accuracy, duration)
	}

	if !timestampRes.Qualified {
		t.Errorf("got %t: expected: %t", timestampRes.Qualified, true)
	}

	if !timestampRes.AddTSACertificate {
		t.Error("TSU certificate must be included in timestamp response")
	}
}

// Sign with TSU and include certificate chain
func TestSignWithTSUIncludeCertificateChain(t *testing.T) {
	tsuKey := getTSURSAKey()
	tsuCert := getTSUCert()
	tsaCert := getTSACert()

	h := sha256.New()
	_, err := h.Write([]byte("Hello World"))
	if err != nil {
		t.Fatal(err)
	}

	genTime := time.Now().UTC()

	nonce := big.NewInt(0)
	nonce = nonce.SetBytes([]byte{0x1, 0x2, 0x3})

	duration, _ := time.ParseDuration("1s")

	timestamp := Timestamp{
		HashAlgorithm:     crypto.SHA256,
		HashedMessage:     h.Sum(nil),
		Time:              genTime,
		Nonce:             nonce,
		Policy:            asn1.ObjectIdentifier{2, 4, 5, 6},
		Ordering:          true,
		Accuracy:          duration,
		Qualified:         true,
		AddTSACertificate: true,
		Certificates:      []*x509.Certificate{tsaCert}, // add parent certificate
	}
	timestampBytes, err := timestamp.CreateResponse(tsuCert, tsuKey)
	if err != nil {
		t.Fatalf("unable to generate time stamp response: %s", err.Error())
	}

	timestampRes, err := ParseResponse(timestampBytes)
	if err != nil {
		t.Fatalf("unable to parse time stamp response: %s", err.Error())
	}

	if timestampRes.HashAlgorithm.HashFunc() != crypto.SHA256 {
		t.Errorf("expected hash algorithm is SHA256")
	}
	if len(timestampRes.HashedMessage) != 32 {
		t.Errorf("got %d: expected: %d", len(timestampRes.HashedMessage), 32)
	}

	if timestampRes.Accuracy != duration {
		t.Errorf("got accuracy %s: expected: %s", timestampRes.Accuracy, duration)
	}

	if !timestampRes.Qualified {
		t.Errorf("got %t: expected: %t", timestampRes.Qualified, true)
	}

	if !timestampRes.AddTSACertificate {
		t.Error("TSA certificate must be included in timestamp response")
	}

	if !timestamp.Certificates[0].Equal(tsaCert) {
		t.Errorf("got certificate %X: expected: %X", timestamp.Certificates[0].SubjectKeyId, tsaCert.SubjectKeyId)
	}
}

func getTSURSAKey() *rsa.PrivateKey {
	tsuRSAKeyPEM := `
-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEA7FzZ2Uff6dPKEgJNgUi3EL21y3CzY3zGE3Wi2pTM1oGAMetP
ecEdFnMM2x76lnGuBk0VW04TdOMymXtW6L7X75Yr73/88b4Ycmrg2Xh/HqiCrtQZ
PrCwOn3V8UFulySqJItBUZ1ih6a0A1NNX6lMmQTbbuxLlyP4iz1HveFmOa/gkLh9
Y1I9yIRTb4Mlm6yISLQkDZn2YMXqlZUv4tyNUmFlSYrMu6lsIwBn2eaBKzQYPZM0
ExuTDD+d7RnrDr1tcfd5cNOq7XDNP2nIDE4rZSwRhOerDKPL9jqd60FKH40oX51/
WssiO1zVmi43uX8gr7iaJrzpkVNDN9m41zLheQIDAQABAoIBAQDA21odghnfjqGY
ZCydSpmknUaSgpi8mnh8NEX3F+azN+ND1/53F+0F/kYFHJfW3VbjaU39vA0AGMmW
lh7ptZ43rU6YEtRu427LHQ3uI/WFLHXE9ObMUhrY/wfr3DnCNXZmbwGS+FoG2SyU
cgn1/guz51Ssgz2CSyVnZ078Tce9VISolxGnxZVhEVcssFazEWLaOe/8t2rMg4e+
RRGiaTcIkTdNTFgk15JrkSRyvU/538vVwNJ67hAwlA011l3XDpVybwEQQ46bG09J
uSNOEm4XMNRxjPS6A2Fd+jtSiP4iYEwVoJU+/P7IRdoVvBjwcXnl7tX9i7o8xwhr
es+NzS8BAoGBAOxxSLk8QVYGoHn+H/I+SHBY8ipY/3mwCzH3shodfAejZ+eoMVI+
cWJl67QKhsFzlAfRj1YF92h5Hx8oXVR7te7KcjXVyStPlAUigls253GaczSnXjA0
0WxpFthES8TSbSfD1VTbU1JB6evUe8KkemEw+9vbl573U3FgEeu67LnxAoGBAP/p
4HNC2JTFGox7IJoEvO2qeJfxtlyJ5vb/RwlJu/kn7mAFzzjIE1xYFDbImKN8OYA4
BWXyZ+5WP+HZOBW16HXoYybT02ufFC3lp9p8xOSBs2DQKfrhqgJTOVjq7b4ovSj5
3DOnR1YojbiDHQpK2IVIj1Iz0DiIFqPhtB7CzdgJAoGANC3f7bkldhWqTqHNbQlf
tSN79eqEHtfB8LoIHQlKuOjP4mjU0aCkJyH0/VuhV4npLjyKFGLmsbChNKAU0LMo
eFVHFShj5+H8+ZEfEYAxXXnHWORiveK6IOGkP//6dKo3mqH2L27jmXCgbgILee4Q
b+h+fIuej19nk8quycYLvhECgYAwoJsyq6AF3NInoXnXalEQBBV4IcjaGqYVhvpT
jHw4YtsLye7PRk1PfbkRk9pVLlSqxXpZHc+b3S20V5ctoOw0A11b0mJZD9hAxGO5
w32SQgb4vXVMo7avTGsYN0PHn2waLigmdIG8oGYVimxpOUGdSeVZ5FiLdWh/6XJV
agS9KQKBgGn6Gfp9/67F0zgJ5gh5DsuY6At9VcGnSPmKaO//ME05KMazF6XW2gWs
/VmHEjBRBqXPylh3xmr5DMm95OeQm3QUsNf75aPFnkukRmgmeVIfICjx0twzls4O
vwxcYS6/uRJ1O1K0U2KZgqY9HGSg4Mm4Zs8mAe86evHEGX7g7N6y
-----END RSA PRIVATE KEY-----
`

	tsuKeyPEMBlock, _ := pem.Decode([]byte(tsuRSAKeyPEM))
	pvtKey, _ := x509.ParsePKCS1PrivateKey(tsuKeyPEMBlock.Bytes)

	return pvtKey
}
func getTSARSAKey() *rsa.PrivateKey {
	tsaRSAKeyPEM := `
-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEApeencH+4Wo3Ry65t2/FdZkHLyQcizv8Xu+4NTEGF502fPV2y
May4/ZU+GXeVTPhwfJuFj1D8Id6skgZ1DlAz+cpIqQQoaUuVM6M5MMJ6Ycf27KAs
knQiEMI7BcyJBni1c/aspLMd3AwPn/4XVweX+KL8FtbNouakKOvKT9MH23hUqJuY
aKxyxRABtuRYaq3PrAvR55gb/f/zLHvPh57vALi4J3WSIePXNpNzpOpZIj+J+UMQ
NQVWVPzRW7Wf057o9cvSl/P/eChKWIeMAsYE5+7Rybj7MnRi5XqDFDCdPLmHaT6/
ZcE6ijHSjoETi/Ut9BMOyIUqpQAs0uZH39FFwwIDAQABAoIBADuncUh9VD+TUQWJ
Ac2dGzVioTD2lOiTRuh3L2blBI3oFkMNhr5f2eCsojisDA4yIthbX4np188h7zFO
ixaLdjTyLHBBo3pBCDQaE71ZoIG6UipBaeV7Rqh5/pkWM4sVKkG5R9is4ya1W4Tu
61uKynVHvZdEw4o4nnxsVEGhouih5q/fmETi7XTCYSCe4gljVDtRpvFQBOrrhye/
BT38SvrXQR2WmgLLpfo+1VR5zcm9bXJXrkOKYNXWDxl9kpY+hwXD0IhTXl4GkqEe
8CP4WFHtX5WA4s9qLATp/zT7fme2Ojh+NkIdU0FMI9lf4pNX+URxii+hn15vrtCi
UxaSVtECgYEA0FobH8XOw7SWjJRs9wfLoF/Wl3s4ET9neJwx047Xlop8QAwHYzo7
CiEH+aodgr/UC8KM62+3y4pZgn3Bmt3/p/WyKOsfG3TZXqvuSGqTXO9sn3T1Z552
jVT/1/3qapHODL4ct52FHxrr243Jp2vfeMciU0tLdsx5FIgRCScqm0sCgYEAy9h/
qnDAC1fI4eEDYgj+kIUDyQegeKbi79U3aF5QjYSgvYm1pev/Zac8+x9X/zQupObB
FmgbtPYrXTY5J38qG/ELjDu7aHfXqgHcVTda0MsGsaoSCmaJ3y19ewxsmK9pFaEl
BUTmFd2hywK34RG00dyYcrvmP6M4OP/Do1+WPGkCgYEAv9lYhIcl/rr4rXW2aDk7
XO8ir9V8KRWS91IL51vuU+YsxuTMoKfr2UXVDCWCivSMElAQZnI2cStxhGC7txiX
4lawuFDYEfYkebIi9Xd9PeQQxztxBPq6+yS7eG2MPpkHfGBKHSDkhWHKsB39Azan
TZU/nCcG09sv2qH33c+8wcUCgYEAli3TqKNWqUSsZ9WZ43ES8zA8ILAwxpLVILKq
Foddu1VaAyngnPQofiDe6XgnIYq1TqH+4V4kA4dVXV/kbbffMyS8SD19jbK1PbgP
Nu0ISEk7jkro7aarrrPZ/XyiyT56IghNuPsQtE1LtMA07mlYGUD3Q5gxQvMiKcQs
w0FZ8vkCgYA7wuwLs7d9LJ4KqMNmOe0eRvIxp+Y8psxykMd1wz3PjdPz30U03xe2
o40r2ZNTK/OGYPmAOcwma7SjenBQve19eVUaECUVREmbvaJqVzz0uSrfqXrUVIiJ
YyOfhPUI5XhkyUlunO5pSAd0CtRv7NVW1wKDjMbJvgV0MlbVvGraAg==
-----END RSA PRIVATE KEY-----
`

	tsaKeyPEMBlock, _ := pem.Decode([]byte(tsaRSAKeyPEM))
	pvtKey, _ := x509.ParsePKCS1PrivateKey(tsaKeyPEMBlock.Bytes)

	return pvtKey
}
func getTSUCert() *x509.Certificate {
	tsaCertPEM := `
-----BEGIN CERTIFICATE-----
MIIDrTCCApWgAwIBAgITHtUKVw2T5tfI4jk7zfJvKj39MDANBgkqhkiG9w0BAQsF
ADBdMQswCQYDVQQGEwJBVTETMBEGA1UECAwKU29tZS1TdGF0ZTEhMB8GA1UECgwY
SW50ZXJuZXQgV2lkZ2l0cyBQdHkgTHRkMRYwFAYDVQQDDA1UZXN0IFJTQSBDZXJ0
MB4XDTIyMTAxODA2MzYxNloXDTQyMTAxODA2MzYxNlowYzELMAkGA1UEBhMCQVUx
EzARBgNVBAgTClNvbWUtU3RhdGUxITAfBgNVBAsTGEludGVybmV0IFdpZGdpdHMg
UHR5IEx0ZDEcMBoGA1UEAxMTVGVzdCBSU0EgQ2hpbGQgQ2VydDCCASIwDQYJKoZI
hvcNAQEBBQADggEPADCCAQoCggEBAOxc2dlH3+nTyhICTYFItxC9tctws2N8xhN1
otqUzNaBgDHrT3nBHRZzDNse+pZxrgZNFVtOE3TjMpl7Vui+1++WK+9//PG+GHJq
4Nl4fx6ogq7UGT6wsDp91fFBbpckqiSLQVGdYoemtANTTV+pTJkE227sS5cj+Is9
R73hZjmv4JC4fWNSPciEU2+DJZusiEi0JA2Z9mDF6pWVL+LcjVJhZUmKzLupbCMA
Z9nmgSs0GD2TNBMbkww/ne0Z6w69bXH3eXDTqu1wzT9pyAxOK2UsEYTnqwyjy/Y6
netBSh+NKF+df1rLIjtc1ZouN7l/IK+4mia86ZFTQzfZuNcy4XkCAwEAAaNgMF4w
DgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYEFCFNxjsFpw4I
5p+o62/J17zDEa8FMB8GA1UdIwQYMBaAFIjUWTfL8OkBDBFeGipGGN5BGygKMA0G
CSqGSIb3DQEBCwUAA4IBAQCYdgikay21S2fYz+mk3dnhUPLnXI9Gg6U9ox2js+Yf
dYOMXX5RlG2HKOEnVI7mOPEaEiSAuvE8x6mQ52xVOw4FWjPO7S5pCBO8YPyyZCer
5bOFhz/zbv7xpZISvTTPfOXqSM1MvkcX9kfPDRagGfB6viVQwhWMQK4Sd0d5wSf2
FHsUAuq+EauBDgTe4bdCMS/AQSF/xhFL1KUwvkI8HjDE+JGu7hexWtlxwD/dnvpD
6ZrUnRLuSF3jebNb2DwIXb05ub+YJoBx/0I/zYe3vWISDtO/onNAZz9hnmAzL32f
69EW22PQaOUDtTbfT2PD09uTUfmA+zZggjPgaWjbw+gf
-----END CERTIFICATE-----
`
	certPEMBlock, _ := pem.Decode([]byte(tsaCertPEM))
	tsaCert, _ := x509.ParseCertificate(certPEMBlock.Bytes)

	return tsaCert
}
func getTSACert() *x509.Certificate {
	tsaCertPEM := `
-----BEGIN CERTIFICATE-----
MIIDmzCCAoOgAwIBAgIUTrgB1p7WpwYXjwGs/uwfKJt4cFcwDQYJKoZIhvcNAQEL
BQAwXTELMAkGA1UEBhMCQVUxEzARBgNVBAgMClNvbWUtU3RhdGUxITAfBgNVBAoM
GEludGVybmV0IFdpZGdpdHMgUHR5IEx0ZDEWMBQGA1UEAwwNVGVzdCBSU0EgQ2Vy
dDAeFw0yMDAzMDQyMjA4MDVaFw00MDAyMjgyMjA4MDVaMF0xCzAJBgNVBAYTAkFV
MRMwEQYDVQQIDApTb21lLVN0YXRlMSEwHwYDVQQKDBhJbnRlcm5ldCBXaWRnaXRz
IFB0eSBMdGQxFjAUBgNVBAMMDVRlc3QgUlNBIENlcnQwggEiMA0GCSqGSIb3DQEB
AQUAA4IBDwAwggEKAoIBAQCl56dwf7hajdHLrm3b8V1mQcvJByLO/xe77g1MQYXn
TZ89XbIxrLj9lT4Zd5VM+HB8m4WPUPwh3qySBnUOUDP5ykipBChpS5Uzozkwwnph
x/bsoCySdCIQwjsFzIkGeLVz9qyksx3cDA+f/hdXB5f4ovwW1s2i5qQo68pP0wfb
eFSom5horHLFEAG25Fhqrc+sC9HnmBv9//Mse8+Hnu8AuLgndZIh49c2k3Ok6lki
P4n5QxA1BVZU/NFbtZ/Tnuj1y9KX8/94KEpYh4wCxgTn7tHJuPsydGLleoMUMJ08
uYdpPr9lwTqKMdKOgROL9S30Ew7IhSqlACzS5kff0UXDAgMBAAGjUzBRMB0GA1Ud
DgQWBBSI1Fk3y/DpAQwRXhoqRhjeQRsoCjAfBgNVHSMEGDAWgBSI1Fk3y/DpAQwR
XhoqRhjeQRsoCjAPBgNVHRMBAf8EBTADAQH/MA0GCSqGSIb3DQEBCwUAA4IBAQAP
+jK6M/zPFrO/hrXOXlfEntbKwxFWoil/BRVMkgMp6JO44wn9QS+oRIVKcMToTPe5
XaU4D8YgHPFiyhaTOQ95RDVZuy5VPf1li1oujPHXP6Y9Ps5RF9AKtLYdJa8ZBmRx
Cg3mHV4f6VJWziWz3s5n6DVQ5DDrSkQ0dIRs5Tu9W4+aHJUMwdkSP0klvBnlzPhq
kl++ygWDU5bJMbwD53eGieJyo5wL0SR08ijiGxCTmYOUuPl/C62MTPJU+oR8qRd3
I/rCr/gywfHmAbgupBo9ikC9rrYD5maaC59xr4NjjI1vSeS3nrO9qmd9KnGD98P8
wA4N9tN/F776b2RG2RZD
-----END CERTIFICATE-----
`
	certPEMBlock, _ := pem.Decode([]byte(tsaCertPEM))
	tsaCert, _ := x509.ParseCertificate(certPEMBlock.Bytes)

	return tsaCert
}

func TestCreateResponseSHA384(t *testing.T) {
	tsakey := getTSARSAKey()
	tsaCert := getTSACert()

	h := sha256.New()
	_, err := h.Write([]byte("Hello World"))
	if err != nil {
		t.Fatal(err)
	}

	genTime := time.Now().UTC()

	nonce := big.NewInt(0)
	nonce = nonce.SetBytes([]byte{0x1, 0x2, 0x3})

	duration, _ := time.ParseDuration("1s")

	timestamp := Timestamp{
		HashAlgorithm:     crypto.SHA256,
		HashedMessage:     h.Sum(nil),
		Time:              genTime,
		Nonce:             nonce,
		Policy:            asn1.ObjectIdentifier{2, 4, 5, 6},
		Ordering:          true,
		Accuracy:          duration,
		Qualified:         false,
		AddTSACertificate: false,
	}
	timestampBytes, err := timestamp.CreateResponseWithOpts(tsaCert, tsakey, crypto.SHA384)
	if err != nil {
		t.Errorf("unable to generate time stamp response: %s", err.Error())
	}
	timestampRes, err := ParseResponse(timestampBytes)
	if err != nil {
		t.Fatalf("unable to parse time stamp response: %s", err.Error())
	}

	p7, err := pkcs7.Parse(timestampRes.RawToken)
	if err != nil {
		t.Fatal(err)
	}
	if got := p7.Signers[0].DigestAlgorithm; !got.Algorithm.Equal(pkcs7.OIDDigestAlgorithmSHA384) {
		t.Fatalf("unexpected digest algorithm: want SHA384(%s), got %s", pkcs7.OIDDigestAlgorithmSHA384, got.Algorithm.String())
	}
}
