## JerryScript debugger interface

JerryScript provides a remote debugger which allows debugging
JavaScript programs. The debugger has two main components:
a server which is part of the JerryScript binary and a
separate client application. Currently a Python-based debugger
client is available in the /jerry-debugger subdirectory.
This simple application demonstrates the communication protocol
between the client and server, and can be reused by integrated
development environments.

## Setting up the debugger server

The following arguments must be passed to `tools/build.py`:

`--jerry-debugger=on`

The transport layer of the communication protocol is pluggable.
At the moment, a WebSocket-based implementation is provided as a
JerryScript extension, which transmits messages over TCP/IP networks.
If necessary/implemented, any reliable stream or datagram based
protocol can be used for transmitting debugger messages.

## Debugging JavaScript applications

The debugger client must be connected to the server before the
JavaScript application runs. On-the-fly attachment is supported
for more than one file, right after the engine initialization
(this feature is available with the python client). The debugging
information (e.g. line index of each possible breakpoint location)
is not preserved by JerryScript. The client is expected to be run
on a system with much more resources and it should be capable of
storing this information. JerryScript frees all debug information
after it is transmitted to the client to save memory.

The following argument makes JerryScript wait for a client
connection:

`--start-debug-server`

The following argument makes JerryScript wait for a client
source code:

`--debugger-wait-source`

It is also recommended to increase the log level to see
the *Waiting for client connection* message:

`--log-level 2`

The Python client can connect to the server by specifying its
IP address on the command line. The address can be localhost
if the server and the client are running on the same machine.

After the connection is established the execution can be
controlled by the debugger. The debugger always stops at
the first possible breakpoint location. The effect is the
same as using the `stop` command. This allows inserting
breakpoints right before the meaningful part of the execution
starts.

All available commands of the client can be queried by the
`help` command.

## Integrating debugger support into applications using JerryScript

When using the extension-provided WebSocket transport layer, the
debugger can be enabled by calling `jerryx_debugger_after_connect
(jerryx_debugger_tcp_create (debug_port) && jerryx_debugger_ws_create ())`
after the `jerry_init ()` function. It initializes the debugger and
blocks until a client connects.
(Custom transport layers may be implemented and initialized similarly.
Currently, `jerryx_debugger_rp_create ()` for raw packet transport layer and
`jerryx_debugger_serial_create (const char* config)` for serial protocol
are also available.)

The source name provided to `jerry_parse ()` is used by the client
to identify the source name of the source code. This source name
is usually a file name.

## JerryScript debugger C-API interface

The following section describes the debugger functions
available to the host application.

## JerryScript debugger types

## jerry_debugger_wait_for_source_callback_t

**Summary**

This callback function is called by
[jerry_debugger_wait_for_client_source](#jerry_debugger_wait_for_client_source)
when a source code is received successfully.

**Prototype**

```c
typedef jerry_value_t
(*jerry_debugger_wait_for_source_callback_t) (const jerry_char_t *source_name_p,
                                              size_t source_name_size,
                                              const jerry_char_t *source_p,
                                              size_t source_size,
                                              void *user_p);
```

- `source_name_p` - source (usually a file) name of the source code
- `source_name_size` - size of source name
- `source_p` - source code character data
- `source_size` - size of source code
- `user_p` - custom pointer passed to [jerry_debugger_wait_for_client_source](#jerry_debugger_wait_for_client_source)


## JerryScript debugger functions

### jerry_debugger_is_connected

**Summary**

Returns true if a remote debugger client is connected.

**Prototype**

```c
bool
jerry_debugger_is_connected (void);
```

**Example**

[doctest]: # (test="link")

```c
#include <stdio.h>

#include "jerryscript.h"
#include "jerryscript-ext/debugger.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);
  jerryx_debugger_after_connect (jerryx_debugger_tcp_create (5001)
                                 && jerryx_debugger_ws_create ());

  if (jerry_debugger_is_connected ())
  {
    printf ("A remote debugger client is connected.");
  }

  jerry_cleanup ();
}
```

### jerry_debugger_stop

**Summary**

Stops execution at the next available breakpoint if a remote
debugger client is connected and the engine is not waiting at
a breakpoint. The engine will stop regardless the breakpoint
is enabled or not.

**Prototype**

```c
void
jerry_debugger_stop (void)
```

**Example**

[doctest]: # (test="link")

```c
#include "jerryscript.h"
#include "jerryscript-ext/debugger.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);
  jerryx_debugger_after_connect (jerryx_debugger_tcp_create (5001)
                                 && jerryx_debugger_ws_create ());

  jerry_debugger_stop ();

  jerry_cleanup ();
}
```

**See also**

- [jerry_debugger_continue](#jerry_debugger_continue)

### jerry_debugger_continue

**Summary**

If the engine would stop at the next available breakpoint it
cancels this effect. The engine will still stop at enabled
breakpoints. This function effectively negates the effect of
[jerry_debugger_stop ()](#jerry_debugger_stop) calls or stop
requests issued by the debugger client.

**Prototype**

```c
void
jerry_debugger_continue (void)
```

**Example**

[doctest]: # (test="link")

```c
#include "jerryscript.h"
#include "jerryscript-ext/debugger.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);
  jerryx_debugger_after_connect (jerryx_debugger_tcp_create (5001)
                                 && jerryx_debugger_ws_create ());

  jerry_debugger_continue ();

  jerry_cleanup ();
}
```

**See also**

- [jerry_debugger_stop](#jerry_debugger_stop)

### jerry_debugger_stop_at_breakpoint

**Summary**

Enables or disables stopping at breakpoints. When stopping is
disabled all breakpoints are ignored including user enabled
breakpoints. This allows hidden execution of ECMAScript code.

**Prototype**

```c
void
jerry_debugger_stop_at_breakpoint (bool enable_stop_at_breakpoint)
```

- `enable_stop_at_breakpoint` - enable (=`true`) or disable (=`false`) stopping at breakpoints

**Example**

[doctest]: # (test="link")

```c
#include "jerryscript.h"
#include "jerryscript-ext/debugger.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);
  jerryx_debugger_after_connect (jerryx_debugger_tcp_create (5001)
                                 && jerryx_debugger_ws_create ());

  jerry_debugger_stop_at_breakpoint (true);

  // Protected execution of JavaScript code.
  const jerry_char_t script[] = "42";
  jerry_eval (script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);

  jerry_debugger_stop_at_breakpoint (false);

  jerry_cleanup ();
}
```

### jerry_debugger_wait_for_client_source

**Summary**

Asks the client to provide the next source code. The function
waits until the whole source code is received. As a reply the
the client may request a context reset or notify that no more
source is available. These notifications are passed back as the
return value of the function.

**Prototype**

```c
jerry_debugger_wait_for_source_status_t
jerry_debugger_wait_for_client_source (jerry_debugger_wait_for_source_callback_t callback_p,
                                       void *user_p,
                                       jerry_value_t *return_value)
```

**Example**

[doctest]: # (test="link")

```c
#include "jerryscript.h"
#include "jerryscript-ext/debugger.h"

/**
 * Runs the source code received by jerry_debugger_wait_for_client_source.
 */
static jerry_value_t
wait_for_source_callback (const jerry_char_t *source_name_p, /**< source name */
                          size_t source_name_size, /**< size of source name */
                          const jerry_char_t *source_p, /**< source code */
                          size_t source_size, /**< source code size */
                          void *user_p /**< user pointer */)
{
  (void) user_p;

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string ((const jerry_char_t *) source_name_p,
                                              (jerry_size_t) source_name_size,
                                              JERRY_ENCODING_UTF8);

  jerry_value_t ret_val = jerry_parse (source_p,
                                       source_size,
                                       &parse_options);
  jerry_value_free (parse_options.source_name);

  if (!jerry_value_is_exception (ret_val))
  {
    jerry_value_t func_val = ret_val;
    ret_val = jerry_run (func_val);
    jerry_value_free (func_val);
  }

  return ret_val;
} /* wait_for_source_callback */

int
main (void)
{
  jerry_debugger_wait_for_source_status_t receive_status;

  do
  {
    /* Create a new JerryScript instance when a context reset is
     * received. Applications usually registers their core bindings
     * here as well (e.g. print, setTimeout). */
    jerry_init (JERRY_INIT_EMPTY);
    jerryx_debugger_after_connect (jerryx_debugger_tcp_create (5001)
                                   && jerryx_debugger_ws_create ());

    do
    {
      jerry_value_t run_result;

      receive_status = jerry_debugger_wait_for_client_source (wait_for_source_callback,
                                                              NULL,
                                                              &run_result);

      jerry_value_free (run_result);
    }
    while (receive_status == JERRY_DEBUGGER_SOURCE_RECEIVED);

    jerry_cleanup ();
  }
  while (receive_status == JERRY_DEBUGGER_CONTEXT_RESET_RECEIVED);

  if (receive_status == JERRY_DEBUGGER_SOURCE_RECEIVE_FAILED)
  {
    // Handle the failure (e.g. display an error).
  }
  return 0;
}
```

### jerry_debugger_send_output

**Summary**

Sends the program's output to the debugger client.

**Prototype**

```c
void
jerry_debugger_send_output (const jerry_char_t *buffer, jerry_size_t str_size)
```

**Example**

[doctest]: # (test="link")

```c
#include "jerryscript.h"
#include "jerryscript-ext/debugger.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);
  jerryx_debugger_after_connect (jerryx_debugger_tcp_create (5001)
                                 && jerryx_debugger_ws_create ());

  jerry_char_t my_output[] = "Hey, this should be sent too!";
  jerry_size_t my_output_size = sizeof (my_output);

  jerry_debugger_send_output (my_output, my_output_size);

  jerry_cleanup ();
}
```
