/*
 * Copyright (C) 2014-2025 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QCoreApplication> /* Q_DECLARE_TR_FUNCTIONS */
#include <QString>
#include <QStringList>

/*!
 * @brief Encapsulates INI-style preferences.
 */
class INIPreferences {
	Q_DECLARE_TR_FUNCTIONS(INIPreferences)

public:
	/*!
	 * @brief Constructor.
	 */
	INIPreferences(void);

	/* Set via command line. */
	QString confSubdir; /*!< Configuration directory. */
	QString loadFromConf; /*!< Configuration file to load from. */
	QString saveToConf; /*!< Configuration file to save to. */

	/*!
	 * @brief Code describing whether essential application data can be accessed.
	 *
	 * @note The values apply to the permissions of the user who's winning the application.
	 */
	enum ConfPresence {
		ACCESSIBLE = 0, /*!< Data accessible. */
		CONF_DIR_CANNOT_CREATE, /*!< Directory cannot be created. */
		CONF_DIR_NOT_EXECUTABLE, /*!< Directory doesn't have execute permission. This isn't used on Windows. */
		CONF_DIR_NOT_WRITEABLE, /*!< Directory doesn't have write permission. */
		CONF_DIR_NOT_READABLE, /*!< Directory doesn't have read permission. */
		CONF_FILE_CANNOT_CREATE, /*!< File cannot be created. */
		CONF_FILE_NOT_WRITEABLE, /*!< File doesn't have write permission. */
		CONF_FILE_NOT_READABLE /*!< File doesn't have read permission. */
	};

	/*!
	 * @brief Return localised error message string.
	 *
	 * @param[in] presenceCode Code why the \a problemPath cannot be accessed.
	 * @param[in] problemPath Path related to \a presenceCode.
	 * @return Localised error string.
	 */
	static
	QString confPresenceErrorMessage(enum ConfPresence presenceCode,
	    const QString &problemPath);

	/*!
	 * @brief Return localised error message string.
	 *
	 * @param[in] presenceCode Code why the \a problemPaths cannot be accessed.
	 * @param[in] problemPaths Paths related to \a presenceCode.
	 * @return Localised error string.
	 */
	static
	QString confPresenceErrorMessage2(enum ConfPresence presenceCode,
	    const QStringList &problemPaths);

	/*!
	 * @brief Create configuration file if not present.
	 *
	 * @note Location of the configuration file is taken from this
	 *     preferences structure.
	 *
	 * @param[out] problemPathPtr Pointer to string where to store
	 *                            the problem path on any error.
	 * @return ACCESSIBLE if basic configuration files can be accessed,
	 *      other codes depending on the detected problem.
	 */
	enum ConfPresence ensureConfPresence(QString *problemPathPtr = Q_NULLPTR) const;

	/*!
	 * @brief Check content of configuration directory.
	 *
	 * @note Location of the configuration file is taken from this
	 *     preferences structure.
	 *
	 * @param[out] problemPaths List of problematic paths.
	 * @return ACCESSIBLE if files and directories can be accessed,
	 *      other codes depending on the detected problem.
	 */
	enum ConfPresence checkConfiguration(QStringList &problemPaths) const;

	/*!
	 * @brief Return path to configuration directory.
	 *
	 * @return Path to configuration directory.
	 */
	QString confDir(void) const;

	/*!
	 * @brief Returns whole configuration file path.
	 *
	 * @return Whole path to loading configuration file.
	 */
	QString loadConfPath(void) const;

	/*!
	 * @brief Returns whole configuration file path.
	 *
	 * @return Whole path to saving configuration file.
	 */
	QString saveConfPath(void) const;

	/*!
	 * @brief Returns whole account db path.
	 *
	 * @return Whole path to account database file.
	 */
	QString acntDbPath(void) const;

	/*!
	 * @brief Returns whole tag db path.
	 *
	 * @return Whole path to tag database file.
	 */
	QString tagDbPath(void) const;

	/*!
	 * @brief Returns whole records management db path.
	 *
	 * @return Whole path to records management database file.
	 */
	QString recMgmtDbPath(void) const;

	/*!
	 * @brief Return whole preferences db path.
	 *
	 * @return Whole path to preferences database file.
	 */
	QString prefsDbPath(void) const;
};
