/*
 *  Copyright (C) 2011-2024 Arnaud Quette (Design and part of implementation)
 *  Copyright (C) 2011 - EATON
 *  Copyright (C) 2020-2024 - Jim Klimov <jimklimov+nut@gmail.com> - support and modernization of codebase
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*! \file scan_avahi.c
    \brief detect NUT through Avahi mDNS / DNS-SD services
    \author Frederic Bohe <fredericbohe@eaton.com>
    \author Arnaud Quette <arnaudquette@free.fr>
*/

#include "common.h"
#include "nut-scan.h"

/* externally visible to nutscan-init */
int nutscan_unload_avahi_library(void);

#ifdef WITH_AVAHI

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include "timehead.h"

#include <avahi-client/client.h>
#include <avahi-client/lookup.h>

#include <avahi-common/simple-watch.h>
#include <avahi-common/malloc.h>
#include <avahi-common/error.h>

#include <ltdl.h>

#define SCAN_AVAHI_DRIVERNAME "dummy-ups"

/* dynamic link library stuff */
static lt_dlhandle dl_handle = NULL;
static const char *dl_error = NULL;
static char *dl_saved_libname = NULL;

static AvahiClient* (*nut_avahi_service_browser_get_client)(AvahiServiceBrowser *);
static int (*nut_avahi_simple_poll_loop)(AvahiSimplePoll *s);
static void (*nut_avahi_client_free)(AvahiClient *client);
static int (*nut_avahi_client_errno)(AvahiClient*);
static void (*nut_avahi_free)(void *p);
static void (*nut_avahi_simple_poll_quit)(AvahiSimplePoll *s);
static AvahiClient* (*nut_avahi_client_new)(
		const AvahiPoll *poll_api,
		AvahiClientFlags flags,
		AvahiClientCallback callback,
		void *userdata,
		int *error);
static void (*nut_avahi_simple_poll_free)(AvahiSimplePoll *s);
static AvahiServiceResolver * (*nut_avahi_service_resolver_new)(
		AvahiClient *client,
		AvahiIfIndex interface,
		AvahiProtocol protocol,
		const char *name,
		const char *type,
		const char *domain,
		AvahiProtocol aprotocol,
		AvahiLookupFlags flags,
		AvahiServiceResolverCallback callback,
		void *userdata);
static const char * (*nut_avahi_strerror)(int error);
static AvahiClient* (*nut_avahi_service_resolver_get_client)(AvahiServiceResolver *);
static AvahiServiceBrowser* (*nut_avahi_service_browser_new)(
		AvahiClient *client,
		AvahiIfIndex interface,
		AvahiProtocol protocol,
		const char *type,
		const char *domain,
		AvahiLookupFlags flags,
		AvahiServiceBrowserCallback callback,
		void *userdata);
static int (*nut_avahi_service_resolver_free)(AvahiServiceResolver *r);
static AvahiSimplePoll *(*nut_avahi_simple_poll_new)(void);
static char* (*nut_avahi_string_list_to_string)(AvahiStringList *l);
static int (*nut_avahi_service_browser_free)(AvahiServiceBrowser *);
static char * (*nut_avahi_address_snprint)(char *ret_s, size_t length, const AvahiAddress *a);
static const AvahiPoll* (*nut_avahi_simple_poll_get)(AvahiSimplePoll *s);

/* Return 0 on success, -1 on error e.g. "was not loaded";
 * other values may be possible if lt_dlclose() errors set them;
 * visible externally */
int nutscan_unload_library(int *avail, lt_dlhandle *pdl_handle, char **libpath);
int nutscan_unload_avahi_library(void)
{
	return nutscan_unload_library(&nutscan_avail_avahi, &dl_handle, &dl_saved_libname);
}

/* Return 0 on error; visible externally */
int nutscan_load_avahi_library(const char *libname_path);
int nutscan_load_avahi_library(const char *libname_path)
{
	if (dl_handle != NULL) {
		/* if previous init failed */
		if (dl_handle == (void *)1) {
			return 0;
		}
		/* init has already been done */
		return 1;
	}

	if (libname_path == NULL) {
		upsdebugx(0, "AVAHI client library not found. AVAHI search disabled.");
		return 0;
	}

	if (lt_dlinit() != 0) {
		upsdebugx(0, "%s: Error initializing lt_dlinit", __func__);
		return 0;
	}

	dl_handle = lt_dlopen(libname_path);
	if (!dl_handle) {
		dl_error = lt_dlerror();
		goto err;
	}

	/* Clear any existing error */
	lt_dlerror();

	*(void **) (&nut_avahi_service_browser_get_client) = lt_dlsym(dl_handle, "avahi_service_browser_get_client");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_simple_poll_loop) = lt_dlsym(dl_handle, "avahi_simple_poll_loop");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_client_free) = lt_dlsym(dl_handle, "avahi_client_free");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_client_errno) = lt_dlsym(dl_handle, "avahi_client_errno");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_free) = lt_dlsym(dl_handle, "avahi_free");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_simple_poll_quit) = lt_dlsym(dl_handle, "avahi_simple_poll_quit");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_client_new) = lt_dlsym(dl_handle, "avahi_client_new");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_simple_poll_free) = lt_dlsym(dl_handle, "avahi_simple_poll_free");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_service_resolver_new) = lt_dlsym(dl_handle, "avahi_service_resolver_new");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_strerror) = lt_dlsym(dl_handle, "avahi_strerror");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_service_resolver_get_client) = lt_dlsym(dl_handle, "avahi_service_resolver_get_client");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_service_browser_new) = lt_dlsym(dl_handle, "avahi_service_browser_new");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_service_resolver_free) = lt_dlsym(dl_handle, "avahi_service_resolver_free");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_simple_poll_new) = lt_dlsym(dl_handle, "avahi_simple_poll_new");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_string_list_to_string) = lt_dlsym(dl_handle, "avahi_string_list_to_string");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_service_browser_free) = lt_dlsym(dl_handle, "avahi_service_browser_free");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_address_snprint) = lt_dlsym(dl_handle, "avahi_address_snprint");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	*(void **) (&nut_avahi_simple_poll_get) = lt_dlsym(dl_handle, "avahi_simple_poll_get");
	if ((dl_error = lt_dlerror()) != NULL) {
		goto err;
	}

	if (dl_saved_libname)
		free(dl_saved_libname);
	dl_saved_libname = xstrdup(libname_path);

	return 1;

err:
	upsdebugx(0,
		"Cannot load AVAHI library (%s) : %s. AVAHI search disabled.",
		libname_path, dl_error);
	dl_handle = (void *)1;
	lt_dlexit();
	if (dl_saved_libname) {
		free(dl_saved_libname);
		dl_saved_libname = NULL;
	}
	return 0;
}
/* end of dynamic link library stuff */

static AvahiSimplePoll *simple_poll = NULL;
static nutscan_device_t * dev_ret = NULL;
static useconds_t avahi_usec_timeout = 0;

static void update_device(const char * host_name, const char *ip, uint16_t port, char * text, int proto)
{
	nutscan_device_t * dev = NULL;

	char * t = NULL;
	char * t_saveptr = NULL;
	char * phrase = NULL;
	char * phrase_saveptr = NULL;
	char * word = NULL;
	char * value = NULL;
	char * device = NULL;
	char * device_saveptr = NULL;
	int device_found = 0;
	char buf[6];
	size_t buf_size;

	if (text == NULL) {
		return;
	}

	t = strdup(text);
	phrase = strtok_r(t, "\"", &t_saveptr);
	while (phrase != NULL) {
		word = strtok_r(phrase, "=", &phrase_saveptr);
		if (word == NULL) {
			phrase = strtok_r(NULL, "\"", &t_saveptr);
			continue;
		}
		value = strtok_r(NULL, "=", &phrase_saveptr);
		if (value == NULL) {
			phrase = strtok_r(NULL, "\"", &t_saveptr);
			continue;
		}

		if (strcmp(word, "device_list") != 0) {
			phrase = strtok_r(NULL, "\"", &t_saveptr);
			continue;
		}

		device = strtok_r(value, ";", &device_saveptr);
		while (device != NULL) {
			device_found = 1;
			dev = nutscan_new_device();
			dev->type = TYPE_NUT;
			/* NOTE: There is no driver by such name, in practice it could
			 * be a dummy-ups relay, a clone driver, or part of upsmon config */
			dev->driver = strdup(SCAN_AVAHI_DRIVERNAME);
			if (proto == AVAHI_PROTO_INET) {
				nutscan_add_option_to_device(dev, "desc", "IPv4");
			}
			if (proto == AVAHI_PROTO_INET6) {
				nutscan_add_option_to_device(dev, "desc", "IPv6");
			}

			if (port != PORT) {
				/* +5+1+1+1 is for :
				 - port number (max 65535 so 5 characters),
				 - '@' and ':' characters
				 - terminating 0 */
				buf_size = strlen(device) +
					strlen(host_name) +
					5 + 1 + 1 + 1;
				dev->port = malloc(buf_size);
				if (dev->port) {
					snprintf(dev->port, buf_size, "%s@%s:%" PRIu16,
						device, host_name, port);
				}
			}
			else {
				/*+1+1 is for '@' character and terminating 0 */
				buf_size = strlen(device) + strlen(host_name) + 1 + 1;
				dev->port = malloc(buf_size);
				if (dev->port) {
					snprintf(dev->port, buf_size, "%s@%s",
						device, host_name);
				}
			}
			if (dev->port) {
				dev_ret = nutscan_add_device_to_device(dev_ret, dev);
			}
			else {
				nutscan_free_device(dev);
			}
			device = strtok_r(NULL, ";", &device_saveptr);
		}

		phrase = strtok_r(NULL, "\"", &t_saveptr);
	}
	free(t);

	/* If no device published in avahi data, try to get the device by
	connecting directly to upsd */
	if (!device_found) {
		snprintf(buf, sizeof(buf), "%u", port);
		dev = nutscan_scan_nut(ip, ip, buf, avahi_usec_timeout);
		if (dev) {
			dev_ret = nutscan_add_device_to_device(dev_ret, dev);
		}
		/* add an upsd entry without associated device */
		else {
			dev = nutscan_new_device();
			dev->type = TYPE_NUT;
			dev->driver = strdup(SCAN_AVAHI_DRIVERNAME);
			if (proto == AVAHI_PROTO_INET) {
				nutscan_add_option_to_device(dev, "desc", "IPv4");
			}
			if (proto == AVAHI_PROTO_INET6) {
				nutscan_add_option_to_device(dev, "desc", "IPv6");
			}
			if (port != PORT) {
				/*+1+1 is for ':' character and terminating 0 */
				/*buf is the string containing the port number*/
				buf_size = strlen(host_name) + strlen(buf) + 1 + 1;
				dev->port = malloc(buf_size);
				if (dev->port) {
					snprintf(dev->port, buf_size, "%s:%s",
						host_name, buf);
				}
			}
			else {
				dev->port = strdup(host_name);
			}
			if (dev->port) {
				dev_ret = nutscan_add_device_to_device(dev_ret, dev);
			}
			else {
				nutscan_free_device(dev);
			}
		}
	}
}

static void resolve_callback(
	AvahiServiceResolver *r,
	AvahiIfIndex interface,
	AvahiProtocol protocol,
	AvahiResolverEvent event,
	const char *name,
	const char *type,
	const char *domain,
	const char *host_name,
	const AvahiAddress *address,
	uint16_t port,
	AvahiStringList *txt,
	AvahiLookupResultFlags flags,
	void* userdata)
{

	assert(r);

	NUT_UNUSED_VARIABLE(interface);
	NUT_UNUSED_VARIABLE(protocol);
	NUT_UNUSED_VARIABLE(userdata);

	/* Called whenever a service has been resolved successfully or timed out */

	switch (event) {
		case AVAHI_RESOLVER_FAILURE:
			upsdebugx(0, "%s: "
				"(Resolver) Failed to resolve service '%s' of type '%s' in domain '%s': %s",
				__func__, name, type, domain,
				(*nut_avahi_strerror)((*nut_avahi_client_errno)((*nut_avahi_service_resolver_get_client)(r))));
			break;

		case AVAHI_RESOLVER_FOUND: { /* scoping */
			char a[AVAHI_ADDRESS_STR_MAX], *t;

/*			upsdebugx(1, "%s: Service '%s' of type '%s' in domain '%s':", __func__, name, type, domain); */

			(*nut_avahi_address_snprint)(a, sizeof(a), address);
			t = (*nut_avahi_string_list_to_string)(txt);

			NUT_UNUSED_VARIABLE(flags);
/*
			upsdebugx(1,
				"\t%s:%u (%s)\n"
				"\tTXT=%s\n"
				"\tcookie is %u\n"
				"\tis_local: %i\n"
				"\tour_own: %i\n"
				"\twide_area: %i\n"
				"\tmulticast: %i\n"
				"\tcached: %i\n",
				host_name, port, a,
				t,
				avahi_string_list_get_service_cookie(txt),
				!!(flags & AVAHI_LOOKUP_RESULT_LOCAL),
				!!(flags & AVAHI_LOOKUP_RESULT_OUR_OWN),
				!!(flags & AVAHI_LOOKUP_RESULT_WIDE_AREA),
				!!(flags & AVAHI_LOOKUP_RESULT_MULTICAST),
				!!(flags & AVAHI_LOOKUP_RESULT_CACHED));
*/
			update_device(host_name, a, port, t, address->proto);
			(*nut_avahi_free)(t);
			}
			break;

#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && ( (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_COVERED_SWITCH_DEFAULT) || (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_UNREACHABLE_CODE) )
# pragma GCC diagnostic push
#endif
#ifdef HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_COVERED_SWITCH_DEFAULT
# pragma GCC diagnostic ignored "-Wcovered-switch-default"
#endif
#ifdef HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_UNREACHABLE_CODE
# pragma GCC diagnostic ignored "-Wunreachable-code"
#endif
/* Older CLANG (e.g. clang-3.4) seems to not support the GCC pragmas above */
#ifdef __clang__
# pragma clang diagnostic push
# pragma clang diagnostic ignored "-Wunreachable-code"
# pragma clang diagnostic ignored "-Wcovered-switch-default"
#endif
		default:
			/* Must not occur. */
			break;
#ifdef __clang__
# pragma clang diagnostic pop
#endif
#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && ( (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_COVERED_SWITCH_DEFAULT) || (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_UNREACHABLE_CODE) )
# pragma GCC diagnostic pop
#endif
	}

	(*nut_avahi_service_resolver_free)(r);
}

static void browse_callback(
		AvahiServiceBrowser *b,
		AvahiIfIndex interface,
		AvahiProtocol protocol,
		AvahiBrowserEvent event,
		const char *name,
		const char *type,
		const char *domain,
		AvahiLookupResultFlags flags,
		void* userdata)
{

	AvahiClient *c = userdata;
	assert(b);

	NUT_UNUSED_VARIABLE(flags);

	/* Called whenever a new services becomes available on the LAN or is removed from the LAN */

	switch (event) {
		case AVAHI_BROWSER_FAILURE:

			upsdebugx(0, "%s: (Browser) %s",
				__func__,
				(*nut_avahi_strerror)((*nut_avahi_client_errno)((*nut_avahi_service_browser_get_client)(b))));
			(*nut_avahi_simple_poll_quit)(simple_poll);
			return;

		case AVAHI_BROWSER_NEW:
/*			upsdebugx(1, "%s: "(Browser) NEW: service '%s' of type '%s' in domain '%s'", __func__, name, type, domain); */

			/* We ignore the returned resolver object. In the callback
			   function we free it. If the server is terminated before
			   the callback function is called the server will free
			   the resolver for us. */

#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_ASSIGN_ENUM)
# pragma GCC diagnostic push
# pragma GCC diagnostic ignored "-Wassign-enum"
#endif
			/* It seems that avahi-common/defs.h only defines the flags in a
			 * manner similar to bitmask flags to request certain features,
			 * but lacks a value in that enum for lack of flags (unconstrained
			 * lookup). So we have to silence a warning here...
			 */
			if (!((*nut_avahi_service_resolver_new)(c, interface, protocol, name, type, domain, AVAHI_PROTO_UNSPEC, 0, resolve_callback, c)))
#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_ASSIGN_ENUM)
# pragma GCC diagnostic pop
#endif
				upsdebugx(0, "%s: "
					"Failed to resolve service '%s': %s",
					__func__,
					name, (*nut_avahi_strerror)((*nut_avahi_client_errno)(c)));

			break;

		case AVAHI_BROWSER_REMOVE:
			upsdebugx(0, "%s: (Browser) REMOVE: service '%s' of type '%s' in domain '%s'", __func__, name, type, domain);
			break;

		case AVAHI_BROWSER_ALL_FOR_NOW:
			(*nut_avahi_simple_poll_quit)(simple_poll);
			goto fallthrough_AVAHI_BROWSER_CACHE_EXHAUSTED; /* be explicit */

		case AVAHI_BROWSER_CACHE_EXHAUSTED:
		fallthrough_AVAHI_BROWSER_CACHE_EXHAUSTED:
/*			upsdebugx(1, "%s: (Browser) %s", __func__, event == AVAHI_BROWSER_CACHE_EXHAUSTED ? "CACHE_EXHAUSTED" : "ALL_FOR_NOW"); */
			break;

#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && ( (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_COVERED_SWITCH_DEFAULT) || (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_UNREACHABLE_CODE) )
# pragma GCC diagnostic push
#endif
#ifdef HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_COVERED_SWITCH_DEFAULT
# pragma GCC diagnostic ignored "-Wcovered-switch-default"
#endif
#ifdef HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_UNREACHABLE_CODE
# pragma GCC diagnostic ignored "-Wunreachable-code"
#endif
/* Older CLANG (e.g. clang-3.4) seems to not support the GCC pragmas above */
#ifdef __clang__
# pragma clang diagnostic push
# pragma clang diagnostic ignored "-Wunreachable-code"
# pragma clang diagnostic ignored "-Wcovered-switch-default"
#endif
		default:
			/* Must not occur. */
			break;
#ifdef __clang__
# pragma clang diagnostic pop
#endif
#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && ( (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_COVERED_SWITCH_DEFAULT) || (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_UNREACHABLE_CODE) )
# pragma GCC diagnostic pop
#endif
	}
}

static void client_callback(AvahiClient *c, AvahiClientState state, void * userdata) {
	assert(c);
	NUT_UNUSED_VARIABLE(userdata);

	/* Called whenever the client or server state changes */

	if (state == AVAHI_CLIENT_FAILURE) {
		upsdebugx(0, "%s: "
			"Server connection failure: %s",
			__func__,
			(*nut_avahi_strerror)((*nut_avahi_client_errno)(c)));
		(*nut_avahi_simple_poll_quit)(simple_poll);
	}
}

static AvahiClient* wrap_nut_avahi_client_new(int *error)
{
	/* Allocate a new client */
#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_ASSIGN_ENUM)
# pragma GCC diagnostic push
# pragma GCC diagnostic ignored "-Wassign-enum"
#endif
	/* It seems that avahi-common/defs.h only defines the flags in a
	 * manner similar to bitmask flags to request certain features,
	 * but lacks a value in that enum for lack of flags (unconstrained
	 * lookup). So we have to silence a warning here...
	 */
	return (*nut_avahi_client_new)((*nut_avahi_simple_poll_get)(simple_poll), 0, client_callback, NULL, error);
#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_ASSIGN_ENUM)
# pragma GCC diagnostic pop
#endif
}

nutscan_device_t * nutscan_scan_avahi(useconds_t usec_timeout)
{
	/* Example service publication
	 * $ avahi-publish -s nut _upsd._tcp 3493 txtvers=1 protovers=1.0.0 device_list="dev1;dev2"
	 */
	AvahiClient *client = NULL;
	AvahiServiceBrowser *sb = NULL;
	int error;

	if (!nutscan_avail_avahi) {
		return NULL;
	}

	avahi_usec_timeout = usec_timeout;

	/* Allocate main loop object */
	if (!(simple_poll = (*nut_avahi_simple_poll_new)())) {
		upsdebugx(0, "%s: Failed to create Avahi simple poll object.",
			__func__);
		goto fail;
	}

	/* Allocate a new client */
	client = wrap_nut_avahi_client_new(&error);

	/* Check wether creating the client object succeeded */
	if (!client) {
		upsdebugx(0, "%s: "
			"Failed to create Avahi client: %s",
			__func__, (*nut_avahi_strerror)(error));
		goto fail;
	}

	/* Create the service browser */
#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_ASSIGN_ENUM)
# pragma GCC diagnostic push
# pragma GCC diagnostic ignored "-Wassign-enum"
#endif
	/* See comments about flags just a bit above */
	if (!(sb = (*nut_avahi_service_browser_new)(
		client, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC,
		"_upsd._tcp", NULL, 0, browse_callback, client))
	) {
#if (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_PUSH_POP) && (defined HAVE_PRAGMA_GCC_DIAGNOSTIC_IGNORED_ASSIGN_ENUM)
# pragma GCC diagnostic pop
#endif
		upsdebugx(0, "%s: "
			"Failed to create Avahi service browser: %s",
			__func__,
			(*nut_avahi_strerror)((*nut_avahi_client_errno)(client)));
		goto fail;
	}

	/* Run the main loop */
	(*nut_avahi_simple_poll_loop)(simple_poll);

fail:

	/* Cleanup things */
	if (sb)
		(*nut_avahi_service_browser_free)(sb);

	if (client)
		(*nut_avahi_client_free)(client);

	if (simple_poll)
		(*nut_avahi_simple_poll_free)(simple_poll);

	return nutscan_rewind_device(dev_ret);
}

#else  /* not WITH_AVAHI */

/* stub function */
nutscan_device_t * nutscan_scan_avahi(useconds_t usec_timeout)
{
	NUT_UNUSED_VARIABLE(usec_timeout);

	return NULL;
}

int nutscan_unload_avahi_library(void)
{
	return 0;
}
#endif /* WITH_AVAHI */
