import type ts from "typescript/lib/tsserverlibrary"
import type tsModule from "./tsserverlibrary.shim"

const SIDE_EFFECT_PROJECTS_LIMIT = 20
const SIDE_EFFECT_FILES_LIMIT = 30

// Spreading expiration time to avoid closing multiple files at once
const SIDE_EFFECT_EXPIRATION_TIME_MIN = 5 * 60 * 1000 // 5 mins
const SIDE_EFFECT_EXPIRATION_TIME_MAX = 10 * 60 * 1000 // 10 mins

export class IdeProjectService {

  private openedFilesByIde: Set<ts.server.NormalizedPath> = new Set()
  private openedProjectsByIde: Set<ts.server.NormalizedPath> = new Set()
  private openedFilesBySideEffect: Map<ts.server.NormalizedPath, number /*expirationTime */> = new Map()
  private openedProjectsBySideEffect: Map<ts.server.NormalizedPath, number /*expirationTime */> = new Map()

  private fallbackToOldEvalOutsideOfImportGraph: boolean = process.argv.slice(2).indexOf("--noFallbackToOldEvalOutsideOfImportGraph") < 0

  private readonly originalOpenClientFileWithNormalizedPath: (
    fileName: tsModule.server.NormalizedPath,
    _fileContent?: string,
    _scriptKind?: tsModule.ScriptKind,
    _hasMixedContent?: boolean,
    _projectRootPath?: tsModule.server.NormalizedPath
  ) => tsModule.server.OpenConfiguredProjectResult

  private readonly originalCloseClientFile: (
    uncheckedFileName: string
  ) => void

  private readonly originalCloseExternalProject: (
    uncheckedFileName: string
  ) => void

  constructor(private readonly ts: typeof import("typescript/lib/tsserverlibrary"),
              private readonly projectService: ts.server.ProjectService,
  ) {
    let thisService = this
    const openClientFileWithNormalizedPath = projectService.openClientFileWithNormalizedPath
    const closeClientFile = projectService.closeClientFile
    const closeExternalProject = projectService.closeExternalProject
    const openExternalProject = projectService.openExternalProject

    if (openClientFileWithNormalizedPath) {
      this.originalOpenClientFileWithNormalizedPath = openClientFileWithNormalizedPath.bind(projectService)
      projectService.openClientFileWithNormalizedPath = function (
        fileName: tsModule.server.NormalizedPath,
        _fileContent?: string,
        _scriptKind?: tsModule.ScriptKind,
        _hasMixedContent?: boolean,
        _projectRootPath?: tsModule.server.NormalizedPath
      ): tsModule.server.OpenConfiguredProjectResult {
        thisService.fileOpenedByIde(fileName)
        return openClientFileWithNormalizedPath.apply(projectService, arguments as any);
      }
    } else {
      this.originalOpenClientFileWithNormalizedPath = () => {
        throw new Error("Unsupported version of compiler plugin")
      };
    }

    if (closeClientFile) {
      this.originalCloseClientFile = closeClientFile.bind(projectService)
      projectService.closeClientFile = function (
        uncheckedFileName: string
      ) {
        thisService.fileClosedByIde(uncheckedFileName)
        closeClientFile.apply(projectService, arguments as any);
      }
    } else {
      this.originalCloseClientFile = () => {
        throw new Error("Unsupported version of compiler plugin")
      };
    }

    if (openExternalProject) {
      projectService.openExternalProject = function (
        proj: ts.server.protocol.ExternalProject
      ) {
        thisService.projectOpenedByIde(proj.projectFileName)
        openExternalProject.apply(projectService, arguments as any);
      }
    }

    if (closeExternalProject) {
      this.originalCloseExternalProject = closeExternalProject.bind(projectService)
      projectService.closeExternalProject = function (
        uncheckedFileName: string
      ) {
        thisService.projectClosedByIde(uncheckedFileName)
        closeExternalProject.apply(projectService, arguments as any);
      }
    } else {
      this.originalCloseExternalProject = () => {
        throw new Error("Unsupported version of compiler plugin")
      };
    }
  }

  closeClientFileSafely(fileName: string) {
    let normalizedFileName = this.ts.server.toNormalizedPath(fileName)
    this.fileClosedByIde(normalizedFileName)
    this.closeSafely(normalizedFileName)
  }

  getProjectAndSourceFile(fileName: string, projectFileName: string | undefined): {
    project?: ts.server.Project,
    sourceFile?: ts.SourceFile
  } {
    let normalizedProjectFileName = projectFileName ? this.ts.server.toNormalizedPath(projectFileName) : undefined;
    let normalizedFileName = this.ts.server.toNormalizedPath(fileName)

    let project = this.findOrOpenProject(normalizedProjectFileName)
    let sourceFile = project?.getLanguageService(true)?.getProgram()?.getSourceFile(normalizedFileName)

    if (projectFileName && !sourceFile && this.fallbackToOldEvalOutsideOfImportGraph) {
      let error = new Error("File outside of import graph")
      error.isFileOutsideOfImportGraphError = true
      throw error
    }

    if (!project || !sourceFile) {
      // Try to fallback to default project for the file
      project = this.findOrOpenDefaultProjectForFile(normalizedFileName)
      sourceFile = project?.getLanguageService(true)?.getProgram()?.getSourceFile(fileName)
    }
    return {project, sourceFile}
  }

  tryOpenProject(projectFileName: string): boolean {
    if (this.internalTryOpenProject(projectFileName)) {
      this.projectOpenedByIde(projectFileName)
      return true;
    }
    return false;
  }

  private closeSafely(fileName: ts.server.NormalizedPath) {
    const wasOpened = this.projectService.openFiles.has(fileName)
        || (!this.ts.sys.useCaseSensitiveFileNames && this.projectService.openFiles.has(fileName.toLowerCase()))
    if (wasOpened) {
      this.projectService.logger.info(`IdeProjectService.closeSafely: closing file - ${fileName}`)
      this.originalCloseClientFile(fileName)
    }
    else {
      this.projectService.logger.info(`IdeProjectService.closeSafely: file is no longer open - ${fileName}`)
    }
  }

  private findOrOpenProject(projectFileName: ts.server.NormalizedPath | undefined): ts.server.Project | undefined {
    if (!projectFileName) return undefined
    let project = this.projectService.findProject(projectFileName)
    if (!project) {
      this.openProjectBySideEffect(projectFileName)
      project = this.projectService.findProject(projectFileName)
    }
    else {
      this.usedAsSideEffect('project', projectFileName)
    }
    return project
  }

  private openProjectBySideEffect(projectFileName: ts.server.NormalizedPath) {
    this.projectService.logger.info(`IdeProjectService.openProjectBySideEffect: opening project by side effect - ${projectFileName}`)
    if (this.internalTryOpenProject(projectFileName)) {
      this.openedAsSideEffect('project', projectFileName)
      this.cleanupOpenedProjectsBySideEffect()
    }
  }

  private findOrOpenDefaultProjectForFile(fileName: ts.server.NormalizedPath): ts.server.Project | undefined {
    let project = this.projectService.getDefaultProjectForFile(fileName, false)
    if (!project) {
      this.openFileBySideEffect(fileName)
      project = this.projectService.getDefaultProjectForFile(fileName, true)
    }
    else {
      this.usedAsSideEffect('file', fileName)
    }
    return project
  }

  private openFileBySideEffect(fileName: ts.server.NormalizedPath) {
    if (this.openedFilesByIde.has(fileName)) {
      throw new Error(`IdeProjectService.openFileBySideEffect: File ${fileName} is already opened by IDE`)
    }
    this.projectService.logger.info(`IdeProjectService.openFileBySideEffect: opening file by side effect - ${fileName}`)
    this.originalOpenClientFileWithNormalizedPath(fileName)
    this.openedAsSideEffect('file', fileName)
    this.cleanupOpenedFilesBySideEffect()
  }

  private cleanupOpenedProjectsBySideEffect() {
    for (let projectFileName of getNamesToClose(this.openedProjectsBySideEffect, SIDE_EFFECT_PROJECTS_LIMIT)) {
      this.closeProjectBySideEffect(projectFileName)
      this.openedProjectsBySideEffect.delete(projectFileName)
    }
  }

  private cleanupOpenedFilesBySideEffect() {
    for (let fileName of getNamesToClose(this.openedFilesBySideEffect, SIDE_EFFECT_FILES_LIMIT)) {
      this.closeFileBySideEffect(fileName)
      this.openedFilesBySideEffect.delete(fileName)
    }
  }

  private closeProjectBySideEffect(projectFileName: ts.server.NormalizedPath) {
    if (this.openedProjectsByIde.has(projectFileName)) {
      this.projectService.logger.info(`IdeProjectService.closeProjectBySideEffect: not closing project - project is opened by ide - ${projectFileName}`)
      return
    }
    let project = this.projectService.findProject(projectFileName)
    if (!project) {
      this.projectService.logger.info(`IdeProjectService.closeProjectBySideEffect: project already closed - ${projectFileName}`)
      return
    }

    for (let [openFile] of this.projectService.openFiles as Map<string, ts.server.NormalizedPath>) {
      if (openFile && (this.projectService.getScriptInfo(openFile)?.containingProjects?.indexOf(project) ?? -1) >= 0) {
        this.projectService.logger.info(`IdeProjectService.closeProjectBySideEffect: not closing project ${projectFileName} - in use by ${openFile}`)
        return
      }
    }
    this.projectService.logger.info(`IdeProjectService.closeProjectBySideEffect: closing project - ${projectFileName}`)
    this.originalCloseExternalProject(projectFileName)
  }

  private closeFileBySideEffect(fileName: ts.server.NormalizedPath) {
    if (this.openedFilesByIde.has(fileName)) {
      this.projectService.logger.info(`IdeProjectService.closeFileBySideEffect: not closing file - file is opened by ide - ${fileName}`)
      return
    }
    this.projectService.logger.info(`IdeProjectService.closeFileBySideEffect: trying to close file by side effect - ${fileName}`)
    this.closeSafely(fileName)
  }

  private openedAsSideEffect(entityKind: 'project' | 'file', fileName: ts.server.NormalizedPath) {
    let map = entityKind == 'project' ? this.openedProjectsBySideEffect : this.openedFilesBySideEffect
    let now = new Date().getTime()
    // use delete to move the entry to the end of the map
    map.delete(fileName)
    map.set(fileName, now + randomizedExpirationTime())
  }

  private usedAsSideEffect(entityKind: 'project' | 'file', fileName: ts.server.NormalizedPath) {
    let map = entityKind == 'project' ? this.openedProjectsBySideEffect : this.openedFilesBySideEffect
    // use delete to move the entry to the end of the map
    if (map.delete(fileName)) {
      let now = new Date().getTime()
      map.set(fileName, now + randomizedExpirationTime())
    }
  }

  private fileOpenedByIde(fileName: ts.server.NormalizedPath) {
    this.openedFilesByIde.add(fileName);
  }

  private fileClosedByIde(uncheckedFileName: string) {
    let normalizedFileName = this.ts.server.toNormalizedPath(uncheckedFileName)
    this.openedFilesByIde.delete(normalizedFileName)
  }

  private projectClosedByIde(projectName: string) {
    let normalizedProjectName = this.ts.server.toNormalizedPath(projectName)
    this.openedProjectsByIde.delete(normalizedProjectName)
  }

  private projectOpenedByIde(projectName: string) {
    let normalizedProjectName = this.ts.server.toNormalizedPath(projectName)
    this.openedProjectsByIde.add(normalizedProjectName);
  }

  private internalTryOpenProject(projectFileName: string): boolean {
    const tsConfigPath = this.ts.server.toNormalizedPath(projectFileName);
    const _projectService = this.projectService as any
    if (_projectService.createAndLoadConfiguredProject && _projectService.externalProjectToConfiguredProjectMap) {
      _projectService.externalProjectToConfiguredProjectMap.set(tsConfigPath, [tsConfigPath]);
      const project = _projectService.createAndLoadConfiguredProject(
        tsConfigPath, `Creating own configured project in external project: ${tsConfigPath}`);
      project.updateGraph();
      if (project.addExternalProjectReference) {
        project.addExternalProjectReference();
      }
      return true
    }

    if (_projectService.createConfiguredProject && _projectService.externalProjectToConfiguredProjectMap) {
      // Like in ProjectService.openExternalProject()
      const project = _projectService.createConfiguredProject(tsConfigPath, `Creating configured project in external project: ${tsConfigPath}`)
      if (!this.projectService.getHostPreferences().lazyConfiguredProjectsFromExternalProject) {
        project.updateGraph()
      }
      _projectService.externalProjectToConfiguredProjectMap.set(tsConfigPath, new Set([project]))
      return true
    }
    return false
  }
}

function randomizedExpirationTime() {
  return SIDE_EFFECT_EXPIRATION_TIME_MIN + Math.floor(Math.random() * (SIDE_EFFECT_EXPIRATION_TIME_MAX - SIDE_EFFECT_EXPIRATION_TIME_MIN));
}

function getNamesToClose<T>(map: Map<T, number /*expirationTime */>, limit: number): T[] {
  if (map.size > limit) {
    let allEntries: [T, number][] = [...map.entries()]
    // oldest entries are at the beginning of the map
    return allEntries
      // close a little bit more to allow for expiration code to trigger next time
      .slice(0, map.size - limit + 2)
      .map(entry => entry[0])
  }
  else {
    const toClose: T[] = []
    const now = new Date().getTime()

    for (let [projectFileName, expiration] of map) {
      if (expiration < now) {
        toClose.push(projectFileName)
      }
    }
    return toClose
  }
}