# -*- coding: utf-8 -*-
# vim:fenc=utf-8

# Copyright (C) 2012-2019 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# X2Go Session Broker is free software; you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# X2Go Session Broker is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

class X2GoBrokerNameService(object):

    def has_user(self, username):
        """\
        Provide information, if the broker knows a given user (or not).

        :param username: name of the user to check
        :type username: ``str``

        :returns: ``True`` if the user is known to the broker, ``False`` if not
        :rtype: ``bool``

        """
        return username in self.get_users()

    def get_users(self):
        """\
        Retrieve list of users known to the broker.

        :returns: list of known user names
        :rtype: ``list``

        """
        return []

    def get_primary_group(self, username):
        """\
        Get the primary group of a given user. If the nameservices
        backend in use does not support primary groups, an empty string
        will be returned.

        :param username: name of the user to get the primary group for
        :type username: ``str``

        :returns: name of the primary group of the given user
        :rtype: ``str``

        """
        return ''

    def has_group(self, group):
        """\
        Provide information, if the broker knows a given group (or not).

        :param group: name of the group to check
        :type group: ``str``

        :returns: ``True`` if the group is known to the broker, ``False`` if not
        :rtype: ``bool``

        """
        return group in self.get_groups()

    def get_groups(self):
        """\
        Retrieve list of groups known to the broker.

        :returns: list of known group names
        :rtype: ``list``

        """
        return []

    def is_group_member(self, username, group, primary_groups=False):
        """\
        Check, if a given user is member of a given group.

        Optionally, primary group memberships can be considered (or not).

        :param username: name of the user to check
        :type username: ``str``
        :param group: name of the group to check
        :type group: ``str``
        :param primary_groups: take primary group membership into consideration
            or not
        :type primary_groups: ``bool``

        :returns: ``True`` if the user is member of the given group, ``False`` if not
        :rtype: ``bool``

        """
        _members = self.get_group_members(group, primary_groups=primary_groups)
        return username in _members

    def get_group_members(self, group, primary_groups=False):
        """\
        Retrieve a list of users being members of a given group.

        Optionally, primary group memberships can be considered (or not).

        :param group: name of the group to retrieve members of
        :type group: ``str``
        :param primary_groups: take primary group membership into consideration
            or not
        :type primary_groups: ``bool``

        :returns: list of users that are members of the given group
        :rtype: ``list``

        """
        return []

    def get_user_groups(self, username, primary_groups=False):
        """\
        Retrieve a list of groups that a given user is member of.

        Optionally, primary group memberships can be considered (or not).

        :param username: name of the user to retrieve groupm memberships of
        :type username: ``str``
        :param primary_groups: take primary group membership into consideration
            or not
        :type primary_groups: ``bool``

        :returns: list of groups that the given user is member of
        :rtype: ``list``

        """
        _groups = []
        for _group in self.get_groups():
            if self.is_group_member(username=username, group=_group, primary_groups=primary_groups):
                _groups.append(_group)
        return _groups
