/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_x509_utils.c
    \brief  This file contains utility functions needed for x509 credential
            handling (openssl)
    \author Martijn Steenbakkers for EGEE
    This file contains utility functions needed for x509 credential
    handling (openssl).
    Contains code contributed by Andrew McNab.

*/

#ifndef LCAS_X509_UTILS_C
#define LCAS_X509_UTILS_C

/******************************************************************************
                             Include header files
******************************************************************************/
#define DEBUG 0
#define HACK 0

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* For X509 and STACK_OF(X509) structs (output) */
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/x509v3.h>
#include <openssl/err.h>

/* LCAS includes */
#include "lcas_log.h"
#include "_lcas_x509_utils.h"

/******************************************************************************
                             Defines
******************************************************************************/

void lcas_print_x509_to_string (X509 *, char *);
void lcas_print_stack_of_x509_to_string (STACK_OF(X509) *, char *);


/******************************************************************************
Function:       lcas_x509_free_chain()
Description:    Properly free a previously allocated STACK_OF(X509) *chain
		hence needs a STACK_OF(X509) **chain
Parameters:
                chain: pointer to STACK_OF(X509)
******************************************************************************/
void lcas_x509_free_chain(
        STACK_OF(X509) **  chain
)
{
     if (*chain)
     {
         sk_X509_pop_free( *chain, X509_free );
         /* sk_X509_free( chain ); */
         *chain = NULL;
     }
}

/******************************************************************************
Function:       lcas_x509_to_dn()
Description:    Get the name of the X509 cert
Parameters:
                px509: x509 certificate
Returns:        dn (which should be freed)
******************************************************************************/
char * lcas_x509_to_dn(
        X509 *  px509
)
{
    char * logstr = "lcas_x509_to_dn()";
    if (!px509)
    {
        lcas_log(1, "%s: No certificate found as input.\n", logstr);
        return NULL;
    }
        
    return X509_NAME_oneline(X509_get_subject_name(px509),NULL,0);
}

/******************************************************************************
Function:       lcas_x509_chain_to_dn()
Description:    Get the base DN of the X509 proxy cert
Parameters:
                px509: x509 certificate
                certstack: certificate chain
Returns:        dn (which should be freed)
******************************************************************************/
char * lcas_x509_chain_to_dn(
        X509 *  px509, STACK_OF(X509) *certstack
)
{
    char * logstr = "lcas_x509_chain_to_dn()";
    int i = 0;
    int amount_of_CAs = 0;
    int depth = sk_X509_num(certstack);

    /* The px509 is not used */

    if (!certstack)
    {
        lcas_log(1, "%s: No stack of certificates found as input.\n", logstr);
        return NULL;
    }

        
    /* Do we have a chain at all... */
    if (depth > 0)
    {
        /* Amount of CAs in chain */
        for (i = 0; i < depth; i++)
        {
            if (lcas_x509IsCA(sk_X509_value(certstack, i)))
                amount_of_CAs++;
        }

        /* Select the User certificate */
        i = depth - (amount_of_CAs + 1);

        if (i < 0)
        {
            i = 0;
            return NULL;
        }

        /* This returned value must be free'd */
        return lcas_x509_to_dn (sk_X509_value(certstack, i));
    }
    else
        return NULL;
}

/******************************************************************************
Function:       lcas_pem_string_to_x509
Description:    Reads a X509 certificate from a PEM-encoded string
Parameters:
                px: pointer to a X509 certificate
                cert_string: PEM-encoded string 
Returns:        0 on success, non-zero otherwise
******************************************************************************/
int lcas_pem_string_to_x509(X509 **px, char *certstring)
{
/*     BIO *certbio = NULL; */

//    *px = X509_new();
//    if (*px == NULL) return -1;
//    /* clean it first */
//    X509_free(*px);

/*
 *  Simular fix to this plugin as for LCMAPS
 */

//    certbio = BIO_new_mem_buf(certstring, -1);
//    if (certbio)
//    {
//        /* This call allocates new X509 struct, so no X509_new() needed */
//        *px = PEM_read_bio_X509(certbio, NULL, 0, NULL);
//    }
//
//    BIO_free(certbio);
//    return 0;
//    int rval = 0;


    STACK_OF (X509) * chain = NULL;
    char *dn;


    if (lcas_pem_string_to_x509_chain (&chain, certstring) == 0)
    {
        int amount_of_CAs = 0;
        int depth = sk_X509_num(chain);
        int i = 0;
        X509 * cert = NULL;


        // How many CA certs are there in the chain?
        for (i = 0; i < depth; i++)
        {
            if (lcas_x509IsCA(sk_X509_value(chain, i)))
                amount_of_CAs++;
        }

        /* Was: Getting the final proxy (the Leaf proxy) */
        /* Is : Getting the first and primary proxy (could be the MyProxy... 00ps ... yes, this is a problem) */
        /* Is again: Getting the final proxy (the Leaf proxy) */

        /* i = depth - (amount_of_CAs + 2); */

        /* if (i < 0) */
            /* i = 0; */

        /* lcas_log(0, "Value was %d, hardwired to 0\n", i);

        i = 0;
        */

        /* Zero is the LEAF proxy, the final certificate (most likely a proxy) in the chain */
        i = 0;
        cert = sk_X509_value(chain, i);
        *px  = X509_dup (cert);
    }

    /*
     * DEBUG the chain, this will output the chain or just one specific certificate to a file - like: openssl x509 -text -noout -in <cert_file>
     *
     * lcas_print_x509_to_string (*px, "/tmp/lcas-output.cert.log");
     * lcas_print_stack_of_x509_to_string (px, "/tmp/lcas-output.chain.log");
     */



/*    if (chain)  sk_X509_free (chain); */
    lcas_x509_free_chain( &chain ); /* NOTE: needs to change chain -> NULL */

    if (!px) return -1;
    dn = lcas_x509_to_dn( *px );
    if (dn)
    {
        lcas_log_debug (5, "Got individual certificate with subject: %s\n", dn);
        free( dn );
    } else
	return -1;

    return 0;
}

/******************************************************************************
Function:       lcas_pem_string_to_x509_chain
Description:    Creates a dynamically allocated stack of X509 certificate objects 
                by walking through the PEM-encoded X509 certificates. 
                Copied from gridsite (Andrew McNab), original name:
                    GRSTx509StringToChain()
Parameters:
                certstack: pointer to a stack of X509 certificates
                cert_string: PEM-encoded string 
Returns:        0 on success, non-zero otherwise
******************************************************************************/
int lcas_pem_string_to_x509_chain(STACK_OF(X509) **certstack, char *certstring)
{
    STACK_OF(X509_INFO) *sk=NULL;
    BIO *certbio = NULL;
    X509_INFO *xi;
#if DEBUG
    int stack_size = 0;
#endif

    *certstack = sk_X509_new_null();
    if ((*certstack == NULL) || (certstring == NULL)) return -1;

    certbio = BIO_new_mem_buf(certstring, -1);
  
    if (!(sk=PEM_X509_INFO_read_bio(certbio, NULL, NULL, NULL)))
    {
#if DEBUG
        fprintf(stderr, "Error reading stack from certbio\n");
#endif
        BIO_free(certbio);
        sk_X509_INFO_free(sk);
        lcas_x509_free_chain( certstack ); /* *certstack -> NULL */
        return 1;
    }
  
    while (sk_X509_INFO_num(sk))
    {
#if DEBUG
        char * dn = NULL;
        fprintf(stderr, "current X509_INFO stack size: %d, DN = ", sk_X509_INFO_num(sk));
#endif
        xi=sk_X509_INFO_shift(sk);
        if (xi->x509 != NULL)
        {
            sk_X509_push(*certstack, xi->x509);
#if DEBUG
            dn = lcas_x509_to_dn(xi->x509);
            if (dn) {
		fprintf(stderr, "%s\n", dn);
		free(dn);
	    }
	    else
	    {
                fprintf(stderr, "NULL!\n");
            }
#endif
            xi->x509=NULL;
        }
        X509_INFO_free(xi);
    }
#if DEBUG
    stack_size = sk_X509_INFO_num(sk);
    fprintf(stderr, "final size of stack of X509_INFO: %d, so it's %s\n",
        stack_size, (stack_size?"not yet empty! (NOT OK)":"empty (OK)."));
    fprintf(stderr, "size of stack of X509: %d\n", sk_X509_num(*certstack));
    {
        char * subject = NULL;
        X509 * cert = NULL;
        int i = 0;

        for (i = 0; i < sk_X509_num(*certstack); i++)
        {
            cert = sk_X509_value(*certstack, i);
            subject = lcas_x509_to_dn(cert);
            if (subject) {
		fprintf(stderr, "    dn of cert %d: %s\n", i, subject);
		free(subject);
	    }
	    else
	    {
                fprintf(stderr, "NULL!\n");
	    }
        }
    }
#endif

    /*
     * DEBUG the chain, this will output the chain or just one specific certificate to a file - like: openssl x509 -text -noout -in <cert_file>
     *
     * lcas_print_x509_to_string (px509_cred, "/tmp/lcaps-output.cert.log");
     * lcas_print_stack_of_x509_to_string (px509_chain, "/tmp/lcaps-output.chain.log");
     */


   if (!sk_X509_num(*certstack))
     {
#if DEBUG
        fprintf(stderr, "Found empty stack!\n");
#endif
       BIO_free(certbio);
       sk_X509_INFO_free(sk);
       lcas_x509_free_chain( certstack ); /* *certstack -> NULL */
       return 1;
     }

   BIO_free(certbio);
   sk_X509_INFO_free(sk);
   ERR_remove_state(0);
   
   return 0;
}



/******************************************************************************
Function:   lcas_x509IsCA
Description:
    Tests if the X509 * cert is a CA certificate or not
Parameters:
    A X509 pointer
Returns:
    0      : Not a CA cert
    1      : This is a CA cert
******************************************************************************/
int lcas_x509IsCA (X509 *cert)
{
   /* int idret, purpose_id; */
   int purpose_id;

   purpose_id = X509_PURPOSE_get_by_sname("sslclient");

   /* final argument to X509_check_purpose() is whether to check for CAness */

   if (X509_check_purpose(cert, purpose_id + X509_PURPOSE_MIN, 1))
        return 1;
   else return 0;
}


/******************************************************************************
Function:    lcas_print_stack_of_x509_to_string
Description:
    Extract from X509 certificate chain (STACK_OF(X509)) and
        write the output to a file.
    Simulair to `openssl x509 -in <file.pem> -text -noout

    Note: only for debugging purposes
Parameters:
    px509_chain:    pointer to a x509 chain
Returns:
    void
******************************************************************************/
void lcas_print_stack_of_x509_to_string (STACK_OF(X509) * px509_chain, char * output_file)
{
    char *            logstr = "\tlcas_print_stack_of_x509_to_string()";

    if (px509_chain)
    {
        STACK_OF(X509) *  dupChain = NULL;
        X509 *            cert = NULL;

/*        dupChain = (STACK_OF(type) *) sk_X509_dup(px509_chain);*/
        dupChain = sk_X509_dup(px509_chain);
        lcas_log(0,"%s\n", logstr);

        while ((cert = sk_X509_pop(dupChain)))
        {
            lcas_print_x509_to_string (cert, output_file);
        }

        lcas_x509_free_chain( &dupChain ); /* dupChain -> NULL */
    }
    else
    {
        lcas_log(0,"%s: no input X509 chain!\n", logstr);
    }
}


/******************************************************************************
Function:    lcas_print_x509_to_string
Description:
    Extract from X509 certificate and
        write the output to a file.
    Simulair to `openssl x509 -in <file.pem> -text -noout

    Note: only for debugging purposes
Parameters:
    px509:    pointer to a x509 cert
Returns:
    void
******************************************************************************/
void lcas_print_x509_to_string (X509 * px509, char * output_file)
{
    char *            logstr = "\tlcas_print_x509_to_string()";

    if (px509)
    {
        FILE *            tmpFile = NULL;

        lcas_log(0,"%s\n", logstr);

        tmpFile = fopen(output_file, "a");
        X509_print_fp(tmpFile, px509);
        fclose (tmpFile);
    }
    else
    {
        lcas_log(0,"%s: no input X509 cert!\n", logstr);
    }
}

#endif /* LCAS_X509_UTILS_C */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/grid_credential_handling/x509_handling/lcas_x509_utils.c,v $
    $Date: 2010-05-03 10:42:50 $
    $Revision: 1.24 $
    $Author: okoeroo $
******************************************************************************/
