#ifndef H_CDW_DEBUG
#define H_CDW_DEBUG

#include <assert.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include "cdw_ncurses.h"

/**
   \file cdw_debug.h
   \brief Utilities used during development phase to print debug messages

   Some utilities that can replace calls to fprintf(stderr, ...).
*/







/**
 * \brief Print debug message - verbose version
 *
 * This macro behaves much like fprintf(stderr, ...) function,
 * caller only have to provide format string with conversion specifiers
 * and list of arguments for this format string.
 *
 * Each message is preceeded with name of function that called the macro.
 *
 * See "C: A Reference Manual", chapter 3.3.10 for more information
 * on variable argument lists in macros (it requires C99).
 */
#ifndef NDEBUG
#define cdw_vdm(...) fprintf(stderr, "%s():%d: ", __func__, __LINE__); fprintf(stderr, __VA_ARGS__); fflush(stderr);
#else
#define cdw_vdm(...)
#endif





/** \brief Print debug message - silent version */
#define cdw_sdm(...)





/**
  \brief Print debug message for ncurses return values - verbose version

  This macro behaves much like fprintf(stderr, ...) function,
  caller only have to provide format string with converesion specifiers
  and list of arguments for this format string.

  Each message is preceeded with name of function that called the macro.

  See "C: A Reference Manual", chapter 3.3.10 for more information
  on variable argument lists in macros (it requires C99).
*/
#ifndef NDEBUG
#define cdw_vdm_n(m_fname, m_rv, ...)					\
	{								\
		int m_e = errno;					\
		fprintf(stderr, "%s():%d: ", __func__, __LINE__);	\
		if (m_rv == E_OK) {					\
			fprintf(stderr, "INFO: %s() returns E_OK %s\n", m_fname, __VA_ARGS__); \
		} else {						\
			fprintf(stderr, "ERROR: %s() returns %s %s\n", m_fname, cdw_ncurses_error_string(rv), __VA_ARGS__); \
			if (m_rv == E_SYSTEM_ERROR) {			\
				fprintf(stderr, "%s():%d: ", __func__, __LINE__); \
				fprintf(stderr, "ERROR: %s(): error = %s\n", m_fname, strerror(m_e)); \
			}						\
		}							\
		fflush(stderr);						\
	}
#else
#define cdw_vdm_n(m_fname, m_rv, ...)
#endif





/**
  \brief Print debug message for ncurses return values - less verbose version

  This macro behaves much like fprintf(stderr, ...) function,
  caller only have to provide format string with converesion specifiers
  and list of arguments for this format string.

  If returned value is E_OK, nothing is printed.

  Each message is preceeded with name of function that called the macro.

  See "C: A Reference Manual", chapter 3.3.10 for more information
  on variable argument lists in macros (it requires C99).
*/
#ifndef NDEBUG
#define cdw_vdm_n2(m_fname, m_rv, ...)					\
	{								\
		int m_e = errno;					\
		if (m_rv == E_OK) {					\
			; /* Be silent about no-error situations */	\
		} else {						\
			fprintf(stderr, "%s():%d: ", __func__, __LINE__);	\
			fprintf(stderr, "ERROR: %s() returns %s %s\n", m_fname, cdw_ncurses_error_string(rv), __VA_ARGS__); \
			if (m_rv == E_SYSTEM_ERROR) {			\
				fprintf(stderr, "%s():%d: ", __func__, __LINE__); \
				fprintf(stderr, "ERROR: %s(): error = %s\n", m_fname, strerror(m_e)); \
			}						\
			fflush(stderr);					\
		}							\
	}
#else
#define cdw_vdm_n2(m_fname, m_rv, ...)
#endif





/** \brief Print debug message for ncurses return values - silent version */
#define cdw_sdm_n(...)





/**
 * \brief Print literal cdw return value - verbose version
 *
 * Print literal corresponding to integer value of given variable of type
 * cdw_rv_t.
 *
 * \param func - name of function that returns a cdw_rv_t value ('char *' string)
 * \param val - value of type cdw_rv_t that should be interpreted by this macro
 */
#ifndef NDEBUG
#define cdw_vcrv(func, val) \
if (val == CDW_OK) { \
	cdw_vdm("%s() returns CDW_OK\n", func); \
} else if (val == CDW_NO) { \
	cdw_vdm("%s() returns CDW_NO\n", func); \
} else if (val == CDW_CANCEL) { \
	cdw_vdm("%s() returns CDW_CANCEL\n", func); \
} else if (val == CDW_GEN_ERROR) { \
	cdw_vdm("%s() returns CDW_GEN_ERROR\n", func); \
} else if (val == CDW_MEM_ERROR) { \
	cdw_vdm("%s() returns CDW_MEM_ERROR\n", func); \
} else if (val == CDW_SYS_ERROR) { \
	cdw_vdm("%s() returns CDW_SYS_ERROR\n", func); \
} else if (val == CDW_FILE_INVALID) { \
	cdw_vdm("%s() returns CDW_FILE_INVALID\n", func); \
} else if (val == CDW_ARG_ERROR) { \
	cdw_vdm("%s() returns CDW_ARG_ERROR\n", func); \
} else if (val == CDW_FAILSAFE_MODE) { \
	cdw_vdm("%s() returns CDW_FAILSAFE_MODE\n", func); \
} else { \
	cdw_vdm("%s() returns unknown crv value: %d\n", func, val); \
}
#else
#define cdw_vcrv(func, val)
#endif




/** \brief Print literal cdw return value - silent version of cdw_vcrv */
#define cdw_scrv(func, val)





/**
 * \brief Assert macro with message
 */
#ifndef NDEBUG
#define cdw_assert(expr, ...)                   \
if (! (expr)) {                                 \
 fprintf(stderr, "\n\nassertion failed in:\n"); \
 fprintf(stderr, "file %s\n", __FILE__);        \
 fprintf(stderr, "line %d\n", __LINE__);        \
 cdw_vdm (__VA_ARGS__);                         \
 fprintf(stderr, "\n\n");                       \
assert (expr);                                  \
}
#else
#define cdw_assert(expr, ...)
#endif


/**
 * \brief Assert macro with message - version for unit tests
 */
#ifdef CDW_UNIT_TEST_CODE
#define cdw_assert_test(expr, ...)				\
	if (! (expr)) {						\
		fprintf(stderr, "\n\nassertion failed in:\n");	\
		fprintf(stderr, "file %s\n", __FILE__);			\
		fprintf(stderr, "line %d\n", __LINE__);			\
		fprintf(stderr, "%s():%d: ", __func__, __LINE__); fprintf(stderr, __VA_ARGS__); \
		fprintf(stderr, "\n\n");				\
		exit(0);						\
	}
#else
#define cdw_assert_test(expr, ...)
#endif


#endif /* H_CDW_DEBUG */
