// Copyright (C) 2023 JiDe Zhang <zhangjide@deepin.org>.
// SPDX-License-Identifier: Apache-2.0 OR LGPL-3.0-only OR GPL-2.0-only OR GPL-3.0-only

#include "wqmldynamiccreator_p.h"

#include <QJSValue>
#include <QQuickItem>
#include <QQmlInfo>
#include <private/qqmlcomponent_p.h>

WAYLIB_SERVER_BEGIN_NAMESPACE

WAbstractCreatorComponent::WAbstractCreatorComponent(QObject *parent)
    : QObject(parent)
{

}

void WAbstractCreatorComponent::remove(QSharedPointer<WQmlCreatorDelegateData>)
{

}

void WAbstractCreatorComponent::remove(QSharedPointer<WQmlCreatorData>)
{

}

QList<QSharedPointer<WQmlCreatorDelegateData>> WAbstractCreatorComponent::datas() const
{
    return {};
}

WQmlCreator *WAbstractCreatorComponent::creator() const
{
    return m_creator;
}

void WAbstractCreatorComponent::setCreator(WQmlCreator *newCreator)
{
    if (m_creator == newCreator)
        return;
    auto oldCreator = m_creator;
    m_creator = newCreator;
    creatorChange(oldCreator, newCreator);

    Q_EMIT creatorChanged();
}

void WAbstractCreatorComponent::creatorChange(WQmlCreator *oldCreator, WQmlCreator *newCreator)
{
    if (oldCreator)
        oldCreator->removeDelegate(this);

    if (newCreator)
        newCreator->addDelegate(this);
}

void WAbstractCreatorComponent::notifyCreatorObjectAdded(WQmlCreator *creator, QObject *object,
                                                         const QJSValue &initialProperties)
{
    Q_EMIT creator->objectAdded(this, object, initialProperties);
}

void WAbstractCreatorComponent::notifyCreatorObjectRemoved(WQmlCreator *creator, QObject *object,
                                                           const QJSValue &initialProperties)
{
    Q_EMIT creator->objectRemoved(this, object, initialProperties);
}

WQmlCreatorDataWatcher::WQmlCreatorDataWatcher(QObject *parent)
    : WAbstractCreatorComponent(parent)
{

}

QSharedPointer<WQmlCreatorDelegateData> WQmlCreatorDataWatcher::add(QSharedPointer<WQmlCreatorData> data)
{
    Q_EMIT added(data->owner, data->properties);
    return nullptr;
}

void WQmlCreatorDataWatcher::remove(QSharedPointer<WQmlCreatorData> data)
{
    Q_EMIT removed(data->owner, data->properties);
}

WQmlCreatorComponent::WQmlCreatorComponent(QObject *parent)
    : WAbstractCreatorComponent(parent)
{

}

WQmlCreatorComponent::~WQmlCreatorComponent()
{
    if (m_creator)
        m_creator->removeDelegate(this);

    clear();
}

bool WQmlCreatorComponent::checkByChooser(const QJSValue &properties) const
{
    if (m_chooserRole.isEmpty())
        return true;
    return properties.property(m_chooserRole).toVariant() == m_chooserRoleValue;
}

QQmlComponent *WQmlCreatorComponent::delegate() const
{
    return m_delegate;
}

void WQmlCreatorComponent::setDelegate(QQmlComponent *component)
{
    if (m_delegate) {
        qmlEngine(this)->throwError(QJSValue::GenericError, "Property 'delegate' can only be assigned once.");
        return;
    }

    m_delegate = component;
}

QSharedPointer<WQmlCreatorDelegateData> WQmlCreatorComponent::add(QSharedPointer<WQmlCreatorData> data)
{
    if (!checkByChooser(data->properties))
        return nullptr;

    QSharedPointer<WQmlCreatorDelegateData> d(new WQmlCreatorDelegateData());
    d->data = data;
    m_datas << d;
    create(d);

    return d;
}

void WQmlCreatorComponent::remove(QSharedPointer<WQmlCreatorData> data)
{
    for (const auto &d : std::as_const(data->delegateDatas)) {
        if (d.first != this)
            continue;
        if (d.second)
            remove(d.second.toStrongRef());
    }
}

void WQmlCreatorComponent::destroy(QSharedPointer<WQmlCreatorDelegateData> data)
{
    if (data->object) {
        auto obj = data->object.get();
        data->object.clear();
        const QJSValue p = data->data.lock()->properties;
        Q_EMIT objectRemoved(obj, p);
        notifyCreatorObjectRemoved(m_creator, obj, p);

        if (m_autoDestroy)
            obj->deleteLater();
    }
}

void WQmlCreatorComponent::remove(QSharedPointer<WQmlCreatorDelegateData> data)
{
    bool ok = m_datas.removeOne(data);
    Q_ASSERT(ok);
    destroy(data);
}

void WQmlCreatorComponent::clear()
{
    for (auto d : std::as_const(m_datas)) {
        d->data.lock()->delegateDatas.removeOne({this, d});
        destroy(d);
    }

    m_datas.clear();
}

void WQmlCreatorComponent::reset()
{
    clear();

    if (m_creator)
        m_creator->createAll(this);
}

void WQmlCreatorComponent::create(QSharedPointer<WQmlCreatorDelegateData> data)
{
    auto parent = m_parent ? m_parent : QObject::parent();

    if (data->object)
        destroy(data);

    Q_ASSERT(data->data);
    Q_ASSERT(m_delegate);

    auto d = QQmlComponentPrivate::get(m_delegate);
#if QT_VERSION >= QT_VERSION_CHECK(6, 5, 0)
    if (d->state.isCompletePending()) {
        QMetaObject::invokeMethod(this, "create", Qt::QueuedConnection, data, parent, data->data.lock()->properties);
#else
    if (d->state.completePending) {
        QMetaObject::invokeMethod(this, "create", Qt::QueuedConnection,
                                  Q_ARG(QSharedPointer<WQmlCreatorDelegateData>, data),
                                  Q_ARG(QObject *, parent),
                                  Q_ARG(const QJSValue &, data->data.lock()->properties));
#endif
    } else {
        create(data, parent, data->data.lock()->properties);
    }
}

void WQmlCreatorComponent::create(QSharedPointer<WQmlCreatorDelegateData> data, QObject *parent, const QJSValue &initialProperties)
{
    auto d = QQmlComponentPrivate::get(m_delegate);

#if QT_VERSION >= QT_VERSION_CHECK(6, 5, 0)
    Q_ASSERT(!d->state.isCompletePending());
#else
    Q_ASSERT(!d->state.completePending);
#endif
    // Don't use QVariantMap instead of QJSValue, because initial properties may be
    // contains some QObject property , if that QObjects is destroyed in future,
    // the QVariantMap's property would not update, you will get a invalid QObject pointer
    // if you using it after it's destroyed, but the QJSValue will watching that QObjects,
    // you will get a null pointer if you using after it's destroyed.
    const auto tmp = qvariant_cast<QVariantMap>(initialProperties.toVariant());

#if QT_VERSION >= QT_VERSION_CHECK(6, 5, 0)
    data->object = d->createWithProperties(parent, tmp, qmlContext(this));
#else
    // The `createWithInitialProperties` provided by QQmlComponent cannot set parent
    // during the creation process. use `setParent` will too late as the creation has
    // complete, lead to the windows of WOutputRenderWindow get empty...

    // for Qt 6.5 The createWithInitialProperties with the parent parameter provided by
    // QQmlComponentPrivate can solve this problem.
    // Qt 6.4 requires beginCreate -> setInitialProperties/setParent -> completeCreate

    QObject *rv = m_delegate->beginCreate(qmlContext(this));
    if (rv) {
        m_delegate->setInitialProperties(rv, tmp);
        rv->setParent(parent);
        if (auto item = qobject_cast<QQuickItem*>(rv))
            item->setParentItem(qobject_cast<QQuickItem*>(parent));
        m_delegate->completeCreate();
        if (!d->requiredProperties().empty()) {
            for (const auto &unsetRequiredProperty : std::as_const(d->requiredProperties())) {
                const QQmlError error = QQmlComponentPrivate::unsetRequiredPropertyToQQmlError(unsetRequiredProperty);
                qmlWarning(rv, error);
            }
            d->requiredProperties().clear();
            rv->deleteLater();
            rv = nullptr;
        }
    }
    data->object = rv;
#endif

    if (data->object) {
        Q_EMIT objectAdded(data->object, initialProperties);
        notifyCreatorObjectAdded(m_creator, data->object, initialProperties);
    }
}

QObject *WQmlCreatorComponent::parent() const
{
    return m_parent;
}

void WQmlCreatorComponent::setParent(QObject *newParent)
{
    if (m_parent == newParent)
        return;
    m_parent = newParent;
    Q_EMIT parentChanged();
}

QList<QSharedPointer<WQmlCreatorDelegateData> > WQmlCreatorComponent::datas() const
{
    return m_datas;
}

// If autoDestroy is disabled, can using this interface to manual manager the object's life.
void WQmlCreatorComponent::destroyObject(QObject *object)
{
    object->deleteLater();
}

QString WQmlCreatorComponent::chooserRole() const
{
    return m_chooserRole;
}

void WQmlCreatorComponent::setChooserRole(const QString &newChooserRole)
{
    if (m_chooserRole == newChooserRole)
        return;
    m_chooserRole = newChooserRole;
    reset();

    Q_EMIT chooserRoleChanged();
}

QVariant WQmlCreatorComponent::chooserRoleValue() const
{
    return m_chooserRoleValue;
}

void WQmlCreatorComponent::setChooserRoleValue(const QVariant &newChooserRoleValue)
{
    if (m_chooserRoleValue == newChooserRoleValue)
        return;
    m_chooserRoleValue = newChooserRoleValue;
    reset();

    Q_EMIT chooserRoleValueChanged();
}

bool WQmlCreatorComponent::autoDestroy() const
{
    return m_autoDestroy;
}

void WQmlCreatorComponent::setAutoDestroy(bool newAutoDestroy)
{
    if (m_autoDestroy == newAutoDestroy)
        return;
    m_autoDestroy = newAutoDestroy;
    Q_EMIT autoDestroyChanged();
}

WQmlCreator::WQmlCreator(QObject *parent)
    : QObject{parent}
{

}

WQmlCreator::~WQmlCreator()
{
    clear(false);

    for (auto delegate : std::as_const(m_delegates)) {
        Q_ASSERT(delegate->creator() == this);
        delegate->setCreator(nullptr);
    }
}

QList<WAbstractCreatorComponent*> WQmlCreator::delegates() const
{
    return m_delegates;
}

int WQmlCreator::count() const
{
    return m_datas.size();
}

void WQmlCreator::add(const QJSValue &initialProperties)
{
    add(nullptr, initialProperties);
}

void WQmlCreator::add(QObject *owner, const QJSValue &initialProperties)
{
    QSharedPointer<WQmlCreatorData> data(new WQmlCreatorData());
    data->owner = owner;
    data->properties = initialProperties;

    for (auto delegate : m_delegates) {
        if (auto d = delegate->add(data.toWeakRef()))
            data->delegateDatas.append({delegate, d});
    }

    m_datas << data;

    if (owner) {
        connect(owner, &QObject::destroyed, this, [this] {
            bool ok = removeByOwner(sender());
            Q_ASSERT(ok);
        });
    }

    Q_EMIT countChanged();
}

bool WQmlCreator::removeIf(QJSValue function)
{
    return remove(indexOf(function));
}

bool WQmlCreator::removeByOwner(QObject *owner)
{
    return remove(indexOfOwner(owner));
}

void WQmlCreator::clear(bool notify)
{
    if (m_datas.isEmpty())
        return;

    for (auto data : std::as_const(m_datas))
        destroy(data);

    m_datas.clear();

    if (notify)
        Q_EMIT countChanged();
}

QObject *WQmlCreator::get(int index) const
{
    if (index < 0 || index >= m_datas.size())
        return nullptr;

    auto data = m_datas.at(index);
    if (data->delegateDatas.isEmpty())
        return nullptr;
    return data->delegateDatas.first().second.lock()->object.get();
}

QObject *WQmlCreator::get(WAbstractCreatorComponent *delegate, int index) const
{
    if (index < 0 || index >= m_datas.size())
        return nullptr;

    auto data = m_datas.at(index);
    for (const auto &d : std::as_const(data->delegateDatas)) {
        if (d.first != delegate)
            continue;
        return d.second ? d.second.lock()->object.get() : nullptr;
    }

    return nullptr;
}

QObject *WQmlCreator::getIf(QJSValue function) const
{
    for (auto delegate : m_delegates) {
        auto obj = getIf(delegate, function);
        if (obj)
            return obj;
    }
    return nullptr;
}

QObject *WQmlCreator::getIf(WAbstractCreatorComponent *delegate, QJSValue function) const
{
    return get(delegate, indexOf(function));
}

QObject *WQmlCreator::getByOwner(QObject *owner) const
{
    for (auto delegate : m_delegates) {
        auto obj = getByOwner(delegate, owner);
        if (obj)
            return obj;
    }
    return nullptr;
}

QObject *WQmlCreator::getByOwner(WAbstractCreatorComponent *delegate, QObject *owner) const
{
    return get(delegate, indexOfOwner(owner));
}

void WQmlCreator::destroy(QSharedPointer<WQmlCreatorData> data)
{
    if (data->owner)
        data->owner->disconnect(this);

    for (auto delegate : m_delegates)
        delegate->remove(data);
}

bool WQmlCreator::remove(int index)
{
    if (index < 0 || index >= m_datas.size())
        return false;
    auto data = m_datas.takeAt(index);
    destroy(data);

    Q_EMIT countChanged();

    return true;
}

int WQmlCreator::indexOfOwner(QObject *owner) const
{
    for (int i = 0; i < m_datas.size(); ++i) {
        if (m_datas.at(i)->owner == owner)
            return i;
    }

    return -1;
}

int WQmlCreator::indexOf(QJSValue function) const
{
    auto engine = qmlEngine(this);
    Q_ASSERT(engine);

    for (int i = 0; i < m_datas.size(); ++i) {
        if (function.call({engine->toScriptValue(m_datas.at(i)->properties)}).toBool())
            return i;
    }

    return -1;
}

void WQmlCreator::addDelegate(WAbstractCreatorComponent *delegate)
{
    Q_ASSERT(!m_delegates.contains(delegate));
    m_delegates.append(delegate);
    createAll(delegate);
}

void WQmlCreator::createAll(WAbstractCreatorComponent *delegate)
{
    for (const auto &data : std::as_const(m_datas)) {
        if (auto d = delegate->add(data))
            data->delegateDatas.append({delegate, d});
    }
}

void WQmlCreator::removeDelegate(WAbstractCreatorComponent *delegate)
{
    bool ok = m_delegates.removeOne(delegate);
    Q_ASSERT(ok);

    for (auto d : delegate->datas()) {
        bool ok = d->data.lock()->delegateDatas.removeOne({delegate, d});
        Q_ASSERT(ok);
    }
}

WAYLIB_SERVER_END_NAMESPACE
