// SPDX-FileCopyrightText: 2025 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include <gtest/gtest.h>
#include <QTemporaryDir>
#include <QFile>
#include <QTextStream>
#include <QUrl>
#include <QDir>

#include "stubext.h"

#include <dfm-base/base/schemefactory.h>
#include <dfm-base/file/local/syncfileinfo.h>
#include <dfm-base/file/local/asyncfileinfo.h>
#include <dfm-base/file/local/desktopfileinfo.h>
#include <dfm-base/interfaces/fileinfo.h>
#include <dfm-base/dfm_global_defines.h>
#include <dfm-io/dfile.h>

// Include the target files
#include "fileoperations/fileoperationutils/filenameutils.h"

DFMBASE_USE_NAMESPACE
DPFILEOPERATIONS_USE_NAMESPACE

class TestFileNameUtils : public testing::Test
{
public:
    void SetUp() override
    {
        // Register FileInfo classes like in core.cpp
        UrlRoute::regScheme(Global::Scheme::kFile, "/");
        UrlRoute::regScheme(Global::Scheme::kAsyncFile, "/");
        InfoFactory::regClass<SyncFileInfo>(Global::Scheme::kFile);
        InfoFactory::regClass<AsyncFileInfo>(Global::Scheme::kAsyncFile);

        // Create temporary directory for test files
        tempDir = std::make_unique<QTemporaryDir>();
        ASSERT_TRUE(tempDir->isValid());

        tempDirPath = tempDir->path();
        tempDirUrl = QUrl::fromLocalFile(tempDirPath);

        // Create target directory FileInfo
        targetDirInfo = InfoFactory::create<FileInfo>(tempDirUrl);
        ASSERT_TRUE(targetDirInfo);
    }

    void TearDown() override
    {
        stub.clear();
        targetDirInfo.reset();
        tempDir.reset();
    }

protected:
    /*!
     * \brief Create a test file with given name and return its FileInfo
     */
    FileInfoPointer createTestFile(const QString &fileName)
    {
        QString filePath = tempDirPath + QDir::separator() + fileName;
        QFile file(filePath);
        EXPECT_TRUE(file.open(QIODevice::WriteOnly));
        QTextStream stream(&file);
        stream << "test content";
        file.close();

        QUrl fileUrl = QUrl::fromLocalFile(filePath);
        return InfoFactory::create<FileInfo>(fileUrl);
    }

    /*!
     * \brief Create a test directory with given name and return its FileInfo
     */
    FileInfoPointer createTestDir(const QString &dirName)
    {
        QString dirPath = tempDirPath + QDir::separator() + dirName;
        QDir().mkpath(dirPath);

        QUrl dirUrl = QUrl::fromLocalFile(dirPath);
        return InfoFactory::create<FileInfo>(dirUrl);
    }

    stub_ext::StubExt stub;
    std::unique_ptr<QTemporaryDir> tempDir;
    QString tempDirPath;
    QUrl tempDirUrl;
    FileInfoPointer targetDirInfo;
};

// ========== FileNameParser Tests ==========

TEST_F(TestFileNameUtils, FileNameParser_parseFileName_RegularFile)
{
    // Test regular file with extension
    auto fileInfo = createTestFile("document.txt");
    ASSERT_TRUE(fileInfo);

    FileNameComponents components = FileNameParser::parseFileName(fileInfo);

    EXPECT_EQ(components.fileName, "document.txt");
    EXPECT_EQ(components.baseName, "document");
    EXPECT_EQ(components.completeSuffix, "txt");
}

TEST_F(TestFileNameUtils, FileNameParser_parseFileName_MultipleExtensions)
{
    // Test file with multiple extensions
    auto fileInfo = createTestFile("archive.tar.gz");
    ASSERT_TRUE(fileInfo);

    FileNameComponents components = FileNameParser::parseFileName(fileInfo);

    EXPECT_EQ(components.fileName, "archive.tar.gz");
    EXPECT_EQ(components.baseName, "archive");
    EXPECT_EQ(components.completeSuffix, "tar.gz");
}

TEST_F(TestFileNameUtils, FileNameParser_parseFileName_NoExtension)
{
    // Test file without extension
    auto fileInfo = createTestFile("README");
    ASSERT_TRUE(fileInfo);

    FileNameComponents components = FileNameParser::parseFileName(fileInfo);

    EXPECT_EQ(components.fileName, "README");
    EXPECT_EQ(components.baseName, "README");
    EXPECT_EQ(components.completeSuffix, "");
}

TEST_F(TestFileNameUtils, FileNameParser_parseFileName_Directory)
{
    // Test directory
    auto dirInfo = createTestDir("testdir");
    ASSERT_TRUE(dirInfo);

    FileNameComponents components = FileNameParser::parseFileName(dirInfo);

    EXPECT_EQ(components.fileName, "testdir");
    EXPECT_EQ(components.baseName, "testdir");
    EXPECT_EQ(components.completeSuffix, "");
}

TEST_F(TestFileNameUtils, FileNameParser_parseFileName_HiddenFileWithExtension)
{
    // Test hidden file with extension
    auto fileInfo = createTestFile(".config.json");
    ASSERT_TRUE(fileInfo);

    FileNameComponents components = FileNameParser::parseFileName(fileInfo);

    EXPECT_EQ(components.fileName, ".config.json");
    EXPECT_EQ(components.baseName, ".config");
    EXPECT_EQ(components.completeSuffix, "json");
}

TEST_F(TestFileNameUtils, FileNameParser_parseFileName_HiddenFileNoExtension)
{
    // Test hidden file without extension
    auto fileInfo = createTestFile(".bashrc");
    ASSERT_TRUE(fileInfo);

    FileNameComponents components = FileNameParser::parseFileName(fileInfo);

    EXPECT_EQ(components.fileName, ".bashrc");
    EXPECT_EQ(components.baseName, ".bashrc");
    EXPECT_EQ(components.completeSuffix, "");
}

TEST_F(TestFileNameUtils, FileNameParser_parseFileName_HiddenFileMultipleExtensions)
{
    // Test hidden file with multiple extensions
    auto fileInfo = createTestFile(".backup.tar.gz");
    ASSERT_TRUE(fileInfo);

    FileNameComponents components = FileNameParser::parseFileName(fileInfo);

    EXPECT_EQ(components.fileName, ".backup.tar.gz");
    EXPECT_EQ(components.baseName, ".backup");
    EXPECT_EQ(components.completeSuffix, "tar.gz");
}

TEST_F(TestFileNameUtils, FileNameParser_parseFileName_NullFileInfo)
{
    // Test with null file info
    FileNameComponents components = FileNameParser::parseFileName(nullptr);

    EXPECT_TRUE(components.fileName.isEmpty());
    EXPECT_TRUE(components.baseName.isEmpty());
    EXPECT_TRUE(components.completeSuffix.isEmpty());
}



// ========== FileExistenceChecker Tests ==========

TEST_F(TestFileNameUtils, FileExistenceChecker_isValidTargetDirectory_ValidDirectory)
{
    // Test with valid directory
    bool result = FileExistenceChecker::isValidTargetDirectory(targetDirInfo);
    EXPECT_TRUE(result);
}

TEST_F(TestFileNameUtils, FileExistenceChecker_isValidTargetDirectory_NullPointer)
{
    // Test with null pointer
    bool result = FileExistenceChecker::isValidTargetDirectory(nullptr);
    EXPECT_FALSE(result);
}

TEST_F(TestFileNameUtils, FileExistenceChecker_isValidTargetDirectory_NonExistentDirectory)
{
    // Test with non-existent directory
    QUrl nonExistentUrl = QUrl::fromLocalFile("/nonexistent/path");
    auto nonExistentInfo = InfoFactory::create<FileInfo>(nonExistentUrl);

    bool result = FileExistenceChecker::isValidTargetDirectory(nonExistentInfo);
    EXPECT_FALSE(result);
}

TEST_F(TestFileNameUtils, FileExistenceChecker_isValidTargetDirectory_FileNotDirectory)
{
    // Test with file instead of directory
    auto fileInfo = createTestFile("test.txt");
    ASSERT_TRUE(fileInfo);

    bool result = FileExistenceChecker::isValidTargetDirectory(fileInfo);
    EXPECT_FALSE(result);
}

TEST_F(TestFileNameUtils, FileExistenceChecker_fileExists_ExistingFile)
{
    // Create a test file
    createTestFile("existing.txt");

    bool result = FileExistenceChecker::fileExists(targetDirInfo, "existing.txt");
    EXPECT_TRUE(result);
}

TEST_F(TestFileNameUtils, FileExistenceChecker_fileExists_NonExistingFile)
{
    // Test with non-existing file
    bool result = FileExistenceChecker::fileExists(targetDirInfo, "nonexistent.txt");
    EXPECT_FALSE(result);
}

TEST_F(TestFileNameUtils, FileExistenceChecker_fileExists_EmptyFileName)
{
    // Test with empty file name
    bool result = FileExistenceChecker::fileExists(targetDirInfo, "");
    EXPECT_FALSE(result);
}

TEST_F(TestFileNameUtils, FileExistenceChecker_fileExists_InvalidDirectory)
{
    // Test with invalid directory
    bool result = FileExistenceChecker::fileExists(nullptr, "test.txt");
    EXPECT_FALSE(result);
}

TEST_F(TestFileNameUtils, FileExistenceChecker_fileExists_ExistingDirectory)
{
    // Create a test directory
    createTestDir("testsubdir");

    bool result = FileExistenceChecker::fileExists(targetDirInfo, "testsubdir");
    EXPECT_TRUE(result);
}

// ========== ConflictNameGenerator Tests ==========

TEST_F(TestFileNameUtils, ConflictNameGenerator_generateCopySuffix_FirstCopy)
{
    // Test first copy (number = 0)
    QString suffix = ConflictNameGenerator::generateCopySuffix(0);
    EXPECT_TRUE(suffix.contains("copy"));
    EXPECT_FALSE(suffix.contains("1"));
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_generateCopySuffix_NumberedCopy)
{
    // Test numbered copy
    QString suffix = ConflictNameGenerator::generateCopySuffix(1);
    EXPECT_TRUE(suffix.contains("copy"));
    EXPECT_TRUE(suffix.contains("1"));

    suffix = ConflictNameGenerator::generateCopySuffix(5);
    EXPECT_TRUE(suffix.contains("copy"));
    EXPECT_TRUE(suffix.contains("5"));
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_constructFileName_WithExtension)
{
    // Test constructing file name with extension
    FileNameComponents components("document", "txt", "document.txt");
    QString copySuffix = " (copy)";

    QString result = ConflictNameGenerator::constructFileName(components, copySuffix);
    EXPECT_EQ(result, "document (copy).txt");
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_constructFileName_WithoutExtension)
{
    // Test constructing file name without extension
    FileNameComponents components("README", "", "README");
    QString copySuffix = " (copy)";

    QString result = ConflictNameGenerator::constructFileName(components, copySuffix);
    EXPECT_EQ(result, "README (copy)");
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_constructFileName_MultipleExtensions)
{
    // Test constructing file name with multiple extensions
    FileNameComponents components("archive", "tar.gz", "archive.tar.gz");
    QString copySuffix = " (copy 2)";

    QString result = ConflictNameGenerator::constructFileName(components, copySuffix);
    EXPECT_EQ(result, "archive (copy 2).tar.gz");
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_constructFileName_HiddenFile)
{
    // Test constructing hidden file name
    FileNameComponents components(".config", "json", ".config.json");
    QString copySuffix = " (copy)";

    QString result = ConflictNameGenerator::constructFileName(components, copySuffix);
    EXPECT_EQ(result, ".config (copy).json");
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_generateUniqueFileName_NoConflict)
{
    // Test generating unique name when no conflict exists
    FileNameComponents components("newfile", "txt", "newfile.txt");

    QString result = ConflictNameGenerator::generateUniqueFileName(components, targetDirInfo);
    EXPECT_TRUE(result.contains("newfile"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.endsWith(".txt"));
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_generateUniqueFileName_WithConflicts)
{
    // Create conflicting files
    createTestFile("conflict.txt");
    createTestFile("conflict (copy).txt");
    createTestFile("conflict (copy 1).txt");

    FileNameComponents components("conflict", "txt", "conflict.txt");

    QString result = ConflictNameGenerator::generateUniqueFileName(components, targetDirInfo);
    EXPECT_TRUE(result.contains("conflict"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.contains("2"));
    EXPECT_TRUE(result.endsWith(".txt"));
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_generateUniqueFileName_InvalidDirectory)
{
    // Test with invalid directory
    FileNameComponents components("test", "txt", "test.txt");

    QString result = ConflictNameGenerator::generateUniqueFileName(components, nullptr);
    EXPECT_TRUE(result.isEmpty());
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_generateUniqueFileName_EmptyBaseName)
{
    // Test with empty base name
    FileNameComponents components("", "txt", ".txt");

    QString result = ConflictNameGenerator::generateUniqueFileName(components, targetDirInfo);
    EXPECT_TRUE(result.isEmpty());
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_generateUniqueFileName_DirectoryConflict)
{
    // Test with directory conflicts
    createTestDir("testdir");
    createTestDir("testdir (copy)");

    FileNameComponents components("testdir", "", "testdir");

    QString result = ConflictNameGenerator::generateUniqueFileName(components, targetDirInfo);
    EXPECT_TRUE(result.contains("testdir"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.contains("1"));
}

TEST_F(TestFileNameUtils, ConflictNameGenerator_generateUniqueFileName_HiddenFileConflict)
{
    // Test with hidden file conflicts
    createTestFile(".hidden");
    createTestFile(".hidden (copy)");

    FileNameComponents components(".hidden", "", ".hidden");

    QString result = ConflictNameGenerator::generateUniqueFileName(components, targetDirInfo);
    EXPECT_TRUE(result.contains(".hidden"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.contains("1"));
}

// ========== FileNamingUtils Tests ==========

TEST_F(TestFileNameUtils, FileNamingUtils_generateNonConflictingName_RegularFile)
{
    // Test generating non-conflicting name for regular file
    auto sourceFile = createTestFile("source.txt");
    ASSERT_TRUE(sourceFile);

    QString result = FileNamingUtils::generateNonConflictingName(sourceFile, targetDirInfo);
    EXPECT_TRUE(result.contains("source"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.endsWith(".txt"));
}

TEST_F(TestFileNameUtils, FileNamingUtils_generateNonConflictingName_Directory)
{
    // Test generating non-conflicting name for directory
    auto sourceDir = createTestDir("sourcedir");
    ASSERT_TRUE(sourceDir);

    QString result = FileNamingUtils::generateNonConflictingName(sourceDir, targetDirInfo);
    EXPECT_TRUE(result.contains("sourcedir"));
    EXPECT_TRUE(result.contains("copy"));
}

TEST_F(TestFileNameUtils, FileNamingUtils_generateNonConflictingName_HiddenFile)
{
    // Test generating non-conflicting name for hidden file
    auto hiddenFile = createTestFile(".hidden.conf");
    ASSERT_TRUE(hiddenFile);

    QString result = FileNamingUtils::generateNonConflictingName(hiddenFile, targetDirInfo);
    EXPECT_TRUE(result.contains(".hidden"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.endsWith(".conf"));
}

TEST_F(TestFileNameUtils, FileNamingUtils_generateNonConflictingName_NullSourceInfo)
{
    // Test with null source info
    QString result = FileNamingUtils::generateNonConflictingName(nullptr, targetDirInfo);
    EXPECT_TRUE(result.isEmpty());
}

TEST_F(TestFileNameUtils, FileNamingUtils_generateNonConflictingName_NullTargetDir)
{
    // Test with null target directory
    auto sourceFile = createTestFile("test.txt");
    ASSERT_TRUE(sourceFile);

    QString result = FileNamingUtils::generateNonConflictingName(sourceFile, nullptr);
    EXPECT_TRUE(result.isEmpty());
}

TEST_F(TestFileNameUtils, FileNamingUtils_generateNonConflictingName_MultipleConflicts)
{
    // Create source file and multiple conflicts in target
    auto sourceFile = createTestFile("document.pdf");
    createTestFile("document.pdf");
    createTestFile("document (copy).pdf");
    createTestFile("document (copy 1).pdf");
    createTestFile("document (copy 2).pdf");

    // Create another temp directory as target
    QTemporaryDir targetTempDir;
    ASSERT_TRUE(targetTempDir.isValid());

    QUrl targetUrl = QUrl::fromLocalFile(targetTempDir.path());
    auto targetInfo = InfoFactory::create<FileInfo>(targetUrl);
    ASSERT_TRUE(targetInfo);

    // Create conflicting files in target directory
    QString targetPath = targetTempDir.path();
    QFile(targetPath + "/document.pdf").open(QIODevice::WriteOnly);
    QFile(targetPath + "/document (copy).pdf").open(QIODevice::WriteOnly);
    QFile(targetPath + "/document (copy 1).pdf").open(QIODevice::WriteOnly);

    QString result = FileNamingUtils::generateNonConflictingName(sourceFile, targetInfo);
    EXPECT_TRUE(result.contains("document"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.contains("2"));
    EXPECT_TRUE(result.endsWith(".pdf"));
}

// ========== Integration Tests ==========

TEST_F(TestFileNameUtils, Integration_CompleteWorkflow_RegularFile)
{
    // Test complete workflow: parse -> check -> generate
    auto sourceFile = createTestFile("integration.doc");
    ASSERT_TRUE(sourceFile);

    // Parse file name components
    FileNameComponents components = FileNameParser::parseFileName(sourceFile);
    EXPECT_EQ(components.fileName, "integration.doc");
    EXPECT_EQ(components.baseName, "integration");
    EXPECT_EQ(components.completeSuffix, "doc");

    // Check if target directory is valid
    EXPECT_TRUE(FileExistenceChecker::isValidTargetDirectory(targetDirInfo));

    // Generate unique name
    QString uniqueName = ConflictNameGenerator::generateUniqueFileName(components, targetDirInfo);
    EXPECT_TRUE(uniqueName.contains("integration"));
    EXPECT_TRUE(uniqueName.contains("copy"));
    EXPECT_TRUE(uniqueName.endsWith(".doc"));

    // Verify the generated name doesn't exist
    EXPECT_FALSE(FileExistenceChecker::fileExists(targetDirInfo, uniqueName));
}

TEST_F(TestFileNameUtils, Integration_CompleteWorkflow_HiddenFileWithMultipleExtensions)
{
    // Test complete workflow with complex hidden file
    auto hiddenFile = createTestFile(".backup.data.gz");
    ASSERT_TRUE(hiddenFile);

    // Use the main API
    QString result = FileNamingUtils::generateNonConflictingName(hiddenFile, targetDirInfo);
    EXPECT_TRUE(result.contains(".backup"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.endsWith(".data.gz"));

    // Verify it doesn't exist
    EXPECT_FALSE(FileExistenceChecker::fileExists(targetDirInfo, result));
}

// ========== Error Handling and Edge Cases ==========

TEST_F(TestFileNameUtils, EdgeCase_VeryLongFileName)
{
    // Test with very long file name (within filesystem limits)
    QString longName = QString("a").repeated(200) + ".txt";
    auto longFile = createTestFile(longName);
    ASSERT_TRUE(longFile);

    QString result = FileNamingUtils::generateNonConflictingName(longFile, targetDirInfo);
    EXPECT_FALSE(result.isEmpty());
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.endsWith(".txt"));
}

TEST_F(TestFileNameUtils, EdgeCase_SpecialCharactersInFileName)
{
    // Test with special characters (that are valid in filenames)
    auto specialFile = createTestFile("file-with_special@chars#.txt");
    ASSERT_TRUE(specialFile);

    QString result = FileNamingUtils::generateNonConflictingName(specialFile, targetDirInfo);
    EXPECT_FALSE(result.isEmpty());
    EXPECT_TRUE(result.contains("file-with_special@chars#"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.endsWith(".txt"));
}

TEST_F(TestFileNameUtils, EdgeCase_FileNameWithSpaces)
{
    // Test with file name containing spaces
    auto spaceFile = createTestFile("file with spaces.txt");
    ASSERT_TRUE(spaceFile);

    QString result = FileNamingUtils::generateNonConflictingName(spaceFile, targetDirInfo);
    EXPECT_FALSE(result.isEmpty());
    EXPECT_TRUE(result.contains("file with spaces"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.endsWith(".txt"));
}

TEST_F(TestFileNameUtils, EdgeCase_NumbersInFileName)
{
    // Test with file name containing numbers
    auto numberFile = createTestFile("file123.txt");
    ASSERT_TRUE(numberFile);

    QString result = FileNamingUtils::generateNonConflictingName(numberFile, targetDirInfo);
    EXPECT_FALSE(result.isEmpty());
    EXPECT_TRUE(result.contains("file123"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.endsWith(".txt"));
}

// ========== Stubbing Tests for Error Conditions ==========

TEST_F(TestFileNameUtils, Stub_DFileExistsFailure)
{
    // Stub DFile::exists to always return false to test error handling
    stub.set_lamda(static_cast<bool (DFMIO::DFile::*)() const>(&DFMIO::DFile::exists),
                   []() -> bool {
                       __DBG_STUB_INVOKE__
                       return false;
                   });

    // This should affect FileExistenceChecker::isValidTargetDirectory
    bool result = FileExistenceChecker::isValidTargetDirectory(targetDirInfo);
    EXPECT_FALSE(result);
}

TEST_F(TestFileNameUtils, Stub_FileInfoNameOfFailure)
{
    // Test behavior when FileInfo::nameOf returns empty strings
    auto testFile = createTestFile("test.txt");
    ASSERT_TRUE(testFile);

    stub.set_lamda(VADDR(SyncFileInfo, nameOf),
                   [](FileInfo *, NameInfoType type) -> QString {
                       __DBG_STUB_INVOKE__
                       return QString();   // Return empty string
                   });

    FileNameComponents components = FileNameParser::parseFileName(testFile);
    EXPECT_TRUE(components.fileName.isEmpty());
    EXPECT_TRUE(components.baseName.isEmpty());
    EXPECT_TRUE(components.completeSuffix.isEmpty());
}

// ========== Performance and Stress Tests ==========

TEST_F(TestFileNameUtils, Performance_ManyConflicts)
{
    // Create many conflicting files to test performance
    const int conflictCount = 50;

    // Create base file
    auto sourceFile = createTestFile("performance.txt");
    ASSERT_TRUE(sourceFile);

    // Create many conflicts
    for (int i = 0; i < conflictCount; ++i) {
        QString conflictName;
        if (i == 0) {
            conflictName = "performance.txt";
        } else if (i == 1) {
            conflictName = "performance (copy).txt";
        } else {
            conflictName = QString("performance (copy %1).txt").arg(i - 1);
        }
        createTestFile(conflictName);
    }

    // This should still work efficiently
    QString result = FileNamingUtils::generateNonConflictingName(sourceFile, targetDirInfo);
    EXPECT_FALSE(result.isEmpty());
    EXPECT_TRUE(result.contains("performance"));
    EXPECT_TRUE(result.contains("copy"));
    EXPECT_TRUE(result.contains(QString::number(conflictCount - 1)));
    EXPECT_TRUE(result.endsWith(".txt"));
}
