package utils

import (
	"context"
	"fmt"
	"pkg.deepin.io/daemon/sync/infrastructure/storage"
	"sync"
	"time"
)

// CloudStorage wrap storage
type CloudStorage struct {
	core  *storage.Storage
	token storage.TokenSource
	hwID  string
}

const (
	// background file upload cost many time
	cloudTimeout = time.Second * 30
)

var (
	_cloud       *CloudStorage
	_cloudLocker sync.Mutex
)

func CloudConnect(token storage.TokenSource, hwID string) (*CloudStorage, error) {
	if nil == token {
		return nil, fmt.Errorf("empty token")
	}
	_cloudLocker.Lock()
	defer _cloudLocker.Unlock()
	if _cloud == nil {
		s, err := storage.NewStorage(token, hwID)
		if err != nil {
			return nil, err
		}
		_cloud = &CloudStorage{
			core:  s,
			token: token,
			hwID:  hwID,
		}
	}
	return _cloud, nil
}

func (cloud *CloudStorage) GetLatest(key string) (*storage.GetResult, error) {
	return cloud.Get(key, &storage.GetOptions{Version: storage.VersionLatest})
}

func (cloud *CloudStorage) Get(key string, opt *storage.GetOptions) (*storage.GetResult, error) {
	ctx, cancel := context.WithTimeout(context.Background(), cloudTimeout)
	defer cancel()
	return cloud.core.Get(ctx, key, opt)
}

func (cloud *CloudStorage) Set(key string, data []byte, opt *storage.SetOptions) (string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), cloudTimeout)
	defer cancel()
	return cloud.core.Set(ctx, key, data, opt)
}

func GetDataPath(name string) string {
	return fmt.Sprintf("%s/data", name)
}

func GetFilePath(name, basename string) string {
	return fmt.Sprintf("%s/files/%s", name, basename)
}
