/* makedoc.c -- make doc.c and funs.h from input files.

   Copyright 1993-2026 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <https://www.gnu.org/licenses/>.

   Originally written by Brian Fox. */

/* This program grovels the contents of the source files passed as arguments
   and writes out a file of function pointers and documentation strings, and
   a header file which describes the contents.  This only does the functions
   declared with DECLARE_INFO_COMMAND. */

#include "system.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>

/* From gnulib */
#include "xalloc.h"

#if !defined (whitespace)
#  define whitespace(c) ((c == ' ') || (c == '\t'))
#endif /* !whitespace */

#if !defined (whitespace_or_newline)
#  define whitespace_or_newline(c) (whitespace (c) \
                                    || (c == '\n') || (c == '\r'))
#endif /* !whitespace_or_newline */


char *program_name = "makedoc";

static void fatal_file_error (char *filename);

/* Name of the header file which receives the declarations of functions. */
static char *funs_filename = "funs.h";

/* Name of the documentation to function pointer file. */
static char *doc_filename = "doc.c";

static char *doc_header[] = {
  "/* doc.c -- Generated structure containing function names and doc strings.",
  "",
  "   This file was automatically made from various source files with the",
  "   command `%s'.",
  "   DO NOT EDIT THIS FILE, only `%s.c'.",
  NULL
};

static char *doc_header_1[] = {
  "   An entry in the array FUNCTION_DOC_ARRAY is made for each command",
  "   found in the above files; each entry consists of a function pointer,",
  "   a string which is the user-visible name of the function,",
  "   and a string which documents its purpose. */",
  "",
  "#include \"info.h\"",
  "#include \"doc.h\"",
  "#include \"funs.h\"",
  "",
  "InfoCommand function_doc_array[] = {",
  "",
  NULL
};

#define DECLARATION_STRING "\nDECLARE_INFO_COMMAND"

static void process_one_file (char *filename, FILE *doc_stream,
                              FILE *funs_stream);
static FILE *must_fopen (char *filename, char *mode);
static void init_func_key (unsigned int val);
static unsigned int next_func_key (void);

int
main (int argc, char **argv)
{
  register int i;
  FILE *funs_stream, *doc_stream;

#if STRIP_DOT_EXE
  {
    char *dot = strrchr (argv[0], '.');

    if (dot && FILENAME_CMP (dot, ".exe") == 0)
      *dot = 0;
  }
#endif

  /* The order of these calls depends exactly on the order in the
     Makefile.{in,am}, or they might fail on filesystems with
     high-precision times; see also the fclose calls below.  */
  funs_stream = must_fopen (funs_filename, "w");
  doc_stream = must_fopen (doc_filename, "w");

  fprintf (funs_stream,
      "/* %s -- Generated declarations for Info commands. */\n\n"
      "#include \"info.h\"\n"
      "#include \"window.h\"\n",
      funs_filename);

  for (i = 0; doc_header[i]; i++)
    {
      fprintf (doc_stream, doc_header[i], argv[0], argv[0]);
      fprintf (doc_stream, "\n");
    }

  fprintf (doc_stream,
           _("   Source files groveled to make this file include:\n\n"));

  for (i = 1; i < argc; i++)
    fprintf (doc_stream, "\t%s\n", argv[i]);

  fprintf (doc_stream, "\n");
  for (i = 0; doc_header_1[i]; i++)
    fprintf (doc_stream, "%s\n", doc_header_1[i]);

  init_func_key(0);

  for (i = 1; i < argc; i++)
    {
      char *curfile;
      curfile = argv[i];

      if (*curfile == '-')
        continue;

      fprintf (doc_stream, "/* Commands found in \"%s\". */\n", curfile);
      fprintf (funs_stream, "\n/* Functions declared in \"%s\". */\n",
               curfile);

      process_one_file (curfile, doc_stream, funs_stream);
    }

  fprintf (doc_stream, "   { NULL, NULL, NULL, NULL }\n};\n");
  fprintf (funs_stream, "\n#define A_NCOMMANDS %u\n", next_func_key());

  /* The order of these calls also depends exactly on the order in the
   * Makefile.{in,am}; see the must_fopen calls above.  */
  fclose (funs_stream);
  fclose (doc_stream);

  return 0;
}

/* Read the file represented by FILENAME into core, and search it for Info
   function declarations.  Output the declarations in various forms to the
   DOC_STREAM and FUNS_STREAM. */
static void
process_one_file (char *filename, FILE *doc_stream, FILE *funs_stream)
{
  int descriptor, decl_len;
  char *buffer, *decl_str;
  struct stat finfo;
  long offset;
  long file_size;

  if (stat (filename, &finfo) == -1)
    fatal_file_error (filename);

  file_size = (long) finfo.st_size;
  /* This shouldn't happen but stops a gcc warning with -Wstringop-overflow. */
  if (file_size < 0)
    fatal_file_error (filename);

  descriptor = open (filename, O_RDONLY, 0666);
  if (descriptor == -1)
    fatal_file_error (filename);

  buffer = xmalloc (1 + file_size);
  /* On some systems, the buffer will actually contain
     less characters than the full file's size, because
     the CR characters are removed from line endings.  */
  file_size = read (descriptor, buffer, file_size);
  close (descriptor);

  offset = 0;
  decl_str = DECLARATION_STRING;
  decl_len = strlen (decl_str);

  while (1)
    {
      long point = 0;
      int line_number = 0;

      char *func, *doc;
      char *func_name;

      for (; offset < (file_size - decl_len); offset++)
        {
          if (buffer[offset] == '\n')
            {
              line_number++;
            }

          if (strncmp (buffer + offset, decl_str, decl_len) == 0)
            {
              offset += decl_len;
              point = offset;
              break;
            }
        }

      if (!point)
        break;

      /* Skip forward until we find the open paren. */
      while (point < file_size)
        {
          if (buffer[point] == '\n')
            {
              line_number++;
            }
          else if (buffer[point] == '(')
            break;

          point++;
        }

      while (point++ < file_size)
        {
          if (!whitespace_or_newline (buffer[point]))
            break;
          else if (buffer[point] == '\n')
            {
              line_number++;
            }
        }

      if (point >= file_size)
        break;

      /* Now looking at name of function.  Get it. */
      for (offset = point; buffer[offset] != ','; offset++);
      func = xmalloc (1 + (offset - point));
      strncpy (func, buffer + point, offset - point);
      func[offset - point] = '\0';

      /* Generate the user-visible function name from the function's name. */
      {
        register int i;
        char *name_start;

        name_start = func;

        if (strncmp (name_start, "info_", 5) == 0)
          name_start += 5;

        func_name = xstrdup (name_start);

        /* Fix up "ea" commands. */
        if (strncmp (func_name, "ea_", 3) == 0)
          {
            char *temp_func_name;

            temp_func_name = xmalloc (10 + strlen (func_name));
            strcpy (temp_func_name, "echo_area_");
            strcat (temp_func_name, func_name + 3);
            free (func_name);
            func_name = temp_func_name;
          }

        for (i = 0; func_name[i]; i++)
          if (func_name[i] == '_')
            func_name[i] = '-';
      }

      /* Find doc string. */
      point = offset + 1;

      while (point < file_size)
        {
          if (buffer[point] == '\n')
            {
              line_number++;
            }

          if (buffer[point] == '"')
            break;
          else
            point++;
        }

      offset = point + 1;

      while (offset < file_size)
        {
          if (buffer[offset] == '\n')
            {
              line_number++;
            }

          if (buffer[offset] == '\\')
            offset += 2;
          else if (buffer[offset] == '"')
            break;
          else
            offset++;
        }

      offset++;
      if (offset >= file_size)
        {
          free (func_name);
          free (func);
          break;
        }

      doc = xmalloc (1 + (offset - point));
      strncpy (doc, buffer + point, offset - point);
      doc[offset - point] = '\0';

      fprintf (doc_stream,
          "   { %s, \"%s\", (FUNCTION_KEYSEQ *)0, %s },\n",
          func, func_name, doc);

      free (func_name);

      fprintf (funs_stream, "#define A_%s %u\n", func, next_func_key());
      fprintf (funs_stream,
          "extern void %s (WINDOW *window, int count);\n",
          func);
      free (func);
      free (doc);
    }
  free (buffer);
}

static void
fatal_file_error (char *filename)
{
  fprintf (stderr, _("Couldn't manipulate the file %s.\n"), filename);
  exit (EXIT_FAILURE);
}

static FILE *
must_fopen (char *filename, char *mode)
{
  FILE *stream;

  stream = fopen (filename, mode);
  if (!stream)
    fatal_file_error (filename);

  return stream;
}

static unsigned int func_key;

static void
init_func_key(unsigned int val)
{
	func_key = val;
}

static unsigned int
next_func_key(void)
{
	return func_key++;
}


