/*
 * Copyright (c) 2017, 2026, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test
 * @bug 8186046
 * @summary Test invalid name in name and type
 * @library /java/lang/invoke/common
 * @build test.java.lang.invoke.lib.InstructionHelper
 * @run junit/othervm -XX:+UnlockDiagnosticVMOptions -XX:UseBootstrapCallInfo=3 CondyTypeValidationTest
 */

import test.java.lang.invoke.lib.InstructionHelper;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.lang.invoke.MethodType.methodType;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class CondyTypeValidationTest {
    static final MethodHandles.Lookup L = MethodHandles.lookup();
    static final String BSM_TYPE = methodType(Object.class, MethodHandles.Lookup.class, String.class, Object.class)
            .toMethodDescriptorString();

    public static Object[][] invalidTypesProvider() {
        return Stream.of(
//                         ByteCode API checks for the following invalid types
//                         "",
//                         "[",
//                         "A",
//                         "a",
                new Object[]{"L/java/lang/Object", "not a valid reference type descriptor"},
                new Object[]{
                        Stream.generate(() -> "[").limit(256)
                                .collect(Collectors.joining("", "", "I")),
                        "Cannot create an array type descriptor with more than 255 dimensions"
                }
        ).toArray(Object[][]::new);
    }

    @ParameterizedTest
    @MethodSource("invalidTypesProvider")
    public void testInvalidTypes(String type, String errMessContent) throws Exception {
        var e = assertThrows(IllegalArgumentException.class, () -> {
            MethodHandle mh = InstructionHelper.ldcDynamicConstant(
                    L, "name", type,
                    "bsm", BSM_TYPE
            );
        });
        assertTrue(e.getMessage().contains(errMessContent));
    }

    public static Object[][] validTypesProvider() {
        List<String> t = new ArrayList<>(List.of("B", "C", "D", "F", "I", "J", "Ljava/lang/Object;", "S", "Z"));
        int l = t.size();
        for (int i = 0; i < l; i++) {
            t.add("[" + t.get(i));
        }

        return t.stream()
                .map(e -> new Object[]{e}).toArray(Object[][]::new);
    }

    @ParameterizedTest
    @MethodSource("validTypesProvider")
    public void testValidTypes(String type) throws Exception {
        MethodHandle mh = InstructionHelper.ldcDynamicConstant(
                L, "name", type,
                "bsm", BSM_TYPE
        );
    }
}
