/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2026 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - MassXpert, model polymer chemistries and simulate mass spectrometric data;
 * - MineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Local includes
#include "MzCalculationTreeViewModel.hpp"
#include "MzCalculationTreeViewItem.hpp"


namespace MsXpS
{

namespace MassXpert
{


MzCalculationTreeViewModel::MzCalculationTreeViewModel(
  std::vector<IonSPtr> *ions_p, QObject *parent)
  : QAbstractItemModel(parent)
{
  QList<QVariant> rootData;

  Q_ASSERT(ions_p);
  Q_ASSERT(parent);

  mp_parentDlg = static_cast<MzCalculationDlg *>(parent);

  rootData << tr("Charge") << tr("Mono") << tr("Avg");

  mpa_rootItem = new MzCalculationTreeViewItem(rootData);

  mp_ions = ions_p;

  setupModelData(mpa_rootItem);
}


MzCalculationTreeViewModel::~MzCalculationTreeViewModel()
{
  delete mpa_rootItem;
}


MzCalculationDlg *
MzCalculationTreeViewModel::getParentDlg()
{
  return mp_parentDlg;
}


void
MzCalculationTreeViewModel::setTreeView(QTreeView *treeView)
{
  Q_ASSERT(treeView);

  mp_treeView = treeView;
}


QTreeView *
MzCalculationTreeViewModel::treeView()
{
  return mp_treeView;
}


QVariant
MzCalculationTreeViewModel::data(const QModelIndex &index, int role) const
{
  if(!index.isValid())
    return QVariant();

  if(role != Qt::DisplayRole)
    return QVariant();

  MzCalculationTreeViewItem *item =
    static_cast<MzCalculationTreeViewItem *>(index.internalPointer());

  return item->data(index.column());
}


QVariant
MzCalculationTreeViewModel::headerData(int section,
                                       Qt::Orientation orientation,
                                       int role) const
{
  if(orientation == Qt::Horizontal && role == Qt::DisplayRole)
    return mpa_rootItem->data(section);

  return QVariant();
}


Qt::ItemFlags
MzCalculationTreeViewModel::flags(const QModelIndex &index) const
{
  if(!index.isValid())
    return Qt::ItemIsEnabled;

  return Qt::ItemIsEnabled | Qt::ItemIsSelectable;
}


QModelIndex
MzCalculationTreeViewModel::index(int row,
                                  int column,
                                  const QModelIndex &parent) const
{
  MzCalculationTreeViewItem *parentItem = 0;

  if(!parent.isValid())
    parentItem = mpa_rootItem;
  else
    parentItem =
      static_cast<MzCalculationTreeViewItem *>(parent.internalPointer());

  MzCalculationTreeViewItem *childItem = parentItem->child(row);
  if(childItem != 0)
    return createIndex(row, column, childItem);
  else
    return QModelIndex();
}


QModelIndex
MzCalculationTreeViewModel::parent(const QModelIndex &index) const
{
  if(!index.isValid())
    return QModelIndex();

  MzCalculationTreeViewItem *childItem =
    static_cast<MzCalculationTreeViewItem *>(index.internalPointer());

  MzCalculationTreeViewItem *parentItem = childItem->parent();

  if(parentItem == mpa_rootItem)
    return QModelIndex();

  return createIndex(parentItem->row(), 0, parentItem);
}


int
MzCalculationTreeViewModel::rowCount(const QModelIndex &parent) const
{
  MzCalculationTreeViewItem *parentItem;

  if(!parent.isValid())
    parentItem = mpa_rootItem;
  else
    parentItem =
      static_cast<MzCalculationTreeViewItem *>(parent.internalPointer());

  return parentItem->childCount();
}


int
MzCalculationTreeViewModel::columnCount(const QModelIndex &parent) const
{
  if(parent.isValid())
    return static_cast<MzCalculationTreeViewItem *>(parent.internalPointer())
      ->columnCount();
  else
    return mpa_rootItem->columnCount();
}


void
MzCalculationTreeViewModel::removeAll()
{
  std::size_t count = rowCount();

  // qDebug() << "There are currently" << count << "items in the treeview.";

  if(!count)
    return;

  std::size_t index = 0;

  beginRemoveRows(
    QModelIndex(), /*start index*/ 0, /*stop index included*/ count - 1);

  while(index < count)
    {
      // qDebug() << "Now deleting child:" << index <<
      // mpa_rootItem->child(0)->getIon()->mono;

      // Each time we call this function we remove the first item from the list
      // of items. So we shift the list upwards. We thus continue to remove
      // the first item of the list until the right count is reached.
      // Tested OK on Debian GNU/Linux.
      delete(mpa_rootItem->takeChild(0));

      // qDebug() << "Deleted item, remaining items:" <<
      // mpa_rootItem->childCount();

      ++index;
    }

  endRemoveRows();

  mp_ions->clear();

  emit layoutChanged();
}


void
MzCalculationTreeViewModel::addIon(IonSPtr ion_sp)
{
  if(ion_sp == nullptr)
    qFatal() << "Progamming error. Pointer cannot be nullptr.";

  QList<QVariant> columnData;

  int charge = ion_sp->ionizer_sp->charge();

  QString mono =
    QString::number(ion_sp->mono, 'f', libXpertMassCore::OLIGOMER_DEC_PLACES);
  QString avg =
    QString::number(ion_sp->avg, 'f', libXpertMassCore::OLIGOMER_DEC_PLACES);

  columnData << charge << mono << avg;

  // Create an item with those columnar strings. The parent of the
  // current item is the root item.
  MzCalculationTreeViewItem *newItem =
    new MzCalculationTreeViewItem(columnData, mpa_rootItem);

  newItem->setIon(ion_sp);

  int newRow = rowCount();

  mpa_rootItem->appendChild(newItem);

  //   qDebug() << __FILE__ << __LINE__
  // 	    << "Appended new item:" << newItem
  // 	    << "with ion_sp:" << newItem->ion_sp();

  // There is no need to produce a model index for the parent, as we
  // know we are going to insert new row in the root of the tree(this
  // tree has no branches). Hence the QModelIndex() calls below.

  beginInsertRows(QModelIndex(), newRow, newRow);
  insertRow(newRow, QModelIndex());
  endInsertRows();

  emit layoutChanged();
}


bool
MzCalculationTreeViewModel::setupModelData(MzCalculationTreeViewItem *parent)
{
  QList<MzCalculationTreeViewItem *> parents;
  MzCalculationTreeViewItem *currentItem = 0;

  Q_ASSERT(parent);

  // Start with populating the very first item of the treeviewmodel.
  parents << parent;

  // We have the mp_ions pointer that points to a QList<Ionizable*>
  // list.

  for(const IonSPtr &ion_sp : *mp_ions)
    {
      QList<QVariant> columnData;

      int charge =
        ion_sp->ionizer_sp->getNominalCharge() * ion_sp->ionizer_sp->getLevel();

      QString mono =
        QString::number(ion_sp->mono, 'f', libXpertMassCore::OLIGOMER_DEC_PLACES);
      QString avg =
        QString::number(ion_sp->avg, 'f', libXpertMassCore::OLIGOMER_DEC_PLACES);

      columnData << charge << mono << avg;


      // Create an item with those columnar strings. The parent of the
      // current item is parents.
      currentItem = new MzCalculationTreeViewItem(columnData, parents[0]);

      currentItem->setIon(ion_sp);

      //       qDebug() << __FILE__ << __LINE__
      // 		<< "Set new ion:" << ion;

      // We should append that item right now.
      parents[0]->appendChild(currentItem);

      // At this point we have finished setting up the Model data.
    }

  return true;
}

} // namespace MassXpert

} // namespace MsXpS
