// bibiman - a TUI for managing BibLaTeX databases
// Copyright (C) 2025  lukeflo
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.
/////

use crate::bibiman::bibisetup::{BibiData, SanitizedBibiData};

mod optimized_sanitize;
use optimized_sanitize::optimized_sanitize;

/// Helper macro to sanitize bibidata structs.
/// Here lives the code that generates SanitizedBibiData
/// structs from BibiData structs.
macro_rules! optimized_sanitize_bibidata {
    ($bibidata:expr) => {
        SanitizedBibiData {
            title: optimized_sanitize(false, &$bibidata.title),
            subtitle: match &$bibidata.subtitle {
                None => None,
                Some(subtitle) => Some(optimized_sanitize(false, subtitle)),
            },
            abstract_text: optimized_sanitize(false, &$bibidata.abstract_text),
        }
    };
}

/// Sanitize one BibiData and return a SanitizedBibiData struct.
/// This function does ignore any existing sanitization of the bibidata struct.
pub fn sanitize_one_bibidata(bibidata: &BibiData) -> SanitizedBibiData {
    optimized_sanitize_bibidata!(bibidata)
}

pub fn sanitize_single_string_fully(input: &str) -> String {
    optimized_sanitize(true, input)
}
