#ifdef GTK_FACE
/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * GtkSCalendar Copyright (C) 1998 Stefan Ondrejicka <ondrej@idata.sk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"

#ifndef _GTK_FEATURES_1_2

#include <time.h>
#include <stdio.h>
#include "gtkscalendar.h"
#include <gtk/gtksignal.h>

static gchar *en_month_names[] = {
	"January" ,
	"February" ,
	"March" ,
	"April" ,
	"May" ,
	"Jun" ,
	"July" ,
	"August" ,
	"September" ,
	"October" ,
	"November" ,
	"December"
};

static gchar *en_day_names[] = {
	"Sun" ,
	"Mon" ,
	"Tue" ,
	"Wed" ,
	"Thu" ,
	"Fri" ,
	"Sat" ,
};

static void   gtk_scalendar_class_init	(GtkSCalendarClass	*klass);
static void   gtk_scalendar_init	(GtkSCalendar		*scalendar);
static void   gtk_scalendar_realize	(GtkWidget		*widget);
static void   gtk_scalendar_size_request(GtkWidget		*widget ,
					 GtkRequisition		*requisition);
static gint   gtk_scalendar_expose	(GtkWidget		*widget ,
					 GdkEventExpose		*event);
static gint   gtk_scalendar_button_press(GtkWidget		*widget,
					 GdkEventButton		*event);
static void   gtk_scalendar_draw	(GtkWidget 		*widget);
static void   gtk_scalendar_draw_cursor	(GtkWidget		*widget,
					 gint			draw);
static gchar* gtk_scalendar_cvt_month	(GtkSCalendar 		*scalendar ,
					 guint 			month);
static gchar* gtk_scalendar_cvt_day	(GtkSCalendar		*scalendar,
					 guint 			day);
static guint  gtk_scalendar_days_of_month(guint 		month,
					 guint 			year);
static guint  gtk_scalendar_day_of_week	(guint 			day,
					 guint 			month,
					 guint 			year);

enum {
	CHANGED ,
	LAST_SIGNAL
};

static gint scalendar_signals[LAST_SIGNAL] = { 0 };

guint gtk_scalendar_get_type ()
{
	static guint scalendar_type = 0;

	if (!scalendar_type)
	{
		GtkTypeInfo scalendar_info =
		{
			"GtkSCalendar",
			sizeof (GtkSCalendar),
			sizeof (GtkSCalendarClass),
			(GtkClassInitFunc) gtk_scalendar_class_init,
			(GtkObjectInitFunc) gtk_scalendar_init,
			(GtkArgSetFunc) NULL,
		        (GtkArgGetFunc) NULL,
		};

		scalendar_type = gtk_type_unique (gtk_widget_get_type (), &scalendar_info);
	}

	return scalendar_type;
}


static void gtk_scalendar_class_init (GtkSCalendarClass *class)
{
	GtkWidgetClass *widget_class;
	GtkObjectClass *object_class;

	widget_class = (GtkWidgetClass*) class;
	object_class = (GtkObjectClass*) class;

	widget_class->realize = gtk_scalendar_realize;
	widget_class->size_request = gtk_scalendar_size_request;
	widget_class->expose_event = gtk_scalendar_expose;
	widget_class->button_press_event = gtk_scalendar_button_press;

	scalendar_signals[CHANGED] = gtk_signal_new ("changed",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (GtkSCalendarClass, changed),
			gtk_signal_default_marshaller,
			GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, scalendar_signals, LAST_SIGNAL);
}

static void gtk_scalendar_init (GtkSCalendar      *scalendar)
{
	time_t t;
	struct tm *ltime;

	t = time(NULL);

	ltime = localtime(&t);

	scalendar->day = ltime->tm_mday;
	scalendar->month = ltime->tm_mon;
	scalendar->year = ltime->tm_year;

	if (ltime->tm_year < 100)
	{
		if (ltime->tm_year > 69) scalendar->year += 1900;
		else scalendar->year += 2000;
	}

	scalendar->show_month_year = TRUE;
	scalendar->spacing = 0;

	scalendar->month_names = en_month_names;
	scalendar->day_names = en_day_names;
}


GtkWidget* gtk_scalendar_new (gchar **day_names , gchar **month_names)
{
	GtkSCalendar *scalendar;

	scalendar = gtk_type_new (gtk_scalendar_get_type ());

	if (month_names)
		scalendar->month_names = month_names;

	if (day_names)
		scalendar->day_names = day_names;

	return GTK_WIDGET (scalendar);
}


static void
gtk_scalendar_realize (GtkWidget *widget)
{
	GtkSCalendar *scalendar;
	GdkWindowAttr attributes;
	gint attributes_mask;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SCALENDAR (widget));

	scalendar = GTK_SCALENDAR (widget);
	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);
	attributes.event_mask = gtk_widget_get_events (widget) |
				GDK_EXPOSURE_MASK |
				GDK_BUTTON_PRESS_MASK;

	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

	widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
	gdk_window_set_user_data (widget->window, scalendar);

	widget->style = gtk_style_attach (widget->style, widget->window);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
}


static void
gtk_scalendar_size_request (
GtkWidget	*widget,
GtkRequisition	*requisition)
{
	GtkSCalendar *scalendar;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SCALENDAR (widget));
	g_return_if_fail (requisition != NULL);

	scalendar = GTK_SCALENDAR (widget);
  
	requisition->width =  8 * scalendar->spacing + scalendar->misc.xpad * 2 +
			21 * gdk_string_width(widget->style->font , "W") + 1;

	requisition->height = (GTK_WIDGET (scalendar)->style->font->ascent +
			GTK_WIDGET (scalendar)->style->font->descent + scalendar->spacing) *
			(7 + scalendar->show_month_year) + scalendar->spacing +
			scalendar->misc.ypad * 2 + 1;
}

static gint gtk_scalendar_expose (
GtkWidget	*widget,
GdkEventExpose	*event)
{
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_SCALENDAR (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	gtk_scalendar_draw(widget);

	return TRUE;
}

static gint gtk_scalendar_button_press (
GtkWidget      *widget,
GdkEventButton *event)
{
	GtkSCalendar *scalendar;
	gint xpos,ypos,dow0;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_SCALENDAR (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	

	scalendar = GTK_SCALENDAR(widget);

	if ((gint) event->y < scalendar->base) return FALSE;

	ypos = ((gint) event->y - scalendar->base ) / 
		(widget->style->font->ascent + widget->style->font->descent +
		 scalendar->spacing);

	xpos = ((int) event->x ) / (3 * gdk_string_width(widget->style->font ,"W") + 
		scalendar->spacing);

	dow0 = gtk_scalendar_day_of_week(1, scalendar->month , scalendar->year);

	if (xpos > 6) return FALSE;
	if ((7*ypos+xpos) < dow0) return FALSE;
	if ((7*ypos+xpos) > (dow0 - 1 + gtk_scalendar_days_of_month(scalendar->month,
		scalendar->year))) return FALSE;

	gtk_scalendar_draw_cursor(widget , FALSE);

	scalendar->day = 7 * ypos + xpos - dow0 + 1;

	gtk_scalendar_draw_cursor(widget , TRUE);

	gtk_signal_emit (GTK_OBJECT (scalendar), scalendar_signals[CHANGED]);

	return TRUE;
}

static void gtk_scalendar_draw(GtkWidget *widget)
{
	GtkSCalendar *scalendar;
	gchar mtxt[50];
	gint twidth;
	gint ndays;
	gint i;
	gint dow;
	gint ypos,xpos,width;

	if (!GTK_WIDGET_VISIBLE (widget) || !GTK_WIDGET_MAPPED (widget))
		return;

	scalendar = GTK_SCALENDAR(widget);
	
	if (scalendar->show_month_year)
	{
		sprintf(mtxt,"%s  %d", gtk_scalendar_cvt_month(scalendar , scalendar->month),
			scalendar->year);

		twidth = gdk_string_width(widget->style->font , mtxt);

		gdk_draw_string (widget->window, widget->style->font,
			GTK_WIDGET_IS_SENSITIVE(widget) ? 
			widget->style->fg_gc[GTK_STATE_PRELIGHT] :
			widget->style->fg_gc[GTK_STATE_INSENSITIVE],
			(widget->allocation.width - twidth) / 2 ,
			scalendar->spacing + widget->style->font->descent +
			widget->style->font->ascent , mtxt);
	}

	width = (3 * gdk_string_width(widget->style->font , "W"));

	ypos = (1 + scalendar->show_month_year) * (scalendar->spacing +
		widget->style->font->descent +
		widget->style->font->ascent);

	scalendar->base = (1 + scalendar->show_month_year) * (scalendar->spacing +
		widget->style->font->descent +
		widget->style->font->ascent) + 
		scalendar->spacing;

	for (i = 0 ; i < 7 ; i ++)
	{
		sprintf(mtxt,"%s" , gtk_scalendar_cvt_day(scalendar , i));

		xpos = i * (3 * gdk_string_width(widget->style->font , "W") + 
			scalendar->spacing) + scalendar->spacing + scalendar->misc.xpad +
			(width - gdk_string_width(widget->style->font , mtxt))/2;

		gdk_draw_string(widget->window, widget->style->font,
			GTK_WIDGET_IS_SENSITIVE(widget) ? 
			widget->style->fg_gc[GTK_STATE_NORMAL] :
			widget->style->fg_gc[GTK_STATE_INSENSITIVE],
			xpos , ypos , mtxt);
	}

	ypos += widget->style->font->ascent +
		widget->style->font->descent +
		scalendar->spacing ;

	ndays = gtk_scalendar_days_of_month(scalendar->month , scalendar->year);
	dow = gtk_scalendar_day_of_week(1 , scalendar->month , scalendar->year);

	for(i = 0 ; i < ndays ; i++)
	{
		sprintf(mtxt,"%d", i + 1);

		xpos = ((i + dow) % 7) * (3 * gdk_string_width(widget->style->font ,"W") + 
			scalendar->spacing) + scalendar->spacing +
			(width - gdk_string_width(widget->style->font , mtxt))/2;


		if (!((i + dow) % 7) && i)
			ypos += widget->style->font->ascent +
				widget->style->font->descent +
				scalendar->spacing ;

		gdk_draw_string(widget->window , widget->style->font ,
			GTK_WIDGET_IS_SENSITIVE(widget) ? 
			widget->style->fg_gc[GTK_STATE_NORMAL] :
			widget->style->fg_gc[GTK_STATE_INSENSITIVE], 
			xpos , ypos , mtxt);
	}

	if (scalendar->day > gtk_scalendar_days_of_month(scalendar->month , scalendar->year)) 
		scalendar->day = 1;

	gtk_scalendar_draw_cursor(widget , TRUE);
}

static void gtk_scalendar_draw_cursor(
GtkWidget	*widget,
gint		draw)
{
	GtkSCalendar *scalendar;
	gint width,height;
	gint dow , dow0;

	if (!GTK_WIDGET_VISIBLE (widget) || !GTK_WIDGET_MAPPED (widget))
		return;

	scalendar = GTK_SCALENDAR(widget);

	dow = gtk_scalendar_day_of_week(scalendar->day , scalendar->month , scalendar->year);

	dow0 = gtk_scalendar_day_of_week(1, scalendar->month , scalendar->year);

	width = (3 * gdk_string_width(widget->style->font , "W"));
	height = widget->style->font->ascent + widget->style->font->descent;

	gdk_draw_rectangle(widget->window , 
			GTK_WIDGET_IS_SENSITIVE(widget) ? 
			(draw ? widget->style->bg_gc[GTK_STATE_SELECTED] : 
			widget->style->bg_gc[GTK_STATE_NORMAL])
			: widget->style->fg_gc[GTK_STATE_INSENSITIVE] , 0 ,
			dow * (width + scalendar->spacing) + scalendar->spacing / 2,
			((dow0 + scalendar->day - 1) / 7) * (height + 
			scalendar->spacing) + scalendar->base ,
			width , height);
}

gint gtk_scalendar_set_date(
GtkSCalendar	*scalendar,
gint 		day,
gint		month,
gint		year)
{
	GtkWidget *widget;
	gboolean redraw = FALSE;
	gboolean change_crsr = FALSE;

	g_return_val_if_fail (scalendar != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_SCALENDAR (scalendar), FALSE);

	widget = GTK_WIDGET(scalendar);

	if (day != scalendar->day)
	{
		gtk_scalendar_draw_cursor(widget , FALSE);
		scalendar->day = (day >= 0) ? day : scalendar->day;
		change_crsr = TRUE;
	}

	if (month != scalendar->month)
	{
		scalendar->month = (month >= 0) ? month : scalendar->month;
		redraw = TRUE;
	}

	if (year != scalendar->year)
	{
		scalendar->year = (year >= 0) ? year : scalendar->year;
		redraw = TRUE;
	}

	if (redraw)
	{
		if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget))
		{
			gdk_window_clear(widget->window);
			gtk_scalendar_draw(widget);
		}
	}
	else if (change_crsr)
	{
		gtk_scalendar_draw_cursor(widget , TRUE);
	}

	if (redraw || change_crsr)	
		gtk_signal_emit (GTK_OBJECT (scalendar), scalendar_signals[CHANGED]);
	
	return TRUE;
}

gint gtk_scalendar_show_month_label(
GtkSCalendar	*scalendar,
gint		show)
{
	GtkWidget *widget;

	g_return_val_if_fail (scalendar != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_SCALENDAR (scalendar), FALSE);

	widget = GTK_WIDGET(scalendar);

	if (show != scalendar->show_month_year)
	{
		scalendar->show_month_year = show;
		gtk_widget_queue_resize (widget);
		if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget))
		{
			gdk_window_clear(widget->window);
			gtk_scalendar_draw(widget);
		}
	}

	return TRUE;
}


static gchar * gtk_scalendar_cvt_month (
GtkSCalendar	*scalendar,
guint 		month)
{
	if (month < 12) return scalendar->month_names[month];
	else return "";
}

static gchar * gtk_scalendar_cvt_day(
GtkSCalendar	*scalendar,
guint 		day)
{
	if (day < 7) return scalendar->day_names[day];
	else return "";
}

static guint gtk_scalendar_days_of_month(
guint month,
guint year)
{
	switch (month)
	{
		case 0: return 31;
		case 1: return ((year < 1753) && (year % 4 == 0)) ||
			 	((year > 1753) && (year % 4 == 0) && 
			 	((year % 100 != 0) || (year % 400 == 0))) ?
				29 : 28;
		case 2: return 31;
		case 3: return 30;
		case 4: return 31;
		case 5: return 30;
		case 6: return 31;
		case 7: return 31;
		case 8: return 30;
		case 9: return 31;
		case 10: return 30;
		case 11: return 31;
		default: return 0;
	}
}

#define gtk_scalendar_days_of_year(year) ((year % 4) ? 365 : 366)

static guint gtk_scalendar_day_of_week(
guint day,
guint month,
guint year)
{
	guint smonth = 0;
	long dow = 4;
	int pyear;

	pyear = (year - 1)/4 - (year - 1 - 1700)/100 + (year - 1)/400;

	dow += (pyear) * 366 + 
	       (year - pyear - 1) * 365;

	while (smonth != month)
	{
		dow += gtk_scalendar_days_of_month(smonth,year);
		smonth++;
	}

	return (int) ((dow + day) % 7);
}

#endif /* _GTK_FEATURES_1_2 */
#endif /* GTK_FACE */

