/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
 */
package gov.nasa.worldwind.servers.wms;

import java.util.Enumeration;

/**
 * @author brownrigg
 * @version $Id$
 */

// TODO: This class in large measure is a subset of behavior/state embodied by WMSGetMapRequest.
public class WMSGetImageryListRequest extends WMSRequest {

    public WMSGetImageryListRequest(WMSHttpServletRequest req) throws WMSServiceException {
        super(req);
        Enumeration params = req.getParameterNames();
        isValid = true;  // assume the best until proven otherwise...

        while(params.hasMoreElements()) {
            String param = (String)params.nextElement();
            String value = req.getParameter(param);
            if (value == null) continue;

            // nothing fancy here; just a rather tedious enumeration...
            if (param.equalsIgnoreCase(REQUEST)) {
                if (value.equals(GETIMAGERYLIST))
                    hasRequestParam = true;
                continue;
            }

            if (param.equalsIgnoreCase(LAYERS)) {
                layers = req.getParameterValues(param);
                continue;
            }

            if (param.equalsIgnoreCase(BBOX)) {
                String[] boxStr = req.getParameterValues(param);
                if (boxStr.length != 4) {
                    errorMsg.append("Invalid BBOX specification; ");
                    isValid = false;
                    continue;
                }
                bbox = new double[4];
                try {
                    for (int i=0; i<4; i++) {
                        bbox[i] = Double.parseDouble(boxStr[i]);
                    }
                } catch (NumberFormatException ex) {
                    errorMsg.append("invalid numeric values in BBOX specification; ");
                    isValid = false;
                    continue;
                }
                if (bbox[XMIN] >= bbox[XMAX] || bbox[YMIN] >= bbox[YMAX]) {
                    errorMsg.append("BBOX min(s) greater than max(s); ");
                    isValid = false;
                }

                continue;
            }


            //******************************************************************
            // Note the spec allows for optional "other sample dimensions"
            // Not implemented here.
            // *****************************************************************
        }

        // Check that all the mandatory parameters are present...
        if (!hasRequestParam)
            missingParam(REQUEST);

        if (layers == null)
            missingParam(LAYERS);

        if (bbox == null)
            missingParam(BBOX);

    }

    public boolean isValid() {
        return isValid;
    }

    public String getParseReport() {
        return errorMsg.toString();
    }

    public String toString() {
        StringBuffer str = new StringBuffer();
        str.append("GetImageryList: ");
        str.append(" layers(");
        for (int i=0; i<layers.length; i++) {
            if (i > 0) str.append(", ");
            str.append(layers[i]);
        }
        str.append(") bbox(");
        for (int i=0; i<bbox.length; i++) {
            if (i > 0) str.append(", ");
            str.append(bbox[i]);
        }
        str.append(")");
        return str.toString();
    }

    // Convenience method; this pattern of error reporting is repeated a number of times.
    private void missingParam(String param) {
        errorMsg.append("Missing required ");
        errorMsg.append(param);
        errorMsg.append(" parameter; ");
        isValid = false;
    }

    public String[] getLayers() { return layers; }
    public double[] getBBox() { return bbox; }
    public double getBBoxXMin() { return bbox[XMIN]; }
    public double getBBoxXMax() { return bbox[XMAX]; }
    public double getBBoxYMin() { return bbox[YMIN]; }
    public double getBBoxYMax() { return bbox[YMAX]; }

    private boolean isValid;
    private boolean hasRequestParam = false;
    private String[] layers = null;
    private double[] bbox = null;
    private StringBuffer errorMsg = new StringBuffer();


    // request-specific parameter names...
    private static final String LAYERS = "LAYERS";
    private static final String BBOX = "BBOX";

    // indices into bbox
    private static int XMIN = 0;
    private static int YMIN = 1;
    private static int XMAX = 2;
    private static int YMAX = 3;
}
