/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.painter.decoration;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.awt.image.BufferedImage;

import javax.swing.*;

import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Implementation of {@link SubstanceDecorationPainter} that uses "arc" painting
 * on title panes and lighter gradient near the center of the application frame.
 * 
 * @author Kirill Grouchnikov
 */
public class ArcDecorationPainter implements SubstanceDecorationPainter {
	/**
	 * The display name for the decoration painters of this class.
	 */
	public static final String DISPLAY_NAME = "Arc";

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.SubstanceTrait#getDisplayName()
	 */
	public String getDisplayName() {
		return DISPLAY_NAME;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.decoration.SubstanceDecorationPainter#paintDecorationArea(java.awt.Graphics2D,
	 *      java.awt.Component,
	 *      org.jvnet.substance.painter.decoration.DecorationAreaType, int, int,
	 *      org.jvnet.substance.theme.SubstanceTheme)
	 */
	public void paintDecorationArea(Graphics2D graphics, Component comp,
			DecorationAreaType decorationAreaType, int width, int height,
			SubstanceTheme theme) {
		switch (decorationAreaType) {
		case PRIMARY_TITLE_PANE:
		case SECONDARY_TITLE_PANE:
			this.paintTitleBackground(graphics, comp, width, height, theme);
			break;
		default:
			this.paintExtraBackground(graphics, SubstanceCoreUtilities
					.getHeaderParent(comp), comp, width, height, theme);
		}
	}

	private void paintTitleBackground(Graphics2D graphics, Component comp,
			int width, int height, SubstanceTheme theme) {
		// create rectangular background and later draw it on
		// result image with contour clip.
		BufferedImage rectangular = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D rgraphics = (Graphics2D) rectangular.getGraphics();
		rgraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		ColorScheme scheme = theme.getColorScheme();

		// Fill background
		GeneralPath clipTop = new GeneralPath();
		clipTop.moveTo(0, 0);
		clipTop.lineTo(width, 0);
		clipTop.lineTo(width, height / 2);
		clipTop.quadTo(width / 2, height / 4, 0, height / 2);
		clipTop.lineTo(0, 0);

		rgraphics.setClip(clipTop);
		GradientPaint gradientTopLeft = new GradientPaint(0, 0, scheme
				.getLightColor(), width / 2, 0, scheme.getUltraLightColor());
		rgraphics.setPaint(gradientTopLeft);
		rgraphics.fillRect(0, 0, width / 2, height);
		GradientPaint gradientTopRight = new GradientPaint(width / 2, 0, scheme
				.getUltraLightColor(), width, 0, scheme.getLightColor());
		rgraphics.setPaint(gradientTopRight);
		rgraphics.fillRect(width / 2, 0, width - width / 2, height);

		// Color darker = SubstanceColorUtilities.getInterpolatedColor(scheme
		// .getMidColor(), scheme.getDarkColor(), 0.8);
		GradientPaint gradientBottomLeft = new GradientPaint(0, height / 2,
				scheme.getMidColor(), width / 2, height / 2, scheme
						.getLightColor());
		GradientPaint gradientBottomRight = new GradientPaint(width / 2,
				height / 2, scheme.getLightColor(), width, height / 2, scheme
						.getMidColor());

		GeneralPath clipBottom = new GeneralPath();
		clipBottom.moveTo(0, height);
		clipBottom.lineTo(width, height);
		clipBottom.lineTo(width, height / 2);
		clipBottom.quadTo(width / 2, height / 4, 0, height / 2);
		clipBottom.lineTo(0, height);

		rgraphics.setClip(clipBottom);
		rgraphics.setPaint(gradientBottomLeft);
		rgraphics.fillRect(0, 0, width / 2, height);
		rgraphics.setPaint(gradientBottomRight);
		rgraphics.fillRect(width / 2, 0, width - width / 2, height);

		GeneralPath mid = new GeneralPath();
		mid.moveTo(width, height / 2);
		mid.quadTo(width / 2, height / 4, 0, height / 2);
		rgraphics.setPaint(gradientBottomLeft);
		rgraphics.setClip(new Rectangle(0, 0, width / 2, height));
		rgraphics.draw(mid);
		rgraphics.setPaint(gradientBottomRight);
		rgraphics
				.setClip(new Rectangle(width / 2, 0, width - width / 2, height));
		rgraphics.draw(mid);

		graphics.drawImage(rectangular, 0, 0, null);
	}

	private void paintExtraBackground(Graphics2D graphics, Container parent,
			Component comp, int width, int height, SubstanceTheme theme) {

		// System.out.println(comp.getClass().getSimpleName() + " -- W: " +
		// width
		// + ", H: " + height);
		// System.out.println(graphics.getTransform());
		// System.out.println(graphics.getClip());

		JRootPane rootPane = SwingUtilities.getRootPane(parent);
		// fix for bug 234 - Window doesn't have a root pane.
		int dx = 0;
		JComponent titlePane = null;
		if (rootPane != null) {
			titlePane = SubstanceCoreUtilities.getTitlePane(rootPane);
			if (titlePane != null) {
				if (comp.isShowing() && titlePane.isShowing()) {
					dx += (comp.getLocationOnScreen().x - titlePane
							.getLocationOnScreen().x);
				} else {
					// have to traverse the hierarchy
					Component c = comp;
					dx = 0;
					while (c != rootPane) {
						dx += c.getX();
						c = c.getParent();
					}
					c = titlePane;
					while (c != rootPane) {
						dx -= c.getX();
						c = c.getParent();
					}
				}
			}
		}
		// int dy = comp.getY();

		int pWidth = (titlePane == null) ? parent.getWidth() : titlePane
				.getWidth();
		// int pHeight = parent.getHeight();
		// System.out.println("DX: " + dx + ", pW: " + pWidth);// + ", pH: " +
		// pHeight);

		ColorScheme scheme = theme.getColorScheme();
		// System.out.println("Using " + scheme.)
		GradientPaint gradientBottomLeft = new GradientPaint(-dx, 0, scheme
				.getMidColor(), -dx + pWidth / 2, 0, scheme.getLightColor());
		GradientPaint gradientBottomRight = new GradientPaint(-dx + pWidth / 2,
				0, scheme.getLightColor(), -dx + pWidth, 0, scheme
						.getMidColor());

		graphics.setPaint(gradientBottomLeft);
		graphics.fillRect(-dx, 0, pWidth / 2, height);
		graphics.setPaint(gradientBottomRight);
		// fix for defect 195 - fill the remaining width (don't use
		// pWidth / 2 since this will leave the last pixel on odd widths).
		graphics.fillRect(-dx + pWidth / 2, 0, pWidth - pWidth / 2, height);

		// System.out.println("left " + (-dx) + ":" + 0 + ":" + pWidth / 2 + ":"
		// + height);
		// System.out.println("right " + (-dx + pWidth / 2) + ":" + 0 + ":"
		// + (pWidth - pWidth / 2) + ":" + height);

		// graphics.translate(dx, 0);
	}
}
