/*
 *   meiogtwy.c -- provides gateway between MEIO API functions and MEIO core functions
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <port_types.h> 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include "meio_api.h"          /* MEIO API Definititions                  */
#include "meioddr2.h"          /* MEIO Device Driver definitions          */
#include "meiomacs.h"          /* to provide reference to MEIO core       */
#include "meiogtwy.h"          /* definitions for gateway structures      */

#if defined(PMGT_ENABLE)       // are we enabled for Notification Server?
extern void EnablePMGT(void);         // function to setup intf to NS
extern void DisablePMGT(void);       // function to remove intf to NS
#endif

/*************************************************************************/
/*  MeioOwner[] - Array of MEIO handle data                              */
/*************************************************************************/
MEIO_OWNER MeioOwner[ MAX_MEIO_HANDLES ];

/*************************************************************************/
/*  MeioConnection[] - Array of MEIO Connection data                     */
/*************************************************************************/
MEIO_CONNECTION MeioConnection[ MAX_MEIO_CONNECTIONS ];

/*************************************************************************/
/*  MeioDspData[] - Array of MEIO DSP data                               */
/*************************************************************************/
MEIO_DSP   MeioDspData[ MAX_MEIO_DSPS ];

/**************************************************************************/
/*  InitializeMEIO()  - initialize MEIO State, called at Driver Init time */
/**************************************************************************/
int InitializeMEIO( char far *lpszCmdLine )
{
  int i;
  (void)lpszCmdLine;                    
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::InitializeMEIO entry lpszCmdLine %s\n",lpszCmdLine);
  
  for (i=0; i<MAX_MEIO_HANDLES; i++) {
    MeioOwner[i].usObjectState = MEIO_OBJ_UNUSED;  
  } 

  for (i=0; i<MAX_MEIO_CONNECTIONS; i++) {
    MeioConnection[i].usObjectState = MEIO_OBJ_UNUSED;
  }

  memset(MeioDspData,0,MAX_MEIO_DSPS * sizeof(MEIO_DSP));

#if defined(PMGT_ENABLE)      
  EnablePMGT();              
#endif
  
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meiogtwy::InitializeMEIO exit\n");

  return(0);
}


/**************************************************************************/
/*  GetHandleIndex() - checks MEIO Handle for validity,                   */
/*                     returns index or -1 (if invalid).                  */
/**************************************************************************/
static int GetHandleIndex(HMEIO hMeio)
{
  HMEIO_PARTS Mp;
  USHORT      i;
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetHandleIndex entry hMeio %lx\n",hMeio);
  Mp.Handle = hMeio;           // get parts of handle
  
  i = Mp.Part.HandleIndex;    // strip index from handle
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetHandleIndex exit i %x\n",i);
  if ( i < MAX_MEIO_HANDLES  &&                        // valid index?
       MeioOwner[i].usObjectState == MEIO_OBJ_USED  &&  // still used?
       Mp.Part.OwnersSegment == MeioOwner[i].usOwnerKey)  // owners match?
    return(i);
  else
    return(-1);
}


/* GetMEIOOwner() - function to return ptr to MEIO owner data */
MEIO_OWNER FAR *GetMEIOOwner(HMEIO hMeio)
{
  int i;
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetMEIOOwner entry hMeio %lx\n",hMeio);
  i = GetHandleIndex(hMeio);
  
  if (i>=0) {
    MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetMEIOOwner exit owner %p\n",&MeioOwner[i]);
    return(&MeioOwner[i]);
  } else {
    return((MEIO_OWNER FAR *)0);
  } 
}

/**************************************************************************/
/*  GetConnectionIndex() - checks Connection Handle for validity,         */
/*                     returns index or -1 (if invalid).                  */
/**************************************************************************/
static int GetConnectionIndex(HCONNECTION hConn)
{
  HMEIO_PARTS Mp, Co;
  USHORT      i;
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetConnectionIndex entry hConn %lx\n",hConn);
  Mp.Handle = (HMEIO)hConn;    
  
  i = Mp.Part.HandleIndex;    

  if ( i < MAX_MEIO_CONNECTIONS ) {
    Co.Handle = MeioConnection[i].hResourceOwner;
    if ( MeioConnection[i].usObjectState == MEIO_OBJ_USED  && 
	 Mp.Part.OwnersSegment == Co.Part.OwnersSegment ) {
      MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetConnectionIndex exit i %x\n",i);
      return(i);
    }
  }
  return(-1);
}


/*   ReleaseConnection() - local function to release a connection's resources */
static ULONG ReleaseConnection(int index)
{
  ULONG ulRC;
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetReleaseConnection entry index %x\n",index);
  if ( index < MAX_MEIO_CONNECTIONS) {
    ulRC = RS_SendMsg(&MeioConnection[index]
		      ,GETRP(MeioConnection[index].lpResourceSet)
		      ,mRELEASEmsg
		      ,0
		      ,0
		      );
    MeioConnection[index].usObjectState = MEIO_OBJ_UNUSED;  
  } else {
    ulRC = MEIO_ND_INVALID_HANDLE;
  } 
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetReleaseConnection exit ulRC %lx\n",ulRC);
  return (ulRC);
}

/*   MeioShutdown() - function to release ALL MEIO resources */
ULONG MeioShutdown(void)
{
  ULONG ulRC=MEIO_NOERROR;
  int   i;
  
#if defined(PMGT_ENABLE)    
  DisablePMGT();            
#endif
  
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meiogtwy::MeioShutdown entry\n");
  for (i=0; i<MAX_MEIO_CONNECTIONS; i++) {
    if (MeioConnection[i].usObjectState == MEIO_OBJ_USED ) {
      ReleaseConnection(i);   
    }
  } 

  for (i=0; i<MAX_MEIO_HANDLES; i++) {
    if (MeioOwner[i].usObjectState == MEIO_OBJ_UNUSED) continue;
    MeioOwner[i].usObjectState = MEIO_OBJ_UNUSED;  
  } 

  for (i=0; i<MAX_MEIO_DSPS; i++) {
    if (MeioDspData[i].usObjectState != MEIO_OBJ_UNUSED) {
      dspFreeModule(MeioDspData[i].hMod);       
      MeioDspData[i].usObjectState = MEIO_OBJ_UNUSED;
    } 
  } 
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::MeioShutdown exit ulRC %lx\n",ulRC);
  return ulRC;
}

/*  Function to resume MEIO operations in all DSPs */
void MEIOrestart(void)
{
  int i;
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meiogtwy::MEIOrestart entry\n");
  for (i=0; i<MAX_MEIO_DSPS; i++) {
    if (MeioDspData[i].hMod ) {
      InitializeMEIODsp(&MeioDspData[i],0);
    } 
  } 
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meiogtwy::MEIOrestart exit\n");
}

/*  Function to suspend MEIO operations in all DSPs */
void MEIOstop(void)
{
  int i;
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meiogtwy::MEIOstop entry\n");
  for (i=0; i<MAX_MEIO_DSPS; i++) {
    if (MeioDspData[i].hMod ) {
      
#if defined(PMGT_ENABLE)      
      MEIO_Suspend(&MeioDspData[i]);
#endif
      
      dspFreeModule(MeioDspData[i].hMod);      
    } 
  } 
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meiogtwy::MEIOstop exit\n");
}


/*   GetMeioDsp() - function to get the DSP data for a matching hDsp */
PMEIO_DSP FAR GetMeioDsp(HDSP hDsp)
{
  int i;
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::GetMeioDsp entry hDsp %p\n",hDsp);
  
  for (i=0; i<MAX_MEIO_DSPS; i++) {
    if (MeioDspData[i].usObjectState != MEIO_OBJ_UNUSED &&
	MeioDspData[i].hDsp == hDsp) {
      MW_SYSLOG_4(TRACE_MEIO_CORE,"meiogtwy::GetMeioDsp exit, %x of %x usObjectState %x\n", i,MAX_MEIO_DSPS,MeioDspData[i].usObjectState);
      return(&MeioDspData[i]);
    } else {
      if (MeioDspData[i].usObjectState == MEIO_OBJ_UNUSED) {
	MeioDspData[ i ].hDsp          = hDsp;
	MeioDspData[ i ].hMod          = 0;
	MeioDspData[ i ].usObjectState = MEIO_OBJ_USED;
	MW_SYSLOG_3(TRACE_MEIO_CORE,"meiogtwy::GetMeioDsp exit, %x of %x setting usObjectState to MEIO_OBJ_USED\n",i,MAX_MEIO_DSPS);
	return(&MeioDspData[ i ]);
      }
    }
  } 

  MW_SYSLOG_1(TRACE_MEIO_CORE,"meiogtwy::GetMeioDsp exit 0\n");
  return((PMEIO_DSP)0);
}

/*   FreeMEIOTask() - function to release MEIO task */
void FreeMEIOTask(PMEIO_DSP pMDsp)
{
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::FreeMEIOTask entry pMDsp %p\n",pMDsp);
  dspFreeModule(pMDsp->hMod);  
  pMDsp->hMod=0;               
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meiogtwy::FreeMEIOTask exit\n");
}

/*************************************************************************/
/*   BEGIN MEIO API gateway functions                                    */
/*************************************************************************/

/*************************************************************************/
/*                                                                       */
/*  dspMeioOpen() - open the MEIO interface, get an owner handle.        */
/*                  On first call the connection to the MEIO Device      */
/*                  Driver must be made.                                 */
/*                                                                       */
/*************************************************************************/
ULONG APIENTRY dspMeioOpeni(HDSP       hDsp,    /* Handle of DSP device from DSPMGR  */
			    HMEIO FAR *phMeio,  /* Handle of MEIO Instance returned  */
			    ULONG      Options) /* reserved - must be 0              */
{
  ULONG       ulRC = MEIO_NOERROR;
  HMEIO_PARTS Mp;
  USHORT      i;

  MW_SYSLOG_3(TRACE_MEIO_CORE,"meiogtwy::dspMeioOpen entry hDsp %p Options %lx\n",hDsp,Options);
  for (i=0; i<MAX_MEIO_HANDLES; i++)
    if (MeioOwner[i].usObjectState==MEIO_OBJ_UNUSED)  break;  

  if (i < MAX_MEIO_HANDLES) {
    Mp.Handle = (HMEIO)phMeio;  
    Mp.Part.HandleIndex = i;    
    
    MeioOwner[i].usObjectState = MEIO_OBJ_USED;
    MeioOwner[i].pMeioDsp      = GetMeioDsp(hDsp);
    MeioOwner[i].usOwnerKey    = Mp.Part.OwnersSegment;
    MeioOwner[i].ulOptions     = Options;
    
    MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::dspMeioOpeni, loading *phMeio with %lx\n",Mp.Handle);
    *phMeio = Mp.Handle;           
  } else  {
    *phMeio = (HMEIO)0; 
    ulRC = MEIO_FAIL_NO_HANDLES;
  } 
  MW_SYSLOG_3(TRACE_MEIO_CORE,"meiogtwy::dspMeioOpen exit phMeio %lx ulRC %lx\n",*phMeio,ulRC);
  return(ulRC);
}


/*************************************************************************/
/*                                                                       */
/*  dspMeioClose()- close a MEIO handle, possibly disconnect connections */
/*                  owned by this handle.                                */
/*                                                                       */
/*************************************************************************/
ULONG dspMeioClosei(HMEIO      hMeio,   /* Handle of MEIO Instance returned  */
			   ULONG      Options) /* reserved - must be 0              */
{
  ULONG       ulRC = MEIO_NOERROR;
  int         hidx, conidx;
  int         numconnections=0;
  PMEIO_DSP   pMeioDsp;

  (void)Options;      

  MW_SYSLOG_3(TRACE_MEIO_CORE,"meiogtwy::dspMeioClose entry hMeio %lx Options %lx\n",hMeio,Options);
  hidx = GetHandleIndex(hMeio);

  if ( hidx >= 0 ) {
    pMeioDsp = MeioOwner[hidx].pMeioDsp;
    
    for (conidx=0; conidx<MAX_MEIO_CONNECTIONS; conidx++) {
      if (MeioConnection[conidx].usObjectState == MEIO_OBJ_USED) {
	if (MeioConnection[conidx].hResourceOwner == hMeio) {
	  ReleaseConnection(conidx);    
	} else {
	  if (MeioConnection[conidx].pMeioDsp == pMeioDsp) {
	    numconnections++;       
	  } 
	}
      }
    } 

    if (numconnections==0) {
      FreeMEIOTask(pMeioDsp);
    } 
    MeioOwner[hidx].usObjectState = MEIO_OBJ_UNUSED;    
  } else {
    ulRC = MEIO_INVALID_MEIO_HANDLE;
  } 

  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::dspMeioClose exit ulRC %lx\n",ulRC);
  return(ulRC);
}


/*------------------------------------------------------------------------*/
/* MEIO QUERY API                                                         */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioQueryi(HMEIO           hMeio,        /* MEIO Handle                     */
			     MEIO_QUERYTYPE  Request,      /* type of Query                   */
			     ULONG    FAR   *fpBufferSize, /* size of buffer for returned data*/
			     VOID     FAR   *fpBuffer)     /* buffer for returned data        */
{
  ULONG                  ulRC = MEIO_NOERROR;
  long                   lSize;
  HMEIO_PARTS            Mp;
  MEIO_CONNECTION       *pConnection; 
  PRSET                  lpSet;
  struct MEIOQCINFO FAR *pQCInfo;
  int                    i,j;

  (void)Request;
  (void)fpBuffer;
  (void)fpBufferSize;
  
  MW_SYSLOG_3(TRACE_MEIO_CORE,"meiogtwy::dspMeioQuery entry hMeio %lx Request %x\n",hMeio,Request);
  if (((USHORT)Request)==MEIOQCINFOREQ) {
    lSize = ((long)*fpBufferSize)-sizeof(struct MEIOQCINFO);
    
    pQCInfo = (struct MEIOQCINFO FAR *)fpBuffer;
    for (i=0, j=0, pConnection = MeioConnection;
	 i < MAX_MEIO_CONNECTIONS;
	 i++, pConnection++) {
      
      if (  pConnection->usObjectState == MEIO_OBJ_USED ) {
	lpSet = pConnection->lpResourceSet->lpResourceSet;
	
	if (lpSet[0]->usHandle == pQCInfo->OwnerHandle &&
	    lpSet[1]->usHandle == pQCInfo->UserHandle ) {
	  if (lSize < 0) return MEIO_INSUFFICIENT_BUFFER;
	  lSize -= sizeof(struct MEIOQCDATA);
	  
	  pQCInfo->Data[j].hMeio = Mp.Handle = pConnection->hResourceOwner;
	  Mp.Part.HandleIndex = (USHORT)i;
	  pQCInfo->Data[j].hConnection = Mp.Handle;
	  j++;
	}
      }
    }
    pQCInfo->OwnerCount = j;

  } else {
    i = GetHandleIndex(hMeio);   
    if (i >= 0) {
      ulRC = MEIO_Query_Abilities(MeioOwner[i].pMeioDsp,
				  Request,
				  fpBufferSize,
				  fpBuffer);

    } else {
      *fpBufferSize = 0;  
      ulRC = MEIO_INVALID_MEIO_HANDLE;  
    } 
  } 
   MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::dspMeioQuery exit ulRC %lx\n",ulRC);
   return(ulRC);
}

/*------------------------------------------------------------------------*/
/* MEIO CONNECT API                                                       */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioConnecti(
            HMEIO            hMeio,          /* MEIO Handle               */
            HCONNECTION FAR *fpConnection,   /* THE Connection is returned*/
	    
            ULONG            OwnerPort,      /* Connection Owner          */
            ULONG            UserPort,       /* Connection User           */
	    
            ULONG            SharingOptions, /* willingness to share      */
            ULONG            ulState,        /* set switch to this valueCH02*/
            ULONG            Reserved)       /* reserved - must be 0      */
{
  ULONG             ulRC = MEIO_NOERROR;
  USHORT            usOwner = (USHORT)OwnerPort;
  USHORT            usUser  = (USHORT)UserPort;
  HMEIO_PARTS       Mp;
  MEIO_CONNECTION  *pConnection;   
  int               hidx, conidx;
  
  MW_SYSLOG_6(TRACE_MEIO_CORE,"meiogtwy::dspMeioConnect entry hMeio %lx OwnerPort %lx UserPort %lx SharingOptions %lx ulState %lx\n",
	 hMeio,OwnerPort,UserPort,SharingOptions,ulState);
  hidx = GetHandleIndex(hMeio);  

  if (hidx >= 0) {
    for (conidx=0, pConnection = MeioConnection;
	 conidx < MAX_MEIO_CONNECTIONS;
	 conidx++, pConnection++) {
      if (pConnection->usObjectState==MEIO_OBJ_UNUSED)  break;
    }  
    
    if (conidx < MAX_MEIO_CONNECTIONS) {
      Mp.Handle = hMeio; 
      Mp.Part.HandleIndex = (USHORT)conidx;

      if (0==(MeioOwner[hidx].pMeioDsp)->hMod) {
	ulRC = InitializeMEIODsp(MeioOwner[hidx].pMeioDsp, 1);
	if (ulRC) return(ulRC);
      }

      ulRC = meFindResourceSet(hMeio,usOwner,usUser,
			       &(pConnection->lpResourceSet));

      if (ulRC) return(ulRC);

      pConnection->hResourceOwner  = hMeio;
      pConnection->pMeioDsp = MeioOwner[hidx].pMeioDsp;
      pConnection->ulSharingOptions= SharingOptions;
      pConnection->ulReserved      = Reserved;
      
      ulRC = RS_SendMsg(pConnection
			,GETRP(pConnection->lpResourceSet)
			,mTESTmsg
			,MEIO_CONNECTION_SHARING
			,SharingOptions
			);
      if (ulRC) {
	return(ulRC); 
      }

      ulRC = RS_SendMsg(pConnection
			,GETRP(pConnection->lpResourceSet)
			,mRESERVEmsg
			,MEIO_CONNECTION_SHARING
			,SharingOptions
			);
      if (ulRC) {
	RS_SendMsg(pConnection
		   ,GETRP(pConnection->lpResourceSet)
		   ,mRELEASEmsg
		   ,0
		   ,0
		   );
	return(ulRC);
      } 

      ulRC = RS_SendMsg(pConnection
			,GETRP(pConnection->lpResourceSet)
			,mUPDATEmsg
			,MEIO_CONNECTION_STATE
			,ulState
			);

      if (ulRC == MEIO_NOERROR) {
	pConnection->usObjectState   = MEIO_OBJ_USED;

	*fpConnection = (HCONNECTION)Mp.Handle;

	/*  Update the External I/O ports */
	ulRC = SendScoreBoards(pConnection->pMeioDsp);
      } else {
	*fpConnection = (HCONNECTION)0;
      } 
    } else {
      ulRC = MEIO_FAIL_NO_HANDLES;
      *fpConnection = (HCONNECTION)0;
    }
  } else {
    ulRC = MEIO_INVALID_MEIO_HANDLE;
    *fpConnection = (HCONNECTION)0;
  } 

  MW_SYSLOG_3(TRACE_MEIO_CORE,"meiogtwy::dspMeioConnect exit hConnection %lx ulRC %lx\n",
	 *fpConnection,ulRC);
  return(ulRC);
}


/*------------------------------------------------------------------------*/
/* MEIO DISCONNECT API                                                    */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioDisconnecti(
               HCONNECTION hConnection,         /* handle returned by     */
                                                /* dspMeioConnect()       */
               ULONG       ulReserved)          /* must be 0          CH02*/
{
  ULONG        ulRC = MEIO_NOERROR;
  int          hidx;
  int          i;
  int          numconnections;
  PMEIO_DSP    pMeioDsp;
  
  (void)ulReserved;     
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::dspMeioDisconnect entry hConnection %lx\n",
	 hConnection);
  hidx = GetConnectionIndex(hConnection);  

  if (hidx >= 0) {
    pMeioDsp = MeioConnection[hidx].pMeioDsp;
    ReleaseConnection(hidx);   
    for (i=0, numconnections=0; i<MAX_MEIO_CONNECTIONS; i++) {
      if (MeioConnection[i].usObjectState == MEIO_OBJ_USED &&
	  MeioConnection[i].pMeioDsp == pMeioDsp) {
	numconnections++;             
      } 
    } 

    if (numconnections==0) {
      FreeMEIOTask(pMeioDsp);
    } 
  } else {
    ulRC = MEIO_ND_INVALID_HANDLE;
  }  

  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::dspMeioDisconnect exit ulRC %lx\n",
	 ulRC);
  return(ulRC);
}

												  
/*------------------------------------------------------------------------*/
/* MEIO Update Connection Attributes                                      */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioUpdateConnectioni(
    HCONNECTION           hConnection,  /* handle returned by             */
                                        /* dspMeioConnect()               */
    MEIO_CONNECTATTRIBUTE Attribute,    /* define attribute to update.    */
    LONG                  lValue)       /* value of attribute.            */
{
  ULONG        ulRC = MEIO_NOERROR;
  int          i;
  
  MW_SYSLOG_4(TRACE_MEIO_CORE,"meiogtwy::dspMeioUpdateConnection entry hConnection %lx Attribute %x lValue %lx\n",
	 hConnection,Attribute,lValue);
  i = GetConnectionIndex(hConnection);

  if (i >= 0) {
    if (Attribute == MEIO_CONNECTION_SHARING) {
      ulRC = RS_SendMsg(&MeioConnection[i]
			,GETRP(MeioConnection[i].lpResourceSet)
			,mTESTUPDATEmsg
			,(ULONG)Attribute
			,(ULONG)lValue
			);
      
    }
     
    if (ulRC==0)
      ulRC = RS_SendMsg(&MeioConnection[i]
			,GETRP(MeioConnection[i].lpResourceSet)
			,mUPDATEmsg
			,(ULONG)Attribute
			,(ULONG)lValue
			);
    
    if (ulRC == MEIO_NOERROR) {
      if (Attribute == MEIO_CONNECTION_SHARING)
	MeioConnection[i].ulSharingOptions = (ULONG)lValue;

      ulRC = SendScoreBoards(MeioConnection[i].pMeioDsp);
    } 
  } else {
    ulRC = MEIO_NC_INVALID_HANDLE;
  }  

  MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::dspMeioUpdateConnection exit ulRC %lx\n",
	 ulRC);
  return(ulRC);
}



/*------------------------------------------------------------------------*/
/* MEIO Query Connection Attributes                                       */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioQueryConnectioni(
    HCONNECTION           hConnection,  /* handle returned by             */
                                        /* dspMeioConnect()               */
    MEIO_CONNECTATTRIBUTE Attribute,    /* define attribute to query.     */
    LONG             FAR *fpValue)      /* value of attribute.            */
{
  ULONG        ulRC = MEIO_NOERROR;
  int          i;

  MW_SYSLOG_3(TRACE_MEIO_CORE,"meiogtwy::dspMeioQueryConnection entry hConnection %lx Attribute %x\n",
	 hConnection,Attribute);
  i = GetConnectionIndex(hConnection);  
  
  if (i >= 0) {
    *fpValue = 0;
    
    ulRC = RS_SendMsg(&MeioConnection[i]
		      ,GETRP(MeioConnection[i].lpResourceSet)
		      ,mQUERYmsg
		      ,(ULONG)Attribute
		      ,(ULONG)fpValue
		      );
    
  } else {
    ulRC = MEIO_NC_INVALID_HANDLE;
  }  

  MW_SYSLOG_4(TRACE_MEIO_CORE,"meiogtwy::dspMeioQueryConnection exit lValue %lx ulRC %lx Attribute %x\n",
	 *fpValue,ulRC,Attribute);
  return(ulRC);
}


/*------------------------------------------------------------------------*/
/* MEIO RESYNC API - reload MEIO Dsp task, etc.                           */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioResynci(
               MEIO_OBJ_TYPE ObjectType,    /* type of handle being passed*/
               ULONG         Object,        /* some Mwave handle          */
                                            /* dspMeioConnect()           */
               ULONG         ulOptions)     /* resync options, must be 0  */
{
  ULONG        ulRC = MEIO_NOERROR;
  PMEIO_DSP    InitArg=0;                    /* argument for Init...()     */
  int          i;

  (void)ulOptions;  

  MW_SYSLOG_4(TRACE_MEIO_CORE,"meiogtwy::dspMeioResync entry ObjectType %x Object %lx ulOptions %lx\n",
	 ObjectType,Object,ulOptions);
  switch (ObjectType) {

  case HMEIO_OBJ_TYPE:       /*  MEIO Handle                        */
    i = GetHandleIndex((HMEIO)Object);
    if (i >= 0) {
      InitArg = MeioOwner[i].pMeioDsp;
    } else {
      ulRC = MEIO_INVALID_MEIO_HANDLE;
    } 
    break;

  case HDSP_OBJ_TYPE       :       /*  DSP  Handle                        */
    InitArg = GetMeioDsp((HDSP)Object);
    break;

  case HCONNECTION_OBJ_TYPE:       /*  Connection Handle                  */
    i = GetConnectionIndex((HCONNECTION)Object);
    if (i >= 0) {
      InitArg = MeioConnection[i].pMeioDsp;
    } else {
      ulRC = MEIO_NC_INVALID_HANDLE;
    } 
    break;

  default:
    ulRC = MEIO_INVALID_MEIO_TYPE;
    break;
  } /* endswitch */


  if (ulRC==MEIO_NOERROR) {
    ulRC = InitializeMEIODsp( InitArg, 0 );
  } 

   MW_SYSLOG_2(TRACE_MEIO_CORE,"meiogtwy::dspMeioResync exit ulRC %lx\n",
	   ulRC);
   return(ulRC);
}
