/*
 *   mwwttbl.h
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
//-----------------------------------------------------------------------------
//
//  Important Info:
//  ---------------
//
//  MWWTTBL.H - header file for World Trade Table Access LIB
//  This file provides the defines, data structures, and prototypes needed
//  to access the World Trade Table database.  Access through this LIB
//  is read only.
//
//  WTT.H - this header file defines all the location of all parameters in
//  the World Trade Table.  Use this header file with the WTT_QueryInfo API.
//
//  MWWTTCUS.H - this header file defines settings in the Mwave.ini file for
//  overriding the World Trade Table settings per country.  Overrides can
//  only be applied if the USER_AF parameter in the Platform Specific block
//  allows it.
//
//
//  Notes for Drivers:
//  ------------------
//
//  The old API's will allocate memory for data storage.  You should call WtFree
//  when you have finished with a field or record to free the storage.  The data
//  is only valid at the time of a function call.  The WtQuery routines are not
//  designed to be accessed at interrupt time.  They allocate memory and the
//  record cache is a global structure which is not protected by a semaphore or
//  other mechanism.
//
//
//  Notes for the Table Maintenance Utility:
//  ----------------------------------------
//
//  The LIB will do all reads of the table on a country record basis.  A cache
//  will be maintained of the last requested record.  If the table maintenance
//  utility changes a record it should call WtCountryRecordChanged() so the
//  cache will be refreshed the next time data is requested.
//
//
//  World Trade Country Database File Naming Convention:
//  ----------------------------------------------------
//
//  This version of the MWWTTBL LIB only supports multi-country files.
//  The multi-country file must also contain a mapping record.  The name
//  and location of a multi-country file is fully qualified in the Database=
//  entry of the [Worldtrd] section of the Mwave.ini file.  An example is
//  Database=c:\mww\worldtrd\tpad1020.wtt.  A provision has been made to use an
//  alternate statement such as Database1= to get multiple databases.
//

#ifndef __WTTBL_H
#define __WTTBL_H

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//----helpful defines----------------------------------------------------------
//

// Note:  The API's in this file require defines such as HDSP, ULONG,
//        and USHORT that are in the Mwave.h file.


                             // This is assumed to be the same size as USHORT
#define TWO_BYTES          2 // which is two bytes.  The structure of the WT Tables
                             // requires two byte entries and this program assumes
                             // that the definition of unsigned short will not change
                             // in future compilers.

#define WT_ENTRY_SIZE      TWO_BYTES            // size of a parameter entry
                                                // used in WtQueryField and
                                                // WtQueryItem functions.




#define WT_TABLE_FORMAT_5  5


#define WT_MAP_TABLE_FORMAT_1 1


#define MAX_RECORD_SIZE  8192   // This is the maximum size record that can be
                                // read into the pusRecord array.



//----data types---------------------------------------------------------------
//

// The table format #5 is defined in MWWTTBL.FM5

typedef struct TAGRECHEADER
{
    unsigned short usHeaderLength;           // Length of the header in words
    unsigned short usWTTableVersion;         // WT table version number
    unsigned short usFormat;                 // table format number
    unsigned short usCountryNum;             // country number of this file

    unsigned short usYear;                   // Year stamp of this file
    BYTE           ubMonth;                  // Month stamp of this file
    BYTE           ubDay;                    // Day stamp of this file
    BYTE           ubHour;                   // Hour stamp of this file
    BYTE           ubMinute;                 // Minute stamp of this file
    unsigned short usRecLength;              // record length of this file in words
} RECHEADER;


typedef struct TAGBLOCKHEADER
{
    unsigned short usBlockLength;
    unsigned short usBlockType;
    unsigned short usYear;                   // Year stamp of this block
    BYTE           ubMonth;                  // Month stamp of this block
    BYTE           ubDay;                    // Day stamp of this block
    BYTE           ubHour;                   // Hour stamp of this block
    BYTE           ubMinute;                 // Minute stamp of this block
} BLOCKHEADER;

typedef struct TAGMAPENTRY  // Map file format #1
{
    unsigned short usAccessCode;    // international telephone access number
    unsigned short usCountryNum;    // Mwave WT country number, WT_COUNTRY below
    unsigned short usWTT;           // Associtated WTT to use the same one in many countries
    unsigned short usReserved;
} MAPENTRY;


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  Country Numbers
//
//  Maintaince note:  changes to the country numbers affect MWWTTBL.H, MWWTTBL.C,
//                    MWWTTBL.RC, MWWT_RES.H, VOICECTL.H and MWCNAME.RC
//
//

#ifndef __WT_COUNTRY_NUMBERS
#define __WT_COUNTRY_NUMBERS

#ifndef __WT_COUNTRY_CODES
#define __WT_COUNTRY_CODES

#define WT_COUNTRY_USA                      1  // removed CANADA from the USA selection
#define WT_COUNTRY_BELGIUM                  2
#define WT_COUNTRY_HONG_KONG                3
#define WT_COUNTRY_SINGAPORE                4
#define WT_COUNTRY_NEW_ZEALAND              5
#define WT_COUNTRY_JAPAN                    6
#define WT_COUNTRY_PORTUGAL                 7
#define WT_COUNTRY_IRELAND                  8
#define WT_COUNTRY_GENERIC                  9
#define WT_COUNTRY_SPAIN                   10
#define WT_COUNTRY_GREECE                  11
#define WT_COUNTRY_ISRAEL                  12
#define WT_COUNTRY_TAIWAN                  13
#define WT_COUNTRY_AUSTRALIA               14
#define WT_COUNTRY_AUSTRIA                 15
#define WT_COUNTRY_MEXICO                  16
#define WT_COUNTRY_SOUTH_AFRICA            17
#define WT_COUNTRY_CHILE                   18
#define WT_COUNTRY_SWITZERLAND             19
#define WT_COUNTRY_GERMANY                 20
#define WT_COUNTRY_BRAZIL                  21
#define WT_COUNTRY_RUSSIA                  22
#define WT_COUNTRY_YUGOSLAVIA              23
#define WT_COUNTRY_HUNGARY                 24
#define WT_COUNTRY_CZECHREPUBLIC           25
#define WT_COUNTRY_LUXEMBORG               26
#define WT_COUNTRY_LUXEMBOURG              26
#define WT_COUNTRY_NORWAY                  27
#define WT_COUNTRY_DENMARK                 28
#define WT_COUNTRY_FRANCE                  29
#define WT_COUNTRY_NETHERLANDS             30
#define WT_COUNTRY_U_K                     31
#define WT_COUNTRY_SWEDEN                  32
#define WT_COUNTRY_ITALY                   33
#define WT_COUNTRY_FINLAND                 34
#define WT_COUNTRY_THAILAND                35
#define WT_COUNTRY_KOREA                   36
#define WT_COUNTRY_KOREA_SOUTH             36
#define WT_COUNTRY_MALAYSIA                37
#define WT_COUNTRY_PRC                     38
#define WT_COUNTRY_SLOVAKIA                39

#define WT_COUNTRY_ALGERIA                 40
#define WT_COUNTRY_ARGENTINA               41
#define WT_COUNTRY_ARMENIA                 42
#define WT_COUNTRY_AZERBAIJAN              43
#define WT_COUNTRY_BELARUS                 44
#define WT_COUNTRY_BOLIVIA                 45
#define WT_COUNTRY_BRUNEI                  46
#define WT_COUNTRY_BULGARIA                47
#define WT_COUNTRY_BURMA                   48
#define WT_COUNTRY_COLOMBIA                49

#define WT_COUNTRY_DEBUG_50                50
#define WT_COUNTRY_DEBUG_51                51
#define WT_COUNTRY_DEBUG_52                52
#define WT_COUNTRY_DEBUG_53                53
#define WT_COUNTRY_DEBUG_54                54
#define WT_COUNTRY_DEBUG_55                55

#define WT_COUNTRY_COSTA_RICA              56
#define WT_COUNTRY_CUBA                    57
#define WT_COUNTRY_ECUADOR                 58
#define WT_COUNTRY_EGYPT                   59
#define WT_COUNTRY_EL_SALVADOR             60
#define WT_COUNTRY_GUATEMALA               61
#define WT_COUNTRY_GUYANA                  62
#define WT_COUNTRY_HONDURAS                63
#define WT_COUNTRY_INDIA                   64
#define WT_COUNTRY_INDONESIA               65
#define WT_COUNTRY_NICARAGUA               66
#define WT_COUNTRY_PANAMA                  67
#define WT_COUNTRY_PARAGUAY                68
#define WT_COUNTRY_PERU                    69
#define WT_COUNTRY_PHILIPPINES             70
#define WT_COUNTRY_POLAND                  71
#define WT_COUNTRY_SAUDI_ARABIA            72
#define WT_COUNTRY_TURKEY                  73
#define WT_COUNTRY_UKRAINE                 74
#define WT_COUNTRY_URUGUAY                 75
#define WT_COUNTRY_VENEZUELA               76
#define WT_COUNTRY_VIETNAM                 77
#define WT_COUNTRY_WEST_INDIES             78

#define WT_COUNTRY_ANTIGUA_BARBUDA         79
#define WT_COUNTRY_ARUBA                   80
#define WT_COUNTRY_BAHAMAS                 81
#define WT_COUNTRY_BARBADOS                82
#define WT_COUNTRY_BERMUDA                 83
#define WT_COUNTRY_CAYMAN_ISLANDS          84
#define WT_COUNTRY_GRENADA                 85
#define WT_COUNTRY_JAMAICA                 86
#define WT_COUNTRY_NETH_ANTILLES           87
#define WT_COUNTRY_ST_KITTS_NEVIS          88
#define WT_COUNTRY_ST_LUCIA                89
#define WT_COUNTRY_ST_MARTIN               90
#define WT_COUNTRY_ST_VINCENT              91
#define WT_COUNTRY_SURINAME                92
#define WT_COUNTRY_TRINIDAD_TOBAGO         93
#define WT_COUNTRY_TURKS_CAICOS            94
#define WT_COUNTRY_VIRGIN_IS_BRITISH       95
#define WT_COUNTRY_VIRGIN_IS_USA           96
#define WT_COUNTRY_JORDAN                  97
#define WT_COUNTRY_KUWAIT                  98
#define WT_COUNTRY_OMAN                    99
#define WT_COUNTRY_UNITED_ARAB_EMIRATES   100
#define WT_COUNTRY_YEMAN                  101
#define WT_COUNTRY_CYPRUS                 102
#define WT_COUNTRY_PAKISTAN               103
#define WT_COUNTRY_ROMANIA                104
#define WT_COUNTRY_SLOVENIA               105
#define WT_COUNTRY_RESERVED5              106
#define WT_COUNTRY_RESERVED6              107
#define WT_COUNTRY_RESERVED7              108
#define WT_COUNTRY_RESERVED8              109
#define WT_COUNTRY_RESERVED9              110
#define WT_COUNTRY_RESERVED10             111
#define WT_COUNTRY_CANADA                 112

#define WT_COUNTRY_MAX                    112

#define WT_COUNTRY_INVALID             0xFFFF  //This defines an invalid country number and
                                               //is used in mapping access codes to country
                                               //numbers.

#endif
#endif

// Define the maximum allowed telephone network access code.
// This is used in mapping access codes to Mwave country numbers and
// into actual WT Tables.
#define ACCESS_CODE_MAX                  10000

//-----------------------------------------------------------------------------



///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//
// Start of New API definition, the old API is listed later in the file.
//
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


// New API return codes
// These need to be a negative copy of the old API.  Additional ones need to
// leave room to support the old API.


// This is returned by functions if no error was detected.
#define WTT_SUCCESSFUL                                        0L

//------ Initialization problems -----------------------------------------------

  // This is returned any time a function is called and the default WT Table has
  // not been loaded into memory.
  #define WTTPOSERR_INIT_FAILED                             100L

  // This specifies that the WT Table format is incorrect for the level of MWWTTBL functions.
  #define WTTPOSERR_INVALID_TABLE_FORMAT                    101L

  // This is returned when information is requested about a country that does not
  // exist in the current WT Table Database.
  #define WTTPOSERR_COUNTRY_NOT_AVAIL                       102L

  // This is return if the WTTversion field in the WT Table does not match the WTTversion
  // setting in the Mwave.ini file.
  #define WTTPOSERR_TABLE_VERSION_INVALID                   103L


  // This will occur if the WT Table becomes too large to load into memory such
  // as when there is too much file history text.
  #define WTTPOSERR_RECORD_TOO_LARGE                        104L

  // This will be returned if the configuration  can not be correctly read.
  #define WTTPOSERR_CONFIG_INVALID                        105L

  // Failed to access the DOS Mwave.ini file to update the country setting.
  #define WTTPOSERR_DOS_MWAVE_INI_ACCESS_FAILED             106

  #define WTTPOSERR_OS_VERSION_MISSING                      107

  #define WTTPOSERR_SETUPX_LIB_LOAD_FAILED                  108


  //--------Server problems-----------------------------------------------------
  // This is returned during initialization if the connection to the Mwave
  // Notification Server failed.
  #define WTTPOSERR_MWAVE_SERVER_FAILED                     110L

  // This is returned during initialization if the connection to the Mwave
  // Notification Server failed.
  #define WTTPOSERR_MWAVE_NOTIFY_FAILED                     111L

  // This is returned if all the available client slots are used up.
  #define WTTPOSERR_TOO_MANY_CLIENTS                        112L

//------ Map file problems: ----------------------------------------------------

  // This is returned by the mapping functions if the map file has not been
  // correctly read.
  #define WTTPOSERR_NO_MAPPING_INFO                         120L

  // This specifies that the map file format is incorrect for the level of MWWTTBL functions.
  #define WTTPOSERR_INVALID_MAP_TBL_FORMAT                  121L

  // Occurs when the map file is too large, i.e., it has more bytes than required by all
  // the possible access codes.
  #define WTTPOSERR_MAP_FILE_CORRUPTED                      122L

  // Occurs when a bad access code value is present in the map file.  It is too large.
  #define WTTPOSERR_MAP_ACCESSCODE_TOO_LRG                  123L

  // Occurs when the telephone access code used in international dialing is larger than
  // ACCESS_CODE_MAX.  This value is read from the telephon.ini file.
  #define WTTPOSERR_ACCESS_CODE_TOO_LARGE                   124L



//------ .ini access problems: -------------------------------------------------
  // This error occurs when reading the Mwave.ini file if no Database = entry is
  // present in the WORLDTRADE section.
  #define WTTPOSERR_GETTING_DB_NAME                         130L

  // This is returned if an invalid setting is in the Mwave.ini file for the Country = entry.
  #define WTTPOSERR_INVALID_MWAVEINI_COUNTRY                131L

  // This is returned when the Mwave.ini does not have a valid setting in the WTTversion = entry
  // of the Worldtrade section.
  #define WTTPOSERR_WTTVERSION_INVALID                      132L

  // This occurs if a write to the Mwave.ini file fails.  This could be for the CurrentCountry=
  // entry or writting the Country= entry if using the telephon.ini file as the country
  // selection source.
  #define WTTPOSERR_FAILED_MWAVE_INI_WRITE                  133L

  #define WTTPOSERR_TELEPHON_INI_INVALID                    134L

  // This is returned if an error occurs getting the Mwave Country Select Utility path and
  // name out of the mwave.ini file.
  #define WTTPOSERR_GETTING_COUNTRY_SELECT_PATH             135L

//------ Database access problems: ---------------------------------------------

  // This is returned if the database specified in the Mwave.ini file can not be openned
  // for binary read access.  This is probably because the Database= entry in the Worldtrade
  // section of the Mwave.ini file is pointing to an invalid file.  It could also be caused
  // if the file is locked up by someother program which is not sharing access.
  #define WTTPOSERR_DATABASE_OPEN_FAILED                    140L

  // This occurs when the correct number of bytes is not read from the file for either the
  // record header or the country record.  This means that the file is corrupt.
  #define WTTPOSERR_DATABASE_READ_FAILED                    141L



//------ Corrupt WT Table problems: --------------------------------------------

  // This occurs when the country number embedded in the WT Table is out of range.  This is
  // also returned when mapping entries are out of range.  The proper range is defined in
  // the country list above.  This means that the WT Table is corrupt.
  #define WTTPOSERR_INVALID_COUNTRY                         150L

  // Occurs when the maximum power level read from the Platform Specific block is greater
  // than 0dBm.
  #define WTTPOSERR_INVALID_MAX_TX_PWR_LVL                  151L

  // This occurs when the WTT contains an invalid checksum
  #define WTTPOSERR_INVALID_CHECKSUM                        152L

//------ WTT_ function usage errors: -------------------------------------------

  // This is returned by the WTT_QueryInfo function when an offset is specified
  // which is outside the block size.
  #define WTTPOSERR_INVALID_OFFSET                          160L

  // This is returned if the WT Table is searched for a data block that is not present.
  // This would probably mean that the WT Table is corrupted or an invalid block
  // type was specified.
  #define WTTPOSERR_BLOCK_TYPE_NOT_FOUND                    161L

  // This is returned if a block in the WT Table has a block length of zero.
  #define WTTPOSERR_BLOCK_LENGTH_INVALID                    162L

  // This is returned if a function needs to access the WT Table and finds that
  // it is not loaded into memory.
  #define WTTPOSERR_COUNTRY_NOT_LOADED                      163L

  // This is returned by obsolete functions that are not supported in the current
  // version of the MWWTTBL LIB.
  #define WTTPOSERR_OBSOLETE_FUNCTION                       164L

  // This is returned if a country name string can not be loaded from the
  // resources.
  #define WTTPOSERR_INVALID_RES_STR_TABLE                   165L

  // This is returned if the WTT_CheckCountryNumberStatus() API returns
  // an invalid status value.
  #define WTTPOSERR_INVALID_CN_STATUS                       166L

  // This is returned if there is a problem calling CreateEvent
  // in WTT_RegisterCallback.
  #define WTTPOSERR_CALLBACK_CREATE_EVENT_FAILED            167L

  // This is returned if there is a problem calling CreateThread
  // in WTT_RegisterCallback.
  #define WTTPOSERR_CALLBACK_CREATE_THREAD_FAILED           168L



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_GetNumCountries
//
//     Retrieves the maximum country number defined in the current revision
//     of this LIB.  This is used by the country selection utility to
//     search the database to find all installed WT Tables.
//
//     Params:   hDsp             - The DSP handle, one WT Table per DSP.
//               lpulNumCountries - Pointer to ULONG location where the
//                                  country number should be returned in.
//               ulReserved       - A reserved location for future API
//                                  expansion.
//
//     Returns:  ULONG WTPOS_ERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_GetNumCountries (HDSP hDsp, ULONG FAR* lpulNumCountries, ULONG ulReserved );


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_GetCountryName
//
//     Retrieves the name of the country associated with the given country
//     number.  Due to WT Table mapping, a particular WT Table could be
//     used in a different country than it's number.
//
//     Params:   hDsp            - The DSP handle, one WT Table per DSP.
//               ulCountryNum    - Country number to be translated into a
//                                 country name string.
//               lpszCountryName - Pointer to a string location to copy the
//                                 country name into.
//               ulBufSize       - Size in bytes of the lpszCountryName string
//                                 location.
//               ulReserved      - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_GetCountryName (HDSP hDsp, ULONG ulCountryNum, ULONG ulAccessCode, char FAR* lpszCountryName, ULONG ulBufSize, ULONG ulReserved );


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_GetCountryCouplerID
//
//     Retrieves the WTTCID setting from the WTITCB block of the specifed
//     country number.  The coupler ID is only valid if a valid WT Table
//     exists for the country number.
//
//
//     Params:   hDsp               - The DSP handle, one WT Table per DSP.
//               ulCountryNum       - Country number to look up the coupler ID.
//               lpulCouplerID      - The CouplerID from the WT Table.
//               ulReserved         - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_GetCountryCouplerID (HDSP hDsp, ULONG ulCountryNum, ULONG *lpulCouplerID, ULONG ulReserved );


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_SetCouplerIDfilter
//
//     Sets the Coupler ID filter.  Once this filter is set, only WT Tables
//     with an identical WTCID setting will be valid.  This will limit the
//     number of countries listed in the country selection application.
//
//
//     Params:   hDsp         - The DSP handle, one WT Table per DSP.
//               ulCouplerID  - Coupler ID that the WTTCID setting must match
//                              for the WT Table to be valid.
//
//               ulDisableRestriction = 0 - Enable restriction
//                                      1 = Disable restriction
//
//               ulReserved   - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_SetCouplerIDfilter (HDSP hDsp, ULONG ulCouplerID, ULONG ulDisableRestriction, ULONG ulReserved );


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_GetCurrentCountry
//
//     Retrieves the current country number in use for a particular DSP.  This
//     function also verifies that the current country being used by the system
//     is the one specified in the Mwave.ini or Telephon.ini files.  If the
//     specified country has changed, the new country is loaded in and
//     a WT_TABLECHANGED notification is sent to all registered clients.
//
//     Params:   hDsp                  - The DSP handle, one WT Table per DSP.
//               lpulCurrentCountryNum - Pointer to location where the current
//                                       country number should be returned in.
//               ulReserved            - A reserved location for future API
//                                       expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_GetCurrentCountry (HDSP hDsp, ULONG FAR* lpulCurrentCountryNum, ULONG ulReserved );


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_GetCurrentWTTversion
//
//     Retrieves the WTTversion of the current WT Table.  This can be used
//     to determine what level of Mwave communications code mix is on the
//     system.
//
//     Params:   hDsp                  - The DSP handle, one WT Table per DSP.
//               lpulCurrentWTTversion - Pointer to location where the current
//                                       WTTversion setting should be returned.
//               ulReserved            - A reserved location for future API
//                                       expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_GetCurrentWTTversion (HDSP hDsp, ULONG FAR* lpulCurrentWTTversion, ULONG ulReserved );


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_ChangeWTTable
//
//     Tells the LIB to update the mwave.ini file country= setting and
//     reload the WT Table.  All registered clients are informed that
//     the WT Table may have changed.
//
//     Params:   hDsp       - The DSP handle, one WT Table per DSP.
//               ulCountryNum    - Country number to be translated into a
//                                 country name string.
//               ulReserved - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_ChangeWTTable (HDSP hDsp, ULONG ulCountryNum, ULONG ulReserved );



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_WTTableHasChanged
//
//     This function is called when the WT Table has changed.  This is called
//     by the WTT_ChangeWTTable() function to reload the WT Table and to
//     notify users of the change.  
//
//     Params:   hDsp            - The DSP handle, one WT Table per DSP.
//               ulReserved      - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_WTTableHasChanged (HDSP hDsp, ULONG ulReserved );


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_CheckCountryNumber
//
//     Params:   hDsp                         - The DSP handle, one WT Table per DSP.
//               lpbCountryNumberIsOK         - TRUE if country selections match,
//                                              FALSE if there is a mismatch
//               BOOL bCallMwaveCountrySelect - Set to TRUE if the Mwave Country
//                                              selection utility should be called
//                                              for a mismatch.
//               ulReserved                   - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_CheckCountryNumber (HDSP hDsp, BOOL FAR *lpbCountryNumberIsOK,
                                       BOOL bCallMwaveCountrySelect, ULONG ulReserved );

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_CheckCountryNumberStatus
//
//     Params:   hDsp                    - The DSP handle, one WT Table per DSP.
//               lpulCountryNumberStatus - Pointer to return the status of the
//                                         country number in.
//               ulReserved              - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list in mwwttbl.h
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_CheckCountryNumberStatus (HDSP hDsp, ULONG FAR *lpulCountryNumberStatus,
                                             ULONG ulReserved );


// lpulCountryNumberStatus return codes:
#define CS_OK                            100
#define NOT_YET_AVAILABLE                101
#define NEED_DP_SET                      102
#define DP_CHANGED                       103
#define UNSUPPORTED_COUNTRY              104
#define UNSUPPORTED_BUT_EQUAL_TO_LAST_DP 105  // The country is not supported but is the
                                              // same as was previously loaded.

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_CheckForCountryNumberChange
//
//     This function performs the same check as WTT_CheckCountryNumberStatus
//     but does not perform and INI/REG updates.
//
//     Params:   hDsp                    - The DSP handle, one WT Table per DSP.
//               lpulCountryNumberStatus - Pointer to return the status of the
//                                         country number in.
//               ulReserved              - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list in mwwttbl.h
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_CheckForCountryNumberChange (HDSP hDsp, BOOL FAR *lpbCountryNumberChange,
                                                ULONG ulReserved );


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_QueryBlockLength
//
//     Provides a means to get the length of a block in the current WT record.
//
//     Params:   hDsp          - The DSP handle, one WT Table per DSP.
//               usWTTBlockTag - WTITCB, Blacklist, etc. tag from WTT.H file.
//               lpulNumShorts - A pointer to where the block length in words
//                               (shorts) will be returned.
//               ulReserved    - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_QueryBlockLength (HDSP hDsp, USHORT usWTTBlockTag, ULONG FAR* lpulNumShorts, ULONG ulReserved );



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_QueryInfo
//
//     Provides a means to get information from the WT Table.  Specifications
//     for what is in the WT Table come from the WTT.H file.
//
//     Params:   hDsp          - The DSP handle, one WT Table per DSP.
//               usWTTBlockTag - WTITCB, Blacklist, etc. tag from WTT.H file.
//               usOffect      - Parameter or field starting location from
//                               WTT.H file.
//               ulNumShorts   - Number words to copy from the WTT starting
//                               with the offset into the block.  Field lengths
//                               are defined in WTT.H file.  For full block lengths,
//                               use the WTT_QueryBlockLength function.
//               lpsBuffer     - Location to return information in
//               ulReserved    - A reserved location for future API expansion.
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
ULONG APIENTRY WTT_QueryInfo (HDSP hDsp, USHORT usWTTBlockTag, USHORT usOffset, ULONG ulNumShorts, short FAR *lpsBuffer, ULONG ulReserved );

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_RegisterCallback
//
//     Registers a callback function from the clients to use for
//     country change notifications.  This bypasses the mwagent approach.
//
//     Params:   CLIPROC
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
//ULONG APIENTRY WTT_RegisterCallback(CLIPROC NotifyClientDirect);

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WTT_UnregisterCallback
//
//     Performs cleanup when a client closes the callback function
//
//     Params:   none
//
//     Returns:  ULONG WTTPOSERR_... from list above
//
//-----------------------------------------------------------------------------
//ULONG APIENTRY WTT_UnregisterCallback();


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//
// Start of OLD API definition
//
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
// Errors returned by the old API:

// In some cases zero may be an error.
// Errors are a negative copy of WTTPOSERR_... errors listed above except
// the specific ones listed below.

#define WT_ERR_MEM_ALLOCATION                           -7
#define WT_ERR_INVALID_FIELD                           -11
#define WT_ERR_INVALID_ITEM                            -10
#define WT_ERR_WTFREE_FAILED                           -15

//-----------------------------------------------------------------------------



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
// World Trade Server Messages:
//
//
//-----------------------------------------------------------------------------
// WT_TABLECHANGED:
//    Sent to all registered clients when the world trade table changes.
//
//    Params:  None

#define WT_TABLECHANGED                  CLI_USERBASE




//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtGetNumCountries
//
//     Retrieves the number of country records in the World Trade
//     database.
//
//     Params:   None
//
//     Returns:  (short) - number of country records
//
//-----------------------------------------------------------------------------
short APIENTRY WtGetNumCountries( void );



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtGetCountryName
//
//     Retrieves the country name corresponding to a record number if there
//     is a corresponding country table available on the system.
//
//     Params:   lCountryNum  - country record number
//               lpstrName    - pointer to buffer where name string will be
//                              placed
//               nBufSize     - size of name string buffer
//
//     Returns:  Successful:  lCount - number of bytes copied into buffer
//               Error:       (long) < 0 (-WTTPOSERR_...)
//
//-----------------------------------------------------------------------------
long APIENTRY WtGetCountryName(long lCountryNum, char *lpstrName, int nBufSize);



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtGetCurrentCountry
//
//     Retrieves the country number currently in use.
//
//     Params:   none
//
//     Returns:  Successful:  lCN - current country number
//
//-----------------------------------------------------------------------------
long APIENTRY WtGetCurrentCountry(void);



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtChangeCountry
//
//     Sets the country number based on the input parameter.
//
//     Params:   lCountryNum - country record number
//
//     Returns:  Successful:  lCount - number of bytes copied into buffer
//               Error:       (long) < 0 ( -WTTPOSERR_...)
//
//-----------------------------------------------------------------------------
long APIENTRY WtChangeCountry(long lCountryNum);


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  This function is obsoleted by WtChangeCountry.
//
//-----------------------------------------------------------------------------
void APIENTRY WtDefaultCountryHasChanged(void);


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtQueryRecord
//
//     Obsolete, this function is not implemented.
//
//     Returns:  Failure:   -WTPOSERR_OBSOLETE_FUNCTION
//
//-----------------------------------------------------------------------------
long APIENTRY WtQueryRecord(long lCountryNum, unsigned short FAR* FAR* lplpRecordBuf);



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtQueryField
//
//     Allocates and fills a buffer with the data for the country field
//     requested.  This function will be used primarily by the MME drivers
//     which need to transfer 'fields' of information to the DSP.  The caller
//     must call WtFree() to release the storage allocated by this function.
//
//     Params:   lpszAccessCode - FAR pointer to null terminated string
//                                containing international access code.  If
//                                NULL then the record for the default country
//                                will be retrieved.
//
//               lFieldQuery    - one of the constants listed below specifying
//                                the field
//
//               lpFieldBuf     - FAR pointer which will be filled with a
//                                pointer to the field for the country
//                                requested.
//
//     Returns:  Successful:  lBufSize > 0 is the size of the record in words
//               Failure:     (long) < 0 ( -WTTPOSERR_... above)
//                            The data in the buffer is not valid.
//
//-----------------------------------------------------------------------------
long APIENTRY WtQueryField(char FAR* lpszAccessCode, long lFieldQuery, unsigned short FAR* FAR* lplpFieldBuf);

// Field Codes:

#define WT_FIELD_BLACKLIST                      0L
#define WT_FIELD_PSTN_CALL_PROGRESS             1L
#define WT_FIELD_TAIO                           2L
#define WT_FIELD_DESCRIPTION                    3L
#define WT_FIELD_PLATFORM_SPECIFIC              4L
#define WT_VOICE_CONTROLS                       5L
#define WT_NUMBER_FIELDS                        6L



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtQueryItem
//
//     Returns a single 'item' of information.  This function will be used
//     primarily by the MME drivers to query information such as functions
//     allowed.  ItemQuery values will be added as requested by developers.
//
//     Params:   lpszAccessCode - FAR pointer to null terminated string
//                                containing internationl access code.  If NULL
//                                then the record for the default country will
//                                be retrieved.
//
//               lItemQuery     - one of the constants listed below specifying
//                                the item to be retrieved
//
//               lpultem        - FAR pointer to storage for retrieved item
//
//     Returns:  Successful:    (long) 0
//               Failure:       (long) < 0 (WT_ERR_... above)
//
//-----------------------------------------------------------------------------
long APIENTRY WtQueryItem(char FAR* lpszAccessCode, long lItemQuery, unsigned long FAR* lpulItem);

// Item Codes: - Note:  some of these defines mis-represent the parameter
//               contents.  A better approach is to use the WTT_QueryInfo function.


#define WT_SMART_CABLE_ID               /* WTTCID   */  0L
#define WT_ALLOWED_FUNC_1               /* WTTAF1   */  1L
#define WT_ALLOWED_FUNC_2               /* WTTAF2   */  2L
#define WT_ALLOWED_FUNC_3               /* WTTAF3   */  3L
#define WT_G3_FAX_CTRL                  /* WTTG3FAX */  4L
#define WT_DTMF_ALLOWED_SET             /* WTTDTADS */  5L
#define WT_CALL_PROG_CTRL               /* WTTCPM   */  6L
#define WT_MIN_RINGS                    /* WTTRNGS  */  7L
#define WT_PULSE_DIAL_MODE              /* WTTDMODE */  8L
#define WT_BLIND_DIAL_WAIT_TIME         /* WTTBDWT  */  9L
#define WT_DIAL_TONE_WAIT_TIME          /* WTTDTWT  */ 10L
#define WT_CARRIER_WAIT_TIME            /* WTTCW    */ 11L
#define WT_AUTO_MODEM_DISCONNECT_TIME   /* WTTAMD   */ 12L
#define WT_LOSS_OF_CARRIER_WAVE_TIME    /* WTTLCW   */ 13L
#define WT_FULL_DUPLEX_TX_LVL           /* WTTFDXL  */ 14L
#define WT_HALF_DUPLEX_TX_LVL           /* WTTHDXL  */ 15L
#define WT_AVG_VOICE_TX_LVL             /* WTTVXL   */ 16L
#define WT_AUTO_CALL_DELAY              /* WTTACD   */ 17L
#define WT_FLASH_HOOK_TIME              /* WTTFH    */ 18L
#define WT_RESERVED_1                                  19L
#define WT_RESERVED_2                                  20L
#define WT_VOICE_SUPPORT_FLAGS                         21L
#define WT_BLIND_DIAL_WAIT_TIME_MAX     /* WTTBDWTX */ 22L
#define WT_CARRIER_WAIT_TIME_MAX        /* WTTCWX   */ 23L
#define WT_LOSS_OF_CARRIER_WAVE_MAX     /* WTTLCWX  */ 24L
#define WT_DTMF_SIGNAL_DURATION         /* WTTDTSD  */ 25L
#define WT_MIC1_GAIN_MAX                               26L
#define WT_MIC1_GAIN_MIN                               27L
#define WT_MIC1_GAIN_NOM                               28L
#define WT_MIC2_GAIN_MAX                               29L
#define WT_MIC2_GAIN_MIN                               30L
#define WT_MIC2_GAIN_NOM                               31L
#define WT_MAX_SPEAKER1_VOLUME                         32L
#define WT_MAX_SPEAKER2_VOLUME                         33L
#define WT_WTTCFSR   /* V.34 Carrier Frequency and Symbol Rate  */ 34L
#define WT_WTT34FBR  /* V.34 Features and Bit Rate              */ 35L

#define WT_NUMBER_ITEMS                                36L


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//  WtFree
//
//     Frees storage allocated by a previous WtQueryField call.
//
//     Params:   lpBuffer - FAR pointer to FAR pointer to buffer which was
//                          allocated by a previous query.
//
//     Returns:  Successful:  (long) 0, storage was freed
//               Failure:     (long) < 0 (WT_ERR_...)
//
//-----------------------------------------------------------------------------
long APIENTRY WtFree(unsigned short FAR* FAR* lplpBuffer);



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtCountryRecordChanged
//
//     Called by the Table maintenance utility when it changes a record.  This
//     routine will refresh the data if the record changed is currently the
//     default.  In this case, WT_TABLECHANGED message will be sent to all
//     registered clients.  It is their responsibility to refresh data obtained
//     from the database.
//
//     Params:   lCountryNum - country number of record changed
//
//     Returns:  None
//
//-----------------------------------------------------------------------------
void APIENTRY WtCountryRecordChanged(long lCountryNum);


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtGetDatabaseName
//
//     Obsolete
//
//     Params:   lCountryNum  - country ID number as defined by WtQueryRecord
//               lpszName     - FAR pointer to storage for database name
//
//     Returns:
//               Failure:     FALSE
//
//-----------------------------------------------------------------------------
BOOL APIENTRY WtGetDatabaseName(long lCountryNum, char FAR* lpszName);



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtCPsourceFileName
//
//     Obsolete
//
//     Returns:  Failure:  FALSE
//
//
//-----------------------------------------------------------------------------
BOOL APIENTRY WtCPsourceFileName(long lCountryNum, char FAR* lpszName);



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WtUseAlternateDatabase
//
//     This function switches which database is the current default.  In the
//     MWAVE.INI file under the [Worldtrade] section is a Database= statement.
//     This is the default database used when this LIB is loaded.  Alternate
//     databases add a number to the Database= statement.  Alternate set
//     1 is Database1=, alternate set 2 is Database2=.  To get back to the
//     original Database= statement, specify set 0.
//
//     Params:   usDatabaseSetNumber - The number of the Database set desired
//
//     Returns:  Successful:  TRUE
//               Failure:     FALSE if there is no corresponding Database? statement
//
//-----------------------------------------------------------------------------
BOOL APIENTRY WtUseAlternateDatabase(unsigned short usLocalDatabaseSetNumber);



//-----------------------------------------------------------------------------

#endif                                  // End of #ifndef __WTTBL_H
