/*
 * seti_control.c
 * 
 * Description: Source file that provides functions to control
 * the setiathome process.
 *
 * Author: Matt Herbert <mherbert@bit-net.com>
 *
 * Date Feb 20, 2000
 *
 * modified: 26/3/00 by Craig Orsinger (CJO) <cjorsinger@earthlink.net>
 *            - removed embedded slashes from comment blocks to
 *              prevent annoying gcc warnings.
 */

#include <gnome.h>
#include <applet-widget.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>

#include <glibtop.h>
#include <glibtop/procstate.h>

#include "seti_control.h"
#include "seti_applet.h"

/*******************************************************************************
 *
 * int is_seti_process( pid_t pid )
 *
 * This function will check a process id to see if it is indeed a setiathome
 * process.
 *
 * returns: 0 if it's not a seti process, 1 if it is
 *
 ******************************************************************************/
static int
is_seti_process ( pid_t pid )
{
	glibtop_proc_state *proc_buf = g_malloc0 ( sizeof ( glibtop_proc_state ) );
	gchar  * return_str;

	glibtop_get_proc_state( proc_buf, pid );

	/* look for setiathome in the cmdline of the process */
	return_str = strstr( proc_buf->cmd, "setiathome" );

	/* clean up */
	g_free( proc_buf );

	/* null pointer returned, that means setiathome wasn't found in the
	 * command line of the process, and it's not a setiathome process!
	 */
	if ( ! return_str )
		return(0);

	/* process exists and is a setiathome process */
	return(1);
}

/*******************************************************************************
 * is_seti_running, check if the seti process we have found is running.
 *
 * This method is present because even if we find a seti process on the machine,
 * it could be a zombie, meaning it isn't really running (just waiting for the
 * parent process to exit).
 *
 * Returns: FALSE if the process is a zombie, TRUE otherwise
 ******************************************************************************/
static int
is_seti_running ( pid_t pid )
{
	glibtop_proc_state *proc_buf = g_malloc0 ( sizeof ( glibtop_proc_state ) );
	gchar  status;

	glibtop_get_proc_state( proc_buf, pid );

	/* look for setiathome in the cmdline of the process */
	status = proc_buf->state;

	/* clean up */
	g_free( proc_buf );

	if ( status == 'Z' )
	{
		return FALSE;
	}

	return TRUE;
}

/*******************************************************************************
 *
 * get_seti_pid( setiapplet *sa )
 *
 * This function will get the process id of a setiathome process from the
 * pid.sah file that setiathome creates.
 *
 * returns: pid_t of the setiathome process, or 0 if it's not found
 *
 ******************************************************************************/
static pid_t
get_seti_pid ( setiapplet *sa )
{
	gchar *file_contents;
	gchar *path_to_pid_file;
	pid_t pid;

	/* Try the new seti pid.sah file name */
	path_to_pid_file = g_strdup_printf ( "%s/%s", sa->setidir, PID_SAH );
	file_contents = re_read_file ( path_to_pid_file );

	/* Try the old seti pid.txt file name */
	if ( !file_contents )
	{
		g_free ( path_to_pid_file );
		path_to_pid_file = g_strdup_printf ( "%s/%s", sa->setidir, PID_TXT );
		file_contents = re_read_file ( path_to_pid_file );

		/* couldn't open any process ID file, bail out */
		if ( !file_contents )
		{
			g_free( path_to_pid_file );
			return( 0 );
		}
	}

	/* convert the pid from the file into an integer */
	pid = atoi( file_contents );

	/* free any memory we allocated */
	g_free( path_to_pid_file );
	g_free( file_contents );

	return(pid);
}

/*******************************************************************************
 *
 * unlink_seti_lock_file( setiapplet *sa )
 *
 * This function will try to unlink the setiathome lock file.  It will
 * try both the seti2 and seti1 lock file names.
 * 
 * TODO: This function could probably use some error checking...
 *
 ******************************************************************************/
static void
unlink_seti_lock_file( setiapplet *sa )
{
	gchar *lock_file_path;

	/*Try the new seti lock.sah file name*/
	lock_file_path = g_strdup_printf ( "%s/%s", sa->setidir, LOCK_SAH );
	unlink( lock_file_path );

	g_free ( lock_file_path );

	/* try the old seti lock.txt file name */
	lock_file_path = g_strdup_printf( "%s/%s", sa->setidir, LOCK_TXT);
	unlink( lock_file_path );

	/* clean up */
	g_free( lock_file_path );
}


/*******************************************************************************
 *
 * seti_status( setiapplet *sa )
 *
 * This function will check to see if setiathome is running.
 *
 * returns: SETI_RUNNING if seti is running and SETI_STOPPED if it's not running
 *
 ******************************************************************************/
int
seti_status ( setiapplet *sa )
{
	int    fd;
	int    seti_pid;
	int    stat_rv;
	struct stat *stat_buf = g_malloc0 ( sizeof ( struct stat ) );
	gchar  *lock_file_path;

	/*Try the new seti lock.sah file name*/
	lock_file_path = g_strdup_printf ( "%s/%s", sa->setidir, LOCK_SAH );

	/* Check if the lock file exists */
	stat_rv = stat ( lock_file_path, stat_buf );
	if ( stat_rv == -1 )
	{
		/* Also look for the old seti lock.txt file name */
		g_free ( lock_file_path );
		lock_file_path = g_strdup_printf( "%s/%s", sa->setidir, LOCK_TXT);
		stat_rv = stat ( lock_file_path, stat_buf );
		if ( stat_rv == -1 )
		{
			/* Nope there is no lock file! that means we are ok to
			 * start the seti client */
			g_free( lock_file_path );
			g_free( stat_buf );
			return SETI_STOPPED;
		}
	}

	/* clean up */
	g_free( lock_file_path );
	g_free( stat_buf );

	/* ok, there is a lock file, lets check the process and see if it's
	 * actually a setiathome process...
	 */
	seti_pid = get_seti_pid ( sa );
	if ( ! is_seti_process( seti_pid ) )
	{
		/* that isn't a setiathome process! it must have died, lets
		 * remove the lock
		 */
		unlink_seti_lock_file( sa );
		return SETI_STOPPED;
	}
	/*RK 29/2/00 - check if the process is alive, if zombie, return 
	 * SETI_STOPPED else return SETI_RUNNING */
	if ( is_seti_running ( seti_pid ) )
	{
		/* seti is running */
		return SETI_RUNNING;
	}
	else
	{
		return SETI_STOPPED;
	}

}


/*******************************************************************************
 *
 * stop_seti( setiapplet *sa )
 *
 * This function will stop the currently running seti process
 *
 ******************************************************************************/
static void
stop_seti ( setiapplet *sa )
{
	pid_t pid;
	gchar *lock_file_path;
	int   rc;

	/* get the process id from the pid.sah file */
	pid = get_seti_pid( sa );

	/* check to make sure this is actually a setiathome process */
	if ( ! is_seti_process( pid ) )
	{
		/* that pid doesn't belong to a setiathome process, that means
		 * seti probably died, and left a mess lying around.  We
		 * should probably clean up after it by removing the lock file.
		 */
		unlink_seti_lock_file( sa );
		return;
	}


	/* kill the setiathome process, trying the "nice" signal first */
	kill( pid, 2 );

	/* give the setiathome process time to die */
	sleep(2);
	
	/* check to see if it actually died... */
	if ( is_seti_process( pid ) )
	{
		/* the process could not be interupted, try SIGKILL (-9) */
		kill( pid, 9 );
		if ( is_seti_process( pid ) )
		{
			/* succesfully killed setiathome with a mean
			 * kill signal.  we need to remove the lock file.
			 */
			 unlink_seti_lock_file( sa );
		}
		else
		{
			/* maybe popup a dialog box here?? */
			printf("Unable to kill setiathome process!\n");
		}

	}
}

/*******************************************************************************
 *
 * start_seti( setiapplet *sa )
 *
 * This function will start the setiathome process
 *
 ******************************************************************************/
static void
start_seti ( setiapplet *sa )
{
	int pid, rc;
	gchar **argsr;
	gchar *seti_loc;
	

	pid = fork();
	if ( pid == 0 )
	{
		argsr = get_extra_args(sa);
		/*
 		 * NOTE:we do not care about memory cleaning as this method is
		 * called only after a fork and before an exec to start the seti
		 * process.
		 */
		chdir( sa->setidir );
		if ( sa->separateexedir )
		{
			seti_loc = g_strdup_printf ( "%s/setiathome", sa->setiexedir );
		}
		else
		{
			seti_loc = g_strdup_printf ( "%s/setiathome", sa->setidir );
		}
		rc = execv ( seti_loc, argsr );
		printf("could not exec setiathome: errno=%d", rc);
		_exit(127);
	}

	/* because it takes a moment for setiathome to startup, we
	 * need to sleep for a second and let it get going.  This will
	 * allow setiathome to create it's lock file
	 */
	sleep(1);
}

/*******************************************************************************
 * get_extra_args( setiapplet *sa )
 *
 * This method parses the extraparams member of the setiapplet structure to
 * retrieve any extra command line parameters for starting the seti client
 *
 *
 * Returns: An array of gchar* containing each of the extra parameters
 ******************************************************************************/
static gchar**
get_extra_args ( setiapplet *sa )
{
	gchar *nice_concat;

	if ( sa->extraparams )
	{
		nice_concat = g_strdup_printf ( "setiathome -nice %d %s", sa->nicevalue, sa->extraparams );
	}
	else
	{
		nice_concat = g_strdup_printf ( "setiathome -nice %d", sa->nicevalue );
	}
	return g_strsplit ( nice_concat, " ", 19 );
}

/*******************************************************************************
 *
 * start_stop_seti_cb( AppletWidget *widget, gpointer sa )
 *
 * This is the callback for the stop/start seti menu pick of the applet
 *
 ******************************************************************************/
void
start_stop_seti_cb ( AppletWidget *widget, gpointer sa )
{
	if ( seti_status(sa) == SETI_RUNNING )
	{
		stop_seti(sa);
	}
	else
	{
		start_seti(sa);
	}

	/* Change the menu entry */
	update_applet_menu();
}

