// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/interface/gc_for_orp.h,v 1.2 2001/12/18 15:58:39 rlhudson Exp $
//


// This is a global include file which provides the ORP an interface to the
// GC. This interface is the only supported interface that the ORP should call
// to talk to the GC. All routines in this C interface will begin with "gc_"
//
// The GC expects that there is a orp_for_gc.h file holding the only  
// interface that the GC will use to talk to the GC.
// 

#ifndef _GC_FOR_ORP_H_
#define _GC_FOR_ORP_H_

#include "platform.h"
#include "gc_header_format.h"
#include "object_layout.h"
#include "orp_types.h"
#include "jni.h"


#ifdef __cplusplus
extern "C" {
#endif


//
// Reveal the parts of the VTable and the Class data types that 
// the GC needs to know about.
//
#ifdef GC_REWORK
typedef VTable Partial_Reveal_VTable;
typedef Class Partial_Reveal_Class;
#else
#error "this is old code that won't work with NEW_OBJ_LAYOUT"
//
// forward declarations
//
typedef struct Partial_Reveal_Class Partial_Reveal_Class;

typedef struct Partial_Reveal_VTable {
	Partial_Reveal_Class	*clss;
} Partial_Reveal_VTable;

typedef struct Partial_Reveal_Class {

/////////////////////////////////////////////////////////////////////
//////// The first few fields can not be changed without reflecting
//////// the changes in the type Partial_Class in orp_for_gc.h
////////////////////////////////////////////////////////////////////

    // NOTE: this p_vtable field must be first since it points to vtable 
    Partial_Reveal_VTable  *p_vtable;  

	void *signers_field_located_in_java_lang_class_in_gnu_classpath;

	//
	// super class of this class; initiially, it is the string name of super
	// class; after super class is loaded, it becomes a pointer to class
	// structure of the super class.
    //
    // The offset of this field is returned by class_get_super_offset.
    // Make sure to update this function if the field is moved around.
	//
    void *not_revealed;
//	union {
//		String  *super_name;
//		Partial_Reveal_Class	*super_class;
//    };


    //
    // See the masks in orp_for_gc.h.
    //

    uint32 class_properties;


 
    // This typically holds an array of offsets to the pointers in 
    // and instance of this class. What is here is totally up to the
    // garbage collector side of the interface.
    // *** Union with array_element_size at some point. ***

    unsigned int *gc_information;

    // The number of bytes allocated for this object. It is the same as
    // instance_data_size with the constraint bit cleared. This includes
    // the OBJECT_HEADER_SIZE as well as the OBJECT_VTABLE_POINTER_SIZE
    unsigned int allocated_size;

    unsigned int array_element_size;
 

/////////////////////////////////////////////////////////////////////
//////// Fields above this point can not be moved without redefining
//////// the data structure Partial_Class in orp_for_gc.h
/////////////////////////////////////////////////////////////////////
} Partial_Class;

#endif

typedef struct Partial_Reveal_JavaArray {
	Partial_Reveal_VTable	*vt;

#ifndef OLD_OBJ_LAYOUT
    POINTER_SIZE_INT obj_info;
#endif
    int32 length;
} Partial_Reveal_JavaArray;

typedef struct Partial_Reveal_JavaObject {
	Partial_Reveal_VTable	*vt;
    
#ifndef OLD_OBJ_LAYOUT
    POINTER_SIZE_INT obj_info;
#endif
} Partial_Reveal_JavaObject;

//
// In order to eliminate dependency on certain types such as (VTable *) we 
// have eliminated them from this interface and replace them with (void *).
// While this might appear to be unfortunate it allows us to eliminate any 
// knowledge of the class and VTable structures that are not of interest 
// to the GC.
//

//
// DLL stuff
//

#ifdef USE_GC_DLL
#ifdef BUILDING_GC
#define GCExport __declspec(dllexport)
#else
#define GCExport __declspec(dllimport)
#endif
#else // US_GC_DLL
// GC is part of the ORP, not part of the JIT, so we use BUILDING_ORP

#ifdef BUILDING_ORP
#define GCExport __declspec(dllexport)
#else 
#define GCExport __declspec(dllimport)
#endif
#endif // US_GC_DLL

// *****
// *
// *  Routines to support the initialization and termination of GC.
// * 
// *****

//
// API for the ORP to hand to the GC any arguments starting with -gc. 
// It is up to the GC to parse these arguments.
// In order to maintain compatability with other systems -mx and -ms will
// also be passed to this routine.
// For example -gcverbose and -gcplan are reasonable formats of params to pass.
//
// Input: name - a string holding the name of the parameter 
//               assumed to begin with "-gc"
//        arg  - a string holding the argument following name on 
//               the command line.
//
GCExport void gc_next_command_line_argument(const char *name, const char *arg);

//
// gc_init initialized the GC internal data structures.
// This routine is called after the last call to gc_next_command_line_argument.
// The ORP should call this *before* any other calls to this interface except
// calls to gc_next_command_line_argument.
//
GCExport void gc_init();

//
// This API is used by the ORP to notify the GC that the
// ORP has completed bootstrapping and initialization, and 
// is henceforth ready to field requests for enumerating 
// live references.
//
// Prior to this function being called the GC might see some
// strange sights such as NULL or incomplete vtables. The GC will
// need to consider these as normal and work with the ORP to ensure 
// that bootstrapping works. This means that the GC will make few
// demands on the ORP prior to this routine being called.
//
// However, once called the GC will feel free to do 
// stop-the-world collections and will assume that the entire
// orp_for_gc interface is available and fully functioning.
//
// If this routine is called twice the result is undefined.
//
GCExport void gc_orp_initialized();

//
// This is called once the ORP has no use for the heap or the 
// garbage collector data structures. The assumption is that the 
// ORP is exiting but needs to give the GC time to run destructors 
// and free up memory it has gotten from the OS.
// After this routine has been called the ORP can not relie on any
// data structures created by the GC.
//
// Errors: If gc_enumerate_finalizable_objects has been called and
//         gc_wrapup gc discovers an object that has not had it
//         finalizer run then it will attempt to report an error.
//
GCExport void gc_wrapup();

// *****
// *
// *  Routines to support finalization of objects.
// * 
// *****

//
// This function is provided by the ORP and called by GC 
// when an object becomes "f-reachable, finalizable"
// The ORP later finalizes those objects in a way that
// is not part of this interface.
// This routine will be available from orp_for_gc.h
//
// ORPExport void orp_finalize_object(Java_java_lang_Object *p_obj);
//

//
// A function called by the ORP before exit.
//
// The GC assumes that all Java objects are dead and
// enumerates all those that have overridden finalize()
// using the orp_finalize_object API.
//
// The GC may assume that finalizers are the only Java
// code that runs after that function has been called.
// 
// The GC should assume that all objects are live
// and shouldn't reclaim any memory.
//
// The expected scenario is:
// 1. The ORP calls gc_enumerate_finalizable_objects()
// 2. The ORP runs all finalizers (note that in
//    a pathological case this could be a non-trivial
//    piece of code and multiple garbage collections
//    may happen in that phase).
// 3. The ORP calls gc_wrapup()
//
// Issues:
// 1. What happens when a finalizer creates new objects
//    which require finalization?  Do we keep calling
//    gc_enumerate_finalizable_objects() until convergence?
//    - For now gc_enumerate_finalizable_objects will be
//      called only once and finalizers that create finalizable
//      object will have undefined results.
//
GCExport void gc_enumerate_finalizable_objects();

// *****
// *
// *  Routines to support querying the layout of objects.
// * 
// *****

//
// This routine gets from the ORP a zero terminated array of offsets
// of all the pointers in an instance of a the class. This routine
// is then responsible for remembering these locations in whatever 
// format seems appropriate. 
// The purpose of this slot is to allow the GC to associate the
// information provided in an optimized permanent form. 
// 
//
// Input: p_class - a pointer to the base of a class data structure. 
//        ref_array - a zero deliminated transient array holding offsets
//                    to all the pointers in an instance of p_class.          
//              
#ifdef GC_REWORK
GCExport void gc_class_ref_map (Partial_Reveal_Class *p_class,
                                unsigned int *ref_array);
#else
GCExport void gc_class_ref_map (Partial_Reveal_Class *p_class,
                                unsigned int *ref_array);
#endif
// This routine provides the ORP with a read only zero terminated array 
// of offsets of all pointers in the instance of a class. The 
// array is in the same format as the array passed to it in 
// gc_class_ref_map. The ORP will not alter this array or expect it to
// persist. ref_array is populated.
// 
// Input:    p_class - a pointer to the base of a class data structure.
//           length - the size of the array to be filled. If this is not
//                    sufficient then it is an internal error.
// Returns   ref_array - a zero deliminated potentially transient array holding offsets
//                       to all the pointers in an instance of p_class. 

#ifdef GC_REWORK  
GCExport void gc_get_class_ref_map (VTable *p_class,
                                    unsigned int length,
                                    unsigned int *ref_array);
#else
GCExport void gc_get_class_ref_map (Partial_Reveal_Class *p_class,
                                    unsigned int length,
                                    unsigned int *ref_array);
#endif
// *****
// *
// *  Routines to support various write barriers.
// * 
// *****

//
// In order to improve performance several of these routines could be inlined
// by the JIT and/or ORP. When we measured the speedup of the JIT inlining the card marking
// routines we saw a 3-5% speedup on 3 or the Spec routines and none on the others.
//

enum GC_Barriers {
    GC_NO_BARRIER,
    GC_CARD_MARK_WRITE_BARRIER,
    GC_SAPPHIRE_WRITE_BARRIER,
    GC_TRACE_BARRIER
}; //GC_Barriers

//
// This API is used by the ORP (JIT) to find out if
// the GC requires read or write barriers. If the GC requests
// write barriers and the JIT does not support write barriers the results
// are undefined.
//
// Output: 
//         GC_NO_BARRIER - JIT does not have to emit write barriers.
//         GC_CARD_MARK_WRITE_BARRIER if the garbage collector expects to 
//              be notified whenever a slot holding a pointer to an object 
//              is modified in the heap. Marking the card is sufficient if
//              the JIT does not allow GC safepoints between the mark and the
//              heap write.
//         GC_SAPPHIRE_WRITE_BARRIER if JIT must do a write barrier for every
//              write to the heap using the interface provided below.
//
// Comments: Future version might extend possible return values 
//           so that the system can support other barriers such as a 
//           read barrier or a write barrier on all heap writes, not 
//           just pointer writes.
//
//
           
GCExport GC_Barriers gc_requires_barriers();

#if 0
//
// If gc_requires_write_barriers() returns true, the Jit-ted code 
// should call this after doing the putfield of an object.
//
// Likewise the ORP needs to call it whenever it stores a pointer into the
// heap.
//
// Input: p_base_of_obj_with_ref - the base of the object that
//                                 had a pointer stored in it.
//
#ifdef BUILDING_ORP
ORPExport 
#else
GCExport
#endif
#endif

// These interfaces are marked for replacement for the IA64 by
// the following gc_heap_write_mumble interface.

GCExport void gc_write_barrier(Java_java_lang_Object *p_base_of_obj_with_slot);
GCExport void __fastcall gc_write_barrier_fastcall(Java_java_lang_Object *p_base_of_obj_with_slot);

GCExport void gc_write_barrier_atomic(Java_java_lang_Object *p_base_of_object_with_slot,
                               Java_java_lang_Object **p_slot,
                               Java_java_lang_Object *value);

  
GCExport void gc_write_barrier_atomic_non_ref(Java_java_lang_Object *p_base_of_object_with_slot,
                                       Java_java_lang_Object **p_slot,
                                       Java_java_lang_Object *value);
//
// Code for gc algorithms requiring write barriers.
//
// The following routines are the only way to alter any value in the gc heap.  
//

//
// There are two flavors for historical reasons. The compiler for IA32 will produce code
// for the version using an offset.
//

GCExport void gc_heap_wrote_object (Java_java_lang_Object *p_base_of_object_just_written);


GCExport void gc_heap_write_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                                 unsigned offset,
                                 Java_java_lang_Object *value);
GCExport void gc_heap_slot_write_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                                 Java_java_lang_Object **p_slot,
                                 Java_java_lang_Object *value);


GCExport void gc_heap_write_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                                  unsigned offset,
                                  int8 value);
GCExport void gc_heap_slot_write_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                                  int8 *p_slot,
                                  int8 value);


GCExport void gc_heap_write_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                   unsigned offset,
                                   int16 value);
GCExport void gc_heap_slot_write_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                   int16 *p_slot,
                                   int16 value);


GCExport void gc_heap_write_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                    unsigned offset,
                                    uint16 value);
GCExport void gc_heap_slot_write_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                    uint16 *p_slot,
                                    uint16 value);


GCExport void gc_heap_write_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                                   unsigned offset,
                                   int32 value);
GCExport void gc_heap_slot_write_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                                   int32 *p_slot,
                                   int32 value);


GCExport void gc_heap_write_float (Java_java_lang_Object *p_base_of_object_with_slot,
                                   unsigned offset,
                                   float value);
GCExport void gc_heap_slot_write_float (Java_java_lang_Object *p_base_of_object_with_slot,
                                   float *p_slot,
                                   float value);


GCExport void gc_heap_write_double (Java_java_lang_Object *p_base_of_object_with_slot,
                                    unsigned offset,
                                    double value);
GCExport void gc_heap_slot_write_double (Java_java_lang_Object *p_base_of_object_with_slot,
                                    double *p_slot,
                                    double value);


GCExport void gc_heap_write_pointer_size_int (Java_java_lang_Object *p_base_of_object_with_slot,
                                              unsigned offset,
                                              POINTER_SIZE_INT value);
GCExport void gc_heap_slot_write_pointer_size_int (Java_java_lang_Object *p_base_of_object_with_slot,
                                              POINTER_SIZE_INT *p_slot,
                                              POINTER_SIZE_INT value);


GCExport void gc_heap_write_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                                   unsigned offset,
                                   int64 value);
GCExport void gc_heap_slot_write_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                                   int64 *p_slot,
                                   int64 value);

// There are some global slots that are shared by different threads. Sapphire 
// needs to know about writes to these slots. One example of such slots is in
// the string pools used by the class loader.
GCExport void gc_heap_write_global_slot(Java_java_lang_Object **p_slot,
                                        Java_java_lang_Object *value);

//   
// On IA64 the following should generate a st.rel. If the Java spec is changed
// to state that reads and writes of volatiles are sequentially consistence 
// then the st.rel will need to be followed by a memory fence.
//
GCExport void gc_volatile_heap_write_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                                          unsigned offset,
                                          Java_java_lang_Object *value);
GCExport void gc_volatile_heap_slot_write_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                                          Java_java_lang_Object **p_slot,
                                          Java_java_lang_Object *value);

// Non reference writes caller does conversion.


GCExport void gc_volatile_heap_write_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                                           unsigned offset,
                                           int8 value);
GCExport void gc_volatile_heap_slot_write_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                                           int8 *p_slot,
                                           int8 value);


GCExport void gc_volatile_heap_write_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            unsigned offset,
                                            int16 value);
GCExport void gc_volatile_heap_slot_write_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            int16 *p_slot,
                                            int16 value);


GCExport void gc_volatile_heap_write_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                             unsigned offset,
                                             uint16 value);
GCExport void gc_volatile_heap_slot_write_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                             uint16 *p_slot,
                                             uint16 value);


GCExport void gc_volatile_heap_write_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            unsigned offset,
                                            int32 value);
GCExport void gc_volatile_heap_slot_write_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            int32 *p_slot,
                                            int32 value);


GCExport void gc_volatile_heap_write_float (Java_java_lang_Object *p_base_of_object_with_slot,
                                            unsigned offset,
                                            float value);
GCExport void gc_volatile_heap_slot_write_float (Java_java_lang_Object *p_base_of_object_with_slot,
                                            float *p_slot,
                                            float value);


GCExport void gc_volatile_heap_write_double (Java_java_lang_Object *p_base_of_object_with_slot,
                                             unsigned offset,
                                             double value);
GCExport void gc_volatile_heap_slot_write_double (Java_java_lang_Object *p_base_of_object_with_slot,
                                             double *p_slot,
                                             double value);


GCExport void gc_volatile_heap_write_pointer_size_int (Java_java_lang_Object *p_base_of_object_with_slot,
                                                       unsigned offset,
                                                       POINTER_SIZE_INT value);
GCExport void gc_volatile_heap_slot_write_pointer_size_int (Java_java_lang_Object *p_base_of_object_with_slot,
                                                       POINTER_SIZE_INT *p_slot,
                                                       POINTER_SIZE_INT value);


GCExport void gc_volatile_heap_write_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            unsigned offset,
                                            int64 value);
GCExport void gc_volatile_heap_slot_write_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            int64 *p_slot,
                                            int64 value);

#ifdef GC_SAPPHIRE
//
// The monitor enter/exit code needs to know lock the "to" object when we are running
// Sapphire. This routine will translate "from" objects into "to" objects. If the object
// is not a "from" object it is just returned without any translation.
//
Java_java_lang_Object *gc_sapphire_get_lock_object (Java_java_lang_Object *p_obj);

enum sapphire_tri_color {
	white,
	gray,
	black
};

#endif
//
// gc_sapphire_equal - do a pointer compare.
//
// Input: p_obj1, p_obj2 - the objects to compare.
//
// Comment - Since there can be two version of an object during certain Sapphire phases
//                  equal must be smarter than a normal equal. 
//   If we are in phases where the two version are visible to application threadst then
//   if either of the objects have a "to" version then we will use the "to" version to do
//   the pointer compare.
//
int gc_sapphire_equal (Java_java_lang_Object *p_obj1, Java_java_lang_Object *p_obj2);


/****
*
*  Routines to support read barriers
*
****/

#ifdef GC_READ_BARRIERS 
// non inlined version needed for JIT. and  GC_SAPPHIRE
//
// Code for the gc algorithms and traces requiring read barriers.
//
// The following routines are to be called whenever a value in the
// gc heap is read  
//

// In a place with gc disabled an entire object could be read, for example inside
// clone. This means that the entire object must be scanned and treated as if
// all the fields had been read. 

void gc_heap_read_object (Java_java_lang_Object *p_base_of_object)
{
}

Java_java_lang_Object *gc_heap_read_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
// The slot versions of the read barrier routines just call the normal barriers.
Java_java_lang_Object *gc_heap_slot_read_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                         Java_java_lang_Object **p_slot);

// Non reference writes caller does conversion.
int8 gc_heap_read_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
int8 gc_heap_slot_read_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                         int8 *p_slot);

int16 gc_heap_read_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
int16 gc_heap_slot_read_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                         int16 *p_slot);

uint16 gc_heap_read_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
uiint16 gc_heap_slot_read_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                         uint16 *p_slot);

int32 gc_heap_read_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
int32 gc_heap_slot_read_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                         int32 *p_slot);

float gc_heap_read_float (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
float gc_heap_slot_read_float (Java_java_lang_Object *p_base_of_object_with_slot,
                         float *p_slot);

double gc_heap_read_double (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);

double gc_heap_slot_read_double (Java_java_lang_Object *p_base_of_object_with_slot,
                         double *p_slot);

POINTER_SIZE_INT gc_heap_read_pointer_size_int (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);

POINTER_SIZE_INT gc_heap_slot_read_pointer_size_int (Java_java_lang_Object *p_base_of_object_with_slot,
                         POINTER_SIZE_INT *p_slot);

int64 gc_heap_read_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);

int64 gc_heap_slot_read_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                         int64 *p_slot);
  
// There are some global slots that are shared by different threads. Some gc  
// algorithms needs to know about reads from these slots.
Java_java_lang_Object *gc_heap_read_global_slot(Java_java_lang_Object **p_slot);

// The following should generate a ls.acq and a mfence to get sequential consistency
// for volatiles. As of June 12, 2000  this is my (RLH) assumption of what the Memory
// access ordering spec will/should be. Though the need for the mfence might not be 
// required.

Java_java_lang_Objec *gc_volatile_heap_read_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
Java_java_lang_Object *gc_volatile_heap_slot_read_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                         Java_java_lang_Object **p_slot);

// Non reference writes caller does conversion.
int8 gc_volatile_heap_read_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
int8 gc_volatile_heap_slot_read_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                         int8 *p_slot);

int16 gc_volatile_heap_read_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
int16 gc_volatile_heap_slot_read_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                         int16 *p_slot);

uint16 gc_volatile_heap_read_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
uint16 gc_volatile_heap_slot_read_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                         uint16 *p_slot);

int32 gc_volatile_heap_read_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
int32 gc_volatile_heap_slot_read_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                         int32 *p_slot);

float gc_volatile_heap_read_float (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
float gc_volatile_heap_slot_read_float (Java_java_lang_Object *p_base_of_object_with_slot,
                         float *p_slot);

double gc_volatile_heap_read_double (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
double gc_volatile_heap_slot_read_double (Java_java_lang_Object *p_base_of_object_with_slot,
                         double *p_slot);

POINTER_SIZE_INT gc_volatile_heap_read_pointer_size_int (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
POINTER_SIZE_INT gc_volatile_heap_slot_read_pointer_size_int (Java_java_lang_Object *p_base_of_object_with_slot,
                         POINTER_SIZE_INT *p_slot);

utin64 gc_volatile_heap_read_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                         unsigned offset);
uint64 gc_volatile_heap_slot_read_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                         int64 *p_slot);

#endif // GC_READ_BARRIERS

// *****
// *
// *  Routines to support building the root set during a collection.
// * 
// *****

//
// Call from the ORP to the GC to enumerate another
// live reference.
//
// Input: ref - the location of a slot holding a pointer that
//              is NULL or points to a valid object in the heap.
//
GCExport void gc_add_root_set_entry(Java_java_lang_Object **ref);


// 
// Call from the ORP to the gc to enumerate an interior pointer. **ref is a slot holding a pointer
// into the interior of an object. The base of the object is located at *ref - offset. The strategy
// employed is to place the slot, the object base and the offset into a slot_base_offset table. We then
// call gc_add_root_set_entry with the slot in the table holding the base of the object. Upon completion
// of the garbage collection the routine fixup_interior_pointers is called and the slot_base_offset table
// is traversed and the new interior pointer is calculated by adding the base of the object and the offset.
// This new interior pointer value is then placed into the slot.
//
// This routine can be called multiple times with the same interiour pointer without any problems.
// The offset is checked to make sure it is positive but the logic is not dependent on this fact.
GCExport void gc_add_root_set_entry_interior_pointer (void **slot, int offset);
 



// *****
// *
// *  Routines to support the allocation and initialization of objects.
// * 
// *****

//
// Allocation of objects.
//
// There is a tension between fast allocation of objects and 
// honoring various constraints the ORP might place on the object. 
// These constraints include registering the objects for 
// finalization, aligning the objects on multiple word boundaries, 
// pinning objects for performance reasons, registering objects 
// related to weak pointers and so forth.
//
// We have tried to resolve this tension by overloading the 
// size argument that is passed to the allocation routine. If 
// the size of the argument has a high bit of 0, then the 
// allocation routine will assume that no constraints exist 
// on the allocation of this object and allocation can potentially 
// be made very fast. If on the other hand the size is large then 
// the routine will query the class data structure to determine 
// what constraints are being made on the allocation of this object.
//
// The orp_for_gc interface will provide the following masks
// to allow the gc to quickly determine the constraints.
// NON_REF_ARRAY_MASK 0x1000
// ARRAY_MASK         0x2000
// PINNED             0x4000
// FINALIZABLE        0x8000
// ALIGN              0x0FFF
//

//
// This routine is the primary routine used to allocate objects. 
// It assumes nothing about the state of the ORP internal data 
// structures or the runtime stack. If gc_malloc_or_null is able 
// to allocate the object without invoking a GC or calling the ORP
// then it does so. It places p_vtable into the object, ensures 
// that the object is zeroed and then returns a Java_java_lang_Object 
// pointer to the object. If it is not able to allocate the object 
// without invoking a GC then it returns NULL.
//
// Input: size - the size of the object to allocate. If the high bit
//               set then various constraints as described above are
//               placed on the allocation of this object.
//        p_vtable - a pointer to the vtable of the class being 
//                   allocated. This routine will place this value 
//                   in the appropriate slot of the new object.
//
#ifdef GC_REWORK
GCExport Java_java_lang_Object *gc_malloc_or_null(unsigned size, 
                                                  VTable *p_vtable);
#else
GCExport Java_java_lang_Object *gc_malloc_or_null(unsigned size, 
                                                  Partial_Reveal_VTable *p_vtable);

#endif
//
// This routine is used to allocate an object. See the above 
// discussion on the overloading of size.
// The GC assumes that the ORP is ready to support a GC if it 
// calls this function.
//
// Input: size - the size of the object to allocate. If the high bit
//               set then various constraints as described above are
//               placed on the allocation of this object.
//        p_vtable - a pointer to the vtable of the class being allocated.
//                   This routine will place this value in the 
//                   appropriate slot of the new object.
//
#ifdef GC_REWORK
GCExport Java_java_lang_Object *gc_malloc(unsigned size, VTable *p_vtable);
#else
GCExport Java_java_lang_Object *gc_malloc(unsigned size, Partial_Reveal_VTable *p_vtable);
#endif
//
// For bootstrapping situations, when we still don't have
// a class for the object. This routine is only available prior to 
// a call to the call gc_orp_initialized. If it is called after
// the call to gc_orp_initialized then the results are undefined. 
// The GC places NULL in the vtable slot of the newly allocated
// object.
// 
// The object allocated will be pinned, not finalizable and not an array.
//
// Input: size - the size of the object to allocate. The high bit
//               will never be set on this argument.
// Output: The newly allocated object
//
GCExport Java_java_lang_Object *gc_pinned_malloc_noclass(unsigned size);

// Currently unused area_id included to that certain objects can be clustered.
// This could result on better code on machine that limit the size of offsets
// in some of their addressing modes. It is only used as a hint and the caller
// should never depend on the clustering.
typedef unsigned int area_id;

//
// It is convenient for the ORP to pin certain objects. For example, if statics
// are pinned then the JIT can produce more efficient code. If all objects of
// a given class are to be pinned then you should *not* use this interface, instead
// you should us the PINNED mask and structures in the class structure.
// 
// Input: size - the size of the object to allocate. If the high bit
//               set then various constraints as described above are
//               placed on the allocation of this object.
//        p_vtable - a pointer to the vtable of the class being allocated.
//                   This routine will place this value in the 
//                   appropriate slot of the new object.
// Output: The newly allocated pinned object
//
#ifdef GC_REWORK
GCExport Java_java_lang_Object *gc_pinned_malloc(unsigned size, 
                                            VTable *p_vtable,
                                            area_id area_hint); 
#else
GCExport Java_java_lang_Object *gc_pinned_malloc(unsigned size, 
                                            Partial_Reveal_VTable *p_vtable,
                                            area_id area_hint); 
#endif

// *****
// *
// *  Routines to support soft, weak, and phantom reference objects.
// * 
// *****
//
// Input:	reference   - the reference object to register.
//			referent    - the referent of the reference object that is to be retrieved with
//						- the get method.
//
// The weak reference code written in Java and the support code provide by the ORP 
// must agree on what the layout of a Java.lang.ref.Reference object looks like 
// and agree that any subclassing will only append fields to the agreed upon layout. 
// This seems reasonable.
//
// In addition the support code will have exclusive knowledge and control of a single 
// field (called the_referent) which holds the reference to the target object. 
// The java code will assume that this field is a read only integer and should
// not be traced by the gc. The Java.lang.ref.ReferenceQueue layout needs to also 
// be known by the supporting code so that it can move reference objects onto the 
// queues at the appropriate times. The Java code uses normal mechanisms to load 
// the Reference classes and to create a reference.
// 
// The constructor code however needs to call the appropriate register function 
// listed below based upon whether we have a soft, weak, or phantom reference. 
// The ORP support code will fill in the referent field. The routine 
// gc_get_referent will return the value in this field.
// Note that the phantom reference method get will not use the gc_get_referent 
// but instead just return NULL as required by the spec.
// 

JNIEXPORT void JNICALL 
Java_java_lang_ref_Reference_enqueue_reference (JNIEnv *the_env, 
                                                jobject p_obj);

JNIEXPORT jobject JNICALL 
Java_java_lang_ref_Reference_get (JNIEnv *the_env, 
                                  jobject p_obj);

JNIEXPORT void JNICALL 
Java_java_lang_ref_Reference_register_phantom_ref (JNIEnv *the_env, jobject p_obj, 
                                                   jobject referent);

JNIEXPORT void JNICALL 
Java_java_lang_ref_Reference_register_soft_ref (JNIEnv *the_env, jobject p_obj, 
                                                jobject referent);

JNIEXPORT void JNICALL 
Java_java_lang_ref_Reference_register_weak_ref (JNIEnv *the_env, jobject p_obj, 
                                                jobject referent);

GCExport void gc_register_soft_ref (Java_java_lang_Object *reference,
									Java_java_lang_Object *referent);

GCExport void gc_register_weak_ref (Java_java_lang_Object *reference,
									Java_java_lang_Object *referent);

GCExport void gc_register_phantom_ref (Java_java_lang_Object *reference,
 									   Java_java_lang_Object *referent);

GCExport Partial_Reveal_JavaObject *gc_get_referent (Java_java_lang_Object *reference);

GCExport Partial_Reveal_JavaObject *gc_clear_referent (Java_java_lang_Object *reference);



//
// *****
// *
// *  Routines to support threads.
// * 
// *****
//
// This routine is called during thread startup to set
// an initial nursery for the thread.
//
// Comment - gc_thread_init and gc_thread_kill assume that
//           the current thread is the one we are interested in
//           If we passed in the thread then these things could be
//           cross inited and cross killed.
//
GCExport void gc_thread_init();

// Release the nursery if it is non NULL.
GCExport void gc_release_nursery (void *p_nursery);

//
// This is called just before the thread is reclaimed.
//
GCExport void gc_thread_kill();

// *****
// *
// *  Routines to support the functionality required by the Java language specification.
// * 
// *****

//
// API for the ORP to force a GC, typically in response to a call to 
// java.lang.Runtime.gc
//
GCExport void gc_force_gc();

//
// API for the ORP to determine the total GC heap, typically in response to a
// call to java.lang.Runtime.totalMemory
//
GCExport int64 gc_total_memory();

//
// API for the ORP to get an approximate view of the free space, 
// typically in response to a call to java.lang.Runtime.freeMemory
//
GCExport int64 gc_free_memory();

//
// The ORP need to manage a nursery so that it can respond to
// the orp_get_nursery call in the orp_for_gc.h interface. Typically gc_get_new_nursery
// this is called during initialization of a thread after gc_thread_init
// is called. If there is thread/CPU affinity then the thread 
// initialization may use a nursery associated with a thread and avoid calling
// gc_get_new_nursery for every thread.
//
// The important thing to note is that this decision is on the orp side of this
// interface.
GCExport void *gc_get_new_nursery();

//
// There is strong evidence that inlining the marking of cards has
// a positive affect on performance. The number of bits one needs
// to shift a heap address to get the card table index along
// with the virtual base of the card table are the only two pieces of
// information needed for this.
//

#ifdef POINTER64
#define GC_CARD_SHIFT_COUNT 13
#else
#define GC_CARD_SHIFT_COUNT 12
#endif

#ifdef OBJECT_SPLITTING
int gc_get_cold_region_offset();
#endif // OBJECT_SPLITTING

/****
*
*  Routines to support the functionality required by Jini to see if an object is pinned.
* 
*****/

#ifdef USE_GC_DLL
GCExport
#endif

bool gc_is_object_pinned (Java_java_lang_Object *obj);


#ifdef GC_SAPPHIRE
//
// Main stop-the-world entry/exit hooks, needed by sapphire to monitor and collect
// enumeration times.
//

enum thread_suspend_state {
    SUSPENDED_AT_SAFE_POINT,
    SUSPENDED_BY_ENABLED_WILL_BLOCK,
    SUSPENDED_AT_UNSAFE_POINT,
};
void thread_enum_start_hook();
void thread_enum_end_hook(thread_suspend_state how_suspended);
#endif // GC_SAPPHIRE

//
// End of the supported interface.
//

// *************************************************************************
//
//      ***********************************************************
//      *                                                         *
//      *    Warning                                   Warning    *
//      *           Warning                     Warning           *
//      *                  Warning       Warning                  *
//      *                         Warning                         *
//      *                  Warning       Warning                  *
//      *           Warning                     Warning           *
//      *    Warning                                   Warning    *
//      *                                                         *
//      ***********************************************************
//
// Code found below this line is provided for debugging and tuning 
// the current system. Implementors should not assume that these 
// routines will be available later today, much less tomorrow. The 
// mere thought that they might be around at the next release should 
// be dealt with by discussing if they should be moved above this line.
//
// *************************************************************************

//
// The class loader is notifying the GC that another class has
// just been loaded.
//
#ifdef GC_REWORK
GCExport void gc_class_loaded (VTable *p_vt);
#else
GCExport void gc_class_loaded (Partial_Reveal_VTable *p_vt);
#endif

//
// Objects are aligned on 4 or 8 bytes. If they are aligned on 8 bytes then
// Arrays will be required to start on the indicated alignement. This means that
// for 8 byte alignment on the IA32 the header will look like this
//
// uint32 gc_header_lock_hash
// VTable *vt
// uint32 array_length
// uint32 padding
// the array elements.
//
#ifdef POINTER64
#define GC_OBJECT_ALIGNMENT 8
#else

#ifdef EIGHT_BYTE_ALIGN
#define GC_OBJECT_ALIGNMENT 8
#elif SIXTY_FOUR_BYTE_ALIGN
#define GC_OBJECT_ALIGNMENT 64
#else
#define GC_OBJECT_ALIGNMENT 4
#endif
#endif

// This can be called from anywhere to trace what an object is doing.
// Input object   - the object that is being manipulated.
//       string_x - If object matches one of the objects being traced then
//                  this string is printed out.
//
// This is only valid if GC_DEBUG>0, in other cases an empty inlined routine is
// called which should result in a NOP. Others have done this kind of stuff with
// macros and I respect the power of macros but I need to debug this code.
//
#ifdef _DEBUG
void gc_trace (void *object, const char *string_x);

void gc_trace_slot (void **object_slot, void *object, const char *string_x);
#else
void inline gc_trace(void *object, const char *string_x) 
{
    // set up a latency free routine when not debugging....
}

void inline gc_trace_slot (void **object_slot, void *object, const char *string_x)
{
    // set up a latency free routine when not debugging....
}
#endif // (GC_DEBUG>0) 

#ifdef __cplusplus
}
#endif

#endif // _GC_FOR_ORP_H_
