// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/block_store.cpp,v 1.11 2002/01/11 15:47:38 weldon Exp $
//

#include "gc_for_orp.h"
 
#include "block_store.h"
#include "generation.h"
#include "gc_space.h"
#include "obsolete_space.h"
#include "car.h"
#include "train.h"
 
#include "gc_plan.h"
#include "gc_hooks.h"
#include "orp_synch.h"
#include "gc_debug.h"
#include "mrl_gc_v1.h"

#ifdef GC_PT_WB
#include <windows.h>
#include <winbase.h>
#include <winnt.h>
#include <stdio.h>    
#endif

// At somepoint this could be a table lookup to get a single block.
// If extend is true we weill extend the heap instead of calling gc.
block_info *p_get_new_block(bool extend, bool collect_returns_null, bool in_a_gc)
{
    return p_global_bs->p_get_multi_block (GC_BLOCK_ALLOC_SIZE, extend, collect_returns_null, in_a_gc);
}

int
_initialize_and_count_blocks (void *p_start_address,
                              void *p_last_block) 
{
    int count = 0;

    // Here we link the blocks together through the next field.
    block_info *p_block_info;
    for (p_block_info = (block_info *)p_start_address; 
         p_block_info != (block_info *)p_last_block;
         p_block_info = (block_info *)((POINTER_SIZE_INT)p_block_info + (POINTER_SIZE_INT)GC_BLOCK_SIZE_BYTES) ) 
         {
             gc_trace_block (p_block_info, " in _initialize_and_count_blocks");
             p_block_info->next = NULL;
             count++;
         }
    p_block_info->next = NULL;
    count++;

    assert (p_last_block == p_block_info);
    return count;
}

//
// The following routine is called when the block store is created.
// It is used to create one large unallocated super-block. 
// The fact that it is unallocated is indicated by the fact that the
// first sub-block has a NULL container, and all the block sizes ,
// except the first, are -1.
//
//
void
Block_Store::_initialize_block_tables(void *p_start_address,
									  void *p_last_block)
{
    // Split this in half so that half goes into the single block bucket.
    int count = _initialize_and_count_blocks (p_start_address, p_last_block);
    int i;
    for (i=0; i<MAX_BUCKET; i++) {
        free_blocks[i] = NULL;
        // Calculate the size for each bucket and make sure it is correct.
    }
    ((block_info *)p_start_address)->block_status_table_index = 0;
    ((block_info *)p_start_address)->number_of_blocks = count; 
    free_blocks[MAX_BUCKET - 1] = (block_info *)p_start_address;
    ((block_info *)p_start_address)->in_free_p = true;
#ifdef _DEBUG
    // Everything should be free.
    for (i = 0; i < count; i++) {
        assert(gc_block_status_table[i] == block_in_free);
    }
#endif
    block_group_list = (block_group_link *) malloc (sizeof (block_group_link));
    block_group_list->next = NULL;
    block_group_list->block_list = (block_info *)p_start_address;
    
    return;
}

Block_Store::Block_Store(Gc_Fast_Hooks *p_gc_hooks,
                         Gc_Plan       *p_gc_plan,
                         unsigned int  sub_block_size_bytes)
{
	_p_gc_plan = p_gc_plan;

    //
    // Create an obsolete space for debugging purposes.
    //
//    _p_global_obsolete_space = new Obsolete_Space(NULL);

    //
    // Initialize the lock.
    //
    InitializeCriticalSection(&_lock);

	//
	// Get information about the current system. (Page size, num processors, etc.)
	//
	_get_system_info();
	//
	// Initialize the Block Store parameters.
	//
    _sub_block_size_bytes    = sub_block_size_bytes;

#ifdef GC_NONE_V0
    _table_size_entries = 0;
#else

#endif

    // Allow a little bit of head room so that we can extend to deal with -mx N -ms N 
    // param arguments.
	unsigned long maximum_size_to_reserve = 
		final_heap_size_bytes + _p_gc_plan->heap_extension_increment_bytes();

#if 0 
    // Increase _initial_heap_size_bytes if needed
    if (initial_heap_size_bytes * 2 > final_heap_size_bytes) {
        initial_heap_size_bytes = final_heap_size_bytes / 2;
        initial_heap_size_bytes = initial_heap_size_bytes & (~(GC_BLOCK_SIZE_BYTES - 1));
        current_heap_size_bytes = initial_heap_size_bytes;
    }
#endif    
    //
	// Reserve enough contiguous space to be able to grow the heap.
	//
    //    orp_cout << " VirtualAlloc of " << maximum_size_to_reserve << endl;


#ifdef GC_NONE_V0
    // No Block store concept in GC_NONE_V0..just return!!!
	return;
#endif

#ifdef ORP_POSIX
    
    // DO RAW malloc() for now...
    _gc_free_list = malloc(maximum_size_to_reserve + 0xFFFF);
    
    // Save gc_free_list value so that it can be used to "free" the memory returned by malloc.
    _p_real_heap_base = _gc_free_list;
    
    // To make sure that _gc_free_list is 64KB aligned.
    _gc_free_list = (void *) (((POINTER_SIZE_INT)_gc_free_list + 0xFFFF) & (0xFFFF0000));
    
    
    if (_gc_free_list == NULL) { 
        
        DWORD error_code = GetLastError();
        orp_cout << "Error: Garbage Collector failed to reserve ";
        orp_cout << final_heap_size_bytes;
        orp_cout << " bytes of virtual address space. Error code = ";
        orp_cout << error_code << endl;
        assert(0);
        orp_exit(error_code);
    }
#else
    
    // This is needed but is not available from the winnt.h along with things like MEM_RESERVE
    // and MEM_COMMIT. I am guessing at the value...
#define MEM_WRITE_WATCH 0x200000
    
    
    if ((_gc_free_list = VirtualAlloc(NULL,
                maximum_size_to_reserve,
#ifdef GC_PT_WB // garbage collector page table write barrier
                // Let the OS handle card marking a page at a time
                MEM_WRITE_WATCH |
#endif  // GC_PT_WB                              
                MEM_RESERVE,
                PAGE_NOACCESS)) == NULL) {

        DWORD error_code = GetLastError();
        orp_cout << "Error: Garbage Collector failed to reserve ";
        orp_cout << final_heap_size_bytes;
        orp_cout << " bytes of virtual address space. Error code = ";
        orp_cout << error_code << endl;
        assert(0);
        orp_exit(error_code);
    }
#endif

    // malloc up the status table and initialize it as all free...
    // 
    int number_of_blocks = (maximum_size_to_reserve + GC_BLOCK_SIZE_BYTES - 1) / GC_BLOCK_SIZE_BYTES;
    int status_table_size = number_of_blocks * sizeof(gc_block_status);
    gc_block_status_table = (gc_block_status *)malloc(status_table_size);
    assert (gc_block_status_table);
    // set status_table[n] to free.
    memset(gc_block_status_table, 0, status_table_size);
    // orp_cout << "Doing a VirtualAlloc of " << maximum_size_to_reserve << " at " << _gc_free_list << endl;
	//
	// Populate the initial heap portion with real pages.
	//
	unsigned long initial_size_to_commit =
		initial_heap_size_bytes;


#ifdef ORP_POSIX 

    // No need to commit specifically for LINUX yet...

#else
    // Commit of NT (and W2K
	if ((_gc_free_list = VirtualAlloc(_gc_free_list,
		                              initial_size_to_commit,
                                      MEM_COMMIT,
								      PAGE_READWRITE)) == NULL) {
		DWORD error_code = GetLastError();
		orp_cout << "Error: GC failed to commit initial heap of size ";
		orp_cout << initial_heap_size_bytes << " bytes. Error code = " ;
		orp_cout << error_code << endl;
        assert(0);
		orp_exit(error_code);
	}
    
//    orp_cout << "Doing a VirtualAlloc commit of " << initial_size_to_commit << " at " << _gc_free_list << endl;
#endif

	if (stats_gc) {
		orp_cout << "Garbage collected heap size is " << initial_size_to_commit << " bytes" << endl;
	}

#ifdef ORP_NT

    //
    // Save it away for later use (ex: deallocation)
    //
    // This differs from _p_real_heap_base since it include the block tables and other meta data.
    //
    _p_real_heap_base = _gc_free_list;

#endif


    //
    // Save global pointers for use in JIT debugging.
    //
    p_gc_base    = _gc_free_list;
    void *_p_gc_limit   =  (void *)((char *)p_gc_base + initial_size_to_commit);
    p_gc_ceiling =  (void *)((char *)p_gc_base + maximum_size_to_reserve);
#if (GC_DEBUG>0)
	if (stats_gc) {
		orp_cout << "Start = " << p_gc_base << ", ceiling = " << _p_gc_limit << endl;
	}
#endif
	//
	// Now that we have allocated all the meta data, find out how big
	// the effective heap is.
	//
	_p_effective_heap_base = _gc_free_list;
  
#ifdef POINTER64
    assert((POINTER_SIZE_INT)((Byte *)_p_gc_limit - (Byte *)_gc_free_list) 
        <= (POINTER_SIZE_INT)0xFFFFFFFF);
#endif
    _effective_heap_size_bytes = (unsigned long)((Byte *)_p_gc_limit - (Byte *)_gc_free_list);

	//
	// Record the last block in the currently committed space.
	//
	_p_last_block =
		(void *)((((_effective_heap_size_bytes / _sub_block_size_bytes) - 1) * 
		          _sub_block_size_bytes) 
		         + (Byte *)_p_effective_heap_base);

    assert (_sub_block_size_bytes == (1 << GC_BLOCK_SHIFT_COUNT));
    //
	// Link all the free blocks into a free list.
	//

    _initialize_block_tables(_gc_free_list, _p_last_block);

    return;
}


Block_Store::~Block_Store()
{

    // NOTE NOTE NOTE: replace this with RTTI
	// Gc_Space *p_space;
    //
    // Clean up any obsolete containers still hanging around.
    //
#if 0 // FIX
    for (unsigned long idx = 0; idx < _table_size_entries; idx++) {

	    if (p_space = 
            dynamic_cast<Obsolete_Space*>(_p_block_table[idx])) {
		        delete p_space;
	    }

    }
#endif
    //
    // Release the memory we obtained for the GC heap, including
	// the ancilliary table(s).
    //

#ifdef ORP_POSIX
	
	free(_p_real_heap_base);
	
#else
    if (!VirtualFree(_p_real_heap_base, 0, MEM_RELEASE)) {
        cerr << "Error: failed to deallocate GC heap.";
        cerr << " error code = " << GetLastError() << endl;
        assert(0);
        orp_exit(GetLastError());
    }
#endif

//    orp_cout << "VirtualFree of " << _p_real_heap_base << endl;

}

//
// The Block Store is being asked to extend the heap by the specified
// number of bytes.
//
bool
Block_Store::_extend_heap(int block_size_bytes)
{
    heap_was_extended = true;
    int delta_bytes = _p_gc_plan->heap_extension_increment_bytes();
    if (delta_bytes < block_size_bytes) {
        delta_bytes = block_size_bytes;
    }

    if (verbose_gc) {
    	orp_cout << "Extending the heap by " << delta_bytes << " bytes" << endl;
    }
    int block_status_table_index_max = current_heap_size_bytes / GC_BLOCK_SIZE_BYTES;

#if 1
    static boolean already_extended_once_past_final_heap_size = false;
    // final_heap_size_bytes it is a bogus concept ....
	if (current_heap_size_bytes + delta_bytes >= final_heap_size_bytes) {
        if (already_extended_once_past_final_heap_size) {
    		orp_cout << "Error: attempt to grow the heap past the maximum size." << endl;
	    	orp_exit(1);
		    return false; // why exit? (caller may be able to recover.)  
        } else {
            already_extended_once_past_final_heap_size = true;
        }
	}
#endif

	//
	// Commit the specified number of bytes 
	//
	void *p_new_region = 
		(void *)((Byte *)_p_effective_heap_base + _effective_heap_size_bytes);
	
#ifdef ORP_NT
    if ((p_new_region = VirtualAlloc(p_new_region,
		                             delta_bytes,
                                     MEM_COMMIT,
								     PAGE_READWRITE)) == NULL) {
		DWORD error_code = GetLastError();
		orp_cout << "Error: GC failed to extend heap by ";
		orp_cout << initial_heap_size_bytes << " bytes. Error code = " ;
		orp_cout << error_code << endl;

		orp_exit(error_code);

		return false; // why exit? (caller may be able to recover.)
	}
#else
    // For Linux, malloc() is equivalent to MEM_RESERVE and MEM_COMMIT. 
    // So this step a NOP.

#endif // ORP_NT

    char *p_end_of_region = (char *)p_new_region + delta_bytes;
    //
	// Update the current heap size. BUG: NEED TO ROUND UP delta_bytes to
	// THE VirtualAlloc GRAIN OF ALLOCATION (FIRST FIGURE THIS OUT IN 
	// _get_system_info()).
	//
	current_heap_size_bytes   += delta_bytes;
	_effective_heap_size_bytes += delta_bytes;
	//
	// Record the last block in the currently committed space.
	//
	void * old_last_block = (void *)((((_effective_heap_size_bytes / GC_BLOCK_SIZE_BYTES) - 1) * 
		                      GC_BLOCK_SIZE_BYTES) 
		+ (Byte *)_p_effective_heap_base);
    _p_last_block = (char *)p_end_of_region - GC_BLOCK_SIZE_BYTES;
    int block_count = _initialize_and_count_blocks (p_new_region,  
                                                    _p_last_block);
    
    ((block_info *)p_new_region)->block_status_table_index = block_status_table_index_max + 1;
    assert(gc_block_status_table[((block_info *)p_new_region)->block_status_table_index] == block_in_free);
    link_free_blocks ((block_info *)p_new_region, block_count) ;
    gc_trace_block(p_new_region, " in link_free_block");
    block_group_link *this_group_list = block_group_list;
    block_group_link *new_group_link = (block_group_link *)malloc (sizeof(block_group_link));
    new_group_link->next = block_group_list;
    new_group_link->block_list = (block_info *)p_new_region;
    block_group_list = new_group_link;
                        
#ifdef GC_PT_WB
    UINT foo = dll_ResetWriteWatch (p_new_region, delta_bytes);
    if (foo != 0) {
        orp_cout << "Bad ResetWriteWatch." << endl;
        exit (99);
    }
#endif // GC_PT_WB

    return true;
}


//
// Card table logic needs to know where heap starts to calculate
// card.

Byte * 
Block_Store::get_heap_base() 
{
	return (Byte *)_p_effective_heap_base;
}

void 
Block_Store::_get_system_info()
{
	SYSTEM_INFO system_info;

	GetSystemInfo(&system_info);
	_machine_page_size_bytes     = system_info.dwPageSize;
	_number_of_active_processors = system_info.dwNumberOfProcessors;
}




// 
// Given a number of bytes size up to the next number of blocks.
//
unsigned int 
Block_Store::size_up_to_los_block_size (unsigned int number_of_bytes)
{
	if (((number_of_bytes / _p_gc_plan->los_block_size_bytes()) * _p_gc_plan->los_block_size_bytes()) ==
		number_of_bytes) {
		return number_of_bytes;
	} else {
		return ((number_of_bytes / _p_gc_plan->los_block_size_bytes()) + 1) * _p_gc_plan->los_block_size_bytes();
	}
}

// The number of buckets holding blocks. 2 ** bucket <= size < 2 ** bucket+1
// so bucket 0 holds the single block list.

void 
Block_Store::init_block (block_info *result, unsigned int size_in_blocks)
{
    // Clear the block info information 0, leave the rest of the block dirty...
    block_info *old_all_blocks_next = result->all_blocks_next;
    int old_block_status_table_index = result->block_status_table_index;
    assert (result->in_free_p);
    result->in_free_p = false;
    memset (result, 0x0, GC_BLOCK_INFO_SIZE_BYTES); 
    result->all_blocks_next = old_all_blocks_next;
    result->block_status_table_index = old_block_status_table_index;
    assert (gc_block_status_table[old_block_status_table_index] == block_in_free);
    result->c_area_p = false;
    result->free = GC_BLOCK_ALLOC_START(result);
    result->ceiling = (void *)((POINTER_SIZE_INT)result + (size_in_blocks * GC_BLOCK_SIZE_BYTES) - 1);
    result->scan = result->free;
    result->next = NULL;
    result->number_of_blocks = size_in_blocks;
}

// p_get_head_ts, ts means thread safe.
//  This routine takes the address of the head of a list and delinks
//  the first item on the list. It is a thread safe routine.
// If the head of the list is NULL return NULL;
block_info *
Block_Store::p_get_head_ts (block_info **head)
{
    block_info *result;
    block_info *next;
    while (true) {
        if ((result = *head) == (block_info *)NULL) {
             return NULL;
        }
        next = result->next;
        if (InterlockedCompareExchangePointer((PVOID *)head, (PVOID)next, (PVOID)result) == result) {
            return result; // We have a block of the correct size.
        }
    }
    assert (0); // Should never get here. either a result will be delinked or the list will be emptied.
                // and we will return from the routine.
    return NULL;
}

void
Block_Store::link_block_ts (block_info **head, block_info *block)
{
    gc_trace_block (block, "Linking this block onto the free list line 985.");
    block_info *old_head = *head;
    block->next = old_head;
    while ((InterlockedCompareExchangePointer((PVOID *)head, (PVOID)block, (PVOID)old_head) != old_head)) {
        block_info *old_head = *head;
        block->next = old_head;
    }
    return;
}

void
Block_Store::link_free_blocks (block_info *freed_block, unsigned int blocks)
{
    gc_trace_block (freed_block, "Linking this block onto the free list line 998.");
#ifdef _DEBUG
    memset (GC_BLOCK_ALLOC_START(freed_block), '*', GC_BLOCK_ALLOC_SIZE);
#endif //_DEBUG
    freed_block->number_of_blocks = blocks;
    unsigned int bucket_index = 0;
    unsigned int bucket_block_size = 1;
    block_info *old_all_blocks_next = freed_block->all_blocks_next;
    int old_block_status_table_index = freed_block->block_status_table_index;
    unsigned int old_number_of_blocks = freed_block->number_of_blocks;
    memset (freed_block, 0, GC_BLOCK_INFO_SIZE_BYTES);
    freed_block->all_blocks_next = old_all_blocks_next;
    freed_block->number_of_blocks = old_number_of_blocks;
    freed_block->in_free_p = true;
    freed_block->block_status_table_index = old_block_status_table_index;
    unsigned int i;
    for (i=freed_block->block_status_table_index; i < freed_block->block_status_table_index + freed_block->number_of_blocks; i++ ) {
        gc_block_status_table[i] = block_in_free;    
    }
    while (bucket_index < MAX_BUCKET) {
        if (blocks < bucket_block_size) {
#ifdef _DEBUG
            block_info *check_block = free_blocks[bucket_index - 1];
            while (check_block != NULL) {
                assert (check_block != freed_block);
                check_block = check_block->next;
            }
#endif // _DEBUG
            link_block_ts (&free_blocks[bucket_index - 1], freed_block);
            return;
        }
        bucket_index++;
        bucket_block_size = 2 * bucket_block_size; // double the size for the next bucket
    }
    link_block_ts (&free_blocks[MAX_BUCKET - 1], freed_block);
}

void clear_free_block_buckets ()
{  
    int bucket_index = 0;
    while (bucket_index < MAX_BUCKET) {
        p_global_bs->free_blocks[bucket_index] = NULL;
        bucket_index++;
    }
}

//
// This routine runs through all the blocks in the system. If it finds two adjacent blocks
// that are free then these two blocks will be combined into a single block.
//
// All blocks are members of a block group and this group is linked using the block_group_list field in block_info.
// All block groups are linked through block_groups_list.
//
// This algorithm in linear with respect to the number of blocks.
//

/*************
remove
*************/
void verify_all_blocks_list();


void Block_Store::coalesce_free_blocks()
{
    
    verify_all_blocks_list();
    // First empty out the free blocks structure, it will be rebuilt in this routine.
     clear_free_block_buckets();
    
    block_group_link *the_group_list = block_group_list;
    while (the_group_list) {
        block_info *the_blocks = the_group_list->block_list;
        while (the_blocks) {
            block_info *next_block = the_blocks->all_blocks_next;
            if (the_blocks->in_free_p) {
                // We have a free block, lets see if we can do some coalescing
                bool combine;
                if (next_block) {
                    combine = next_block->in_free_p;
                } else {
                    combine = false;
                }
                while (combine) {
                    // coalesce this block.
                    assert (next_block->in_free_p);
                    assert (!next_block->in_los_p);
                    assert (!next_block->in_nursery_p);
                    assert (!next_block->in_step_p);
                    assert (next_block->train_birthday == 0);
                    assert (next_block->car_birthday == 0);
                    the_blocks->number_of_blocks = the_blocks->number_of_blocks + next_block->number_of_blocks;
                    next_block = next_block->all_blocks_next;
                    the_blocks->all_blocks_next = next_block;
                    if (next_block) {
                        // loop and coalesce the next block also
                        combine = next_block->in_free_p;
                    } else {
                        combine = false;
                    }
                }
                p_global_bs->link_free_blocks (the_blocks, the_blocks->number_of_blocks);
            } // end if (the_blocks->in_free_p)
            the_blocks = next_block;
        }
        the_group_list = the_group_list->next;
        verify_all_blocks_list();
    }
}

bool 
Block_Store::is_block_available (unsigned int size)
{
    bool done = false;
    POINTER_SIZE_INT size_in_blocks = ( (((size + GC_BLOCK_INFO_SIZE_BYTES)) + (GC_BLOCK_SIZE_BYTES -1) ) / GC_BLOCK_SIZE_BYTES);
    unsigned int bucket_index = 0;
    unsigned int bucket_block_size = 1;
    assert ((size + GC_BLOCK_INFO_SIZE_BYTES) <= ((size_in_blocks * GC_BLOCK_SIZE_BYTES)));
    assert ((size + GC_BLOCK_INFO_SIZE_BYTES) > ((size_in_blocks - 1)*GC_BLOCK_SIZE_BYTES));
    bucket_index = 0;
    while (bucket_index < MAX_BUCKET) {
        if (size_in_blocks <= bucket_block_size) {
            if (free_blocks[bucket_index]) {
                return true;
            }
        }
        bucket_index++;
        bucket_block_size = 2 * bucket_block_size; // double the size for the next bucket
    }
    return false;
}

int total_blocks_available = 0;
int blocks_in_use = 0;

int get_multi_count = 0;
// This must be thread safe...
block_info *
Block_Store::p_get_multi_block(unsigned int size, bool extend, bool collect_returns_null, bool in_a_gc) 
{


    get_multi_count++;
    bool done = false;
    POINTER_SIZE_INT size_in_blocks = ( (((size + GC_BLOCK_INFO_SIZE_BYTES)) + (GC_BLOCK_SIZE_BYTES -1) ) / GC_BLOCK_SIZE_BYTES);
    block_info *result;
    unsigned int bucket_index = 0;
    unsigned int bucket_block_size = 1;
    assert ((size + GC_BLOCK_INFO_SIZE_BYTES) <= ((size_in_blocks * GC_BLOCK_SIZE_BYTES)));
    assert ((size + GC_BLOCK_INFO_SIZE_BYTES) > ((size_in_blocks - 1)*GC_BLOCK_SIZE_BYTES));
    if (size_in_blocks == 0) {
        assert (0);
    }
#ifdef _DEBUG
    while (bucket_index < MAX_BUCKET) {
        if (free_blocks[bucket_index]) {
            if (free_blocks[bucket_index]->number_of_blocks > 1000) {
                int foo = MAX_BUCKET;    
            }
        }
        bucket_index++;
    }
#endif

#if 0
    //
    // The problem is that we can allocate so many fixed objects that we run out of space
    // to collect the non-pinned objects.
    //
    if (!in_a_gc) {
        int total_number_of_blocks = get_total_block_count();
        int blocks_in_los = get_los_block_count();
        int free_blocks = get_free_non_nursery_block_count();
        int blocks_in_non_los_use = total_number_of_blocks - free_blocks - blocks_in_los;
        int free_blocks_needed_for_a_gc = blocks_in_non_los_use * 2;
        // We are not in a GC so we need to check if we should do a GC.
        if (free_blocks_needed_for_a_gc > free_blocks) {
            p_gc->reclaim_heap(size, false);
        }
    }
#endif
    bucket_index = 0;
    while (bucket_index < MAX_BUCKET) {
        if (size_in_blocks <= bucket_block_size) {
            result = p_get_head_ts (&free_blocks[bucket_index]);
            if (result) {
                
                int freed_block_status_index = result->block_status_table_index + size_in_blocks;
                unsigned int extra_blocks = result->number_of_blocks - size_in_blocks;
                assert (result->in_free_p);
                init_block (result, size_in_blocks); // Clears the info block (not the data area) and sets up a few common fields.
                
                //                orp_cout <<"-- Alloc of multi block at " << (void *)result << " of size " << (int) size_in_blocks << endl;
                if (extra_blocks) {
                    block_info *freed_block = (block_info *)((POINTER_SIZE_INT)result + (size_in_blocks * GC_BLOCK_SIZE_BYTES));
                    freed_block->block_status_table_index = freed_block_status_index;
                    gc_trace_block (freed_block, "Relinking this block after.");
                    link_free_blocks (freed_block, extra_blocks);
                    freed_block->block_status_table_index = freed_block_status_index;
                    assert (freed_block->in_free_p);
                    assert ((gc_block_status_table[freed_block_status_index]) == block_in_free);
                    // We split a block so update all_blocks_next list.
                    freed_block->all_blocks_next = result->all_blocks_next;                    
                    result->all_blocks_next = freed_block;
                }
                gc_trace_block (result, "In get_a_new_block allocating this block at line 1075");
//                result->in_free_p = false; // We are no longer free but the caller must handle setting this..
                blocks_in_use += size_in_blocks;
                return result;
            }
        }
        bucket_index++;
        bucket_block_size = 2 * bucket_block_size; // double the size for the next bucket
    }
    
    if (extend) {
        _extend_heap(0); // Extend the heap by some default value.
    } else {
        
        // Grab the GC lock and reclaim the heap.
        
        orp_enable_gc();
        
        p_gc_lock->_lock_enum_or_null(false);
        
        p_gc->reclaim_heap(size, false); // force = true
        
        p_gc_lock->_unlock_enum_or_null();
        
        orp_disable_gc();
        
        if (collect_returns_null) {
            return NULL;
        }
    }
    return p_get_multi_block(size, extend, collect_returns_null, in_a_gc);
}

// end file gc\block_store.cpp
