// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/local_reg_alloc.h,v 1.2 2001/08/13 09:54:43 xhshi Exp $
//


#include "../x86/x86.h"
#include "dlink.h"
//
// forward declarations
//
class Operand;
class Physical_Reg; 
class Reg_Operand;
class Inst;
class Call_Inst;

#ifndef _LOCAL_REG_ALLOC_H_
#define _LOCAL_REG_ALLOC_H_

#define MAX_LOCAL_REGS n_reg
#define NOT_YET_ASSIGNED n_reg
//
// Local_Reg_Manager keeps track of which local registers are used by
// which local live ranges.
// _avail_local_regs: if the corresponding bit of reg is set, then the reg is 
//     available for assignment.
// _pregs: Physical Reg of X86 registers
// _ready_be_assigned: those operands are put into local_reg_manager and waiting to 
//     get assigned regs
// _free_bv[i]: registers that _ready_be_assigned[i] can use.
// _free_entry: bit vector that indicates if there is a free entry in _ready_be_assigned
// _live_preg_lr: recording the live ranges of physical regs.  If the bit of a preg
//     is set, then the preg is live at this point.
// 
class Local_Reg_Manager {
public:
    Local_Reg_Manager(unsigned avail_regs) : _avail_local_regs(avail_regs) {
        int i;
		for (i = MAX_LOCAL_REGS; i != 0; i--)
			if (_avail_local_regs & (1<<i)) break;
		assert(i != 0);
		_max_local_regs = i+1;
        reset();
        _ever_used_pregs = 0;
    }
    void reset() {
        _live_preg_lr = NULL;
        _free_entry = (1<<_max_local_regs) - 1;
        _next_free = 0;
        unsigned i;
        for (i = 0; i < _max_local_regs; i++) {
            _free_bv[i] = NULL;
            _ready_be_assigned[i] = NULL;
            _preference[i] = n_reg;
        }
    }
    //
    // mark (X86_Reg_No)i is busy
    //
    void mark_preg_busy(unsigned i) {
        unsigned j;
        for (j = 0; j < _max_local_regs; j++)
            _free_bv[j] &= ~(1<<i);
        _ever_used_pregs |= (1<<i);
    }
    void starts_live_range(Operand *opnd);
    void ends_live_range(Operand *opnd);
    void call_live_range(Call_Inst *call);
    void set_preference(Reg_Operand *dst, Reg_Operand *src);
    unsigned ever_used_pregs() {return _ever_used_pregs;}
private:
    unsigned const _avail_local_regs;
	unsigned _max_local_regs;
    unsigned _live_preg_lr;
    unsigned _next_free;
    unsigned _free_entry;
    unsigned _ever_used_pregs; // pregs that have been used
    Physical_Reg *_pregs[MAX_LOCAL_REGS];
    Reg_Operand  *_ready_be_assigned[MAX_LOCAL_REGS];
    unsigned     _free_bv[MAX_LOCAL_REGS];
    X86_Reg_No   _preference[MAX_LOCAL_REGS];
};

class Fold_Link : public Dlink {
public:
    Fold_Link() : elem(NULL) {} // dummy header
    Fold_Link(Inst *i) : elem(i) {}
    Inst *const elem;
    Fold_Link *next() {return (Fold_Link*)_next;}
    Fold_Link *prev() {return (Fold_Link*)_prev;}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
};


//
// Dlink is for linking elements of hash table
// list_next is for linking elements of static_list and ary_fld_list
//
#define NUM_FOLD_PER_ENTRY 8
#define MAX_FOLD_SIZE 16
class Folding {
public:
    Folding(Mem_Manager& mm) :  _mem(mm) {reset();}
    void reset() {
        _ary_fld = NULL;
        int i;
        for (i = 0; i < MAX_FOLD_SIZE; i++) {
            _size[i] = 0;
            _table[i].unlink();
        }
    }
    void insert(Inst *i);
    void killed_by_ary_fld(Operand *opnd);
    void killed_by(Operand *opnd);
    Operand *lookup_replace_opnd(Operand *src);
    void replace_reg_opnd(Reg_Operand*&);
private:
    Mem_Manager& _mem;
    Fold_Link _table[MAX_FOLD_SIZE];
    unsigned  _size[MAX_FOLD_SIZE];  // _size[i] tells how many elements in _table[i]
    Inst *_ary_fld;  // array and field access
};

#endif // _LOCAL_REG_ALLOC_H_
