// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/cg_prepass.cpp,v 1.2 2001/08/13 09:54:55 xhshi Exp $
//

#include "defines.h"
#include <iostream.h>
#include <assert.h>
#include <string.h>  // for memset()
#include "cg_prepass.h"
#include "jit_common.h"

#define OFFSET2(array,idx) ((((char*)array)[idx] << 8) | (array)[(idx)+1])
#define OFFSET4(array,idx) (((array)[idx] << 24) | ((array)[(idx)+1] << 16) | ((array)[(idx)+2] << 8) | (array)[(idx)+3])

// In some cases, we have seen bytecode sequences like this:
//   ret 2
//   return
//
// Here, the "return" statement is dead code, determined statically.
// This kind of dead code causes problems for the flow graph, since
// the flow graph routines look for the last statement of the basic
// block to determine the out-edges.  The POTENTIAL_DEAD_CODE macro
// makes sure that the beginning of the dead code is marked as a new
// basic block, to avoid confusing the flow graph routines.
#define POTENTIAL_DEAD_CODE(i)                                    \
    do {                                                          \
    if ((i) < code_length) bytecode_info[(i)].is_block_entry = 1; \
    } while (0);


CG_Prepass::CG_Prepass(Compile_Handle comp_handle,
                       Class_Handle ch,
                       Method_Handle mh,
                       Mem_Manager& mem_manager,
                       const unsigned char *first,unsigned length,
                       unsigned maxStk, unsigned maxLocals):
first_bc(first), code_length(length),class_handle(ch),
_mm(mem_manager), num_jsr(0), result(JIT_SUCCESS), has_fp(false)
{
    
    bytecode_info =
        (Bytecode_Info*)mem_manager.alloc(code_length * sizeof(Bytecode_Info));
    
    // initialization
    memset(bytecode_info, '\0', code_length*sizeof(struct Bytecode_Info));
    
    // perform prepass (from first_bc and all exception handler)
    find_labels(comp_handle,first_bc,0);
    if (result != JIT_SUCCESS)
        return;
    unsigned cEH = method_get_num_handlers(mh);
    unsigned e;
    for(e=0; e<cEH; e++) {
        unsigned tryBegOffsPtr, tryEndOffsPtr, handlerOffsPtr, handlerTypePtr;
        method_get_handler_info(mh,e, &tryBegOffsPtr, &tryEndOffsPtr,
            &handlerOffsPtr, &handlerTypePtr);
        Loader_Exception exc;
        if (handlerTypePtr != 0)
            resolve_class(comp_handle, ch, handlerTypePtr, &exc);
        assert(tryBegOffsPtr < code_length);
        bytecode_info[tryBegOffsPtr].is_block_entry = 1;
        bytecode_info[tryEndOffsPtr].is_block_entry = 1;
        // pass 1 as stack_depth because stack contains only exception object
        find_labels(comp_handle,first_bc,handlerOffsPtr);
        if (result != JIT_SUCCESS)
            return;
    }
}

void CG_Prepass::find_labels(Compile_Handle comp_handle,
                             const unsigned char *bc,
                             unsigned idx)
{
    assert(idx < code_length);
    Bytecode_Info *byte_info = &bytecode_info[idx];
    //
    // is a label as well as block entry
    // 
    byte_info->is_block_entry = 1;
    //
    // check if the current bb has been visited
    //
    if (byte_info->is_visited)
        return;
    
    int offset;             // for branches: offset of branch
    unsigned index; // index of constant pools or variables
    Loader_Exception exc;
    Java_Type ty;
    
    //
    // traverse code within the current bb
    //
    while (!byte_info->is_visited && idx < code_length) {
        byte_info->is_visited = 1;
        switch (bc[idx])
        {
        case 0x99: case 0x9a:   // if{eq,ne,lt,ge,gt,le}
        case 0x9b: case 0x9c:
        case 0x9d: case 0x9e:
            offset = OFFSET2(bc,idx+1);
            find_labels(comp_handle,bc, idx+offset);      // target bb
            assert(idx+3 < code_length);
            bytecode_info[idx+3].is_block_entry = 1;
            break;
        case 0x9f: case 0xa0:   // if_icmp{eq,ne,lt,ge,gt,le}
        case 0xa1: case 0xa2:
        case 0xa3: case 0xa4:
        case 0xa5: case 0xa6:   // if_acmp{eq,ne}
            offset = OFFSET2(bc,idx+1);
            find_labels(comp_handle,bc, idx+offset);      // target bb
            assert(idx+3 < code_length);
            bytecode_info[idx+3].is_block_entry = 1;
            break;
        case 0xa7:      // goto
            offset = OFFSET2(bc,idx+1);
            find_labels(comp_handle,bc, idx+offset);              // target bb
            POTENTIAL_DEAD_CODE(idx+3);
            return; 
            break;
        case 0xa8:      // jsr
            num_jsr ++;
            offset = OFFSET2(bc,idx+1);
            //
            // stack: ... ==> ...,address
            // the address of the inst I immediately following this jsr is pushed.
            // However, the callee pops the address and saves it in a local
            // variable.  At the time when the routine returns, the stack_depth
            // is the same as that prior to pushing the address.
            //        ^^^^ WRONG!!!!!
            //
            // search target (no edge to target)
            find_labels(comp_handle,bc, idx+offset);
            POTENTIAL_DEAD_CODE(idx+3);
            // We go ahead and continue from the successor instruction, even though
            // it might actually be dead code that is not the target of any "ret".
            break;
        case 0xa9: // ret
            POTENTIAL_DEAD_CODE(idx+2);
            return; // it is the end of bb
            break;
        case 0xaa: // tableswitch
            {
                unsigned new_idx = ((idx + 4) & (~3));
                int default_offset = OFFSET4(bc,new_idx);
                int low = OFFSET4(bc,new_idx+4);
                int high = OFFSET4(bc,new_idx+8);
                new_idx += 12;
                int n_entries = high - low + 1;
                find_labels(comp_handle,bc, idx+default_offset);
                int i;
                for (i = 0; i < n_entries; i++)
                {
                    offset = OFFSET4(bc,new_idx);
                    new_idx += 4;
                    find_labels(comp_handle,bc, idx+offset);
                }
                POTENTIAL_DEAD_CODE(new_idx);
                return; // it is the end of bb
            }
            break;
        case 0xab:      // lookupswitch (key match and jump)
            {
                unsigned new_idx = ((idx + 4) & (~3));
                int default_offset = OFFSET4(bc,new_idx);
                int npairs = OFFSET4(bc,new_idx+4);
                new_idx += 8;
                int i;
                for (i = 0; i < npairs; i++)
                {
                    int offset = OFFSET4(bc,new_idx+4);
                    find_labels(comp_handle,bc, idx+offset);
                    new_idx += 8;
                }
                find_labels(comp_handle,bc, idx+default_offset);
                POTENTIAL_DEAD_CODE(new_idx);
                return; // it is the end of bb
            }
            break;
        case 0xae:      // freturn
        case 0xaf:      // dreturn
            has_fp = true;
            // no break
        case 0xac:      // ireturn
        case 0xb0:      // areturn
        case 0xad:      // lreturn
        case 0xb1:      // return
            POTENTIAL_DEAD_CODE(idx+1);
            return; // it is the end of bb
            break;
        case 0xb2: // getstatic
            {
                index = OFFSET2(bc,idx+1);
                Field_Handle fh = resolve_static_field(comp_handle,class_handle,index,&exc);
                if (fh == NULL)
                {
                    // resolution error
                    result = JIT_RESOLVE_ERROR;
                    break;
                }
                ty = field_get_type(fh);
                if (ty == JAVA_TYPE_DOUBLE || ty == JAVA_TYPE_FLOAT)
                    has_fp = true;
            }
            break;
        case 0xb3: // putstatic
            {
                index = OFFSET2(bc,idx+1);
                Field_Handle fh = resolve_static_field(comp_handle,class_handle,index,&exc);
                if (fh == NULL)
                {
                    // resolution error
                    result = JIT_RESOLVE_ERROR;
                    break;
                }
                ty = field_get_type(fh);
                if (ty == JAVA_TYPE_DOUBLE || ty == JAVA_TYPE_FLOAT)
                    has_fp = true;
            }
            break;
        case 0xb4:      // getfield
            {
                index = OFFSET2(bc,idx+1);
                Field_Handle fh = resolve_nonstatic_field(comp_handle,class_handle,index,&exc);
                if (fh == NULL)
                {
                    // resolution error
                    result = JIT_RESOLVE_ERROR;
                    break;
                }
                ty = field_get_type(fh);
                if (ty == JAVA_TYPE_DOUBLE || ty == JAVA_TYPE_FLOAT)
                    has_fp = true;
            }
            break;
        case 0xb5:      // putfield
            {
                index = OFFSET2(bc,idx+1);
                Field_Handle fh = resolve_nonstatic_field(comp_handle,class_handle,index,&exc);
                if (fh == NULL)
                {
                    // resolution error
                    result = JIT_RESOLVE_ERROR;
                    break;
                }
                ty = field_get_type(fh);
                if (ty == JAVA_TYPE_DOUBLE || ty == JAVA_TYPE_FLOAT)
                    has_fp = true;
            }
            break;
        case 0xb6: // invokevirtual
            {
                index = OFFSET2(bc,idx+1);
                Method_Handle mh = resolve_virtual_method(comp_handle,class_handle,index,&exc);
                if (mh == NULL)
                {
                    // resolution error
                    result = JIT_RESOLVE_ERROR;
                    break;
                }
                ty = method_get_return_type(mh);
                if (ty == JAVA_TYPE_DOUBLE || ty == JAVA_TYPE_FLOAT)
                    has_fp = true;
            }
            break;
        case 0xb7: // invokespecial
            {
                index = OFFSET2(bc,idx+1);
                Method_Handle mh = resolve_special_method(comp_handle,class_handle,index,&exc);
                if (mh == NULL)
                {
                    // resolution error
                    result = JIT_RESOLVE_ERROR;
                    break;
                }
                ty = method_get_return_type(mh);
                if (ty == JAVA_TYPE_DOUBLE || ty == JAVA_TYPE_FLOAT)
                    has_fp = true;
            }
            break;
        case 0xb8:      // invokestatic
            {
                index = OFFSET2(bc,idx+1);
                // invokestatic does not have objectref
                Method_Handle mh = resolve_static_method(comp_handle,class_handle,index,&exc);
                if (mh == NULL)
                {
                    // resolution error
                    result = JIT_RESOLVE_ERROR;
                    break;
                }
                ty = method_get_return_type(mh);
                if (ty == JAVA_TYPE_DOUBLE || ty == JAVA_TYPE_FLOAT)
                    has_fp = true;
            }
            break;
        case 0xb9:      // invokeinterface
            {
                index = OFFSET2(bc,idx+1);
                Method_Handle mh = resolve_interface_method(comp_handle,class_handle,index,&exc);
                if (mh == NULL)
                {
                    // resolution error
                    result = JIT_RESOLVE_ERROR;
                    break;
                }
                ty = method_get_return_type(mh);
                if (ty == JAVA_TYPE_DOUBLE || ty == JAVA_TYPE_FLOAT)
                    has_fp = true;
            }
            break;
        case 0xbb:      // new
            index = OFFSET2(bc,idx+1);
            // Pre-resolve for code generation
            resolve_class(comp_handle,class_handle, index,&exc);
            break;
        case 0xbd:      // anewarray
            index = OFFSET2(bc,idx+1);
            // Pre-resolve for code generation
            resolve_class(comp_handle,class_handle, index,&exc);
            break;
        case 0xbf:      // athrow
            POTENTIAL_DEAD_CODE(idx+1);
            return;
            break;
        case 0xc0:      // checkcast
        case 0xc1:      // instanceof
            index = OFFSET2(bc,idx+1);
            // Pre-resolve for code generation
            resolve_class(comp_handle,class_handle, index,&exc);
            break;
        case 0xc4:      // wide
            switch (bc[idx+1])
            {
            case 0x17: // fload
            case 0x18: // dload
            case 0x38: // fstore
            case 0x39: // dstore
                has_fp = true;
                break;
            case 0xa9: // ret
                POTENTIAL_DEAD_CODE(idx+4);
                return; // it is the end of bb
                break;
            default:
                break;
            }
            break;
        case 0xc5:      // multianewarray
            index = OFFSET2(bc,idx+1);
            // Pre-resolve for code generation
            resolve_class(comp_handle,class_handle, index,&exc);
            break;
        case 0xc6:      // ifnull
        case 0xc7:      // ifnonnull
            offset = OFFSET2(bc,idx+1);
            find_labels(comp_handle,bc, idx+offset);      // target bb
            assert(idx+3 < code_length);
            bytecode_info[idx+3].is_block_entry = 1;
            break;
        case 0xc8:      // goto_w
            offset = OFFSET4(bc,idx+1);
            find_labels(comp_handle,bc, idx+offset);
            POTENTIAL_DEAD_CODE(idx+5);
            return;  // it is the end of bb
            break;
        case 0xc9:      // jsr_w
            // See jsr.
            num_jsr ++;
            offset = OFFSET4(bc,idx+1);
            find_labels(comp_handle,bc, idx+offset);
            POTENTIAL_DEAD_CODE(idx+3);
            break;
        case 0x12: // ldc
            ty = class_get_const_type(class_handle, bc[idx+1]);
            if (ty == JAVA_TYPE_FLOAT || ty == JAVA_TYPE_DOUBLE)
                has_fp = true;
            break;
        case 0x13: // ldc_w
        case 0x14: // ldc2_w
            ty = class_get_const_type(class_handle, (bc[idx+1] << 8) + bc[idx+2]);
            if (ty == JAVA_TYPE_FLOAT || ty == JAVA_TYPE_DOUBLE)
                has_fp = true;
            break;
        case 0x0b: // fconst_0
        case 0x0c: // fconst_1
        case 0x0d: // fconst_2
        case 0x0e: // dconst_0
        case 0x0f: // dconst_1
        case 0x17: // fload
        case 0x18: // dload
        case 0x22: // fload_0
        case 0x23: // fload_1
        case 0x24: // fload_2
        case 0x25: // fload_3
        case 0x26: // dload_0
        case 0x27: // dload_1
        case 0x28: // dload_2
        case 0x29: // dload_3
        case 0x30: // faload
        case 0x31: // daload
        case 0x38: // fstore
        case 0x39: // dstore
        case 0x43: // fstore_0
        case 0x44: // fstore_1
        case 0x45: // fstore_2
        case 0x46: // fstore_3
        case 0x47: // dstore_0
        case 0x48: // dstore_1
        case 0x49: // dstore_2
        case 0x4a: // dstore_3
        case 0x51: // fastore
        case 0x52: // dastore
        case 0x62: // fadd
        case 0x63: // dadd
        case 0x66: // fsub
        case 0x67: // dsub
        case 0x6a: // fmul
        case 0x6b: // dmul
        case 0x6e: // fdiv
        case 0x6f: // ddiv
        case 0x72: // frem
        case 0x73: // drem
        case 0x76: // fneg
        case 0x77: // dneg
        case 0x86: // i2f
        case 0x87: // i2d
        case 0x89: // l2f
        case 0x8a: // l2d
        case 0x8b: // f2i
        case 0x8c: // f2l
        case 0x8d: // f2d
        case 0x8e: // d2i
        case 0x8f: // d2l
        case 0x90: // d2f
            has_fp = true;
            break;
        case 0x94: // lcmp
        case 0x95: // fcmpl
        case 0x96: // fcmpg
        case 0x97: // dcmpl
        case 0x98: // dcmpg
            //
            // fall back to O1 if next_bc is not in ifeq, ..., ifle
            //
            {
                unsigned next_bc = bc[idx + instruction_length(bc, idx)];
                if (next_bc < 0x99 /* ifeq */ || next_bc > 0x9e /* ifle */)
                    result = JIT_FAILURE;
                if (bc[idx] != 0x94) // floating comparison
                    has_fp = true;
            }
            break;
        default:
            break;
    } // switch
    if (result != JIT_SUCCESS)
        return;
    
    idx += instruction_length(bc, idx);
    // no assertion that idx<code_length
    byte_info = &bytecode_info[idx];
  }
  if (idx < code_length && result == JIT_SUCCESS)
  {
      // we have exited the while loop (always guaranteed to enter
      // it if it is reached) and we are not at the end of the
      // bytecodes, so we must have run into code that has already
      // been visited. if so, then it must have been jumped to,
      // which means it is a block entry.
      assert(byte_info->is_block_entry == 1);
  }
}
