/*
 * Package org.OpenJUMP.layer.style.classification pour JUMP
 *
 * Copyright (C) 2004
 * Olivier Bedel, ingnieur informaticien laboratoire Reso UMR ESO 6590, Bassin versant du Jaudy-Guindy-Bizien.
 * Cline Foureau, stagiaire MIAGE, laboratoire Reso UMR ESO 6590.
 * Erwan Bocher, doctorant en gographie, laboratoire Reso UMR ESO 6590, Bassin versant du Jaudy-Guindy-Bizien
 *
 * Date de cration : 19 aout 2004
 *
 * Dvelopp dans le cadre du Projet APARAD 
 *  (Laboratoire Reso UMR ESO 6590 CNRS / Bassin Versant du Jaudy-Guindy-Bizien)
 *    Responsable : Erwan BOCHER
 *    Dveloppeurs : Cline FOUREAU, Olivier BEDEL
 *
 * olivier.bedel@uhb.fr ou olivier.bedel@yahoo.fr
 * erwan.bocher@uhb.fr ou erwan.bocher@free.fr
 * celine.foureau@uhb.fr ou celine.foureau@wanadoo.fr
 * 
 * Ce package hrite de la licence GPL de JUMP. Il est rgi par la licence CeCILL soumise au droit franais et
 * respectant les principes de diffusion des logiciels libres. (http://www.cecill.info)
 * 
 */

package org.openjump.core.ui.style.classification;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import com.vividsolutions.jts.util.Assert;
import com.vividsolutions.jump.I18N;
import com.vividsolutions.jump.util.CollectionUtil;
import com.vividsolutions.jump.workbench.ui.renderer.style.ColorScheme;
import com.vividsolutions.jump.workbench.ui.renderer.style.ColorThemingStylePanel;
import com.vividsolutions.jump.workbench.ui.renderer.style.ColorThemingTableModel;

/**
 * @author FOUREAU_C
 */

public class QuantileColorThemingState implements ColorThemingStylePanel.State {

    private ColorThemingStylePanel stylePanel;

    private DefaultComboBoxModel comboBoxModel = new DefaultComboBoxModel();

    private JComboBox comboBox = new JComboBox(comboBoxModel);

    private static final String QUANTILE_COUNT_KEY = QuantileColorThemingState.class
            .getName()
            + " - QUANTILE COUNT";

    private JLabel label = new JLabel(
            I18N
                    .get("org.OpenJUMP.layer.style.classification.QuantileColorThemingState.range-count"));

    private JButton reverseButton = new JButton(
            I18N
                    .get("org.OpenJUMP.layer.style.classification.QuantileColorThemingState.reverse-colors"));

    private JPanel panel = new JPanel(new GridBagLayout()) {
        public void setEnabled(boolean enabled) {
            comboBox.setEnabled(enabled);
            label.setEnabled(enabled);
            reverseButton.setEnabled(enabled);
            super.setEnabled(enabled);
        }
    };

    private boolean reversingColorScheme = false;

    public QuantileColorThemingState(final ColorThemingStylePanel stylePanel) {
        this.stylePanel = stylePanel;
        addComboBoxItems();
        comboBox.setSelectedItem(stylePanel.getLayer().getLayerManager()
                .getBlackboard().get(QUANTILE_COUNT_KEY, new Integer(5)));
        //Don't add action listeners until items have been added to the
        //combo box. [Jon Aquino]
        comboBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                stylePanel.populateTable();
                stylePanel.getLayer().getLayerManager().getBlackboard().put(
                        QUANTILE_COUNT_KEY, comboBox.getSelectedItem());
            }
        });
        panel.add(label, new GridBagConstraints(1, 0, 1, 1, 0, 0,
                GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(2,
                        2, 2, 2), 0, 0));
        panel.add(comboBox, new GridBagConstraints(2, 0, 1, 1, 0, 0,
                GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(2,
                        2, 2, 2), 0, 0));
        panel.add(reverseButton, new GridBagConstraints(3, 0, 1, 1, 0, 0,
                GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(2,
                        2, 2, 2), 0, 0));
        reverseButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                reversingColorScheme = !reversingColorScheme;
                stylePanel.applyColorScheme();
            }
        });
    }

    public String getAllOtherValuesDescription() {
        return I18N
                .get("org.OpenJUMP.layer.style.classification.QuantileColorThemingState.OtherValues");
    }

    public ColorScheme filterColorScheme(ColorScheme colorScheme) {
        if (!reversingColorScheme) {
            return colorScheme;
        }
        List colors = new ArrayList(colorScheme.getColors());
        Collections.reverse(colors);
        return new ColorScheme(colorScheme.getName(), colors);
    }

    public void activate() {
        stylePanel.tableModel().addTableModelListener(tableModelListener);
    }

    public void deactivate() {
        stylePanel.tableModel().removeTableModelListener(tableModelListener);
    }

    public Collection getColorSchemeNames() {
        return ColorScheme.rangeColorSchemeNames();
    }

    public void applyColorScheme(ColorScheme scheme) {
        stylePanel.tableModel().apply(
                new ColorScheme(null, CollectionUtil.stretch(
                        scheme.getColors(), new ArrayList(), stylePanel
                                .tableModel().getRowCount())), false);
    }

    public Collection filterAttributeValues(SortedSet attributeValues) {
        return null;
    }

    public Collection filterAttributeValues(List attributeValues) {
        List filterAttributeValues = new ArrayList();
        ComputeQuantile quantile = new ComputeQuantile(attributeValues,
                getNbClasses());
        filterAttributeValues = quantile.getClasses();
        return filterAttributeValues;
    }

    public String getAttributeValueColumnTitle() {
        return I18N
                .get("org.OpenJUMP.layer.style.classification.QuantileColorThemingState.minimum-attribute-values");
    }

    public JComponent getPanel() {
        return panel;
    }

    public Map toExternalFormat(Map attributeValueToBasicStyleMap) {
        if (attributeValueToBasicStyleMap.isEmpty()) {
            return attributeValueToBasicStyleMap;
        }
        //Turn the values into ranges. Validations have already ensured that
        //the values are unique and contain no nulls. [Jon Aquino]
        Assert.isTrue(attributeValueToBasicStyleMap instanceof SortedMap);
        TreeMap newMap = new Quantile.ClassificationQuantileTreeMap();
        Object previousValue = null;
        for (Iterator i = attributeValueToBasicStyleMap.keySet().iterator(); i
                .hasNext();) {
            Object value = i.next();
            try {
                if (previousValue == null) {
                    //Let the default style handle values from negative
                    // infinity to
                    //the first value. [Jon Aquino]
                    continue;
                }
                //Make one side inclusive and the other exclusive to ensure no
                //overlaps. [Jon Aquino]
                newMap.put(new Quantile(previousValue, true, value, false),
                        attributeValueToBasicStyleMap.get(previousValue));
            } finally {
                previousValue = value;
            }
        }
        newMap.put(new Quantile(previousValue, true,
                new Quantile.PositiveInfinity(), true),
                attributeValueToBasicStyleMap.get(previousValue));

        /*
         * for (Iterator i = attributeValueToBasicStyleMap.keySet().iterator();
         * i .hasNext();) { Interval value = (Interval) i.next(); newMap.put(new
         * Quantile(value.getMin(), value.isIncludingMin(), value.getMax(),
         * value.isIncludingMax()), attributeValueToBasicStyleMap.get(value)); }
         */
        return newMap;
    }

    public Map fromExternalFormat(Map attributeValueToBasicStyleMap) {
        TreeMap newMap = new TreeMap();
        for (Iterator i = attributeValueToBasicStyleMap.keySet().iterator(); i
                .hasNext();) {
            Quantile quantile = (Quantile) i.next();
            newMap.put(quantile.getMin(), attributeValueToBasicStyleMap
                    .get(quantile));
        }
        return newMap;
    }

    private void addComboBoxItems() {
        int maxColorSchemeSize = -1;
        for (Iterator i = ColorScheme.rangeColorSchemeNames().iterator(); i
                .hasNext();) {
            String rangeColorSchemeName = (String) i.next();
            maxColorSchemeSize = Math.max(maxColorSchemeSize, ColorScheme
                    .create(rangeColorSchemeName).getColors().size());
        }
        for (int i = 3; i <= maxColorSchemeSize; i++) {
            comboBoxModel.addElement(new Integer(i));
        }
    }

    private TableModelListener tableModelListener = new TableModelListener() {
        public void tableChanged(TableModelEvent e) {
            if (e instanceof ColorThemingTableModel.AttributeValueTableModelEvent) {
                Object attributeValue = stylePanel.tableModel().getValueAt(
                        e.getFirstRow(),
                        ColorThemingTableModel.ATTRIBUTE_COLUMN);
                stylePanel.tableModel().sort(
                        stylePanel.tableModel().wasLastSortAscending());
                //I'd like to scroll to the row at this point, but the user
                // probably
                //finished the edit by clicking on another cell, so even if I
                // scroll
                //to the row, it scrolls back to where the user clicked. [Jon
                // Aquino]
            }
        }
    };

    private int getNbClasses() {
        return ((Integer) comboBox.getSelectedItem()).intValue();
    }
}