/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


//---------------------------------------------------------------------------

#ifndef PLATFORM_UNIX
#include <vcl.h>
#pragma hdrstop
#include <io.h>
#endif
#include <math.h>
#include <fcntl.h>

#include "map.h"
#include "art.h"
#include "console.h"
#include "Render.h"
#include "FixPoly.h"
#ifndef PLATFORM_UNIX
#include "SoftRender.h"
#else
#include "SDLRender.h"
#include "linux_inc.h"
#endif

#define ReadWord(x) (*((short *)&(x)))
#define ReadLong(x) (*((long *)&(x)))

//---------------------------------------------------------------------------

float Map::CalcFloorZ(int i,int wallIndex)
{
    return CalcFloorZAtPt(i,wall[wallIndex].x,wall[wallIndex].y);
}

float Map::CalcCeilingZ(int i,int wallIndex)
{
    return CalcCeilingZAtPt(i,wall[wallIndex].x,wall[wallIndex].y);
}

float Map::CalcFloorZAtPt(int i,float x,float y)
{
    // Find Z coordinate of floor at a point
    float vectx,vecty,vectd2,vectd;
    vectx=wall[wall[sector[i].wallptr].point2].x-wall[sector[i].wallptr].x;
    vecty=wall[wall[sector[i].wallptr].point2].y-wall[sector[i].wallptr].y;
    vectd2=vectx*vectx+vecty*vecty;
    vectd=sqrt(vectd2);
    float xd,yd;
    xd=x-wall[sector[i].wallptr].x;
    yd=y-wall[sector[i].wallptr].y;
    float zvectd,z;
    zvectd=(-xd*vecty+yd*vectx)/vectd;
    z=sector[i].floorz+zvectd*sector[i].floorheinum/256.0;
    return z;
}

float Map::CalcCeilingZAtPt(int i,float x,float y)
{
    // Find Z coordinate of ceiling at a point
    float vectx,vecty,vectd2,vectd;
    vectx=wall[wall[sector[i].wallptr].point2].x-wall[sector[i].wallptr].x;
    vecty=wall[wall[sector[i].wallptr].point2].y-wall[sector[i].wallptr].y;
    vectd2=vectx*vectx+vecty*vecty;
    vectd=sqrt(vectd2);
    float xd,yd;
    xd=x-wall[sector[i].wallptr].x;
    yd=y-wall[sector[i].wallptr].y;
    float zvectd,z;
    zvectd=(-xd*vecty+yd*vectx)/vectd;
    z=sector[i].ceilingz+zvectd*sector[i].ceilingheinum/256.0;
    return z;
}

Map::Map(GroupFile *_group,char *name,Art *_art): group(_group), art(_art)
{
    hideSpecialSprites=true;
    // Read the map file into the sector, wall, and
    // sprite arrays
    char *buf;
    if (group->FileSize(name)==-1)
        throw "Map does not exist\n";
    buf=new char[group->FileSize(name)];
    group->LoadFile(name,buf);
    if (ReadLong(buf[0])!=7)
        throw "Incorrect map version\n";
    startX=ReadLong(buf[4]);
    startY=ReadLong(buf[8]);
    startZ=ReadLong(buf[12]);
    startAng=ReadWord(buf[16]);
    startSect=ReadWord(buf[18]);
    numSectors=ReadWord(buf[20]);
    int index=22;
    sector=new SectorType[numSectors];
    sectorDrawn=new int[numSectors];
    memcpy(sector,&buf[index],sizeof(SectorType)*numSectors);
    index+=sizeof(SectorType)*numSectors;
    numWalls=ReadWord(buf[index]);
    index+=2;
    wall=new WallType[numWalls];
    memcpy(wall,&buf[index],sizeof(WallType)*numWalls);
    index+=sizeof(WallType)*numWalls;
    numSprites=ReadWord(buf[index]);
    index+=2;
    sprite=new SpriteType[numSprites];
    spriteDrawn=new int[numSprites];
    memcpy(sprite,&buf[index],sizeof(SpriteType)*numSprites);
    delete[] buf;
    // Convert sectors into lists of polygons
    sectorPolys=new PolyList[numSectors];
    sectorWindowPolys=new PolyList[numSectors];
    // Set floor/ceiling slopes to zero if the "sloped"
    // flag is not set for the sector
    for (int i=0;i<numSectors;i++)
    {
        if (!(sector[i].floorstat&2))
            sector[i].floorheinum=0;
        if (!(sector[i].ceilingstat&2))
            sector[i].ceilingheinum=0;
    }
    for (int i=0;i<numSectors;i++)
        ConvertSectorToPolyList(i);
    // Convert sprites into lists of polygons
    spritePolys=new PolyList[numSprites];
    for (int i=0;i<numSprites;i++)
        ConvertSpriteToPolyList(i);
    cprintf("Loaded map %s\n",name);
}

Map::~Map()
{
    delete[] sector;
    delete[] wall;
    delete[] sprite;
}

int Map::IsInSector(int sectNum,int x,int y,int z)
{
    // Return false if the sector number is invalid
    if (sectNum==-1) return 0;
    // Make an array of the wall indicies for all
    // of the sector's vertices
    static int pt[1024];
    int n=0;
    int wallIndex=sector[sectNum].wallptr;
    while (1)
    {
        pt[n++]=wallIndex;
        wallIndex=wall[wallIndex].point2;
        if (wallIndex==sector[sectNum].wallptr)
            break;
    }
    // Determine if the passed coordinate is within
    // this polygon
    int c=0;
    for (int i=0,j=n-1;i<n;j=i++)
    {
        if ((((wall[pt[i]].y<=y)&&(y<wall[pt[j]].y))||
            ((wall[pt[j]].y<=y)&&(y<wall[pt[i]].y)))&&
            (x<(wall[pt[j]].x-wall[pt[i]].x)*
            (y-wall[pt[i]].y)/(wall[pt[j]].y-
            wall[pt[i]].y)+wall[pt[i]].x))
            c=!c;
    }
    if (!c) return 0;
    // Check to see if the height of the coordinate
    // passed is between the floor and the ceiling
    if ((z<=CalcFloorZAtPt(sectNum,x,y))&&(z>=CalcCeilingZAtPt(sectNum,x,y)))
        return 1;
    return 0;
}

void Map::AddWallPoly(int i,int wallIndex)
{
    // Convert the shade value for this wall into
    // a D3D color value
    int shade=wall[wallIndex].shade*4;
    if (shade<-128) shade=-128;
    if (shade>127) shade=127;
    unsigned char light=~(shade+0x80);
    int color=0xff000000|(light<<16)|(light<<8)|light;
    // Make the polygon translucent if the wall flags
    // request it
    if ((wall[wallIndex].nextsector!=-1)&&(wall[wallIndex].cstat&0x80))
        color&=0x80ffffff;
    // Only create the wall polygon if it is not a "wall"
    // that divides sectors or if it is a window
    if ((wall[wallIndex].nextsector==-1)||(wall[wallIndex].cstat&0x20)||(wall[wallIndex].cstat&0x10))
    {
        float tu1=0,tv1=0,tu2=0,tv2=0;
        float tv3=0,tv4=0;
        // Get texture number for the wall
        int texnum=wall[wallIndex].picnum;
        if ((wall[wallIndex].nextsector!=-1)&&(wall[wallIndex].cstat&0x10))
            texnum=wall[wallIndex].overpicnum;
        // Calculate height values for the wall
        float floorz1=CalcFloorZ(i,wallIndex);
        float ceilingz1=CalcCeilingZ(i,wallIndex);
        float floorz2=CalcFloorZ(i,wall[wallIndex].point2);
        float ceilingz2=CalcCeilingZ(i,wall[wallIndex].point2);
        if (wall[wallIndex].nextsector!=-1)
        {
            // The height values are obtained from the
            // sector on the other side of the wall if
            // it is a window
            if ((wall[wallIndex].x==wall[wall[wallIndex].nextwall].x)&&
                (wall[wallIndex].y==wall[wall[wallIndex].nextwall].y))
            {
                floorz1=CalcFloorZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall);
                ceilingz1=CalcCeilingZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall);
                floorz2=CalcFloorZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2);
                ceilingz2=CalcCeilingZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2);
            }
            else
            {
                floorz1=CalcFloorZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2);
                ceilingz1=CalcCeilingZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2);
                floorz2=CalcFloorZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall);
                ceilingz2=CalcCeilingZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall);
            }
        }
        art->RequestTexture(texnum,0);
        if (art->tex[0][texnum])
        {
            // Calculate the texture coordaintes of each
            // wall vertex
            tu1=((float)wall[wallIndex].xpanning)/art->sizeX[texnum];
            tu2=((float)wall[wallIndex].xpanning+8.0*wall[wallIndex].xrepeat)/art->sizeX[texnum];
            if (wall[wallIndex].cstat&4)
            {
                if (wall[wallIndex].nextsector!=-1)
                {
                    tv1=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((ceilingz1-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                    tv2=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((floorz1-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                    tv3=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((ceilingz2-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                    tv4=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((floorz2-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                }
                else
                {
                    tv1=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((ceilingz1-sector[i].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                    tv2=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((floorz1-sector[i].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                    tv3=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((ceilingz2-sector[i].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                    tv4=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((floorz2-sector[i].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                }
            }
            else
            {
                tv1=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((ceilingz1-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                tv2=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((floorz1-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                tv3=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((ceilingz2-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
                tv4=(wall[wallIndex].ypanning*(art->sizeY[texnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((floorz2-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[texnum];
            }
            if (wall[wallIndex].cstat&8)
            {
                // Handle the swap x/y flag
                int temp=tu1;
                tu1=tu2;
                tu2=temp;
            }
        }
        // Generate the polygon
        Poly p;
        p.AddVert(wall[wallIndex].x/64.0,-floorz1/1024.0,
            -wall[wallIndex].y/64.0,tu1,tv2,0,0,color);
        p.AddVert(wall[wallIndex].x/64.0,-ceilingz1/1024.0,
            -wall[wallIndex].y/64.0,tu1,tv1,0,0,color);
        p.AddVert(wall[wall[wallIndex].point2].x/64.0,
            -ceilingz2/1024.0,-wall[wall[wallIndex].point2].y/64.0,
            tu2,tv3,0,0,color);
        p.AddVert(wall[wall[wallIndex].point2].x/64.0,
            -floorz2/1024.0,-wall[wall[wallIndex].point2].y/64.0,
            tu2,tv4,0,0,color);
        if ((!(wall[wallIndex].cstat&0x20))&&(!(wall[wallIndex].cstat&0x10)))
            p.flags|=PF_TWOSIDED;
        p.texture=art->tex[0][texnum];
        p.id=wallIndex;
        p.sect=i;
        // Add it to the polygon list
        if (wall[wallIndex].nextsector==-1)
            AddPolyToList(sectorPolys[i],p);
        else
            AddPolyToList(sectorWindowPolys[i],p);
    }
}

void Map::AddFloorToFloorWallPoly(int i,int wallIndex)
{
    // Generates polygons for the walls that result
    // when the floor changes height
    // Convert the shade value for this wall into
    // a D3D color value
    int shade=wall[wallIndex].shade*4;
    if (shade<-128) shade=-128;
    if (shade>127) shade=127;
    unsigned char light=~(shade+0x80);
    int color=0xff000000|(light<<16)|(light<<8)|light;
    // Only try to generate the polygon if there is a
    // sector on the other side
    if (wall[wallIndex].nextsector!=-1)
    {
        // The wall only belongs to this sector if the
        // floor is higher in the next sector than it is
        // in this one
        if ((sector[i].floorz>sector[wall[wallIndex].nextsector].floorz)&&(!(sector[i].floorstat&1))&&(!(sector[wall[wallIndex].nextsector].floorstat&1)))
        {
            float tu1=0,tv1=0,tu2=0,tv2=0;
            float tv3=0,tv4=0;
            art->RequestTexture(wall[wallIndex].picnum,0);
            if (art->tex[0][wall[wallIndex].picnum])
            {
                // Calculate texture coordinates of each
                // wall vertex
                tu1=((float)wall[wallIndex].xpanning)/art->sizeX[wall[wallIndex].picnum];
                tu2=((float)wall[wallIndex].xpanning+8.0*wall[wallIndex].xrepeat)/art->sizeX[wall[wallIndex].picnum];
                if ((wall[wallIndex].x==wall[wall[wallIndex].nextwall].x)&&
                    (wall[wallIndex].y==wall[wall[wallIndex].nextwall].y))
                {
                    if (wall[wallIndex].cstat&4)
                    {
                        tv2=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(i,wallIndex)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv1=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv4=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(i,wall[wallIndex].point2)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv3=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                    }
                    else
                    {
                        tv2=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(i,wallIndex)-sector[wall[wallIndex].nextsector].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv1=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)-sector[wall[wallIndex].nextsector].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv4=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(i,wall[wallIndex].point2)-sector[wall[wallIndex].nextsector].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv3=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)-sector[wall[wallIndex].nextsector].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                    }
                }
                else
                {
                    if (wall[wallIndex].cstat&4)
                    {
                        tv2=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(i,wallIndex)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv1=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv4=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(i,wall[wallIndex].point2)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv3=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                    }
                    else
                    {
                        tv2=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(i,wallIndex)-sector[wall[wallIndex].nextsector].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv1=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)-sector[wall[wallIndex].nextsector].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv4=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(i,wall[wallIndex].point2)-sector[wall[wallIndex].nextsector].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv3=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcFloorZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)-sector[wall[wallIndex].nextsector].floorz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                    }
                }
                if (wall[wallIndex].cstat&8)
                {
                    int temp=tu1;
                    tu1=tu2;
                    tu2=temp;
                }
            }
            Poly p;
            // Generate the polygon, making sure the wall vertices
            // from this sector match up with the wall vertices
            // from the next sector
            if ((wall[wallIndex].x==wall[wall[wallIndex].nextwall].x)&&
                (wall[wallIndex].y==wall[wall[wallIndex].nextwall].y))
            {
                p.AddVert(wall[wallIndex].x/64.0,-CalcFloorZ(i,wallIndex)/1024.0,
                    -wall[wallIndex].y/64.0,tu1,tv2,0,0,color);
                p.AddVert(wall[wallIndex].x/64.0,-CalcFloorZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)/1024.0,
                    -wall[wallIndex].y/64.0,tu1,tv1,0,0,color);
                p.AddVert(wall[wall[wallIndex].point2].x/64.0,
                    -CalcFloorZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)/1024.0,-wall[wall[wallIndex].point2].y/64.0,
                    tu2,tv3,0,0,color);
                p.AddVert(wall[wall[wallIndex].point2].x/64.0,
                    -CalcFloorZ(i,wall[wallIndex].point2)/1024.0,-wall[wall[wallIndex].point2].y/64.0,
                    tu2,tv4,0,0,color);
            }
            else
            {
                p.AddVert(wall[wallIndex].x/64.0,-CalcFloorZ(i,wallIndex)/1024.0,
                    -wall[wallIndex].y/64.0,tu1,tv2,0,0,color);
                p.AddVert(wall[wallIndex].x/64.0,-CalcFloorZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)/1024.0,
                    -wall[wallIndex].y/64.0,tu1,tv1,0,0,color);
                p.AddVert(wall[wall[wallIndex].point2].x/64.0,
                    -CalcFloorZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)/1024.0,-wall[wall[wallIndex].point2].y/64.0,
                    tu2,tv3,0,0,color);
                p.AddVert(wall[wall[wallIndex].point2].x/64.0,
                    -CalcFloorZ(i,wall[wallIndex].point2)/1024.0,-wall[wall[wallIndex].point2].y/64.0,
                    tu2,tv4,0,0,color);
            }
            p.flags|=PF_TWOSIDED;
            p.texture=art->tex[0][wall[wallIndex].picnum];
            p.id=wallIndex;
            p.sect=i;
            // Add the polygon to the list
            AddPolyToList(sectorPolys[i],p);
        }
    }
}

void Map::AddCeilingToCeilingWallPoly(int i,int wallIndex)
{
    // Generates polygons for the walls that result
    // when the ceiling changes height
    // Convert the shade value for this wall into
    // a D3D color value
    int shade=wall[wallIndex].shade*4;
    if (shade<-128) shade=-128;
    if (shade>127) shade=127;
    unsigned char light=~(shade+0x80);
    int color=0xff000000|(light<<16)|(light<<8)|light;
    // Only try to generate the polygon if there is a
    // sector on the other side
    if (wall[wallIndex].nextsector!=-1)
    {
        // The wall only belongs to this sector if the
        // ceiling is lower in the next sector than it is
        // in this one
        if ((sector[i].ceilingz<sector[wall[wallIndex].nextsector].ceilingz)&&(!((sector[i].ceilingstat&1)&&(sector[wall[wallIndex].nextsector].ceilingstat&1))))
        {
            float tu1=0,tv1=0,tu2=0,tv2=0;
            float tv3=0,tv4=0;
            art->RequestTexture(wall[wallIndex].picnum,0);
            if (art->tex[0][wall[wallIndex].picnum])
            {
                // Calculate texture coordinates of each
                // wall vertex
                tu1=((float)wall[wallIndex].xpanning)/art->sizeX[wall[wallIndex].picnum];
                tu2=((float)wall[wallIndex].xpanning+8.0*wall[wallIndex].xrepeat)/art->sizeX[wall[wallIndex].picnum];
                if ((wall[wallIndex].x==wall[wall[wallIndex].nextwall].x)&&
                    (wall[wallIndex].y==wall[wall[wallIndex].nextwall].y))
                {
                    if (wall[wallIndex].cstat&4)
                    {
                        tv1=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(i,wallIndex)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv2=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv3=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(i,wall[wallIndex].point2)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv4=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                    }
                    else
                    {
                        tv1=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(i,wallIndex)-sector[wall[wallIndex].nextsector].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv2=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)-sector[wall[wallIndex].nextsector].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv3=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(i,wall[wallIndex].point2)-sector[wall[wallIndex].nextsector].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv4=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)-sector[wall[wallIndex].nextsector].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                    }
                }
                else
                {
                    if (wall[wallIndex].cstat&4)
                    {
                        tv1=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(i,wallIndex)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv2=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv3=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(i,wall[wallIndex].point2)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv4=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)-sector[i].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                    }
                    else
                    {
                        tv1=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(i,wallIndex)-sector[wall[wallIndex].nextsector].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv2=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)-sector[wall[wallIndex].nextsector].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv3=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(i,wall[wallIndex].point2)-sector[wall[wallIndex].nextsector].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                        tv4=(wall[wallIndex].ypanning*(art->sizeY[wall[wallIndex].picnum]/256.0)+((wall[wallIndex].cstat&0x100)?-1:1)*(((CalcCeilingZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)-sector[wall[wallIndex].nextsector].ceilingz)*wall[wallIndex].yrepeat)/2048.0))/art->sizeY[wall[wallIndex].picnum];
                    }
                }
                if (wall[wallIndex].cstat&8)
                {
                    int temp=tu1;
                    tu1=tu2;
                    tu2=temp;
                }
            }
            Poly p;
            // Generate the polygon, making sure the wall vertices
            // from this sector match up with the wall vertices
            // from the next sector
            if ((wall[wallIndex].x==wall[wall[wallIndex].nextwall].x)&&
                (wall[wallIndex].y==wall[wall[wallIndex].nextwall].y))
            {
                p.AddVert(wall[wallIndex].x/64.0,-CalcCeilingZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)/1024.0,
                    -wall[wallIndex].y/64.0,tu1,tv2,0,0,color);
                p.AddVert(wall[wallIndex].x/64.0,-CalcCeilingZ(i,wallIndex)/1024.0,
                    -wall[wallIndex].y/64.0,tu1,tv1,0,0,color);
                p.AddVert(wall[wall[wallIndex].point2].x/64.0,
                    -CalcCeilingZ(i,wallIndex)/1024.0,-wall[wall[wallIndex].point2].y/64.0,
                    tu2,tv3,0,0,color);
                p.AddVert(wall[wall[wallIndex].point2].x/64.0,
                    -CalcCeilingZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)/1024.0,-wall[wall[wallIndex].point2].y/64.0,
                    tu2,tv4,0,0,color);
            }
            else
            {
                p.AddVert(wall[wallIndex].x/64.0,-CalcCeilingZ(wall[wallIndex].nextsector,wall[wall[wallIndex].nextwall].point2)/1024.0,
                    -wall[wallIndex].y/64.0,tu1,tv2,0,0,color);
                p.AddVert(wall[wallIndex].x/64.0,-CalcCeilingZ(i,wallIndex)/1024.0,
                    -wall[wallIndex].y/64.0,tu1,tv1,0,0,color);
                p.AddVert(wall[wall[wallIndex].point2].x/64.0,
                    -CalcCeilingZ(i,wallIndex)/1024.0,-wall[wall[wallIndex].point2].y/64.0,
                    tu2,tv3,0,0,color);
                p.AddVert(wall[wall[wallIndex].point2].x/64.0,
                    -CalcCeilingZ(wall[wallIndex].nextsector,wall[wallIndex].nextwall)/1024.0,-wall[wall[wallIndex].point2].y/64.0,
                    tu2,tv4,0,0,color);
            }
            p.flags|=PF_TWOSIDED;
            p.texture=art->tex[0][wall[wallIndex].picnum];
            p.id=wallIndex;
            p.sect=i;
            // Add the polygon to the list
            AddPolyToList(sectorPolys[i],p);
        }
    }
}

void Map::AddFloorPoly(int i)
{
    Poly p;
    // Do not draw the floor if it is paralaxing
    if (!(sector[i].floorstat&1))
    {
        // Convert the shade value for the floor into
        // a D3D color value
        int shade=sector[i].floorshade*4;
        if (shade<-128) shade=-128;
        if (shade>127) shade=127;
        unsigned char light=~(shade+0x80);
        int color=0xff000000|(light<<16)|(light<<8)|light;
        // Store the starting wall and the wall count
        int wallIndex=sector[i].wallptr;
        int count=sector[i].wallnum;
        float vectx,vecty,vectd2,vectd;
        // Compute a vector from the first to the second
        // vertex for texture alignment later on
        vectx=wall[wall[sector[i].wallptr].point2].x-wall[sector[i].wallptr].x;
        vecty=wall[wall[sector[i].wallptr].point2].y-wall[sector[i].wallptr].y;
        vectd2=vectx*vectx+vecty*vecty;
        vectd=sqrt(vectd2);
        // Store the value for how stretched the floor
        // texture is
        float div=16.0;
        if (sector[i].floorstat&8)
            div=8.0;
        // Create an initialize an array of flags that
        // reports whether each wall vertex has been used
        int *wallUsed=new int[count];
        memset(wallUsed,0,4*count);
        // Save the starting wall index
        int startWall=wallIndex;
        // Set up a SplitPolys structure to contain
        // the floor polygon
        SplitPolys polys;
        polys.polys=1;
        polys.poly[0].points=0;
        art->RequestTexture(sector[i].floorpicnum,0);
        for (;count;count--)
        {
            // Add the wall vertex to the floor polygon
            polys.poly[0].point[polys.poly[0].points].x=wall[wallIndex].x;
            polys.poly[0].point[polys.poly[0].points++].y=wall[wallIndex].y;
            // Mark the wall as used
            wallUsed[wallIndex-sector[i].wallptr]=1;
            // Go to the next wall
            wallIndex=wall[wallIndex].point2;
            // Check to see if we are back where we started
            if (wallIndex==startWall)
            {
                // The polygon is done, fix it so that all
                // concave polygons are converted into multiple
                // convex polygons
                FixPoly(polys);
                // Loop though each convex polygon
                for (int pn=0;pn<polys.polys;pn++)
                {
                    p.Clear();
                    // Loop through each vertex
                    for (int pt=0;pt<polys.poly[pn].points;pt++)
                    {
                        // Calculate the texture coordinates
                        // for this point
                        float tu=0,tv=0;
                        if (art->tex[0][sector[i].floorpicnum])
                        {
                            if (sector[i].floorstat&0x40)
                            {
                                float xd,yd;
                                xd=polys.poly[pn].point[pt].x-wall[sector[i].wallptr].x;
                                yd=polys.poly[pn].point[pt].y-wall[sector[i].wallptr].y;
                                float uvectd;
                                uvectd=(xd*vectx+yd*vecty)/vectd;
                                if (sector[i].floorstat&0x10)
                                    uvectd=-uvectd;
                                tu=(uvectd/div)/(float)art->sizeX[sector[i].floorpicnum]+(sector[i].floorxpanning/256.0);
                                float vvectd;
                                vvectd=(-xd*vecty+yd*vectx)/vectd;
                                if (sector[i].floorstat&0x20)
                                    vvectd=-vvectd;
                                tv=(vvectd/div)/(float)art->sizeY[sector[i].floorpicnum]+(sector[i].floorypanning/256.0);
                            }
                            else
                            {
                                float xd,yd;
                                xd=polys.poly[pn].point[pt].x;
                                if (sector[i].floorstat&0x10)
                                    xd=-xd;
                                yd=polys.poly[pn].point[pt].y;
                                if (sector[i].floorstat&0x20)
                                    yd=-yd;
                                tu=(xd/div)/(float)art->sizeX[sector[i].floorpicnum]+(sector[i].floorxpanning/256.0);
                                tv=(yd/div)/(float)art->sizeY[sector[i].floorpicnum]+(sector[i].floorypanning/256.0);
                            }
                            if (sector[i].floorstat&4)
                            {
                                float temp=tu;
                                tu=tv*(float)art->sizeY[sector[i].floorpicnum]/(float)art->sizeX[sector[i].floorpicnum];
                                tv=temp*(float)art->sizeX[sector[i].floorpicnum]/(float)art->sizeY[sector[i].floorpicnum];
                            }
                        }
                        // Add the vertex to the polygon object
                        p.AddVert(polys.poly[pn].point[pt].x/64.0,
                            -CalcFloorZAtPt(i,polys.poly[pn].point[pt].x,polys.poly[pn].point[pt].y)/1024.0,
                            -polys.poly[pn].point[pt].y/64.0,tu,tv,0,0,color);
                    }
                    // Set the texture for the polygon
                    p.texture=art->tex[0][sector[i].floorpicnum];
                    p.flags|=PF_TWOSIDED;
                    p.id=WALLINDEX_FLOOR;
                    p.sect=i;
                    // Add the polygon to the list
                    AddPolyToList(sectorPolys[i],p);
                }
                if (count>1)
                {
                    // If there are still vertices left, find
                    // one of them and get ready to add that
                    // polygon
                    polys.polys=1;
                    polys.poly[0].points=0;
                    for (wallIndex=sector[i].wallptr;;wallIndex++)
                    {
                        if (!wallUsed[wallIndex-sector[i].wallptr])
                            break;
                    }
                    startWall=wallIndex;
                }
            }
        }
        delete[] wallUsed;
    }
}

void Map::AddCeilingPoly(int i)
{
    Poly p;
    // Do not draw the ceiling if it is paralaxing
    if (!(sector[i].ceilingstat&1))
    {
        // Convert the shade value for the ceiling into
        // a D3D color value
        int shade=sector[i].ceilingshade*4;
        if (shade<-128) shade=-128;
        if (shade>127) shade=127;
        unsigned char light=~(shade+0x80);
        int color=0xff000000|(light<<16)|(light<<8)|light;
        // Store the starting wall and the wall count
        int wallIndex=sector[i].wallptr;
        int count=sector[i].wallnum;
        float vectx,vecty,vectd2,vectd;
        // Compute a vector from the first to the second
        // vertex for texture alignment later on
        vectx=wall[wall[sector[i].wallptr].point2].x-wall[sector[i].wallptr].x;
        vecty=wall[wall[sector[i].wallptr].point2].y-wall[sector[i].wallptr].y;
        vectd2=vectx*vectx+vecty*vecty;
        vectd=sqrt(vectd2);
        // Store the value for how stretched the ceiling
        // texture is
        float div=16.0;
        if (sector[i].ceilingstat&8)
            div=8.0;
        // Create an initialize an array of flags that
        // reports whether each wall vertex has been used
        int *wallUsed=new int[count];
        memset(wallUsed,0,4*count);
        // Save the starting wall index
        int startWall=wallIndex;
        // Set up a SplitPolys structure to contain
        // the ceiling polygon
        SplitPolys polys;
        polys.polys=1;
        polys.poly[0].points=0;
        art->RequestTexture(sector[i].ceilingpicnum,0);
        for (;count;count--)
        {
            // Add the wall vertex to the ceiling polygon
            polys.poly[0].point[polys.poly[0].points].x=wall[wallIndex].x;
            polys.poly[0].point[polys.poly[0].points++].y=wall[wallIndex].y;
            // Mark the wall as used
            wallUsed[wallIndex-sector[i].wallptr]=1;
            // Go to the next wall
            wallIndex=wall[wallIndex].point2;
            if (wallIndex==startWall)
            {
                // The polygon is done, fix it so that all
                // concave polygons are converted into multiple
                // convex polygons
                FixPoly(polys);
                // Loop though each convex polygon
                for (int pn=0;pn<polys.polys;pn++)
                {
                    p.Clear();
                    // Loop through each vertex
                    for (int pt=0;pt<polys.poly[pn].points;pt++)
                    {
                        // Calculate the texture coordinates
                        // for this point
                        float tu=0,tv=0;
                        if (art->tex[0][sector[i].ceilingpicnum])
                        {
                            if (sector[i].ceilingstat&0x40)
                            {
                                float xd,yd;
                                xd=polys.poly[pn].point[pt].x-wall[sector[i].wallptr].x;
                                yd=polys.poly[pn].point[pt].y-wall[sector[i].wallptr].y;
                                float uvectd;
                                uvectd=(xd*vectx+yd*vecty)/vectd;
                                if (sector[i].ceilingstat&0x10)
                                    uvectd=-uvectd;
                                tu=(uvectd/div)/(float)art->sizeX[sector[i].ceilingpicnum]+(sector[i].ceilingxpanning/256.0);
                                float vvectd;
                                vvectd=(-xd*vecty+yd*vectx)/vectd;
                                if (sector[i].ceilingstat&0x20)
                                    vvectd=-vvectd;
                                tv=(vvectd/div)/(float)art->sizeY[sector[i].ceilingpicnum]+(sector[i].ceilingypanning/256.0);
                            }
                            else
                            {
                                float xd,yd;
                                xd=polys.poly[pn].point[pt].x;
                                if (sector[i].ceilingstat&0x10)
                                    xd=-xd;
                                yd=polys.poly[pn].point[pt].y;
                                if (sector[i].ceilingstat&0x20)
                                    yd=-yd;
                                tu=(xd/div)/(float)art->sizeX[sector[i].ceilingpicnum]+(sector[i].ceilingxpanning/256.0);
                                tv=(yd/div)/(float)art->sizeY[sector[i].ceilingpicnum]+(sector[i].ceilingypanning/256.0);
                            }
                            if (sector[i].ceilingstat&4)
                            {
                                float temp=tu;
                                tu=tv*(float)art->sizeY[sector[i].ceilingpicnum]/(float)art->sizeX[sector[i].ceilingpicnum];
                                tv=temp*(float)art->sizeX[sector[i].ceilingpicnum]/(float)art->sizeY[sector[i].ceilingpicnum];
                            }
                        }
                        // Add the vertex to the polygon object
                        p.AddVert(polys.poly[pn].point[pt].x/64.0,
                            -CalcCeilingZAtPt(i,polys.poly[pn].point[pt].x,polys.poly[pn].point[pt].y)/1024.0,
                            -polys.poly[pn].point[pt].y/64.0,tu,tv,0,0,color);
                    }
                    // Set the texture for the polygon
                    p.texture=art->tex[0][sector[i].ceilingpicnum];
                    p.flags|=PF_TWOSIDED;
                    p.id=WALLINDEX_CEILING;
                    p.sect=i;
                    // Add the polygon to the list
                    AddPolyToList(sectorPolys[i],p);
                }
                if (count>1)
                {
                    // If there are still vertices left, find
                    // one of them and get ready to add that
                    // polygon
                    polys.polys=1;
                    polys.poly[0].points=0;
                    for (wallIndex=sector[i].wallptr;;wallIndex++)
                    {
                        if (!wallUsed[wallIndex-sector[i].wallptr])
                            break;
                    }
                    startWall=wallIndex;
                }
            }
        }
        delete[] wallUsed;
    }
}

void Map::ConvertSectorToPolyList(int i)
{
    // Delete old polygons for this sector if this is
    // an update operation (i.e. moving sector)
    FreeList(sectorPolys[i]);
    FreeList(sectorWindowPolys[i]);
    // Add all wall polygons
    for (int wallIndex=sector[i].wallptr;wallIndex<sector[i].wallptr+sector[i].wallnum;wallIndex++)
    {
        AddWallPoly(i,wallIndex);
        AddFloorToFloorWallPoly(i,wallIndex);
        AddCeilingToCeilingWallPoly(i,wallIndex);
    }
    // Add floor and ceiling polygons
    AddFloorPoly(i);
    AddCeilingPoly(i);
}

float Map::AngleToRadian(int ang)
{
    // Converts a Build-style angle into radians
    return -(ang*2.0*M_PI/2048.0);
}

void Map::ConvertSpriteToPolyList(int i)
{
    // Delete old polygon for sprite in case this
    // is an update operation
    FreeList(spritePolys[i]);
    // Return if invisible sprite
    if (sprite[i].cstat&0x8000) return;
    // Return if special (i.e. sector effector) sprite
    if (hideSpecialSprites)
    {
        if (sprite[i].picnum<=10) return;
        if (sprite[i].picnum==1405) return; // spawn point
    }
    // Return if completely "squished"
    if (sprite[i].xrepeat<=4) return;
    // Request texture for dynamic loading
    art->RequestTexture(sprite[i].picnum,0);
    if (((sprite[i].cstat>>4)&3)<2)
    {
        // Sprite is a "facing" or "wall" sprite
        // Calculate height values
        float z1=(sprite[i].cstat&0x80)?(sprite[i].z-art->origSizeY[sprite[i].picnum]*2.0*sprite[i].yrepeat):(sprite[i].z-art->origSizeY[sprite[i].picnum]*4.0*sprite[i].yrepeat);
        float z2=(sprite[i].cstat&0x80)?(sprite[i].z+art->origSizeY[sprite[i].picnum]*2.0*sprite[i].yrepeat):sprite[i].z;
        Poly p;
        // Get shade value and calculate the color
        // value from it
        int shade=sprite[i].shade*4;
        if (shade<-128) shade=-128;
        if (shade>127) shade=127;
        unsigned char light=~(shade+0x80);
        int color=((sprite[i].cstat&2)?0x80000000:0xff000000)|(light<<16)|(light<<8)|light;
        // Calculate the correct x and y coordinates
        // using the angle in the sprite data
        float x1=sprite[i].x+(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*cos(AngleToRadian(sprite[i].ang+512));
        float x2=sprite[i].x+(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*cos(AngleToRadian(sprite[i].ang-512));
        float y1=sprite[i].y-(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*sin(AngleToRadian(sprite[i].ang+512));
        float y2=sprite[i].y-(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*sin(AngleToRadian(sprite[i].ang-512));
        // Calculate texture coordinates
        float tu=0;
        float tv=0;
        float halfpixelu=0;
        float halfpixelv=0;
        if ((art->sizeX[sprite[i].picnum]!=0)&&(art->sizeY[sprite[i].picnum]!=0))
        {
            // Only attempt to calulate texture coordinates
            // if the texture actually exists
            tu=(float)art->origSizeX[sprite[i].picnum]/(float)art->sizeX[sprite[i].picnum];
            tv=(float)art->origSizeY[sprite[i].picnum]/(float)art->sizeY[sprite[i].picnum];
            halfpixelu=0.5/art->sizeX[sprite[i].picnum];
            halfpixelv=0.5/art->sizeY[sprite[i].picnum];
        }
        // Add the vertices to a Poly structure
        p.AddVert(x1/64.0,-z1/1024.0,-y1/64.0,halfpixelu,halfpixelv,0,0,color);
        p.AddVert(x2/64.0,-z1/1024.0,-y2/64.0,tu-halfpixelu,halfpixelv,0,0,color);
        p.AddVert(x2/64.0,-z2/1024.0,-y2/64.0,tu-halfpixelu,tv-halfpixelv,0,0,color);
        p.AddVert(x1/64.0,-z2/1024.0,-y1/64.0,halfpixelu,tv-halfpixelv,0,0,color);
        // Set the texture
        p.texture=art->tex[0][sprite[i].picnum];
        // Make the polygon two-sided if the sprite
        // flags are not set to one-sided
        if (!(sprite[i].cstat&0x40))
            p.flags|=PF_TWOSIDED;
        // Make the polygon always show up over a
        // wall to prevent sprite flickering problems
        // when the sprite and wall are in the same plane
        p.flags|=PF_ZBIAS;
        // Add the polygon to the polygon list for
        // this sprite
        AddPolyToList(spritePolys[i],p);
    }
    else
    {
        Poly p;
        // Get shade value and calculate the color
        // value from it
        int shade=sprite[i].shade*4;
        if (shade<-128) shade=-128;
        if (shade>127) shade=127;
        unsigned char light=~(shade+0x80);
        int color=((sprite[i].cstat&2)?0x80000000:0xff000000)|(light<<16)|(light<<8)|light;
        // Calculate the correct x and y coordinates
        // using the angle in the sprite data
        float x1=sprite[i].x+(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*cos(AngleToRadian(sprite[i].ang+512))+(art->origSizeY[sprite[i].picnum]/8.0*sprite[i].yrepeat)*cos(AngleToRadian(sprite[i].ang));
        float x2=sprite[i].x+(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*cos(AngleToRadian(sprite[i].ang-512))+(art->origSizeY[sprite[i].picnum]/8.0*sprite[i].yrepeat)*cos(AngleToRadian(sprite[i].ang));
        float x3=sprite[i].x+(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*cos(AngleToRadian(sprite[i].ang+512))+(art->origSizeY[sprite[i].picnum]/8.0*sprite[i].yrepeat)*cos(AngleToRadian(sprite[i].ang+1024));
        float x4=sprite[i].x+(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*cos(AngleToRadian(sprite[i].ang-512))+(art->origSizeY[sprite[i].picnum]/8.0*sprite[i].yrepeat)*cos(AngleToRadian(sprite[i].ang+1024));
        float y1=sprite[i].y-(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*sin(AngleToRadian(sprite[i].ang+512))-(art->origSizeY[sprite[i].picnum]/8.0*sprite[i].yrepeat)*sin(AngleToRadian(sprite[i].ang));
        float y2=sprite[i].y-(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*sin(AngleToRadian(sprite[i].ang-512))-(art->origSizeY[sprite[i].picnum]/8.0*sprite[i].yrepeat)*sin(AngleToRadian(sprite[i].ang));
        float y3=sprite[i].y-(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*sin(AngleToRadian(sprite[i].ang+512))-(art->origSizeY[sprite[i].picnum]/8.0*sprite[i].yrepeat)*sin(AngleToRadian(sprite[i].ang+1024));
        float y4=sprite[i].y-(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*sin(AngleToRadian(sprite[i].ang-512))-(art->origSizeY[sprite[i].picnum]/8.0*sprite[i].yrepeat)*sin(AngleToRadian(sprite[i].ang+1024));
        // Calculate texture coordinates
        float tu=0;
        float tv=0;
        float halfpixelu=0;
        float halfpixelv=0;
        if ((art->sizeX[sprite[i].picnum]!=0)&&(art->sizeY[sprite[i].picnum]!=0))
        {
            // Only attempt to calulate texture coordinates
            // if the texture actually exists
            tu=(float)art->origSizeX[sprite[i].picnum]/(float)art->sizeX[sprite[i].picnum];
            tv=(float)art->origSizeY[sprite[i].picnum]/(float)art->sizeY[sprite[i].picnum];
            halfpixelu=0.5/art->sizeX[sprite[i].picnum];
            halfpixelv=0.5/art->sizeY[sprite[i].picnum];
        }
        // Add the vertices to a Poly structure
        p.AddVert(x1/64.0,-sprite[i].z/1024.0,-y1/64.0,halfpixelu,halfpixelv,0,0,color);
        p.AddVert(x2/64.0,-sprite[i].z/1024.0,-y2/64.0,tu-halfpixelu,halfpixelv,0,0,color);
        p.AddVert(x4/64.0,-sprite[i].z/1024.0,-y4/64.0,tu-halfpixelu,tv-halfpixelv,0,0,color);
        p.AddVert(x3/64.0,-sprite[i].z/1024.0,-y3/64.0,halfpixelu,tv-halfpixelv,0,0,color);
        // Set the texture
        p.texture=art->tex[0][sprite[i].picnum];
        // Make the polygon two-sided so it always
        // displays correctly
        p.flags|=PF_TWOSIDED;
        // Make the polygon always show up over a
        // floor/ceiling to prevent sprite flickering
        // problems when the sprite and floor/ceiling
        // are in the same plane
        p.flags|=PF_ZBIAS;
        // Add the polygon to the polygon list for
        // this sprite
        AddPolyToList(spritePolys[i],p);
    }
}

void Map::MakeSpriteFaceViewer(int i)
{
    // Return if invisible sprite
    if (sprite[i].cstat&0x8000) return;
    // Return if not a "facing" sprite
    if (((sprite[i].cstat>>4)&3)!=0) return;
    // Return if sprite does not have a polygon generated
    if (!spritePolys[i].first) return;
    // Calculate the new x and y coordinates of the
    // sprite so that it faces the viewer
    float ang=render->GetYRotate()+M_PI/2;
    int x1=sprite[i].x+(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*cos(ang+M_PI/2);
    int x2=sprite[i].x+(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*cos(ang-M_PI/2);
    int y1=sprite[i].y-(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*sin(ang+M_PI/2);
    int y2=sprite[i].y-(art->origSizeX[sprite[i].picnum]/8.0*sprite[i].xrepeat)*sin(ang-M_PI/2);
    // Set the polygon vertices to the newly
    // calculated values
    spritePolys[i].first->poly.v[0].x=x1/64.0;
    spritePolys[i].first->poly.v[0].z=-y1/64.0;
    spritePolys[i].first->poly.v[1].x=x2/64.0;
    spritePolys[i].first->poly.v[1].z=-y2/64.0;
    spritePolys[i].first->poly.v[2].x=x2/64.0;
    spritePolys[i].first->poly.v[2].z=-y2/64.0;
    spritePolys[i].first->poly.v[3].x=x1/64.0;
    spritePolys[i].first->poly.v[3].z=-y1/64.0;
}

void Map::HideSpecialSprites(bool hide)
{
    hideSpecialSprites=hide;
    // Regenerate sprite polygons with the new setting
    for (int i=0;i<numSprites;i++)
        ConvertSpriteToPolyList(i);
}

bool Map::IsPtInTriangle(ProjVertex *v1,ProjVertex *v2,ProjVertex *v3,int x,int y)
{
    bool c=false;
    ProjVertex* v[3]; v[0]=v1; v[1]=v2; v[2]=v3;
    for (int i=0,j=2;i<3;j=i++)
    {
        if ((((v[i]->y<=y)&&(y<v[j]->y))||
            ((v[j]->y<=y)&&(y<v[i]->y)))&&
            (x<(v[j]->x-v[i]->x)*(y-v[i]->y)/
            (v[j]->y-v[i]->y)+v[i]->x))
            c=!c;
    }
    return c;
}

bool Map::IsPtInPoly(Poly& poly,int ptX,int ptY,float& z)
{
    // Check to see if (ptX,ptY) is in this polygon
    bool isInPoly=false;
    int n=poly.nVert;
    Vertex *tv=new Vertex[n];
    ProjVertex pv1,pv2,pv3;
    int start=-1;
    // Translate each vertex to view-relative coordinates
    for (int i=0;i<n;i++)
    {
        tv[i]=poly.v[i];
        render->TranslateVertex(tv[i]);
        if ((start==-1)&&(tv[i].z>=1))
            start=i;
    }
    // If there was no vertex visible, discard the polygon
    if (start==-1)
    {
        delete[] tv;
        return 0;
    }
    float projYMult=0.5*(float)render->GetScrnX()/(float)render->GetScrnY();
    pv1.w=1.0/tv[start].z;
    pv1.x=(tv[start].x*pv1.w*0.5+0.5)*render->GetScrnX();
    pv1.y=(-tv[start].y*pv1.w*projYMult+0.5)*render->GetScrnY();
    bool pt2valid=false;
    bool split=false;
    int i;
    for (i=(start+1)%n;i!=start;i=(i+1)%n)
    {
        if (isInPoly) break;
        if (tv[i].z<1)
        {
            if (split) continue;
            pv2=pv3;
            float t=(tv[i].z-1)/(tv[i].z-tv[(i+n-1)%n].z);
            float x=t*(tv[(i+n-1)%n].x-tv[i].x)+tv[i].x;
            float y=t*(tv[(i+n-1)%n].y-tv[i].y)+tv[i].y;
            pv3.w=1.0;
            pv3.x=(x*0.5+0.5)*render->GetScrnX();
            pv3.y=(-y*projYMult+0.5)*render->GetScrnY();
            if (pt2valid)
                isInPoly=IsPtInTriangle(&pv1,&pv2,&pv3,ptX,ptY);
            else
                pt2valid=true;
            split=true;
            continue;
        }
        pv2=pv3;
        if (split)
        {
            float t;
            if (tv[i].z!=tv[(i+n-1)%n].z)
                t=(tv[i].z-1)/(tv[i].z-tv[(i+n-1)%n].z);
            else
                t=0;
            float x=t*(tv[(i+n-1)%n].x-tv[i].x)+tv[i].x;
            float y=t*(tv[(i+n-1)%n].y-tv[i].y)+tv[i].y;
            pv3.w=1.0;
            pv3.x=(x*0.5+0.5)*render->GetScrnX();
            pv3.y=(-y*projYMult+0.5)*render->GetScrnY();
            if (pt2valid)
                isInPoly=IsPtInTriangle(&pv1,&pv2,&pv3,ptX,ptY);
            else
                pt2valid=true;
            split=false;
            i=(i+n-1)%n;
            continue;
        }
        pv3.w=1.0/tv[i].z;
        pv3.x=(tv[i].x*pv3.w*0.5+0.5)*render->GetScrnX();
        pv3.y=(-tv[i].y*pv3.w*projYMult+0.5)*render->GetScrnY();
        if (pt2valid)
            isInPoly=IsPtInTriangle(&pv1,&pv2,&pv3,ptX,ptY);
        else
            pt2valid=true;
    }
    if (split&&(!isInPoly))
    {
        pv2=pv3;
        float t;
        if (tv[i].z!=tv[(i+n-1)%n].z)
            t=(tv[i].z-1)/(tv[i].z-tv[(i+n-1)%n].z);
        else
            t=0;
        float x=t*(tv[(i+n-1)%n].x-tv[i].x)+tv[i].x;
        float y=t*(tv[(i+n-1)%n].y-tv[i].y)+tv[i].y;
        pv3.w=1.0;
        pv3.x=(x*0.5+0.5)*render->GetScrnX();
        pv3.y=(-y*projYMult+0.5)*render->GetScrnY();
        if (pt2valid)
            isInPoly=IsPtInTriangle(&pv1,&pv2,&pv3,ptX,ptY);
    }
    if (isInPoly)
    {
        // Calculate Z coordinate of hit point
        // z=-D/(AI/K+BJ/K+C)
        Vector normal=VectorCrossProduct(Vector(tv[1].x-tv[0].x,
            tv[1].y-tv[0].y,tv[1].z-tv[0].z),Vector(tv[2].x-
            tv[0].x,tv[2].y-tv[0].y,tv[2].z-tv[0].z));
        Plane p(normal.x,normal.y,normal.z,-(normal.x*
            tv[0].x+normal.y*tv[0].y+normal.z*tv[0].z));
        Vector viewVect(((float)ptX/(float)render->GetScrnX())*2.0-1.0,
            ((float)-ptY/(float)render->GetScrnY())*2.0+1.0,1);
        z=-p.D/(p.A*viewVect.x+p.B*viewVect.y+p.C);
    }
    delete[] tv;
    return isInPoly;
}

int Map::FindPolyAtPoint(int ptX,int ptY,int& type,int& index,int& sect)
{
    // Find which polygon is at 2D point (ptX,ptY) on the screen
    float bestZ=999999999.0;
    type=-1;
    index=-1;
    // Update view matrix for translating verticies
    // (Not done automatically in Direct3D renderer)
    render->UpdateViewMatrix();
    for (int sectNum=0;sectNum<numSectors;sectNum++)
    {
        PolyListElem *ptr=sectorPolys[sectNum].first;
        while (ptr)
        {
            float z;
            if (IsPtInPoly(ptr->poly,ptX,ptY,z))
            {
                if (z<bestZ)
                {
                    bestZ=z;
                    if (ptr->poly.id==WALLINDEX_CEILING)
                    {
                        type=POLYTYPE_CEILING;
                        index=sect=sectNum;
                    }
                    else if (ptr->poly.id==WALLINDEX_FLOOR)
                    {
                        type=POLYTYPE_FLOOR;
                        index=sect=sectNum;
                    }
                    else
                    {
                        type=POLYTYPE_WALL;
                        index=ptr->poly.id;
                        sect=ptr->poly.sect;
                    }
                }
            }
            ptr=ptr->next;
        }
    }
    if ((type!=-1)&&(index!=-1))
        return 1;
    return 0;
}

#pragma package(smart_init)
