/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.iep.editor;

import java.awt.BorderLayout;
import java.awt.Image;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.BeanInfo;
import java.io.IOException;
import java.util.logging.Level;
import javax.swing.ActionMap;
import javax.swing.text.DefaultEditorKit;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.CannotRedoException;
import javax.swing.SwingUtilities;
import javax.swing.Action;

import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.awt.UndoRedo;
import org.openide.cookies.SaveCookie;
import org.openide.nodes.Node.Cookie;
import org.openide.nodes.Node;
import org.openide.nodes.NodeAdapter;
import org.openide.nodes.NodeListener;
import org.openide.util.HelpCtx;
import org.openide.windows.Mode;
import org.openide.windows.TopComponent;
import org.openide.util.Lookup;
import org.openide.util.lookup.Lookups;
import org.openide.util.lookup.ProxyLookup;
import org.netbeans.spi.palette.PaletteController;

import org.netbeans.modules.iep.editor.designer.PlanDesigner;
import org.netbeans.modules.iep.editor.model.Plan;
import org.netbeans.modules.iep.editor.palette.IepPaletteFactory;
import org.openide.util.NbBundle;


/**
 * This class provides a container for a PlanDesigner to reside in.
 *
 * @author Bing Lu
 */
public class PlanTopComponent extends TopComponent {
    /**
     *  Logger.
     */
    private static final java.util.logging.Logger mLog = java.util.logging.Logger.getLogger(PlanTopComponent.class.getName());

    public static final String MODE_NAME = "EMS_PLAN_MODE";

    private PlanDesigner mPlanDesigner;

    private PlanNode mPlanNode;
    
    private PropertyChangeListener mDesignerListener;
    
    private NodeListener mNodeListener;
    
    private boolean mPlanDiscarded = false;
    
    /**
     * Constructor to create the top level IEP editor view panels
     *
     * @param node the associated the plan node
     */
    public PlanTopComponent(PlanNode node) {
        super();
        mPlanDesigner = new PlanDesigner(this);
        mDesignerListener = new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                if (evt.getPropertyName().equals(PlanDesigner.PLAN_DESIGNER_CHANGE)) {
                    addSaveCookie();
                }
            }
        };
        mPlanDesigner.getPropertyChangeSupport().addPropertyChangeListener(mDesignerListener);
        
        mPlanNode = node;
        mNodeListener = new NodeAdapter() {
            public void propertyChange(PropertyChangeEvent evt) {
                if (evt.getPropertyName().equals(PlanNode.PROP_DISPLAY_NAME)) {
                    if (PlanTopComponent.this.isModified()) {
                        PlanTopComponent.this.setNameLater(mPlanNode.getDisplayName() + "*");
                    } else {
                        PlanTopComponent.this.setNameLater(mPlanNode.getDisplayName());
                    }
                }
            }
        };
        mPlanNode.addNodeListener(mNodeListener);
        
        init();
    }

    /**
     * Constructor PlanTopComponent, for serialization only
     */
    public PlanTopComponent() {
        mLog.info("PlanTopComponent : empty Constructor");
    }
    
    private void init() {
        setLayout(new BorderLayout());
        add(mPlanDesigner, BorderLayout.CENTER);
        putClientProperty("PersistenceType", "Never");
        setEnabled(true);
        setName(mPlanNode.getDisplayName());
        setDisplayName(mPlanNode.getDisplayName());
        
        ActionMap map = this.getActionMap();
        map.put(DefaultEditorKit.copyAction, mPlanDesigner.mCopyAction);
        map.put(DefaultEditorKit.cutAction, mPlanDesigner.mCutAction);
        map.put(DefaultEditorKit.pasteAction, mPlanDesigner.mPasteAction);
        map.put("delete", mPlanDesigner.mDeleteAction);
        
        try {
            PaletteController pc = IepPaletteFactory.getPalette();
            Lookup lookup = new IepProxyLookup(new Lookup[] {
                Lookups.fixed(new Object[] {map, pc}),
                mPlanNode.getLookup()
            });                     
            associateLookup(lookup);
        } catch (Exception e) {
            mLog.log(Level.SEVERE, NbBundle.getMessage(PlanTopComponent.class,"PlanTopComponent.Fail_to_associate_palette_with_PlanTopComponent"), e);
        }

        initDocument();
    }

    public HelpCtx getHelpCtx() {
        return new HelpCtx("org.netbeans.modules.iep.editor.PlanTopComponent");   
    }

    public void initDocument() {
        mLog.info("PlanTopComponent : initDocument");
        Plan obj = mPlanNode.getPlan();

        if (obj == null) {
            mLog.info(NbBundle.getMessage(PlanTopComponent.class,"PlanTopComponent.plan_is_null"));
            return;
        }

        mPlanDesigner.setPlan(obj);
    }
    

    public void save() {
        if (mPlanDesigner != null) {
            mPlanDesigner.save();
            removeSaveCookie();
        }
    }


    /** Finds the workspace mode this business process belongs to (could be an eInsight, eVision
     *  or eXchange BP).
     *
     * @param   create      Flag to create one if it doesn't exists.
     * @return  the workspace mode or <code>null</code> if not found.
     */
    private Mode findMode(boolean create) {
        Mode mode = null;
        return mode;
    }

    /**
     * open the editor panel
     *
     */
    public void open() {
        mLog.info("PlanTopComponent : open");
        super.open();
        this.requestActive();
    }
    
    public Image getIcon() {
        return mPlanNode.getOpenedIcon(BeanInfo.ICON_COLOR_16x16);
    }

   //=========================================================================

   //=========================================================================
    /**
     * Implement the Save cookie
     * Cookie Support implementation for the SaveCookie
     */
    private SaveCookie mSaveCookie = new SaveCookie() {
        /**
         * Method save
         * @throws IOException  when encounter errors
         */
        public void save() throws IOException {
	    PlanTopComponent.this.save();
            removeSaveCookie();
        }
    };
    
    /** Helper method. Adds save cookie to the data object. */
    private void addSaveCookie() {
        PlanDataObject obj = (PlanDataObject)mPlanNode.getDataObject();
        obj.setModified(true);
        // Adds save cookie to the data object.
        if(obj.getCookie(SaveCookie.class) == null) {
            obj.addCookie(mSaveCookie);
            obj.setModified(true);
            setNameLater(mPlanNode.getDisplayName() + "*");
        }
    }

    /** Helper method. Removes save cookie from the data object. */
    private void removeSaveCookie() {
        PlanDataObject obj = (PlanDataObject)mPlanNode.getDataObject();
        
        // Remove save cookie from the data object.
        Cookie cookie = obj.getCookie(SaveCookie.class);

        if(cookie != null && cookie.equals(mSaveCookie)) {
            obj.removeCookie(mSaveCookie);
            obj.setModified(false);
            setNameLater(mPlanNode.getDisplayName());
        }
    }    
    
    private void setNameLater(final String name) {
        SwingUtilities.invokeLater(new Runnable() {
           public void run() {
               PlanTopComponent.this.setName(name);
               PlanTopComponent.this.setDisplayName(name);
           } 
        });
    }
    private boolean isModified() {
        if (mPlanNode != null) {
            PlanDataObject obj = (PlanDataObject)mPlanNode.getDataObject();
            return obj.isModified();
        }
        return false;
    }

    public boolean canClose() {
        boolean ans = true;
    	if (isModified()) {
            NotifyDescriptor d = new NotifyDescriptor.Confirmation(
                NbBundle.getMessage(PlanTopComponent.class,"PlanTopComponent.Intelligent_Event_Processor") 
                + " " 
                + mPlanNode.getDisplayName() 
                + " " + NbBundle.getMessage(PlanTopComponent.class,"PlanTopComponent.is_modified_Save_it"),
                NotifyDescriptor.YES_NO_CANCEL_OPTION);
            DialogDisplayer.getDefault().notify(d);
            if (NotifyDescriptor.YES_OPTION == d.getValue()) {
                try {
                    save();
                } catch (Exception e) {
                    e.printStackTrace();
                } finally {
                    //blow away everything and keep the state since it is alive
                    //this.mUndoManager.discardAllEdits();
                }
                ans = true;
            } else if (NotifyDescriptor.CANCEL_OPTION == d.getValue()) {
                ans = false;
            } else  if (NotifyDescriptor.NO_OPTION == d.getValue()) {
                discardPlan();
                removeSaveCookie();
//                this.mUndoManager.end(); //transform to compoundedit
//                try {
//                    this.mUndoManager.undo();
//                } catch (CannotUndoException e) {
//                    mLog.log(Level.SEVERE, e.getMessage(), e);
//                } finally {
//                    this.fireIrreversibleStateReached();
//                    //this.mUndoManager = new UndoManager();
//                }
                ans = true;
            }
        }
        if (ans) {
            setActivatedNodes(new Node[]{mPlanNode});
        }
        return ans;
    }
    
    private void discardPlan() {
        mPlanDiscarded = true;
        mPlanNode.discardPlan();
    }

    public boolean isPlanDiscarded() {
        return mPlanDiscarded;
    }
    //========================================================
    //========================================================
    public int getPersistenceType() { 
        return PERSISTENCE_NEVER;
    } 

    protected String preferredID() { 
        return MODE_NAME + Math.random();
    } 
    
    //======================== Undo & Redo ====================
    
    //======================== Undo & Redo ====================
    public UndoRedo getUndoRedo() { 
        return new PdUndoRedo();
    } 

    public class PdUndoRedo extends UndoRedo.Manager { 
        
        public String getUndoPresentationName() { 
            return (String)mPlanDesigner.mUndoAction.getValue(Action.SHORT_DESCRIPTION);
        } 

        public String getRedoPresentationName() { 
            return (String)mPlanDesigner.mRedoAction.getValue(Action.SHORT_DESCRIPTION);
        } 

        public boolean canUndo() { 
            return mPlanDesigner.mUndoAction.canAct();
        } 

        public boolean canRedo() { 
            return mPlanDesigner.mRedoAction.canAct();
        } 

        public void undo() throws CannotUndoException { 
            mPlanDesigner.mUndoAction.actionPerformed(null);
        } 

        public void redo() throws CannotRedoException { 
            mPlanDesigner.mRedoAction.actionPerformed(null);
        } 
    } 
    
    private static final long serialVersionUID = -4406860224835839120L;
    
    static class IepProxyLookup extends ProxyLookup {
        
        IepProxyLookup(Lookup[] lookups) {
            super(lookups);
        }
        
        Lookup[] getSubLookups() {
            return getLookups();
        }
        
        void setSubLookups(Lookup[] lookups) {
            setLookups(lookups);
        }
        
    }
    
}



