/* 
 *   Creation Date: <1999/03/18 03:28:10 samuel>
 *   Time-stamp: <2001/06/18 19:56:26 samuel>
 *   
 *	<os_interface.c>
 *	
 *	Interface to the OS beeing emulated (calls are made through 
 *	the 'sc' instruction).
 *   
 *   Copyright (C) 1999, 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include <time.h>

#include "mol_assert.h"
#include "mac_registers.h"
#include "molcpu.h"
#include "os_interface.h"
#include "debugger.h"
#include "memory.h"
#include "rvec.h"

#define DEBUG
/* #define DISABLE_DRIVERS  */

static int osip_enabled( int, int *params );
static int osip_debugger( int, int *params );
static int osip_log_num( int, int *params );
static int osip_log_hex( int, int *params );
static int osip_log_str( int, int *params );
static int osip_log_putc( int, int *params );
static int osip_get_date( int, int *params );
static int osip_exit( int, int *params );
static int osip_bad_vector( int, int *params );

static int rvec_osi_syscall( int dummy_rvec );

static osi_proc osi_selectors[NUM_OSI_SELECTORS];

/* 
 * This function is called whenever the a 'sc' instruction is issued
 * and r3 & r4 contains the magic constants. The routine selector
 * is passed in r5, parameters in r6..r31. The return value goes
 * in r3. 
 *
 * Pointers passed *MUST* by MAC-PHYSICAL. There are
 * two main reasons for this:
 *
 *	1) MMU-mapped memory might not have a valid translation 
 *	   at the moment (a DSI-exception might be necessary)
 *
 *	2) Even if a mapping had a valid translation, it is quite 
 *	   expensive to do the translation from here.
 *
 * If the client has problems to allocate physical RAM (for instance
 * a user process in linux) we could on request map some space to a 
 * "high" memory addresss, but below 0x80000000 in order to avoid 
 * conflicts with dynamically mapped PCI space).
 */

void
os_interface_init( void )
{
	int i;
	for( i=0; i<NUM_OSI_SELECTORS; i++ )
		os_interface_add_proc( i, osip_bad_vector );

	os_interface_add_proc( OSI_ENABLED, osip_enabled );
	os_interface_add_proc( OSI_DEBUGGER, osip_debugger );
	os_interface_add_proc( OSI_LOG_NUM, osip_log_num );
	os_interface_add_proc( OSI_LOG_STR, osip_log_str );
	os_interface_add_proc( OSI_LOG_HEX, osip_log_hex );
	os_interface_add_proc( OSI_LOG_PUTC, osip_log_putc );
	os_interface_add_proc( OSI_GET_DATE, osip_get_date );
	os_interface_add_proc( OSI_EXIT, osip_exit );

	set_rvector( RVEC_OSI_SYSCALL, rvec_osi_syscall, "OSI SysCall", kFPU_Unsafe );
}

void
os_interface_cleanup( void )
{
}

int 
rvec_osi_syscall( int dummy_rvec )
{
	int sel = mregs->gpr[5];
#ifdef DEBUG
	mregs->dbg_last_osi = sel;
#endif
	if( (sel < 0 || sel >= NUM_OSI_SELECTORS) ) {
		printm("Bogus selector %d in os_interface_call!\n", sel);
		return 0;
	}
	mregs->gpr[3] = osi_selectors[sel](sel, (int*)&mregs->gpr[6] );
	return 0;
}

int
os_interface_add_proc( int sel, osi_proc proc )
{
	assert( sel >= 0 && sel < NUM_OSI_SELECTORS );
	assert( !osi_selectors[sel] || osi_selectors[sel] == osip_bad_vector );

	osi_selectors[sel] = proc;
	return 0;
}

int
os_interface_remove_proc( int sel )
{
	assert( sel >= 0 && sel < NUM_OSI_SELECTORS );

	if( !osi_selectors[sel] ) {
		printm("os_interface: No such selector (%d)\n", sel);
		return 1;
	}
	osi_selectors[sel]=NULL;
	return 0;
}



/************************************************************************/
/*	OSI functions							*/
/************************************************************************/

static int
osip_enabled( int sel, int *params )
{
	/* printm("************ osip_enabled queried *************\n"); */
#ifdef DISABLE_DRIVERS
	/* It is difficult to turn of drivers - holding down
	 * shift doesn't help. It is simpler to let the driver
	 * check this flag and recompile.
	 */
	printm("os_interface: DISABLED\n");
	return 0;
#else
	return 1;
#endif
}

static int
osip_debugger( int sel, int *params )
{
	printm("*** osi user break [%d] ***\n", params[0] );
	stop_emulation();
	return 0;
}

static int
osip_log_num( int sel, int *params )
{
	printm("osi_log: %d\n",params[0] );
	return 0;
}

static int
osip_log_hex( int sel, int *params )
{
	printm("osi_log: %08X\n",params[0] );
	return 0;
}

static int
osip_log_str( int sel, int *params )
{
	char *str, *p;
	
	if( mphys_to_lvptr( params[0], &str ) < 0 ){
		printm("osi_log_str: Address not in RAM!\n");
		return -1;
	}

	/* substitute '\r' to '\n' */
	for(p=str; *p; p++ )
		if( *p == '\r' )
			*p = '\n';

	printm("<*> %s", str );
	return 0;
}

static int
osip_log_putc( int sel, int *params )
{
	printm("%c", params[0] );
	return 0;
}

static int
osip_get_date( int sel, int *params )
{
	time_t gmt;
	struct tm *ltime;                                                                 

	gmt = time( NULL );
	ltime = localtime( &gmt );                                                        	

	/* 1904 -> 1970 conversion with time zone adjustment */
	return gmt + 0x7C25B080 - timezone + (ltime->tm_isdst ? 3600 : 0);                
}

static int
osip_exit( int sel, int *params )
{
	exit(0);
	return 0;
}

static int
osip_bad_vector( int sel, int *params )
{
	printm("Uninitialized osip vector %d\n", sel);
	return 0;
}

