/* 
 *   Creation Date: <1999/05/23 00:00:00 BenH>
 *   
 *	<osi_enet.c>
 *	
 *	 Mac-On-Linux project
 *
 *   OSI Ethernet interface
 *   
 *   Copyright (C) 1999, 2000, 2001 Benjamin Herrenschmidt (bh40@calva.net)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

/* #define VERBOSE */

#include "mol_config.h"
#include <signal.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/poll.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <asm/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/wait.h>
#include <linux/netlink.h>
#include <linux/if.h>

#include "booter.h"
#include "debugger.h"
#include "os_interface.h"
#include "osi_driver.h"
#include "osi_enet.h"
#include "memory.h"
#include "verbose.h"
#include "res_manager.h"
#include "timer.h"
#include "ioports.h"
#include "pic.h"
#include "driver_mgr.h"
#include "async.h"
#include "session.h"
#include "ip.h"

#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#define HAS_TUN
#include <linux/if_tun.h>
#endif

SET_VERBOSE_NAME("osi_enet")

#define RX_QUEUE_SIZE			32

#define ETH_ADDR_MULTICAST		0x1
#define ETH_ADDR_LOCALLY_DEFINED	0x2

/* From sheep.c */
#define SIOC_MOL_GET_IPFILTER		SIOCDEVPRIVATE
#define SIOC_MOL_SET_IPFILTER		(SIOCDEVPRIVATE + 1)

/* virtual ethernet addressed used */
static char sheep_virtual_hw_addr[6] = {0xFE, 0xFD, 0xDE, 0xAD, 0xBE, 0xEF };
#define TAP_VIRTUAL_HW_ADDR 		0xDEADBEEF
#define TUN_VIRTUAL_HW_ADDR 		0x0DEADBEE

/************************************************************************/
/*	PCI constants (XXX: hardcoded to slot C1)			*/
/************************************************************************/

#define VENDOR		0x6666
#define DEVICE_ID	0x7777
#define CLASS_CODE	0x0200

static pci_dev_info_t enet_pci_config = {
	/* vendor, device ID, revision, class */
	VENDOR, DEVICE_ID, 0x00, CLASS_CODE
};

/************************************************************************/
/*	static declarations						*/
/************************************************************************/

struct enet_iface;

static int 	osi_enet_init( void );
static void 	osi_enet_cleanup( void );
static int 	add_netdev( char *str, int flags );
static int	save_enet_state( void );
static void	load_enet_state( void );

static int	osip_open( int sel, int *params );
static int	osip_close( int sel, int *params );
static int	osip_get_address( int sel, int *params );
static int	osip_get_status( int sel, int *params );
static int	osip_control( int sel, int *params );
static int	osip_get_packet( int sel, int *params );
static int	osip_send_packet( int sel, int *params );
static int	osip_add_multicast( int sel, int *params );
static int	osip_del_multicast( int sel, int *params );

static void	check_interrupts( struct enet_iface *is );
static void 	rx_packet_handler( int fd, int events );


/************************************************************************/
/*	packet drivers							*/
/************************************************************************/

static int	tap_open( struct enet_iface *is, char *intf_name, int *sigio_capable );
static int	sheep_open( struct enet_iface *is, char *intf_name, int *sigio_capable );
static int	sheep_add_multicast( struct enet_iface *is, char *addr );
static int	sheep_del_multicast( struct enet_iface *is, char *addr );
static int	sheep_load_save_state( struct enet_iface *is, struct enet_iface *load_is, int index, int loading );
#ifdef HAS_TUN
static int	tun_open( struct enet_iface *is, char *intf_name, int *sigio_capable );
#endif

typedef struct
{
	char *name;
	int packet_driver_id;	/* f_xxx */

	int (*open)( struct enet_iface *is, char *intf_name, int *sigio_capable );
	int (*close)( struct enet_iface  *is );
	int (*add_multicast)( struct enet_iface *is, char *addr );
	int (*del_multicast)( struct enet_iface *is, char *addr );
	int (*load_save_state)( struct enet_iface *is, struct enet_iface *load_is, int index, int loading );
} packet_driver_t;

#define PD	(*is->pd)

enum{ f_tap=1, f_sheep=2, f_tun=4, f_driver_id_mask=7 };

static packet_driver_t sheep_pd = {
	name:			"sheep",
	packet_driver_id: 	f_sheep,
	open: 			sheep_open,
	add_multicast:		sheep_add_multicast,
	del_multicast:		sheep_del_multicast,
	load_save_state:	sheep_load_save_state
};
static packet_driver_t tap_pd = {
	name:			"tap",
	packet_driver_id: 	f_tap,
	open: 			tap_open,
};

#ifdef HAS_TUN
static packet_driver_t tun_pd = {
	name:			"tun",
	packet_driver_id: 	f_tun,
	open: 			tun_open,
};
#endif

static packet_driver_t *packet_drivers[] = {
#ifdef HAS_TUN
	&tun_pd,
#endif
	&sheep_pd, 
	&tap_pd,
	NULL
};

static opt_entry_t netdev_opts[] = {
#ifdef HAS_TUN
	{"-tun",	f_tun },
#endif
	{"-tap",	f_tap },
	{"-sheep",	f_sheep },
	{NULL, 0 }
};


/************************************************************************/
/*	State structs							*/
/************************************************************************/

#define MAX_ENET_IFS		3

#define TAP_PACKET_PAD		2		/* used by read & writes */
#define MAX_PACKET_SIZE		1514
#define PACKET_BUF_SIZE		(1514 + TAP_PACKET_PAD)


driver_interface_t osi_enet_driver = {
    "osi_enet", osi_enet_init, osi_enet_cleanup
};

typedef int (*add_multicast_proc_t)( struct enet_iface *is, char *addr );
typedef int (*del_multicast_proc_t)( struct enet_iface *is, char *addr );
typedef int (*load_save_state_proc_t)( struct enet_iface *is, struct enet_iface *load_is, int index, int loading );

typedef struct rx_qentry {
	int		 p_size;		/* including pad bytes */
	unsigned char 	 packet[PACKET_BUF_SIZE];
	struct rx_qentry *next;
	struct rx_qentry *prev;
} rx_qentry_t;

typedef struct enet_iface {
	struct osi_driver *osi_driver;

	/* Rx queue */
	rx_qentry_t 	*rx_qhead;
	rx_qentry_t	*rx_qtail;
	rx_qentry_t	*free_head;
	char		*queue_space;
	
	/* Kernel device */
	int		net_dev;
	
	/* Ethernet address */
	unsigned char	eth_addr[6];
	
	int		irq_level;
	volatile int	started;

	/* Packet driver */
	packet_driver_t	*pd;

	int		packet_pad;		/* #bytes before the ethernet packet */
	char		drv_name[26];

	/* Sheep private */
	ulong		ipfilter;		/* for save/load state only */ 
} enet_iface_t;

static int 		numifs;
static enet_iface_t	enetif[MAX_ENET_IFS];
static int 		initialized=0;


/************************************************************************/
/*	Init / Cleanup							*/
/************************************************************************/

static int
osi_enet_init( void )
{
	int 	i, flags=0;
	char	*str;
	
	initialized = 0;
	numifs = 0;
	memset( (char*)enetif, 0, sizeof(enetif) );
#if 0
	ip_init();
#endif
	/* parse netdev resource (netdev: if [-sheep|-tap]) */
	for( i=0; ; i++ ) {
		if( !(str=get_str_res_ind("netdev",i,0 )) )
			break;
		if( numifs >= MAX_ENET_IFS ){
			printm("---> Too many netdevices specified\n");
			continue;
		}
		flags = parse_res_options("netdev", i, 1, netdev_opts, "---> Invalid netdev flag");

		if( str[0] != 't' )
			flags &= ~f_tap;
		if( !(flags & (f_sheep | f_tap | f_tun)) ) 
			flags |= (str[0] == 't')? f_tap : f_sheep;

		add_netdev( str, flags );
	}

	/* add osi functions */
	os_interface_add_proc( OSI_ENET_OPEN, osip_open );
	os_interface_add_proc( OSI_ENET_CLOSE, osip_close );
	os_interface_add_proc( OSI_ENET_GET_ADDR, osip_get_address );
	os_interface_add_proc( OSI_ENET_GET_STATUS, osip_get_status );
	os_interface_add_proc( OSI_ENET_CONTROL, osip_control );
	os_interface_add_proc( OSI_ENET_GET_PACKET, osip_get_packet );
	os_interface_add_proc( OSI_ENET_SEND_PACKET, osip_send_packet );
	os_interface_add_proc( OSI_ENET_ADD_MULTI, osip_add_multicast );
	os_interface_add_proc( OSI_ENET_DEL_MULTI, osip_del_multicast );

	session_save_proc( save_enet_state, NULL, kDynamicChunk );
	if( loading_session() )
		load_enet_state();

	initialized = 1;
	return 1;
}

static int
check_netdev( char *ifname )
{
	struct ifreq	ifr;
	int 		fd;
	int		ret=-1;
	
	if( (fd = socket( AF_INET, SOCK_DGRAM, 0 )) < 0 )
		return -1;

	memset( &ifr, 0, sizeof(ifr) );
	strncpy( ifr.ifr_name, ifname, sizeof(ifr.ifr_name) );
	if( ioctl( fd, SIOCGIFFLAGS, &ifr ) < 0 ){
		perrorm("SIOCGIFFLAGS");
		goto out;
	}
	if( !(ifr.ifr_flags & IFF_RUNNING ) ){
		printm("---> The network interface '%s' is not configured!\n", ifname);
		goto out;
	}
	if( (ifr.ifr_flags & IFF_NOARP ) ){
		printm("WARNING: Turning on ARP for device '%s'.\n", ifname);
		ifr.ifr_flags &= ~IFF_NOARP;
		if( ioctl( fd, SIOCSIFFLAGS, &ifr ) < 0 ) {
			perrorm("SIOCSIFFLAGS");
			goto out;
		}
	}
	ret = 0;
 out:
	close(fd);
	return ret;
}

static int 
add_netdev( char *str, int flags )
{
	enet_iface_t *is = &enetif[numifs];
	int 	i, sigio_capable=0;
	char	name[20];
	packet_driver_t **pd;
	
	memset( (char*)is, 0, sizeof(enet_iface_t) );

	for( pd=&packet_drivers[0]; *pd; pd++ )
		if( (**pd).packet_driver_id == (flags & f_driver_id_mask) )
			break;
	if( !*pd ) {
		printf("add_netdev: Internal error\n");
		return 1;
	}
	is->pd = *pd;

	/* Initialize device */
	if( PD.open( is, str, &sigio_capable ) ){
		printm("Failed to initialize the %s-<%s> device\n", PD.name, str);
		return 1;
	}

	snprintf( name, sizeof(name), "mol-enet-%d", numifs );
	if( !(is->osi_driver = register_osi_driver( "enet", name, &enet_pci_config )) ) {
		printm("Could not register the ethernet driver\n");
		return 1;
	}

	/* allocate rx queue elements */
	is->queue_space = calloc( RX_QUEUE_SIZE, sizeof(rx_qentry_t) );
	for(i=0; i<RX_QUEUE_SIZE; i++ ){
		rx_qentry_t *pkg = &((rx_qentry_t*)is->queue_space)[i];
		pkg->next = is->free_head;
		is->free_head = pkg;
	}

	/* Async I/O */
	if( add_async_handler( is->net_dev, POLLIN | POLLPRI, rx_packet_handler, sigio_capable ) < 0 ){
		LOG("-----> Could not add async handler!\n");
	}

	/* Initialize */
	is->irq_level = 0;
	is->started = 0;

	printm("Ethernet Interface (port %d) '%s' @ %02X:%02X:%02X:%02X:%02X:%02X\n",
	    numifs+1, is->drv_name, is->eth_addr[0], is->eth_addr[1], is->eth_addr[2],
		is->eth_addr[3], is->eth_addr[4], is->eth_addr[5]);

	numifs++;
	return 0;
}

static void
osi_enet_cleanup( void )
{
	int i;

	if( !initialized )
		return;
	initialized = 0;

	os_interface_remove_proc( OSI_ENET_OPEN );
	os_interface_remove_proc( OSI_ENET_CLOSE );
	os_interface_remove_proc( OSI_ENET_GET_ADDR );
	os_interface_remove_proc( OSI_ENET_GET_STATUS );	
	os_interface_remove_proc( OSI_ENET_CONTROL );	
	os_interface_remove_proc( OSI_ENET_GET_PACKET );
	os_interface_remove_proc( OSI_ENET_SEND_PACKET );
	os_interface_remove_proc( OSI_ENET_ADD_MULTI );
	os_interface_remove_proc( OSI_ENET_DEL_MULTI );

	for(i=0; i<numifs; i++ ) {
		enet_iface_t *is = &enetif[i];
		if( PD.close )
			PD.close(is);

		if (is->net_dev > 0)
			close(is->net_dev);
		is->net_dev = -1;

		if( is->queue_space )
			free( is->queue_space );
		free_osi_driver( is->osi_driver );
	}
	numifs = 0;
#if 0
	ip_cleanup();
#endif

	VPRINT("Enet cleanup done!\n");
}

/************************************************************************/
/*	Session support							*/
/************************************************************************/

static int
save_enet_state( void )
{
	enet_iface_t *is=enetif;
	int i;
	for(i=0; i<numifs; i++, is++ ) {
		if( PD.load_save_state && PD.load_save_state(is, NULL, i, 0) )
			return -1;
		if( write_session_data( "enet", i, (char*)is, sizeof(enet_iface_t) ))
			return -1;
	}
	return 0;
}

static void
load_enet_state( void )
{
	enet_iface_t lis, *is = enetif;
	int i;

	for(i=0; i<MAX_ENET_IFS; i++, is++ ){
		if( read_session_data( "enet", i, (char*)&lis, sizeof(lis) ) )
			break;
		if( PD.load_save_state && PD.load_save_state(is, &lis, i, 1) )
			session_failure("Unexpected error in load_save_state hook\n");

		is->started = lis.started;
	}
}


/************************************************************************/
/*	OS interface							*/
/************************************************************************/

#define ENETIF_FROM_ID( id ) ({ if( (id)<0 || (id)>=numifs ) return -1; &enetif[(id)]; })

static int
osip_open( int sel, int *params )
{
	int i;
	struct osi_driver *d;
	
	d = get_osi_driver_from_id( params[0] );
	
	for(i=0; d && i<numifs && enetif[i].osi_driver != d; i++ )
		;
	if( i>=numifs )
		return -1;
	return i;
}

static int
osip_close( int sel, int *params )
{
	VPRINT("osip_close()\n");
	return 0;
}

static int
osip_get_address( int sel, int *params )
{
	enet_iface_t *is = ENETIF_FROM_ID( params[0] );
	unsigned char* address;

	VPRINT("osip_get_address(@0x%08x)\n", params[1]);

	if( mphys_to_lvptr( params[1], (char**)&address ) < 0 ) {
		LOG("enet, osip_get_address(): Address not in RAM!\n");
		return -2;
	}

	/* XXX: This is more easily done by returning values in several registers 
	 * - avoids the necessity to pass locked, physically continuous memory. 
	 */
	address[0] = is->eth_addr[0];
	address[1] = is->eth_addr[1];
	address[2] = is->eth_addr[2];
	address[3] = is->eth_addr[3];
	address[4] = is->eth_addr[4];
	address[5] = is->eth_addr[5];
	return 0;
}

static int
osip_get_status( int sel, int *params )
{
	enet_iface_t *is = ENETIF_FROM_ID( params[0] );
	int result = 0;
	
	is->irq_level = 0;
	check_interrupts(is);
	
	if (is->started && is->rx_qhead ) {
		result |= ((is->rx_qhead->p_size - is->packet_pad) & 0xFFFFUL);
		result |= kEnetStatusRxPacket;
	}

	VPRINT("osip_get_status() -> 0x%08x\n", result);
	return result;
}

static int
osip_control( int sel, int *params )
{
	enet_iface_t *is = ENETIF_FROM_ID( params[0] );
	int command = params[1];

	VPRINT("osip_control(cmd: 0x%x, p:0x%x)\n", command, params[2]);
	
	switch(command) {
	case kEnetCommandStart:
		is->started = 1;
		break;
	case kEnetCommandStop:
		is->started = 0;

		/* flush queue */
		if( is->rx_qtail ) {
			is->rx_qtail->next = is->free_head;
			is->free_head = is->rx_qhead;
			is->rx_qtail = NULL;
			is->rx_qhead = NULL;
		}
		break;
	}
	check_interrupts(is);	
	return 0;
}

static int
osip_get_packet( int sel, int *params )
{
	enet_iface_t 	*is = ENETIF_FROM_ID( params[0] );
	int		result = -1;
	unsigned  char* address;	
	rx_qentry_t	*pkg;
	
	if ( !is->rx_qhead )		
		goto bail;
	
	/* copy and free package */
	pkg = is->rx_qhead;
	if( params[1] ) {
		if( mphys_to_lvptr( params[1], (char**)&address ) < 0 )
			LOG("enet, osip_get_packet(): Address not in RAM!\n");
		else
			memcpy(address, pkg->packet + is->packet_pad, pkg->p_size - is->packet_pad );
	}
	is->rx_qhead = is->rx_qhead->next;
	if( !is->rx_qhead )
		is->rx_qtail = NULL;
	pkg->next = is->free_head;
	is->free_head = pkg;
	result = 0;

bail:
	return result;
}

static int
osip_send_packet( int sel, int *params )
{
	enet_iface_t 	*is = ENETIF_FROM_ID( params[0] );
	int		i, size = params[2];
	unsigned char*	address;	
	char		buf[ PACKET_BUF_SIZE ];
	
	if( mphys_to_lvptr( params[1], (char**)&address ) < 0 ) {
		LOG("enet, osip_send_packet(): Address not in RAM!\n");
		return -1;
	}
	if( size > MAX_PACKET_SIZE ) {
		LOG("tx-packet too big\n");
		return -1;
	}

#if 0
	/* Put TCP/UDP packets into sockets */
	/* IMPORTANT: don't modify the packet - MacOS might want to reuse certain fields. */
	if( tx_packet_hook( address, size, is ) )
		return 0;
#endif
	/* Insert the PACKET_PAD bytes in front of the packet */
	memmove( buf + is->packet_pad, address, size );
	for(i=0; i<is->packet_pad; i++ )
		buf[i] = 0;
#if 0
	if( size<60 ) {
		printm("Too small outgoing packet (%d)\n", size );
		for(i=size; i<60; i++ )
			buf[i+is->packet_pad] = 0;
//		size = 60;
	}
#endif

	if( write(is->net_dev, buf, size + is->packet_pad ) != size + is->packet_pad ) {
		printm("osi_enet write failed\n");
		return -1;
	}
	return 0;
}

static int
osip_add_multicast( int sel, int *params )
{
	enet_iface_t 	*is = ENETIF_FROM_ID( params[0] );
	unsigned char*	address;	

	if( mphys_to_lvptr( params[1], (char**)&address ) < 0 ) {
		LOG("enet, osip_add_multicast(): Address not in RAM!\n");
		return -1;
	}
	
	VPRINT("osip_add_multicast, address is: %02x %02x %02x %02x %02x %02x\n",
		address[0], address[1], address[2],
		address[3], address[4], address[5]);

	if( PD.add_multicast )
		return (PD.add_multicast)( is, address );	
	return 0;
}


static int
osip_del_multicast( int sel, int *params )
{
	enet_iface_t 	*is = ENETIF_FROM_ID( params[0] );
	unsigned  char* address;

	if( mphys_to_lvptr( params[1], (char**)&address ) < 0 ) {
		LOG("enet, osip_del_multicast(): Address not in RAM!\n");
		return -1;
	}
	
	VPRINT("osip_del_multicast, address is: %02x %02x %02x %02x %02x %02x\n",
		address[0], address[1], address[2],
		address[3], address[4], address[5]);

	if( is->pd->del_multicast )
		return (*is->pd->del_multicast)( is, address );
	return 0;
}


/************************************************************************/
/*	Low-level implementation					*/
/************************************************************************/

static void
check_interrupts( enet_iface_t *is )
{
	if( is->irq_level ){
		osi_irq_hi( is->osi_driver );
	} else {
		osi_irq_low( is->osi_driver );
	}
}

/* space for the ethernet header should be included (but this
 * function will initialize it properly).
 */
int
synthesize_eth_packet( char *ethpacket, int size, void *iface )
{
	enet_iface_t 	*is = (enet_iface_t*)iface;
	rx_qentry_t 	*pkg;
	char 		*buf;
	
	// printm("Got synthesized packet size %d\n", size );

	if( size > MAX_PACKET_SIZE ) {
		printm("synthesize_ip_packet: Too big packet detected (%d)!\n", size);
		return -1;
	}
	if( !is->free_head ) {
		printm("No free buffers!\n");
		return -1;
	}
	buf = is->free_head->packet + is->packet_pad;
	memcpy( buf, ethpacket, size );
	memcpy( buf, &is->eth_addr, 6 );

#if 0
{
	int i;
	for(i=0; i<size; i++ ){
		printm("%02x ", buf[i] );
		if( i%32==31 )
			printm("\n");
	}
	printm("\n");
	//return 0;
}
#endif

	// Queue packet
	pkg = is->free_head;
	is->free_head = is->free_head->next;

	pkg->p_size = size;
	pkg->next = NULL;

	if( is->rx_qtail ){
		pkg->prev = is->rx_qtail;
		is->rx_qtail->next = pkg;
		is->rx_qtail = pkg;
	} else {
		is->rx_qtail = pkg;
		is->rx_qhead = pkg;
	}
	is->irq_level = 1;
	check_interrupts(is);

	return 0;
}

static void 
rx_packet_handler( int fd, int events )
{
	char discardbuf[PACKET_BUF_SIZE];
	rx_qentry_t 	*pkg;
	char 		*buf;
	int		i, size;
	enet_iface_t	*is;
	
	for( is=&enetif[0], i=0; i<numifs && is->net_dev != fd ; i++, is++ )
		;
	if( i >= numifs ){
		printm("rx_packet_handler: bad fd\n");
		return;
	}

	if( !(events & POLLIN) )
		printm("rx_packet_handler called with events %08X\n", events);

	for( ;; ) {
		if( is->free_head )
			buf = is->free_head->packet;
		else {
			static int warned=0;

			if( ++warned < 10 )
				printm("WARNING: Ethernet packet dropped\n");
			buf = discardbuf;
		}

		/* Read packet from device */
		if( (size = read(is->net_dev, buf, PACKET_BUF_SIZE )) < 0 ) {
			if( errno == EAGAIN )
				break;
			else
				perrorm("packet read");
		}
		
		if( size < 14 + is->packet_pad || !is->started || !is->free_head )
			continue;

		if( !(buf[0+is->packet_pad] & ETH_ADDR_MULTICAST) && memcmp( is->eth_addr, buf+is->packet_pad, 6) ) {
			// printm("Dropping packet with wrong HW-addr\n");
			continue;
		}
#if 0
		// Reception packet hook
		if( rx_packet_hook( buf + is->packet_pad, size - is->packet_pad, is ))
			continue;
#if 0
		if( size - is->packet_pad < 60 ) {
			for(i=size - is->packet_pad; i<60; i++ )
				buf[i+is->packet_pad] = 0;
			size = 60 + is->packet_pad;
		}
#endif
#endif

		/* Queue packet */

		// printm("Got packet size %d\n", size );
		pkg = is->free_head;
		is->free_head = is->free_head->next;

		/* Truncate packets containing more than 1514 bytes */
		if( size > is->packet_pad + MAX_PACKET_SIZE )
			size = is->packet_pad + MAX_PACKET_SIZE;

		pkg->p_size = size;
		pkg->next = NULL;

		if( is->rx_qtail ){
			pkg->prev = is->rx_qtail;
			is->rx_qtail->next = pkg;
			is->rx_qtail = pkg;
		} else {
			is->rx_qtail = pkg;
			is->rx_qhead = pkg;
		}
		is->irq_level = 1;
		check_interrupts(is);
	}
}


/************************************************************************/
/*	SHEEP Packet Drivers						*/
/************************************************************************/

static int
sheep_open( enet_iface_t *is, char *intf_name, int *sigio_capable )
{
	int 	fd;
	int 	non_block;
	
	/* verify that the device is up and running */
	if( check_netdev( intf_name ) )
		return 1;
	
	/* open sheep_net device */
	if( (fd=open("/dev/sheep_net", O_RDWR)) < 0 ) {
		printm("-----> Can't open /dev/sheep_net, please check module is present !\n");
		return 1;
	}

	/* attach to selected Ethernet card */
	if (ioctl(fd, SIOCSIFLINK, intf_name) < 0) {
		printm("-----> Can't attach to interface <%s>\n", intf_name);
		goto err;
	}

	/* set nonblocking I/O */
	non_block = 1;
	if( ioctl(fd, FIONBIO, &non_block) < 0 )
		perrorm("ioctl FIONBIO");
	
	/* get/set ethernet address */
	if( ioctl(fd, SIOCSIFADDR, sheep_virtual_hw_addr ) < 0 )
		printm("----> An old version of the sheep_net kernel module is probably running!\n");
	ioctl(fd, SIOCGIFADDR, is->eth_addr);

	/* set device specific variables */ 
	is->packet_pad = 0;
	is->net_dev = fd;
	*sigio_capable = 0;

	snprintf(is->drv_name, sizeof(is->drv_name), "sheep-<%s>", intf_name );
	return 0;

err:
	close(fd);
	return 1;
}

static int
sheep_add_multicast( enet_iface_t *is, char *addr )
{
	if( ioctl(is->net_dev, SIOCADDMULTI, addr) < 0 ) {
		LOG("sheep_add_multicast failed\n");
		return -1;
	}
	return 0;
}

static int 
sheep_del_multicast( enet_iface_t *is, char *addr )
{
	if( ioctl(is->net_dev, SIOCDELMULTI, addr) < 0 ) {
		LOG("sheep_del_multicast failed\n");
		return -1;
	}
	return 0;
}

static int
sheep_load_save_state( enet_iface_t *is, enet_iface_t *load_is, int index, int loading )
{
	/* XXX: Multicast addresses are not handled */
	if( loading ){
		if( ioctl( is->net_dev, SIOC_MOL_SET_IPFILTER, load_is->ipfilter ) < 0 )
			perrorm("SIOC_MOL_SET_IPFILTER");
	} else {
		if( ioctl( is->net_dev, SIOC_MOL_GET_IPFILTER, &is->ipfilter ) < 0 )
			perrorm("SIOC_MOL_GET_IPFILTER");
	}
	return 0;
}

/************************************************************************/
/*	TAP Packet Driver						*/
/************************************************************************/

static int
tap_open( enet_iface_t *is, char *intf_name, int *sigio_capable )
{
	struct sockaddr_nl nladdr;
	int fd, tapnum=0;
	char buf[16];
	
	if( intf_name ) {
		if( sscanf(intf_name, "tap%d", &tapnum)==1 ) {
			if( tapnum<0 || tapnum>15 ) {
				printf("Invalid tap device %s. Using tap0 instead\n", intf_name );
				intf_name = NULL;
				tapnum = 0;
			}
		} else {
			printm("Bad tapdevice interface '%s'\n", intf_name );
			printm("Using default tap device (tap0)\n");
			intf_name = NULL;
		}
	}
	if( !intf_name ) {
		sprintf(buf, "tap0");
		intf_name = buf;
	}

	/* verify that the device is up and running */
	if( check_netdev( intf_name ) )
		return 1;

	if( (fd = socket( PF_NETLINK, SOCK_RAW, NETLINK_TAPBASE+tapnum )) < 0 ) {
		perrorm("socket");
		printm("Does the kernel lack netlink support (CONFIG_NETLINK)?\n");
		return 1;
	}
	memset( &nladdr, 0, sizeof(nladdr) );
	nladdr.nl_family = AF_NETLINK;
	nladdr.nl_groups = ~0;
	nladdr.nl_pid = TAP_VIRTUAL_HW_ADDR;
	/* nladdr.nl_pid = getpid(); */
	if( bind( fd, (struct sockaddr*)&nladdr, sizeof(nladdr) ) < 0 ) {
		perrorm("bind");
		close( fd );
		return 1;
	}

	is->eth_addr[0] = is->eth_addr[1] = 0;
	*(ulong*)&is->eth_addr[2] = TAP_VIRTUAL_HW_ADDR;
	/* *(ulong*)&is->eth_addr[2] = getpid(); */

	is->net_dev = fd;
	is->packet_pad = TAP_PACKET_PAD;
	*sigio_capable = 1;
	snprintf(is->drv_name, sizeof(is->drv_name), "tap-<tap%d>", tapnum );
	return 0;
}


/************************************************************************/
/*	TUN/TAP Packet Driver						*/
/************************************************************************/

#ifdef HAS_TUN

static int 
script_exec( char *name, char *arg1, char *arg2 )
{
	int status;
	pid_t pid;
	
	if( !(pid=fork()) ){
		char buf[256];
		getcwd( buf, sizeof(buf));
		strncat(buf, "/bin/", sizeof(buf));
		strncat(buf, name, sizeof(buf));

		/* The enviornment is cleared to avoid security holes */
		if( arg2 )
			execle( buf, name, arg1, arg2, NULL, NULL);
		else if( arg1 )
			execle( buf, name, arg1, NULL, NULL);
		else
			execle( buf, name, NULL, NULL);
		perrorm("Failed to execute '%s'", buf);
		_exit(-1);
	}

	(void)wait(&status);
	if( !WIFEXITED(status) ) {
		printm("WIFEXITED() != 0\n");
		return -1;
	}
	return WEXITSTATUS(status);
}


static int
tun_open( enet_iface_t *is, char *intf_name, int *sigio_capable )
{
	struct ifreq ifr;
	int fd;
	
	if( !intf_name )
		intf_name = "mol";

	/* allocate tun/tap device */ 
	if( (fd = open("/dev/net/tun", O_RDWR)) < 0 ) {
		perrorm("Failed to open /dev/net/tun");
		return 1;
	}
	memset( &ifr, 0, sizeof(ifr) );
	ifr.ifr_flags = IFF_TAP | IFF_NO_PI;

	strncpy(ifr.ifr_name, intf_name, IFNAMSIZ );
	if( ioctl(fd, TUNSETIFF, &ifr) < 0 ){
		perrorm("TUNSETIFF");
		goto out;
	}

	/* don't checksum */
	ioctl( fd, TUNSETNOCSUM, 1 );

	/* Configure device */
	script_exec("tunconfig", intf_name, NULL );

	/* set HW address */
	is->eth_addr[0] = is->eth_addr[1] = 0;
	*(ulong*)&is->eth_addr[2] = TUN_VIRTUAL_HW_ADDR;

	/* finish... */
	is->net_dev = fd;
	*sigio_capable = 1;
	snprintf(is->drv_name, sizeof(is->drv_name), "tun-<%s>", intf_name );
	return 0;

 out:
	close(fd);
	return 1;
}

#endif /* HAS_TUN */
